;****************************************************************************
;*																			*
;*					LCD Controller fr Dualscan 640x480 LCD					*
;*																			*
;*								fr FT245									*
;*																			*
;*								Version 0.24								*
;*																			*
;*								 by Benedikt								*
;*																			*
;*						Email:	benedikt83 t gmx.net						*
;*																			*
;****************************************************************************
;*																			*
;*	Die Software darf frei kopiert und verndert werden, solange sie nicht	*
;*	ohne meine Erlaubnis fr kommerzielle Zwecke eingesetzt wird.			*
;*																			*
;****************************************************************************
;*																			*
;*	Einfacher LCD Controller fr ein SW 640x480 LCD, wie es in vielen alten	*
;*	Notebooks verbaut ist.													*
;*	Auer einem mega8515, einem 74x02, dem 74x573 Adresslatch im Datenbus, 	*
;*	und einem 64kB SRAM (mit 55ns oder schneller) werden keine weiteren		*
;*	Bauteile bentig. Neue Daten werden ber ein paralleles Interface mit	*
;*	bis zu 150kByte in den Speicher geladen.							 	*			
;*	Aufgrund der hohen Datenrate des Parallelports und der LCD Ausgang, 	*
;*	luft der AVR mit 16,000MHz und steuert das LCD mit rund 60Hz an.		*
;*																			*
;*	Fuse Bit High: 11001001	= 0XC9	(CKOPT enabled)							*
;*	Fuse Bit Low:  00101110	= 0x2E	(externer Quarz, BOD enabled)			*
;*																			*
;*																			*
;*	Befehle:																*
;*																			*
;*	001		Set Cursor. Gefolgt von zwei Parametern, nmlich der Zeilen		*
;*			und Spaltenadresse. Die Zeilenadresse liegt im Bereich 0-239,	*
;*			und die Spaltenadresse im Bereich 0-79 (plus 128, falls die 	*
;*			Position in der unteren Displayhlfte liegt.) 					*
;*																			*
;*	002		Set PWM. Stellt das Tastverhltnis des PWM Ausgangs OC1B ein.	*
;*			Das Tastverhltnis folgt auf diesen Befehl als Parameter.		*
;*																			*
;*	003		Set X Start. Stellt den Startwert fr den Zeilenbeginn ein.    	*
;*			Damit lsst sich ein Ausschnitt aus dem virtuellen 1000x480 	*
;*			Bild im Speicher auf dem LCD anzeigen.							*
;*																			*
;*	011		Clear LCD. Dieser Befehl lscht das LCD in extrem kurzer Zeit.	*
;*			Aber wie auch schon beim Clear Line Befehl, bentigt er einiges	*
;*			an Zeit um den kompletten Speicher zu lschen. Das LCD Timing	*
;*			wird zwar nicht ganz eingehalten, aber man merkt nichts davon,	*
;*			da ab und zu dummy Daten ans LCD gesendet werden.				*
;*																			*
;*	012		Read Byte. Es wird ein Byte gelesen.							*
;*																			*
;*	016		Setze Pixel 0 des Datenbytes an aktueller Adresse				*
;*	017		Setze Pixel 1 des Datenbytes an aktueller Adresse				*
;*	018		Setze Pixel 2 des Datenbytes an aktueller Adresse				*
;*	019		Setze Pixel 3 des Datenbytes an aktueller Adresse				*
;*	020		Setze Pixel 4 des Datenbytes an aktueller Adresse				*
;*	021		Setze Pixel 5 des Datenbytes an aktueller Adresse				*
;*	022		Setze Pixel 6 des Datenbytes an aktueller Adresse				*
;*	023		Setze Pixel 7 des Datenbytes an aktueller Adresse				*
;*																			*
;*	192		Display Character. Mit diesem Befehl wird ein ASCII Zeichen 	*
;*			mittels des interenen Character Generator dargestellt. Das     	*
;*			gewnschte Zeichen wird als Parameter bergeben. Die 			*
;*			Darstellung erfolgt als 8x8 Zeichen. Die Schriftart entspricht 	*
;*			der auf einem PC im 640x480 Grafikmodus.						*
;*																			*
;*	Daten:	Nach dem Senden eines Datenwertes wird dieser in den Speicher	*
;*			geschrieben, und der Adresszhler anschlieend erhht.			*
;*			Jedes Byte enthlt 8 nebeneinanderliegende Pixel. Im SRAM 		*
;*			werden dagegen 4 nebeneinander liegende Pixel aus der oberen,	*
;*			und 4 nebeneinander liegende Pixel aus der unteren Hlfte 		*
;*			gespeichert. Das mach das Laden von Daten etwas komplizierter,	* 
;*			vereinfacht und beschleunigt die Ausgabe aber extrem.			*
;*																			*
;*	IO Pins:																*
;*																			*
;*	PA0-7	Adr/Dat	Daten/Low Adresse fr den SRAM							*
;*	PB0-7	Daten	8 Bit Befehle/Daten fr den Controller					*
;*	PC0-7	Adresse	High Adresse fr den SRAM								*
;*	PD0		RXF\	RX Full von FT245										*
;*	PD1		RD\		Read Impuls an FT245									*
;*	PD2		NC																*
;*	PD3		EXSCL	Aktivierung fr XSCL. Dieser Pin aktiviert ber ein 	*
;*					74HC02 den Schiebetakt fr die Daten					*
;*	PD4		LP		Latchpuls. Der Impuls ld die Daten aus dem Schiebe-	*
;*					register ins Display. Dient als HSync.					*
;*	PD5		FLM		VSync fr das LCD. Das LCD springt in die erste Zeile.	*
;*	PD6		VWR\	Write Enable fr den SRAM.								*
;*	PD7		VRD\	Output Enable fr den SRAM und XSCL ber den 74HC32 	*
;*					fr das LCD. Bei jedem Lesevorgang werden die Daten ins	*
;*					LCD geladen												*
;*	PE0		EnLCD	Schaltet ber einen PNP und einen NPN Transistor die	*
;*					LCD Spannung, damit im Falle eines Fehlers (z.B. Ausfall*
;*					der +5V) das LCD nicht zerstrt wird.					*
;*	PE1		ALE 	Adress Latch fr Low Adressen des SRAMs                	*	
;*	PE2		PWM 	PWM Ausgang, z.B. fr Displaykontrast					*	
;*																			*
;****************************************************************************

.include "m8515def.inc"

.def null		=r0
.def voll		=r1
.def tempm		=r2
.def regsave	=r3

.def XStart		=r5				;Beginn einer Spalte ab Nibble
.def XScan		=r6				;Endadresse bei der Bildausgabe in X Richtung

.def temp		=r16
.def temp2		=r17
.def temp3		=r18
.def NextOp		=r19			;Zhler fr Parameter eines Befehls
.def Flags		=r20			;Allgemeine Bits
.def tempi		=r21
.def tempi2		=r22

.def XAdresseLow=r26			;Writepointer Spalte
.def ZeileL		=r27			;Writepointer Zeile LSB
.def AdresseLow	=r28			;Readpointer Spalte
.def AdresseHigh=r29			;Readpointer Zeile


;PortA
.equ 	RXF		=0				;RXF\
.equ 	TXE		=1				;TXE\


;PortD
.equ 	WR		=0				;WR\
.equ 	RD		=1				;RD\
.equ 	MOD		=2				;MOD
.equ 	XSCL	=3				;Enable XSCL
.equ 	LP		=4				;HSync
.equ 	FLM		=5				;VSync

;PortE
.equ 	EnLCD	=0				;LCD Spannung ein/ausschalten

;Flags
.equ 	ZeileH	=7				;Writepointer Zeile MSB, entscheidet zwischen obere und untere Displayhlte, also zwischen High und Low Nibble eines RAM Bytes
.equ	HSync	=5				;Flag fr LP

;NextOPs						;Werte um die nchste Operation nach einem Befehl festzulegen
.equ SetCurL	=1				
.equ SetCurH	=2
.equ SetPWMV	=3
.equ CGChar		=4
.equ SetScroll	=5

.equ Escape		=14				;Wert des Escape Zeichens (=Beginn eines Befehls)
.equ CD			=7				;Bit um zwischen Befehl/Daten zu unterscheiden


.equ XSize		=160				;Bildgre in X Richtung in Nibbles
.equ YSize		=240				;Bildgre in Y Richtung pro Halbbild

.equ XOScan		=0					;Anzahl der zustzlich auszugebenden Nibbles in X Richtung
.equ YOScan		=2					;Anzahl der zustzlich auszugebenden Zielen in Y Richtung (muss >=2 sein, sonst geht z.B. das Pollin LCD kaputt !)

.equ XMax		=250				;Maximale Speicheradresse in XRichtung

.equ YMin		=4					;Minimale Speicheradresse in Y Richtung
.equ YMax		=YMin+YSize			;Maximale Speicheradresse in Y Richtung

.equ YScan		=YMin+YSize+YOScan	;Endadresse bei der Bildausgabe in Y Richtung

.equ XPos		=200			;Position des Logos in X Richtung
.equ YPos		=216			;Position des Logos in Y Richtung
.equ LXSize		=30				;Gre des Logos in X Richtung durch 8
.equ LYSize		=48				;Gre des Logos in Y Richtung
								;Maximale Gre des Logos: 43008 Pixel

.equ CPUClock	=18432000		;Taktfrequenz

.equ FrameFreq	=60				;Bildwiederholfrequenz in Hz	
								;60Hz verursachen 55% CPU Auslastung bei 16MHz
.equ ACFreq		=180			;AC Drive (M) Frequenz in Hz

.equ TReload	= (CPUClock/8/(FrameFreq*(YSize+YOScan)))-1	;Wert fr Timer: Timertakt/Zeilenfrequenz-1
.equ ACDrive	= (CPUClock/2048/ACFreq)-1					;Wert fr CTC: Timertakt/2/Frequenz-1

#define		PWM					;Funktion von OC1B: PWM (fr Kontrast) oder M (AC Drive fr LCD)


;****************************************************************************
;	Interrupt Vektoren
;****************************************************************************

.org 0
rjmp Reset
.org int2addr     
reti
.org OC0addr
	sbi portd, LP			;HSync
	in regsave, sreg
	ori Flags, 1<<HSync		;HSync Flag setzen
	out sreg, regsave
	cbi portd, LP			;HSync
	cbi portd, FLM			;VSync Ende
reti

;****************************************************************************
;	CPU Register Initialisierung
;****************************************************************************

.org 32
reti

Reset:

ldi	temp, low(RAMEND)
out	SPL, temp					;setup stack pointer

ldi	temp, high(RAMEND)
out	SPH, temp					;setup stack pointer

clr null

ser temp
mov voll, temp

ldi temp, (1<<RD)|(1<<WR)|(1<<XSCL)|(1<<LP)|(1<<FLM)|(1<<MOD)|192
out ddrd, temp

sbi porte, EnLCD				;Disable LCD Voltage

out portb, voll
ldi temp, (1<<RD)|(1<<XSCL)|(1<<LP)|(1<<FLM)|(1<<MOD)|192
out portd, temp
out ddre, voll
sbi portd, XSCL

ldi temp, TReload				;Timer Reloadwert
out OCR0, temp

ldi temp, (1<<WGM01)|2			;2MHz Timer Takt, CTC
out TCCR0, temp								
						
ldi temp, 1<<OCIE0							
out timsk, temp

ldi temp, (1<<SRE)|(1<<ISC01)	;Enable XRAM
out mcucr, temp

#ifdef PWM
ldi temp, (1<<COM1B1)|(1<<COM1B0)|(1<<WGM10)	;Enable PWM (Kontrast)
out TCCR1A, temp

ldi temp, 192
out OCR1BL, temp				;PWM: 75% Duty Cycle

ldi temp, (1<<CS10)				;31,25kHz PWM Frequenz
out TCCR1B, temp
#endif

#ifdef M
ldi temp, (1<<COM1B0)
out TCCR1A,temp					;Toggle OC1B at CTC

ldi temp, ACDrive
out OCR1AL, temp				;M (AC Drive)

ldi temp, (1<<WGM12)|5	
out TCCR1B,temp					;Enable CTC, Clock= 18kHz
#endif

	clr XAdresseLow				;Adresse auf XRAM Beginn setzten
	ldi ZeileL, YMin
czeilc:							;SRAM lschen
	st X+, null
	cpi ZeileL, 255
	brne czeilc

ldi ZL, low(2*BMP)
ldi ZH, high(2*BMP)

clr Flags

ldi r22, LYSize					;Logo laden
.if (YPos>=YSize)				;Logo beginnt in unterer Bildhlfte
	ldi XH, YMin+YPos-1-YSize
	sbr Flags, 1<<ZeileH
.else							;Logo beginnt in oberer Bildhlfte
	ldi XH, YMin+YPos-1
.endif


YLoop:
	ldi r23, LXSize
	ldi XL, XPos/4
	inc XH
XLoop:
		lpm temp, Z+
		rcall Writebyte
		cpi ZeileL, YMax
		brne czeill
		ldi ZeileL, YMin
		ldi temp, (1<<ZeileH)
		eor Flags, temp
czeill:
		dec r23
		brne XLoop
	dec r22
	brne YLoop

wdr								;Watchdogtimer starten
in temp, WDTCR
ori temp, (1<<WDCE) | (1<<WDE)	;Zeit ca. 0,26s
out WDTCR, temp
ldi temp, (1<<WDE) | (1<<WDP2)
out WDTCR, temp
wdr

;****************************************************************************
;	Variablen Initialisierung
;****************************************************************************

clr temp
clr temp2
clr temp3

clr ZL
clr ZH
		
clr XAdresseLow	
ldi ZeileL, YMin			;XRAM Start Adress

clr Flags		
clr NextOp

clr AdresseLow				
ldi AdresseHigh, YMin		;Begin XRAM

ldi temp, 0
mov XStart, temp
ldi temp2, XSize+XOScan
add temp, temp2
mov XScan, temp

cbi portd, XSCL

sei

cbi porte, EnLCD			;Displayspannung einschalten

;****************************************************************************
;	Hauptprogramm mit Datenausgabe zum LCD
;****************************************************************************


Mainloop:
	sbis pinb, RXF			;Daten im Puffer des FT245 ?
		rcall Write

	sbrs Flags, HSync		;Delay um exakte Framerate zu erreichen. Eventuelle Verzgerungen durch Datenempfang werden dadurch ausgeglichen
		rjmp Mainloop

	cbr Flags, 1<<HSync		;Clear Flag
	mov AdresseLow, XStart	;Adresszhler auf nchste Zeile setzen
	inc AdresseHigh

Xshift:
	ld temp, Y+
	ld temp, Y+
	ld temp, Y+
	ld temp, Y+
	cp AdresseLow, XScan
	brlo XShift

	cpi AdresseHigh, YMin	;VSync ?
	brne SkipFLM
	sbi portd, FLM
	in temp, portb
	ldi temp2, 1<<MOD
	eor temp, temp2
	out portb, temp
SkipFLM:
	cpi AdresseHigh, YScan	;Adresszhler Reset ?
	brlo SkipAdRes
	wdr						;Watchdog Reset
	ldi AdresseHigh, YMin-1	;Adresszhler Reset
SkipAdRes:
	rjmp Mainloop

;****************************************************************************
;	Empfangsroutine fr Daten und Befehle
;****************************************************************************

Write:
	cbi portd, RD			;Byte lesen
	nop
	nop
	nop
	in temp, PINB
	sbi portd, RD

	sbrc NextOp, CD			;Befehl oder Daten ?	(Bit gesetzt fr Com, d.h. vorhergehendes Byte war Escape)
	rjmp Comm

	cpi temp, Escape		;Akteller Wert = Escape ?
	brne Daten				;Wenn nein, dann Bilddaten	
	sbr NextOp, 1<<CD		;Wenn ja, dann ist nchster Wert ein Befehl -> NextOp.Escape setzen.
ret

;*** Bearbeitung der Daten	***

Daten:
	cpi NextOp, 0			;Paramter oder Daten ?
	brne Next

	sbi portd, XSCL			;Datenverbindung zum LCD unterbrechen
	sbrc Flags, ZeileH		;Pixel in der oberen oder unteren Displayhlfte ?
	rjmp untereHalfte
obereHalfte:
	ld temp2, X				;Lader Wert aus SRAM
	andi temp2, 15			;obere Datenhlfte der SRAM Daten behalten
	mov temp3, temp
	andi temp3, 240			;erste Hlfte der Daten behalten
	or temp2, temp3			;Beide Hlften zusammenfgen
	st X+,temp2

	ld temp2, X
	andi temp2, 15			;obere Datenhlfte der SRAM Daten behalten
	swap temp
	andi temp, 240			;zweite Hlfte der Daten behalten
	or temp2, temp
	st X+,temp2

	cpi XAdresseLow, XMax	;Spalten Adresse > Max ?
	brlo normal
	clr XAdresseLow			;Spalten Adresse = 0
	inc ZeileL				;Zeilen Adresse ++
	cpi ZeileL, YMax		;Zeilen Adresse > Max ?
	brlo normal
	ldi ZeileL, YMin		;Zeile = Beginn XRAM
	sbr Flags, (1<<ZeileH)	;MSB Spalten Adresse = Displayhlfte
	cbi portd, XSCL
ret

untereHalfte:
	swap temp
	ld temp2, X
	andi temp2, 240			;untere Datenhlfte der SRAM Daten behalten
	mov temp3, temp
	andi temp3, 15			;erste Hlfte der Daten behalten
	or temp2, temp3
	st X+,temp2

	ld temp2, X
	andi temp2, 240			;untere Datenhlfte der SRAM Daten behalten
	swap temp
	andi temp, 15			;zweite Hlfte der Daten behalten
	or temp2, temp
	st X+,temp2

	cpi XAdresseLow, XMax	;Spalten Adresse > Max ?
	brlo normal
	clr XAdresseLow			;Spalten Adresse = 0
	inc ZeileL			
	cpi ZeileL, YMax		;Zeilen Adresse > Max ?
	brlo normal
	ldi ZeileL, YMin		;Zeile = Beginn XRAM
	cbr Flags, (1<<ZeileH)	;MSB Spalten Adresse = Displayhlfte
normal:
	cbi portd, XSCL			
ret

;*** Bearbeitung der Parameter	***

Next:
	cpi Nextop, SetCurL			;Lade Zeilenadresse
	brne skipn1	
	ldi ZeileL, YMin
	add ZeileL, temp
	brcs overflow
	ldi Nextop, SetCurH
ret

overflow:						;Ungltiger Wert: Reset auf Beginn
	ldi ZeileL, YMin
	ldi Nextop, SetCurH
ret

skipn1:
	cpi Nextop, SetCurH			;Lade Spaltenadresse + MSB Zeilenadresse
	brne skipn2
	mov XAdresseLow, temp
	lsl XAdresseLow

	bst temp, 7					;Lade Zeilenadresse MSB
	bld Flags, ZeileH
	ldi Nextop,0
ret

skipn2:
	cpi Nextop, SetPWMV			;Lade PWM
	brne skipn3
#ifdef PWM
	out OCR1BL, temp
#endif
	ldi Nextop, 0
ret

skipn3:
	cpi Nextop, CGChar			;Generiere Zeichen
	brne skipn4
	sbi portd, XSCL
	mov ZL, temp
	clr ZH
	lsl ZL
	rol ZH
	lsl ZL
	rol ZH
	lsl ZL
	rol ZH
	subi ZH, -4				;Add 3: Offset: 1024

	lpm temp, Z+
	rcall WriteByte
	inc ZeileL				;Zeilen Adresse ++
	subi XAdresseLow, 2

	lpm temp, Z+
	rcall WriteByte
	inc ZeileL				;Zeilen Adresse ++
	subi XAdresseLow, 2

	lpm temp, Z+
	rcall WriteByte
	inc ZeileL				;Zeilen Adresse ++
	subi XAdresseLow, 2

	lpm temp, Z+
	rcall WriteByte
	inc ZeileL				;Zeilen Adresse ++
	subi XAdresseLow, 2

	lpm temp, Z+
	rcall WriteByte
	inc ZeileL				;Zeilen Adresse ++
	subi XAdresseLow, 2

	lpm temp, Z+
	rcall WriteByte
	inc ZeileL				;Zeilen Adresse ++
	subi XAdresseLow, 2

	lpm temp, Z+
	rcall WriteByte
	inc ZeileL				;Zeilen Adresse ++
	subi XAdresseLow, 2

	lpm temp, Z+
	rcall WriteByte
	subi ZeileL, 7

	cpi XAdresseLow, 160	;Spalten Adresse > Max ?
	brne normal2
	clr XAdresseLow			;Spalten Adresse = 0
	subi ZeileL, -8			;Zeilen Adresse ++
	cpi ZeileL, YMax			;Zeilen Adresse > Max ?
	brlo normal2
	ldi ZeileL, YMin		;Zeile = Beginn XRAM
	ldi temp,  (1<<ZeileH)	;toggle Displayhlfte
	eor Flags, temp			;MSB Spalten Adresse = Displayhlfte
normal2:
	cbi portd, XSCL
	ldi Nextop, 0
ret

skipn4:
	cpi Nextop, SetScroll	;Lade Startwert fr Zeilenbeginn
	brne skipn5

	cpi temp, 90			;Zulssig: 0-89, anonsten crashed der Controller !
	brsh skipset
	mov XStart, temp
	ldi temp2, XSize+XOScan
	add temp, temp2
	mov XScan, temp
skipset:
	ldi Nextop,0
ret

skipn5:						;Ungltiger Wert: Reset auf 0
	ldi Nextop,0
ret

;*** Bearbeitung der Befehle	***

Datenjmp:
rjmp Daten

Comm:
	cbr Nextop, 1<<CD
	cpi temp, Escape		;Aktelles Zeichen = Escape ?
	breq Datenjmp			;Wenn ja dann war es ein doppeltes Escape, und das sind Daten

	clr Nextop
	cpi temp, 1				;Set Adress
	brne skip1
	ldi Nextop, SetCurL
ret

skip1:
	cpi temp, 2				;Set PWM
	brne skip10
	ldi Nextop, SetPWMV
ret

skip10:
	cpi temp, 11			;Clear LCD
	brne skip11
	sbi portd, XSCL
	clr XAdresseLow
	ldi ZeileL, YMin
czeil2:	
	st X+, null
	st X+, null
	st X+, null
	st X+, null
	st X+, null
	st X+, null
	st X+, null
	st X+, null
	cpi XAdresseLow, 2
	brsh czeil2
	sbi portd, LP			;HSync simulieren
	nop
	nop
	cbi portd, LP			;HSync simulieren
	cpi ZeileL, YMax
	brne czeil2
	ldi ZeileL, YMin
	cbr Flags, (1<<ZeileH)
	cbi portd, XSCL
ret

skip11:
	cpi temp, 12			;Read Byte
	brne skip12

	sbi portd, XSCL
	ld temp, X+
	ld temp2, X+

	sbrc Flags, ZeileH		;Pixel in der oberen oder unteren Displayhlfte ?
	rjmp untereHalfte3
obereHalfte3:
	andi temp, 240
	andi temp2, 240
	swap temp2
	or temp, temp2
	rjmp gesamt
untereHalfte3:
	andi temp, 15
	andi temp2, 15
	swap temp
	or temp, temp2
gesamt:
	cpi XAdresseLow, XMax	;Spalten Adresse > Max ?
	brlo normal3
	clr XAdresseLow			;Spalten Adresse = 0
	inc ZeileL			
	cpi ZeileL, YMax		;Zeilen Adresse > Max ?
	brlo normal3
	ldi ZeileL, YMin		;Zeile = Beginn XRAM
	cbr Flags, (1<<ZeileH)	;MSB Spalten Adresse = Displayhlfte
normal3:
	cbi portd, XSCL		

readwarte:
	sbic pinb, TXE			;Warten bis Platz im Puffer
	rjmp readwarte

	out ddrb, voll
	out portb, temp
	clr Nextop
	sbi portd, WR
	nop
	nop
	nop
	nop
	cbi portd, WR
	out ddrb, null
ret

skip12:	
	mov temp2, temp			;Set Pixel
	andi temp2, 248
	cpi temp2, 16
	brne skip16
	sbi portd, XSCL
	sbrc temp, 2
	inc XAdresseLow			;Nibble berspringen

	mov temp3, temp
	andi temp3, 3
	ldi temp2, 8
shift:						;temp2 = 1<<temp3
	cpi temp3, 0
	breq exitshift
	lsr temp2
	dec temp3
	rjmp shift
exitshift:
	sbrs Flags, ZeileH		;Pixel in der oberen oder unteren Displayhlfte ?
	swap temp2				;Untere Hlfte -> Nibble vertauschen
	ld temp3, X				;Lade Wert aus SRAM
	or temp3, temp2			;Pixel setzen
	st X+,temp3				;Wert in SRAM schreiben
	sbrs temp, 2
	inc XAdresseLow			;Nibble berspringen (so dass wieder eine gerade Adresse entsteht)
	cpi XAdresseLow, XMax	;Spalten Adresse > Max ?
	brlo normalp
	clr XAdresseLow			;Spalten Adresse = 0
	inc ZeileL			
	cpi ZeileL, YMax		;Zeilen Adresse > Max ?
	brlo normalp
	ldi ZeileL, YMin		;Zeile = Beginn XRAM
	cbr Flags, (1<<ZeileH)	;MSB Spalten Adresse = Displayhlfte
normalp:
	cbi portd, XSCL			
ret
		
skip16:			
	cpi temp, 192			;Character Generator
	brne skip17
	ldi Nextop, CGChar
ret

skip17:			
	cpi temp, 3				;Set Scroll
	brne skip255
	ldi Nextop, SetScroll
ret

skip255:					;Ungltiger Befehl: ignorieren und weitermachen

exint:
ret


WriteByte:					;Allgemeine Routine um ein Byte in den Speicher zu schreiben
	sbrc Flags, ZeileH		;Pixel in der oberen oder unteren Displayhlfte ?
	rjmp untereHalfte2

obereHalfte2:
	ld temp2, X				;Lader Wert aus SRAM
	andi temp2, 15			;obere Datenhlfte der SRAM Daten behalten
	mov temp3, temp
	andi temp3, 240			;erste Hlfte der Daten behalten
	or temp2, temp3			;Beide Hlften zusammenfgen
	st X+,temp2

	ld temp2, X
	andi temp2, 15			;obere Datenhlfte der SRAM Daten behalten
	swap temp
	andi temp, 240			;zweite Hlfte der Daten behalten
	or temp2, temp
	st X+,temp2
ret

untereHalfte2:
	swap temp
	ld temp2, X
	andi temp2, 240			;untere Datenhlfte der SRAM Daten behalten
	mov temp3, temp
	andi temp3, 15			;erste Hlfte der Daten behalten
	or temp2, temp3
	st X+,temp2

	ld temp2, X
	andi temp2, 240			;untere Datenhlfte der SRAM Daten behalten
	swap temp
	andi temp, 15			;zweite Hlfte der Daten behalten
	or temp2, temp
	st X+,temp2
ret

;*** CG ROM Daten *** (8x8 Zeichensatz)
.org 512
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 
.db  126 , 129 , 165 , 129 , 189 , 153 , 129 , 126 
.db  126 , 255 , 219 , 255 , 195 , 231 , 255 , 126 
.db  108 , 254 , 254 , 254 , 124 , 56 , 16 , 0 
.db  16 , 56 , 124 , 254 , 124 , 56 , 16 , 0 
.db  56 , 124 , 56 , 254 , 254 , 124 , 56 , 124 
.db  16 , 16 , 56 , 124 , 254 , 124 , 56 , 124 
.db  16 , 16 , 56 , 124 , 254 , 124 , 56 , 124 
.db  255 , 255 , 231 , 195 , 195 , 231 , 255 , 255 
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 
.db  127 , 99 , 127 , 99 , 99 , 103 , 230 , 192 
.db  153 , 90 , 60 , 231 , 231 , 60 , 90 , 153 
.db  128 , 224 , 248 , 254 , 248 , 224 , 128 , 0 
.db  2 , 14 , 62 , 254 , 62 , 14 , 2 , 0 
.db  24 , 60 , 126 , 24 , 24 , 126 , 60 , 24 
.db  102 , 102 , 102 , 102 , 102 , 0 , 102 , 0 
.db  127 , 219 , 219 , 123 , 27 , 27 , 27 , 0 
.db  62 , 99 , 56 , 108 , 108 , 56 , 204 , 120 
.db  0 , 0 , 0 , 0 , 126 , 126 , 126 , 0 
.db  24 , 60 , 126 , 24 , 126 , 60 , 24 , 255 
.db  24 , 60 , 126 , 24 , 24 , 24 , 24 , 0 
.db  24 , 24 , 24 , 24 , 126 , 60 , 24 , 0 
.db  0 , 24 , 12 , 254 , 12 , 24 , 0 , 0 
.db  0 , 48 , 96 , 254 , 96 , 48 , 0 , 0 
.db  0 , 48 , 96 , 254 , 96 , 48 , 0 , 0 
.db  0 , 48 , 96 , 254 , 96 , 48 , 0 , 0 
.db  0 , 48 , 96 , 254 , 96 , 48 , 0 , 0 
.db  0 , 48 , 96 , 254 , 96 , 48 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 
.db  48 , 120 , 120 , 48 , 48 , 0 , 48 , 0 
.db  108 , 108 , 108 , 0 , 0 , 0 , 0 , 0 
.db  108 , 108 , 254 , 108 , 254 , 108 , 108 , 0 
.db  48 , 124 , 192 , 120 , 12 , 248 , 48 , 0 
.db  0 , 198 , 204 , 24 , 48 , 102 , 198 , 0 
.db  56 , 108 , 56 , 118 , 220 , 204 , 118 , 0 
.db  96 , 96 , 192 , 0 , 0 , 0 , 0 , 0 
.db  24 , 48 , 96 , 96 , 96 , 48 , 24 , 0 
.db  96 , 48 , 24 , 24 , 24 , 48 , 96 , 0 
.db  0 , 102 , 60 , 255 , 60 , 102 , 0 , 0 
.db  0 , 48 , 48 , 252 , 48 , 48 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 48 , 48 , 96 
.db  0 , 0 , 0 , 252 , 0 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 48 , 48 , 0 
.db  6 , 12 , 24 , 48 , 96 , 192 , 128 , 0 
.db  124 , 198 , 206 , 222 , 246 , 230 , 124 , 0 
.db  48 , 112 , 48 , 48 , 48 , 48 , 252 , 0 
.db  120 , 204 , 12 , 56 , 96 , 204 , 252 , 0 
.db  120 , 204 , 12 , 56 , 12 , 204 , 120 , 0 
.db  28 , 60 , 108 , 204 , 254 , 12 , 30 , 0 
.db  252 , 192 , 248 , 12 , 12 , 204 , 120 , 0 
.db  56 , 96 , 192 , 248 , 204 , 204 , 120 , 0 
.db  252 , 204 , 12 , 24 , 48 , 48 , 48 , 0 
.db  120 , 204 , 204 , 120 , 204 , 204 , 120 , 0 
.db  120 , 204 , 204 , 124 , 12 , 24 , 112 , 0 
.db  0 , 48 , 48 , 0 , 0 , 48 , 48 , 0 
.db  0 , 48 , 48 , 0 , 0 , 48 , 48 , 96 
.db  24 , 48 , 96 , 192 , 96 , 48 , 24 , 0 
.db  0 , 0 , 252 , 0 , 0 , 252 , 0 , 0 
.db  96 , 48 , 24 , 12 , 24 , 48 , 96 , 0 
.db  120 , 204 , 12 , 24 , 48 , 0 , 48 , 0 
.db  124 , 198 , 222 , 222 , 222 , 192 , 120 , 0 
.db  48 , 120 , 204 , 204 , 252 , 204 , 204 , 0 
.db  252 , 102 , 102 , 124 , 102 , 102 , 252 , 0 
.db  60 , 102 , 192 , 192 , 192 , 102 , 60 , 0 
.db  248 , 108 , 102 , 102 , 102 , 108 , 248 , 0 
.db  254 , 98 , 104 , 120 , 104 , 98 , 254 , 0 
.db  254 , 98 , 104 , 120 , 104 , 96 , 240 , 0 
.db  60 , 102 , 192 , 192 , 206 , 102 , 62 , 0 
.db  204 , 204 , 204 , 252 , 204 , 204 , 204 , 0 
.db  120 , 48 , 48 , 48 , 48 , 48 , 120 , 0 
.db  30 , 12 , 12 , 12 , 204 , 204 , 120 , 0 
.db  230 , 102 , 108 , 120 , 108 , 102 , 230 , 0 
.db  240 , 96 , 96 , 96 , 98 , 102 , 254 , 0 
.db  198 , 238 , 254 , 254 , 214 , 198 , 198 , 0 
.db  198 , 230 , 246 , 222 , 206 , 198 , 198 , 0 
.db  56 , 108 , 198 , 198 , 198 , 108 , 56 , 0 
.db  252 , 102 , 102 , 124 , 96 , 96 , 240 , 0 
.db  120 , 204 , 204 , 204 , 220 , 120 , 28 , 0 
.db  252 , 102 , 102 , 124 , 108 , 102 , 230 , 0 
.db  120 , 204 , 224 , 112 , 28 , 204 , 120 , 0 
.db  252 , 180 , 48 , 48 , 48 , 48 , 120 , 0 
.db  204 , 204 , 204 , 204 , 204 , 204 , 252 , 0 
.db  204 , 204 , 204 , 204 , 204 , 120 , 48 , 0 
.db  198 , 198 , 198 , 214 , 254 , 238 , 198 , 0 
.db  198 , 198 , 108 , 56 , 56 , 108 , 198 , 0 
.db  204 , 204 , 204 , 120 , 48 , 48 , 120 , 0 
.db  254 , 198 , 140 , 24 , 50 , 102 , 254 , 0 
.db  120 , 96 , 96 , 96 , 96 , 96 , 120 , 0 
.db  192 , 96 , 48 , 24 , 12 , 6 , 2 , 0 
.db  120 , 24 , 24 , 24 , 24 , 24 , 120 , 0 
.db  16 , 56 , 108 , 198 , 0 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 255 
.db  48 , 48 , 24 , 0 , 0 , 0 , 0 , 0 
.db  0 , 0 , 120 , 12 , 124 , 204 , 118 , 0 
.db  224 , 96 , 96 , 124 , 102 , 102 , 220 , 0 
.db  0 , 0 , 120 , 204 , 192 , 204 , 120 , 0 
.db  28 , 12 , 12 , 124 , 204 , 204 , 118 , 0 
.db  0 , 0 , 120 , 204 , 252 , 192 , 120 , 0 
.db  56 , 108 , 96 , 240 , 96 , 96 , 240 , 0 
.db  0 , 0 , 118 , 204 , 204 , 124 , 12 , 248 
.db  224 , 96 , 108 , 118 , 102 , 102 , 230 , 0 
.db  48 , 0 , 112 , 48 , 48 , 48 , 120 , 0 
.db  12 , 0 , 12 , 12 , 12 , 204 , 204 , 120 
.db  224 , 96 , 102 , 108 , 120 , 108 , 230 , 0 
.db  112 , 48 , 48 , 48 , 48 , 48 , 120 , 0 
.db  0 , 0 , 204 , 254 , 254 , 214 , 198 , 0 
.db  0 , 0 , 248 , 204 , 204 , 204 , 204 , 0 
.db  0 , 0 , 120 , 204 , 204 , 204 , 120 , 0 
.db  0 , 0 , 220 , 102 , 102 , 124 , 96 , 240 
.db  0 , 0 , 118 , 204 , 204 , 124 , 12 , 30 
.db  0 , 0 , 220 , 118 , 102 , 96 , 240 , 0 
.db  0 , 0 , 124 , 192 , 120 , 12 , 248 , 0 
.db  16 , 48 , 124 , 48 , 48 , 52 , 24 , 0 
.db  0 , 0 , 204 , 204 , 204 , 204 , 118 , 0 
.db  0 , 0 , 204 , 204 , 204 , 120 , 48 , 0 
.db  0 , 0 , 198 , 214 , 254 , 254 , 108 , 0 
.db  0 , 0 , 198 , 108 , 56 , 108 , 198 , 0 
.db  0 , 0 , 204 , 204 , 204 , 124 , 12 , 248 
.db  0 , 0 , 252 , 152 , 48 , 100 , 252 , 0 
.db  28 , 48 , 48 , 224 , 48 , 48 , 28 , 0 
.db  24 , 24 , 24 , 0 , 24 , 24 , 24 , 0 
.db  224 , 48 , 48 , 28 , 48 , 48 , 224 , 0 
.db  118 , 220 , 0 , 0 , 0 , 0 , 0 , 0 
.db  0 , 16 , 56 , 108 , 198 , 198 , 254 , 0 
.db  120 , 204 , 192 , 204 , 120 , 24 , 12 , 120 
.db  0 , 204 , 0 , 204 , 204 , 204 , 126 , 0 
.db  28 , 0 , 120 , 204 , 252 , 192 , 120 , 0 
.db  126 , 195 , 60 , 6 , 62 , 102 , 63 , 0 
.db  204 , 0 , 120 , 12 , 124 , 204 , 126 , 0 
.db  224 , 0 , 120 , 12 , 124 , 204 , 126 , 0 
.db  48 , 48 , 120 , 12 , 124 , 204 , 126 , 0 
.db  0 , 0 , 120 , 192 , 192 , 120 , 12 , 56 
.db  126 , 195 , 60 , 102 , 126 , 96 , 60 , 0 
.db  204 , 0 , 120 , 204 , 252 , 192 , 120 , 0 
.db  224 , 0 , 120 , 204 , 252 , 192 , 120 , 0 
.db  204 , 0 , 112 , 48 , 48 , 48 , 120 , 0 
.db  124 , 198 , 56 , 24 , 24 , 24 , 60 , 0 
.db  224 , 0 , 112 , 48 , 48 , 48 , 120 , 0 
.db  198 , 56 , 108 , 198 , 254 , 198 , 198 , 0 
.db  48 , 48 , 0 , 120 , 204 , 252 , 204 , 0 
.db  28 , 0 , 252 , 96 , 120 , 96 , 252 , 0 
.db  0 , 0 , 127 , 12 , 127 , 204 , 127 , 0 
.db  62 , 108 , 204 , 254 , 204 , 204 , 206 , 0 
.db  120 , 204 , 0 , 120 , 204 , 204 , 120 , 0 
.db  0 , 204 , 0 , 120 , 204 , 204 , 120 , 0 
.db  0 , 224 , 0 , 120 , 204 , 204 , 120 , 0 
.db  120 , 204 , 0 , 204 , 204 , 204 , 126 , 0 
.db  0 , 224 , 0 , 204 , 204 , 204 , 126 , 0 
.db  0 , 204 , 0 , 204 , 204 , 124 , 12 , 248 
.db  195 , 24 , 60 , 102 , 102 , 60 , 24 , 0 
.db  204 , 0 , 204 , 204 , 204 , 204 , 120 , 0 
.db  24 , 24 , 126 , 192 , 192 , 126 , 24 , 24 
.db  56 , 108 , 100 , 240 , 96 , 230 , 252 , 0 
.db  204 , 204 , 120 , 252 , 48 , 252 , 48 , 48 
.db  248 , 204 , 204 , 250 , 198 , 207 , 198 , 199 
.db  14 , 27 , 24 , 60 , 24 , 24 , 216 , 112 
.db  28 , 0 , 120 , 12 , 124 , 204 , 126 , 0 
.db  56 , 0 , 112 , 48 , 48 , 48 , 120 , 0 
.db  0 , 28 , 0 , 120 , 204 , 204 , 120 , 0 
.db  0 , 28 , 0 , 204 , 204 , 204 , 126 , 0 
.db  0 , 248 , 0 , 248 , 204 , 204 , 204 , 0 
.db  252 , 0 , 204 , 236 , 252 , 220 , 204 , 0 
.db  60 , 108 , 108 , 62 , 0 , 126 , 0 , 0 
.db  56 , 108 , 108 , 56 , 0 , 124 , 0 , 0 
.db  48 , 0 , 48 , 96 , 192 , 204 , 120 , 0 
.db  0 , 0 , 0 , 252 , 192 , 192 , 0 , 0 
.db  0 , 0 , 0 , 252 , 12 , 12 , 0 , 0 
.db  195 , 198 , 204 , 222 , 51 , 102 , 204 , 15 
.db  195 , 198 , 204 , 219 , 55 , 111 , 207 , 3 
.db  24 , 24 , 0 , 24 , 24 , 24 , 24 , 0 
.db  0 , 51 , 102 , 204 , 102 , 51 , 0 , 0 
.db  0 , 204 , 102 , 51 , 102 , 204 , 0 , 0 
.db  34 , 136 , 34 , 136 , 34 , 136 , 34 , 136 
.db  85 , 170 , 85 , 170 , 85 , 170 , 85 , 170 
.db  219 , 119 , 219 , 238 , 219 , 119 , 219 , 238 
.db  24 , 24 , 24 , 24 , 24 , 24 , 24 , 24 
.db  24 , 24 , 24 , 24 , 248 , 24 , 24 , 24 
.db  24 , 24 , 248 , 24 , 248 , 24 , 24 , 24 
.db  54 , 54 , 54 , 54 , 246 , 54 , 54 , 54 
.db  0 , 0 , 0 , 0 , 254 , 54 , 54 , 54 
.db  0 , 0 , 248 , 24 , 248 , 24 , 24 , 24 
.db  54 , 54 , 246 , 6 , 246 , 54 , 54 , 54 
.db  54 , 54 , 54 , 54 , 54 , 54 , 54 , 54 
.db  0 , 0 , 254 , 6 , 246 , 54 , 54 , 54 
.db  54 , 54 , 246 , 6 , 254 , 0 , 0 , 0 
.db  54 , 54 , 54 , 54 , 254 , 0 , 0 , 0 
.db  24 , 24 , 248 , 24 , 248 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 248 , 24 , 24 , 24 
.db  24 , 24 , 24 , 24 , 31 , 0 , 0 , 0 
.db  24 , 24 , 24 , 24 , 255 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 255 , 24 , 24 , 24 
.db  24 , 24 , 24 , 24 , 31 , 24 , 24 , 24 
.db  0 , 0 , 0 , 0 , 255 , 0 , 0 , 0 
.db  24 , 24 , 24 , 24 , 255 , 24 , 24 , 24 
.db  24 , 24 , 31 , 24 , 31 , 24 , 24 , 24 
.db  54 , 54 , 54 , 54 , 55 , 54 , 54 , 54 
.db  54 , 54 , 55 , 48 , 63 , 0 , 0 , 0 
.db  0 , 0 , 63 , 48 , 55 , 54 , 54 , 54 
.db  54 , 54 , 247 , 0 , 255 , 0 , 0 , 0 
.db  0 , 0 , 255 , 0 , 247 , 54 , 54 , 54 
.db  54 , 54 , 55 , 48 , 55 , 54 , 54 , 54 
.db  0 , 0 , 255 , 0 , 255 , 0 , 0 , 0 
.db  54 , 54 , 247 , 0 , 247 , 54 , 54 , 54 
.db  24 , 24 , 255 , 0 , 255 , 0 , 0 , 0 
.db  54 , 54 , 54 , 54 , 255 , 0 , 0 , 0 
.db  0 , 0 , 255 , 0 , 255 , 24 , 24 , 24 
.db  0 , 0 , 0 , 0 , 255 , 54 , 54 , 54 
.db  54 , 54 , 54 , 54 , 63 , 0 , 0 , 0 
.db  24 , 24 , 31 , 24 , 31 , 0 , 0 , 0 
.db  0 , 0 , 31 , 24 , 31 , 24 , 24 , 24 
.db  0 , 0 , 0 , 0 , 63 , 54 , 54 , 54 
.db  54 , 54 , 54 , 54 , 255 , 54 , 54 , 54 
.db  24 , 24 , 255 , 24 , 255 , 24 , 24 , 24 
.db  24 , 24 , 24 , 24 , 248 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 31 , 24 , 24 , 24 
.db  255 , 255 , 255 , 255 , 255 , 255 , 255 , 255 
.db  0 , 0 , 0 , 0 , 255 , 255 , 255 , 255 
.db  240 , 240 , 240 , 240 , 240 , 240 , 240 , 240 
.db  15 , 15 , 15 , 15 , 15 , 15 , 15 , 15 
.db  255 , 255 , 255 , 255 , 0 , 0 , 0 , 0 
.db  0 , 0 , 118 , 220 , 200 , 220 , 118 , 0 
.db  0 , 120 , 204 , 248 , 204 , 248 , 192 , 192 
.db  0 , 252 , 204 , 192 , 192 , 192 , 192 , 0 
.db  0 , 254 , 108 , 108 , 108 , 108 , 108 , 0 
.db  252 , 204 , 96 , 48 , 96 , 204 , 252 , 0 
.db  0 , 0 , 126 , 216 , 216 , 216 , 112 , 0 
.db  0 , 102 , 102 , 102 , 102 , 124 , 96 , 192 
.db  0 , 118 , 220 , 24 , 24 , 24 , 24 , 0 
.db  252 , 48 , 120 , 204 , 204 , 120 , 48 , 252 
.db  56 , 108 , 198 , 254 , 198 , 108 , 56 , 0 
.db  56 , 108 , 198 , 198 , 108 , 108 , 238 , 0 
.db  28 , 48 , 24 , 124 , 204 , 204 , 120 , 0 
.db  0 , 0 , 126 , 219 , 219 , 126 , 0 , 0 
.db  6 , 12 , 126 , 219 , 219 , 126 , 96 , 192 
.db  56 , 96 , 192 , 248 , 192 , 96 , 56 , 0 
.db  120 , 204 , 204 , 204 , 204 , 204 , 204 , 0 
.db  0 , 252 , 0 , 252 , 0 , 252 , 0 , 0 
.db  48 , 48 , 252 , 48 , 48 , 0 , 252 , 0 
.db  96 , 48 , 24 , 48 , 96 , 0 , 252 , 0 
.db  24 , 48 , 96 , 48 , 24 , 0 , 252 , 0 
.db  14 , 27 , 27 , 24 , 24 , 24 , 24 , 24 
.db  24 , 24 , 24 , 24 , 24 , 216 , 216 , 112 
.db  48 , 48 , 0 , 252 , 0 , 48 , 48 , 0 
.db  0 , 118 , 220 , 0 , 118 , 220 , 0 , 0 
.db  56 , 108 , 108 , 56 , 0 , 0 , 0 , 0 
.db  0 , 0 , 0 , 24 , 24 , 0 , 0 , 0 
.db  0 , 0 , 0 , 0 , 24 , 0 , 0 , 0 
.db  15 , 12 , 12 , 12 , 236 , 108 , 60 , 28 
.db  120 , 108 , 108 , 108 , 108 , 0 , 0 , 0 
.db  112 , 24 , 48 , 96 , 120 , 0 , 0 , 0 
.db  0 , 0 , 60 , 60 , 60 , 60 , 0 , 0 
.db  0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 

BMP:
.db 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xE,0x0,0x0,0x7,0xF0,0x0,0xFF,0xF8,0x0,0x0,0x0,0xFE,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x7,0xE,0x0,0x0,0x0,0x0,0x0,0x0,0xE,0x0,0x0,0x1F,0xFC,0x0,0xFF,0xFE,0x0,0x0,0x3,0xFF,0x80,0x0,0x0,0x0,0x0,0x0,0x40,0x0,0x0,0x0,0x7,0xE,0x0,0x0,0x0,0x0,0x0,0x0,0xE,0x0,0x0,0x3F,0xFE
.db 0x0,0xFF,0xFF,0x80,0x0,0x7,0xFF,0xC0,0x0,0x0,0x0,0x0,0x1,0xC0,0x0,0x0,0x0,0x7,0xE,0x0,0x0,0x0,0x0,0x0,0x0,0xE,0x0,0x0,0x7C,0x1F,0x0,0xE0,0x7,0x80,0x0,0xF,0x83,0xE0,0x0,0x0,0x0,0x0,0x1,0xC0,0x0,0x0,0x0,0x7,0xE,0x0,0x0,0x0,0x0,0x0,0x0,0xE,0x0,0x0,0xF0,0x7,0x80,0xE0,0x3,0xC0,0x0,0x1E,0x0,0xF0,0x0,0x0,0x0,0x0,0x1,0xC0,0x0,0x0,0x0,0x7,0xE,0x0,0x0,0x0,0x0,0x0,0x0,0xE,0x0,0x1,0xE0,0x3,0x80,0xE0,0x1,0xE0,0x0,0x3C
.db 0x0,0x70,0x0,0x0,0x0,0x0,0x1,0xC0,0x0,0x0,0x0,0x7,0xE,0x0,0x0,0x0,0x0,0x0,0x0,0xE,0x0,0x1,0xC0,0x3,0xC0,0xE0,0x0,0xE0,0x0,0x38,0x0,0x78,0x0,0x0,0x0,0x0,0x1,0xC0,0x0,0x0,0x0,0x7,0xE,0x0,0x0,0x0,0x0,0x0,0x0,0xE,0x0,0x1,0xC0,0x1,0x80,0xE0,0x0,0xE0,0x0,0x38,0x0,0x30,0x3,0xF0,0xE,0x3F,0x7,0xF8,0xE7,0x81,0xF8,0x7,0xE,0x1,0xF8,0x7,0x3C,0x0,0x0,0xE,0x0,0x3,0xC0,0x0,0x0,0xE0,0x0,0x70,0x0,0x78,0x0,0x0,0xF,0xFC,0xE,0xFF
.db 0x87,0xF8,0xEF,0x87,0xFE,0x7,0xE,0x7,0xFE,0x7,0x7C,0x0,0x0,0xE,0x0,0x3,0x80,0x0,0x0,0xE0,0x0,0x70,0x0,0x70,0x0,0x0,0x1F,0xFE,0xF,0xFF,0xC7,0xF8,0xFF,0xF,0xFF,0x7,0xE,0xF,0xFF,0x7,0xF8,0x0,0x0,0xE,0x0,0x3,0x80,0x0,0x0,0xE0,0x0,0x70,0x0,0x70,0x0,0x0,0x3E,0x1F,0xF,0xC3,0xE1,0xC0,0xF8,0x1F,0xF,0x87,0xE,0x1F,0x7,0x87,0xC0,0x0,0x0,0xE,0x0,0x3,0x80,0x0,0x0,0xE0,0x0,0x70,0x0,0x70,0x0,0x0,0x38,0x7,0xF,0x1,0xE1,0xC0,0xF0,0x1C,0x3,0x87
.db 0xE,0x1C,0x3,0x87,0x80,0x0,0x0,0xE,0x0,0x3,0x80,0x0,0x0,0xE0,0x0,0x70,0x0,0x70,0x0,0x0,0x78,0x7,0x8F,0x0,0xE1,0xC0,0xE0,0x3C,0x3,0xC7,0xE,0x38,0x1,0xC7,0x0,0x0,0x0,0xE,0x0,0x3,0x80,0x0,0x0,0xE0,0x0,0x70,0x0,0x70,0x0,0x0,0x70,0x3,0x8E,0x0,0xE1,0xC0,0xE0,0x38,0x1,0xC7,0xE,0x38,0x1,0xC7,0x0,0x0,0x0,0xE,0x0,0x3,0x80,0x0,0x0,0xE0,0x0,0x70,0x0,0x70,0x0,0x0,0x70,0x3,0x8E,0x0,0xE1,0xC0,0xE0,0x38,0x1,0xC7,0xE,0x3F,0xFF,0xC7,0x0,0x0
.db 0x0,0xE,0x0,0x3,0x80,0x0,0x0,0xE0,0x0,0x70,0x0,0x70,0x0,0x0,0x70,0x3,0x8E,0x0,0xE1,0xC0,0xE0,0x38,0x1,0xC7,0xE,0x3F,0xFF,0xC7,0x0,0x0,0x0,0xE,0x0,0x1,0xC0,0x0,0xC0,0xE0,0x0,0x70,0x0,0x38,0x0,0x18,0x70,0x3,0x8E,0x0,0xE1,0xC0,0xE0,0x38,0x1,0xC7,0xE,0x3F,0xFF,0xC7,0x0,0x0,0x0,0xE,0x0,0x1,0xC0,0x0,0xE0,0xE0,0x0,0xE0,0x0,0x38,0x0,0x1C,0x70,0x3,0x8E,0x0,0xE1,0xC0,0xE0,0x38,0x1,0xC7,0xE,0x38,0x0,0x7,0x0,0x0,0x0,0xE,0x0,0x1,0xC0,0x1
.db 0xC0,0xE0,0x0,0xE0,0x0,0x38,0x0,0x38,0x70,0x3,0x8E,0x0,0xE1,0xC0,0xE0,0x38,0x1,0xC7,0xE,0x38,0x0,0x7,0x0,0x0,0x0,0xE,0x0,0x0,0xE0,0x3,0xC0,0xE0,0x1,0xE0,0x0,0x1C,0x0,0x78,0x78,0x7,0x8E,0x0,0xE1,0xC0,0xE0,0x3C,0x3,0xC7,0xE,0x3C,0x0,0x7,0x0,0x0,0x0,0xE,0x0,0x0,0xF0,0x7,0xC0,0xE0,0x3,0xC0,0x0,0x1E,0x0,0xF8,0x38,0x7,0xE,0x0,0xE1,0xC0,0xE0,0x1C,0x3,0x87,0xE,0x1C,0x1,0xC7,0x0,0x0,0x0,0xE,0x0,0x0,0x7C,0xF,0x80,0xE0,0xF,0x80,0x0,0xF
.db 0x81,0xF0,0x3E,0x1F,0xE,0x0,0xE1,0xC0,0xE0,0x1F,0xF,0x87,0xE,0x1F,0x7,0x87,0x0,0x0,0x0,0xF,0xFF,0xE0,0x3F,0xFF,0x0,0xFF,0xFF,0x80,0x0,0x7,0xFF,0xE0,0x1F,0xFE,0xE,0x0,0xE1,0xF8,0xE0,0xF,0xFF,0x7,0xE,0xF,0xFF,0x87,0x0,0x0,0x0,0xF,0xFF,0xE0,0x1F,0xFE,0x0,0xFF,0xFE,0x0,0x0,0x3,0xFF,0xC0,0xF,0xFC,0xE,0x0,0xE0,0xF8,0xE0,0x7,0xFE,0x7,0xE,0x7,0xFF,0x7,0x0,0x0,0x0,0xF,0xFF,0xE0,0x7,0xF8,0x0,0xFF,0xF8,0x0,0x0,0x0,0xFF,0x0,0x3,0xF0,0xE,0x0
.db 0xE0,0x78,0xE0,0x1,0xF8,0x7,0xE,0x1,0xFC,0x7,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0
.db 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0
.db 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xFC,0x2,0x0,0x0,0x3F,0xC0,0x0,0x0,0x0,0x4,0x88,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x1,0x2,0x2,0x0,0x0,0x20,0x20,0x0,0x0,0x0,0x4,0x8,0x8,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x2,0x79,0x2,0x0,0x0,0x20,0x20,0x0,0x0,0x0,0x4,0x8,0x8,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0
.db 0x0,0x4,0x84,0x82,0xE2,0x8,0x20,0x21,0xC2,0xE0,0xE0,0x74,0x88,0x3E,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x4,0x80,0x83,0x12,0x8,0x20,0x42,0x23,0x11,0x10,0x8C,0x88,0x48,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x4,0x80,0x82,0xA,0x10,0x3F,0x84,0x12,0x12,0x9,0x4,0x88,0x88,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x4,0x80,0x82,0x9,0x10
.db 0x20,0x44,0x12,0x12,0x9,0x4,0x89,0x8,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x4,0x84,0x82,0x9,0x10,0x20,0x27,0xF2,0x13,0xF9,0x4,0x8B,0x8,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x4,0x78,0x82,0x8,0xA0,0x20,0x24,0x2,0x12,0x1,0x4,0x8C,0x88,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x2,0x1,0x2,0x8,0xA0,0x20,0x24,0x12,0x12,0x9,0x4
.db 0x88,0x88,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x1,0x2,0x3,0x10,0xA0,0x20,0x42,0x22,0x11,0x10,0x8C,0x88,0x48,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xFC,0x2,0xE0,0x40,0x3F,0x81,0xC2,0x10,0xE0,0x74,0x88,0x2E,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x40,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0
.db 0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x40,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x1,0x80,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0

