#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#define BAUD 115200                  // define Baud Rate
#define USART0_RX_BUFFER_SIZE 32     // define Receive Buffer Size. Size can be 2, 4, 8, 16, 32, 64, 128 or 256 Byte
#define USART0_TX_BUFFER_SIZE 64     // define Transmit Buffer Size. Size can be 2, 4, 8, 16, 32, 64, 128 or 256 Byte

#define F_CPU 20000000
#include "usart.h"
#include <util/delay.h>

const char cmd[] PROGMEM = "sh /flash2/motorola.sh\r\n";

/* Static Variables */
static char USART0_RxBuf[USART0_RX_BUFFER_SIZE];
static volatile char USART0_RxHead;
static volatile char USART0_RxTail;
static char USART0_TxBuf[USART0_TX_BUFFER_SIZE];
static volatile char USART0_TxHead;
static volatile char USART0_TxTail;

/* Initialize USART */
void usart0_init( void )
{
    UBRR0L = UBRRL_VALUE;
    UBRR0H = UBRRH_VALUE;
    UCSR0B = ( ( 1 << RXCIE0 ) | ( 1 << RXEN0 ) | ( 1 << TXEN0 ) );
    #ifdef URSEL
        UCSRC = (1<<URSEL)|(1<<UCSZ1)|(1<<UCSZ0);
    #else
        UCSR0C = (1<<UCSZ01)|(1<<UCSZ00);
    #endif
#if USE_2X
    UCSR0A |= (1 << U2X0);
#endif /*USE_2X*/
    USART0_RxTail = 0;
    USART0_RxHead = 0;
    USART0_TxTail = 0;
    USART0_TxHead = 0;
}


ISR(UART0_RECV_IRQ)
{
    char data;
    unsigned char tmp;
    data = UDR0;
    tmp = ( USART0_RxHead + 1 ) & USART0_RX_BUFFER_MASK;
    USART0_RxHead = tmp;
    USART0_RxBuf[tmp] = data;
}

ISR(UART0_DATA_IRQ)
{
    unsigned char tmp;

    if ( USART0_TxHead != USART0_TxTail )
    {
        tmp = ( USART0_TxTail + 1 ) & USART0_TX_BUFFER_MASK;
        USART0_TxTail = tmp;
        UDR0 = USART0_TxBuf[tmp];
    }
    else
    {
        UCSR0B &= ~(1<<UDRIE0);
    }
}

char usart0_getc( void )
{
    unsigned char tmp;

    while ( USART0_RxHead == USART0_RxTail );
    tmp = ( USART0_RxTail + 1 ) & USART0_RX_BUFFER_MASK;
    USART0_RxTail = tmp;
    return USART0_RxBuf[tmp];
}

void usart0_putc( char data )
{
    unsigned char tmp;
    tmp = ( USART0_TxHead + 1 ) & USART0_TX_BUFFER_MASK;
    while ( tmp == USART0_TxTail );
    USART0_TxBuf[tmp] = data;
    USART0_TxHead = tmp;
    UCSR0B |= (1<<UDRIE0);
}

char usart0_data_received( void )
{
    return ( USART0_RxHead != USART0_RxTail );
}

void wait_for_sh()
{
	while(1)
	{
		if(usart0_getc()==' ')
			if(usart0_getc()=='#')
				if(usart0_getc()==' ')
					return;
	}
}

int main(void)
{
	int i;
	DDRD |= 0b00001100;
	PORTD |= 0b00001000;
	usart0_init();
	sei();
	for (;;)
	{
		wait_for_sh();
		for(i=0;i<100;i++) _delay_ms(20);
		for(i=0;i<sizeof(cmd);i++)
			usart0_putc(pgm_read_byte(&cmd[i]));
		PORTD &= 0b11110111;
		PORTD |= 0b00000100;
		while(1) usart0_getc();
	}
}

