/***********************************************
***
***  eeeeee eeeee eeeee eeeee eeeeee eeeeee
***  8    8 8     8     8     8   88 8    8
***  8e   8 8     8     8     8    8 8    8
***  88   8 8eee  8eee  8e    8    8 8e   8
***  88   8 8e    8e    88    8    8 88   8
***  88   e 88    88    88    8    e 88   8
***  88eee8 88eee 88    88eee 88eee8 88   8
***
***        Coded by Crashdemon....
***         HAVE A LOT OF PHUN!
***
***			 Motorcycle IMU 	    
***     inertial measurement unit
***         ver 0.1 (STM32)
***			 ARM Cortex-M3
***        
***********************************************/

#include <math.h>
#include <stdio.h>
#include <stdint.h>
#include <stddef.h>

#include <stm32f10x.h>

// Constants
#define M_PI 3.14159265358979323846

// USART
#define BAUD 115200 // Baudrate fr serielle Kommunikation
#define BLIND_CHAR "," 

// LCD
#define	LCD_DATA_PINS GPIO_Pin_0 | GPIO_Pin_1 | GPIO_Pin_2 | GPIO_Pin_3
#define LCD_RS_PIN GPIO_Pin_4
#define LCD_EN_PIN GPIO_Pin_5
#define CLEAR_DISPLAY 0x01
#define CURSOR_HOME 0x02 

// Pre-calc.
#define RAD_TO_DEG (180.0 / M_PI)
#define DEG_TO_RAD (M_PI / 180.0)

// Our covariance matrix.  This is updated at every time step to
// determine how well the sensors are tracking the actual state.
static double P[2][2] = 
{

	{1, 0},
	{0, 1}
};

// Our two states, the angle and the gyro bias.  As a byproduct of computing
// the angle, we also have an unbiased angular rate available.   These are
// read-only to the user of the module.
float fDt; 
float fAngleEstimate; 		// deg
float fGyroRateBias;		// deg per second	
float fGyroRateEstimate;	// deg per second

// R represents the measurement covariance noise.  In this case,
// it is a 1x1 matrix that says that we expect 0.3 rad jitter
// from the accelerometer.
static const double R_angle = 0.3;

// Q is a 2x2 matrix that represents the process covariance noise.
// In this case, it indicates how much we trust the acceleromter
// relative to the gyros.
static const double Q_angle	= 0.001;
static const double Q_gyro	= 0.003;

volatile unsigned int systicks = 0;

// Prototypendeklaration
// RCC
void RCC_Configuration(void);

// NVIC
void NVIC_Configuration(void);

// RTC
void RTCSetup(void);

// ADC
void ADC_Configuration(void);
uint16_t ADC_Read(uint8_t);
uint16_t ADC_Read_Avg(uint8_t, uint8_t);

// USART
void USART_Configuration(void);
void USART_Send(char*);

// LCD
void LCD_init(void); 
void LCD_home(void);
void LCD_clear(void);
void LCD_enable(void);
void LCD_string(char*);
void LCD_data(unsigned char);
void LCD_command(unsigned char);
void LCD_set_cursor(uint8_t, uint8_t);

// KALMAN
void state_update(const float);
void kalman_update(float);

// SYSTICK
void SysTick_init(void);
void SysTick_Handler(void);

// DELAY
void delay_us(unsigned int);
void delay_ms(unsigned int);

/********************************** MAIN ******************************************/
int main(void)
{
	// Messwerte der ADC Kanle C0-C5 (0-1024) 12 Bit Auflsung
	// 0-2: Messwerte des Beschleunigungssensors (X, Y, Z)-Achse
	// 3: Messwert des Gyros Z-Achse
	// 4: Messwert des Gyros Z-Achse, vierfach verstrkt
	// 5: Referenzspannung des Gyros
	uint16_t meas[6];

	// Die Winkel zwischen den verschiedenen Achsen,
	// des Beschleunigungssensor ermitteln.
	// angle[0] = YX
	// angle[1] = ZY
	// angle[2] = ZX
	float angle[3];

	RCC_Configuration(); // System Clocks Configuration
	NVIC_Configuration(); // NVIC Configuration
	RTCSetup();	// RTC Setup
	ADC_Configuration(); // ADC initialisieren
	USART_Configuration(); // USART initialisieren
	LCD_init(); // Initialisiert das LCD

	while(1)
	{
		// Messwerte vom Beschleunigungssensor holen
		meas[0] = ADC_Read(ADC_Channel_0); // 12Bit - Messwert der x-Achse
		meas[1] = ADC_Read(ADC_Channel_1); // 12Bit - Messwert der y-Achse
		meas[2] = ADC_Read(ADC_Channel_2); // 12Bit - Messwert der z-Achse	

		// Gyro 
		meas[3] = ADC_Read(ADC_Channel_3); // 10Bit - Messwert der z-Achse 
		meas[4] = ADC_Read(ADC_Channel_4); // 10Bit - Messwert der z-Achse, 4-fach Verstrkt
		meas[5] = ADC_Read(ADC_Channel_5); // Referenzspannung des Gyros   

		// Winkel berechnen, und in Grad umrechnen
		angle[0] = atan2(meas[1], meas[0]) * RAD_TO_DEG;
		angle[1] = atan2(meas[2], meas[1]) * RAD_TO_DEG;
		angle[2] = atan2(meas[2], meas[0]) * RAD_TO_DEG;
		
		USART_Send("Hallo"); 
		LCD_string("Hallo"); 
		USART_Send("\n");
	}
}

/********************************** RCC ******************************************/
/******** RCC initialisieren ********/
void RCC_Configuration(void)
{
	ErrorStatus HSEStartUpStatus;

  	RCC_DeInit(); // RCC system reset(for debug purpose)
  	RCC_HSEConfig(RCC_HSE_ON); // Enable HSE

  	HSEStartUpStatus = RCC_WaitForHSEStartUp(); // Wait till HSE is ready

  	if(HSEStartUpStatus == SUCCESS)
  	{
    	FLASH_PrefetchBufferCmd(FLASH_PrefetchBuffer_Enable); // Enable Prefetch Buffer
    	FLASH_SetLatency(FLASH_Latency_2); // Flash 2 wait state 

    	RCC_HCLKConfig(RCC_SYSCLK_Div1); // HCLK = SYSCLK 
    	RCC_PCLK2Config(RCC_HCLK_Div1); // PCLK2 = HCLK
    	RCC_PCLK1Config(RCC_HCLK_Div2); // PCLK1 = HCLK/2

    	RCC_PLLConfig(RCC_PLLSource_HSE_Div1, RCC_PLLMul_9); // PLLCLK = 8MHz * 9 = 72 MHz
    	RCC_PLLCmd(ENABLE); // Enable PLL
    	while(RCC_GetFlagStatus(RCC_FLAG_PLLRDY) == RESET); // Wait till PLL is ready

    	RCC_SYSCLKConfig(RCC_SYSCLKSource_PLLCLK); // Select PLL as system clock source

    	while(RCC_GetSYSCLKSource() != 0x08); // Wait till PLL is used as system clock source
  	}
}

/********************************** NVIC ******************************************/
/******** NVIC initialisieren ********/
void NVIC_Configuration(void)
{
	NVIC_SetVectorTable(NVIC_VectTab_RAM, 0x0); // Set the Vector Table base location at 0x20000000
}

/********************************** RTC ******************************************/
/******** RTC initialisieren ********/
void RTCSetup(void)
{
  	RCC_APB1PeriphClockCmd(RCC_APB1Periph_PWR | RCC_APB1Periph_BKP, ENABLE); // Enable PWR and BKP clocks
  	PWR_BackupAccessCmd(ENABLE); // Allow access to BKP Domain
  	BKP_DeInit(); // Reset Backup Domain

  	RCC_LSEConfig(RCC_LSE_ON); // Enable LSE
  	while(RCC_GetFlagStatus(RCC_FLAG_LSERDY) == RESET); // Wait till LSE is ready

  	RCC_RTCCLKConfig(RCC_RTCCLKSource_LSE); // Select LSE as RTC Clock Source
 	RCC_RTCCLKCmd(ENABLE); // Enable RTC Clock
  	RTC_WaitForSynchro(); // Wait for RTC registers synchronization
  	RTC_WaitForLastTask(); // Wait until last write operation on RTC registers has finished
  	RTC_ITConfig(RTC_IT_SEC, ENABLE); // Enable the RTC Second
  	RTC_WaitForLastTask(); // Wait until last write operation on RTC registers has finished

  	// Set RTC prescaler: set RTC period to 1sec 
  	RTC_SetPrescaler(32767); // RTC period = RTCCLK/RTC_PR = (32.768 KHz)/(32767+1) 
	RTC_WaitForLastTask(); // Wait until last write operation on RTC registers has finished
}

/********************************** ADC ******************************************/
/******** ADC initialisieren ********/
void ADC_Configuration(void)
{
	ADC_InitTypeDef ADC_InitStructure;
  	
	// PCLK2 is the APB2 clock 
  	// ADCCLK = PCLK2/6 = 72/6 = 12MHz
  	RCC_ADCCLKConfig(RCC_PCLK2_Div6);

  	RCC_APB2PeriphClockCmd(RCC_APB2Periph_ADC1, ENABLE); // Enable ADC1 clock so that we can talk to it
  	ADC_DeInit(ADC1); // Put everything back to power-on defaults

   	// ------------------------------ ADC1 Configuration ------------------------------
  	ADC_InitStructure.ADC_Mode = ADC_Mode_Independent; // ADC1 and ADC2 operate independently 
  	ADC_InitStructure.ADC_ScanConvMode = DISABLE; // Disable the scan conversion so we do one at a time
  	ADC_InitStructure.ADC_ContinuousConvMode = DISABLE; // Don't do contimuous conversions - do them on demand
  	ADC_InitStructure.ADC_ExternalTrigConv = ADC_ExternalTrigConv_None; // Start conversion by software, not an external trigger
  	ADC_InitStructure.ADC_DataAlign = ADC_DataAlign_Right; // Conversions are 12 bit - put them in the lower 12 bits of the result
  	ADC_InitStructure.ADC_NbrOfChannel = 1; // Say how many channels would be used by the sequencer
  	ADC_Init(ADC1, &ADC_InitStructure); // Now do the setup
  	ADC_Cmd(ADC1, ENABLE); // Enable ADC1

  	ADC_ResetCalibration(ADC1); // Enable ADC1 reset calibaration register 
  	while(ADC_GetResetCalibrationStatus(ADC1)); // Check the end of ADC1 reset calibration register

  	ADC_StartCalibration(ADC1); // Start ADC1 calibaration
  	while(ADC_GetCalibrationStatus(ADC1)); // Check the end of ADC1 calibration 
}

/******** Holt den Messwert vom AD-Channel ********/
uint16_t ADC_Read(uint8_t channel)
{
	ADC_RegularChannelConfig(ADC1, channel, 1, ADC_SampleTime_1Cycles5);
  	ADC_SoftwareStartConvCmd(ADC1, ENABLE); // Start the conversion

  	while(ADC_GetFlagStatus(ADC1, ADC_FLAG_EOC) == RESET); // Wait until conversion completion
  
  	return ADC_GetConversionValue(ADC1); // Get the conversion value
}

/******** Holt Wert vom bergebenen AD-Channel und bildet den Mittelwert ********/
uint16_t ADC_Read_Avg(uint8_t channel, uint8_t average)
{
	uint8_t index = 0;
	uint32_t result = 0;	

   	ADC_RegularChannelConfig(ADC1, channel, 1, ADC_SampleTime_1Cycles5);

	// Eigentliche Messung - Mittelwert aus 'average' aufeinanderfolgenden Wandlungen 
  	for(index = average; index > 0; index--) 
	{
  		ADC_SoftwareStartConvCmd(ADC1, ENABLE); // Start the conversion

  		while(ADC_GetFlagStatus(ADC1, ADC_FLAG_EOC) == RESET); // Wait until conversion completion
  
  		// Get the conversion value
  		result += ADC_GetConversionValue(ADC1);	// Wandlungsergebnisse aufaddieren
	}

	result /= average; // arithmethischer Mittelwert

	return result;
}

/********************************** USART ******************************************/
/******** USART initialisieren ********/
void USART_Configuration(void)
{
	USART_InitTypeDef USART_InitStructure;
	GPIO_InitTypeDef GPIO_InitStructure; 

	RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOB, ENABLE); // Enable GPIOB peripheral clock
	RCC_APB2PeriphClockCmd(RCC_APB2Periph_AFIO, ENABLE); 
	RCC_APB1PeriphClockCmd(RCC_APB1Periph_USART3, ENABLE); // Enable USART3 Clock
	
	GPIO_StructInit(&GPIO_InitStructure); // Set GPIO to default values
	GPIO_InitStructure.GPIO_Pin = GPIO_Pin_10; // Set TX(PB10)
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP; // Set as alternate function Push-Pull
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz; // Set freq. of PortPin
	GPIO_Init(GPIOB, &GPIO_InitStructure); 	// Initialize 

	GPIO_InitStructure.GPIO_Pin = GPIO_Pin_11; // Set RX(PB11)
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING ; // Set as floating input
	GPIO_Init(GPIOB, &GPIO_InitStructure); // Initialize
	
	USART_StructInit(&USART_InitStructure); // Initializes structure with default values
  	USART_DeInit(USART3); // Put everything back to power-on defaults

	// ------------------------------ USART3 Configuration ------------------------------
	USART_InitStructure.USART_BaudRate = BAUD; // The Baudrate
	USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None; // Software or Hardware FlowControl
	USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx; // Enable RX and TX
	USART_InitStructure.USART_Parity = USART_Parity_No; // No Parity Bit
	USART_InitStructure.USART_StopBits = USART_StopBits_1; // One Stopbit
	USART_InitStructure.USART_WordLength = USART_WordLength_8b; // 8-Bit Word length of Data
	USART_Init(USART3, &USART_InitStructure); // Now do the setup

	USART_Cmd(USART3, ENABLE); // Enable USART3
}

/******** Daten ber den USART senden ********/
void USART_Send(char *data) 
{
	while(*data) // Write as long as no String-EndChar.
	{
		USART_SendData(USART3, *data); // Send one byte out
		while(USART_GetFlagStatus(USART3, USART_FLAG_TXE) == RESET); // Loop until USART3 DR register is empty 
	}
}

/********************************** LCD ******************************************/
/******** LCD initialisieren ********/
void LCD_init(void)
{	
	//LCD_DDR = LCD_DDR | 0x0F | (1 << LCD_RS) | (1 << LCD_EN); // Port auf Ausgang schalten
	GPIO_InitTypeDef GPIO_InitStructure;

	RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOC, ENABLE); // Enable GPIOC peripheral clock

	GPIO_StructInit(&GPIO_InitStructure); // Set GPIO to default values
	GPIO_InitStructure.GPIO_Pin = LCD_DATA_PINS | LCD_EN_PIN | LCD_RS_PIN; // Set all Output pins needed by LCD
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP; // Set as output Push-Pull
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz; // Set freq. of PortPin
	GPIO_Init(GPIOC, &GPIO_InitStructure); 	// Initialize 

	delay_ms(15);
	
	//LCD_PORT &= 0xF0;        
	//LCD_PORT |= 0x03;      
	GPIO_WriteBit(GPIOC, ((LCD_DATA_PINS) & 0x03), Bit_SET);      

	//LCD_PORT &= ~(1 << LCD_RS); // Register Select auf "Transferring Instruction Data" setzen
	GPIO_WriteBit(GPIOC, LCD_RS_PIN, Bit_RESET);

	LCD_enable();
	delay_ms(5);
   	LCD_enable();
 
   	delay_ms(1);
   	LCD_enable();
   	delay_ms(1);

	// 4 Bit Modus aktivieren 
	//LCD_PORT &= 0xF0;
	//LCD_PORT |= 0x02;
	GPIO_WriteBit(GPIOC, ((LCD_DATA_PINS) & 0x02), Bit_SET);

	LCD_enable();
	
	delay_ms(1); 
   	LCD_command(0x28); // 4Bit / 2 Zeilen / 5x7
    LCD_command(0x0C); // Display ein / Cursor aus / kein Blinken
 	LCD_command(0x06); // inkrement / kein Scrollen
 
   	LCD_clear();
}

/******** Befehl an das LCD senden ********/
void LCD_command(unsigned char temp1)
{
	unsigned char temp2 = temp1;
 
	//LCD_PORT &= ~(1 << LCD_RS); // Register Select auf "Transfering Instruction Data" setzen
	GPIO_WriteBit(GPIOC, LCD_RS_PIN, Bit_RESET);
 
	temp1 = temp1 >> 4; // Oberes Nibble holen
	temp1 = temp1 & 0x0F; // Maskieren

	//LCD_PORT &= 0xF0;
	//LCD_PORT |= temp1; // Daten Bits setzen
	GPIO_WriteBit(GPIOC, ((LCD_DATA_PINS) & temp1), Bit_SET);

	LCD_enable();
 
	temp2 = temp2 & 0x0F; // unteres Nibble holen und maskieren
	//LCD_PORT &= 0xF0;
	//LCD_PORT |= temp2; // Daten Bits setzen
	GPIO_WriteBit(GPIOC, ((LCD_DATA_PINS) & temp2), Bit_SET);

   	LCD_enable();
    delay_us(42);
}

/******** Datenbyte an das LCD senden ********/
void LCD_data(unsigned char temp1)
{
	unsigned char temp2 = temp1;
 
	//LCD_PORT |= (1 << LCD_RS); // Register Select auf "Transferring Display Data" setzen
	GPIO_WriteBit(GPIOC, LCD_RS_PIN, Bit_SET);
 
	temp1 = temp1 >> 4;
   	temp1 = temp1 & 0x0F;
   	
	//LCD_PORT &= 0xF0;
   	//LCD_PORT |= temp1; // Daten Bits setzen
	GPIO_WriteBit(GPIOC, ((LCD_DATA_PINS) & temp1), Bit_SET);

   	LCD_enable();

   	temp2 = temp2 & 0x0F;

   	//LCD_PORT &= 0xF0;
   	//LCD_PORT |= temp2; // Daten Bits setzen
	GPIO_WriteBit(GPIOC, ((LCD_DATA_PINS) & temp2), Bit_SET);

   	LCD_enable();
   
   	delay_us(42);
}

/******** setzt Cursor in Zeile y (1..4) Spalte x (0..15) ********/
void LCD_set_cursor(uint8_t x, uint8_t y)
{
	uint8_t tmp;
 
  	switch (y) 
	{
    	case 1: // 1. Zeile 
			tmp = 0x80 + 0x00 + x; 
		break;

    	case 2: // 2. Zeile 
			tmp = 0x80 + 0x40 + x; 
		break;

    	case 3: // 3. Zeile 
			tmp = 0x80 + 0x10 + x; 
		break;

    	case 4: // 4. Zeile 
			tmp = 0x80 + 0x50 + x; 
		break;

    	default: 
			return; // fr den Fall einer falschen Zeile
  	}

  	LCD_command(tmp);
}

/******** schreibt String auf LCD ********/
void LCD_string(char *data)
{
	while(*data) 
	{
    	LCD_data(*data);
        data++;
    }
}

/******** erzeugt den Enable-Puls ********/
void LCD_enable(void)
{
	//LCD_PORT |= (1 << LCD_EN); // LCD_ENABLED = 1 an LCD Port senden
	GPIO_WriteBit(GPIOC, LCD_EN_PIN, Bit_SET);

  	delay_us(2); // kurze Pause zum verarbeiten des Befehls

  	//LCD_PORT &= ~(1 << LCD_EN); // LCD_ENABLED = 0 an LCD Port senden
	GPIO_WriteBit(GPIOC, LCD_EN_PIN, Bit_RESET);
}

/******** LCD lschen ********/
void LCD_clear(void)
{
	LCD_command(CLEAR_DISPLAY);
   	delay_ms(5);
}

/******** LCD Cursor Home ********/
void LCD_home(void)
{
	LCD_command(CURSOR_HOME);
   	delay_ms(5);
}

/********************************** KALMAN ******************************************/
// state_update is called every fDt with a biased gyro measurement
// by the user of the module.  It updates the current angle and
// rate estimate.
// Parameter: rad per second?
void state_update(const float q)
{
	float Pdot[2][2];
	
	Pdot[0] [0] = Q_angle - P[0][1] - P[1][0];	/* 0,0 */
	Pdot[0] [1] = -P[1][1];		            	/* 0,1 */
	Pdot[1] [0] = -P[1][1];		            	/* 1,0 */
	Pdot[1] [1] = Q_gyro;                       /* 1,1 */
		        
	/* Store our unbias gyro estimate */
	fGyroRateEstimate = q;
	fAngleEstimate += q * fDt;

	P[0][0] += Pdot[0] [0] * fDt;
	P[0][1] += Pdot[0] [1] * fDt;
	P[1][0] += Pdot[1] [0] * fDt;
	P[1][1] += Pdot[1] [1] * fDt;
}

// kalman_update is called by a user of the module when a new
// accelerometer measurement is available.  angle_m not
// need to be scaled into actual units, but must be zeroed and have
// the same scale.
void kalman_update(float angle_m)
{
	float angle_err;
    float C_0 = 1;	
         
    float PCt_0;
    float PCt_1; 
        
    float E;   
        
    float K_0;
    float K_1; 
        
    float t_0;
    float t_1;  
	
	angle_err = angle_m - fAngleEstimate;

	PCt_0 = C_0 * P[0][0]; /* + C_1 * P[0][1] = 0 */
	PCt_1 = C_0 * P[1][0]; /* + C_1 * P[1][1] = 0 */
	E = R_angle + C_0 * PCt_0;
	K_0 = PCt_0 / E;
	K_1 = PCt_1 / E;
		
	t_0 = PCt_0; /* C_0 * P[0][0] + C_1 * P[1][0] */
	t_1 = C_0 * P[0][1]; /* + C_1 * P[1][1]  = 0 */

	P[0][0] -= K_0 * t_0;
	P[0][1] -= K_0 * t_1;
	P[1][0] -= K_1 * t_0;
	P[1][1] -= K_1 * t_1;
	
	fAngleEstimate	+= K_0 * angle_err;
	fGyroRateBias	+= K_1 * angle_err;
}

/********************************** SYSTICK ******************************************/
/******** Initialisiert die Zeitbasis ********/
void SysTick_init(void)
{
	SysTick_CLKSourceConfig(SysTick_CLKSource_HCLK); // Main HSE as Clock-Source	

	//SysTick_Config(72000); // HSE 72Mhz / 72k = 1000 -> 1/1000 = 1ms
	SysTick_Config(72); // HSE 72Mhz / 72 = 1M -> 1/1M = 1us
}

/******** SysTick Interrupt ********/
void SysTick_fkt(void)
{
	systicks++;
}

/********************************** DELAY ******************************************/
/******** Verzgert um usec ********/
void delay_us(unsigned int usec)
{
	systicks = 0;
	
	while(systicks <= usec);
}

/******** Verzgert um msec ********/
void delay_ms(unsigned int msec)
{
	systicks = 0;
	msec *= 1000;
	
	while(systicks <= msec);
}



#ifdef  USE_FULL_ASSERT

/**
  * @brief  Reports the name of the source file and the source line number
  *   where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t* file, uint32_t line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

/**
  * @}
  */

