

; ringpuffer implementation in asm
; by Clemens Helfmeier
; ClemensHelfmeier (at) gmx (dot) de
;
; this makes only macros :)
; thus may be included only once.

#ifndef _ringpuffer_asm_
#define _ringpuffer_asm_

.equ	rp_FirstIndexOffset	= 0
.equ	rp_LengthOffset		= 1
.equ	rp_StateOffset		= 2
.equ	rp_StreamOffset		= 3

; bit numbers in rp_State
.equ	rp_bXON			= 0

; some debug definitions
#ifdef DEBUG
	#define rp_DEBUG
#endif
; #define rp_OnceOnOff

; --- MODIFY HERE ---
; these macros must be implemented by the main program
;.macro rp_Full
	; this macro is called when the ringpuffer is going to be filled up.
	; registers to be used are rArg0 all other register _must_ not be used!
	; this function should not block since it is called from the interrupt!
;.endm
;.macro rp_Empty
	; this macro is called when the ringpuffer is going to be empty again
	; registers to be used are rArg0 all other register _must_ not be used!
;.endm
; -----  don't modify the following functions ... -----


.macro rp_Init
	; clear the length and firstindex
	; the parameters are:
	; @0	Base address of memory space
	; @1	macro to be called when buffer is full
	clr rArg0
	sts @0 + rp_LengthOffset, rArg0
	sts @0 + rp_FirstIndexOffset, rArg0
	@1
.endm
	


.macro rp_popZ
	pop zh
	pop zl
.endm
.macro rp_pushZ
	push zl
	push zh
.endm
.macro rp_addi16
	subi @0, low(-(@2))
	sbci @1, high(-(@2))
.endm
	

.macro rp_Put
	; this macro puts the data into the ring-puffer
	; argument is rArg0, needs rTemp0

	; the parameters are:
	; @0	Base address of memory space
	; @1	Buffer length (in bytes) must be a power of 2!
	; @2	macro to be called when buffer is full
	
	rp_pushZ
	
	lds zl, @0 + rp_FirstIndexOffset
	lds rTemp0, @0 + rp_LengthOffset
	
	add zl, rTemp0				; firstIndex + Length
	
	andi zl, @1-1				; mask into stream space
	clr zh
	
	rp_addi16 zl, zh, @0 + rp_StreamOffset	; put to stream space
	
	st z, rArg0				; store the new byte
	
	inc rTemp0
	sts @0 + rp_LengthOffset, rTemp0	; increase length and store again

	#ifdef rp_DEBUG
		cpi rTemp0, @1
		brlo rp_Put_NotFull
			rp_BufferOverrun
	rp_Put_NotFull:
		cpi rTemp0, (@1)-10
		brlo rp_Put_OKa
			rp_BufferFilling
	rp_Put_OKa:
	#endif

	cpi rTemp0, (@1)*8/10
	brlo rp_Put_OK				; if not as full as rp_BufferMaxFill
		#ifdef rp_OnceOnOff
			lds rTemp0, @0 + rp_StateOffset
			sbrs rTemp0, rp_bXON
				rjmp rp_Put_OK		; only send one time
			andi rTemp0, ~(1<<rp_bXON)
			sts @0 + rp_StateOffset, rTemp0
		#endif
		@2				; rp_Full
rp_Put_OK:
	rp_popZ
.endm

.macro rp_Get
	; this macro gets the next data from the ringpuffer
	; return value is rArg0, needs rTemp0
	
	; the parameters are:
	; @0	Base address of memory space
	; @1	Buffer length (in bytes) must be a power of 2!
	; @2	macro to be called when buffer is empty
	
	rp_pushZ
	#ifdef SLEEP
		rjmp rp_Get_WaitForData_a
	#endif
rp_Get_WaitForData:
	#ifdef SLEEP
		in rArg0, MCUCR
		andi rArg0, ~((1<<SM2) | (1<<SM1) | (1<<SM0))
		ori rArg0, (1<<SE)
		out MCUCR, rArg0
		sleep
	#endif
rp_Get_WaitForData_a:
	lds rArg0, @0 + rp_LengthOffset
	cpi rArg0, 0x00
	breq rp_Get_WaitForData			; wait here until data available
	
	cpi rArg0, 0x03
	brne rp_Get_OK				; if buffer empty
		#ifdef rp_OnceOnOff
			lds rTemp0, @0 + rp_StateOffset
			sbrc rTemp0, rp_bXON		; and not send yet
				rjmp rp_Get_OK
			ori rTemp0, (1<<rp_bXON)
			sts @0 + rp_StateOffset, rTemp0
		#endif
		@2
rp_Get_OK:
	
	lds zl, @0 + rp_FirstIndexOffset	; get index
	clr zh					; this doesn't belong to critical since this is only modified by this function
	rp_addi16 zl, zh, @0 + rp_StreamOffset	; move to stream
	
	in rTemp0, SREG
	cli					; don't write now to the stream (critical section)
	
	lds rArg0, @0 + rp_LengthOffset		; refresh the length ( in case there was an interrupt)
	dec rArg0				; decrease length by one
	sts @0 + rp_LengthOffset, rArg0		; store length
	
	ld rArg0, z				; read and move on
	
	lds zl, @0 + rp_FirstIndexOffset	; get the index again
	inc zl					; increase it (next index next time)
	andi zl, @1 - 1				; keep in stream space
	sts @0 + rp_FirstIndexOffset, zl	; store the next address
	
	out SREG, rTemp0			; restore interrupt (end of ciritcal section)
	
	rp_popZ
.endm


#endif
