
 // SVF to mySVF2 Translator by Clemens Helfmeier
 // ClemensHelfmeier (at) gmx (dot) de
 //
 // compiles cleanly under:
 //  g++ (GCC) 4.0.1 (Debian 4.0.1-2, linux)
 //  devcpp 4.9.9.2 (windows)


#include <string>
#include <iostream>
#include <fstream>
using namespace std;

// version information
#define Version_maj	0
#define Version_min	1
#define Version_ref	1

// verbose constants
#define ERRORS		0
#define WARNINGS	1
#define LOG		2
int verbose = WARNINGS;

// action constants
#define showHelp	0
#define convert		1
int action = showHelp;


// strings ...
#define sRUNTEST	"RUNTEST"
#define sSDR		"SDR"
#define sSIR		"SIR"
#define sHDR		"HDR"
#define sTDR		"TDR"
#define sHIR		"HIR"
#define sTIR		"TIR"
#define sSTATE		"STATE"

#define sIDLE		"IDLE"
#define sRESET		"RESET"

// mysvf2 values ...
#define vRUNTEST	0x07
#define vSIR		0x08
#define vSDR		0x09
#define vTLR		0x0A
#define vCOMMENT	0xFE
#define vRESET		0xAA
#define vESC		0xAE
#define vTERMINATOR	0x00


#define bActionCompare	1
#define bReadInput	4
#define bReadOutput	5
#define bReadMask	6

struct ShiftInstruction {
	unsigned char nBits;
	char* tdi;
	char* tdo;
	char* mask;
	char actionMask;
} typedef ShInstr;


istream* is = NULL;
ostream* os = NULL;

void translate( void );
int nextSpace( char* str );
int removeSpaces( char* );
int strcmpA( char*, char* );

void printString( char*, int );
char btoh( char b );
char htob( char b );
bool arrayEquals( char* str1, char* str2 );

char* Escape( char* );

char* iRuntest( char* str );
char* iState( char* str );
char* iHDR( char* str );
char* iHIR( char* str );
char* iTDR( char* str );
char* iTIR( char* str );
char* iSDR( char* str );
char* iSIR( char* str );

void ReadShiftInstruction( ShInstr* si, char* str );
char* ReadBytes( char* );

// these should be defined as constant, but i don't know how do do that :/
char*           sInstructions[] = { sRUNTEST, sSDR, sSIR, sHDR, sTDR, sHIR, sTIR, sSTATE };
char* (*iInstructions[])(char*) = { iRuntest, iSDR, iSIR, iHDR, iTDR, iHIR, iTIR, iState };
// the number of instructions (length of the above arrays)
#define nInstructions		8


// the global memories
ShInstr HDR, HIR, TDR, TIR, SIR, SDR;


int main(int argc, char *argv[]) {
	char * srcFile = NULL;
	char * destFile = NULL;


	bool flagsEnd = false;
	for (int i = 1; i<argc; i++) {
		if ((argv[i][0] == '-') & !flagsEnd) {
			for (int j = 1; argv[i][j] != 0; j++) {
				if (argv[i][j] == 'v') verbose++;
				else if (argv[i][j] == 'V') verbose--;
				else if (argv[i][j] == 'h') action = showHelp;
				else if (argv[i][j] == 'c') {
					srcFile = strcpy( new char[10], "__stdin__" );
					action = convert;
				}
			}
			if (argv[i][1] == 0) flagsEnd = true;		// a single dash
		} else {
			if (srcFile == NULL) {
				srcFile = strcpy( new char[strlen(argv[i])+1], argv[i] );
				action = convert;
			}
			else if (destFile == NULL) {
				destFile = strcpy( new char[strlen(argv[i])+1], argv[i] );
				action = convert;
			}
		}
	}
	if (verbose >= ERRORS) cout << "mysvf2 by Clemens Helfmeier, ClemensHelfmeier (at) gmx (dot) de\n";
	
	if (verbose >= LOG) {
		clog << "---------------------------------------------------------------\n";
		clog << "parsing commandline\n";
		clog << "verbose = " << verbose << "\n";
	 	clog << "action = " << action << "\n";
		if (srcFile != NULL) clog << "srcFile = " << srcFile << "\n";
		if (destFile != NULL) clog << "destFile = " << destFile << "\n";
		clog << "parsing complete\n";
	}

	if (verbose >= ERRORS) cout << "\n";
	if (srcFile == NULL) {
		if (verbose >= ERRORS) cerr << "no input file specified.\n";
		action == showHelp;
	}

	switch (action) {
		case showHelp:
			// just show help and exit
			cout << "Version " << Version_maj << "." << Version_min << "." << Version_ref << "\n";
			cout << "Help\n";
			cout << argv[0] << " [-v|V][-h] [-] srcFile [destFile]\n";
			cout << " -v        put to more verbose\n";
			cout << " -V        put to less verbose\n";
			cout << " -c        read from console and output to console or file (if given)\n";
			cout << " -h        show this help\n";
			cout << "\n";
			cout << " srcFile   source file to process (SVF format)\n";
			cout << "           \"__stdin__\" tells the program to read from console input.\n";
			cout << " destFile  destination file to write to (mySVF2 format)\n";
			cout << "           \"__stdout__\" tells the program to write to console output.\n";
			cout << "\n";
			cout << "Supported SVF instructions:   SDR, SIR, STATE RESET IDLE, RUNTEST\n";
			cout << " HDR, TDR, HIR, TIR are recognized but don't work yet...\n";
			cout << "\n";
			break;
		case convert:
			// open the input and output files
			if (strcmpA( srcFile, "__stdin__" ) == 0) {
				is = &cin;
			} else {
				ifstream* is_ = new ifstream();
				is_->open( srcFile, ios::in );
				is = is_;
			}
			if (destFile == NULL) {
				if (strcmpA( srcFile, "__stdin__" ) == 0) {
					destFile = strcpy( new char[11], "__stdout__" );
				} else if (strcmpA( srcFile + strlen( srcFile ) - 4, ".svf") != 0) {
					if (verbose >= WARNINGS) cout << "unknown extension at source file.\n";
					destFile = new char[ strlen( srcFile ) + 8 ];
					strcpy( destFile, srcFile );
					strcpy( destFile + strlen( srcFile ), ".mysvf2\0" );
				} else {
					destFile = new char[ strlen( srcFile ) + 4 ];
					strcpy( destFile, srcFile );
					strcpy( destFile + strlen( srcFile ) - 4, ".mysvf2\0" );
				}
				if (verbose >= WARNINGS) cout << "no output file specified. using " << destFile << "\n";
			}
			if (strcmpA( destFile, "__stdout__" ) == 0) {
				os = &cout;
			} else {
				os = new ofstream( destFile, ios::out | ios::binary | ios::trunc );
			}

			
			if (verbose >= LOG) clog << "translating from " << srcFile << " to " << destFile << "\n";
			// convert the file
			translate();

			os->flush();
			// close the files again
			if (strcmpA( srcFile, "__stdin__" ) == 0) ((ifstream*)is)->close();
			if (strcmpA( destFile, "__stdout__" ) == 0) ((ofstream*)os)->close();
			break;
	} 
		

	if (srcFile != NULL) delete srcFile;
	if (destFile != NULL) delete destFile;
	
	if (verbose >= LOG) clog << "mysvf2 exiting ...\n";
	if (verbose >= LOG) clog << "------------------\n";
	return 0;
}

void translate( void ) {

	//translate from input file to output file
	//(is) > (os)
	// reset device
	//char reset[] = { vRESET, vRESET, vCOMMENT, vCOMMENT, vCOMMENT, vCOMMENT, vCOMMENT, vCOMMENT, vCOMMENT, vCOMMENT, vCOMMENT, vTERMINATOR };
	//os->write( reset, 2 );
		
	char line[256];

	// proceed whole input data
	while (is->good()) {
		is->getline( line, 256 );
		
		if (verbose >= LOG) clog << "parsing " << line << " ...\n";

		if (strlen( line )==0) continue;

		if (strcmpA( line, ".EXIT" )==0) break;
		
		int i;
		i = removeSpaces( line );		// start with the command

		// find out the instruction number
		for (int j = 0; j<nInstructions; j++) {
			if (strcmpA( &line[i], sInstructions[j]) == 0) {
				i += nextSpace( &line[i] );
				char* buf;
				buf = iInstructions[j]( &line[i] );
				char* buf2 = Escape( buf );
				if (buf2 != NULL) {
					((ofstream*)os)->write( &buf2[1], buf2[0] );
					delete buf;
					delete buf2;
				}
				break;
			}
		} // for
		
	}
	
	return;
}

char* Escape( char* buf ) {
	if (buf == NULL) return NULL;

	int nEsc = 0;
	// count the number of escapes
	for (int i = 1; i<buf[0]; i++)
		if (buf[i] == (char)vRESET) nEsc++;

	char* buf2 = new char[ buf[0] + 1 + nEsc ];
	buf2[0] = buf[0] + nEsc;

	int n = 1;
	for (int i = 1; i<=buf[0]; i++)
		if (buf[i] != (char)vRESET) buf2[n++] = buf[i];
		else {
			buf2[n++] = vESC;
			buf2[n++] = buf[i];
		}
	return buf2;
}

int removeSpaces( char* str ) {
	int i = 0;
	while ((str[i] == ' ') & (str[i] != 0)) i++;
	return i;
}
int nextSpace( char* str ) {
	// goto behind next space in str
	int i = 0;
	while ((str[i] != ' ') & (str[i] != 0)) i++;
	while ((str[i] == ' ') & (str[i] != 0)) i++;
	return i;
}
int strcmpA( char* str1, char* str2 ){
	if ((strlen( str1 ) == 0) | (strlen( str2 ) == 0)) return -1;
	for (int i = 0; (i<strlen( str1 )) && (i<strlen( str2 )); i++) {
		if (str1[i] != str2[i]) return -1;
	}
	return 0;
}


char* iRuntest( char* str ) {
	if (verbose >= LOG) clog << "iRuntest " << str << "\n";

	// perform runtest instruction
	// the number of clock cycles to wait is specified after
	// the RUNTEST command
	
	char* ret;
	int clk = (int)(atoi( str ));
	if (clk <= 65535) {
		ret = new char[4];
		ret[0] = 3;
		ret[1] = vRUNTEST;
		ret[2] = clk & 0xFF;
		ret[3] = (clk >> 8 ) & 0xFF;
		return ret;
	} else {
		char l = (clk >> 16);
		if ((clk & 0xFFFF) != 0) l++;

		ret = new char[ 1 + 3 * l ];
		ret[0] = 3 * l;
		for (int i = 0; i<l; i++) {
			ret[1+i*3] = vRUNTEST;
			ret[1+i*3+1] = ((int)(clk/l)) & 0xFF;
			ret[1+i*3+2] = ((int)(clk/l)>>8) & 0xFF;
		}
		return ret;
	}
}
char* iState( char* str ) {
	if (verbose >= LOG) clog << "iState " << str << "\n";
	
	// reset the target with vTLR
	// any other state is not implemented by programmer
	
	char* ret;
	ret = new char[2];
	ret[0] = 1;
	ret[1] = vTLR;
	return &ret[0];
}
char* iHDR( char* str ) {
	if (verbose >= LOG) clog << "iHDR " << str << "\n";

	// take the bits and mask into memory
	// no data is sent.

	ReadShiftInstruction( &HDR, str );
	return NULL;
}
char* iHIR( char* str ) {
	if (verbose >= LOG) clog << "iHIR " << str << "\n";
	
	// take the bits and mask into memory
	// no data is sent.
	
	ReadShiftInstruction( &HIR, str );
	return NULL;
}
char* iTDR( char* str ) {
	if (verbose >= LOG) clog << "iTDR " << str << "\n";
	
	// take the bits and mask into memory
	// no data is sent.
	ReadShiftInstruction( &TDR, str );
	return NULL;
}
char* iTIR( char* str ) {
	if (verbose >= LOG) clog << "iTIR " << str << "\n";

	// take the bits and mask into memory
	// no data is sent.
	ReadShiftInstruction( &TIR, str );
	return NULL;
}
char* iSDR( char* str ) {
	if (verbose >= LOG) clog << "iSDR " << str << "\n";

	ReadShiftInstruction( &SDR, str );

	// here starts action.
	// put the HDR, DR and TDR together and send the resulting bits.
	char nBits = SDR.nBits;

	if (nBits <= 0) return NULL;
	
	char nBytes = nBits / 8;
	if (nBytes * 8 < nBits) nBytes++;

	int nRegs = 0;
	if ((SDR.actionMask & (1<<bReadInput)) != 0) nRegs++;
	if ((SDR.actionMask & (1<<bReadOutput)) != 0) nRegs++;
	if ((SDR.actionMask & (1<<bReadMask)) != 0) nRegs++;
	
	char* ret = new char[nRegs * nBytes + 4];
	ret[0] = nBytes * nRegs + 3;
	ret[1] = vSDR;
	ret[2] = nBits;
	ret[3] = SDR.actionMask;

	int n = 4;
	if ((SDR.actionMask & (1<<bReadInput)) != 0)
		for (int i = 0; i<nBytes; i++) ret[n++] = SDR.tdi[i+1];
	if ((SDR.actionMask & (1<<bReadOutput)) != 0)
		for (int i = 0; i<nBytes; i++) ret[n++] = SDR.tdo[i+1];
	if ((SDR.actionMask & (1<<bReadMask)) != 0)
		for (int i = 0; i<nBytes; i++) ret[n++] = SDR.mask[i+1];
	
	return ret;
}
char* iSIR( char* str ) {
	if (verbose >= LOG) clog << "iSIR " << str << "\n";
	
	ReadShiftInstruction( &SIR, str );
	// here starts action.
	// put the HIR, IR and TIR together and send the resulting bits.
	char nBits = SIR.nBits;

	if (nBits <= 0) return NULL;
	
	char nBytes = nBits / 8;
	if (nBytes * 8 < nBits) nBytes++;
	
	int nRegs = 0;
	if ((SIR.actionMask & (1<<bReadInput)) != 0) nRegs++;
	if ((SIR.actionMask & (1<<bReadOutput)) != 0) nRegs++;
	if ((SIR.actionMask & (1<<bReadMask)) != 0) nRegs++;

	char* ret = new char[nRegs * nBytes + 4];
	ret[0] = nBytes*nRegs + 3;
	ret[1] = vSIR;
	ret[2] = nBits;
	ret[3] = SIR.actionMask;

	int n = 4;
	if ((SIR.actionMask & (1<<bReadInput)) != 0)
		for (int i = 0; i<nBytes; i++) ret[n++] = SIR.tdi[i+1];
	if ((SIR.actionMask & (1<<bReadOutput)) != 0)
		for (int i = 0; i<nBytes; i++) ret[n++] = SIR.tdo[i+1];
	if ((SIR.actionMask & (1<<bReadMask)) != 0)
		for (int i = 0; i<nBytes; i++) ret[n++] = SIR.mask[i+1];
	
	return ret;
}

void ReadShiftInstruction( ShInstr* psi, char* str ) {
	//[nLen] TDI (hex) TDO (hex) MASK (hex) SMASK (hex);

	// read the number of bits to handle
	psi->nBits = atoi( str );
	str += nextSpace( str );
	psi->actionMask = 0;
	
	if (psi->nBits > 0 ) {
		// read the bits if needed
		while (strlen( str )>0) {
			if (strcmpA( str, "TDI" ) == 0) {
				// there is a TDI data following
				str += nextSpace( str );
				char * newtdi = ReadBytes( str );
				if (!arrayEquals( newtdi, psi->tdi )) {
					psi->actionMask |= (1<<bReadInput);
					delete psi->tdi;
					psi->tdi = newtdi;
				}
			} else if (strcmpA( str, "TDO" ) == 0 ) {
				// TDO data
				str += nextSpace( str );
				char * newtdo = ReadBytes( str );
				if (!arrayEquals( newtdo, psi->tdo )) {
					psi->actionMask |= (1<<bReadOutput);
					delete psi->tdo;
					psi->tdo = newtdo;
				}
				psi->actionMask |= (1<<bActionCompare);
			} else if (strcmpA( str, "MASK" ) == 0 ) {
				// mask following
				str += nextSpace( str );
				char * newmask = ReadBytes( str );
				if (!arrayEquals( newmask, psi->mask )) {
					psi->actionMask |= (1<<bReadMask);
					delete psi->mask;
					psi->mask = newmask;
				}
				psi->actionMask |= (1<<bActionCompare);
			} else {
				// if (verbose >= WARNINGS) cout << "unknown Argument: " << str << "\n";
			}
			str += nextSpace( str );
		}
		if (verbose >= LOG) {
			clog << "bits to read " << (int)psi->nBits << " tdi: ";
			if (psi->tdi != NULL) printString( &psi->tdi[1], psi->tdi[0] );
			clog << " tdo: ";
			if (psi->tdo != NULL) printString( &psi->tdo[1], psi->tdo[0] );
			clog << " mask: ";
			if (psi->mask != NULL) printString( &psi->mask[1], psi->mask[0] );
			clog << "\n";
		}
	} else {
		// no bits to read
		// remove all buffer
		if (verbose >= LOG) clog << "no bits to read\n";
		delete psi->tdi, psi->tdo, psi->mask;
	}
}


char* ReadBytes( char* str ) {

	str += removeSpaces( str );

	if (str[0] == '(') {
		// hex values
		int i = 0;
		for (i = 0; str[i] != ')'; i++);

		char* ret = new char[ (i-1)/2 +1 ];
		ret[0] = (i-1)/2;

		int n = 1;
		int j;
		for (j = i-2; j>=1; j-=2)
			ret[n++] = (htob( str[j] )<<4) | htob( str[j+1] );
		if (j==0)
			ret[n] = htob( str[1] );
		return ret;
	} else {
		// error
		if (verbose >= ERRORS) clog << "Unknown number format " << str << "\n";
		return NULL;
	}
}

void printString( char* str, int length ) {
	for (int i = 0; i<length; i++) {
		clog << btoh((str[i]>>4) & 0x0F);
		clog << btoh(str[i] & 0x0F);
	}
}

char btoh( char b ) {
	if (b >= 10) return 'A'+b-10;
	return b+'0';
}
char htob( char b ) {
	char ret;
	if (('F'>=b) & (b >= 'A')) ret = b-'A'+10;
	else if (('f'>=b) & (b >= 'a')) ret = b-'a'+10;
	else ret = b-'0';
	return ret;
}

bool arrayEquals( char* str1, char* str2 ) {
	if ((str1 == 0) && (str2 == 0)) return true;
	if ((str1 == 0) || (str2 == 0)) return false;
	if (str1[0] != str2[0]) return false;
	for (int i = 1; i<=str1[0]; i++)
		if (str1[i]!=str2[i]) return false;
	return true;
}

