#include "log.h"
#include "lcd.h"
#include "i2c.h"
#include "uart.h"

void ADC_Init(void) 
{
 
    uint16_t result;
    
    ADMUX = (0<<REFS1) | (1<<REFS0);      // AVcc als Referenz benutzen
    //ADMUX = (1<<REFS1) | (1<<REFS0);      // interne Referenzspannung nutzen
    ADCSRA = (1<<ADPS2) | (1<<ADPS0);     // Frequenzvorteiler
    ADCSRA |= (1<<ADEN);                  // ADC aktivieren
    
    /* nach Aktivieren des ADC wird ein "Dummy-Readout" empfohlen, man liest
    also einen Wert und verwirft diesen, um den ADC "warmlaufen zu lassen" */
    
    ADCSRA |= (1<<ADSC);                  // eine ADC-Wandlung 
    while (ADCSRA & (1<<ADSC) ) {}        // auf Abschluss der Konvertierung warten
    /* ADCW muss einmal gelesen werden, sonst wird Ergebnis der nchsten
    Wandlung nicht bernommen. */
    result = ADCW;
}
 
/* ADC Einzelmessung */
uint16_t ADC_Read( uint8_t channel )
{
    // Kanal waehlen, ohne andere Bits zu beeinfluen
    ADMUX = (ADMUX & ~(0x1F)) | (channel & 0x1F);
    ADCSRA |= (1<<ADSC);            // eine Wandlung "single conversion"
    while (ADCSRA & (1<<ADSC) ) {}  // auf Abschluss der Konvertierung warten
    return ADCW;                    // ADC auslesen und zurckgeben
}

uint16_t ADC_Read16(uint8_t channel)
{
    uint32_t result = 0;
    for(uint16_t i = 0; i < 4096; i++)
        result += ADC_Read(channel);
    return result >> 6;
}

uint16_t ADC_Read13(uint8_t channel)
{
    uint32_t result = 0;
    for(uint16_t i = 0; i < 64; i++)
        result += ADC_Read(channel);
    return result >> 3;
}

void print13(uint16_t value)
{
    uint32_t result = value * (uint32_t)625;
    result = result >> 10;
    char buffer[] = "x0000V";
    while(result >= 1000)
    {
        result -= 1000;
        buffer[1]++;
    }
    while(result >= 100)
    {
        result -= 100;
        buffer[2]++;
    }
    while(result >= 10)
    {
        result -= 10;
        buffer[3]++;
    }
    buffer[4] += result;
    buffer[0] = buffer[1];
    buffer[1] = '.';
    lcd_puts(buffer);
}
void print13s(uint16_t value)
{
    uint32_t result = value * (uint32_t)25;
    result = result >> 12;
    char buffer[] = "x00";
    while(result >= 10)
    {
        result -= 10;
        buffer[1]++;
    }
    buffer[2] += result;
    buffer[0] = buffer[1];
    buffer[1] = '.';
    lcd_puts(buffer);
}

void itoa8(uint8_t result, char* buffer)
{
    buffer[0] = '0';
    buffer[1] = '0';
    while(result >= 100)
    {
        result -= 100;
        buffer[0]++;
    }
    if(buffer[0] == '0')
        buffer[0] = ' ';
    while(result >= 10)
    {
        result -= 10;
        buffer[1]++;
    }
    if(buffer[1] == '0')
        buffer[1] = ' ';
    buffer[2] = result + '0';
    buffer[3] = '\0';
}
void itoa16(uint16_t result, char* buffer)
{
    buffer[0] = '0';
    buffer[1] = '0';
    buffer[2] = '0';
    buffer[3] = '0';
    while(result >= 10000)
    {
        result -= 10000;
        buffer[0]++;
    }
    if(buffer[0] == '0')
        buffer[0] = ' ';
    while(result >= 1000)
    {
        result -= 1000;
        buffer[1]++;
    }
    if(buffer[1] == '0')
        buffer[1] = ' ';
    while(result >= 100)
    {
        result -= 100;
        buffer[2]++;
    }
    if(buffer[2] == '0')
        buffer[2] = ' ';
    while(result >= 10)
    {
        result -= 10;
        buffer[3]++;
    }
    if(buffer[3] == '0')
        buffer[3] = ' ';
    buffer[4] = result + '0';
    buffer[5] = '\0';
}

//Stream
#define BUFFERLEN 64
#define WRITE(pos,buf,count) i2cMemWrite(0xA0,pos,buf,count,0)
#define READ(pos,buf,count) i2cMemRead(0xA0,pos,buf,count)

uint16_t streampos = 0;
uint8_t buffer[BUFFERLEN];
uint8_t bufferpos = 0;
uint8_t newstream = 0;

uint16_t GetStreamEnd(void)
{
    uint8_t rbuffer[BUFFERLEN];    
    for(uint16_t pos = 0; pos < 16384 ; pos += BUFFERLEN)
    {
        READ(pos, rbuffer, BUFFERLEN);
        for(uint8_t i = 0; i < BUFFERLEN; i += 2)
            if(rbuffer[i] == 0xFF)
                return pos + i;
    }
    return 0;
}

void NewStream(uint8_t channels)
{
    if(streampos == 0)
        streampos = GetStreamEnd();
    newstream = channels << 5;
}

void WriteBuffer13(uint16_t data)
{
    WriteBuffer8((uint8_t)((data >> 8) | newstream));
    WriteBuffer8((uint8_t) data);
    newstream = 0;
}

void WriteBuffer8(uint8_t data)
{
    buffer[bufferpos++] = data;
    streampos++;
    if((((uint8_t)streampos) & (BUFFERLEN - 1)) == 0 || bufferpos == BUFFERLEN)
        ClearBuffer();
}

void ClearBuffer(void)
{
    WRITE(streampos - bufferpos, buffer, bufferpos);
    buffer[BUFFERLEN - 1] = 0;
    uart_puts(buffer);
    bufferpos = 0;
}


