/*----------------------------------------------------------------------------
 Copyright:      Ulrich Radig (mail@ulrichradig.de)
 Author:         Ulrich Radig
 Remarks:        
 known Problems: none
 Version:        23.06.2011
 Description:    Brushless Motor Controller for ATmega48/88/168
 
 Dieses Programm ist freie Software. Sie knnen es unter den Bedingungen der 
 GNU General Public License, wie von der Free Software Foundation verffentlicht, 
 weitergeben und/oder modifizieren, entweder gem Version 2 der Lizenz oder 
 (nach Ihrer Option) jeder spteren Version. 

 Die Verffentlichung dieses Programms erfolgt in der Hoffnung, 
 da es Ihnen von Nutzen sein wird, aber OHNE IRGENDEINE GARANTIE, 
 sogar ohne die implizite Garantie der MARKTREIFE oder der VERWENDBARKEIT 
 FR EINEN BESTIMMTEN ZWECK. Details finden Sie in der GNU General Public License. 

 Sie sollten eine Kopie der GNU General Public License zusammen mit diesem 
 Programm erhalten haben. 
 Falls nicht, schreiben Sie an die Free Software Foundation, 
 Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA. 
------------------------------------------------------------------------------*/

#include <avr/interrupt.h>
#include <avr/io.h>


//PHASE1 (U)
#define UH_DDR	DDRB |= (1<<3);
#define UH_ON	TCCR2A |= (1<<COM2A1);
#define UH_OFF	TCCR2A &= ~(1<<COM2A1);

//PHASE1 (U)
#define UL_DDR	DDRB |= (1<<1);
#define UL_ON	PORTB |= (1<<1);
#define UL_OFF	PORTB &= ~(1<<1);


//PHASE2 (V)
#define VH_DDR	DDRD |= (1<<5);
#define VH_ON	TCCR0A |= (1<<COM0B1);
#define VH_OFF	TCCR0A &= ~(1<<COM0B1);

//PHASE2 (V)
#define VL_DDR	DDRB |= (1<<2);
#define VL_ON	PORTB |= (1<<2);
#define VL_OFF	PORTB &= ~(1<<2);


//PHASE3 (W)
#define WH_DDR	DDRD |= (1<<3);
#define WH_ON	TCCR2A |= (1<<COM2B1);
#define WH_OFF	TCCR2A &= ~(1<<COM2B1);

//PHASE3 (W)
#define WL_DDR	DDRC |= (1<<3);
#define WL_ON	PORTC |= (1<<3);
#define WL_OFF	PORTC &= ~(1<<3);


#define PHASE_ALL_OFF	UH_OFF;UL_OFF;VH_OFF;VL_OFF;WH_OFF;WL_OFF;

#define SENSE_U		ADMUX = 0;
#define SENSE_V		ADMUX = 1;
#define SENSE_W		ADMUX = 2;

#define SENSE_H		(ACSR&(1<<ACO))

volatile unsigned char pwm = 20;
volatile unsigned char rotor_state = 0;
volatile unsigned char rotor_run = 0;

//############################################################################
void next_commutate_state (unsigned char startup)
//############################################################################
{
	switch (rotor_state)
	{
		case (0):
			if(!SENSE_H || startup)
			{
				WH_OFF;
				SENSE_W;
				UH_ON;
				rotor_state = 1;
				TCNT1 = 1;
			}
			break;

		case (1):
			if(SENSE_H || startup)
			{
				VL_OFF;
				SENSE_V;
				WL_ON;
				rotor_state = 2;
				TCNT1 = 1;
			}
			break;

		case (2):
			if(!SENSE_H || startup)
			{
				UH_OFF;
				SENSE_U;
				VH_ON;
				rotor_state = 3;
				TCNT1 = 1;
			}
			break;
	
		case (3):
			if(SENSE_H || startup)
			{
				WL_OFF;
				SENSE_W;
				UL_ON;
				rotor_state = 4;
				TCNT1 = 1;
			}
			break;

		case (4):
			if(!SENSE_H || startup)
			{
				VH_OFF;
				SENSE_V;
				WH_ON;
				rotor_state = 5;
				TCNT1 = 1;
			}
			break;

		case (5):
			if(SENSE_H || startup)
			{
				UL_OFF;
				SENSE_U;
				VL_ON;
				rotor_state = 0;
				TCNT1 = 1;
			}
			break;
	}
}

//############################################################################
//back EMF zero crossing detection
ISR (ANALOG_COMP_vect) 
//############################################################################
{
	next_commutate_state (0);
	rotor_run++;
	if(rotor_run > 200)
	{
		rotor_run = 200;
	}
}

//############################################################################
ISR (TIMER1_OVF_vect)
//############################################################################
{	
	rotor_run = 0;
	OCR2A = 10;
	OCR2B = 10;
	OCR0B = 10;
}

//############################################################################
//Hauptprogramm
int main (void) 
//############################################################################
{
	unsigned char pwm_tmp = 10;
	//Watchdog off
	asm("wdr");
	MCUSR &= ~(1<<WDRF);
	//Write logical one to WDCE and WDE
	WDTCSR |= (1<<WDCE) | (1<<WDE);
	//Turn off WDT
	WDTCSR = 0x00;
	
	//Motordriver output
	UH_DDR;
	VH_DDR;
	WH_DDR;
	UL_DDR;
	VL_DDR;
	WL_DDR;

	PHASE_ALL_OFF;
	
	//PWM for PHASE_A_H (16KHz)
	TCCR0A |= (1<<COM0B1|1<<WGM01|1<<WGM00);
	TCCR0B |= (1<<CS00);
	
	//PWM for PHASE_B_H and WH (16KHz)
	TCCR2A |= (1<<COM2A1|1<<COM2B1|1<<WGM21|1<<WGM20);
	TCCR2B |= (1<<CS20);

	PHASE_ALL_OFF;

	//TIMER1 for start commutation or Back EMF lost
	TCCR1B |= (1<<CS10);
	TIMSK1 |= (1<<TOIE1);

	//Comperator init for back EMF
	ADCSRB	|= (1<<ACME);
	DIDR1	|= (1<<AIN0D);
	ACSR	|= (1<<ACIE);
	sei();


	while(1)
	{	
		for(unsigned long b=0;b<30000;b++)asm("nop");
		//Motor anlaufen lassen wenn kein BACK_EMF vorhanden
		if(rotor_run < 100)
		{
			ACSR &=~(1<<ACIE);
			for(unsigned char a = 0;a<255;a++)
			{	
				if(a > 220) ACSR|=(1<<ACIE);
				
				OCR2A = 10;
				OCR2B = 10;
				OCR0B = 10;
				pwm_tmp = 8;
				
				next_commutate_state (1);
				for(unsigned long b=0;b<(40*(255-a));b++)asm("nop");
			}
		}
		else
		{
			if(pwm > pwm_tmp) pwm_tmp++;
			if(pwm < pwm_tmp) pwm_tmp--;
				
			OCR2A = pwm_tmp;
			OCR2B = pwm_tmp;
			OCR0B = pwm_tmp;
		}
	}
}

