/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * irmp.c - infrared multi-protocol decoder, supports several remote control protocols
 *
 * Copyright (c) 2009-2010 Frank Meyer - frank(at)fli4l.de
 *
 * ATMEGA88 @ 8 MHz
 *
 * Typical manufacturers:
 *
 * SIRCS      - Sony
 * NEC        - NEC, Yamaha, Canon, Tevion, Harman/Kardon, Hitachi, JVC, Pioneer, Toshiba, Xoro, Orion, and many other Japanese manufacturers
 * SAMSUNG    - Samsung
 * MATSUSHITA - Matsushita
 * KASEIKYO   - Panasonic, Denon & other Japanese manufacturers (members of "Japan's Association for Electric Home Application")
 * RECS80     - Philips, Nokia, Thomson, Nordmende, Telefunken, Saba
 * RC5        - Philips and other European manufacturers
 * DENON      - Denon
 * RC6        - Philips and other European manufacturers
 * PANASONIC  - Panasonic (older, yet not implemented)
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   SIRCS
 *   -----
 *
 *   frame: 1 start bit + 12-20 data bits + no stop bit
 *   data:  7 command bits + 5 address bits + 0 to 8 additional bits
 *
 *   start bit:                           data "0":                 data "1":                 stop bit:
 *   -----------------_________           ------_____               ------------______
 *       2400us         600us             600us 600us               1200us      600 us        no stop bit
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   NEC + extended NEC
 *   -------------------------
 *
 *   frame: 1 start bit + 32 data bits + 1 stop bit
 *   data NEC:          8 address bits + 8 inverted address bits + 8 command bits + 8 inverted command bits
 *   data extended NEC: 16 address bits + 8 command bits + 8 inverted command bits
 *
 *   start bit:                           data "0":                 data "1":                 stop bit:
 *   -----------------_________           ------______              ------________________    ------______....
 *       9000us        4500us             560us  560us              560us    1690 us          560us
 *
 *
 *   Repetition frame:
 *
 *   -----------------_________------______  .... ~100ms Pause, then repeat
 *       9000us        2250us   560us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   SAMSUNG
 *   -------
 *
 *   frame: 1 start bit + 16 data(1) bits + 1 sync bit + additional 20 data(2) bits + 1 stop bit
 *   data(1): 16 address bits
 *   data(2): 4 ID bits + 8 command bits + 8 inverted command bits
 *
 *   start bit:                           data "0":                 data "1":                 sync bit:               stop bit:
 *   ----------______________             ------______              ------________________    ------______________    ------______....
 *    4500us       4500us                 550us  450us              550us    1450us           550us    4500us         550us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   SAMSUNG32
 *   ----------
 *
 *   frame: 1 start bit + 32 data bits + 1 stop bit
 *   data: 16 address bits + 16 command bits
 *
 *   start bit:                           data "0":                 data "1":                 stop bit:
 *   ----------______________             ------______              ------________________    ------______....
 *    4500us       4500us                 550us  450us              550us    1450us           550us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   MATSUSHITA
 *   ----------
 *
 *   frame: 1 start bit + 24 data bits + 1 stop bit
 *   data:  6 custom bits + 6 command bits + 12 address bits
 *
 *   start bit:                           data "0":                 data "1":                 stop bit:
 *   ----------_________                  ------______              ------________________    ------______....
 *    3488us     3488us                   872us  872us              872us    2616us           872us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   KASEIKYO
 *   --------
 *
 *   frame: 1 start bit + 48 data bits + 1 stop bit
 *   data:  16 manufacturer bits + 4 parity bits + 4 genre1 bits + 4 genre2 bits + 10 command bits + 2 id bits + 8 parity bits
 *
 *   start bit:                           data "0":                 data "1":                 stop bit:
 *   ----------______                     ------______              ------________________    ------______....
 *    3380us   1690us                     423us  423us              423us    1269us           423us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   RECS80
 *   ------
 *
 *   frame: 2 start bits + 10 data bits + 1 stop bit
 *   data:  1 toggle bit + 3 address bits + 6 command bits
 *
 *   start bit:                           data "0":                 data "1":                 stop bit:
 *   -----_____________________           -----____________         -----______________       ------_______....
 *   158us       5060us                   158us   5060us            158us    7600us           158us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   RC5 + RC5X
 *   ----------
 *
 *   RC5 frame:  2 start bits + 12 data bits + no stop bit
 *   RC5 data:   1 toggle bit + 5 address bits + 6 command bits
 *   RC5X frame: 1 start bit +  13 data bits + no stop bit
 *   RC5X data:  1 inverted command bit + 1 toggle bit + 5 address bits + 6 command bits
 *
 *   start bit:              data "0":                data "1":
 *   ______-----             ------______             ______------
 *   889us 889us             889us 889us              889us 889us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   DENON
 *   -----
 *
 *   frame: 0 start bits + 16 data bits + stop bit + 65ms pause + 16 inverted data bits + stop bit
 *   data:  5 address bits + 10 command bits
 *
 *   data "0":                 data "1":
 *   ------________________    ------______________
 *   275us      1050us         275us   1900us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   RC6
 *   ---
 *
 *   RC6 frame:  1 start bit + 1 bit "1" + 3 mode bits + 1 toggle bit + 16 data bits + 2666 s pause
 *   RC6 data:   8 address bits + 8 command bits
 *
 *   start  bit               toggle bit "0":      toggle bit "1":     data/mode "0":      data/mode "1":
 *   ____________-------      _______-------       -------_______      _______-------      -------_______
 *      2666us    889us        889us  889us         889us  889us        444us  444us        444us  444us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   APPLE
 *   -----
 *
 *   frame: 1 start bit + 32 data bits + 1 stop bit
 *   data:  16 address bits + 11100000 + 8 command bits
 *
 *   start bit:                           data "0":                 data "1":                 stop bit:
 *   -----------------_________           ------______              ------________________    ------______....
 *       9000us        4500us             560us  560us              560us    1690 us          560us
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 *   PANASONIC (older protocol, yet not implemented, see also MATSUSHITA, timing very similar)
 *   -----------------------------------------------------------------------------------------
 *
 *   frame: 1 start bit + 22 data bits + 1 stop bit
 *   22 data bits = 5 custom bits + 6 data bits + 5 inverted custom bits + 6 inverted data bits
 *
 *   European version:      T = 456us
 *   USA & Canada version:  T = 422us
 *
 *   start bit:                           data "0":                 data "1":                 stop bit:
 *        8T            8T                 2T   2T                   2T      6T                2T
 *   -------------____________            ------_____               ------_____________       ------_______....
 *      3648us        3648us              912us 912us               912us    2736us           912us                (Europe)
 *      3376us        3376us              844us 844us               844us    2532us           844us                (US)
 *
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */

#ifdef unix                                                                 // test on linux/unix
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#define DEBUG

#else // not unix:

#ifdef WIN32
#include <stdio.h>
typedef unsigned char   uint8_t ;
typedef unsigned short  uint16_t ;
#define DEBUG

#else

#ifndef CODEVISION
#include <inttypes.h>
#include <stdio.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/pgmspace.h>
#endif  // CODEVISION

#endif // windows
#endif // unix

#include "irmp.h"

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * Change settings from 1 to 0 if you want to disable one or more decoders.
 * This saves program space.
 * 1 enable  decoder
 * 0 disable decoder
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define IRMP_SUPPORT_SIRCS_PROTOCOL             1       // flag: support SIRCS                      uses ~100 bytes
#define IRMP_SUPPORT_NEC_PROTOCOL               1       // flag: support NEC + APPLE                uses ~250 bytes
#define IRMP_SUPPORT_SAMSUNG_PROTOCOL           1       // flag: support Samsung + Samsung32        uses ~250 bytes
#define IRMP_SUPPORT_MATSUSHITA_PROTOCOL        1       // flag: support Matsushita                 uses ~100 bytes
#define IRMP_SUPPORT_KASEIKYO_PROTOCOL          1       // flag: support Kaseikyo                   uses ~100 bytes
#define IRMP_SUPPORT_RECS80_PROTOCOL            1       // flag: support RECS80                     uses ~100 bytes
#define IRMP_SUPPORT_RC5_PROTOCOL               1       // flag: support RC5                        uses ~250 bytes
#define IRMP_SUPPORT_DENON_PROTOCOL             1       // flag: support DENON                      uses ~250 bytes
#define IRMP_SUPPORT_RC6_PROTOCOL               1       // flag: support RC6                        uses ~200 bytes

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * Change hardware pin here:
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define IRMP_PORT                               PORTD
#define IRMP_DDR                                DDRD
#define IRMP_PIN                                PIND
#define IRMP_BIT                                2       // use PD2 as IR input

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 * Set IRMP_LOGGING to 1 if want to log data to UART with 9600Bd
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define IRMP_LOGGING                            0                             // 1: log IR signal (scan), 0: do not (default)

#define IRMP_TIMEOUT                            120                           // timeout after 12 ms darkness
#define IRMP_REPETITION_TIME                    (uint16_t)(F_INTERRUPTS * 100.0e-3 + 0.5)  // autodetect key repetition within 100 msec

#define MIN_TOLERANCE_10                        0.9                           // -10%
#define MAX_TOLERANCE_10                        1.1                           // +10%

#define MIN_TOLERANCE_20                        0.8                           // -20%
#define MAX_TOLERANCE_20                        1.2                           // +20%

#define MIN_TOLERANCE_30                        0.7                           // -30%
#define MAX_TOLERANCE_30                        1.3                           // +30%

#define MIN_TOLERANCE_40                        0.6                           // -40%
#define MAX_TOLERANCE_40                        1.4                           // +40%

#define MIN_TOLERANCE_50                        0.5                           // -50%
#define MAX_TOLERANCE_50                        1.5                           // +50%

#define MIN_TOLERANCE_60                        0.4                           // -60%
#define MAX_TOLERANCE_60                        1.6                           // +60%

#define SIRCS_START_BIT_PULSE_LEN_MIN           (uint8_t)(F_INTERRUPTS * SIRCS_START_BIT_PULSE_TIME * MIN_TOLERANCE_10 + 0.5)
#define SIRCS_START_BIT_PULSE_LEN_MAX           (uint8_t)(F_INTERRUPTS * SIRCS_START_BIT_PULSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define SIRCS_START_BIT_PAUSE_LEN_MIN           (uint8_t)(F_INTERRUPTS * SIRCS_START_BIT_PAUSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define SIRCS_START_BIT_PAUSE_LEN_MAX           (uint8_t)(F_INTERRUPTS * SIRCS_START_BIT_PAUSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define SIRCS_1_PULSE_LEN_MIN                   (uint8_t)(F_INTERRUPTS * SIRCS_1_PULSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define SIRCS_1_PULSE_LEN_MAX                   (uint8_t)(F_INTERRUPTS * SIRCS_1_PULSE_TIME * MAX_TOLERANCE_30 + 0.5)
#define SIRCS_0_PULSE_LEN_MIN                   (uint8_t)(F_INTERRUPTS * SIRCS_0_PULSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define SIRCS_0_PULSE_LEN_MAX                   (uint8_t)(F_INTERRUPTS * SIRCS_0_PULSE_TIME * MAX_TOLERANCE_50 + 0.5)
#define SIRCS_PAUSE_LEN_MIN                     (uint8_t)(F_INTERRUPTS * SIRCS_PAUSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define SIRCS_PAUSE_LEN_MAX                     (uint8_t)(F_INTERRUPTS * SIRCS_PAUSE_TIME * MAX_TOLERANCE_50 + 0.5)

#define NEC_START_BIT_PULSE_LEN_MIN             (uint8_t)(F_INTERRUPTS * NEC_START_BIT_PULSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define NEC_START_BIT_PULSE_LEN_MAX             (uint8_t)(F_INTERRUPTS * NEC_START_BIT_PULSE_TIME * MAX_TOLERANCE_40 + 0.5)
#define NEC_START_BIT_PAUSE_LEN_MIN             (uint8_t)(F_INTERRUPTS * NEC_START_BIT_PAUSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define NEC_START_BIT_PAUSE_LEN_MAX             (uint8_t)(F_INTERRUPTS * NEC_START_BIT_PAUSE_TIME * MAX_TOLERANCE_40 + 0.5)
#define NEC_REPEAT_START_BIT_PAUSE_LEN_MIN      (uint8_t)(F_INTERRUPTS * NEC_REPEAT_START_BIT_PAUSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define NEC_REPEAT_START_BIT_PAUSE_LEN_MAX      (uint8_t)(F_INTERRUPTS * NEC_REPEAT_START_BIT_PAUSE_TIME * MAX_TOLERANCE_40 + 0.5)
#define NEC_PULSE_LEN_MIN                       (uint8_t)(F_INTERRUPTS * NEC_PULSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define NEC_PULSE_LEN_MAX                       (uint8_t)(F_INTERRUPTS * NEC_PULSE_TIME * MAX_TOLERANCE_40 + 0.5)
#define NEC_1_PAUSE_LEN_MIN                     (uint8_t)(F_INTERRUPTS * NEC_1_PAUSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define NEC_1_PAUSE_LEN_MAX                     (uint8_t)(F_INTERRUPTS * NEC_1_PAUSE_TIME * MAX_TOLERANCE_40 + 0.5)
#define NEC_0_PAUSE_LEN_MIN                     (uint8_t)(F_INTERRUPTS * NEC_0_PAUSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define NEC_0_PAUSE_LEN_MAX                     (uint8_t)(F_INTERRUPTS * NEC_0_PAUSE_TIME * MAX_TOLERANCE_40 + 0.5)

#define SAMSUNG_START_BIT_PULSE_LEN_MIN         (uint8_t)(F_INTERRUPTS * SAMSUNG_START_BIT_PULSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define SAMSUNG_START_BIT_PULSE_LEN_MAX         (uint8_t)(F_INTERRUPTS * SAMSUNG_START_BIT_PULSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define SAMSUNG_START_BIT_PAUSE_LEN_MIN         (uint8_t)(F_INTERRUPTS * SAMSUNG_START_BIT_PAUSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define SAMSUNG_START_BIT_PAUSE_LEN_MAX         (uint8_t)(F_INTERRUPTS * SAMSUNG_START_BIT_PAUSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define SAMSUNG_PULSE_LEN_MIN                   (uint8_t)(F_INTERRUPTS * SAMSUNG_PULSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define SAMSUNG_PULSE_LEN_MAX                   (uint8_t)(F_INTERRUPTS * SAMSUNG_PULSE_TIME * MAX_TOLERANCE_50 + 0.5)
#define SAMSUNG_1_PAUSE_LEN_MIN                 (uint8_t)(F_INTERRUPTS * SAMSUNG_1_PAUSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define SAMSUNG_1_PAUSE_LEN_MAX                 (uint8_t)(F_INTERRUPTS * SAMSUNG_1_PAUSE_TIME * MAX_TOLERANCE_50 + 0.5)
#define SAMSUNG_0_PAUSE_LEN_MIN                 (uint8_t)(F_INTERRUPTS * SAMSUNG_0_PAUSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define SAMSUNG_0_PAUSE_LEN_MAX                 (uint8_t)(F_INTERRUPTS * SAMSUNG_0_PAUSE_TIME * MAX_TOLERANCE_50 + 0.5)

#define MATSUSHITA_START_BIT_PULSE_LEN_MIN      (uint8_t)(F_INTERRUPTS * MATSUSHITA_START_BIT_PULSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define MATSUSHITA_START_BIT_PULSE_LEN_MAX      (uint8_t)(F_INTERRUPTS * MATSUSHITA_START_BIT_PULSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define MATSUSHITA_START_BIT_PAUSE_LEN_MIN      (uint8_t)(F_INTERRUPTS * MATSUSHITA_START_BIT_PAUSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define MATSUSHITA_START_BIT_PAUSE_LEN_MAX      (uint8_t)(F_INTERRUPTS * MATSUSHITA_START_BIT_PAUSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define MATSUSHITA_PULSE_LEN_MIN                (uint8_t)(F_INTERRUPTS * MATSUSHITA_PULSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define MATSUSHITA_PULSE_LEN_MAX                (uint8_t)(F_INTERRUPTS * MATSUSHITA_PULSE_TIME * MAX_TOLERANCE_40 + 0.5)
#define MATSUSHITA_1_PAUSE_LEN_MIN              (uint8_t)(F_INTERRUPTS * MATSUSHITA_1_PAUSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define MATSUSHITA_1_PAUSE_LEN_MAX              (uint8_t)(F_INTERRUPTS * MATSUSHITA_1_PAUSE_TIME * MAX_TOLERANCE_40 + 0.5)
#define MATSUSHITA_0_PAUSE_LEN_MIN              (uint8_t)(F_INTERRUPTS * MATSUSHITA_0_PAUSE_TIME * MIN_TOLERANCE_40 + 0.5)
#define MATSUSHITA_0_PAUSE_LEN_MAX              (uint8_t)(F_INTERRUPTS * MATSUSHITA_0_PAUSE_TIME * MAX_TOLERANCE_40 + 0.5)

#define KASEIKYO_START_BIT_PULSE_LEN_MIN        (uint8_t)(F_INTERRUPTS * KASEIKYO_START_BIT_PULSE_TIME * MIN_TOLERANCE_30 + 0.5)
#define KASEIKYO_START_BIT_PULSE_LEN_MAX        (uint8_t)(F_INTERRUPTS * KASEIKYO_START_BIT_PULSE_TIME * MAX_TOLERANCE_30 + 0.5)
#define KASEIKYO_START_BIT_PAUSE_LEN_MIN        (uint8_t)(F_INTERRUPTS * KASEIKYO_START_BIT_PAUSE_TIME * MIN_TOLERANCE_30 + 0.5)
#define KASEIKYO_START_BIT_PAUSE_LEN_MAX        (uint8_t)(F_INTERRUPTS * KASEIKYO_START_BIT_PAUSE_TIME * MAX_TOLERANCE_30 + 0.5)
#define KASEIKYO_PULSE_LEN_MIN                  (uint8_t)(F_INTERRUPTS * KASEIKYO_PULSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define KASEIKYO_PULSE_LEN_MAX                  (uint8_t)(F_INTERRUPTS * KASEIKYO_PULSE_TIME * MAX_TOLERANCE_60 + 0.5)
#define KASEIKYO_1_PAUSE_LEN_MIN                (uint8_t)(F_INTERRUPTS * KASEIKYO_1_PAUSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define KASEIKYO_1_PAUSE_LEN_MAX                (uint8_t)(F_INTERRUPTS * KASEIKYO_1_PAUSE_TIME * MAX_TOLERANCE_50 + 0.5)
#define KASEIKYO_0_PAUSE_LEN_MIN                (uint8_t)(F_INTERRUPTS * KASEIKYO_0_PAUSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define KASEIKYO_0_PAUSE_LEN_MAX                (uint8_t)(F_INTERRUPTS * KASEIKYO_0_PAUSE_TIME * MAX_TOLERANCE_50 + 0.5)

#define RECS80_START_BIT_PULSE_LEN_MIN          (uint8_t)(F_INTERRUPTS * RECS80_START_BIT_PULSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define RECS80_START_BIT_PULSE_LEN_MAX          (uint8_t)(F_INTERRUPTS * RECS80_START_BIT_PULSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define RECS80_START_BIT_PAUSE_LEN_MIN          (uint8_t)(F_INTERRUPTS * RECS80_START_BIT_PAUSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define RECS80_START_BIT_PAUSE_LEN_MAX          (uint8_t)(F_INTERRUPTS * RECS80_START_BIT_PAUSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define RECS80_PULSE_LEN_MIN                    (uint8_t)(F_INTERRUPTS * RECS80_PULSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define RECS80_PULSE_LEN_MAX                    (uint8_t)(F_INTERRUPTS * RECS80_PULSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define RECS80_1_PAUSE_LEN_MIN                  (uint8_t)(F_INTERRUPTS * RECS80_1_PAUSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define RECS80_1_PAUSE_LEN_MAX                  (uint8_t)(F_INTERRUPTS * RECS80_1_PAUSE_TIME * MAX_TOLERANCE_20 + 0.5)
#define RECS80_0_PAUSE_LEN_MIN                  (uint8_t)(F_INTERRUPTS * RECS80_0_PAUSE_TIME * MIN_TOLERANCE_20 + 0.5)
#define RECS80_0_PAUSE_LEN_MAX                  (uint8_t)(F_INTERRUPTS * RECS80_0_PAUSE_TIME * MAX_TOLERANCE_20 + 0.5)

#define RC5_START_BIT_LEN_MIN                   (uint8_t)(F_INTERRUPTS * RC5_BIT_TIME * MIN_TOLERANCE_20 + 0.5)
#define RC5_START_BIT_LEN_MAX                   (uint8_t)(F_INTERRUPTS * RC5_BIT_TIME * MAX_TOLERANCE_20 + 0.5)
#define RC5_BIT_LEN_MIN                         (uint8_t)(F_INTERRUPTS * RC5_BIT_TIME * MIN_TOLERANCE_20 + 0.5)
#define RC5_BIT_LEN_MAX                         (uint8_t)(F_INTERRUPTS * RC5_BIT_TIME * MAX_TOLERANCE_20 + 0.5)

#define DENON_PULSE_LEN_MIN                     (uint8_t)(F_INTERRUPTS * DENON_PULSE_TIME * MIN_TOLERANCE_50 + 0.5)
#define DENON_PULSE_LEN_MAX                     (uint8_t)(F_INTERRUPTS * DENON_PULSE_TIME * MAX_TOLERANCE_50 + 0.5)
#define DENON_1_PAUSE_LEN_MIN                   (uint8_t)(F_INTERRUPTS * DENON_1_PAUSE_TIME * MIN_TOLERANCE_30 + 0.5)
#define DENON_1_PAUSE_LEN_MAX                   (uint8_t)(F_INTERRUPTS * DENON_1_PAUSE_TIME * MAX_TOLERANCE_30 + 0.5)
#define DENON_0_PAUSE_LEN_MIN                   (uint8_t)(F_INTERRUPTS * DENON_0_PAUSE_TIME * MIN_TOLERANCE_30 + 0.5)
#define DENON_0_PAUSE_LEN_MAX                   (uint8_t)(F_INTERRUPTS * DENON_0_PAUSE_TIME * MAX_TOLERANCE_30 + 0.5)

#define RC6_START_BIT_PULSE_LEN_MIN             (uint8_t)(F_INTERRUPTS * RC6_START_BIT_PULSE_TIME * MIN_TOLERANCE_30 + 0.5)
#define RC6_START_BIT_PULSE_LEN_MAX             (uint8_t)(F_INTERRUPTS * RC6_START_BIT_PULSE_TIME * MAX_TOLERANCE_30 + 0.5)
#define RC6_START_BIT_PAUSE_LEN_MIN             (uint8_t)(F_INTERRUPTS * RC6_START_BIT_PAUSE_TIME * MIN_TOLERANCE_30 + 0.5)
#define RC6_START_BIT_PAUSE_LEN_MAX             (uint8_t)(F_INTERRUPTS * RC6_START_BIT_PAUSE_TIME * MAX_TOLERANCE_30 + 0.5)
#define RC6_TOGGLE_BIT_LEN_MIN                  (uint8_t)(F_INTERRUPTS * RC6_TOGGLE_BIT_TIME * MIN_TOLERANCE_20 + 0.5)
#define RC6_TOGGLE_BIT_LEN_MAX                  (uint8_t)(F_INTERRUPTS * RC6_TOGGLE_BIT_TIME * MAX_TOLERANCE_20 + 0.5)
#define RC6_BIT_LEN_MIN                         (uint8_t)(F_INTERRUPTS * RC6_BIT_TIME * MIN_TOLERANCE_30 + 0.5)
#define RC6_BIT_LEN_MAX                         (uint8_t)(F_INTERRUPTS * RC6_BIT_TIME * MAX_TOLERANCE_30 + 0.5)

#define AUTO_REPETITION_LEN                     (uint16_t)(F_INTERRUPTS * AUTO_REPETITION_TIME + 0.5)       // use uint16_t!

#ifdef DEBUG
#define DEBUG_PUTCHAR(a)                        { if (! silent) { putchar (a);              } }
#define DEBUG_PRINTF1(a)                        { if (! silent) { printf (a);               } }
#define DEBUG_PRINTF2(a,b)                      { if (! silent) { printf (a,b);             } }
#define DEBUG_PRINTF3(a,b,c)                    { if (! silent) { printf (a,b,c);           } }
#define DEBUG_PRINTF4(a,b,c,d)                  { if (! silent) { printf (a,b,c,d);         } }
#define DEBUG_PRINTF5(a,b,c,d,e)                { if (! silent) { printf (a,b,c,d,e);       } }
#define DEBUG_PRINTF6(a,b,c,d,e,f)              { if (! silent) { printf (a,b,c,d,e,f);     } }
#define DEBUG_PRINTF7(a,b,c,d,e,f,g)            { if (! silent) { printf (a,b,c,d,e,f,g);   } }
static int silent;
#else
#define DEBUG_PUTCHAR(a)
#define DEBUG_PRINTF1(a)
#define DEBUG_PRINTF2(a,b)
#define DEBUG_PRINTF3(a,b,c)
#define DEBUG_PRINTF4(a,b,c,d)
#define DEBUG_PRINTF5(a,b,c,d,e)
#define DEBUG_PRINTF6(a,b,c,d,e,f)
#define DEBUG_PRINTF7(a,b,c,d,e,f,g)
#endif

#if IRMP_LOGGING == 1
#define irmp_logIsr()                           irmp_logIr((IRMP_PIN & (1<<IRMP_BIT))?1:0)
#define UART_BAUD                               9600L

// calculate real baud rate:
#define UBRR_VAL                                ((F_CPU+UART_BAUD*8)/(UART_BAUD*16)-1)      // round
#define BAUD_REAL                               (F_CPU/(16*(UBRR_VAL+1)))                   // real baudrate

#ifdef CODEVISION
#if ((BAUD_REAL*1000)/UART_BAUD-1000) > 10
#  error Error of baud rate of RS232 UARTx is more than 1%. That is too high!
#endif

#else // not CODEVISION

#define BAUD_ERROR                              ((BAUD_REAL*1000)/UART_BAUD-1000)           // error in promille

#if ((BAUD_ERROR > 10) || (-BAUD_ERROR < 10))
#  error Error of baud rate of RS232 UARTx is more than 1%. That is too high!
#endif

#endif // CODEVISION

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Initialize  UART
 *  @details  Initializes UART
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
irmp_uart_init (void)
{
    UCSR0B |= (1<<TXEN0);                                                         // activate UART0 TX
    UBRR0H = UBRR_VAL >> 8;                                                       // store baudrate (upper byte)
    UBRR0L = UBRR_VAL & 0xFF;                                                     // store baudrate (lower byte)
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Send character
 *  @details  Sends character
 *  @param    ch character to be transmitted
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
irmp_uart_putc (unsigned char ch)
{
    while (!(UCSR0A & (1<<UDRE0)))
    {
        ;
    }

    UDR0 = ch;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Log IR signal
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#define c_startcycles                     2                         // min count of zeros before start of logging
#define c_endBits                      1000                         // log buffer size
#define c_datalen                       700                         // number of sequenced highbits to detect end

static void
irmp_logIr (uint8_t val)
{
    static uint8_t  s_data[c_datalen];                                // logging buffer
    static uint16_t s_dataIdx;                                        // number of written bits
    static uint8_t  s_startcycles;                                    // current number of start-zeros
    static uint16_t s_ctr;                                            // counts sequenced highbits - to detect end

    if ((val == 0) && (s_startcycles < c_startcycles) && !s_dataIdx)  // prevent that single random zeros init logging
    {
        ++s_startcycles;
    }
    else
    {
        s_startcycles = 0;

        if (    (val == 0)                                              // start or continue logging on "0"
            || ((val == 1) && (s_dataIdx != 0)))                        // "1" cannot init logging
        {
            if (val)
            {                                                             // set or clear bit in bitarray
                s_data[(s_dataIdx / 8)] |=  (1<<(s_dataIdx % 8));
            }
            else
            {
                s_data[(s_dataIdx / 8)] &= ~(1<<(s_dataIdx % 8));
            }

            ++s_dataIdx;

            if (val)
            {                                                             // if high received then look at log-stop condition
                ++s_ctr;

                if (s_ctr > c_endBits)
                {                                                           // if stop condition (200 sequenced ones) meets, output on uart
                    uint16_t i;

                    for (i = 0; i < c_startcycles; ++i)
                    {
                        irmp_uart_putc ('0');                                   // the ignored starting zeros
                    }

                    for (i = 0;i < (s_dataIdx - c_endBits + 20) / 8; ++i)     // transform bitset into uart chars
                    {
                        uint8_t d = s_data[i];
                        uint8_t j;

                        for (j = 0;j<8;++j)
                        {
                            irmp_uart_putc ((d & 1) + '0');
                            d >>= 1;
                        }
                    }

                    irmp_uart_putc ('\n');
                    s_dataIdx = 0;
                }
            }
            else
            {
                s_ctr = 0;
            }
        }
    }
}

#else
#define irmp_logIsr()
#endif

static volatile uint8_t                     irmp_ir_detected;
static volatile uint8_t                     irmp_protocol;
static volatile uint16_t                    irmp_address;
static volatile uint16_t                    irmp_command;
static volatile uint16_t                    irmp_id;                                          // only used for SAMSUNG protocol
static volatile uint8_t                     irmp_flags;

#ifdef DEBUG
static uint8_t                              IRMP_PIN;
#endif

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Initialize IRMP decoder
 *  @details  Configures IRMP input pin
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
#ifndef DEBUG
void
irmp_init (void)
{
    IRMP_PORT &= ~(1<<IRMP_BIT);                                                                  // deactivate pullup
    IRMP_DDR &= ~(1<<IRMP_BIT);                                                                   // set pin to input

#if IRMP_LOGGING == 1
    irmp_uart_init ();
#endif
}
#endif
/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  Get IRMP data
 *  @details  gets decoded IRMP data
 *  @param    pointer in order to store IRMP data
 *  @return    TRUE: successful, FALSE: failed
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
uint8_t
irmp_get_data (IRMP_DATA * irmp_data_p)
{
    uint8_t   rtc = FALSE;

    if (irmp_ir_detected)
    {
        switch (irmp_protocol)
        {
#if IRMP_SUPPORT_SAMSUNG_PROTOCOL == 1
            case IRMP_SAMSUNG_PROTOCOL:
                if ((irmp_command >> 8) == (~irmp_command & 0x00FF))
                {
                    irmp_command &= 0xff;
                    irmp_command |= irmp_id << 8;
                    rtc = TRUE;
                }
                break;
#endif
#if IRMP_SUPPORT_NEC_PROTOCOL == 1
            case IRMP_NEC_PROTOCOL:
                if ((irmp_command >> 8) == (~irmp_command & 0x00FF))
                {
                    irmp_command &= 0xff;
                    rtc = TRUE;
                }
                else if ((irmp_command & 0xFF00) == 0xD100)
                {
                    DEBUG_PRINTF1 ("Switching to APPLE protocol\n");
                    irmp_protocol = IRMP_APPLE_PROTOCOL;
                    irmp_command &= 0xff;
                    rtc = TRUE;
                }
                break;
#endif
            default:
                rtc = TRUE;
        }

        if (rtc)
        {
            irmp_data_p->protocol = irmp_protocol;
            irmp_data_p->address = irmp_address;
            irmp_data_p->command = irmp_command;
            irmp_data_p->flags   = irmp_flags;
            irmp_command = 0;
            irmp_address = 0;
            irmp_flags   = 0;
        }

        irmp_ir_detected = FALSE;
    }

    return rtc;
}

// these statics must not be volatile, because they are only used by irmp_store_bit(), which is called by irmp_ISR()
static uint8_t    irmp_tmp_protocol;                                                        // ir protocol
static uint16_t   irmp_tmp_address;                                                         // ir address
static uint16_t   irmp_tmp_command;                                                         // ir command
#if IRMP_SUPPORT_SAMSUNG_PROTOCOL == 1
static uint16_t   irmp_tmp_id;                                                              // ir id (only SAMSUNG)
#endif
static uint8_t    address_offset;                                                           // address offset
static uint8_t    address_end;                                                              // end of address
static uint8_t    command_offset;                                                           // command offset
static uint8_t    command_end;                                                              // end of command
static uint8_t    irmp_bit;                                                                 // current bit position
static uint8_t    lsb_first;                                                                // flag: LSB first

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  store bit
 *  @details  store bit in temp address or temp command
 *  @param    value to store: 0 or 1
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
static void
irmp_store_bit (uint8_t value)
{
    if (irmp_bit >= address_offset && irmp_bit < address_end)
    {
        if (lsb_first)
        {
            irmp_tmp_address |= (((uint16_t) (value)) << (irmp_bit - address_offset));
        }
        else
        {
            irmp_tmp_address <<= 1;
            irmp_tmp_address |= value;
        }
    }
    else if (irmp_bit >= command_offset && irmp_bit < command_end)
    {
        if (lsb_first)
        {
            irmp_tmp_command |= (((uint16_t) (value)) << (irmp_bit - command_offset));
        }
        else
        {
            irmp_tmp_command <<= 1;
            irmp_tmp_command |= value;
        }
    }
#if IRMP_SUPPORT_SAMSUNG_PROTOCOL == 1
    else if (irmp_tmp_protocol == IRMP_SAMSUNG_PROTOCOL && irmp_bit >= SAMSUNG_ID_OFFSET && irmp_bit < SAMSUNG_ID_OFFSET + SAMSUNG_ID_LEN)
    {
        irmp_tmp_id |= (((uint16_t) (value)) << (irmp_bit - SAMSUNG_ID_OFFSET));              // assume LSB first
    }
#endif
    irmp_bit++;
}

/*---------------------------------------------------------------------------------------------------------------------------------------------------
 *  ISR routine
 *  @details  ISR routine, called 10000 times per second
 *---------------------------------------------------------------------------------------------------------------------------------------------------
 */
void
irmp_ISR (void)
{
    static uint8_t    irmp_start_bit_detected;                                  // flag: start bit detected
    static uint8_t    wait_for_space;                                           // flag: wait for data bit space
    static uint8_t    wait_for_start_space;                                     // flag: wait for start bit space
    static uint8_t    irmp_pulse_time;                                          // count bit time for pulse
    static uint8_t    irmp_pause_time;                                          // count bit time for pause
    static uint16_t   last_irmp_address;                                        // save last irmp address to recognize key repetition
    static uint16_t   last_irmp_command;                                        // save last irmp command to recognize key repetition
    static uint16_t   repetition_counter;                                       // SIRCS repeats frame 2-5 times with 45 ms pause
    static uint8_t    pulse_1_len_min;                                          // minimum length of pulse with bit value 1
    static uint8_t    pulse_1_len_max;                                          // maximum length of pulse with bit value 1
    static uint8_t    pause_1_len_min;                                          // minimum length of pause with bit value 1
    static uint8_t    pause_1_len_max;                                          // maximum length of pause with bit value 1
    static uint8_t    pulse_0_len_min;                                          // minimum length of pulse with bit value 0
    static uint8_t    pulse_0_len_max;                                          // maximum length of pulse with bit value 0
    static uint8_t    pause_0_len_min;                                          // minimum length of pause with bit value 0
    static uint8_t    pause_0_len_max;                                          // maximum length of pause with bit value 0
    static uint8_t    complete_len;                                             // complete length of frame
    static uint8_t    stop_bit;                                                 // flag: frame has stop bit
#if IRMP_SUPPORT_DENON_PROTOCOL == 1
    static uint16_t   last_irmp_denon_command;                                  // save last irmp command to recognize DENON frame repetition
#endif
#if IRMP_SUPPORT_RC5_PROTOCOL == 1
    static uint8_t    rc5_cmd_bit6;                                             // bit 6 of RC5 command is the inverted 2nd start bit
#endif
#if IRMP_SUPPORT_RC5_PROTOCOL == 1 || IRMP_SUPPORT_RC6_PROTOCOL == 1
    static uint8_t    rc5_last_pause;                                           // last pause value
    static uint8_t    rc5_last_value;                                           // last bit value
#endif

    irmp_logIsr();                                                              // log ir signal, if IRMP_LOGGING defined

    if (! irmp_ir_detected)                                                     // ir code already detected?
    {                                                                           // yes... wait for application to get data
        if (! irmp_start_bit_detected)                                          // start bit detected?
        {                                                                       // no...
            if (!(IRMP_PIN & (1 << IRMP_BIT)))                                  // receiving burst?
            {                                                                   // yes...
                irmp_pulse_time++;                                              // increment counter
            }
            else
            {                                                                   // yes...
                if (irmp_pulse_time)                                            // it's dark....
                {                                                               // set flags for counting the dark...
                    irmp_start_bit_detected = 1;
                    wait_for_start_space    = 1;
                    wait_for_space          = 0;
                    irmp_tmp_command        = 0;
                    irmp_tmp_address        = 0;
                    irmp_bit                = 0xff;
                    irmp_pause_time         = 1;                                // 1st pause: set to 1, not to 0!
#if IRMP_SUPPORT_RC5_PROTOCOL == 1
                    rc5_cmd_bit6            = 0;                                // fm 2010-03-07: bugfix: reset it after incomplete RC5 frame!
#endif
                }
                else
                {
                    repetition_counter++;
                }
            }
        }
        else
        {
            if (wait_for_start_space)                                           // we have received start bit and are counting the dark...
            {
                if (IRMP_PIN & (1 << IRMP_BIT))                                 // still dark?
                {                                                               // yes
                    irmp_pause_time++;                                          // increment counter

                    if (irmp_pause_time > IRMP_TIMEOUT)                         // timeout?
                    {                                                           // yes...
                        DEBUG_PRINTF3 ("error 1: pause after start bit %d too long: %d\n", irmp_pulse_time, irmp_pause_time);
                        irmp_start_bit_detected = 0;                            // reset flags, let's wait for another start bit
                        irmp_pulse_time         = 0;
                        irmp_pause_time         = 0;
                    }
                }
                else
                {                                                               // receiving first data pulse!
                    DEBUG_PRINTF3 ("start-bit: pulse = %d, pause = %d\n", irmp_pulse_time, irmp_pause_time);

#if IRMP_SUPPORT_SIRCS_PROTOCOL == 1
                    if (irmp_pulse_time >= SIRCS_START_BIT_PULSE_LEN_MIN && irmp_pulse_time <= SIRCS_START_BIT_PULSE_LEN_MAX &&
                        irmp_pause_time >= SIRCS_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= SIRCS_START_BIT_PAUSE_LEN_MAX)
                    {                                                                                   // it's SIRCS
                        DEBUG_PRINTF5 ("protocol = SIRCS, start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                        SIRCS_START_BIT_PULSE_LEN_MIN, SIRCS_START_BIT_PULSE_LEN_MAX,
                                        SIRCS_START_BIT_PAUSE_LEN_MIN, SIRCS_START_BIT_PAUSE_LEN_MAX);

                        irmp_tmp_protocol   = IRMP_SIRCS_PROTOCOL;
                        pulse_1_len_min     = SIRCS_1_PULSE_LEN_MIN;
                        pulse_1_len_max     = SIRCS_1_PULSE_LEN_MAX;
                        pause_1_len_min     = SIRCS_PAUSE_LEN_MIN;
                        pause_1_len_max     = SIRCS_PAUSE_LEN_MAX;
                        pulse_0_len_min     = SIRCS_0_PULSE_LEN_MIN;
                        pulse_0_len_max     = SIRCS_0_PULSE_LEN_MAX;
                        pause_0_len_min     = SIRCS_PAUSE_LEN_MIN;
                        pause_0_len_max     = SIRCS_PAUSE_LEN_MAX;
                        address_offset      = SIRCS_ADDRESS_OFFSET;
                        address_end         = SIRCS_ADDRESS_OFFSET + SIRCS_ADDRESS_LEN;
                        command_offset      = SIRCS_COMMAND_OFFSET;
                        command_end         = SIRCS_COMMAND_OFFSET + SIRCS_COMMAND_LEN;
                        complete_len        = SIRCS_COMPLETE_DATA_LEN;
                        stop_bit            = SIRCS_STOP_BIT;
                        lsb_first           = SIRCS_LSB;
                    }
                    else
#endif

#if IRMP_SUPPORT_NEC_PROTOCOL == 1
                    if (irmp_pulse_time >= NEC_START_BIT_PULSE_LEN_MIN && irmp_pulse_time <= NEC_START_BIT_PULSE_LEN_MAX &&
                        ((irmp_pause_time >= NEC_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= NEC_START_BIT_PAUSE_LEN_MAX) ||
                         (irmp_pause_time >= NEC_REPEAT_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= NEC_REPEAT_START_BIT_PAUSE_LEN_MAX)))
                    {                                                                 // it's NEC
                        if (irmp_pause_time <= NEC_REPEAT_START_BIT_PAUSE_LEN_MAX)
                        {
                            DEBUG_PRINTF5 ("protocol = NEC (repetition frame), start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                            NEC_START_BIT_PULSE_LEN_MIN, NEC_START_BIT_PULSE_LEN_MAX,
                                            NEC_REPEAT_START_BIT_PAUSE_LEN_MIN, NEC_REPEAT_START_BIT_PAUSE_LEN_MAX);
                        }
                        else
                        {
                            DEBUG_PRINTF5 ("protocol = NEC, start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                            NEC_START_BIT_PULSE_LEN_MIN, NEC_START_BIT_PULSE_LEN_MAX,
                                            NEC_START_BIT_PAUSE_LEN_MIN, NEC_START_BIT_PAUSE_LEN_MAX);
                        }

                        irmp_tmp_protocol   = IRMP_NEC_PROTOCOL;
                        pulse_1_len_min     = NEC_PULSE_LEN_MIN;
                        pulse_1_len_max     = NEC_PULSE_LEN_MAX;
                        pause_1_len_min     = NEC_1_PAUSE_LEN_MIN;
                        pause_1_len_max     = NEC_1_PAUSE_LEN_MAX;
                        pulse_0_len_min     = NEC_PULSE_LEN_MIN;
                        pulse_0_len_max     = NEC_PULSE_LEN_MAX;
                        pause_0_len_min     = NEC_0_PAUSE_LEN_MIN;
                        pause_0_len_max     = NEC_0_PAUSE_LEN_MAX;

                        if (irmp_pause_time <= NEC_REPEAT_START_BIT_PAUSE_LEN_MAX)
                        {
                            address_offset    = 0;
                            address_end       = 0;
                            command_offset    = 0;
                            command_end       = 0;
                            complete_len      = 0;
                        }
                        else
                        {
                            address_offset    = NEC_ADDRESS_OFFSET;
                            address_end       = NEC_ADDRESS_OFFSET + NEC_ADDRESS_LEN;
                            command_offset    = NEC_COMMAND_OFFSET;
                            command_end       = NEC_COMMAND_OFFSET + NEC_COMMAND_LEN;
                            complete_len      = NEC_COMPLETE_DATA_LEN;
                        }
                        stop_bit            = NEC_STOP_BIT;
                        lsb_first           = NEC_LSB;
                    }
                    else
#endif
#if IRMP_SUPPORT_SAMSUNG_PROTOCOL == 1
                    if (irmp_pulse_time >= SAMSUNG_START_BIT_PULSE_LEN_MIN && irmp_pulse_time <= SAMSUNG_START_BIT_PULSE_LEN_MAX &&
                        irmp_pause_time >= SAMSUNG_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= SAMSUNG_START_BIT_PAUSE_LEN_MAX)
                    {                                                                 // it's SAMSUNG
                        DEBUG_PRINTF5 ("protocol = SAMSUNG, start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                        SAMSUNG_START_BIT_PULSE_LEN_MIN, SAMSUNG_START_BIT_PULSE_LEN_MAX,
                                        SAMSUNG_START_BIT_PAUSE_LEN_MIN, SAMSUNG_START_BIT_PAUSE_LEN_MAX);

                        irmp_tmp_protocol   = IRMP_SAMSUNG_PROTOCOL;
                        pulse_1_len_min     = SAMSUNG_PULSE_LEN_MIN;
                        pulse_1_len_max     = SAMSUNG_PULSE_LEN_MAX;
                        pause_1_len_min     = SAMSUNG_1_PAUSE_LEN_MIN;
                        pause_1_len_max     = SAMSUNG_1_PAUSE_LEN_MAX;
                        pulse_0_len_min     = SAMSUNG_PULSE_LEN_MIN;
                        pulse_0_len_max     = SAMSUNG_PULSE_LEN_MAX;
                        pause_0_len_min     = SAMSUNG_0_PAUSE_LEN_MIN;
                        pause_0_len_max     = SAMSUNG_0_PAUSE_LEN_MAX;
                        address_offset      = SAMSUNG_ADDRESS_OFFSET;
                        address_end         = SAMSUNG_ADDRESS_OFFSET + SAMSUNG_ADDRESS_LEN;
                        command_offset      = SAMSUNG_COMMAND_OFFSET;
                        command_end         = SAMSUNG_COMMAND_OFFSET + SAMSUNG_COMMAND_LEN;
                        complete_len        = SAMSUNG_COMPLETE_DATA_LEN;
                        stop_bit            = SAMSUNG_STOP_BIT;
                        lsb_first           = SAMSUNG_LSB;
                    }
                    else
#endif
#if IRMP_SUPPORT_MATSUSHITA_PROTOCOL == 1
                    if (irmp_pulse_time >= MATSUSHITA_START_BIT_PULSE_LEN_MIN && irmp_pulse_time <= MATSUSHITA_START_BIT_PULSE_LEN_MAX &&
                        irmp_pause_time >= MATSUSHITA_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= MATSUSHITA_START_BIT_PAUSE_LEN_MAX)
                    {                                                                 // it's MATSUSHITA
                        DEBUG_PRINTF5 ("protocol = MATSUSHITA, start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                        MATSUSHITA_START_BIT_PULSE_LEN_MIN, MATSUSHITA_START_BIT_PULSE_LEN_MAX,
                                        MATSUSHITA_START_BIT_PAUSE_LEN_MIN, MATSUSHITA_START_BIT_PAUSE_LEN_MAX);

                        irmp_tmp_protocol   = IRMP_MATSUSHITA_PROTOCOL;
                        pulse_1_len_min     = MATSUSHITA_PULSE_LEN_MIN;
                        pulse_1_len_max     = MATSUSHITA_PULSE_LEN_MAX;
                        pause_1_len_min     = MATSUSHITA_1_PAUSE_LEN_MIN;
                        pause_1_len_max     = MATSUSHITA_1_PAUSE_LEN_MAX;
                        pulse_0_len_min     = MATSUSHITA_PULSE_LEN_MIN;
                        pulse_0_len_max     = MATSUSHITA_PULSE_LEN_MAX;
                        pause_0_len_min     = MATSUSHITA_0_PAUSE_LEN_MIN;
                        pause_0_len_max     = MATSUSHITA_0_PAUSE_LEN_MAX;
                        address_offset      = MATSUSHITA_ADDRESS_OFFSET;
                        address_end         = MATSUSHITA_ADDRESS_OFFSET + MATSUSHITA_ADDRESS_LEN;
                        command_offset      = MATSUSHITA_COMMAND_OFFSET;
                        command_end         = MATSUSHITA_COMMAND_OFFSET + MATSUSHITA_COMMAND_LEN;
                        complete_len        = MATSUSHITA_COMPLETE_DATA_LEN;
                        stop_bit            = MATSUSHITA_STOP_BIT;
                        lsb_first           = MATSUSHITA_LSB;
                    }
                    else
#endif
#if IRMP_SUPPORT_KASEIKYO_PROTOCOL == 1
                    if (irmp_pulse_time >= KASEIKYO_START_BIT_PULSE_LEN_MIN && irmp_pulse_time <= KASEIKYO_START_BIT_PULSE_LEN_MAX &&
                        irmp_pause_time >= KASEIKYO_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= KASEIKYO_START_BIT_PAUSE_LEN_MAX)
                    {                                                                 // it's KASEIKYO
                        DEBUG_PRINTF5 ("protocol = KASEIKYO, start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                        KASEIKYO_START_BIT_PULSE_LEN_MIN, KASEIKYO_START_BIT_PULSE_LEN_MAX,
                                        KASEIKYO_START_BIT_PAUSE_LEN_MIN, KASEIKYO_START_BIT_PAUSE_LEN_MAX);

                        irmp_tmp_protocol   = IRMP_KASEIKYO_PROTOCOL;
                        pulse_1_len_min     = KASEIKYO_PULSE_LEN_MIN;
                        pulse_1_len_max     = KASEIKYO_PULSE_LEN_MAX;
                        pause_1_len_min     = KASEIKYO_1_PAUSE_LEN_MIN;
                        pause_1_len_max     = KASEIKYO_1_PAUSE_LEN_MAX;
                        pulse_0_len_min     = KASEIKYO_PULSE_LEN_MIN;
                        pulse_0_len_max     = KASEIKYO_PULSE_LEN_MAX;
                        pause_0_len_min     = KASEIKYO_0_PAUSE_LEN_MIN;
                        pause_0_len_max     = KASEIKYO_0_PAUSE_LEN_MAX;
                        address_offset      = KASEIKYO_ADDRESS_OFFSET;
                        address_end         = KASEIKYO_ADDRESS_OFFSET + KASEIKYO_ADDRESS_LEN;
                        command_offset      = KASEIKYO_COMMAND_OFFSET;
                        command_end         = KASEIKYO_COMMAND_OFFSET + KASEIKYO_COMMAND_LEN;
                        complete_len        = KASEIKYO_COMPLETE_DATA_LEN;
                        stop_bit            = KASEIKYO_STOP_BIT;
                        lsb_first           = KASEIKYO_LSB;
                    }
                    else
#endif
#if IRMP_SUPPORT_RECS80_PROTOCOL == 1
                    if (irmp_pulse_time >= RECS80_START_BIT_PULSE_LEN_MIN && irmp_pulse_time <= RECS80_START_BIT_PULSE_LEN_MAX &&
                        irmp_pause_time >= RECS80_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= RECS80_START_BIT_PAUSE_LEN_MAX)
                    {                                                                 // it's RECS80
                        DEBUG_PRINTF5 ("protocol = RECS80, start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                        RECS80_START_BIT_PULSE_LEN_MIN, RECS80_START_BIT_PULSE_LEN_MAX,
                                        RECS80_START_BIT_PAUSE_LEN_MIN, RECS80_START_BIT_PAUSE_LEN_MAX);

                        irmp_tmp_protocol   = IRMP_RECS80_PROTOCOL;
                        pulse_1_len_min     = RECS80_PULSE_LEN_MIN;
                        pulse_1_len_max     = RECS80_PULSE_LEN_MAX;
                        pause_1_len_min     = RECS80_1_PAUSE_LEN_MIN;
                        pause_1_len_max     = RECS80_1_PAUSE_LEN_MAX;
                        pulse_0_len_min     = RECS80_PULSE_LEN_MIN;
                        pulse_0_len_max     = RECS80_PULSE_LEN_MAX;
                        pause_0_len_min     = RECS80_0_PAUSE_LEN_MIN;
                        pause_0_len_max     = RECS80_0_PAUSE_LEN_MAX;
                        address_offset      = RECS80_ADDRESS_OFFSET;
                        address_end         = RECS80_ADDRESS_OFFSET + RECS80_ADDRESS_LEN;
                        command_offset      = RECS80_COMMAND_OFFSET;
                        command_end         = RECS80_COMMAND_OFFSET + RECS80_COMMAND_LEN;
                        complete_len        = RECS80_COMPLETE_DATA_LEN;
                        stop_bit            = RECS80_STOP_BIT;
                        lsb_first           = RECS80_LSB;
                    }
                    else
#endif

#if IRMP_SUPPORT_RC5_PROTOCOL == 1
                    if (((irmp_pulse_time >= RC5_START_BIT_LEN_MIN && irmp_pulse_time <= RC5_START_BIT_LEN_MAX) ||
                         (irmp_pulse_time >= 2 * RC5_START_BIT_LEN_MIN && irmp_pulse_time <= 2 * RC5_START_BIT_LEN_MAX)) &&
                        ((irmp_pause_time >= RC5_START_BIT_LEN_MIN && irmp_pause_time <= RC5_START_BIT_LEN_MAX) ||
                         (irmp_pause_time >= 2 * RC5_START_BIT_LEN_MIN && irmp_pause_time <= 2 * RC5_START_BIT_LEN_MAX)))
                    {                                                                 // it's RC5
                        DEBUG_PRINTF5 ("protocol = RC5, start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                        RC5_START_BIT_LEN_MIN, RC5_START_BIT_LEN_MAX,
                                        RC5_START_BIT_LEN_MIN, RC5_START_BIT_LEN_MAX);

                        irmp_tmp_protocol   = IRMP_RC5_PROTOCOL;
                        pulse_1_len_min     = RC5_BIT_LEN_MIN;
                        pulse_1_len_max     = RC5_BIT_LEN_MAX;
                        pause_1_len_min     = RC5_BIT_LEN_MIN;
                        pause_1_len_max     = RC5_BIT_LEN_MAX;
                        pulse_0_len_min     = 1;                                // tricky: use this as stop bit length
                        pulse_0_len_max     = 1;
                        pause_0_len_min     = 1;
                        pause_0_len_max     = 1;
                        address_offset      = RC5_ADDRESS_OFFSET;
                        address_end         = RC5_ADDRESS_OFFSET + RC5_ADDRESS_LEN;
                        command_offset      = RC5_COMMAND_OFFSET;
                        command_end         = RC5_COMMAND_OFFSET + RC5_COMMAND_LEN;
                        complete_len        = RC5_COMPLETE_DATA_LEN;
                        stop_bit            = RC5_STOP_BIT;
                        lsb_first           = RC5_LSB;

                        rc5_last_pause      = irmp_pause_time;

                        if ((irmp_pulse_time > RC5_START_BIT_LEN_MAX && irmp_pulse_time <= 2 * RC5_START_BIT_LEN_MAX) ||
                            (irmp_pause_time > RC5_START_BIT_LEN_MAX && irmp_pause_time <= 2 * RC5_START_BIT_LEN_MAX))
                        {
                          rc5_last_value  = 0;
                          rc5_cmd_bit6 = 1<<6;
                        }
                        else
                        {
                          rc5_last_value  = 1;
                        }
                    }
                    else
#endif

#if IRMP_SUPPORT_DENON_PROTOCOL == 1
                    if ( (irmp_pulse_time >= DENON_PULSE_LEN_MIN && irmp_pulse_time <= DENON_PULSE_LEN_MAX) &&
                        ((irmp_pause_time >= DENON_1_PAUSE_LEN_MIN && irmp_pause_time <= DENON_1_PAUSE_LEN_MAX) ||
                         (irmp_pause_time >= DENON_0_PAUSE_LEN_MIN && irmp_pause_time <= DENON_0_PAUSE_LEN_MAX)))
                    {                                                           // it's DENON
                        DEBUG_PRINTF7 ("protocol = DENON, start bit timings: pulse: %2d - %2d, pause: %2d - %2d or %2d - %2d\n",
                                        DENON_PULSE_LEN_MIN, DENON_PULSE_LEN_MAX,
                                        DENON_1_PAUSE_LEN_MIN, DENON_1_PAUSE_LEN_MAX,
                                        DENON_0_PAUSE_LEN_MIN, DENON_0_PAUSE_LEN_MAX);

                        irmp_tmp_protocol   = IRMP_DENON_PROTOCOL;
                        pulse_1_len_min     = DENON_PULSE_LEN_MIN;
                        pulse_1_len_max     = DENON_PULSE_LEN_MAX;
                        pause_1_len_min     = DENON_1_PAUSE_LEN_MIN;
                        pause_1_len_max     = DENON_1_PAUSE_LEN_MAX;
                        pulse_0_len_min     = DENON_PULSE_LEN_MIN;
                        pulse_0_len_max     = DENON_PULSE_LEN_MAX;
                        pause_0_len_min     = DENON_0_PAUSE_LEN_MIN;
                        pause_0_len_max     = DENON_0_PAUSE_LEN_MAX;
                        address_offset      = DENON_ADDRESS_OFFSET;
                        address_end         = DENON_ADDRESS_OFFSET + DENON_ADDRESS_LEN;
                        command_offset      = DENON_COMMAND_OFFSET;
                        command_end         = DENON_COMMAND_OFFSET + DENON_COMMAND_LEN;
                        complete_len        = DENON_COMPLETE_DATA_LEN;
                        stop_bit            = DENON_STOP_BIT;
                        lsb_first           = DENON_LSB;
                    }
                    else
#endif
#if IRMP_SUPPORT_RC6_PROTOCOL == 1
                    if (irmp_pulse_time >= RC6_START_BIT_PULSE_LEN_MIN && irmp_pulse_time <= RC6_START_BIT_PULSE_LEN_MAX &&
                        irmp_pause_time >= RC6_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= RC6_START_BIT_PAUSE_LEN_MAX)
                    {                                                           // it's RC6
                        DEBUG_PRINTF5 ("protocol = RC6, start bit timings: pulse: %2d - %2d, pause: %2d - %2d\n",
                                        RC6_START_BIT_PULSE_LEN_MIN, RC6_START_BIT_PULSE_LEN_MAX,
                                        RC6_START_BIT_PAUSE_LEN_MIN, RC6_START_BIT_PAUSE_LEN_MAX);

                        irmp_tmp_protocol   = IRMP_RC6_PROTOCOL;
                        pulse_1_len_min     = RC6_BIT_LEN_MIN;
                        pulse_1_len_max     = RC6_BIT_LEN_MAX;
                        pause_1_len_min     = RC6_BIT_LEN_MIN;
                        pause_1_len_max     = RC6_BIT_LEN_MAX;
                        pulse_0_len_min     = 1;                                // tricky: use this as stop bit length
                        pulse_0_len_max     = 1;
                        pause_0_len_min     = 1;
                        pause_0_len_max     = 1;
                        address_offset      = RC6_ADDRESS_OFFSET;
                        address_end         = RC6_ADDRESS_OFFSET + RC6_ADDRESS_LEN;
                        command_offset      = RC6_COMMAND_OFFSET;
                        command_end         = RC6_COMMAND_OFFSET + RC6_COMMAND_LEN;
                        complete_len        = RC6_COMPLETE_DATA_LEN_SHORT;
                        stop_bit            = RC6_STOP_BIT;
                        lsb_first           = RC6_LSB;

                        rc5_last_pause      = 0;
                        rc5_last_value      = 0;
                    }
                    else
#endif
                    {
                        DEBUG_PRINTF1 ("protocol = UNKNOWN\n");
                        irmp_start_bit_detected = 0;                            // wait for another start bit...
                    }

                    if (irmp_start_bit_detected)
                    {
                        DEBUG_PRINTF3 ("pulse_1        = %2d - %2d\n", pulse_1_len_min, pulse_1_len_max);
                        DEBUG_PRINTF3 ("pause_1        = %2d - %2d\n", pause_1_len_min, pause_1_len_max);
                        if (irmp_tmp_protocol == IRMP_RC6_PROTOCOL)
                        {
                            DEBUG_PRINTF3 ("pulse_toggle   = %2d - %2d\n", RC6_TOGGLE_BIT_LEN_MIN, RC6_TOGGLE_BIT_LEN_MAX);
                        }
                        DEBUG_PRINTF3 ("pulse_0        = %2d - %2d\n", pulse_0_len_min, pulse_0_len_max);
                        DEBUG_PRINTF3 ("pause_0        = %2d - %2d\n", pause_0_len_min, pause_0_len_max);
                        DEBUG_PRINTF2 ("command_offset = %d\n", command_offset);
                        DEBUG_PRINTF2 ("command_len    = %d\n", command_end - command_offset);
                        DEBUG_PRINTF2 ("complete_len   = %d\n", complete_len);
                        DEBUG_PRINTF2 ("stop_bit       = %d\n", stop_bit);
                    }

                    irmp_bit = 0;

#if IRMP_SUPPORT_RC5_PROTOCOL == 1
                    if (irmp_tmp_protocol == IRMP_RC5_PROTOCOL)
                    {
                        if (irmp_pause_time > RC5_START_BIT_LEN_MAX && irmp_pause_time <= 2 * RC5_START_BIT_LEN_MAX)
                        {
                          DEBUG_PRINTF4 ("[bit %2d: pulse = %2d, pause = %2d] ", irmp_bit, irmp_pulse_time, irmp_pause_time);
                          DEBUG_PUTCHAR ('1');
                          DEBUG_PUTCHAR ('\n');
                          irmp_store_bit (1);
                        }
                        else if (! rc5_last_value)
                        {
                          DEBUG_PRINTF4 ("[bit %2d: pulse = %2d, pause = %2d] ", irmp_bit, irmp_pulse_time, irmp_pause_time);
                          DEBUG_PUTCHAR ('0');
                          DEBUG_PUTCHAR ('\n');
                          irmp_store_bit (0);
                        }
                    }
                    else
#endif // IRMP_SUPPORT_RC5_PROTOCOL == 1

#if IRMP_SUPPORT_DENON_PROTOCOL == 1
                    if (irmp_tmp_protocol == IRMP_DENON_PROTOCOL)
                    {
                        DEBUG_PRINTF4 ("[bit %2d: pulse = %2d, pause = %2d] ", irmp_bit, irmp_pulse_time, irmp_pause_time);

                        if (irmp_pause_time >= DENON_1_PAUSE_LEN_MIN && irmp_pause_time <= DENON_1_PAUSE_LEN_MAX)
                        {                                                       // pause timings correct for "1"?
                          DEBUG_PUTCHAR ('1');                                  // yes, store 1
                          DEBUG_PUTCHAR ('\n');
                          irmp_store_bit (1);
                        }
                        else // if (irmp_pause_time >= DENON_0_PAUSE_LEN_MIN && irmp_pause_time <= DENON_0_PAUSE_LEN_MAX)
                        {                                                       // pause timings correct for "0"?
                          DEBUG_PUTCHAR ('0');                                  // yes, store 0
                          DEBUG_PUTCHAR ('\n');
                          irmp_store_bit (0);
                        }
                    }
#endif // IRMP_SUPPORT_DENON_PROTOCOL == 1

                    irmp_pulse_time = 1;                                        // set counter to 1, not 0
                    irmp_pause_time = 0;
                    wait_for_start_space = 0;
                }
            }
            else if (wait_for_space)                                            // the data section....
            {                                                                   // counting the dark....
                uint8_t got_light = FALSE;

                if (IRMP_PIN & (1 << IRMP_BIT))                                 // still dark?
                {                                                               // yes...
                    if (irmp_bit == complete_len && stop_bit == 1)
                    {
                        if (irmp_pulse_time >= pulse_0_len_min && irmp_pulse_time <= pulse_0_len_max)
                        {
#ifdef DEBUG
                            if (irmp_tmp_protocol != IRMP_RC5_PROTOCOL)
                            {
                                DEBUG_PRINTF1 ("stop bit detected\n");
                            }
#endif
                            stop_bit = 0;
                        }
                        else
                        {
                            DEBUG_PRINTF1 ("stop bit timing wrong\n");

                            irmp_start_bit_detected = 0;                        // wait for another start bit...
                            irmp_pulse_time         = 0;
                            irmp_pause_time         = 0;
                        }
                    }
                    else
                    {
                        irmp_pause_time++;                                      // increment counter

#if IRMP_SUPPORT_SIRCS_PROTOCOL == 1
                        if (irmp_tmp_protocol == IRMP_SIRCS_PROTOCOL &&         // Sony has a variable number of bits:
                            irmp_pause_time > SIRCS_PAUSE_LEN_MAX &&            // minimum is 12
                            irmp_bit >= 12 - 1)                                 // pause too long?
                        {                                                       // yes, break and close this frame
                            complete_len = irmp_bit + 1;                        // set new complete length
                            got_light = TRUE;                                   // this is a lie, but helps (generates stop bit)
                            command_end = command_offset + irmp_bit + 1;        // correct command length
                            irmp_pause_time = SIRCS_PAUSE_LEN_MAX - 1;          // correct pause length
                        }
                        else
#endif
#if IRMP_SUPPORT_RC5_PROTOCOL == 1
                        if (irmp_tmp_protocol == IRMP_RC5_PROTOCOL &&
                            irmp_pause_time > 2 * RC5_BIT_LEN_MAX && irmp_bit >= RC5_COMPLETE_DATA_LEN - 2 && !stop_bit)
                        {                                                       // special rc5 decoder
                            got_light = TRUE;                                   // this is a lie, but generates a stop bit ;-)
                            stop_bit = TRUE;                                    // set flag
                        }
                        else
#endif
#if IRMP_SUPPORT_RC6_PROTOCOL == 1
                        if (irmp_tmp_protocol == IRMP_RC6_PROTOCOL &&
                            irmp_pause_time > 2 * RC6_BIT_LEN_MAX && irmp_bit >= complete_len - 2 && !stop_bit)
                        {                                                       // special rc6 decoder
                            got_light = TRUE;                                   // this is a lie, but generates a stop bit ;-)
                            stop_bit = TRUE;                                    // set flag
                        }
                        else
#endif
                        if (irmp_pause_time > IRMP_TIMEOUT)                     // timeout?
                        {                                                       // yes...
                            if (irmp_bit == complete_len - 1 && stop_bit == 0)
                            {
                                irmp_bit++;
                            }
                            else
                            {
                                DEBUG_PRINTF3 ("error 2: pause %d after data bit %d too long\n", irmp_pause_time, irmp_bit);

                                irmp_start_bit_detected = 0;                    // wait for another start bit...
                                irmp_pulse_time         = 0;
                                irmp_pause_time         = 0;
                            }
                        }
                    }
                }
                else
                {                                                               // got light now!
                    got_light = TRUE;
                }

                if (got_light)
                {
                    DEBUG_PRINTF4 ("[bit %2d: pulse = %2d, pause = %2d] ", irmp_bit, irmp_pulse_time, irmp_pause_time);

#if IRMP_SUPPORT_RC5_PROTOCOL == 1
                    if (irmp_tmp_protocol == IRMP_RC5_PROTOCOL)                 // special rc5 decoder
                    {
                        if (irmp_pulse_time > RC5_BIT_LEN_MAX && irmp_pulse_time <= 2 * RC5_BIT_LEN_MAX)
                        {
                            DEBUG_PUTCHAR ('1');
                            irmp_store_bit (1);
                            DEBUG_PUTCHAR ('0');
                            DEBUG_PUTCHAR ('\n');
                            irmp_store_bit (0);
                            rc5_last_value = 0;
                        }

                        else // if (irmp_pulse_time >= RC5_BIT_LEN_MIN && irmp_pulse_time <= RC5_BIT_LEN_MAX)
                        {
                            uint8_t rc5_value;

                            if (rc5_last_pause > RC5_BIT_LEN_MAX && rc5_last_pause <= 2 * RC5_BIT_LEN_MAX)
                            {
                                rc5_value = rc5_last_value ? 0 : 1;
                                rc5_last_value  = rc5_value;
                            }
                            else
                            {
                                rc5_value = rc5_last_value;
                            }

                            DEBUG_PUTCHAR (rc5_value + '0');
                            DEBUG_PUTCHAR ('\n');
                            irmp_store_bit (rc5_value);
                        }

                        rc5_last_pause = irmp_pause_time;
                        wait_for_space = 0;
                    }
                    else
#endif

#if IRMP_SUPPORT_RC6_PROTOCOL == 1
                    if (irmp_tmp_protocol == IRMP_RC6_PROTOCOL)                 // special rc6 decoder
                    {
                        switch (irmp_bit)
                        {                                                       // handle toggle bit, which is 2 times longer than other bits
                            case 3:
                            case 4:
                            case 5:
                                if (irmp_pulse_time > RC6_TOGGLE_BIT_LEN_MAX && irmp_pause_time > RC6_TOGGLE_BIT_LEN_MAX)
                                {
                                  DEBUG_PUTCHAR ('1');
                                  irmp_store_bit (1);
                                }
                                DEBUG_PUTCHAR ('0');
                                irmp_store_bit (0);
                                rc5_last_value = 0;
                                DEBUG_PUTCHAR ('\n');
                                break;
                            default:
                                if (irmp_pulse_time > RC6_BIT_LEN_MAX && irmp_pulse_time <= 2 * RC6_BIT_LEN_MAX)
                                {
                                  DEBUG_PUTCHAR ('0');
                                  irmp_store_bit (0);
                                  DEBUG_PUTCHAR ('1');
                                  DEBUG_PUTCHAR ('\n');
                                  irmp_store_bit (1);
                                  rc5_last_value = 1;
                                }
                                else // if (irmp_pulse_time >= RC6_BIT_LEN_MIN && irmp_pulse_time <= RC6_BIT_LEN_MAX)
                                {
                                  uint8_t rc5_value;

                                  if (rc5_last_pause > RC6_BIT_LEN_MAX && rc5_last_pause <= 2 * RC6_BIT_LEN_MAX)
                                  {
                                    rc5_value = rc5_last_value ? 0 : 1;
                                    rc5_last_value  = rc5_value;
                                  }
                                  else
                                  {
                                    rc5_value = rc5_last_value;
                                  }

                                  if (irmp_bit == 1 && rc5_value == 0)
                                  {
                                    complete_len = RC6_COMPLETE_DATA_LEN_LONG;
                                  }

                                  DEBUG_PUTCHAR (rc5_value + '0');
                                  DEBUG_PUTCHAR ('\n');
                                  irmp_store_bit (rc5_value);
                                }

                                rc5_last_pause = irmp_pause_time;
                                break;
                        } // switch

                        wait_for_space = 0;
                    }
                    else
#endif

#if IRMP_SUPPORT_SAMSUNG_PROTOCOL == 1
                    if (irmp_tmp_protocol == IRMP_SAMSUNG_PROTOCOL && irmp_bit == 16)       // Samsung: 16th bit
                    {
                        if (irmp_pulse_time >= SAMSUNG_PULSE_LEN_MIN && irmp_pulse_time <= SAMSUNG_PULSE_LEN_MAX &&
                            irmp_pause_time >= SAMSUNG_START_BIT_PAUSE_LEN_MIN && irmp_pause_time <= SAMSUNG_START_BIT_PAUSE_LEN_MAX)
                        {
                          DEBUG_PRINTF1 ("SYNC\n");
                          wait_for_space = 0;
                          irmp_tmp_id = 0;
                          irmp_bit++;
                        }
                        else  if (irmp_pulse_time >= SAMSUNG_PULSE_LEN_MIN && irmp_pulse_time <= SAMSUNG_PULSE_LEN_MAX)
                        {
                            if (irmp_pause_time >= SAMSUNG_1_PAUSE_LEN_MIN && irmp_pause_time <= SAMSUNG_1_PAUSE_LEN_MAX)
                            {
                              DEBUG_PUTCHAR ('1');
                              DEBUG_PUTCHAR ('\n');
                              irmp_store_bit (1);
                              wait_for_space = 0;
                            }
                            else
                            {
                              DEBUG_PUTCHAR ('0');
                              DEBUG_PUTCHAR ('\n');
                              irmp_store_bit (0);
                              wait_for_space = 0;
                            }

                            DEBUG_PRINTF1 ("Switching to SAMSUNG32 protocol\n");

                            irmp_tmp_protocol   = IRMP_SAMSUNG32_PROTOCOL;
                            command_offset      = SAMSUNG32_COMMAND_OFFSET;
                            command_end         = SAMSUNG32_COMMAND_OFFSET + SAMSUNG32_COMMAND_LEN;
                            complete_len        = SAMSUNG32_COMPLETE_DATA_LEN;
                        }
                        else
                        {                                                       // timing incorrect!
                          DEBUG_PRINTF4 ("error 3: timing not correct: data bit %d,  pulse: %d, pause: %d\n", irmp_bit, irmp_pulse_time, irmp_pause_time);
                          irmp_start_bit_detected = 0;                          // reset flags and wait for next start bit
                          irmp_pause_time         = 0;
                        }

                        irmp_pulse_time = 1;                                    // set counter to 1, not 0
                    }
                    else
#endif // IRMP_SUPPORT_SAMSUNG_PROTOCOL

                    if (irmp_pulse_time >= pulse_1_len_min && irmp_pulse_time <= pulse_1_len_max &&
                        irmp_pause_time >= pause_1_len_min && irmp_pause_time <= pause_1_len_max)   // pulse & pause timings correct for "1"?
                    {                                                                               // yes...
                        DEBUG_PUTCHAR ('1');
                        DEBUG_PUTCHAR ('\n');
                        irmp_store_bit (1);
                        wait_for_space = 0;
                    }
                    else if (irmp_pulse_time >= pulse_0_len_min && irmp_pulse_time <= pulse_0_len_max &&
                             irmp_pause_time >= pause_0_len_min && irmp_pause_time <= pause_0_len_max)    // pulse & pause timings correct for "0"?
                    {                                                                                     // yes, store a "0"
                        DEBUG_PUTCHAR ('0');
                        DEBUG_PUTCHAR ('\n');
                        irmp_store_bit (0);
                        wait_for_space = 0;
                    }
                    else
                    {                                                           // timing incorrect!
                        DEBUG_PRINTF4 ("error 3: timing not correct: data bit %d,  pulse: %d, pause: %d\n", irmp_bit, irmp_pulse_time, irmp_pause_time);
                        irmp_start_bit_detected = 0;                            // reset flags and wait for next start bit
                        irmp_pause_time         = 0;
                    }

                    irmp_pulse_time = 1;                                        // set counter to 1, not 0
                }
            }
            else
            {                                                                   // counting the light (pulse)...
                if (!(IRMP_PIN & (1 << IRMP_BIT)))                              // still light?
                {                                                               // yes...
                    irmp_pulse_time++;                                          // increment counter
                }
                else
                {                                                               // now it's dark!
                    wait_for_space  = 1;                                        // let's count it (see above)
                    irmp_pause_time = 1;                                        // set pause counter to 1, not 0
                }
            }

            if (irmp_bit == complete_len && stop_bit == 0)                      // enough bits received?
            {
                // if SIRCS/SAMSUNG32 protocol and the code will be repeated inside of 50 ms, we will ignore it.
                if ((irmp_tmp_protocol == IRMP_SIRCS_PROTOCOL || irmp_tmp_protocol == IRMP_SAMSUNG32_PROTOCOL) &&
                    last_irmp_command == irmp_tmp_command && repetition_counter < AUTO_REPETITION_LEN)
                {
                    DEBUG_PRINTF2 ("code skipped, recognized SIRCS or SAMSUNG32 repetition, counter = %d\n", repetition_counter);
                    repetition_counter = 0;
                }
                else
                {
                    DEBUG_PRINTF2 ("code detected, length = %d\n", irmp_bit);
                    irmp_ir_detected = TRUE;

#if IRMP_SUPPORT_DENON_PROTOCOL == 1
                    if (irmp_tmp_protocol == IRMP_DENON_PROTOCOL)
                    {                                                               // check for repetition frame
                        if ((~irmp_tmp_command & 0x3FF) == last_irmp_denon_command) // command bits must be inverted
                        {
                            irmp_tmp_command = last_irmp_denon_command;             // use command received before!

                            irmp_protocol = irmp_tmp_protocol;                      // store protocol
                            irmp_address = irmp_tmp_address;                        // store address
                            irmp_command = irmp_tmp_command ;                       // store command
                        }
                        else
                        {
                            DEBUG_PRINTF1 ("waiting for inverted command repetition\n");
                            irmp_ir_detected = FALSE;
                            last_irmp_denon_command = irmp_tmp_command;
                        }
                    }
                    else
#endif // IRMP_SUPPORT_DENON_PROTOCOL
#if IRMP_SUPPORT_NEC_PROTOCOL == 1
                    if (irmp_tmp_protocol == IRMP_NEC_PROTOCOL && irmp_bit == 0)    // repetition frame
                    {
                        irmp_protocol = irmp_tmp_protocol;
                        irmp_address = last_irmp_address;                           // address is last address
                        irmp_command = last_irmp_command;                           // command is last command
                        irmp_flags |= IRMP_FLAG_REPETITION;
                    }
                    else
#endif // IRMP_SUPPORT_NEC_PROTOCOL
                    {
                        irmp_protocol = irmp_tmp_protocol;
                        irmp_address = irmp_tmp_address;                            // store address
#if IRMP_SUPPORT_NEC_PROTOCOL == 1
                        last_irmp_address = irmp_tmp_address;                       // store as last address, too
#endif

#if IRMP_SUPPORT_RC5_PROTOCOL == 1
                        irmp_tmp_command |= rc5_cmd_bit6;                           // store bit 6
#endif
                        irmp_command = irmp_tmp_command;                            // store command

#if IRMP_SUPPORT_SAMSUNG_PROTOCOL == 1
                        irmp_id = irmp_tmp_id;
#endif
                    }
                }

                if (irmp_ir_detected)
                {
                    if (last_irmp_command == irmp_command &&
                        last_irmp_address == irmp_address &&
                        repetition_counter < IRMP_REPETITION_TIME)
                    {
                        irmp_flags |= IRMP_FLAG_REPETITION;
                    }

                    last_irmp_address = irmp_tmp_address;                           // store as last address, too
                    last_irmp_command = irmp_tmp_command;                           // store as last command, too

                    repetition_counter = 0;
                }

                irmp_start_bit_detected = 0;                                        // and wait for next start bit
                irmp_tmp_command        = 0;
                irmp_pulse_time         = 0;
                irmp_pause_time         = 0;
            }
        }
    }
}

#ifdef DEBUG

// main function - for unix/linux + windows only!
// AVR: see main.c!
// Compile it under linux with:
// cc irmp.c -o irmp
//
// usage: ./irmp [-v|-s] < file

int
main (int argc, char ** argv)
{
    int         i;
    int         verbose = FALSE;
    int         ch;
    int         last_ch = 0;
    int         pulse = 0;
    int         pause = 0;
    IRMP_DATA   irmp_data;

    if (argc == 2)
    {
        if (! strcmp (argv[1], "-v"))
        {
            verbose = TRUE;
        }
        else if (! strcmp (argv[1], "-s"))
        {
            silent = TRUE;
        }
    }

    while ((ch = getchar ()) != EOF)
    {
        if (ch == '_' || ch == '0')
        {
            if (last_ch != ch)
            {
                if (verbose)
                {
                    printf ("pause: %d\n", pause);
                }
                pause = 0;
            }
            pulse++;
            IRMP_PIN = 0x00;
        }
        else if (ch == 0xaf || ch == '-' || ch == '1')
        {
            if (last_ch != ch)
            {
                if (verbose)
                {
                    printf ("pulse: %d ", pulse);
                }
                pulse = 0;
            }
            pause++;
            IRMP_PIN = 0xff;
        }
        else if (ch == '\n')
        {
            IRMP_PIN = 0xff;

            for (i = 0; i < 8000; i++)                                              // newline: long pause of 800 msec
            {
                irmp_ISR ();
            }
        }
        else if (ch == '#')
        {
            puts ("-------------------------------------------------------------------");
            putchar (ch);

            while ((ch = getchar()) != '\n' && ch != EOF)
            {
                if (ch != '\r')                                                     // ignore CR in DOS/Windows files
                {
                    putchar (ch);
                }
            }
            putchar ('\n');
        }

        last_ch = ch;

        irmp_ISR ();

        if (irmp_get_data (&irmp_data))
        {
            printf ("protcol = %d, address = 0x%04x, code = 0x%04x, flags = 0x%02x\n",
                    irmp_data.protocol, irmp_data.address, irmp_data.command, irmp_data.flags);
        }
    }
    return 0;
}

#endif // DEBUG
