/*
 * user.c
 *
 *  Created on: 29.08.2011
 *      Author: Frank
 */

#include <inttypes.h>
#include <avr/interrupt.h>
#include <stdlib.h>

#include "define.h"

#ifdef DEBUG
#include "base.h"
#include "uart.h"
#endif

#include "user.h"
#ifdef WITH_FB
#include "irmp.h"
#include "wceeprom.h"
#endif
#include "TLC5940.h"
#include "hsv_rgb.h"

#ifdef WITH_FB
#define MAX_COLOR_STEPS      4095
#define MAX_SATURATION_STEPS 4095
#define MAX_PWM_STEPS        64
#define MAX_FADING_STEPS     10
#define MAX_AUTO_LEAVE_STEPS 60
#endif
#define USER_DELAY_BEFORE_SAVE_EEPROM_s 120

#ifdef WITH_FB
#define g_params (&(wcEeprom_getData()->userParams))

uint16_t pwm_table[MAX_PWM_STEPS] /*PROGMEM */ =
{
      0,   18,   34,   50,   66,   82,   98,  114,
    130,  146,  162,  178,  194,  210,  226,  242,
    258,  274,  290,  306,  322,  338,  354,  370,
    386,  402,  418,  434,  450,  466,  482,  498,
    514,  530,  546,  562,  578,  594,  610,  626,
    642,  658,  674,  691,  771,  835,  915,  996,
   1092, 1188, 1301, 1429, 1558, 1750, 1863, 2023,
   2216, 2409, 2634, 2875, 3148, 3437, 4079, 4095
};

static uint8_t      g_keyDelay;
static uint8_t      g_userSwitchedOff;
static uint8_t      g_keyPressTime;

static uint8_t      g_eepromSaveDelay;
#endif

volatile uint16_t   g_colorCounter;
volatile uint8_t    g_colorUpdate;
volatile uint16_t   g_ColorStep;

uint8_t             g_fadingStep;
#ifdef WITH_FB
e_userCommands      g_userCommands;

uint8_t             g_brightnessCounter;
uint8_t             g_lastbrightnessCounter;

uint16_t            g_SaturationCounter;
uint8_t             g_autoleaveMenuModeCounter;
#endif

colorstruc          g_colorStruc;

#ifdef DEBUG
char text[20];
#endif

#ifdef WITH_FB
enum SWITCHED_OFF{
  USO_NORMAL_ON = 0,
  USO_OVERIDE_ON,
  USO_AUTO_OFF,
  USO_MANUAL_OFF,
};

uint8_t handle_standardMode( uint8_t ir_code, uint8_t repetition )
{
	uint8_t handled;
	handled = FALSE;
	if( UI_ONOFF == ir_code )
	{
		if(g_userSwitchedOff == USO_NORMAL_ON)
		{
			g_userSwitchedOff = USO_MANUAL_OFF;
			g_keyDelay = USER_KEY_PRESS_DELAY_100ms;
			pwmOff();
#ifdef DEBUG_USER
			uart_puts("off\n");
#endif
			handled = TRUE;
		}
		else if(g_userSwitchedOff == USO_MANUAL_OFF)
		{
			g_userSwitchedOff = USO_NORMAL_ON;
			g_keyDelay = USER_KEY_PRESS_DELAY_100ms;
			pwmOn();
#ifdef DEBUG_USER
			uart_puts("on\n");
#endif
			handled = TRUE;
		}
#ifdef DEBUG_STATE
			uart_puts("enter writeback\n");
#endif
        if ( handled )
        {
        	wcEeprom_writeback( wcEeprom_getData(), sizeof(WcEepromData)); /* save whole data */
        }
#ifdef DEBUG_STATE
			uart_puts("leave writeback\n");
#endif
        return handled;
    }
	else if ( g_userSwitchedOff == USO_NORMAL_ON )
	{
		if ( repetition )
		{
			if ( UI_BRIGHTNESS_UP == ir_code )
			{
				increaseBrightness();
#ifdef DEBUG_USER
				uart_puts("heller_r\n");
#endif
				return TRUE;
			}
			else if ( UI_BRIGHTNESS_DOWN == ir_code )
			{
				decreaseBrightness();
#ifdef DEBUG_USER
				uart_puts("dunkler_r\n");
#endif
				return TRUE;
			}
		    else
		    {
		    	return FALSE;
		    }
		}
		else
		{
			if ( UI_FADE == ir_code )
			{
				setColorMode( 0 );
				return TRUE;
			}
			else if ( UI_1 == ir_code )
			{
				setColorMode( 1 );
				return TRUE;
			}
			else if ( UI_2 == ir_code )
			{
				setColorMode( 2 );
				return TRUE;
			}
			else if ( UI_3 == ir_code )
			{
				setColorMode( 3 );
				return TRUE;
			}
			else if ( UI_4 == ir_code )
			{
				setColorMode( 4 );
				return TRUE;
			}
			else if ( UI_5 == ir_code )
			{
				setColorMode( 5 );
				return TRUE;
			}
			else if ( UI_6 == ir_code )
			{
				setColorMode( 6 );
				return TRUE;
			}
			else if ( UI_7 == ir_code )
			{
				setColorMode( 7 );
				return TRUE;
			}
			else if ( UI_8 == ir_code )
			{
				setColorMode( 8 );
				return TRUE;
			}
			else if ( UI_9 == ir_code )
			{
				setColorMode( 9 );
				return TRUE;
			}
			else if ( UI_BRIGHTNESS_UP == ir_code )
			{
				increaseBrightness();
#ifdef DEBUG_USER
				uart_puts("heller\n");
#endif
				return TRUE;
			}
			else if ( UI_BRIGHTNESS_DOWN == ir_code )
			{
				decreaseBrightness();
#ifdef DEBUG_USER
				uart_puts("dunkler\n");
#endif
				return TRUE;
			}
			else
			{
				return FALSE;
			}
		}
	}
	return FALSE;
}

uint8_t handle_fadeMode( uint8_t ir_code, uint8_t repetition )
{
	if ( g_userSwitchedOff == USO_NORMAL_ON )
	{
		if ( repetition )
		{
			if ( UI_UP == ir_code )
			{
				increaseFadingStep();
#ifdef DEBUG_USER
				uart_puts("schneller\n");
#endif
				return TRUE;
			}
			else if ( UI_DOWN == ir_code )
			{
				decreaseFadingStep();
#ifdef DEBUG_USER
				uart_puts("langsamer\n");
#endif
				return TRUE;
			}
			else
			{
				return FALSE;
			}
		}
		else
		{
			if ( UI_UP == ir_code )
			{
				decreaseFadingStep();
#ifdef DEBUG_USER
				uart_puts("schneller\n");
#endif
				return TRUE;
			}
			else if ( UI_DOWN == ir_code )
			{
				increaseFadingStep();
#ifdef DEBUG_USER
				uart_puts("langsamer\n");
#endif
				return TRUE;
			}
			else
			{
				return FALSE;
			}
		}
	}
	return FALSE;
}

uint8_t handle_menuMode( uint8_t ir_code, uint8_t repetition )
{
	static uint8_t step = 1;
	static uint8_t stepcounter = 0;

	if ( g_userSwitchedOff == USO_NORMAL_ON )
	{
		if ( repetition )
		{
			if ( UI_BRIGHTNESS_UP == ir_code )
			{
				g_autoleaveMenuModeCounter = 1;
				increaseBrightness();
#ifdef DEBUG_USER
				uart_puts("heller_r\n");
#endif
				return TRUE;
			}
			else if ( UI_BRIGHTNESS_DOWN == ir_code )
			{
				g_autoleaveMenuModeCounter = 1;
				decreaseBrightness();
#ifdef DEBUG_USER
				uart_puts("dunkler_r\n");
#endif
				return TRUE;
			}
			else if ( UI_UP == ir_code )
			{
				stepcounter++;
				if ( stepcounter == 10 )
				{
					step += 50;
				}
				g_autoleaveMenuModeCounter = 1;
				increaseColorStep( step );
#ifdef DEBUG_USER
				uart_puts("Farbe_r +\n");
#endif
				return TRUE;
			}
			else if ( UI_DOWN == ir_code )
			{
				stepcounter++;
				if ( stepcounter == 10 )
				{
					step += 50;
				}
				g_autoleaveMenuModeCounter = 1;
				decreaseColorStep( step );
#ifdef DEBUG_USER
				uart_puts("Farbe_r -\n");
#endif
				return TRUE;
			}
			else if ( UI_SATURATION_DOWN == ir_code )
			{
				stepcounter++;
				if ( stepcounter == 10 )
				{
					step += 50;
				}
				g_autoleaveMenuModeCounter = 1;
				decreaseSaturationStep( step );
#ifdef DEBUG_USER
				uart_puts("Sttigung_r -\n");
#endif
				return TRUE;
			}
			else if ( UI_SATURATION_UP == ir_code )
			{
				stepcounter++;
				if ( stepcounter == 10 )
				{
					step += 50;
				}
				g_autoleaveMenuModeCounter = 1;
				increaseSaturationStep( step );
#ifdef DEBUG_USER
				uart_puts("Sttigung_r +\n");
#endif
				return TRUE;
			}
			else
			{
				return FALSE;
			}
		}
		else
		{
			if ( UI_BRIGHTNESS_UP == ir_code )
			{
				g_autoleaveMenuModeCounter = 1;
				increaseBrightness();
#ifdef DEBUG_USER
				uart_puts("heller\n");
#endif
				return TRUE;
			}
			else if ( UI_BRIGHTNESS_DOWN == ir_code )
			{
				g_autoleaveMenuModeCounter = 1;
				decreaseBrightness();
#ifdef DEBUG_USER
				uart_puts("dunkler\n");
#endif
				return TRUE;
			}
			else if ( UI_UP == ir_code )
			{
				step = 1;
				stepcounter = 0;
				g_autoleaveMenuModeCounter = 1;
				increaseColorStep( step );
#ifdef DEBUG_USER
				uart_puts("Farbe +\n");
#endif
				return TRUE;
			}
			else if ( UI_DOWN == ir_code )
			{
				step = 1;
				stepcounter = 0;
				g_autoleaveMenuModeCounter = 1;
				decreaseColorStep( step );
#ifdef DEBUG_USER
				uart_puts("Farbe -\n");
#endif
				return TRUE;
			}
			else if ( UI_SATURATION_DOWN == ir_code )
			{
				step = 1;
				stepcounter = 0;
				g_autoleaveMenuModeCounter = 1;
				decreaseSaturationStep( step );
#ifdef DEBUG_USER
				uart_puts("Sttigung -\n");
#endif
				return TRUE;
			}
			else if ( UI_SATURATION_UP == ir_code )
			{
				step = 1;
				stepcounter = 0;
				g_autoleaveMenuModeCounter = 1;
				increaseSaturationStep( step );
#ifdef DEBUG_USER
				uart_puts("Sttigung +\n");
#endif
				return TRUE;
			}
			else if ( UI_ENTER == ir_code )
			{
				saveColorStep();
#ifdef DEBUG_USER
				uart_puts("speichern\n");
#endif
				return TRUE;
			}
			else
			{
				return FALSE;
			}
		}
	}
	return FALSE;
}

uint8_t handle_normalMode( uint8_t ir_code, uint8_t repetition )
{
	if ( UI_MENU == ir_code )
	{
#ifdef DEBUG_STATE
		uart_puts( "enter MS_MenuMode " );
#endif
		g_userCommands = MS_MenuMode;
		setColorStep();
		return TRUE;
	}
	return FALSE;
}

void handle_ir_code( void )
{
	IRMP_DATA ir_data;

	if (irmp_get_data (&ir_data))
	{
#ifdef DEBUG_IR_CODE
		uint16ToHexStr(ir_data.protocol, text);
		uart_puts(text);
		uart_puts(" - ");
		uint16ToHexStr(ir_data.address, text);
		uart_puts(text);
		uart_puts(" - ");
		uint16ToHexStr(ir_data.command, text);
		uart_puts(text);
		uart_putc('\n');
#endif
		if(g_keyDelay)
		{
			return;
		}
		g_keyDelay = 0;

		uint8_t ir_code = 0;

		if( g_params->irAddress != ir_data.address )
			return;

		while( ( ir_code < UI_COMMAND_COUNT ) && ( g_params->irCommandCodes[ir_code] != ir_data.command ) )
		{
			++ir_code;
		}

		uint8_t handled = FALSE;
		if ( g_userCommands == MS_normalMode )
		{
			handled = handle_standardMode( ir_code, ir_data.flags & IRMP_FLAG_REPETITION );
			if ( !handled )
			{
				handle_normalMode( ir_code, ir_data.flags & IRMP_FLAG_REPETITION );
			}
		}
		else if ( g_userCommands == MS_fadeMode )
		{
			handled = handle_standardMode( ir_code, ir_data.flags & IRMP_FLAG_REPETITION );
			if ( !handled )
			{
				handle_fadeMode( ir_code, ir_data.flags & IRMP_FLAG_REPETITION );
			}
		}
		else if ( g_userCommands == MS_MenuMode )
		{
			handled = handle_menuMode( ir_code, ir_data.flags & IRMP_FLAG_REPETITION );
		}
	    g_eepromSaveDelay = 0; /* on recognized keypress reset the counter */
		ir_code = UI_COMMAND_COUNT;
	}
}
#endif

void handle_fading( void )
{
	if ( g_colorUpdate == 1 )
	{
		hsv_to_rgb12( &g_colorStruc );

#ifdef DEBUG_FADING
		uart_puts( " H : " );
		itoa( g_colorStruc.H, text, 10 );
		uart_puts( text );
		uart_puts( " S : " );
		itoa( g_colorStruc.S, text, 10 );
		uart_puts( text );
		uart_puts( " V : " );
		itoa( g_colorStruc.V, text, 10 );
		uart_puts( text );
		uart_puts( "\n" );

		uart_puts( " R : " );
		itoa( g_colorStruc.Pin_R, text, 10 );
		uart_puts( text );
		uart_puts( " G : " );
		itoa( g_colorStruc.Pin_G, text, 10 );
		uart_puts( text );
		uart_puts( " B : " );
		itoa( g_colorStruc.Pin_B, text, 10 );
		uart_puts( text );
		uart_puts( "\n" );
#endif
		TLC5940_setLED( 13, g_colorStruc.Pin_B );
		TLC5940_setLED( 14, g_colorStruc.Pin_G );
		TLC5940_setLED( 15, g_colorStruc.Pin_R );

		TLC5940_setLED( 10, g_colorStruc.Pin_B );
		TLC5940_setLED( 11, g_colorStruc.Pin_G );
		TLC5940_setLED( 12, g_colorStruc.Pin_R );

		TLC5940_setLED(  7, g_colorStruc.Pin_B );
		TLC5940_setLED(  8, g_colorStruc.Pin_G );
		TLC5940_setLED(  9, g_colorStruc.Pin_R );

		TLC5940_setLED(  4, g_colorStruc.Pin_B );
		TLC5940_setLED(  5, g_colorStruc.Pin_G );
		TLC5940_setLED(  6, g_colorStruc.Pin_R );

		TLC5940_display();
		g_colorUpdate = 0;
	}
}

void handle_leds( void )
{
	handle_fading();
}

#ifdef WITH_FB
void user_init( void )
{
	g_userCommands = g_params->mode;
	g_userSwitchedOff = USO_NORMAL_ON;
}

void user_isr10Hz(void)
{
	if(g_userSwitchedOff != USO_AUTO_OFF )
	{
		if( g_keyDelay != 0)
		{
			--g_keyDelay;
		}
		++g_keyPressTime;
	}
}

void user_isr1HZ( void )
{
	if ( g_userCommands == MS_MenuMode )
	{
		g_autoleaveMenuModeCounter++;
	}
	if(g_userSwitchedOff != USO_AUTO_OFF )
	{
		#ifdef USER_AUTOSAVE
	    if( g_eepromSaveDelay <= USER_DELAY_BEFORE_SAVE_EEPROM_s )
	    {
	      ++g_eepromSaveDelay;
	    }
	    if(g_eepromSaveDelay == USER_DELAY_BEFORE_SAVE_EEPROM_s)
	    {
	    	wcEeprom_writeback( wcEeprom_getData(), sizeof(WcEepromData)); /* save whole data */
	    }
		# endif
	 }
}
#endif

void color_init( void )
{
	g_colorCounter = 0;
	g_colorUpdate = 1;
#ifdef WITH_FB
	uint16_t* hsv = (uint16_t*)(&g_params->colorPresets[ g_params->curColorProfile ]);
	g_colorStruc.H = hsv[0];
	g_colorStruc.S = hsv[1];
	g_colorStruc.V = hsv[2];
	g_brightnessCounter = getBrightnessValue( g_colorStruc.V );
	g_SaturationCounter = g_colorStruc.S;
	g_fadingStep = g_params->fadingSteps;
#else
	g_fadingStep = FADING_STEP_INIT;
	g_colorStruc.H = 0;
	g_colorStruc.S = 4095;
	g_colorStruc.V = 4095;
#endif
}

void color_isr100Hz(void)
{
#ifdef WITH_FB
	static uint8_t fadingStep = 0;
	if( g_userSwitchedOff != USO_AUTO_OFF && g_userCommands == MS_fadeMode )
	{
		fadingStep++;
		if ( fadingStep < g_fadingStep )
		{
			return;
		}
		fadingStep = 0;
#endif
		if ( !g_colorUpdate )
		{
			g_colorCounter++;
			if ( g_colorCounter > 4095 )
			{
				g_colorCounter = 0;
			}
			g_colorStruc.H = g_colorCounter;
			g_colorUpdate = 1;
		}
#ifdef WITH_FB
	}
#endif
}

#ifdef WITH_FB
void increaseBrightness( void )
{
	if ( g_brightnessCounter < ( MAX_PWM_STEPS - 1 ) )
	{
		g_brightnessCounter++;
		g_colorStruc.V = pwm_table[g_brightnessCounter];
		setHSVValues( g_colorStruc.H, g_colorStruc.S, g_colorStruc.V );
	}
}

void decreaseBrightness( void )
{
	if ( g_brightnessCounter > 0 )
	{
		g_brightnessCounter--;
		g_colorStruc.V = pwm_table[g_brightnessCounter];
		setHSVValues( g_colorStruc.H, g_colorStruc.S, g_colorStruc.V );
	}
}

void increaseFadingStep( void )
{
	if ( g_fadingStep < MAX_FADING_STEPS )
	{
		g_fadingStep++;
		g_params->fadingSteps = g_fadingStep;
	}
}

void decreaseFadingStep( void )
{
	if ( g_fadingStep > 0 )
	{
		g_fadingStep--;
		g_params->fadingSteps = g_fadingStep;
	}
}

void pwmOn( void )
{
	g_colorStruc.V = pwm_table[g_lastbrightnessCounter];
	g_colorUpdate = 1;
}

void pwmOff( void )
{
	g_lastbrightnessCounter = g_brightnessCounter;
	g_colorStruc.V = pwm_table[0];
	g_colorUpdate = 1;
}

void setColorMode( uint8_t mode )
{
	if ( mode == 0 )
	{
#ifdef DEBUG_STATE
		uart_puts( "enter MS_FadeMode " );
#endif
		g_userCommands = MS_fadeMode;
		g_params->curColorProfile = UI_COLOR_PRESET_COUNT - 1;
		uint16_t* hsv = (uint16_t*)(&g_params->colorPresets[ g_params->curColorProfile ]);
		g_colorStruc.H = hsv[0];
		g_colorStruc.S = hsv[1];
		g_colorStruc.V = hsv[2];
		g_brightnessCounter = getBrightnessValue( hsv[2] );
		g_colorUpdate = 1;
	}
	if ( mode > 0 && mode < 10 )
	{
#ifdef DEBUG_STATE
		uart_puts( "enter MS_NormalMode " );
#endif
		g_params->curColorProfile = mode - 1;
		uint16_t* hsv = (uint16_t*)(&g_params->colorPresets[ g_params->curColorProfile ]);
		g_colorStruc.H = hsv[0];
		g_colorStruc.S = hsv[1];
		g_colorStruc.V = hsv[2];
		g_brightnessCounter = getBrightnessValue( hsv[2] );
		g_userCommands = MS_normalMode;
		g_colorUpdate = 1;
	}
	g_params->mode = g_userCommands;
}

void increaseColorStep( uint8_t step )
{
	if ( g_ColorStep < MAX_COLOR_STEPS )
	{
		g_ColorStep += step;
		if ( g_ColorStep >= MAX_COLOR_STEPS )
		{
			g_ColorStep = MAX_COLOR_STEPS;
		}
	}
	g_colorStruc.H = g_ColorStep;
	setHSVValues( g_colorStruc.H, g_colorStruc.S, g_colorStruc.V );
}

void decreaseColorStep( uint8_t step )
{
	if ( g_ColorStep > 0 )
	{
		g_ColorStep -= step;
		if ( g_ColorStep > 4095 )
		{
			g_ColorStep = 0;
		}
	}
	g_colorStruc.H = g_ColorStep;
	setHSVValues( g_colorStruc.H, g_colorStruc.S, g_colorStruc.V );
}

void setColorStep( void )
{
	g_ColorStep = g_colorStruc.H;
	g_SaturationCounter = g_colorStruc.S;
	g_brightnessCounter = getBrightnessValue( g_colorStruc.V );
}

void saveColorStep( void )
{
	setHSVValues( g_colorStruc.H, g_colorStruc.S, g_colorStruc.V );
    wcEeprom_writeback( wcEeprom_getData(), sizeof(WcEepromData));
	g_userCommands = MS_normalMode;
}

void increaseSaturationStep( uint8_t step )
{
	if ( g_SaturationCounter < MAX_SATURATION_STEPS )
	{
		g_SaturationCounter += step;
		if ( g_SaturationCounter > MAX_SATURATION_STEPS )
		{
			g_SaturationCounter = MAX_SATURATION_STEPS;
		}
	}
	g_colorStruc.S = g_SaturationCounter;
	setHSVValues( g_colorStruc.H, g_colorStruc.S, g_colorStruc.V );
}

void decreaseSaturationStep( uint8_t step )
{
	if ( g_SaturationCounter >= 0 )
	{
		g_SaturationCounter -= step;
		if ( g_SaturationCounter > MAX_SATURATION_STEPS )
		{
			g_SaturationCounter = 0;
		}
	}
	g_colorStruc.S = g_SaturationCounter;
	setHSVValues( g_colorStruc.H, g_colorStruc.S, g_colorStruc.V );
}

void autoleaveMenuMode( void )
{
	if ( g_autoleaveMenuModeCounter >= MAX_AUTO_LEAVE_STEPS && g_userCommands == MS_MenuMode )
	{
		saveColorStep();
#ifdef DEBUG_STATE
		uart_puts( "leave MS_MenuMode " );
#endif
	}
}

uint8_t getBrightnessValue( uint16_t V )
{
	uint8_t i;
	i = 0;
	while( ( i < MAX_PWM_STEPS ) && ( pwm_table[i] != V ) )
	{
		++i;
	}
	return i;
}

extern void setHSVValues( uint16_t H, uint16_t S, uint16_t V )
{
    uint16_t* hsv = (uint16_t*)(&g_params->colorPresets[ g_params->curColorProfile ]);
    if ( hsv[0] != H )
    {
    	hsv[0] = H;
    	g_colorUpdate = 1;
    }
    if ( hsv[1] != S )
    {
    	hsv[1] = S;
    	g_colorUpdate = 1;
    }
    if ( hsv[2] != V )
    {
    	hsv[2] = V;
    	g_brightnessCounter = getBrightnessValue( hsv[2] );
    	g_colorUpdate = 1;
    }
}

#endif
