
#include "project.h"

#ifdef DEBUG

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include DEBUG_H_FILEPATH
#include UTIL_H_FILEPATH
#include HARDWARE_H_FILEPATH
#ifdef DEBUG_UPLINK_UART_MAX3100
#include MAX3100_H_FILEPATH
#endif
#ifdef DEBUG_UPLINK_I2C_HW
#include I2C_HW_H_FILEPATH
#endif
#include <stdio.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>

#ifdef __CODEVISIONAVR__
#pragma regalloc-
#endif
unsigned char debug_cmd[DEBUG_MAX_UART_COMMAND];
unsigned char debug_act_cmd = 0;
unsigned char debug_last_key = 0;

#ifdef __CODEVISIONAVR__
#pragma regalloc+

#pragma used+
#pragma used-
#endif

int debug_putchar(unsigned char c);

void debug_do_cmd(void);

void debug_init(void)
{
#if defined(DEBUG_UPLINK_UART0) || defined(DEBUG_UPLINK_UART1)
   unsigned char ucByte;
   unsigned long ulBaudRateCounter;
#define serBAUD_DIV_CONSTANT			((unsigned long )16)
#define ulWantedBaud                    DEBUG_BAUD_RATE
#define serRX_ENABLE					_BV(RXEN)
#define serTX_ENABLE					_BV(TXEN)
#define serEIGHT_DATA_BITS				(((8)-5)<<UCSZ0)

   // falls kein Level-shifter angeschlossen ist, hilft die Aktivierung des
   // Pullups auf der Rx-Leitung
   // die Frame-Errors zu reduzieren
   DEBUG_UART_PULLUP_ENABLE;
      /*
       * Calculate the baud rate register value from the equation in the data
       * sheet.
       */
      ulBaudRateCounter = ((F_CPU/serBAUD_DIV_CONSTANT)/ulWantedBaud) - (unsigned long)1;

      /*
       * Set the baud rate.
       */
      ucByte =(unsigned char)(ulBaudRateCounter & (unsigned long)0xff );
      UBRR0L = ucByte;

      ulBaudRateCounter >>= 8;
      ucByte = (unsigned char)(ulBaudRateCounter & (unsigned long)0xff);
      UBRR0H = ucByte;

      UCSR0B =  serRX_ENABLE | serTX_ENABLE;

#ifdef URSEL
      UCSR0C = _BV(URSEL) | serEIGHT_DATA_BITS;
#else
      UCSR0C = serEIGHT_DATA_BITS;
#endif
#endif
#ifdef DEBUG_UPLINK_UART_MAX3100
   max3100_init();
#endif
#ifdef DEBUG_UPLINK_I2C_HW
   i2c_hw_init();
#endif

#ifndef __CODEVISIONAVR__
    fdevopen( (int(*)(char))debug_putchar, NULL, 0 );
    printf_P(PSTR("Welcome\n"));
DDRG |= (1<<3);
PORTG &= ~(1<<3);
PORTG |= (1<<3);
#endif
}


#if defined(DEBUG_UPLINK_UART1) || defined(DEBUG_UPLINK_UART0)

#if defined(_CHIP_ATMEGA8535_)||defined(_CHIP_ATMEGAS8535L_)||\
    defined(_CHIP_ATMEGA8_)   ||defined(_CHIP_ATMEGA8L_)    ||\
    defined(_CHIP_ATMEGA16_)  ||defined(_CHIP_ATMEGAS16L_)  ||\
    defined(_CHIP_ATMEGA32_)  ||defined(_CHIP_ATMEGA32L_)   ||\
    defined(__AVR_ATmega32__) ||defined(__AVR_ATmega8535__)
#define DEBUG_UDR					UDR
#define DEBUG_UCSR					UCSRA
#else
#ifdef DEBUG_UPLINK_UART0
#define DEBUG_UDR					UDR0
#define DEBUG_UCSR					UCSR0A
#else
#define DEBUG_UDR					UDR1
#define DEBUG_UCSR					UCSR1A
#endif
#endif
#define FRAMING_ERROR				0x10
#define PARITY_ERROR 				0x04
#define DATA_OVERRUN				0x08
#define DATA_REGISTER_EMPTY 		0x20
#define RX_COMPLETE 				0x80

signed int debug_check_rx(void)
{
   unsigned char status, data;

   if (((status = DEBUG_UCSR) & RX_COMPLETE) == 0)
      return -1;
   data = DEBUG_UDR;
   if ((status & (FRAMING_ERROR | PARITY_ERROR | DATA_OVERRUN)) == 0)
      return data;
   else
      return -1;
}


#if defined(DEBUG_UPLINK_UART1) || defined (DEBUG_UPLINK_UART0)
int debug_putchar(unsigned char c)
{
   if(c == '\n')
       debug_putchar('\r');
   while ((DEBUG_UCSR & DATA_REGISTER_EMPTY) == 0)
      ;
   DEBUG_UDR = c;
   return 0;
}
#endif
#endif


#ifdef DEBUG_UPLINK_UART_MAX3100
unsigned char debug_check_rx(void)
{
   if (max3100_poll() == 1)
      return max3100_getchar();
   else
      return -1;
}


int debug_putchar(unsigned char c)
{
   if(c == '\n')
       max3100_putchar('\r');
   max3100_putchar(c);
   return 0;
}
#endif


#ifdef DEBUG_UPLINK_I2C_HW
int debug_putchar(unsigned char c)
{
   i2c_hw_start();
   i2c_hw_write(0x02);
   i2c_hw_write(c);
   i2c_hw_stop();
   return 0;
}


signed int debug_check_rx(void)
{
   unsigned char data;
   unsigned char retval = 0;

   delay_ms(1);
   i2c_hw_start();
   delay_ms(1);
   i2c_hw_write(0x03);
   delay_ms(1);
   retval = i2c_hw_read(&data, 1);
   delay_ms(1);
   if (data == 1)
   {
      // Zeichen vorhanden
      retval = i2c_hw_read(&data, 0);
      delay_ms(1);
      i2c_hw_stop();
      return data;
   }
   else
   {
      // keine Daten vorhanden
      retval = i2c_hw_read(&data, 0);
      delay_ms(1);
      i2c_hw_stop();
      return -1;
   }
}
#endif


const char debug_q1[] PROGMEM = "??:%s\n";
const char debug_q2[] PROGMEM = "??:%s";
unsigned char *debug_get_command(void)
{
   signed int act_key;

   act_key = debug_check_rx();
   if (act_key != -1)
   {
      if (act_key == '\r')
      {
         if (debug_act_cmd != 0)
            debug_cmd[debug_act_cmd] = 0;
         debug_act_cmd = 0;
         printf_P(debug_q1, debug_cmd);
         debug_last_key = (unsigned char) act_key;
         debug_do_cmd();
         return debug_cmd;
      }
      else if (act_key == '\b')
      {
         if (debug_act_cmd > 0)
         {
            debug_act_cmd--;
            printf_P(PSTR(" \b"));
         }
         debug_cmd[debug_act_cmd] = 0;
      }
      else if (act_key == 0x1b)
      {
         if (debug_last_key == 0x0d)
         {
            debug_act_cmd = strlen(debug_cmd);
            printf_P(debug_q2, debug_cmd);
         }
      }
      else
      {
         debug_cmd[debug_act_cmd] = (unsigned char) act_key;
         debug_act_cmd++;
         if (debug_act_cmd >= DEBUG_MAX_UART_COMMAND)
            debug_act_cmd = 0;
      }
      debug_last_key = act_key;
   }
   return 0;
}


unsigned char debug_set_command(PGM_P new_command)
{
   if (strlen_P(new_command) < sizeof(debug_cmd))
   {
      debug_last_key = 0x0d;
      strcpy_P(debug_cmd, new_command);
      return 1;
   }
   return 0;
}

const char P2ukz[]   PROGMEM = "%2u.) ";
const char P2xsp[]   PROGMEM = "%02x ";
const char NewLine[] PROGMEM = "\n";
const char NewLine5sp[] PROGMEM = "\n     ";
void debug_display_frame(unsigned char *p_data, unsigned int size,
                         unsigned char break_count)
{
   unsigned char sign_count = 0;
   unsigned char new_break = 0;

   while (size)
   {
#ifdef __CODEVISIONAVR__
#asm("wdr");
#else
      wdt_reset();
#endif
      if ((break_count != 0) && (sign_count == 0))
      {
         printf_P(P2ukz, new_break);
      }
      printf_P(P2xsp, *p_data);
      p_data++;
      size--;
      sign_count++;
      if (break_count != 0)
      {
         if (break_count <= 24)
         {
            if (sign_count == break_count)
               printf_P(NewLine);
         }
         else
         {
            if (sign_count == break_count)
               printf_P(NewLine);
            else if ((sign_count % 24) == 0)
               printf_P(NewLine5sp);
         }
         if (sign_count == break_count)
         {
            sign_count = 0;
            new_break++;
         }
      }
      else
      {
         if (sign_count == 24)
         {
            sign_count = 0;
            printf_P(NewLine);
         }
      }
   }
   if (sign_count != 24)
      printf_P(NewLine);
}


void debug_display_frame_ee(/*eeprom*/ unsigned char *p_data, unsigned int size,
                            unsigned char break_count)
{
   unsigned char sign_count = 0;
   unsigned char new_break = 0;

   while (size)
   {
#ifdef __CODEVISIONAVR__
#asm("wdr");
#else
      wdt_reset();
#endif
      if ((break_count != 0) && (sign_count == 0))
      {
         printf_P(P2ukz/*PSTR("%2u.) ")*/, new_break);
      }
      printf_P(P2xsp/*PSTR("%02x ")*/, eeprom_read_byte(&p_data[0]));
      p_data++;
      size--;
      sign_count++;
      if (break_count != 0)
      {
         if (break_count <= 24)
         {
            if (sign_count == break_count)
               printf_P(NewLine);
         }
         else
         {
            if (sign_count == break_count)
               printf_P(NewLine);
            else if ((sign_count % 24) == 0)
               printf_P(NewLine5sp);
         }
         if (sign_count == break_count)
         {
            sign_count = 0;
            new_break++;
         }
      }
      else
      {
         if (sign_count == 24)
         {
            sign_count = 0;
            printf_P(NewLine);
         }
      }
   }
   if (sign_count != 24)
      printf_P(NewLine);
}


const char Suuuuc[] PROGMEM = "%S: %u.%u.%u.%u\n";
void debug_print_ip(PGM_P string_data, unsigned char *ip_data)
{
   if (string_data != 0x0)
      printf_P(Suuuuc/*PSTR("%S: %u.%u.%u.%u%c")*/, string_data, ip_data[0], ip_data[1],
             ip_data[2], ip_data[3], '\n');
   else
      printf_P(&Suuuuc[4]/*PSTR("%u.%u.%u.%u ")*/, ip_data[0], ip_data[1], ip_data[2], ip_data[3], ' ');
}


void debug_print_ip_ee(PGM_P string_data,
                       /*eeprom*/ unsigned char *ip_data)
{
   if (string_data != 0x0)
      printf_P(Suuuuc/*PSTR("%S: %u.%u.%u.%u%c")*/, string_data, eeprom_read_byte(&ip_data[0]),
             eeprom_read_byte(&ip_data[1]),
             eeprom_read_byte(&ip_data[2]), eeprom_read_byte(&ip_data[3]), '\n');
   else
      printf_P(&Suuuuc[4]/*PSTR("%u.%u.%u.%u ")*/, eeprom_read_byte(&ip_data[0]),
             eeprom_read_byte(&ip_data[1]), eeprom_read_byte(&ip_data[2]),
             eeprom_read_byte(&ip_data[3]), ' ');
}


void debug_print_mac(PGM_P string_data,
                     unsigned char *mac_data)
{
   if (string_data != NULL)
      printf_P(PSTR("%S: "), string_data);
   printf_P(PSTR("%02x:%02x:%02x:%02x:%02x:%02x "), mac_data[0], mac_data[1],
          mac_data[2], mac_data[3], mac_data[4], mac_data[5]);
   if (string_data != NULL)
      printf_P(NewLine);
}


unsigned char debug_string2ip(unsigned char *command, unsigned char *new_ip,
                              unsigned int *new_port)
{
   unsigned char Lva = 0, Lvb, Lvc;
   unsigned char new_data[10];

   while ((command[Lva] != ' ') && (command[Lva] != '\0'))
      Lva++;
   while ((command[Lva] == ' '))
      Lva++;
   if (command[Lva] != '\0')
   {
      Lvb = 0;
      Lvc = 0;
      for (Lva = Lva; Lva < strlen(debug_cmd); Lva++)
      {
         if (Lvc > 3)
            return 0;
         if (command[Lva] == ' ')
            break;
         if (command[Lva] == '.')
         {
            new_data[Lvb] = 0x0;
            new_ip[Lvc] = atoi(new_data);
            Lvb = 0;
            Lvc++;
         }
         else
         {
            new_data[Lvb] = command[Lva];
            Lvb++;
         }
      }
      if (Lvc < 3)
         return 0;
      new_data[Lvb] = 0x0;
      new_ip[Lvc] = atoi(new_data);
      if (new_port != 0x0)
      {
         // delete the last part of ip-parameter
         while ((command[Lva] != ' ') && (command[Lva] != '\0'))
            Lva++;
         // delete the space-signs between ip and port-parameter
         while ((command[Lva] == ' '))
            Lva++;
         *new_port = 0;
         if (command[Lva] != '\0')
            *new_port = atoi(&command[Lva]);
      }
      return 1;
   }
   return 0;
}



unsigned char debug_string2mac(unsigned char *command, unsigned char *new_mac)
{
   unsigned char Lva = 0, Lvb, Lvc;
   unsigned char new_data[10];

   while ((command[Lva] != ' ') && (command[Lva] != '\0'))
      Lva++;
   while ((command[Lva] == ' '))
      Lva++;
   if (command[Lva] != '\0')
   {
      Lvb = 0;
      Lvc = 0;
      for (Lva = Lva; Lva < strlen(debug_cmd); Lva++)
      {
         if (Lvc > 5)
            return 0;
         if (command[Lva] == ' ')
            break;
         if ((command[Lva] == ':') || (command[Lva] == '-'))
         {
            if (atoi_hex(new_data[1]) == 0xff)
               new_mac[Lvc] = atoi_hex(new_data[0]);
            else
            {
               new_mac[Lvc] = atoi_hex(new_data[0]);
               new_mac[Lvc] <<= 4;
               new_mac[Lvc] |= atoi_hex(new_data[1]);
            }
            Lvb = 0;
            Lvc++;
         }
         else
         {
            new_data[Lvb] = command[Lva];
            Lvb++;
         }
      }
      if (Lvc < 5)
         return 0;
      if (atoi_hex(new_data[1]) == 0xff)
         new_mac[Lvc] = atoi_hex(new_data[0]);
      else
      {
         new_mac[Lvc] = atoi_hex(new_data[0]);
         new_mac[Lvc] <<= 4;
         new_mac[Lvc] |= atoi_hex(new_data[1]);
      }
      return 1;
   }
   return 0;
}



unsigned char debug_atoi_hex_str(unsigned char *src, unsigned char *dest,
                                 unsigned char *length)
{
   unsigned char result;
   unsigned char index = 0;

   while (*src != '\0')
   {
      result = atoi_hex(*src);
      if (result == 0xFF)
         return 0xFF;
      *dest = result << 4;
      src++;
      if (*src == 0x00)
         return 0xFC;
      result = atoi_hex(*src);
      if (result == 0xFF)
         return 0xFF;
      *dest |= result;
      src++;
      dest++;
      index++;
      if (index > *length)
         return 0xFE;
   }
   *length = index;
   return 0x00;
}


unsigned char *debug_search_parameter(unsigned char *command)
{
   while ((*command != ' ') && (*command != '\0'))
      command++;
   while ((*command == ' '))
      command++;
   if (*command != '\0')
      return command;
   return 0x0;
}


signed int debug_integer(unsigned char *command)
{
   command = debug_search_parameter(command);
   if (command != NULL)
      return atoi(command);
   return -1;
}


signed int debug_2nd_integer(unsigned char *command)
{
   command = debug_search_parameter(command);
   command = debug_search_parameter(command);
   if (command != NULL)
      return atoi(command);
   return -1;
}


void debug_help(void)
{
   printf_P(PSTR("\"reset\" - reset the AVR\n"));
}


#ifdef WATCHDOG_ENABLE
#ifdef __CODEVISIONAVR__
flash unsigned int watchdog_time[8] =
   { 16, 32, 65, 130, 260, 520, 1000, 2100 };
#else
PROGMEM unsigned int watchdog_time[8] =
   { 16, 32, 65, 130, 260, 520, 1000, 2100 };
#endif
#endif
void debug_info(PGM_P info_string)
{
   if ((info_string) && (pgm_read_byte(&info_string[0]) != 0))
      printf_P(PSTR("%S\n"), info_string);
   printf_P(PSTR("Project: " PROJECT_NAME " " PROJECT_VERSION " (" __DATE__ "/"
         __TIME__ ")\n"));
#ifdef __CODEVISIONAVR__
   printf_P(PSTR("CodeVisionAVR-Version:%u\n"), __CODEVISIONAVR__);
#else
   printf_P(PSTR("CGG-AVR-Version:%u.%u.%u\n"),
          __GNUC__, __GNUC_MINOR__, __GNUC_PATCHLEVEL__);
#endif
   printf_P(PSTR(MC_TYP_STRING ":" MCU_CLOCK_STRING "\n"));
   printf_P(PSTR("AVR-RESET: "));
   if (hardware_reset_status & 0x10)
      printf_P(PSTR("JTAG, "));
   if (hardware_reset_status & 0x08)
      printf_P(PSTR("Watchdog, "));
   if (hardware_reset_status & 0x04)
      printf_P(PSTR("Brown-out, "));
   if (hardware_reset_status & 0x02)
      printf_P(PSTR("External, "));
   if (hardware_reset_status & 0x01)
      printf_P(PSTR("Power-On, "));
   if (hardware_reset_status == 0x00)
      printf_P(PSTR("no Reset"));
   printf_P(PSTR("\n"));
#ifdef WATCHDOG_ENABLE
   printf_P(PSTR("AVR-Watchdog: %ums\n"), pgm_read_word(&watchdog_time[WDTCR & 0x07]));
#endif
}


void debug_do_cmd(void)
{
   if (strncmp_P(debug_cmd, PSTR("reset"), 5) == 0)
   {
      // Watchdog Timer initialization
#ifdef __CODEVISIONAVR__
#asm("cli");
#else
      cli();
#endif
      WDTCR = 0x18;
      WDTCR = 0x08;
//#asm("sei");
      for (;;)
         ;
   }
   if (strncmp_P(debug_cmd, PSTR("help"), 4) == 0)
   {
      debug_help();
   }
}
#endif // #ifdef DEBUG
