// 3c5x9.c: device driver for the 3c5x9 nic in 8-bit mode.


#include "project.h"
#if defined(ETHERNET_ENABLE) && defined(DEVICE_3c509b)

#include <stdio.h>
#include <string.h>
#include <avr/pgmspace.h>
#include <avr/signal.h>
//#include <avr/delay.h>
extern void delay_ms(unsigned int);
#include <avr/interrupt.h>
#include <avr/io.h>

#include NIC_3c509b_H_FILEPATH
#include DEVICE_H_FILEPATH

#include ETHERNET_H_FILEPATH
#include BUFFER_H_FILEPATH
#include IP_H_FILEPATH
#if defined(ARP_ENABLE) || defined(RARP_ENABLE)
#include ARP_RARP_H_FILEPATH
#endif
#ifdef WEB_DEBUG_MEASURE
#include MEASURE_H_FILEPATH
#endif
#ifdef WEB_DEBUG
#include DEBUG_H_FILEPATH
#include WEB_DEBUG_H_FILEPATH
#endif


#define UIP_BUFSIZE (1500)
//static char uip_buf[UIP_BUFSIZE];
//static char *uip_appdata = NULL;
//static PGM_P uip_PROGDATA = NULL;
//static uint16_t uip_len = 0;

unsigned char mac_of_last_frame[6];


//#include "uip.h"
//#include "uip_arp.h"

//#define NIC_INT       INT6
//#define SIG_3c5x9NIC  SIG_INTERRUPT6
//#define ISC_NICINT_1  ISC61
//#define ISC_NICINT_0  ISC60

//xSemaphoreHandle NIC_RX_sema = NULL;

/*
Die Karte muss vorher mit 3c5x9cfg.exe konfiguriert sein.
Aktuelle Einstellungen: INT3, I/O 310H, TP, PnP aus, Full-Duplex aus.
- 3c509B - I/O cycle time = 200ns
- Bei 16MHz - Dauer 62,5ns - Memory mapped ein WaitState -> 125ns
*/


// ********************* MACROS ***************************

#define nop()  __asm__ volatile ("nop" ::)


/*#######################################################################################
// Belegung fr mein ATmega128 AT-Bus Board
#######################################################################################*/

/*
 * Steuerleitungen
 */
#define CTRL_OUT        *((volatile unsigned char *)0x1101)
#define CTRL_RD         _BV(6)
#define CTRL_WR         _BV(7)
#define CTRL_RESET      _BV(4)

#define RESET_ACTIVE()  (CTRL_OUT = 0xF0)
#define RESET_PASSIVE() (CTRL_OUT = 0xE0)
#define RD_ACTIVE_0(a)  (CTRL_OUT = (a | CTRL_WR))
#define RD_PASSIVE(a)   (CTRL_OUT = (a | (CTRL_RD) | (CTRL_WR)))
#define RD_ACTIVE(a)    RD_PASSIVE(a);RD_ACTIVE_0(a)
#define WR_ACTIVE_0(a)  (CTRL_OUT = (a | CTRL_RD))
#define WR_PASSIVE(a)   (CTRL_OUT = (a | (CTRL_RD) | (CTRL_WR)))
#define WR_ACTIVE(a)    WR_PASSIVE(a);WR_ACTIVE_0(a)

/*
 * Adressleitungen A0-A4
 */
#define ADDRESS_OUT     *((volatile unsigned char *)0x1101)
#define ADDRESS_MASK    (~((1<<CTRL_RD)|(1<<CTRL_WR)|(1<<CTRL_RESET)))

/*
 * Datenleitungen
 */
#define DATA_OUT        *((volatile unsigned char *)0x1100)
#define DATA_IN         *((volatile unsigned char *)0x1100)


// Die wichtigste  IO-Port Adresse als Makro
#define STATUS   0x2E

#define delay(a)      delay_ms(a)
#define delay_100ms() delay_ms(100)
/*
void delay_100ms( void )
{
   for(char i=0; i < 10; i++)
      _delay_ms( 10.0 );
}
*/
// 3c5x9 control macros
#define select_register_window(_window)	    if(1){_3c5x9_window__=_window&7;setreg(0x0800|_3c5x9_window__,0x2E);}
#define global_reset()	                    setreg(0x0000,0x2E)
#define start_coax_tranceiver()             setreg(0x1000,0x2E)
#define rx_disable()                        setreg(0x1800,0x2E)
#define rx_enable()                         setreg(0x2000,0x2E)
#define rx_reset() 	                        setreg(0x2800,0x2E)
#define rx_discard_top_packet()   	        setreg(0x4000,0x2E)
#define tx_enable()   	                    setreg(0x4800,0x2E)
#define tx_disable()  	                    setreg(0x5000,0x2E)
#define request_interrupt()   	            setreg(0x6000,0x2E)
#define statistics_enable()   	            setreg(0xA800,0x2E)
#define statistics_disable()                setreg(0xB000,0x2E)
#define stop_coax_transceiver()  	        setreg(0xB800,0x2E)
#define power_down_full()  	                setreg(0xE000,0x2E)
#define power_auto()  	                    setreg(0xE800,0x2E)

#define set_interrupt_mask(mask) 	        setreg(0x7000 | (mask & 0xFF),0x2E)
#define set_read_zero_mask(mask)	        setreg(0x7800 | (mask & 0xFF),0x2E)
#define set_rx_early_threshold(ths)	        setreg(0x8800 | (ths & 0x7FF),0x2E)
#define set_tx_available_threshold(ths) 	setreg(0x9000 | (ths & 0x7FF),0x2E)
#define set_tx_start_threshold(ths)	        setreg(0x9800 | (ths & 0x7FF),0x2E)
#define set_tx_claim_threshold(ths)  	    setreg(0xC000 | (ths & 0x7FF),0x2E)

// After an underrun or jabber error on transmit, a TX Reset is required.
// Do not use TX Reset unless it is absolutely necessary.
#define  tx_reset()  	                    setreg(0x5800,0x2E)

// Interrupt reasons:
// 0x01 = Interrupt Latch
// 0x02 = Adapter Failure
// 0x04 = TX Complete
// 0x08 = TX Available
// 0x10 = RX Complete
// 0x20 = RX Early
// 0x40 = Interrupt Request
// 0x80 = Update  Statistics
#define ack_interrupt(reason)	            setreg(0x6800|(reason&0xFF),0x2E)

// RX filter mask values
// 0x01 = Individual address
// 0x02 = Multicast address
// 0x04 = broadcast address
// 0x08 = promiscuous mode
#define set_rx_filter(filter)	            setreg((0x8000|(filter&0x0F)),0x2E)

// ***************** DATA *************************

/*
 * Hilfsvariable fr delay()
 */
//static uint16_t delay_count = F_CPU / 4000;

/*
 * Hier wird das aktuell ausgewaehlte Register-Fenster gemerkt fr den
 * Fall dass eine Interruptroutine sie ndert und wieder zurckstellen will.
 * Diese muss dieses erst sichern, kann dann ndern und an Ende einfach wieder herstellen.
 * SIGNAL(...) {
 *      uint8_t safe_win = _3c5x9_window__;
 *      select_register_window(ISR_window);
 *       ...
 *     select_register_window(safe_win);
 * } // end SIGNAL
 */
uint8_t _3c5x9_window__ = 0;

/*
 * Sendedaten Kopfinformation fr 3c5x9 Controler
 */
static uint8_t tx_preamble[] = {
   14 + 80, 0x00, 0x00, 0x00    // TRUE_PACKET_LENGTH, 0, 0, 0
};

// ******************* CODE ********************

/*
 * Ein byte ausgeben
 */
static void outportb100( uint8_t b )
{
   DATA_OUT = b;
   WR_ACTIVE( 0 );              // eigentlich kein nop beim schreiben ntig
   nop(  );
   WR_PASSIVE( 0 );
}


static void outportb31x( uint8_t p, uint8_t b )
{
   DATA_OUT = b;
   WR_ACTIVE( p );              // eigentlich kein nop beim schreiben ntig
   nop(  );
   WR_PASSIVE( p );
}

/*
 * Ein byte einlesen
 */
static uint8_t inportb31x( uint8_t p )
{
   RD_ACTIVE( p );
   nop(  );                     // Minimum 2 nop's beim lesen
   nop(  );
   RD_PASSIVE( p );
   return DATA_IN;
}

/*
 * Ein 2 byte wort ausgeben
 */
// "...commands must be written as a word, ... (executed when the high byte is written)..."

// Die reihenfolge des schreibens der register ist in "Chapter 6" ganz am anfang des datenblatts
// definiert. "...the low byte acceess must be followed by the high byte access,..."
//#define setreg(val,reg) setreg_(reg,val)
//static void setreg_(uint8_t reg, uint16_t val)
static void setreg( uint16_t val, uint8_t reg )
{
   union
   {
      struct
      {
         uint8_t l;
         uint8_t h;
      }
      h;
      uint16_t f;
   }
   st;

   st.f = val;

   DATA_OUT = st.h.l;           // low byte
   reg |= ( CTRL_RD ) | ( CTRL_WR );
   CTRL_OUT = reg;              // Address out, RD WR
   reg &= ~( CTRL_WR );
   CTRL_OUT = reg;              // /RD WR
   nop(  );
   reg |= ( CTRL_WR );
   CTRL_OUT = reg;              // Address+1 RD WR

   DATA_OUT = st.h.h;           // high byte
   reg++;
   CTRL_OUT = reg;              // RD WR
   reg &= ~( CTRL_WR );
   CTRL_OUT = reg;
   reg |= ( CTRL_WR );
   nop(  );
   CTRL_OUT = reg;
}

/*
 * Ein 2 byte wort einlesen
 */
// Die reihenfolge des lesens der register ist in "Chapter 6" ganz am anfang des datenblatts
// definiert. "...the low byte access must be followed by the high byte access,..."
static uint16_t readreg( uint8_t reg )
{
   union
   {
      struct
      {
         uint8_t l;
         uint8_t h;
      }
      h;
      uint16_t f;
   }
   st;

   reg |= ( CTRL_RD ) | ( CTRL_WR );

   CTRL_OUT = reg;              // Address out, RD WR
   reg &= ~( CTRL_RD );
   CTRL_OUT = reg;              // /RD WR
   reg |= ( CTRL_RD );
   reg++;
   CTRL_OUT = reg;              // Address+1 RD WR
   st.h.l = DATA_IN;            // lower byte
   reg &= ~( CTRL_RD );
   CTRL_OUT = reg;
   nop(  );
   reg |= ( CTRL_RD );
   CTRL_OUT = reg;
   st.h.h = DATA_IN;            // upper byte

   return st.f;
}


#if 0/*
 * kein macro
 */
static inline void power_up( void )
{
   setreg( 0xD800, 0x2E );
   tx_reset(  );                // see Page 39 of 3c5x9 manual
}
#endif

#if 0
/*
 * Lese Adaper-Konfinguartionsinformationen aus dem Adapter eeProm
 */
static uint16_t readAdapterEEProm( uint16_t reg )
{
   uint16_t to = 32000;

   reg &= 0x3F;                 // nur die unteren 6 bits gltig
   select_register_window( 0 );
   // Warte wenn eeProm busy
   while ( ( inportb31x( 0x2B ) & 0x80 ) && to-- );     // EEProm busy
   setreg( 0x80 + reg, 0x2A );
   // Warte wenn eeProm busy
   to = 32000;
   while ( ( inportb31x( 0x2B ) & 0x80 ) && to-- );     // EEProm busy
   to = readreg( 0x2C );
   select_register_window( 1 );
   return to;
}
#endif

/*
 * Die netzwerkkarte muss erst "aktiviert" werden damit sie, und nur sie auf dem
 * port 0x1x0 befehle entgegennimmt und informationen zurckgibt.
 * dazu muss ein bestimmtes bytemuster auf den port (ID_PORT) im bereich
 * 0x1x0 (x=1...F) ausgegeben werden, auf den als letztes der wert null
 * geschrieben wurde.
 */
static void activate_3c5x9_nic( void )
{
   uint16_t id;
   uint16_t loop;

   printf_P( PSTR( "3c - activate\n" ) );
   // ----------------
   // Write ID loop
   id = 0xFF;
   outportb100( 0 );            // Enable IDport
   outportb100( 0 );            // Enable IDport
   for ( loop = 0; loop < 255; loop++ )
   {
#if 1
      DATA_OUT = id;
      WR_ACTIVE_0( 0 );
      id <<= 1;
      if ( id & 0x100 )
         id ^= 0xCF;
      WR_PASSIVE( 0 );
#else
      outportb100( id );
      id <<= 1;
      if ( id & 0x100 )
         id ^= 0xCF;
#endif
   }
}

#if 0
/*
 * Entspricht einem hardware-reset
 */
void nic_global_reset( void )
{
   // Write ID loop
   activate_3c5x9_nic(  );
   // do a global reset
   outportb100( 0xCF );
   delay_100ms(  );             // Warte bis reset ausgefhrt
}
#endif

/*
 * Um vom ID_PORT eeprom daten zu lesen, mssen diese 16-bit werte nach anforderung in
 * einer schleife vom bit #0 des ID_PORT eingelsen werden. Hchstwertiges bit zuerst.
 */
static uint16_t readLoop( uint8_t eadr )
{
   uint8_t loop;
   uint16_t st = 0;             // to keep the compiler happy

   // Fordere controller auf die daten von eeprom adresse eadr zu lesen.
   outportb100( 0x80 + eadr );
   delay_100ms(  );             // Warte bis daten im eeprom datenregister
   // zur verfgung stehen

   RD_PASSIVE( 0 );
   for ( loop = 0; loop < 16; loop++ )
   {
      RD_ACTIVE_0( 0 );
      st <<= 1;
      nop(  );
      RD_PASSIVE( 0 );
      st |= ( DATA_IN & 1 );
   }
   return st;
}

static char initOK = 0;

volatile uint8_t i3cFlag = 0;

char nic_start(void)
{
    return initOK;
}

#if 0
void SIG_3c5x9NIC( void ) __attribute__ ( ( signal ) );
// #-#
void SIG_3c5x9NIC( void )
{
   uint8_t v = inportb31x( 0x2E );      // Status

   if((v & (uint8_t)0x01)) // this nic rx complete
   {
        set_interrupt_mask(0);   // disable rx-complete interrupt
        EIMSK &= ~(1<<NIC_INT);  // disable interrupt
        ack_interrupt(1);
 i3cFlag++;
        if(v & (uint8_t)0x10)  // RX complete
        {
            if( xSemaphoreGiveFromISR( NIC_RX_sema, pdFALSE ) )
            {
                vPortYield();
            }
        }
/*
        if(v & (uint8_t)0x..)
        {
        }
*/
   }
}
#endif

void nic_init( uint8_t *addr )
{
   uint16_t st;

   if(initOK)
       return;
//   vSemaphoreCreateBinary( NIC_RX_sema );
   /* We start by 'taking' the semaphore so the ISR can 'give' it when the
	  first interrupt occurs. */
//   if( NIC_RX_sema )
//      xSemaphoreTake( NIC_RX_sema, 0 );

//   initOK = 0;
   RESET_ACTIVE(  );
   printf_P( PSTR( "3c - +reset\n" ) );
   delay_100ms();             // keep reset active
//   printf_P( PSTR( "3c - -reset\n" ) );
   RESET_PASSIVE(  );

   // wait, NIC is initializing
   delay_100ms( );

   // activate the NIC
//   printf_P( PSTR( "3c activate the NIC\n" ) );
   activate_3c5x9_nic(  );

   // read the mac address of this card from eeprom address 10,11, and 12.
   st = readLoop( 10 );
   addr[0] = st >> 8;
   addr[1] = st;
   printf_P( PSTR( "3c - addr %04X" ), st );

   st = readLoop( 11 );
   addr[2] = st >> 8;
   addr[3] = st;
   printf_P( PSTR( "%04X" ), st );

   st = readLoop( 12 );
   addr[4] = st >> 8;
   addr[5] = st;
   printf_P( PSTR( "%04X\n" ), st );

   // set the base I/O addresse to 0x310.
   // activate the adapter at the preconfigured I/O base address and return
   // to the ID_WAIT state.
   outportb100( 0xE0 + ( ( 0x310 - 0x200 ) >> 4 ) );    // 0F1H ->> 310H
   nop(  );
   outportb100( 0x00 );
   printf_P( PSTR( "3c - setio\n" ) );

   delay_100ms(  );
   st = 32000;
   // select_register_window(0); // window 0 already active after reset
   // wait while card is busy
   while ( ( ( readreg( 0x2E ) ) & 0x1000 ) && st-- )
      ;
   // read some informations about the NIC
   st = readreg( 0x20 );        // Manufacturer ID
   // printf("MfgtID: %04X\n", st);
   if ( st != 0x6D50 )          // 3Com ?
   {
      printf_P( PSTR( "3c - NotFoundMfgtID %04X\n" ), st );

      goto Fin;        // not 3com
   }

   // product ID
   st = readreg( 0x22 );
   // printf_P(PSTR("Product ID: %04X\n"), st);
   if ( st != 0x9450 )
   {
      printf_P( PSTR( "3c - NotFoundProdID %04X\n" ), st );
      goto Fin;        // no 3c5x9B
   }
// ------ the sequence of the following code is important -- keep it
   // Enable Adapter
   setreg( readreg( 0x24 ) | 0x01, 0x24 );

   // Transmitter reset
   tx_reset(  );                // take a little break
   delay_100ms(  );

   uint16_t to = 32000;         // init timeout max value
   do  // while card is busy resetting tx
   {
      st = readreg( 0x2E );
   }
   while ( ( st & 0x1000 ) && to-- );

// Receiver reset
   rx_reset(  );                // take a little break
   delay_100ms(  );
   to = 32000;         // init timeout max value
   do  // while card is busy resetting rx
   {
      st = readreg( 0x2E );
   }
   while ( ( st & 0x1000 ) && to-- );

   set_interrupt_mask( 0 );     // disable all interrupts
   set_read_zero_mask( 0xFF );  // P40 (6-8) - mask NOTHING

   /*
    * Receive packet filter
    * own MAC address and broadcasts
    */
   set_rx_filter( 0x05 );       // broadcast + individual
   printf_P( PSTR( "3c - set_rx_filter\n" ) );

   /*
    * Set MAC hardware address
    */
   // Switch to command window 2 (Station address management)
   select_register_window( 2 );
   outportb31x( 0x25, addr[5] );
   outportb31x( 0x24, addr[4] );
   outportb31x( 0x23, addr[3] );
   outportb31x( 0x22, addr[2] );
   outportb31x( 0x21, addr[1] );
   outportb31x( 0x20, addr[0] );


   // Enable Link Beat and Jabber
   select_register_window( 4 );
   st = readreg( 0x2A );
   st |= ( 0x80 /* link beat */  | 0x40 /* jabber */  );
   setreg( st, 0x2A );

   // activate sender and receiver
   tx_enable(  );
   rx_enable(  );
   initOK = ( to > 0 );
   select_register_window( 1 ); // select "Operating Set"
#if 0
   if(initOK)         // then enable RX complete interrupt
   {
     portENTER_CRITICAL(  );
     {
#if (NIC_INT < INT4)    // The low level of INTn generates an interrupt request
       DDRD  &= ~(1<<NIC_INT);  // input
       PORTD |=  (1<<NIC_INT);  // pullup on
       // Interrupt at the rising edge of the signal
       EICRA &= ~((1<<ISC_NICINT_1)|(1<<ISC_NICINT_0));
#else
       DDRE  &= ~(1<<NIC_INT);  // input
       PORTE |=  (1<<NIC_INT);  // pullup on
       // Interrupt at the rising edge of the signal
       EICRB |= ((1<<ISC_NICINT_0)|(1<<ISC_NICINT_0));
#endif
       EIMSK |= (1<<NIC_INT);      // Enable interrupt

       set_interrupt_mask(0x10);   // enable rx-complete interrupt
     }
     portEXIT_CRITICAL(  );
   }
#endif
 Fin:
   printf_P( PSTR( "3c - initOK=%d\n" ), initOK );
}

unsigned char nic_transmit_ready(unsigned int framesize)
{
   if ( !initOK )
      return 0;

   uint16_t i, j;
   static int16_t tx_start_threshold = 0;

//i3cFlag = (uint8_t)tx_start_threshold;
   uint8_t v, sreg;
   sreg = SREG;
   cli();
   //portENTER_CRITICAL(  );
   {

     v = inportb31x( 0x2B );      // TX Status
     if ( v & (uint8_t)0x30 ) // jabber or underrun
     {
        tx_reset(  );
        printf_P( PSTR( "3c - TX Status: Jabber or Underrung - TX reset!\n" ) );
        // wait until reset complete
        j = 0;
        do
        {
           i = readreg( 0x2E );
        }
        while ( ( i & 0x1000 ) && ( j++ < 10000 ) );

        tx_enable(  );
     }
   }
   //portEXIT_CRITICAL(  );
   SREG = sreg;
//printf_P(PSTR("try send %d\n"),uip_len);
   if ( framesize < 54 )
      framesize = 54;

   // calculate tx start threshold (len/4 + len/16 + len/32 + len/64 + "static" threshold)
   i = (framesize >> 2);
   i += (i >> 2)+(i >> 3)+(i >> 4);
   i += tx_start_threshold;

   //portENTER_CRITICAL(  );
   cli();
   {
     j = readreg( 0x2C );      // free tx buffer bytes
   }
   SREG = sreg;
      //portEXIT_CRITICAL(  );
   if( j < i+14 )
       return 0;

   set_tx_start_threshold( i );

   return 1; // OK
}

void nic_write_frame(uint8_t *uip_buf, uint16_t uip_len)
{
   if ( !initOK )
      return;

   uint16_t i, j;
   uint8_t v;
   uint8_t *cp;
   uint8_t sreg;
   static int16_t tx_start_threshold = 0;

 re_send:

//i3cFlag = (uint8_t)tx_start_threshold;
   sreg = SREG;
   cli();
   //portENTER_CRITICAL(  );
   {

     v = inportb31x( 0x2B );      // TX Status
     if ( v & (uint8_t)0x30 ) // jabber or underrun
     {
        tx_reset(  );
        printf_P( PSTR( "3c - TX Status: Jabber or Underrung - TX reset!\n" ) );
        // wait until reset complete
        j = 0;
        do
        {
           i = readreg( 0x2E );
        }
        while ( ( i & 0x1000 ) && ( j++ < 10000 ) );

        tx_enable(  );
     }
   }
   //portEXIT_CRITICAL(  );
   SREG = sreg;
//printf_P(PSTR("try send %d\n"),uip_len);
   if ( uip_len < 54 )
      uip_len = 54;

   // calculate tx start threshold (len/4 + len/16 + len/32 + len/64 + "static" threshold)
   i = (uip_len >> 2);
   i += (i >> 2)+(i >> 3)+(i >> 4);
   i += tx_start_threshold;

   // Wait for buffer space
   do
   {
      //portENTER_CRITICAL(  );
      cli();
      {
        j = readreg( 0x2C );      // free tx buffer bytes
      }
      SREG = sreg;
      //portEXIT_CRITICAL(  );
   } while( j < i+14 );

   set_tx_start_threshold( i );

   // Write the preamble
   *( ( unsigned * ) &tx_preamble[0] ) = uip_len;       // packet length
   cp = &tx_preamble[0];
   j = 0;
   //portENTER_CRITICAL(  );
   cli();
   {
     WR_PASSIVE( 0x20 );

     while ( j < sizeof( tx_preamble ) )
     {
      DATA_OUT = *cp++;         // tx_preamble[j];
      WR_ACTIVE_0( 0x20 );      // DMA port in window 1
      j++;                      // this works for my "delay"
      WR_PASSIVE( 0x20 );
     }

   // Write the packet data

//  initiate transfer(); DONE ===================================================

     cp = &uip_buf[0];
     j = 0;
     while ( j < 54 )
     {
      DATA_OUT = *cp++;         // uip_buf[j];
      WR_ACTIVE_0( 0x20 );
      j++;
      WR_PASSIVE( 0x20 );
     }
     if ( uip_len > 54 )
     {
      i = uip_len - 54;
#if 0
      //cp = (char *)&uip_appdata[0];
      if(uip_PROGDATA)       // PROGMEM data
      {
        while( i )
        {
         DATA_OUT = pgm_read_byte(&cp[0]);      // uip_appdata[j];
         WR_ACTIVE_0( 0x20 );
         cp++;
         --i;
         WR_PASSIVE( 0x20 );
        }
      }
      else
#endif
      {
        while( i )
        {
         DATA_OUT = *cp++;      // uip_appdata[j];
         WR_ACTIVE_0( 0x20 );   //
         --i;
         WR_PASSIVE( 0x20 );
        }
      }
      j = uip_len;
     }
     // send data; DONE
     // ================================
     // Write padding.
     while ( j & 3 )
     {
      WR_ACTIVE_0( 0x20 );
      j++;
      WR_PASSIVE( 0x20 );
     }
     DATA_OUT = 0;
   }
   SREG = sreg;
   // portEXIT_CRITICAL(  );

   // check for jabber or underrun
   //portENTER_CRITICAL(  );
   cli();
   {
     v = inportb31x( 0x2B );      // TX Status
     if ( v & (uint8_t)0x30 ) // jabber or underrun
     {
        tx_reset(  );
//        printf_P( PSTR( "3c - TX Status: Jabber or Underrung - TX reset!\n" ) );
        // wait until reset complete
        j = 0;
        do
        {
           cli();
           i = readreg( 0x2E );
           SREG = sreg;
        }
        while ( ( i & 0x1000 ) && ( j++ < 10000 ) );
        cli();
        tx_enable(  );
     }
   }
   SREG = sreg;
   //portEXIT_CRITICAL(  );
   if( v & (uint8_t)0x30 )
   {
      if( v & (uint8_t)0x10 ) // underrun , correct TX start threshold
      {
        tx_start_threshold++;
      }
      goto re_send;
   }

   return;
}


#if 0
uint16_t nic_poll(uint16_t len)
{
   uint16_t j;

    if ( len > UIP_BUFSIZE )
    {
     //portENTER_CRITICAL(  );
     cli();
     {
      rx_discard_top_packet(  );        // discard oversized packets
     }
     SREG = sreg;
     //portEXIT_CRITICAL(  );
     goto rx_again;
    }

    char *cp = &uip_buf[0];
    j = len;
    //portENTER_CRITICAL(  );
    cli();
    {
      RD_PASSIVE( 0x20 );
      RD_ACTIVE_0( 0x20 );

      j--;
      while ( j )
      {
        RD_PASSIVE( 0x20 );
        *cp++ = DATA_IN;
        --j;
        RD_ACTIVE_0( 0x20 );
      }
      RD_PASSIVE( 0x20 );
      *cp = DATA_IN;

      rx_discard_top_packet(  );   // packet done !

      //EIMSK |= (1<<NIC_INT);       // Enable interrupt
      //set_interrupt_mask(0x10);    // enable rx-complete interrupt
    }
    SREG = sreg;
    //portEXIT_CRITICAL(  );

    return len;
}
#endif

uint16_t nic_get_RX_state(void)
{
  uint16_t j;

   if ( !initOK )
      return 0;

    uint8_t sreg = SREG;
// rx_again:

   /*
    * read the packet
    */
    cli();
    //portENTER_CRITICAL(  );
    {
      j = readreg( 0x28 );       // RX status
    }
    SREG = sreg;
    //portEXIT_CRITICAL(  );
    if ( j & 0x8000 )          // RX packet is incomplete or RX FIFO empty
      return 0;

    if ( ( j & 0x4000 ) != 0 ) // error in RX packet
    {
      if ( (j = 0x3800) == 0 )
      {
         printf_P( PSTR( "\n =====> Overrun <====\n" ) );

      }
      else
      {
         j &= 0x3800;         // isolate error information
         if ( j == 0x1800 )
            printf_P( PSTR( "\n =====> Runt packet error <====\n" ) );
         else if ( j == 0x2000 )
            printf_P( PSTR( "\n =====> Alignment error <====\n" ) );
         else if ( j & 0x2800 )
            printf_P( PSTR( "\n =====> CRC error <====\n" ) );
         else if ( j & 0x800 )
            printf_P( PSTR( "\n =====> Oversized <====\n" ) );

        //portENTER_CRITICAL(  );
        cli();
        {
         rx_discard_top_packet(  );     // discrard this packet
        }
        SREG = sreg;
        //portEXIT_CRITICAL(  );
        return 0;
      }
    }

    /*
     * Obtain the size of the packet and put it into the "len" variable.
     */
    j &= 0x7FF;           // isolate data length

    return j > 0;
}


uint16_t nic_get_RX_len(void)
{
   uint16_t j, len;

   if ( !initOK )
      return 0;

    uint8_t sreg = SREG;
 rx_again:

   /*
    * read the packet
    */
    cli();
    //portENTER_CRITICAL(  );
    {
      len = readreg( 0x28 );       // RX status
    }
    SREG = sreg;
    //portEXIT_CRITICAL(  );

    if ( len & 0x8000 )          // RX packet is incomplete or RX FIFO empty
      return 0;

    if ( ( len & 0x4000 ) != 0 ) // error in RX packet
    {
      j = len & 0x3800;         // isolate error information
      if ( j == 0 )
      {
         printf_P( PSTR( "\n =====> Overrun <====\n" ) );
      }
      else
      {
         if ( j == 0x1800 )
            printf_P( PSTR( "\n =====> Runt packet error <====\n" ) );
         else if ( j == 0x2000 )
            printf_P( PSTR( "\n =====> Alignment error <====\n" ) );
         else if ( j & 0x2800 )
            printf_P( PSTR( "\n =====> CRC error <====\n" ) );
         else if ( j & 0x800 )
            printf_P( PSTR( "\n =====> Oversized <====\n" ) );

        //portENTER_CRITICAL(  );
        cli();
        {
         rx_discard_top_packet(  );     // discrard this packet
        }
        SREG = sreg;
        //portEXIT_CRITICAL(  );
        goto rx_again;
      }
    }

    /*
     * Obtain the size of the packet and put it into the "len" variable.
     */
    len = len & 0x7FF;           // isolate data length
//printf_P(PSTR("packet, %d bytes\n"), len);
    if ( ( len == 0 ) )// || ( len == 0x7FF ) || ( len == 0x7FE ) )
    {
#if 0
       portENTER_CRITICAL(  );
       {
         EIMSK |= (1<<NIC_INT);        // Enable interrupt
         set_interrupt_mask(0x10);     // enable rx-complete interrupt
       }
       portEXIT_CRITICAL(  );
#endif
//       return 0;
    }
    return len;
}

static unsigned char rxd_eth_addr[6];


void DEVICE_READ_FRAME_DUMMY(uint16_t j)
{
    //portENTER_CRITICAL(  );
    uint8_t sreg = SREG;
    cli();
    {
//      RD_PASSIVE( 0x20 );
      RD_ACTIVE_0( 0x20 );

      j--;
      while ( j )
      {
        RD_PASSIVE( 0x20 );
//        *cp++ = DATA_IN;
        --j;
        RD_ACTIVE_0( 0x20 );
      }
      RD_PASSIVE( 0x20 );
//      *cp = DATA_IN;

    }
    SREG = sreg;
    //portEXIT_CRITICAL(  );
}

void DEVICE_READ_FRAME_DATA(uint8_t *cp, uint16_t j)
{
    //portENTER_CRITICAL(  );
    uint8_t sreg = SREG;
    cli();
    {
      RD_PASSIVE( 0x20 );
      RD_ACTIVE_0( 0x20 );

      j--;
      while ( j )
      {
        RD_PASSIVE( 0x20 );
        *cp++ = DATA_IN;
        --j;
        RD_ACTIVE_0( 0x20 );
      }
      RD_PASSIVE( 0x20 );
      *cp = DATA_IN;

    }
    SREG = sreg;
    //portEXIT_CRITICAL(  );
}

uint16_t DEVICE_READ_FRAME_16BIT_LE(void)
{
    union {
       uint16_t w;
       uint8_t b[2];
       } r;

    //portENTER_CRITICAL(  );
    uint8_t sreg = SREG;
    cli();
    {
//      RD_PASSIVE( 0x20 );
      RD_ACTIVE_0( 0x20 );
      RD_PASSIVE( 0x20 );
      r.b[0] = DATA_IN;
      RD_ACTIVE_0( 0x20 );
      RD_PASSIVE( 0x20 );
      r.b[1] = DATA_IN;
    }
    SREG = sreg;
    //portEXIT_CRITICAL(  );
    return r.w;
}

uint16_t DEVICE_READ_FRAME_16BIT(void)
{
    union {
       uint16_t w;
       uint8_t b[2];
       } r;

    //portENTER_CRITICAL(  );
    uint8_t sreg = SREG;
    cli();
    {
//      RD_PASSIVE( 0x20 );
      RD_ACTIVE_0( 0x20 );
      RD_PASSIVE( 0x20 );
      r.b[1] = DATA_IN;
      RD_ACTIVE_0( 0x20 );
      RD_PASSIVE( 0x20 );
      r.b[0] = DATA_IN;
    }
    SREG = sreg;
    //portEXIT_CRITICAL(  );
    return r.w;
}

#define read_rx_mac()  DEVICE_READ_FRAME_DATA(&rxd_eth_addr[0], sizeof(rxd_eth_addr))

void ethernet_process_frame(unsigned char broadcast_flag, uint16_t frame_length)
{
   unsigned int ip_frame_type;

#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_ETHERNET_S)
#endif

   DEVICE_READ_FRAME_DATA(mac_of_last_frame, sizeof(mac_of_last_frame));
   ip_frame_type = DEVICE_READ_FRAME_16BIT();
#ifdef WEB_DEBUG_ETHERNET
   if (web_debug_flag & WEB_DEBUG_ETHERNET_BIT)
   {
      printf_P(PSTR("ETHERNET FRAME-Typ: 0x%04x, %uByte"), ip_frame_type,
             frame_length);
      if (broadcast_flag)
         printf_P(PSTR(" Broadcast\rn"));
      else
         printf_P(PSTR("\n"));
      debug_print_mac(PSTR("ETHERNET MAC"), rxd_eth_addr);
   }
#endif
   switch (ip_frame_type)
   {
#if defined(ARP_ENABLE) || defined(RARP_ENABLE)
   case ARP_FRAME:
   case RARP_FRAME:
      arp_rarp_process_frame(broadcast_flag);
      break;
#endif
#ifdef IP_ENABLE
   case IP_FRAME:
      ip_process_frame(broadcast_flag);
      break;
#endif
   default:
#ifdef WEB_DEBUG_ETHERNET
      if (web_debug_flag & WEB_DEBUG_ETHERNET_BIT)
         printf_P(PSTR("ETHERNET unsupported FRAME-Typ: 0x%04x\r\n"), ip_frame_type);
#endif
      break;
   }
   rx_discard_top_packet(  );   // packet done !

#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_ETHERNET_E);
#endif
}


void ethernet_process(void)
{
   unsigned int rx_len;

#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_NET_S)
#endif
   rx_len = nic_get_RX_len();

   if (rx_len > 14)
   {
      read_rx_mac();
      // is broadcast ?
      if(strncmp_P(rxd_eth_addr, PSTR("\0377\0377\0377\0377\0377\0377"), sizeof(rxd_eth_addr)) == 0)
         ethernet_process_frame(1, rx_len);  // broadcast
      else
         ethernet_process_frame(0, rx_len);
   }
#ifdef ARP_ENABLE
   arp_request_timer(ARP_REQUEST_TIMER_CONTINUE);
#endif
#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_NET_E)
#endif
}


#endif // #if defined(ETHERNET_ENABLE) && defined(DEVICE_3c509b)
