
/*****************************************************************************
 *
 *                           "cs8900.c"
 *                   -----------------------------
 *
 *  Version:    2.04
 *  File:     	..\..\cs8900.c
 *  Created:    10.02.2003
 *  Date:       31.08.2004
 *  Author:     Copyright (C) 2001-2004
 *              Udo Jakobza - FTZ Leipzig; D-04107 Leipzig; Wchterstr. 13
 *				info@easytoweb.net
 *  Func:		ethernet packet-driver for use with LAN-
 *				controller CS8900 from Crystal/Cirrus Logic
 *  license:
 *    This library is free software; you can redistribute it and/or modify it
 *    under the terms of the GNU Lesser General Public License as published by
 *    the Free Software Foundation; either version 2.1 of the License, or
 *    (at your option) any later version. This library is distributed in the hope
 *    that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *    See the GNU Lesser General Public License for more details.
 *	  see: http://www.gnu.org/copyleft/lesser.html
 *    You should have received a copy of the GNU Lesser General Public License
 *    along with this library; if not, write to the Free Software Foundation,
 *    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *    Die Bibliothek ist freie Software; Sie drfen sie unter den Bedingungen der
 *    GNU Lesser General Public License, wie von der Free Software Foundation
 *    verffentlicht, weiterverteilen und/oder modifizieren; entweder gem
 *    Version 2.1 der Lizenz oder (nach Ihrer Option) jeder spteren Version.
 *    Diese Bibliothek wird in der Hoffnung weiterverbreitet, da sie ntzlich
 *    sein wird, jedoch OHNE IRGENDEINE GARANTIE, auch ohne die implizierte
 *    Garantie der MARKTREIFE oder der VERWENDBARKEIT FR EINEN BESTIMMTEN ZWECK.
 *    Mehr Details finden Sie in der GNU Lesser General Public License.
 *	  see: http://www.gnu.org/copyleft/lesser.de.html
 *    Sie sollten eine Kopie der GNU Lesser General Public License zusammen mit
 *    dieser Bibliothek erhalten haben; falls nicht, schreiben Sie an die FSF,
 *    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 *  history:
 *		1.) 08.08.2003 Version 2.00
 *		2.) 26.08.2003 Version 2.01
 *			a.)	- CS8900-Interface supports now "CS8900_IO_PORT8", "CS8900_IO_PORT16"
 *				"CS8900_I2C_HW_8", "CS8900_I2C_HW_16", "CS8900_I2C_SW_8" and "CS8900_I2C_SW_16"
 *		3.) 01.06.2004 Version 2.04
 *			a.)	- new functions:
 *                 "cs8900_rx_skip()" - Skip-Command, see CS8900-PacketPage-Register RxCFG (0x102)
 *                 "cs8900_collision_counter()" - reads the TX-Collision-Counter, see CS8900-PacketPage-Register TxCOL (0x132)
 *                 "cs8900_miss_counter()" - reads the RX-Miss-Counter, see CS8900-PacketPage-Register RxMISS (0x130)
 *		4.) 12.08.2004 Version 2.04
 *			a.)	- function "cs8900_init()" now with timeout (400ms);
 *
 *****************************************************************************/
#include "project.h"

#ifdef DEVICE_CS8900

#if defined(CS8900_I2C_SW_8)||defined(CS8900_I2C_SW_16)
#ifdef __CODEVISIONAVR__
#asm
.equ __i2c_port = 0x15.equ __sda_bit = 1.equ __scl_bit = 0
#endasm
#endif
#include <i2c.h>
#endif
#include <stdio.h>
#ifdef __CODEVISIONAVR__
#include <io.h>
#include <delay.h>
#else
#define nop()  __asm__ volatile ("nop" ::)
#include <avr/io.h>
#include <avr/delay.h>
extern void delay_ms(unsigned int a);
#define delay_us(a) _delay_us(a)
#include <avr/wdt.h>
#include <avr/pgmspace.h>
#endif
#include CS8900_H_FILEPATH
#include DEVICE_H_FILEPATH
#ifdef WEB_DEBUG
#include WEB_DEBUG_H_FILEPATH
#endif

CS8900_INIT_DATA cs8900_init_data[5] =
{
   {CS8900_PP_IA,  MAC_DEFAULT_1 | (MAC_DEFAULT_2 << 8) },
   {CS8900_PP_IA + 2, MAC_DEFAULT_3 | (MAC_DEFAULT_4 << 8)},
   {CS8900_PP_IA + 4, MAC_DEFAULT_5 | (MAC_DEFAULT_6 << 8)},
   {CS8900_PP_LINE_CTL, SERIAL_RX_ON | SERIAL_TX_ON},
#ifdef CS8900_TX_PADDING_DISABLE
   {CS8900_PP_RX_CTL, RX_RUNT_ACCEPT | RX_IA_ACCEPT | RX_BROADCAST_ACCEPT}};
#else
   {CS8900_PP_RX_CTL, RX_OK_ACCEPT | RX_IA_ACCEPT | RX_BROADCAST_ACCEPT}};
#endif

#ifdef CS8900_IO_PORT8
void _cs8900_w_port8(unsigned char address, unsigned char data);
unsigned char _cs8900_r_port8(unsigned char address);
#endif

#ifdef CS8900_IO_PORT16
void _cs8900_w_port16(unsigned char address, unsigned int data);
unsigned int _cs8900_r_port16(unsigned char address, ENDIAN_MODE mode);
#endif

#if defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_HW_16)
void i2c_write_hw(unsigned char i2c_device, unsigned char address_data);
unsigned char i2c_read_hw(unsigned char i2c_device);
void i2c_wait(void);
void i2c_send(void);
#endif

#ifdef CS8900_I2C_HW_8
void _cs8900_w_i2c_hw8(unsigned char address, unsigned char data);
unsigned char _cs8900_r_i2c_hw8(unsigned char address);
#endif

#ifdef CS8900_I2C_HW_16
void _cs8900_w_i2c_hw16(unsigned char address, unsigned int data);
unsigned int _cs8900_r_i2c_hw16(unsigned char address, ENDIAN_MODE mode);
#endif

#if defined(CS8900_I2C_SW_8)||defined(CS8900_I2C_SW_16)
void i2c_write_sw(unsigned char i2c_device, unsigned char address_data);
unsigned char i2c_read_sw(unsigned char i2c_device);
#endif

#ifdef CS8900_I2C_SW_8
void _cs8900_w_i2c_sw8(unsigned char address, unsigned char data);
unsigned char _cs8900_r_i2c_sw8(unsigned char address);
#endif

#ifdef CS8900_I2C_SW_16
void _cs8900_w_i2c_sw16(unsigned char address, unsigned int data);
unsigned int _cs8900_r_i2c_sw16(unsigned char address, ENDIAN_MODE mode);
#endif

void cs8900_set_mac(unsigned char *new_mac)
{
   cs8900_init_data[0].data = new_mac[0] | ((unsigned int) new_mac[1] << 8);
   cs8900_init_data[1].data = new_mac[2] | ((unsigned int) new_mac[3] << 8);
   cs8900_init_data[2].data = new_mac[4] | ((unsigned int) new_mac[5] << 8);
}


unsigned char cs8900_init(void)
{
   unsigned char Lva = 0;

   CS8900_RESET_L;
   CS8900_RESET_OUTPUT;
   CS8900_SLEEP_H;
   CS8900_SLEEP_OUTPUT;

   CS8900_TRAFFIC_RECEIVE_H;
   CS8900_TRAFFIC_RECEIVE_OUTPUT;
   CS8900_TRAFFIC_TRANSMIT_H;
   CS8900_TRAFFIC_TRANSMIT_OUTPUT;

#if defined(CS8900_IO_PORT8)||defined(CS8900_IO_PORT16)||\
    defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_8)||defined(CS8900_I2C_SW_16)
   CS8900_SBHE_H;
   CS8900_SBHE_OUTPUT;
   CS8900_IOR_H;
   CS8900_IOR_OUTPUT;
   CS8900_IOW_H;
   CS8900_IOW_OUTPUT;
#endif

#if defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)||defined(CS8900_IO_PORT16)
   CS8900_SBHE_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_SBHE_H;
#endif

#if defined(CS8900_IO_PORT8)||defined(CS8900_IO_PORT16)
   CS8900_ADR_DDR;
   CS8900_DATA_INPUT;
   CS8900_DATA_W = 0xff;
#endif

#ifdef CS8900_IO_PORT16
   CS8900_DATA16_INPUT;
   CS8900_DATA16_W = 0xff;
#endif

#if defined(CS8900_I2C_SW_8)||defined(CS8900_I2C_SW_16)
   i2c_init();
#endif
#if defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_HW_16)
   TWCR = 0x01;
   TWSR = I2C_TWPS1 | I2C_TWPS0;
   TWBR = 0x03;
#endif

   cs8900_software_wakeup();
   if ((cs8900_read_io_reg16_le(IO_PORT_ADD) & 0x3000) == 0)
   {
#ifdef WEB_DEBUG_CS8900
      if (web_debug_flag & WEB_DEBUG_CS8900_BIT)
         printf_P(PSTR("CS8900 IObase+0Ah: 0x%04x\r\n"),
                cs8900_read_io_reg16_le(IO_PORT_ADD));
#endif
      return 0;
   }
   cs8900_software_reset();
   delay_ms(10);
   cs8900_write_io_reg16(IO_PORT_ADD, CS8900_PP_SELF_ST);
   while (!(cs8900_read_io_reg16_le(IO_PORT_DATA) & SELF_ST_INITD))
   {
      delay_ms(100);
      Lva++;
      if (Lva > 4)
      {
#ifdef WEB_DEBUG_CS8900
         if (web_debug_flag & WEB_DEBUG_CS8900_BIT)
            printf_P(PSTR("CS8900 CS8900_PP_SELF_ST: 0x%04x\r\n"),
                   cs8900_read_io_reg16_le(IO_PORT_DATA));
#endif
         return 0;
      }
   }
   for (Lva = 0; Lva < sizeof(cs8900_init_data) / sizeof(CS8900_INIT_DATA); Lva++)
      cs8900_write_packetpage16(cs8900_init_data[Lva].addr,
                                cs8900_init_data[Lva].data);
   return 1;
}


void cs8900_write_io_reg16(unsigned char address, unsigned int data)
{
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
   ETHER_IO_W(address, LO(data));
   ETHER_IO_W(address + 1, HI(data));
#endif
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
   ETHER_IO_W(address, data);
#endif
}


unsigned int cs8900_read_io_reg16_le(unsigned char address)
{
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
   unsigned char value_low;
   unsigned int value_high;

   value_low = ETHER_IO_R(address);
   value_high = ETHER_IO_R(address + 1);
   value_high <<= 8;
   value_high |= value_low;
   return value_high;
#endif

#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
   return ETHER_IO_R_LE(address);
#endif
}

unsigned int cs8900_read_io_reg16_be(unsigned char address)
{
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
   unsigned int value;

   value = ETHER_IO_R(address);
   value <<= 8;
   value |= ETHER_IO_R(address + 1);
   return value;
#endif
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
   return ETHER_IO_R_BE(address);
#endif
}

unsigned int cs8900_read_io_reg16_HB1ST(unsigned char address)
{
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
   unsigned int value;

   value = ETHER_IO_R(address + 1);
   value <<= 8;
   value |= ETHER_IO_R(address);
   return value;
#endif
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
   return ETHER_IO_R_LE(address);
#endif
}

void cs8900_write_packetpage16(unsigned int address, unsigned int data)
{
   cs8900_write_io_reg16(IO_PORT_ADD, address);
   cs8900_write_io_reg16(IO_PORT_DATA, data);
}

unsigned int cs8900_read_packetpage16_le(unsigned int address)
{
   cs8900_write_io_reg16(IO_PORT_ADD, address);
   return cs8900_read_io_reg16_le(IO_PORT_DATA);
}

unsigned int cs8900_read_packetpage16_be(unsigned int address)
{
   cs8900_write_io_reg16(IO_PORT_ADD, address);
   return cs8900_read_io_reg16_be(IO_PORT_DATA);
}


void cs8900_write_frame(void *source, unsigned int byte_size)
{
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
   unsigned char *p_data;
#endif
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
   unsigned int *p_data;
#endif

   CS8900_TRAFFIC_TRANSMIT_L;
   p_data = source;
   while (byte_size > 1)
   {
      ETHER_IO_W(IO_PORT_TX_RX, *p_data);
      p_data++;
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
      ETHER_IO_W(IO_PORT_TX_RX + 1, *p_data);
      p_data++;
#endif
      byte_size -= 2;
//#asm("wdr");
   wdt_reset();
   }
   if (byte_size)
   {
      ETHER_IO_W(IO_PORT_TX_RX, *p_data);
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
      p_data++;
      ETHER_IO_W(IO_PORT_TX_RX + 1, *p_data);
#endif
   }
   CS8900_TRAFFIC_TRANSMIT_H;
}


void cs8900_read_frame(void *dest, unsigned int byte_size)
{
   unsigned char dummy;
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
   unsigned char *p_data;
#endif
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
   unsigned int *p_data;
#endif

   CS8900_TRAFFIC_RECEIVE_L;
   p_data = dest;
   while (byte_size > 1)
   {
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
      *p_data = ETHER_IO_R(IO_PORT_TX_RX);
      p_data++;
      *p_data = ETHER_IO_R(IO_PORT_TX_RX + 1);
#endif
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
      *p_data = ETHER_IO_R_LE(IO_PORT_TX_RX);
#endif
      p_data++;
      byte_size -= 2;
#ifdef __CODEVISIONAVR__
#asm("wdr");
#else
   wdt_reset();
#endif
   }
   if (byte_size)
   {
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
      *p_data = ETHER_IO_R(IO_PORT_TX_RX);
#ifdef __CODEVISIONAVR__
#pragma warn-
#endif
      dummy = ETHER_IO_R(IO_PORT_TX_RX + 1);

#ifdef __CODEVISIONAVR__
#pragma warn+
#endif
#endif
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
      *p_data = ETHER_IO_R_LE(IO_PORT_TX_RX);
#endif
   }
   CS8900_TRAFFIC_RECEIVE_H;
}


void cs8900_read_dummyframe(unsigned int size)
{
   unsigned char dummy;
   while (size > 1)
   {
      CS8900_TRAFFIC_RECEIVE_L;
#ifdef __CODEVISIONAVR__
#pragma warn-
#endif
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
      ETHER_IO_R_LE(IO_PORT_TX_RX);
#endif
#if defined(CS8900_MEMORY)||defined(CS8900_IO_PORT8)||defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_SW_8)
      dummy = ETHER_IO_R(IO_PORT_TX_RX);
      dummy = ETHER_IO_R(IO_PORT_TX_RX + 1);
#endif
#ifdef __CODEVISIONAVR__
#pragma warn+
#endif
      size -= 2;
#ifdef __CODEVISIONAVR__
#asm("wdr");
#else
      wdt_reset();
#endif
      CS8900_TRAFFIC_RECEIVE_H;
   }
}

unsigned int cs8900_transmit_ready(unsigned int frame_size)
{
#ifdef CS8900_TX_PADDING_DISABLE
   cs8900_write_io_reg16(IO_PORT_TX_CMD,
                         TX_CMD_PAD_DIS | TX_CMD_START_ALL_BYTES);
#else
   cs8900_write_io_reg16(IO_PORT_TX_CMD, TX_CMD_START_ALL_BYTES);
#endif
   cs8900_write_io_reg16(IO_PORT_TX_LEN, frame_size);
   cs8900_write_io_reg16(IO_PORT_ADD, CS8900_PP_BUS_ST);
   return (cs8900_read_io_reg16_le(IO_PORT_DATA) & READY_FOR_TX_NOW);
}

void cs8900_software_sleep(void)
{
   cs8900_write_io_reg16(IO_PORT_ADD, CS8900_PP_SELF_CTL);
   delay_ms(10);
   cs8900_write_packetpage16(CS8900_PP_SELF_CTL, SELF_CTL_SW_SUSPEND);
}

void cs8900_software_wakeup(void)
{
   cs8900_write_io_reg16(IO_PORT_ADD, CS8900_PP_SELF_CTL);
   delay_ms(10);
}

void cs8900_software_reset(void)
{
   cs8900_write_packetpage16(CS8900_PP_SELF_CTL, SELF_CTL_RESET);
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
   CS8900_SBHE_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_SBHE_H;
#endif
}


void cs8900_hardware_reset(void)
{
   CS8900_RESET_H;
   delay_us(20);
   CS8900_RESET_L;
#if defined(CS8900_IO_PORT16)||defined(CS8900_I2C_HW_16)||defined(CS8900_I2C_SW_16)
   CS8900_SBHE_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_SBHE_H;
#endif
}


void cs8900_rx_skip(void)
{
   cs8900_write_packetpage16(CS8900_PP_RX_CFG, SKIP_1);
}


unsigned int cs8900_collision_counter(void)
{
   unsigned int data;

   data = cs8900_read_packetpage16_le(CS8900_PP_TX_COL);
   data >>= 6;
   return data;
}

unsigned int cs8900_miss_counter(void)
{
   unsigned int data;

   data = cs8900_read_packetpage16_le(CS8900_PP_RX_MISS);
   data >>= 6;
   return data;
}


#ifdef CS8900_I2C_SW_8
void _cs8900_w_i2c_sw8(unsigned char address, unsigned char data)
{
   i2c_write_sw(I2C_ADDRESS_EXPANDER & I2C_AND_WRITE_MODE, address);
   i2c_write_sw(I2C_DATA8_EXPANDER & I2C_AND_WRITE_MODE, data);
   CS8900_IOW_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOW_H;
}

unsigned char _cs8900_r_i2c_sw8(unsigned char address)
{
   unsigned char data = 0x00;

   i2c_write_sw(I2C_ADDRESS_EXPANDER & I2C_AND_WRITE_MODE, address);
   CS8900_IOR_L;
   data = i2c_read_sw(I2C_DATA8_EXPANDER | I2C_OR_READ_MODE);
   CS8900_IOR_H;
   return data;
}
#endif


#ifdef CS8900_I2C_SW_16
void _cs8900_w_i2c_sw16(unsigned char address, unsigned int data)
{
   CS8900_SBHE_L;
   i2c_write_sw(I2C_ADDRESS_EXPANDER & I2C_AND_WRITE_MODE, address);
   i2c_write_sw(I2C_DATA16_EXPANDER & I2C_AND_WRITE_MODE, data >> 8);
   i2c_write_sw(I2C_DATA8_EXPANDER & I2C_AND_WRITE_MODE, data);
   CS8900_IOW_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOW_H;
   CS8900_SBHE_H;
}

unsigned int _cs8900_r_i2c_sw16(unsigned char address, ENDIAN_MODE mode)
{
   unsigned int data_2;
   unsigned char data_1;

   CS8900_SBHE_L;
   i2c_write_sw(I2C_ADDRESS_EXPANDER & I2C_AND_WRITE_MODE, address);
   CS8900_IOR_L;
   if (mode == LITTLE_ENDIAN)
   {
      data_2 = i2c_read_sw(I2C_DATA16_EXPANDER | I2C_OR_READ_MODE);
      data_1 = i2c_read_sw(I2C_DATA8_EXPANDER | I2C_OR_READ_MODE);
   }
   else
   {
      data_1 = i2c_read_sw(I2C_DATA16_EXPANDER | I2C_OR_READ_MODE);
      data_2 = i2c_read_sw(I2C_DATA8_EXPANDER | I2C_OR_READ_MODE);
   }
   CS8900_IOR_H;
   CS8900_SBHE_H;
   data_2 <<= 8;
   data_2 |= data_1;
   return data_2;
}
#endif


#ifdef CS8900_I2C_HW_8
void _cs8900_w_i2c_hw8(unsigned char address, unsigned char data)
{
   i2c_write_hw(I2C_ADDRESS_EXPANDER & I2C_AND_WRITE_MODE, address);
   i2c_write_hw(I2C_DATA8_EXPANDER & I2C_AND_WRITE_MODE, data);
   CS8900_IOW_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOW_H;
}

unsigned char _cs8900_r_i2c_hw8(unsigned char address)
{
   unsigned char data = 0x00;

   i2c_write_hw(I2C_ADDRESS_EXPANDER & I2C_AND_WRITE_MODE, address);
   CS8900_IOR_L;
   data = i2c_read_hw(I2C_DATA8_EXPANDER | I2C_OR_READ_MODE);
   CS8900_IOR_H;
   return data;
}
#endif

#ifdef CS8900_I2C_HW_16
void _cs8900_w_i2c_hw16(unsigned char address, unsigned int data)
{
   CS8900_SBHE_L;
   i2c_write_hw(I2C_ADDRESS_EXPANDER & I2C_AND_WRITE_MODE, address);
   i2c_write_hw(I2C_DATA16_EXPANDER & I2C_AND_WRITE_MODE, data >> 8);
   i2c_write_hw(I2C_DATA8_EXPANDER & I2C_AND_WRITE_MODE, data);
   CS8900_IOW_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOW_H;
   CS8900_SBHE_H;
}

unsigned int _cs8900_r_i2c_hw16(unsigned char address, ENDIAN_MODE mode)
{
   unsigned int data_2;
   unsigned char data_1;

   CS8900_SBHE_L;
   i2c_write_hw(I2C_ADDRESS_EXPANDER & I2C_AND_WRITE_MODE, address);
   CS8900_IOR_L;
   if (mode == LITTLE_ENDIAN)
   {
      data_2 = i2c_read_hw(I2C_DATA16_EXPANDER | I2C_OR_READ_MODE);
      data_1 = i2c_read_hw(I2C_DATA8_EXPANDER | I2C_OR_READ_MODE);
   }
   else
   {
      data_1 = i2c_read_hw(I2C_DATA16_EXPANDER | I2C_OR_READ_MODE);
      data_2 = i2c_read_hw(I2C_DATA8_EXPANDER | I2C_OR_READ_MODE);
   }
   CS8900_IOR_H;
   CS8900_SBHE_H;
   data_2 <<= 8;
   data_2 |= data_1;
   return data_2;
}
#endif


#ifdef CS8900_IO_PORT8
void _cs8900_w_port8(unsigned char address, unsigned char data)
{
   CS8900_ADR(address);
   CS8900_DATA_W = data;
   CS8900_DATA_OUTPUT;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOW_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOW_H;
   CS8900_DATA_INPUT;
   CS8900_DATA_W = 0xff;
}

unsigned char _cs8900_r_port8(unsigned char address)
{
   unsigned char data;

   CS8900_ADR(address);
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOR_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#if(_MCU_CLOCK_FREQUENCY_>4000000)
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#endif
#if(_MCU_CLOCK_FREQUENCY_>8000000)
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#endif
#if(_MCU_CLOCK_FREQUENCY_>12000000)
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#endif
#if(_MCU_CLOCK_FREQUENCY_>16000000)
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#endif
   data = CS8900_DATA_R;
   CS8900_IOR_H;
   return data;
}
#endif


#ifdef CS8900_IO_PORT16
void _cs8900_w_port16(unsigned char address, unsigned int data)
{
   CS8900_SBHE_L;
   CS8900_ADR(address);
   CS8900_DATA_W = LO(data);
   CS8900_DATA_OUTPUT;
   CS8900_DATA16_W = HI(data);
   CS8900_DATA16_OUTPUT;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOW_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOW_H;
   CS8900_SBHE_H;
   CS8900_DATA_INPUT;
   CS8900_DATA_W = 0xff;
   CS8900_DATA16_INPUT;
   CS8900_DATA16_W = 0xff;
}


unsigned int _cs8900_r_port16(unsigned char address, ENDIAN_MODE mode)
{
   unsigned int data_2;
   unsigned char data_1;

   CS8900_SBHE_L;
   CS8900_ADR(address);
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
   CS8900_IOR_L;
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#if(_MCU_CLOCK_FREQUENCY_>4000000)
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#endif
#if(_MCU_CLOCK_FREQUENCY_>8000000)
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#endif
#if(_MCU_CLOCK_FREQUENCY_>12000000)
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#endif
#if(_MCU_CLOCK_FREQUENCY_>16000000)
#ifdef __CODEVISIONAVR__
#asm("nop");
#else
    nop();
#endif
#endif
   if (mode == LITTLE_ENDIAN)
   {
      data_1 = CS8900_DATA_R;
      data_2 = CS8900_DATA16_R;
   }
   else
   {
      data_2 = CS8900_DATA_R;
      data_1 = CS8900_DATA16_R;
   }
   CS8900_IOR_H;
   CS8900_SBHE_H;
   data_2 <<= 8;
   data_2 |= data_1;
   return data_2;
}
#endif

#if defined(CS8900_I2C_HW_8)||defined(CS8900_I2C_HW_16)
unsigned char i2c_read_hw(unsigned char i2c_device)
{
   unsigned char data = 0x00;

   TWCR = I2C_TWINT | I2C_TWSTA | I2C_TWEN;
   i2c_wait();
   if (((TWSR & 0xf8) == 0x08) | ((TWSR & 0xf8) == 0x10))
   {
      TWDR = i2c_device;
      i2c_send();
   }
   else
   {
#ifdef WEB_DEBUG_CS8900
      if (web_debug_flag & WEB_DEBUG_CS8900_BIT)
         putsf("CS8900 I2C read-error!!\r");
#endif
      return;
   }
   if ((TWSR & 0xf8) == 0x40)
   {
      TWCR = TWCR & ~I2C_TWEA;
      i2c_wait();
      data = TWDR;
   }
   else
   {
#ifdef WEB_DEBUG_CS8900
      if (web_debug_flag & WEB_DEBUG_CS8900_BIT)
         printf_P(PSTR("CS8900 I2C read-error!!\n"));
#endif
      return;
   }
   if ((i2c_device & I2C_DATA8_EXPANDER) == I2C_DATA8_EXPANDER)
      TWCR = I2C_TWINT | I2C_TWSTO | I2C_TWEN;
   return data;
}


void i2c_write_hw(unsigned char i2c_device, unsigned char address_data)
{
   TWCR = I2C_TWINT | I2C_TWSTA | I2C_TWEN;
   i2c_wait();
   if (((TWSR & 0xf8) == 0x08) | ((TWSR & 0xf8) == 0x10))
   {
      TWDR = i2c_device;
      i2c_send();
   }
   else
   {
#ifdef WEB_DEBUG_CS8900
      if (web_debug_flag & WEB_DEBUG_CS8900_BIT)
         printf_P(PSTR("CS8900 I2C write-error!!\n"));
#endif
      return;
   }
   if ((TWSR & 0xf8) == 0x18)
   {
      TWDR = address_data;
      i2c_send();
   }
   else
   {
#ifdef WEB_DEBUG_CS8900
      if (web_debug_flag & WEB_DEBUG_CS8900_BIT)
         printf_P(PSTR("CS8900 I2C write-error!!\n"));
#endif
      return;
   }
   if ((i2c_device & I2C_DATA8_EXPANDER) == I2C_DATA8_EXPANDER)
      TWCR = I2C_TWINT | I2C_TWSTO | I2C_TWEN;
}

void i2c_wait(void)
{
   while (!(TWCR & I2C_TWINT))
    ;
}

void i2c_send(void)
{
   TWCR = I2C_TWINT | I2C_TWEN;
   i2c_wait();
}
#endif


#if defined(CS8900_I2C_SW_8)||defined(CS8900_I2C_SW_16)
unsigned char i2c_read_sw(unsigned char i2c_device)
{
   unsigned char data;

   if (i2c_start())
   {
      i2c_write(i2c_device);
      data = i2c_read(0);
      if ((i2c_device & I2C_DATA8_EXPANDER) == I2C_DATA8_EXPANDER)
         i2c_stop();
   }
#ifdef WEB_DEBUG_CS8900
   else
   {
      if (web_debug_flag & WEB_DEBUG_CS8900_BIT)
         printf_P(PSTR("CS8900 I2C read-error!!\n"));
   }
#endif
   return data;
}


void i2c_write_sw(unsigned char i2c_device, unsigned char address_data)
{
   if (i2c_start())
   {
      i2c_write(i2c_device);
      i2c_write(address_data);
      if ((i2c_device & I2C_DATA8_EXPANDER) == I2C_DATA8_EXPANDER)
         i2c_stop();
   }
#ifdef WEB_DEBUG_CS8900
   else
   {
      if (web_debug_flag & WEB_DEBUG_CS8900_BIT)
         printf_P(PSTR("CS8900 I2C write-error!!\n"));
   }
#endif
}
#endif

#endif
