
/*****************************************************************************
 *
 *                           "ppp.c"
 *                   -----------------------------
 *
 *  Version:    2.10
 *  File:     	..\..\ppp.c
 *  Created:    09.03.2003
 *  Date:       06.09.2005
 *  Author:     Copyright (C) 2001-2005
 *              Wolfgang Lutsch, Udo Jakobza
 *              FTZ Leipzig; D-04107 Leipzig; Wchterstr. 13
 *				info@easytoweb.net
 *  Func:		opens a serial point-to-point link via standardmodem @ 9600bps
 *				and provides a SAP for a TCP/IP stack
 *  License:
 *    This library is free software; you can redistribute it and/or modify it
 *    under the terms of the GNU Lesser General Public License as published by
 *    the Free Software Foundation; either version 2.1 of the License, or
 *    (at your option) any later version. This library is distributed in the hope
 *    that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *    See the GNU Lesser General Public License for more details.
 *	  see: http://www.gnu.org/copyleft/lesser.html
 *    You should have received a copy of the GNU Lesser General Public License
 *    along with this library; if not, write to the Free Software Foundation,
 *    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *    Die Bibliothek ist freie Software; Sie drfen sie unter den Bedingungen der
 *    GNU Lesser General Public License, wie von der Free Software Foundation
 *    verffentlicht, weiterverteilen und/oder modifizieren; entweder gem
 *    Version 2.1 der Lizenz oder (nach Ihrer Option) jeder spteren Version.
 *    Diese Bibliothek wird in der Hoffnung weiterverbreitet, da sie ntzlich
 *    sein wird, jedoch OHNE IRGENDEINE GARANTIE, auch ohne die implizierte
 *    Garantie der MARKTREIFE oder der VERWENDBARKEIT FR EINEN BESTIMMTEN ZWECK.
 *    Mehr Details finden Sie in der GNU Lesser General Public License.
 *	  see: http://www.gnu.org/copyleft/lesser.de.html
 *    Sie sollten eine Kopie der GNU Lesser General Public License zusammen mit
 *    dieser Bibliothek erhalten haben; falls nicht, schreiben Sie an die FSF,
 *    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 *  history:
 *		1.) 30.07.2003 Version 2.00a
 *			a.)	-
 *
 *****************************************************************************/
#include "project.h"

#include <stdio.h>
#include <string.h>
//#include <avr/delay.h>
#include <avr/wdt.h>
#include <avr/pgmspace.h>

#include HARDWARE_H_FILEPATH
#include DEVICE_H_FILEPATH
#include ATI_H_FILEPATH
#include PPP_H_FILEPATH
#include ETHERNET_H_FILEPATH
#ifdef IP_ENABLE
#include IP_H_FILEPATH
#endif
#ifdef WEB_DEBUG
#include DEBUG_H_FILEPATH
#include WEB_DEBUG_H_FILEPATH
#include PPP_DEBUG_H_FILEPATH
#endif

#ifdef __CODEVISIONAVR__
#pragma regalloc-
#endif
unsigned int ppp_rx_flag;
unsigned char ppp_status;

PPP_STATE_LCP_IPCP lcp_automaton_state;
PPP_STATE_LCP_IPCP ipcp_automaton_state;

PPP_EVENT_TYPE ppp_event;
PPP_PROTOCOL_TYPE ppp_protocol_type;
PPP_PHASE ppp_phase;

unsigned char ppp_behaviour = PPP_SERVER_CLIENT_MODE;

unsigned char ppp_ip_request;

unsigned char ppp_rx_buffer[PPP_HDLC_HEADER + PPP_MRU + PPP_HDLC_TRAILER];
unsigned int ppp_rx_buffersize;
unsigned char *p_ppp_rx_buffer;

PPP_BUFFER_TABLE ppp_rx_table_buffer[PPP_MAX_DATA_PACKETS];
unsigned char ppp_rx_table_write_index;
unsigned char ppp_rx_table_read_index;

unsigned char ppp_rx_new_packet_count;  // flag to signal a new packet, a new
                                        // entry in "ppp_table_buffer"
unsigned char *p_ppp_new_packet;        // pointer to new packet
unsigned int ppp_new_packet_size;
unsigned char *p_ppp_new_packet_data;   // pointer to new packet data
unsigned char *p_ppp_new_packet_option;
unsigned char *p_ppp_rx_data_packet;
unsigned int ppp_rx_data_packet_size = 0;

unsigned char ppp_tx_buffer[PPP_HDLC_HEADER + PPP_MTU + PPP_HDLC_TRAILER];
unsigned char *p_ppp_act_tx_buffer;
unsigned int ppp_tx_buffer_size;

unsigned char ppp_remote_configure_req_id;
unsigned char ppp_remote_terminate_req_id;
unsigned char ppp_remote_max_receive_unit[2];
unsigned char ppp_remote_accm[4];
unsigned char ppp_remote_magic_number[4];
unsigned char ppp_local_configure_request_id;
unsigned char ppp_local_terminate_request_id;
unsigned char ppp_remote_ip[4];
unsigned char ppp_local_ip[4];
unsigned char ppp_gateway_81[4];
unsigned char ppp_gateway_82[4];
unsigned char ppp_gateway_83[4];
unsigned char ppp_gateway_84[4];
unsigned char peer_id[10];
unsigned char peer_id_length;
unsigned char password[10];
unsigned char password_length;

#ifdef __CODEVISIONAVR__
#pragma regalloc+
#endif

void ppp_init(void);
void ppp_this_layer_up(void);
void ppp_this_layer_down(void);
void ppp_this_layer_started(void);
void ppp_this_layer_finished(void);
void ppp_initialize_restart_count(void);
void ppp_zero_restart_count(void);
void ppp_illegal_transition(void);

void ppp_send_configure_request(PPP_PROTOCOL_TYPE prot);
void ppp_send_configure_ack(PPP_PROTOCOL_TYPE prot);
void ppp_send_configure_nak_reject(PPP_PROTOCOL_TYPE prot);
void ppp_send_terminate_req(void);
void ppp_send_terminate_ack(void);
void ppp_send_code_reject(void);
void ppp_send_echo_reply(void);
void ppp_send_authenticate_request(void);

void ppp_packet_header(unsigned char packet_format, unsigned char id);
void ppp_transmit_buffer(unsigned char *pointer, unsigned int size);

void ppp_eval_rx_packet(void);
void ppp_eval_lcp_packet(void);
void ppp_disable_local_options(void);
void ppp_eval_pap_packet(void);
void ppp_eval_ipcp_packet(void);

void ppp_lcp_process(void);
void ppp_ipcp_process(void);
void ppp_pap_process(void);

unsigned int ppp_fcs16(unsigned int fcs, unsigned char *cp, unsigned int len);
static unsigned int fcstab[256] PROGMEM =
   { 0x0000, 0x1189, 0x2312, 0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf,
   0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7,
   0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e,
   0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876,
   0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd,
   0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5,
   0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c,
   0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974,
   0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb,
   0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3,
   0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a,
   0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72,
   0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9,
   0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1,
   0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738,
   0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70,
   0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7,
   0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff,
   0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036,
   0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e,
   0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5,
   0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd,
   0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134,
   0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c,
   0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3,
   0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb,
   0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232,
   0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a,
   0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1,
   0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9,
   0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330,
   0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78
};




#ifdef WEB_DEBUG_PPP

static PROGMEM char dead[]    = "DEAD";
static PROGMEM char estab[]   = "ESTABLISH";
static PROGMEM char authen[]  = "AUTHENTICATE";
static PROGMEM char network[] = "NETWORK";
static PROGMEM char term[]    = "TERMINATE";
static PROGMEM char unreach[] = "UNREACHED";
PGM_P ppp_phase_name[6] PROGMEM = {
   dead, estab, authen, network, term, unreach
};

static PROGMEM char initial[]  = "INITIAL";
static PROGMEM char starting[] = "STARTING";
static PROGMEM char closed[]   = "CLOSED";
static PROGMEM char stopped[]  = "STOPPED";
static PROGMEM char closing[]  = "CLOSING";
static PROGMEM char stopping[] = "STOPPING";
static PROGMEM char req_sent[] = "REQUEST_SENT";
static PROGMEM char ack_recv[] = "ACK_RECEIVED";
static PROGMEM char ack_sent[] = "ACK_SENT";
static PROGMEM char ack_opnd[] = "ACK_OPENED";
//static PROGMEM char unreach[]  = "UNREACHED";
PGM_P ppp_state_name[11] PROGMEM = {
   initial, starting, closed, stopped,
   closing, stopping, req_sent, ack_recv,
   ack_sent, ack_opnd, unreach
};

static PROGMEM char Cup[]    = "UP";
static PROGMEM char Cdown[]  = "DOWN";
static PROGMEM char Copen[]  = "OPEN";
static PROGMEM char Cclose[] = "CLOSE";
static PROGMEM char CtoP[]   = "TO+";
static PROGMEM char CtoM[]   = "TO-";
static PROGMEM char CrcrP[]  = "RCR+";
static PROGMEM char CrcrM[]  = "RCR-";
static PROGMEM char Crca[]   = "RCA";
static PROGMEM char Crcn[]   = "RCN";
static PROGMEM char Crtr[]   = "RTR";
static PROGMEM char Crta[]   = "RTA";
static PROGMEM char Cruc[]   = "RUC";
static PROGMEM char CrxjP[]  = "RXJ+";
static PROGMEM char CrxjM[]  = "RXJ-";
static PROGMEM char Crxr[]   = "RXR";
static PROGMEM char Cnone[]  = "NONE";
//static PROGMEM char unreach[]  = "UNREACHED";
PGM_P ppp_event_name[18] PROGMEM = {
   Cup, Cdown, Copen, Cclose,
   CtoP, CtoM, CrcrP, CrcrM,
   Crca, Crcn, Crtr, Crta,
   Cruc, CrxjP, CrxjM, Crxr,
   Cnone, unreach
};
#endif

void ppp_main(void)
{
#ifdef WEB_DEBUG_PPP
   static PPP_PHASE last_ppp_phase = PPP_PHASE_UNREACHED;
   static PPP_STATE_LCP_IPCP last_ipcp_auto_state = PPP_STATE_UNREACHED;
   static PPP_STATE_LCP_IPCP last_lcp_auto_state = PPP_STATE_UNREACHED;
#endif

#if defined(AT_DEVICE_MODEM) || defined(AT_DEVICE_BLUETOOTH)
   ati_state_machine();
#endif
   switch (ppp_phase)
   {
      // -------------------------------------------------------------------------------------------------------
   case PPP_PHASE_DEAD:
      if (ppp_event == PPP_EVENT_OPEN)
      {
         ppp_event = PPP_EVENT_NONE;
         lcp_automaton_state = PPP_STATE_STARTING;
#if defined(AT_DEVICE_MODEM) || defined(AT_DEVICE_BLUETOOTH)
         switch (ati_get_device_mode(ATI_ANSWER_HIDE))
         {
         case ATI_MODE_ONLINE_ENTERING:
            lcp_automaton_state = PPP_STATE_INITIAL;
            ppp_event = PPP_EVENT_OPEN;
            break;
         case ATI_MODE_ONLINE:
            ppp_event = PPP_EVENT_UP;
            break;
         default:
            ppp_event = PPP_EVENT_DOWN;
         }
#endif
         switch (ppp_event)
         {
         case PPP_EVENT_DOWN:
            ppp_init();
            break;
         case PPP_EVENT_UP:
            ppp_phase = PPP_PHASE_ESTABLISH;
            break;
         default:
            break;
         }
      }
      break;
      // -------------------------------------------------------------------------------------------------------
   case PPP_PHASE_ESTABLISH:
      if (ppp_rx_flag & PPP_RX_FLAG_FAIL)
      {
         ppp_rx_flag &= ~PPP_RX_FLAG_FAIL;
         ppp_init();
         ppp_phase = PPP_PHASE_DEAD;
      }
      else
      {
         if (lcp_automaton_state != PPP_STATE_OPENED)
         {
            ppp_lcp_process();
            ppp_event = PPP_EVENT_NONE;
            if (ppp_rx_new_packet_count > 0)
            {
               ppp_eval_rx_packet();
               if (ppp_protocol_type != PPP_PROTOCOL_LCP)
                  ppp_event = PPP_EVENT_NONE;
            }
         }
         else if (lcp_automaton_state == PPP_STATE_OPENED)
         {
            if (ppp_rx_flag & PPP_RX_FLAG_AUTH)
            {
               ppp_phase = PPP_PHASE_AUTHENTICATE;
               ppp_event = PPP_EVENT_NONE;
            }
            else
            {
               ipcp_automaton_state = PPP_STATE_STARTING;
               ppp_phase = PPP_PHASE_NETWORK;
               ppp_event = PPP_EVENT_UP;
            }
         }
      }  // end if(ppp_rx_flag & PPP_RX_FLAG_FAIL)
      break;
      // -------------------------------------------------------------------------------------------------------
   case PPP_PHASE_AUTHENTICATE:
      while ((ppp_event != PPP_EVENT_UP) & !(ppp_rx_flag & PPP_RX_FLAG_FAIL))
      {
         ppp_pap_process();
         if (ppp_rx_new_packet_count > 0)
         {
            ppp_eval_rx_packet();
            if (ppp_protocol_type == PPP_PROTOCOL_LCP)
            {
               ppp_lcp_process();
               ppp_event = PPP_EVENT_NONE;
            }
         }
      }
      if (ppp_rx_flag & PPP_RX_FLAG_FAIL)
      {
         ppp_rx_flag &= ~PPP_RX_FLAG_FAIL;
         ppp_event = PPP_EVENT_CLOSE;
         ppp_phase = PPP_PHASE_TERMINATE;
      }
      else if (ppp_event == PPP_EVENT_UP)
      {
         ipcp_automaton_state = PPP_STATE_STARTING;
         ppp_phase = PPP_PHASE_NETWORK;
      }
      break;
      // -------------------------------------------------------------------------------------------------------
   case PPP_PHASE_NETWORK:
      if ((ppp_event == PPP_EVENT_CLOSE) || (ppp_rx_flag & PPP_RX_FLAG_FAIL))
      {
         ppp_phase = PPP_PHASE_TERMINATE;
         ppp_event = PPP_EVENT_CLOSE;
      }
      else
      {
         ppp_ipcp_process();
         ppp_event = PPP_EVENT_NONE;
         if (ppp_rx_new_packet_count > 0)
         {
            ppp_eval_rx_packet();
            if (ppp_protocol_type == PPP_PROTOCOL_LCP)
            {
               ppp_lcp_process();
               if (ppp_event == PPP_EVENT_RCR
                   || ppp_event == PPP_EVENT_RCR_BAD)
               {
                  ppp_phase = PPP_PHASE_ESTABLISH;
                  ipcp_automaton_state = PPP_STATE_INITIAL;
               }
               else
               {
                  if (ppp_event == PPP_EVENT_RTR)
                     ppp_phase = PPP_PHASE_TERMINATE;
                  else
                     ppp_event = PPP_EVENT_NONE;
               }
            }
         }
      }
      break;
      // -------------------------------------------------------------------------------------------------------
   case PPP_PHASE_TERMINATE:
      if (ppp_event == PPP_EVENT_CLOSE)
      {
         while ((ppp_event !=
                 PPP_EVENT_RTA) & !(ppp_rx_flag & PPP_RX_FLAG_FAIL))
         {
            ppp_lcp_process();
            if (ppp_rx_new_packet_count > 0)
            {
               ppp_eval_rx_packet();
               if (ppp_protocol_type != PPP_PROTOCOL_LCP)
                  ppp_event = PPP_EVENT_NONE;
            }
         }
      }
#if defined(AT_DEVICE_MODEM) || defined(AT_DEVICE_BLUETOOTH)
      ati_disconnect();
#endif
      ppp_init();
#ifdef WEB_DEBUG_PPP
      if (web_debug_flag & WEB_DEBUG_PPP_BIT)
         printf_P(PSTR("PPP connection closed\n"));
#endif
      break;
   default:
      break;
   }                            // switch(ppp_phase)
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
   {
      if (last_ppp_phase != ppp_phase)
      {
         printf_P(PSTR("PPP(%u) Phase: %S\n"), ppp_frame_count,
                pgm_read_word(&ppp_phase_name[ppp_phase]));
         last_ppp_phase = ppp_phase;
      }
      if (last_ipcp_auto_state != ipcp_automaton_state)
      {
         printf_P(PSTR("PPP(%u) IPCP-State: %S\n"), ppp_frame_count,
                pgm_read_word(&ppp_state_name[ipcp_automaton_state]));
         last_ipcp_auto_state = ipcp_automaton_state;
      }
      if (last_lcp_auto_state != lcp_automaton_state)
      {
         printf_P(PSTR("PPP(%u) LCP-State: %p\n"), ppp_frame_count,
                pgm_read_word(&ppp_state_name[lcp_automaton_state]));
         last_lcp_auto_state = lcp_automaton_state;
      }
      ppp_debug_print_event_change();
   }
#endif
}                               // end main()


void ppp_init(void)
{
   unsigned char i;

   switch (ppp_behaviour)
   {
   case PPP_SERVER:
      ppp_phase = PPP_PHASE_ESTABLISH;
      lcp_automaton_state = PPP_STATE_STOPPED;
#ifdef WEB_DEBUG_PPP
      if (web_debug_flag & WEB_DEBUG_PPP_BIT)
         printf_P(PSTR("PPP SERVER Mode waiting for incoming connection request ..\n"));
#endif
      break;
   case PPP_CLIENT:
      ppp_phase = PPP_PHASE_DEAD;
      lcp_automaton_state = PPP_STATE_INITIAL;
#ifdef WEB_DEBUG_PPP
      if (web_debug_flag & WEB_DEBUG_PPP_BIT)
         printf_P(PSTR("PPP CLIENT Mode\n"));
#endif
      break;
   }
   ipcp_automaton_state = PPP_STATE_INITIAL;
   ppp_event = PPP_EVENT_NONE;
   ppp_status = 0;
   p_ppp_rx_buffer = PPP_RX_BUFFER_START;
   ppp_rx_buffersize = 0;
   p_ppp_act_tx_buffer = PPP_TX_BUFFER_START;
   ppp_tx_buffer_size = 0;
   ppp_rx_table_write_index = 0;
   ppp_rx_table_read_index = 0;
   ppp_rx_new_packet_count = 0;
   ppp_rx_flag =
      PPP_RX_FLAG_MRU + PPP_RX_FLAG_ACCM + PPP_RX_FLAG_PFC +
      PPP_RX_FLAG_ACFC + PPP_RX_FLAG_GATEWAY81 + PPP_RX_FLAG_GATEWAY82 +
      PPP_RX_FLAG_GATEWAY83 + PPP_RX_FLAG_GATEWAY84;
   ppp_local_configure_request_id = 1;
   ppp_local_terminate_request_id = 1;
   for (i = 0; i < 4; i++)
   {
      ppp_remote_ip[i]  = 0;
      ppp_local_ip[i]   = 0;
      ppp_gateway_81[i] = 0;
      ppp_gateway_82[i] = 0;
      ppp_gateway_83[i] = 0;
      ppp_gateway_84[i] = 0;
   }
#ifdef IP_ENABLE
   memcpy(&ppp_local_ip[0], &ip_address[0], sizeof(ppp_local_ip));
#endif
   ppp_rx_data_packet_size = 0;
}

unsigned char ppp_init_interface(void)
{
#if defined(AT_DEVICE_MODEM) || defined(AT_DEVICE_BLUETOOTH)
   ati_init();
#endif
   ppp_init();
   return 1;
}


void ppp_lcp_process(void)
{
   switch (ppp_event)           // which event occured?
   {
   case PPP_EVENT_UP:
      switch (lcp_automaton_state)
      {
      case PPP_STATE_INITIAL:
         lcp_automaton_state = PPP_STATE_CLOSED;
         break;
      case PPP_STATE_STARTING:
         ppp_initialize_restart_count();
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      default:
         ppp_illegal_transition();
         break;
      }
      break;
   case PPP_EVENT_DOWN:
      switch (lcp_automaton_state)
      {
      case PPP_STATE_STOPPED:
         ppp_this_layer_started();
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         break;
      default:
         break;
      }
      if (lcp_automaton_state == PPP_STATE_CLOSED
          || lcp_automaton_state == PPP_STATE_CLOSING)
      {
         lcp_automaton_state = PPP_STATE_INITIAL;
      }
      else if ((lcp_automaton_state == PPP_STATE_STOPPED) ||
               (lcp_automaton_state == PPP_STATE_STOPPING) ||
               (lcp_automaton_state == PPP_STATE_REQUEST_SENT) ||
               (lcp_automaton_state == PPP_STATE_ACK_RECEIVED) ||
               (lcp_automaton_state == PPP_STATE_ACK_SENT) ||
               (lcp_automaton_state == PPP_STATE_OPENED))
      {
         lcp_automaton_state = PPP_STATE_STARTING;
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_OPEN:
      switch (lcp_automaton_state)
      {
      case PPP_STATE_INITIAL:
         ppp_this_layer_started();
         lcp_automaton_state = PPP_STATE_STARTING;
         break;
      case PPP_STATE_CLOSED:
         ppp_initialize_restart_count();
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_CLOSING:
         lcp_automaton_state = PPP_STATE_STOPPING;
         break;
      default:
         break;
      }
      break;
   case PPP_EVENT_CLOSE:
      if (lcp_automaton_state == PPP_STATE_STARTING)
      {
         ppp_this_layer_finished();
         lcp_automaton_state = PPP_STATE_INITIAL;
      }
      else if (lcp_automaton_state == PPP_STATE_STOPPED)
      {
         lcp_automaton_state = PPP_STATE_CLOSED;
      }
      else if (lcp_automaton_state == PPP_STATE_STOPPING ||
               lcp_automaton_state == PPP_STATE_REQUEST_SENT ||
               lcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
               lcp_automaton_state == PPP_STATE_ACK_SENT ||
               lcp_automaton_state == PPP_STATE_OPENED)
      {
         if (lcp_automaton_state == PPP_STATE_OPENED)
            ppp_this_layer_down();
         if (lcp_automaton_state == PPP_STATE_REQUEST_SENT ||
             lcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
             lcp_automaton_state == PPP_STATE_ACK_SENT ||
             lcp_automaton_state == PPP_STATE_OPENED)
         {
            ppp_initialize_restart_count();
            ppp_send_terminate_req();
         }
         lcp_automaton_state = PPP_STATE_CLOSING;
      }
      break;
   case PPP_EVENT_TO:
      if (lcp_automaton_state == PPP_STATE_CLOSING
          || lcp_automaton_state == PPP_STATE_STOPPING)
      {
         ppp_send_terminate_req();
      }
      else if (lcp_automaton_state == PPP_STATE_REQUEST_SENT ||
               lcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
               lcp_automaton_state == PPP_STATE_ACK_SENT)
      {
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         if (lcp_automaton_state == PPP_STATE_ACK_RECEIVED)
            lcp_automaton_state = PPP_STATE_REQUEST_SENT;
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_TO_EX:
      if (lcp_automaton_state == PPP_STATE_CLOSING ||
          lcp_automaton_state == PPP_STATE_STOPPING ||
          lcp_automaton_state == PPP_STATE_REQUEST_SENT ||
          lcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
          lcp_automaton_state == PPP_STATE_ACK_SENT)
      {
         ppp_this_layer_finished();
         if (lcp_automaton_state == PPP_STATE_CLOSING)
         {
            lcp_automaton_state = PPP_STATE_CLOSED;
         }
         else
         {
            lcp_automaton_state = PPP_STATE_STOPPED;
         }
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RCR:
      switch (lcp_automaton_state)
      {
      case PPP_STATE_CLOSED:
         ppp_send_terminate_ack();
         break;
      case PPP_STATE_STOPPED:
         ppp_initialize_restart_count();
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         ppp_send_configure_ack(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_ACK_SENT;
         break;
      case PPP_STATE_CLOSING:
         break;
      case PPP_STATE_STOPPING:
         break;
      case PPP_STATE_REQUEST_SENT:
         ppp_send_configure_ack(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_ACK_SENT;
         break;
      case PPP_STATE_ACK_RECEIVED:
         ppp_send_configure_ack(PPP_PROTOCOL_LCP);
         // ppp_this_layer_up();
         lcp_automaton_state = PPP_STATE_OPENED;
         break;
      case PPP_STATE_ACK_SENT:
         ppp_send_configure_ack(PPP_PROTOCOL_LCP);
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         ppp_send_configure_ack(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_ACK_SENT;
         break;
      default:
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RCR_BAD:
      switch (lcp_automaton_state)
      {
      case PPP_STATE_CLOSED:
         ppp_send_terminate_ack();
         break;
      case PPP_STATE_STOPPED:
         ppp_initialize_restart_count();
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         ppp_send_configure_nak_reject(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_CLOSING:
         break;
      case PPP_STATE_STOPPING:
         break;
      case PPP_STATE_REQUEST_SENT:
         ppp_send_configure_nak_reject(PPP_PROTOCOL_LCP);
         break;
      case PPP_STATE_ACK_RECEIVED:
         ppp_send_configure_nak_reject(PPP_PROTOCOL_LCP);
         break;
      case PPP_STATE_ACK_SENT:
         ppp_send_configure_nak_reject(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         ppp_send_configure_nak_reject(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      default:
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RCA:
      switch (lcp_automaton_state)
      {
      case PPP_STATE_CLOSED:
         ppp_send_terminate_ack();
         break;
      case PPP_STATE_STOPPED:
         ppp_send_terminate_ack();
         break;
      case PPP_STATE_CLOSING:
         break;
      case PPP_STATE_STOPPING:
         break;
      case PPP_STATE_REQUEST_SENT:
         ppp_initialize_restart_count();
         lcp_automaton_state = PPP_STATE_ACK_RECEIVED;
         break;
      case PPP_STATE_ACK_RECEIVED:
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_ACK_SENT:
         ppp_initialize_restart_count();
         // ppp_this_layer_up();
         lcp_automaton_state = PPP_STATE_OPENED;
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      default:
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RCN:
      if (lcp_automaton_state == PPP_STATE_CLOSED
          || lcp_automaton_state == PPP_STATE_STOPPED)
      {
         ppp_send_terminate_ack();
      }
      else if (lcp_automaton_state == PPP_STATE_REQUEST_SENT ||
               lcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
               lcp_automaton_state == PPP_STATE_ACK_SENT ||
               lcp_automaton_state == PPP_STATE_OPENED)
      {
         if (lcp_automaton_state == PPP_STATE_REQUEST_SENT
             || lcp_automaton_state == PPP_STATE_ACK_SENT)
         {
            ppp_initialize_restart_count();
         }
         else if (lcp_automaton_state == PPP_STATE_OPENED)
         {
            ppp_this_layer_down();
         }
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         if (!(lcp_automaton_state == PPP_STATE_ACK_SENT))
            lcp_automaton_state = PPP_STATE_REQUEST_SENT;
      }
      else if (lcp_automaton_state == PPP_STATE_INITIAL
               || lcp_automaton_state == PPP_STATE_STARTING)
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RTR:
      if (!
          (lcp_automaton_state == PPP_STATE_INITIAL
           || lcp_automaton_state == PPP_STATE_STARTING))
      {
         if (lcp_automaton_state == PPP_STATE_OPENED)
         {
            ppp_status &= ~PPP_STATUS_UP;
            // ppp_this_layer_down();
            ppp_zero_restart_count();
         }
         ppp_send_terminate_ack();
         if (lcp_automaton_state == PPP_STATE_ACK_RECEIVED
             || lcp_automaton_state == PPP_STATE_ACK_SENT)
         {
            lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         }
         else if (lcp_automaton_state == PPP_STATE_OPENED)
         {
            lcp_automaton_state = PPP_STATE_STOPPING;
         }
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RTA:
      switch (lcp_automaton_state)
      {
      case PPP_STATE_CLOSING:
         ppp_this_layer_finished();
         lcp_automaton_state = PPP_STATE_CLOSED;
         break;
      case PPP_STATE_STOPPING:
         ppp_this_layer_finished();
         lcp_automaton_state = PPP_STATE_STOPPED;
         break;
      case PPP_STATE_ACK_RECEIVED:
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         ppp_send_configure_request(PPP_PROTOCOL_LCP);
         lcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      default:
         break;
      }
      if (lcp_automaton_state == PPP_STATE_INITIAL
          || lcp_automaton_state == PPP_STATE_STARTING)
         ppp_illegal_transition();
      break;
   case PPP_EVENT_RUC:
      if (!
          (lcp_automaton_state == PPP_STATE_INITIAL
           || lcp_automaton_state == PPP_STATE_STARTING))
      {
         ppp_send_code_reject();
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RXJ:
      if (lcp_automaton_state == PPP_STATE_INITIAL
          || lcp_automaton_state == PPP_STATE_STARTING)
         ppp_illegal_transition();
      break;
   case PPP_EVENT_RXJ_CAT:
      if (lcp_automaton_state == PPP_STATE_INITIAL ||
          lcp_automaton_state == PPP_STATE_STARTING ||
          lcp_automaton_state == PPP_STATE_OPENED)
      {
         if (lcp_automaton_state == PPP_STATE_OPENED)
         {
            ppp_this_layer_down();
            ppp_initialize_restart_count();
            ppp_send_terminate_req();
            lcp_automaton_state = PPP_STATE_STOPPING;
         }
         else
         {
            ppp_illegal_transition();
         }
      }
      else
      {
         ppp_this_layer_finished();
         if (lcp_automaton_state == PPP_STATE_CLOSING)
            lcp_automaton_state = PPP_STATE_CLOSED;
         else
            lcp_automaton_state = PPP_STATE_STOPPED;
      }
      break;
   case PPP_EVENT_RXR:
      if (lcp_automaton_state == PPP_STATE_OPENED)
         ppp_send_echo_reply();
      if (lcp_automaton_state == PPP_STATE_INITIAL
          || lcp_automaton_state == PPP_STATE_STARTING)
         ppp_illegal_transition();
      break;
   default:
      break;
   }                            // switch(ppp_event) // which event occured?
}                               // end ppp_lcp_process()


void ppp_pap_process(void)
{
   ppp_send_authenticate_request();
}

void ppp_ipcp_process(void)
{
   switch (ppp_event)           // which event occured?
   {
   case PPP_EVENT_UP:
      switch (ipcp_automaton_state)
      {
      case PPP_STATE_INITIAL:
         ipcp_automaton_state = PPP_STATE_CLOSED;
         break;
      case PPP_STATE_STARTING:
         ppp_initialize_restart_count();
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      default:
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_DOWN:
      switch (ipcp_automaton_state)
      {
      case PPP_STATE_STOPPED:
         ppp_this_layer_started();
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         break;
      default:
         break;
      }
      if (ipcp_automaton_state == PPP_STATE_CLOSED
          || ipcp_automaton_state == PPP_STATE_CLOSING)
      {
         ipcp_automaton_state = PPP_STATE_INITIAL;
      }
      else if (ipcp_automaton_state == PPP_STATE_STOPPED ||
               ipcp_automaton_state == PPP_STATE_STOPPING ||
               ipcp_automaton_state == PPP_STATE_REQUEST_SENT ||
               ipcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
               ipcp_automaton_state == PPP_STATE_ACK_SENT ||
               ipcp_automaton_state == PPP_STATE_OPENED)
      {
         ipcp_automaton_state = PPP_STATE_STARTING;
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_OPEN:
      switch (ipcp_automaton_state)
      {
      case PPP_STATE_INITIAL:
         ppp_this_layer_started();
         ipcp_automaton_state = PPP_STATE_STARTING;
         break;
      case PPP_STATE_CLOSED:
         ppp_initialize_restart_count();
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_CLOSING:
         ipcp_automaton_state = PPP_STATE_STOPPING;
         break;
      default:
         break;
      }
      break;
   case PPP_EVENT_CLOSE:
      if (ipcp_automaton_state == PPP_STATE_STARTING)
      {
         ppp_this_layer_finished();
         ipcp_automaton_state = PPP_STATE_INITIAL;
      }
      else if (ipcp_automaton_state == PPP_STATE_STOPPED)
      {
         ipcp_automaton_state = PPP_STATE_CLOSED;
      }
      else if (ipcp_automaton_state == PPP_STATE_STOPPING ||
               ipcp_automaton_state == PPP_STATE_REQUEST_SENT ||
               ipcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
               ipcp_automaton_state == PPP_STATE_ACK_SENT ||
               ipcp_automaton_state == PPP_STATE_OPENED)
      {
         if (ipcp_automaton_state == PPP_STATE_OPENED)
            ppp_this_layer_down();
         if (ipcp_automaton_state == PPP_STATE_REQUEST_SENT ||
             ipcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
             ipcp_automaton_state == PPP_STATE_ACK_SENT ||
             ipcp_automaton_state == PPP_STATE_OPENED)
         {
            ppp_initialize_restart_count();
            ppp_send_terminate_req();
         }
         ipcp_automaton_state = PPP_STATE_CLOSING;
      }
      break;
   case PPP_EVENT_TO:
      if (ipcp_automaton_state == PPP_STATE_CLOSING
          || ipcp_automaton_state == PPP_STATE_STOPPING)
      {
         ppp_send_terminate_req();
      }
      else if (ipcp_automaton_state == PPP_STATE_REQUEST_SENT ||
               ipcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
               ipcp_automaton_state == PPP_STATE_ACK_SENT)
      {
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         if (ipcp_automaton_state == PPP_STATE_ACK_RECEIVED)
            ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_TO_EX:
      if (ipcp_automaton_state == PPP_STATE_CLOSING ||
          ipcp_automaton_state == PPP_STATE_STOPPING ||
          ipcp_automaton_state == PPP_STATE_REQUEST_SENT ||
          ipcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
          ipcp_automaton_state == PPP_STATE_ACK_SENT)
      {
         ppp_this_layer_finished();
         if (ipcp_automaton_state == PPP_STATE_CLOSING)
         {
            ipcp_automaton_state = PPP_STATE_CLOSED;
         }
         else
         {
            ipcp_automaton_state = PPP_STATE_STOPPED;
         }
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RCR:
      switch (ipcp_automaton_state)
      {
      case PPP_STATE_CLOSED:
         ppp_send_terminate_ack();
         break;
      case PPP_STATE_STOPPED:
         ppp_initialize_restart_count();
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ppp_send_configure_ack(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_ACK_SENT;
         break;
      case PPP_STATE_CLOSING:
         break;
      case PPP_STATE_STOPPING:
         break;
      case PPP_STATE_REQUEST_SENT:
         ppp_send_configure_ack(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_ACK_SENT;
         break;
      case PPP_STATE_ACK_RECEIVED:
         ppp_send_configure_ack(PPP_PROTOCOL_IPCP);
         ppp_this_layer_up();
         ipcp_automaton_state = PPP_STATE_OPENED;
         break;
      case PPP_STATE_ACK_SENT:
         ppp_send_configure_ack(PPP_PROTOCOL_IPCP);
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ppp_send_configure_ack(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_ACK_SENT;
         break;
      default:
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RCR_BAD:
      switch (ipcp_automaton_state)
      {
      case PPP_STATE_CLOSED:
         ppp_send_terminate_ack();
         break;
      case PPP_STATE_STOPPED:
         ppp_initialize_restart_count();
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ppp_send_configure_nak_reject(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_CLOSING:
         break;
      case PPP_STATE_STOPPING:
         break;
      case PPP_STATE_REQUEST_SENT:
         ppp_send_configure_nak_reject(PPP_PROTOCOL_IPCP);
         break;
      case PPP_STATE_ACK_RECEIVED:
         ppp_send_configure_nak_reject(PPP_PROTOCOL_IPCP);
         break;
      case PPP_STATE_ACK_SENT:
         ppp_send_configure_nak_reject(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ppp_send_configure_nak_reject(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      default:
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RCA:
      switch (ipcp_automaton_state)
      {
      case PPP_STATE_CLOSED:
         ppp_send_terminate_ack();
         break;
      case PPP_STATE_STOPPED:
         ppp_send_terminate_ack();
         break;
      case PPP_STATE_CLOSING:
         break;
      case PPP_STATE_STOPPING:
         break;
      case PPP_STATE_REQUEST_SENT:
         ppp_initialize_restart_count();
         ipcp_automaton_state = PPP_STATE_ACK_RECEIVED;
         break;
      case PPP_STATE_ACK_RECEIVED:
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_ACK_SENT:
         ppp_initialize_restart_count();
         ppp_this_layer_up();
         ipcp_automaton_state = PPP_STATE_OPENED;
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      default:
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RCN:
      if (ipcp_automaton_state == PPP_STATE_CLOSED
          || ipcp_automaton_state == PPP_STATE_STOPPED)
      {
         ppp_send_terminate_ack();
      }
      else if (ipcp_automaton_state == PPP_STATE_REQUEST_SENT ||
               ipcp_automaton_state == PPP_STATE_ACK_RECEIVED ||
               ipcp_automaton_state == PPP_STATE_ACK_SENT ||
               ipcp_automaton_state == PPP_STATE_OPENED)
      {
         if (ipcp_automaton_state == PPP_STATE_REQUEST_SENT
             || ipcp_automaton_state == PPP_STATE_ACK_SENT)
         {
            ppp_initialize_restart_count();
         }
         else if (ipcp_automaton_state == PPP_STATE_OPENED)
         {
            ppp_this_layer_down();
         }
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         if (!(ipcp_automaton_state == PPP_STATE_ACK_SENT))
            ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
      }
      else if (ipcp_automaton_state == PPP_STATE_INITIAL
               || ipcp_automaton_state == PPP_STATE_STARTING)
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RTR:
      if (!
          (ipcp_automaton_state == PPP_STATE_INITIAL
           || ipcp_automaton_state == PPP_STATE_STARTING))
      {
         if (ipcp_automaton_state == PPP_STATE_OPENED)
         {
            ppp_status &= ~PPP_STATUS_UP;
            // ppp_this_layer_down();
            ppp_zero_restart_count();
         }
         ppp_send_terminate_ack();
         if (ipcp_automaton_state == PPP_STATE_ACK_RECEIVED
             || ipcp_automaton_state == PPP_STATE_ACK_SENT)
         {
            ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         }
         else if (ipcp_automaton_state == PPP_STATE_OPENED)
         {
            ipcp_automaton_state = PPP_STATE_STOPPING;
         }
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RTA:
      switch (ipcp_automaton_state)
      {
      case PPP_STATE_CLOSING:
         ppp_this_layer_finished();
         ipcp_automaton_state = PPP_STATE_CLOSED;
         break;
      case PPP_STATE_STOPPING:
         ppp_this_layer_finished();
         ipcp_automaton_state = PPP_STATE_STOPPED;
         break;
      case PPP_STATE_ACK_RECEIVED:
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      case PPP_STATE_OPENED:
         ppp_this_layer_down();
         ppp_send_configure_request(PPP_PROTOCOL_IPCP);
         ipcp_automaton_state = PPP_STATE_REQUEST_SENT;
         break;
      default:
         break;
      }
      if (ipcp_automaton_state == PPP_STATE_INITIAL
          || ipcp_automaton_state == PPP_STATE_STARTING)
         ppp_illegal_transition();
      break;
   case PPP_EVENT_RUC:
      if (!
          (ipcp_automaton_state == PPP_STATE_INITIAL
           || ipcp_automaton_state == PPP_STATE_STARTING))
      {
         ppp_send_code_reject();
      }
      else
      {
         ppp_illegal_transition();
      }
      break;
   case PPP_EVENT_RXJ:
      if (ipcp_automaton_state == PPP_STATE_INITIAL
          || ipcp_automaton_state == PPP_STATE_STARTING)
         ppp_illegal_transition();
      break;
   case PPP_EVENT_RXJ_CAT:
      if (ipcp_automaton_state == PPP_STATE_INITIAL ||
          ipcp_automaton_state == PPP_STATE_STARTING ||
          ipcp_automaton_state == PPP_STATE_OPENED)
      {
         if (ipcp_automaton_state == PPP_STATE_OPENED)
         {
            ppp_this_layer_down();
            ppp_initialize_restart_count();
            ppp_send_terminate_req();
            ipcp_automaton_state = PPP_STATE_STOPPING;
         }
         else
         {
            ppp_illegal_transition();
         }
      }
      else
      {
         ppp_this_layer_finished();
         if (ipcp_automaton_state == PPP_STATE_CLOSING)
            ipcp_automaton_state = PPP_STATE_CLOSED;
         else
            ipcp_automaton_state = PPP_STATE_STOPPED;
      }
      break;
   case PPP_EVENT_RXR:
      if (ipcp_automaton_state == PPP_STATE_OPENED)
         ppp_send_echo_reply();
      if (ipcp_automaton_state == PPP_STATE_INITIAL
          || ipcp_automaton_state == PPP_STATE_STARTING)
         ppp_illegal_transition();
      break;
   default:
      break;
   }                            // switch(ppp_event) // which event occured?
}                               // end ppp_ipcp_process()

#ifdef WEB_DEBUG_PPP
static PROGMEM char Clcp[]  = "LCP";
static PROGMEM char Cpap[]  = "PAP";
static PROGMEM char Cipcp[] = "IPCP";
static PROGMEM char Cip[]   = "IP";
static PROGMEM char Cunkw[] = "UNKNOWN";
PGM_P ppp_protocol_name[5] PROGMEM = {
   Clcp, Cpap, Cipcp, Cip, Cunkw
};
#endif

void ppp_eval_rx_packet(void)
{
   ppp_event = PPP_EVENT_NONE;
   p_ppp_new_packet = ppp_rx_table_buffer[ppp_rx_table_read_index].addr;
   ppp_new_packet_size = ppp_rx_table_buffer[ppp_rx_table_read_index].size;
   p_ppp_new_packet_data = p_ppp_new_packet;
#ifdef WEB_DEBUG_PPP
   ppp_frame_count++;
#endif
   if (ppp_fcs16(PPP_INIT_FCS16, p_ppp_new_packet_data, ppp_new_packet_size)
       == PPP_GOOD_FCS16)
   {
      // FCS war erfolgreich, hier nun LCP packet auswerten
      // --------------
      if (p_ppp_new_packet_data[PPP_ADDRESS_OFS] == PPP_ADDRESS
          && p_ppp_new_packet_data[PPP_CONTROL_OFS] == PPP_CONTROL)
      {
         // wenn die Address-and-Control-Field-Compression nicht aktiv ist,
         // d.h.
         // Address-and-Control-Field mit bertragen werden
         // wird der Buffer Pointer um 2 erhht und zeigt demnach auf das
         // protocol field
         p_ppp_new_packet_data += 2;
      }
      // only valid protocol types will be evaluated
      // c021 : Link Control Protocol
      // c023 : Password Authentication Protocol
      // 8021 : IP Control Protocol
      // 0021 || 21 : Internet Protocol
      if (p_ppp_new_packet_data[PPP_PROTOCOL_OFS] == 0xC0
          && p_ppp_new_packet_data[PPP_PROTOCOL_OFS + 1] == 0x21)
         ppp_protocol_type = PPP_PROTOCOL_LCP;
      else if (p_ppp_new_packet_data[PPP_PROTOCOL_OFS] == 0xC0
               && p_ppp_new_packet_data[PPP_PROTOCOL_OFS + 1] == 0x23)
         ppp_protocol_type = PPP_PROTOCOL_PAP;
      else if (p_ppp_new_packet_data[PPP_PROTOCOL_OFS] == 0x80
               && p_ppp_new_packet_data[PPP_PROTOCOL_OFS + 1] == 0x21)
         ppp_protocol_type = PPP_PROTOCOL_IPCP;
      else
         if ((p_ppp_new_packet_data[PPP_PROTOCOL_OFS] == 0x00
              && p_ppp_new_packet_data[PPP_PROTOCOL_OFS + 1] == 0x21)
             || (p_ppp_new_packet_data[PPP_PROTOCOL_OFS] == 0x21))
      {
         if (p_ppp_new_packet_data[PPP_PROTOCOL_OFS] == 0x21)
         {
            p_ppp_new_packet_data++;
            ppp_new_packet_size -= (1 + PPP_HDLC_TRAILER);
         }
         else
         {
            p_ppp_new_packet_data += 2;
            ppp_new_packet_size -= (2 + PPP_HDLC_TRAILER);
         }
         ppp_protocol_type = PPP_PROTOCOL_IP;
      }
      else
      {
         ppp_protocol_type = PPP_PROTOCOL_UNKNOWN;
         ppp_event = PPP_EVENT_RUC;
      }
#ifdef WEB_DEBUG_PPP
      if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      {
         printf_P(PSTR("PPP-R(%u) %S-%uByte\n"), ppp_frame_count,
                pgm_read_word(&ppp_protocol_name[ppp_protocol_type]), ppp_new_packet_size);
         if (ppp_protocol_type != PPP_PROTOCOL_IP)
            debug_display_frame(p_ppp_new_packet, ppp_new_packet_size, 0);
      }
#endif
      switch (ppp_protocol_type)
      {
      case PPP_PROTOCOL_LCP:
         ppp_eval_lcp_packet();
         break;
      case PPP_PROTOCOL_PAP:
         ppp_eval_pap_packet();
         break;
      case PPP_PROTOCOL_IPCP:
         ppp_eval_ipcp_packet();
         break;
      case PPP_PROTOCOL_IP:
         if ((ipcp_automaton_state == PPP_STATE_OPENED)
             && (lcp_automaton_state == PPP_STATE_OPENED)
             && (ppp_status & PPP_STATUS_UP))
         {
            if ((ppp_status & PPP_STATUS_RX_DATA) == 0)
            {
               ppp_status |= PPP_STATUS_RX_DATA;
               p_ppp_rx_data_packet = p_ppp_new_packet_data;    // init
                                                                // SAP-pointer
               ppp_rx_data_packet_size = ppp_new_packet_size;
#ifdef WEB_DEBUG_PPP
               if (web_debug_flag & WEB_DEBUG_PPP_BIT)
               {


               }
#endif
            }
            else
            {
#ifdef WEB_DEBUG_PPP
               if (web_debug_flag & WEB_DEBUG_PPP_BIT)
                  printf_P(PSTR("PPP-R(%u) data lost: %uByte\n"), ppp_frame_count,
                         ppp_new_packet_size);
#endif
            }
         }
         break;
      default:
            break;
      }
   }                            // end FCS
   ppp_rx_table_read_index++;
   if (ppp_rx_table_read_index > (PPP_MAX_DATA_PACKETS - 1))
      ppp_rx_table_read_index = 0;
   ppp_rx_new_packet_count--;
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      ppp_debug_print_event_change();
#endif
}

#ifdef WEB_DEBUG_PPP
static PROGMEM char Cnix[]     = "";
static PROGMEM char CcfgRequ[] = "CONFIG_REQUEST";
static PROGMEM char CcfgAck[]  = "CONFIG_ACK";
static PROGMEM char CcfgNak[]  = "CONFIG_NAK";
static PROGMEM char CcfgRej[]  = "CONFIG_REJECT";
static PROGMEM char CtrmReq[]  = "TERMINATE_REQUEST";
static PROGMEM char CtrmAck[]  = "TERMINATE_ACK";
static PROGMEM char CcodeRej[] = "CODE_REJECT";
static PROGMEM char CprtlRej[] = "PROTOCOL_REJECT";
static PROGMEM char CechoReq[] = "ECHO_REQUEST";
static PROGMEM char CechoRpl[] = "ECHO_REPLY";
static PROGMEM char CdiscReq[] = "DISCARD_REQUEST";
PGM_P ppp_packet_code_name[12] PROGMEM = {
   Cnix, CcfgRequ, CcfgAck, CcfgNak,
   CcfgRej, CtrmReq, CtrmAck, CcodeRej,
   CprtlRej, CechoReq, CechoRpl, CdiscReq
};

static PROGMEM char Cunsupp[]  = "UNSUPPORTED";
static PROGMEM char CmaxRxU[]  = "MAX_RECEIVE_UNIT";
static PROGMEM char Caccm[]    = "ACCM";
static PROGMEM char CauthTyp[] = "AUTH_TYPE";
static PROGMEM char CqualPro[] = "QUALITY_PROTOCOL";
static PROGMEM char CmagicNm[] = "MAGIC_NUMBER";
static PROGMEM char ClnkQuMn[] = "LINK_QUALITY_MON";
static PROGMEM char CproFlCp[] = "PROTOCOL_FIELD_COMP";
static PROGMEM char CadrCtlF[] = "ADR_CTL_FIELD_COMP";
PGM_P ppp_lcp_options_name[9] PROGMEM = {
   Cunsupp, CmaxRxU, Caccm, CauthTyp,
   CqualPro, CmagicNm, ClnkQuMn,
   CproFlCp, CadrCtlF
};
#endif

void ppp_eval_lcp_packet(void)
{
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      printf_P(PSTR("PPP-R(%u) LCP %S "), ppp_frame_count,
             pgm_read_word(&ppp_packet_code_name[p_ppp_new_packet_data[PPP_CODE_OFS]]));
#endif
   switch (p_ppp_new_packet_data[PPP_CODE_OFS])
   {
   case PPP_CONFIGURE_REQUEST:
      ppp_remote_configure_req_id = p_ppp_new_packet_data[PPP_ID_OFS];
      p_ppp_new_packet_data += PPP_OPTIONS_OFS;
      p_ppp_new_packet_option = p_ppp_new_packet_data;  // for further
      while (p_ppp_new_packet_data <
             p_ppp_new_packet + (ppp_new_packet_size - 2))
      {
#ifdef WEB_DEBUG_PPP
         if (web_debug_flag & WEB_DEBUG_PPP_BIT)
         {
            if (*p_ppp_new_packet_data < 9)
               printf_P(PSTR("%S "), pgm_read_word(&ppp_lcp_options_name[*p_ppp_new_packet_data]));
            else
               printf_P(PSTR("%S "), pgm_read_word(&ppp_lcp_options_name[0]));
         }
#endif
         switch (*p_ppp_new_packet_data)
         {
         case LCP_MAX_RECEIVE_UNIT:
            ppp_remote_max_receive_unit[0] =
               p_ppp_new_packet_data[LCP_OPTIONS_DATA_OFS];
            ppp_remote_max_receive_unit[1] =
               p_ppp_new_packet_data[LCP_OPTIONS_DATA_OFS + 1];
            p_ppp_new_packet_data += LCP_MAX_RECEIVE_UNIT_LEN;
            break;
         case LCP_ACCM:
            memcpy(&ppp_remote_accm[0],
                   &p_ppp_new_packet_data[LCP_OPTIONS_DATA_OFS],
                   sizeof(ppp_remote_accm));
            p_ppp_new_packet_data += LCP_ACCM_LEN;
            break;
         case LCP_AUTHENTICATION_TYPE:
            if (p_ppp_new_packet_data[LCP_OPTIONS_DATA_OFS] == 0xc0
                && p_ppp_new_packet_data[LCP_OPTIONS_DATA_OFS + 1] == 0x23)
               ppp_rx_flag |= PPP_RX_FLAG_AUTH;
            else
               ppp_event = PPP_EVENT_RCR_BAD;
            p_ppp_new_packet_data += LCP_AUTHENTICATION_TYPE_LEN;
            break;
         case LCP_MAGIC_NUMBER:
            memcpy(&ppp_remote_magic_number[0],
                   &p_ppp_new_packet_data[LCP_OPTIONS_DATA_OFS],
                   sizeof(ppp_remote_magic_number));
            p_ppp_new_packet_data += LCP_MAGIC_NUMBER_LEN;
            break;
         case LCP_LINK_QUALITY_MONITORING:
            break;
         case LCP_PROTOCOL_FIELD_COMP:
            ppp_rx_flag |= PPP_RX_FLAG_PFC;
            p_ppp_new_packet_data += LCP_PROTOCOL_FIELD_COMP_LEN;
            break;
         case LCP_ADR_CTL_FIELD_COMP:
            ppp_rx_flag |= PPP_RX_FLAG_ACFC;
            p_ppp_new_packet_data += LCP_ADR_CTL_FIELD_COMP_LEN;
            break;
         default:
            p_ppp_new_packet_data += p_ppp_new_packet_data[1];
            ppp_event = PPP_EVENT_RCR_BAD;
         }
      }                         // end while
      if (ppp_event != PPP_EVENT_RCR_BAD)
         ppp_event = PPP_EVENT_RCR;
      break;
   case PPP_CONFIGURE_ACK:
      ppp_event = PPP_EVENT_RCA;
      break;
   case PPP_CONFIGURE_NAK:
      ppp_disable_local_options();
      break;
   case PPP_CONFIGURE_REJECT:
      ppp_disable_local_options();
      break;
   case PPP_TERMINATE_REQUEST:
      ppp_remote_terminate_req_id = p_ppp_new_packet_data[PPP_ID_OFS];
      ppp_event = PPP_EVENT_RTR;
      break;
   case PPP_TERMINATE_ACK:
      ppp_event = PPP_EVENT_RTA;
      break;
   case PPP_CODE_REJECT:
      ppp_event = PPP_EVENT_RXJ;
      break;
   case PPP_PROTOCOL_REJECT:
      ppp_event = PPP_EVENT_RXJ;
      break;
   case PPP_ECHO_REQUEST:
      ppp_event = PPP_EVENT_RXR;
      break;
   case PPP_ECHO_REPLY:
      ppp_event = PPP_EVENT_RXR;
      break;
   case PPP_DISCARD_REQUEST:
      ppp_event = PPP_EVENT_RXR;
      break;
   default:
      ppp_event = PPP_EVENT_RUC;
   }                            // switch(p_ppp_new_packet_data[PPP_CODE_OFS])
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
   {
      printf_P(PSTR("\n"));
      ppp_debug_print_event_change();
   }
#endif
}                               // end ppp_eval_lcp_packet()


void ppp_disable_local_options(void)
{
   p_ppp_new_packet_data += PPP_OPTIONS_OFS;
   while (p_ppp_new_packet_data <
          p_ppp_new_packet + (ppp_new_packet_size - 2))
   {
      switch (*p_ppp_new_packet_data)
      {
      case LCP_MAX_RECEIVE_UNIT:
         ppp_rx_flag &= ~PPP_RX_FLAG_MRU;
         p_ppp_new_packet_data += LCP_MAX_RECEIVE_UNIT_LEN;
         break;
      case LCP_ACCM:
         ppp_rx_flag &= ~PPP_RX_FLAG_ACCM;
         p_ppp_new_packet_data += LCP_ACCM_LEN;
         break;
      case LCP_PROTOCOL_FIELD_COMP:
         ppp_rx_flag &= ~PPP_RX_FLAG_PFC;
         p_ppp_new_packet_data += LCP_PROTOCOL_FIELD_COMP_LEN;
         break;
      case LCP_ADR_CTL_FIELD_COMP:
         ppp_rx_flag &= ~PPP_RX_FLAG_ACFC;
         p_ppp_new_packet_data += LCP_ADR_CTL_FIELD_COMP_LEN;
         break;
      default:
         p_ppp_new_packet_data += p_ppp_new_packet_data[1];
      }                         // end switch(*pRxBuffer)
   }                            // end while
   ppp_event = PPP_EVENT_RCN;
}                               // end ppp_disable_local_options()


void ppp_eval_pap_packet(void)
{
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      printf_P(PSTR("PPP-R(%u) PAP %S\n"), ppp_frame_count,
             pgm_read_word(&ppp_packet_code_name[p_ppp_new_packet_data[PPP_CODE_OFS]]));
#endif
   if (p_ppp_new_packet_data[PPP_CODE_OFS] == PPP_CONFIGURE_ACK)
      ppp_event = PPP_EVENT_UP;
}


void ppp_eval_ipcp_packet(void)
{
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      printf_P(PSTR("PPP-R(%u) IPCP %S "), ppp_frame_count,
             pgm_read_word(&ppp_packet_code_name[p_ppp_new_packet_data[PPP_CODE_OFS]]));
#endif
   switch (p_ppp_new_packet_data[PPP_CODE_OFS])
   {
   case PPP_CONFIGURE_REQUEST:
      ppp_remote_configure_req_id = p_ppp_new_packet_data[PPP_ID_OFS];
      p_ppp_new_packet_data += PPP_OPTIONS_OFS;
      p_ppp_new_packet_option = p_ppp_new_packet_data;  // for further
      while (p_ppp_new_packet_data <
             p_ppp_new_packet + (ppp_new_packet_size - 2))
      {
#ifdef WEB_DEBUG_PPP
         if (web_debug_flag & WEB_DEBUG_PPP_BIT)
         {
            if (*p_ppp_new_packet_data == IPCP_IP_ADR)
               debug_print_ip(0x0,
                              &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS]);
            else
               printf_P(PSTR("0x%02x "), *p_ppp_new_packet_data);
         }
#endif
         if (*p_ppp_new_packet_data == IPCP_IP_ADR)
         {
            if (p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS] == 0 &&
                p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS + 1] == 0 &&
                p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS + 2] == 0 &&
                p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS + 3] == 0)
            {
               ppp_ip_request = 1;
               ppp_event = PPP_EVENT_RCR_BAD;
            }
            memcpy(&ppp_remote_ip[0],
                   &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS],
                   sizeof(ppp_remote_ip));
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
         }
         else
         {
            p_ppp_new_packet_data += p_ppp_new_packet_data[1];
            ppp_ip_request = 0;
            ppp_event = PPP_EVENT_RCR_BAD;
         }
      }                         // end while
      if (ppp_event != PPP_EVENT_RCR_BAD)
         ppp_event = PPP_EVENT_RCR;
      break;
   case PPP_CONFIGURE_ACK:
      ppp_event = PPP_EVENT_RCA;
      break;
   case PPP_CONFIGURE_NAK:
      p_ppp_new_packet_data += PPP_OPTIONS_OFS;
      while (p_ppp_new_packet_data <
             p_ppp_new_packet + (ppp_new_packet_size - 2))
      {
#ifdef WEB_DEBUG_PPP
         if (web_debug_flag & WEB_DEBUG_PPP_BIT)
         {
            if (*p_ppp_new_packet_data == IPCP_IP_ADR ||
                *p_ppp_new_packet_data == IPCP_GATEWAY_81 ||
                *p_ppp_new_packet_data == IPCP_GATEWAY_82 ||
                *p_ppp_new_packet_data == IPCP_GATEWAY_83 ||
                *p_ppp_new_packet_data == IPCP_GATEWAY_84)
            {
               debug_print_ip(0x0,
                              &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS]);
            }
            else
               printf_P(PSTR("0x%02x "), *p_ppp_new_packet_data);
         }
#endif
         switch (*p_ppp_new_packet_data)
         {
         case IPCP_IP_ADR:
            memcpy(&ppp_local_ip[0],
                   &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS],
                   sizeof(ppp_local_ip));
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         case IPCP_GATEWAY_81:
            memcpy(&ppp_gateway_81[0],
                   &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS],
                   sizeof(ppp_gateway_81));
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         case IPCP_GATEWAY_82:
            memcpy(&ppp_gateway_82[0],
                   &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS],
                   sizeof(ppp_gateway_82));
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         case IPCP_GATEWAY_83:
            memcpy(&ppp_gateway_83[0],
                   &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS],
                   sizeof(ppp_gateway_83));
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         case IPCP_GATEWAY_84:
            memcpy(&ppp_gateway_84[0],
                   &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS],
                   sizeof(ppp_gateway_84));
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         default:
            p_ppp_new_packet_data += p_ppp_new_packet_data[1];
         }
      }                         // end while
      ppp_event = PPP_EVENT_RCN;
      break;
   case PPP_CONFIGURE_REJECT:
      p_ppp_new_packet_data += PPP_OPTIONS_OFS;
      while (p_ppp_new_packet_data <
             p_ppp_new_packet + (ppp_new_packet_size - 2))
      {
#ifdef WEB_DEBUG_PPP
         if (web_debug_flag & WEB_DEBUG_PPP_BIT)
         {
            if (*p_ppp_new_packet_data == IPCP_IP_ADR ||
                *p_ppp_new_packet_data == IPCP_GATEWAY_81 ||
                *p_ppp_new_packet_data == IPCP_GATEWAY_82 ||
                *p_ppp_new_packet_data == IPCP_GATEWAY_83 ||
                *p_ppp_new_packet_data == IPCP_GATEWAY_84)
            {
               debug_print_ip(0x0,
                              &p_ppp_new_packet_data[IPCP_OPTIONS_DATA_OFS]);
            }
            else
               printf_P(PSTR("0x%02x "), *p_ppp_new_packet_data);
         }
#endif
         switch (*p_ppp_new_packet_data)
         {
         case IPCP_GATEWAY_81:
            ppp_rx_flag &= ~PPP_RX_FLAG_GATEWAY81;
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         case IPCP_GATEWAY_82:
            ppp_rx_flag &= ~PPP_RX_FLAG_GATEWAY82;
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         case IPCP_GATEWAY_83:
            ppp_rx_flag &= ~PPP_RX_FLAG_GATEWAY83;
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         case IPCP_GATEWAY_84:
            ppp_rx_flag &= ~PPP_RX_FLAG_GATEWAY84;
            p_ppp_new_packet_data += IPCP_IP_ADR_LEN;
            break;
         default:
            p_ppp_new_packet_data += p_ppp_new_packet_data[1];
         }
      }                         // end while
      ppp_event = PPP_EVENT_RCN;
      break;
   case PPP_TERMINATE_REQUEST:
      ppp_remote_terminate_req_id = p_ppp_new_packet_data[PPP_ID_OFS];
      ppp_event = PPP_EVENT_RTR;
      break;
   case PPP_TERMINATE_ACK:
      ppp_event = PPP_EVENT_RTA;
      break;
   case PPP_CODE_REJECT:
      ppp_event = PPP_EVENT_RXJ;
      break;
   default:
      ppp_event = PPP_EVENT_RUC;
   }                            // switch(p_ppp_new_packet_data[PPP_CODE_OFS])
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      printf_P(PSTR("\r"));
#endif
}                               // end ppp_eval_ipcp_packet


void ppp_this_layer_up(void)
{
   ppp_status |= PPP_STATUS_UP;
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      printf_P(PSTR("PPP(%u) connection established\n"), ppp_frame_count);
#endif
}

void ppp_this_layer_down(void)
{
   ppp_status &= ~PPP_STATUS_UP;
   ppp_event = PPP_EVENT_DOWN;
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      printf_P(PSTR("PPP(%u) connection down\n"), ppp_frame_count);
#endif
}

void ppp_this_layer_started(void)
{
}

void ppp_this_layer_finished(void)
{
}

void ppp_initialize_restart_count(void)
{
}

void ppp_zero_restart_count(void)
{
}


void ppp_illegal_transition(void)
{
#ifdef WEB_DEBUG_PPP
   if (web_debug_flag & WEB_DEBUG_PPP_BIT)
      printf_P(PSTR("PPP(%u) Illegal Transaction\n"), ppp_frame_count);
#endif
}


void ppp_send_configure_request(PPP_PROTOCOL_TYPE prot)
{
   p_ppp_act_tx_buffer = PPP_TX_BUFFER_START;
   p_ppp_act_tx_buffer += PPP_OPTIONS_OFS + 2;
   ppp_tx_buffer_size = PPP_OPTIONS_OFS + 2;

   switch (prot)
   {
   case PPP_PROTOCOL_LCP:
      if (ppp_rx_flag & PPP_RX_FLAG_MRU)
      {
         *p_ppp_act_tx_buffer++ = LCP_MAX_RECEIVE_UNIT;
         *p_ppp_act_tx_buffer++ = LCP_MAX_RECEIVE_UNIT_LEN;
         *p_ppp_act_tx_buffer++ = ((PPP_MRU >> 8) & 0x00ff);
         *p_ppp_act_tx_buffer++ = (PPP_MRU & 0x00ff);
         ppp_tx_buffer_size += LCP_MAX_RECEIVE_UNIT_LEN;
      }
      if (ppp_rx_flag & PPP_RX_FLAG_ACCM)
      {
         *p_ppp_act_tx_buffer++ = LCP_ACCM;
         *p_ppp_act_tx_buffer++ = LCP_ACCM_LEN;
         *p_ppp_act_tx_buffer++ = 0x00;
         *p_ppp_act_tx_buffer++ = 0x0A;
         *p_ppp_act_tx_buffer++ = 0x00;
         *p_ppp_act_tx_buffer++ = 0x00;
         ppp_tx_buffer_size += LCP_ACCM_LEN;
      }
      if (ppp_rx_flag & PPP_RX_FLAG_PFC)
      {
         *p_ppp_act_tx_buffer++ = LCP_PROTOCOL_FIELD_COMP;
         *p_ppp_act_tx_buffer++ = LCP_PROTOCOL_FIELD_COMP_LEN;
         ppp_tx_buffer_size += LCP_PROTOCOL_FIELD_COMP_LEN;
      }
      if (ppp_rx_flag & PPP_RX_FLAG_ACFC)
      {
         *p_ppp_act_tx_buffer++ = LCP_ADR_CTL_FIELD_COMP;
         *p_ppp_act_tx_buffer++ = LCP_ADR_CTL_FIELD_COMP_LEN;
         ppp_tx_buffer_size += LCP_ADR_CTL_FIELD_COMP_LEN;
      }
      break;
   case PPP_PROTOCOL_IPCP:
      *p_ppp_act_tx_buffer++ = IPCP_IP_ADR;
      *p_ppp_act_tx_buffer++ = IPCP_IP_ADR_LEN;
      memcpy(p_ppp_act_tx_buffer, ppp_local_ip, sizeof(ppp_local_ip));
      p_ppp_act_tx_buffer += 4;
      ppp_tx_buffer_size += IPCP_IP_ADR_LEN;
      if (ppp_rx_flag & PPP_RX_FLAG_GATEWAY81)
      {
         *p_ppp_act_tx_buffer++ = IPCP_GATEWAY_81;
         *p_ppp_act_tx_buffer++ = IPCP_IP_ADR_LEN;
         memcpy(p_ppp_act_tx_buffer, ppp_gateway_81, sizeof(ppp_gateway_81));
         p_ppp_act_tx_buffer += 4;
         ppp_tx_buffer_size += IPCP_IP_ADR_LEN;
      }
      if (ppp_rx_flag & PPP_RX_FLAG_GATEWAY82)
      {
         *p_ppp_act_tx_buffer++ = IPCP_GATEWAY_82;
         *p_ppp_act_tx_buffer++ = IPCP_IP_ADR_LEN;
         memcpy(p_ppp_act_tx_buffer, ppp_gateway_82, sizeof(ppp_gateway_82));
         p_ppp_act_tx_buffer += 4;
         ppp_tx_buffer_size += IPCP_IP_ADR_LEN;
      }
      if (ppp_rx_flag & PPP_RX_FLAG_GATEWAY83)
      {
         *p_ppp_act_tx_buffer++ = IPCP_GATEWAY_83;
         *p_ppp_act_tx_buffer++ = IPCP_IP_ADR_LEN;
         memcpy(p_ppp_act_tx_buffer, ppp_gateway_83, sizeof(ppp_gateway_83));
         p_ppp_act_tx_buffer += 4;
         ppp_tx_buffer_size += IPCP_IP_ADR_LEN;
      }
      if (ppp_rx_flag & PPP_RX_FLAG_GATEWAY84)
      {
         *p_ppp_act_tx_buffer++ = IPCP_GATEWAY_84;
         *p_ppp_act_tx_buffer++ = IPCP_IP_ADR_LEN;
         memcpy(p_ppp_act_tx_buffer, ppp_gateway_84, sizeof(ppp_gateway_84));
         p_ppp_act_tx_buffer += 4;
         ppp_tx_buffer_size += IPCP_IP_ADR_LEN;
      }
      break;
   default:
      break;
   }                            // switch(prot)
   ppp_protocol_type = prot;
   ppp_packet_header(PPP_CONFIGURE_REQUEST, ppp_local_configure_request_id++);
   ppp_transmit_buffer((unsigned char *) ppp_tx_buffer, ppp_tx_buffer_size);
}                               // end ppp_send_configure_request()


void ppp_send_configure_ack(PPP_PROTOCOL_TYPE prot)
{
   p_ppp_act_tx_buffer = PPP_TX_BUFFER_START;
   p_ppp_act_tx_buffer += PPP_OPTIONS_OFS + 2;
   ppp_tx_buffer_size = PPP_OPTIONS_OFS + 2;

   while (p_ppp_new_packet_option <
          p_ppp_new_packet + (ppp_new_packet_size - 2))
   {
      *p_ppp_act_tx_buffer++ = *p_ppp_new_packet_option++;
      ppp_tx_buffer_size++;
   }
   ppp_protocol_type = prot;
   ppp_packet_header(PPP_CONFIGURE_ACK, ppp_remote_configure_req_id);
   ppp_transmit_buffer((unsigned char *) ppp_tx_buffer, ppp_tx_buffer_size);
}                               // end ppp_send_configure_ack()


void ppp_send_configure_nak_reject(PPP_PROTOCOL_TYPE prot)
{
   unsigned int i;
   unsigned char PacketCode = PPP_CONFIGURE_REJECT;
   unsigned char ip_remote[IP_ADR_SIZE] = IP_REMOTE_DEFAULT;

   p_ppp_act_tx_buffer = PPP_TX_BUFFER_START;
   p_ppp_act_tx_buffer += PPP_OPTIONS_OFS + 2;
   ppp_tx_buffer_size = PPP_OPTIONS_OFS + 2;

   switch (prot)
   {
   case PPP_PROTOCOL_LCP:
      while (p_ppp_new_packet_option <
             p_ppp_new_packet + (ppp_new_packet_size - 2))
      {
         switch (*p_ppp_new_packet_option)
         {
         case LCP_MAX_RECEIVE_UNIT:
            p_ppp_new_packet_option += LCP_MAX_RECEIVE_UNIT_LEN;
            break;
         case LCP_ACCM:
            p_ppp_new_packet_option += LCP_ACCM_LEN;
            break;
         case LCP_AUTHENTICATION_TYPE:
            if (p_ppp_new_packet_option[LCP_OPTIONS_DATA_OFS] == 0xc0
                && p_ppp_new_packet_option[LCP_OPTIONS_DATA_OFS + 1] == 0x23)
            {
               p_ppp_new_packet_option += LCP_AUTHENTICATION_TYPE_LEN;
            }
            else
            {
               *p_ppp_act_tx_buffer++ = LCP_AUTHENTICATION_TYPE;
               *p_ppp_act_tx_buffer++ = LCP_AUTHENTICATION_TYPE_LEN;
               *p_ppp_act_tx_buffer++ = 0xc0;
               *p_ppp_act_tx_buffer++ = 0x23;
               ppp_tx_buffer_size += LCP_AUTHENTICATION_TYPE_LEN;
               PacketCode = PPP_CONFIGURE_NAK;
               p_ppp_new_packet_option =
                  p_ppp_new_packet + (ppp_new_packet_size - 2);
            }
            break;
         case LCP_MAGIC_NUMBER:
            p_ppp_new_packet_option += LCP_MAGIC_NUMBER_LEN;
            break;
         case LCP_PROTOCOL_FIELD_COMP:
            p_ppp_new_packet_option += LCP_PROTOCOL_FIELD_COMP_LEN;
            break;
         case LCP_ADR_CTL_FIELD_COMP:
            p_ppp_new_packet_option += LCP_ADR_CTL_FIELD_COMP_LEN;
            break;
         default:
            for (i = 0; i < p_ppp_new_packet_option[1]; i++)
            {
               *p_ppp_act_tx_buffer++ = p_ppp_new_packet_option[i];
               ppp_tx_buffer_size++;
            }
            p_ppp_new_packet_option += p_ppp_new_packet_option[1];
         }
      }                         // end while
      break;
   case PPP_PROTOCOL_IPCP:
      while (p_ppp_new_packet_option <
             p_ppp_new_packet + (ppp_new_packet_size - 2))
      {
         switch (*p_ppp_new_packet_option)
         {
         case IPCP_IP_ADR:
            if (ppp_ip_request)
            {
               *p_ppp_act_tx_buffer++ = IPCP_IP_ADR;
               *p_ppp_act_tx_buffer++ = IPCP_IP_ADR_LEN;
               memcpy(p_ppp_act_tx_buffer, ip_remote, IP_ADR_SIZE);
               p_ppp_act_tx_buffer += IP_ADR_SIZE;
               ppp_tx_buffer_size += IPCP_IP_ADR_LEN;
               PacketCode = PPP_CONFIGURE_NAK;
               p_ppp_new_packet_option =
                  p_ppp_new_packet + (ppp_new_packet_size - 2);
            }
            else
            {
               p_ppp_new_packet_option += IPCP_IP_ADR_LEN;
            }
            break;
         default:
            for (i = 0; i < p_ppp_new_packet_option[1]; i++)
            {
               *p_ppp_act_tx_buffer++ = p_ppp_new_packet_option[i];
               ppp_tx_buffer_size++;
            }
            p_ppp_new_packet_option += p_ppp_new_packet_option[1];
         }
      }
      break;
   default:
      break;
   }                            // end switch(ppp_protocol_type)
   ppp_protocol_type = prot;
   ppp_packet_header(PacketCode, ppp_remote_configure_req_id);
   ppp_transmit_buffer((unsigned char *) ppp_tx_buffer, ppp_tx_buffer_size);
}                               // end ppp_send_configure_nak_reject()


void ppp_send_terminate_req(void)
{
   ppp_tx_buffer_size = 8;
   ppp_packet_header(PPP_TERMINATE_REQUEST, ppp_local_terminate_request_id++);
   ppp_transmit_buffer((unsigned char *) ppp_tx_buffer, ppp_tx_buffer_size);
}                               // end ppp_send_terminate_req()

void ppp_send_terminate_ack(void)
{
   ppp_tx_buffer_size = 8;
   ppp_packet_header(PPP_TERMINATE_ACK, ppp_remote_terminate_req_id);
   ppp_transmit_buffer((unsigned char *) ppp_tx_buffer, ppp_tx_buffer_size);
}                               // end ppp_send_terminate_ack()


void ppp_send_code_reject(void)
{
   unsigned char PacketCode = PPP_PROTOCOL_REJECT;

   p_ppp_act_tx_buffer = PPP_TX_BUFFER_START;
   p_ppp_act_tx_buffer += PPP_OPTIONS_OFS + 2;
   ppp_tx_buffer_size = PPP_OPTIONS_OFS + 2;
   p_ppp_new_packet_option = p_ppp_new_packet;
   if ((p_ppp_new_packet_option[0] == PPP_ADDRESS)
       && (p_ppp_new_packet_option[1] == PPP_CONTROL))
      p_ppp_new_packet_option += 2;
   if (ppp_protocol_type == PPP_PROTOCOL_LCP
       || ppp_protocol_type == PPP_PROTOCOL_PAP
       || ppp_protocol_type == PPP_PROTOCOL_IPCP)
   {
      PacketCode = PPP_CODE_REJECT;
      p_ppp_new_packet_option += 2;
   }
   else if (ppp_protocol_type == PPP_PROTOCOL_UNKNOWN)
   {
      ppp_protocol_type = PPP_PROTOCOL_LCP;
// alread set...      PacketCode = PPP_PROTOCOL_REJECT;
   }
   while (p_ppp_new_packet_option <
          p_ppp_new_packet + (ppp_new_packet_size - 2))
   {
      *p_ppp_act_tx_buffer++ = *p_ppp_new_packet_option++;
      ppp_tx_buffer_size++;
   }
   ppp_packet_header(PacketCode, ppp_local_configure_request_id++);
   ppp_transmit_buffer((unsigned char *) ppp_tx_buffer, ppp_tx_buffer_size);
}                               // end ppp_send_code_reject()


void ppp_send_echo_reply(void)
{
}                               // end ppp_send_echo_reply()

void ppp_send_authenticate_request(void)
{
   unsigned char i;

   ppp_protocol_type = PPP_PROTOCOL_PAP;
   p_ppp_act_tx_buffer = PPP_TX_BUFFER_START;
   p_ppp_act_tx_buffer += PPP_OPTIONS_OFS + 2;
   ppp_tx_buffer_size = PPP_OPTIONS_OFS + 2;
   *p_ppp_act_tx_buffer++ = peer_id_length;
   ppp_tx_buffer_size++;
   for (i = 0; i < peer_id_length; i++, ppp_tx_buffer_size++)
   {
      *p_ppp_act_tx_buffer++ = peer_id[i];
   }
   *p_ppp_act_tx_buffer++ = password_length;
   ppp_tx_buffer_size++;
   for (i = 0; i < password_length; i++, ppp_tx_buffer_size++)
   {
      *p_ppp_act_tx_buffer++ = password[i];
   }
   ppp_packet_header(PPP_CONFIGURE_REQUEST, ppp_local_configure_request_id++);
   ppp_transmit_buffer((unsigned char *) ppp_tx_buffer, ppp_tx_buffer_size);
}                               // end ppp_send_authenticate_request()

unsigned int ppp_fcs16(unsigned int fcs, unsigned char *cp, unsigned int len)
{
   while (len--)
      fcs = (fcs >> 8) ^ pgm_read_word(&fcstab[(fcs ^ *cp++) & 0xff]);
   return (fcs);
}                               // end ppp_fcs16()


void ppp_packet_header(unsigned char packet_format, unsigned char id)
{
   p_ppp_act_tx_buffer = PPP_TX_BUFFER_START;
   p_ppp_act_tx_buffer[PPP_ADDRESS_OFS] = PPP_ADDRESS;
   p_ppp_act_tx_buffer[PPP_CONTROL_OFS] = PPP_CONTROL;
   p_ppp_act_tx_buffer += 2;
   switch (ppp_protocol_type)
   {
   case PPP_PROTOCOL_LCP:
      p_ppp_act_tx_buffer[PPP_PROTOCOL_OFS] = 0xc0;
      p_ppp_act_tx_buffer[PPP_PROTOCOL_OFS + 1] = 0x21;
      break;
   case PPP_PROTOCOL_IPCP:
      p_ppp_act_tx_buffer[PPP_PROTOCOL_OFS] = 0x80;
      p_ppp_act_tx_buffer[PPP_PROTOCOL_OFS + 1] = 0x21;
      break;
   case PPP_PROTOCOL_PAP:
      p_ppp_act_tx_buffer[PPP_PROTOCOL_OFS] = 0xc0;
      p_ppp_act_tx_buffer[PPP_PROTOCOL_OFS + 1] = 0x23;
      break;
   default:
      break;
   }
   p_ppp_act_tx_buffer[PPP_CODE_OFS] = packet_format;
   p_ppp_act_tx_buffer[PPP_ID_OFS] = id;
   p_ppp_act_tx_buffer[PPP_LENGTH_OFS] =
      (((ppp_tx_buffer_size - PPP_HDLC_HEADER) >> 8) & 0x00ff);
   p_ppp_act_tx_buffer[PPP_LENGTH_OFS + 1] =
      ((ppp_tx_buffer_size - PPP_HDLC_HEADER) & 0x00ff);
}

void ppp_transmit_buffer(unsigned char *pointer, unsigned int size)
{
   unsigned int i = 0, fcs;

   fcs = ppp_fcs16(PPP_INIT_FCS16, pointer, size);
   fcs ^= 0xffff;
   pointer[size] = (fcs & 0x00ff);
   pointer[size + 1] = ((fcs >> 8) & 0x00ff);

   uart1_putchar(PPP_FLAG_SEQUENZE);
   while (i < size + PPP_HDLC_TRAILER)
   {
      if (pointer[i] < 0x20 || pointer[i] == PPP_FLAG_SEQUENZE
          || pointer[i] == PPP_CONTROL_ESCAPE)
      {
         uart1_putchar(PPP_CONTROL_ESCAPE);
         pointer[i] ^= PPP_CE_BIT;
      }
      uart1_putchar(pointer[i++]);
   }
   uart1_putchar(PPP_FLAG_SEQUENZE);
}

unsigned char ppp_interface_available(void)
{
   if (ppp_status & PPP_STATUS_UP)
      return 1;
   return 0;
}


unsigned char ppp_data_available(void)
{
   if (ppp_status & PPP_STATUS_RX_DATA)
      return 1;
   return 0;
}


void ppp_data_clear(void)
{
   ppp_status &= ~PPP_STATUS_RX_DATA;
   ppp_rx_data_packet_size = 0;
}


unsigned char ppp_data_read16(unsigned int *data)
{
   *data = 0;
   if (ppp_rx_data_packet_size > 1)
   {
      *data = *p_ppp_rx_data_packet;
      *data <<= 8;
      p_ppp_rx_data_packet++;
      *data |= *p_ppp_rx_data_packet;
      p_ppp_rx_data_packet++;
      ppp_rx_data_packet_size -= 2;
      return 1;
   }
   ppp_data_clear();
   return 0;
}


unsigned char ppp_data_read8(unsigned char *data)
{
   *data = 0;
   if (ppp_rx_data_packet_size)
   {
      *data = *p_ppp_rx_data_packet;
      p_ppp_rx_data_packet++;
      ppp_rx_data_packet_size--;
      return 1;
   }
   ppp_data_clear();
   return 0;
}

unsigned char ppp_transmit_data(unsigned int packetSize)
{
   if ((ipcp_automaton_state == PPP_STATE_OPENED)
       && (lcp_automaton_state == PPP_STATE_OPENED)
       && (ppp_status & PPP_STATUS_UP))
   {
      p_ppp_act_tx_buffer = PPP_TX_BUFFER_START;
      p_ppp_act_tx_buffer[PPP_ADDRESS_OFS] = PPP_ADDRESS;
      p_ppp_act_tx_buffer[PPP_CONTROL_OFS] = PPP_CONTROL;
      p_ppp_act_tx_buffer += 2;
      p_ppp_act_tx_buffer[PPP_PROTOCOL_OFS] = 0x00;
      p_ppp_act_tx_buffer[PPP_PROTOCOL_OFS + 1] = 0x21;
      ppp_transmit_buffer(PPP_TX_BUFFER_START, packetSize + PPP_HDLC_HEADER);
      return 1;
   }
   return 0;
}


unsigned char ppp_write_ready(unsigned int count)
{
   if (count < PPP_MTU)
      return 1;
   return 0;
}

unsigned char ppp_write_data(unsigned char *data, unsigned int count)
{
   unsigned int length;

   data += ETH_HEADER_SIZE;
   count -= ETH_HEADER_SIZE;
   if (count > PPP_MTU)
      length = PPP_MTU;
   else
      length = count;
   memcpy(((unsigned char *) ppp_tx_buffer + PPP_HDLC_HEADER), data, length);
   return ppp_transmit_data(length);
}

unsigned int ppp_ready(void)
{
   if ((ppp_status & PPP_STATUS_UP) && (ppp_status & PPP_STATUS_RX_DATA))
      return (DEVICE_READ_READY_RECEIVE | DEVICE_READ_READY_FRAME);
   return 0;
}


unsigned int ppp_read_frame_16bit(void)
{
   unsigned int data;

   ppp_data_read16(&data);
   return data;
}


unsigned int ppp_read_frame_16bit_le(void)
{
   unsigned char data_low;
   unsigned int data_high;

   ppp_data_read8(&data_low);
   ppp_data_read8((unsigned char *) &data_high);
   data_high <<= 8;
   data_high |= data_low;
   return data_high;
}

void ppp_read_frame_dummy(unsigned int size)
{
   unsigned char data;

   while (size)
   {
      ppp_data_read8(&data);
      size--;
#ifdef __CODEVISIONAVR__
#asm("wdr");
#else
      wdt_reset();
#endif
   }
}


void ppp_read_frame(void *dest, unsigned int size)
{
   unsigned char *p_data;

   p_data = dest;
   while (size)
   {
      ppp_data_read8(p_data);
      p_data++;
      size--;
#ifdef __CODEVISIONAVR__
#asm("wdr");
#else
      wdt_reset();
#endif
   }
}


#define MODEM_ESC_CTL			0x01
void ppp_receive_data(unsigned char data)
{
   static unsigned char escape_control = 0;

   switch (data)
   {
   case PPP_FLAG_SEQUENZE:
      if (ppp_rx_buffersize == 0)
      {
         ppp_rx_table_buffer[ppp_rx_table_write_index].addr = p_ppp_rx_buffer;
      }
      else
      {
         ppp_rx_new_packet_count++;
         ppp_rx_table_buffer[ppp_rx_table_write_index++].size =
            ppp_rx_buffersize;
         if (ppp_rx_table_write_index > (PPP_MAX_DATA_PACKETS - 1))
            ppp_rx_table_write_index = 0;
         ppp_rx_buffersize = 0;
         ppp_rx_table_buffer[ppp_rx_table_write_index].addr = p_ppp_rx_buffer;
      }
      break;
   case PPP_CONTROL_ESCAPE:
      escape_control = MODEM_ESC_CTL;
      break;
   default:
      *p_ppp_rx_buffer = data;
      if (escape_control)
      {
         *p_ppp_rx_buffer ^= PPP_CE_BIT;
         escape_control = 0;
      }
      *p_ppp_rx_buffer++;
      if (p_ppp_rx_buffer >= PPP_RX_BUFFER_END)
      {
         p_ppp_rx_buffer = PPP_RX_BUFFER_START;
         while (ppp_rx_table_buffer[ppp_rx_table_write_index].addr <
                PPP_RX_BUFFER_END)
            *p_ppp_rx_buffer++ =
               *ppp_rx_table_buffer[ppp_rx_table_write_index].addr++;
         ppp_rx_table_buffer[ppp_rx_table_write_index].addr =
            PPP_RX_BUFFER_START;
      }
      ppp_rx_buffersize++;
   }                            // end switch(data)
}                               // end RX_UART1() interrupt
