
/*****************************************************************************
 *
 *                           "tcp.c"
 *                   -----------------------------
 *
 *  Version:    2.06
 *  File:     	..\..\tcp.c
 *  Created:    08.04.2003
 *  Date:       24.04.2005
 *  Author:     Copyright (C) 2001-2005
 *              Udo Jakobza - FTZ Leipzig; D-04107 Leipzig; Wchterstr. 13
 *				info@easytoweb.net
 *  Func:		implements the TCP/IP-stack and provides a simple API to the user
 *  License:
 *    This library is free software; you can redistribute it and/or modify it
 *    under the terms of the GNU Lesser General Public License as published by
 *    the Free Software Foundation; either version 2.1 of the License, or
 *    (at your option) any later version. This library is distributed in the hope
 *    that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *    See the GNU Lesser General Public License for more details.
 *	  see: http://www.gnu.org/copyleft/lesser.html
 *    You should have received a copy of the GNU Lesser General Public License
 *    along with this library; if not, write to the Free Software Foundation,
 *    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *    Die Bibliothek ist freie Software; Sie drfen sie unter den Bedingungen der
 *    GNU Lesser General Public License, wie von der Free Software Foundation
 *    verffentlicht, weiterverteilen und/oder modifizieren; entweder gem
 *    Version 2.1 der Lizenz oder (nach Ihrer Option) jeder spteren Version.
 *    Diese Bibliothek wird in der Hoffnung weiterverbreitet, da sie ntzlich
 *    sein wird, jedoch OHNE IRGENDEINE GARANTIE, auch ohne die implizierte
 *    Garantie der MARKTREIFE oder der VERWENDBARKEIT FR EINEN BESTIMMTEN ZWECK.
 *    Mehr Details finden Sie in der GNU Lesser General Public License.
 *	  see: http://www.gnu.org/copyleft/lesser.de.html
 *    Sie sollten eine Kopie der GNU Lesser General Public License zusammen mit
 *    dieser Bibliothek erhalten haben; falls nicht, schreiben Sie an die FSF,
 *    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 *  history:
 *		1.) 03.11.2003 Version 2.02
 *			a.)	- function tcp_handle_retransmission() displays the data frames
 *                inside of Debug-Mode (debug tcp ON);
 *		2.) 08.01.2004 Version 2.03
 *			a.)	- add some debug informations;
 *			b.)	- big Error inside of "tcp_process_frame()"
 *                the Passive Open doesn't work together with "tcp_abort()" and
 *                therefore a new Connection with a other "tcp_sorce_port" doesn't
 *                work - some new Commands to break a Passive Connection
 *		3.) 25.02.2004 Version 2.03
 *			a.)	- functions "tcp_prepare_frame()" and "tcp_prepare_data_frame()" use
 *                now the IP-function "ip_header_write()"
 *		4.) 16.06.2004 Version 2.04
 *			a.)	- new function "tcp_get_struct()" checks the TCP-Connection Handler
 *      5.) 29.06.2004
 *          a.) - new macro ADD_PSH_TO_PAYLOAD adds PSH-Flag to each payload packet
 *                otherwise problems with delays in some windows-tcp/ip-stacks occur
 *      6.) 02.07.2004
 *          a.) - rename macro "ADD_PSH_TO_PAYLOAD" to "TCP_ADD_PSH_FLAG"
 *      7.) 19.08.2004
 *          a.) - new function "tcp_alive_enable()" and "tcp_alive_disable()"
 *              - enbale with Macro "TCP_ALIVE_ENABLE"
 *
 *****************************************************************************/
#include "project.h"

#ifdef TCP_ENABLE

#include <stdio.h>
#include <string.h>
#include <avr/pgmspace.h>
#include HARDWARE_H_FILEPATH
#include DEVICE_H_FILEPATH
#include ETHERNET_H_FILEPATH
#include BUFFER_H_FILEPATH
#include IP_H_FILEPATH
#ifdef IPSEC_ENABLE
#include IPSEC_H_FILEPATH
#endif
#include ARP_RARP_H_FILEPATH
#include ICMP_H_FILEPATH
#include TCP_H_FILEPATH
#ifdef WEB_DEBUG_MEASURE
#include MEASURE_H_FILEPATH
#endif
#if defined(WEB_DEBUG) || defined(WEB_DEBUG_TCP)
#include DEBUG_H_FILEPATH
#include WEB_DEBUG_H_FILEPATH
#include TCP_DEBUG_H_FILEPATH
#endif

#ifdef __CODEVISIONAVR__
#pragma regalloc-
#endif
TCP_STRUCTURE tcp_struct[TCP_MAX_CONNECTIONS];
unsigned int tcp_isn_high_number;


#ifdef WEB_DEBUG_TCP
static PROGMEM char NewLine[] = "\n";

unsigned int tcp_rw_frame_count = 0;
#endif
#ifdef __CODEVISIONAVR__
#pragma regalloc+
#endif
void tcp_tx_buffer_ack(TCP_STRUCTURE * p_struct, unsigned long tcp_ack);
void tcp_rx_buffer_read_frame(unsigned char tcp_connection,
                              unsigned int tcp_data_count,
                              unsigned long tcp_sequence);
void tcp_prepare_frame(unsigned char tcp_connection, unsigned int tcp_code);
unsigned char tcp_prepare_data_frame(unsigned char tcp_connection,
                                     TCP_DATA_FRAME_MODE mode,
                                     unsigned char act_tx_buffer);
void tcp_start_retry_timer(TCP_STRUCTURE * p_struct);
void tcp_start_wait_timer(TCP_STRUCTURE * p_struct);
void tcp_stop_timer(TCP_STRUCTURE * p_struct);
void tcp_handle_retransmission(unsigned char tcp_connection);
void tcp_handle_timeout(TCP_STRUCTURE * p_struct);
void tcp_close_flags(unsigned char tcp_connection);
void tcp_connect_flags(unsigned char tcp_connection);
unsigned char tcp_tx_buffer_free(TCP_STRUCTURE * p_struct);
void tcp_generate_isn(TCP_STRUCTURE * p_struct, TCP_GENERATE_ISN_MODE mode);


TCP_STRUCTURE *tcp_get_struct(unsigned char tcp_connection)
{
   if ((tcp_connection != TCP_WRONG_CONNECTIONS)
       && (tcp_connection < TCP_MAX_CONNECTIONS))
      return &tcp_struct[tcp_connection];
   return 0x0;
}


void tcp_init_entry(TCP_STRUCTURE * p_struct)
{
   memset(p_struct, 0x0, sizeof(TCP_STRUCTURE));
   p_struct->tcp_state_machine = CLOSED;
   p_struct->tcp_socket_status = SOCK_PASSIV_OPEN | SOCK_DISCONNECTED;
   p_struct->tcp_socket_error = SOCK_ERR_OK;
   p_struct->tcp_window_local = TCP_MAX_WINDOW_SIZE;
   p_struct->tcp_rx.gap = TCP_RX_BUFFER_SIZE;
#ifdef WEB_DEBUG_TCP
   p_struct->last_tcp_socket_error = SOCK_ERR_OK;
   p_struct->last_tcp_state_machine = TCP_UNREACHED;
#endif
   memset(p_struct->tcp_ip_remote, 0xff, IP_ADR_SIZE);
#ifdef TCP_ALIVE_ACTIV_ENABLE
   p_struct->tcp_alive_time = TCP_ALIVE_DEFAULT_TIME;
#endif
}

void tcp_init(void)
{
   unsigned char Lva;
   TCP_STRUCTURE *p_struct;

   p_struct = &tcp_struct[0];
   for (Lva = 0; Lva < TCP_MAX_CONNECTIONS; Lva++)
   {
      tcp_init_entry(p_struct);
      p_struct++;
   }
}

void tcp_init_connection(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;

   p_struct = tcp_get_struct(tcp_connection);
   if (p_struct)
   {
      tcp_init_entry(p_struct);
   }
}

unsigned char tcp_passive_open(unsigned int tcp_local_port)
{
   unsigned char Lva;
   TCP_STRUCTURE *p_struct;

   p_struct = &tcp_struct[0];
   for (Lva = 0; Lva < TCP_MAX_CONNECTIONS; Lva++)
   {
      if (!(p_struct->tcp_flags & TCP_FLAG_OPEN) &&
          !(p_struct->tcp_socket_status & SOCK_READY_CONNECT) &&
          (p_struct->tcp_state_machine == CLOSED))
      {
         tcp_init_entry(p_struct);
         p_struct->tcp_flags = TCP_FLAG_PASSIV | TCP_FLAG_OPEN;
         p_struct->tcp_state_machine = LISTEN;
         p_struct->tcp_socket_status =
            SOCK_READY_CONNECT | SOCK_PASSIV_OPEN | SOCK_DISCONNECTED;
         p_struct->tcp_port_local = tcp_local_port;     // set port we want
                                                        // to listen to
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
            printf_P(PSTR("TCP-O(%u,x) Passive Open: %u\n"), Lva, tcp_local_port);
#endif
         return Lva;
      }
      p_struct++;
   }
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      printf_P(PSTR("TCP-O(x,x) Passive Open: %u ERROR\n"), tcp_local_port);
#endif
   return TCP_WRONG_CONNECTIONS;
}


void tcp_passive_reopen(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;
   unsigned int temp_tcp_local_port;

   if (tcp_get_struct(tcp_connection))
   {
      p_struct = &tcp_struct[tcp_connection];
      if (!(p_struct->tcp_flags & TCP_FLAG_OPEN) &&
          (p_struct->tcp_flags & TCP_FLAG_PASSIV) &&
          !(p_struct->tcp_socket_status & SOCK_READY_CONNECT) &&
          (p_struct->tcp_state_machine == CLOSED))
      {
         temp_tcp_local_port = p_struct->tcp_port_local;
         tcp_init_entry(p_struct);
         p_struct->tcp_flags = TCP_FLAG_PASSIV | TCP_FLAG_OPEN;
         p_struct->tcp_state_machine = LISTEN;
         p_struct->tcp_socket_status =
            SOCK_READY_CONNECT | SOCK_PASSIV_OPEN | SOCK_DISCONNECTED;
         p_struct->tcp_port_local = temp_tcp_local_port;        // set port
                                                                // we want to
                                                                // listen to
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
            printf_P(PSTR("TCP-O(%u,x) Passive Reopen: %u\n"), tcp_connection,
                   p_struct->tcp_port_local);
#endif
      }
   }
}


unsigned char tcp_active_open(unsigned char *dest_ip_address,
                              unsigned int dest_port,
                              unsigned int source_port)
{
   unsigned char Lva;
   TCP_STRUCTURE *p_struct;

#ifdef ARP_CACHE_ENABLE
   unsigned char *mac_adr;
#endif

   p_struct = &tcp_struct[0];
   for (Lva = 0; Lva < TCP_MAX_CONNECTIONS; Lva++)
   {
      if (!(p_struct->tcp_flags & TCP_FLAG_OPEN) &&
          ((p_struct->tcp_state_machine == CLOSED)
           || (p_struct->tcp_state_machine == LISTEN)))
      {
         // --------------------------------------------------------------------------------
         // Reset TCP-Struct-Entry
         // --------------------------------------------------------------------------------
         //
         tcp_init_entry(p_struct);
         p_struct->tcp_flags = TCP_FLAG_ACTIVE | TCP_FLAG_OPEN;
         // --------------------------------------------------------------------------------
         // Calculate ARP-Entry
         // --------------------------------------------------------------------------------
         //
#ifdef ARP_ENABLE
#ifdef ARP_CACHE_ENABLE
         mac_adr = arp_cache(dest_ip_address, 0x0);
         if (mac_adr != 0)
         {
            memcpy(p_struct->tcp_mac_remote, mac_adr, 6);
            p_struct->tcp_flags |= TCP_FLAG_IP_ADDR_RESOLVED;
         }
         else
         {
            arp_prepare_request(dest_ip_address, ARP_REQUEST_IP);
            p_struct->tcp_last_frame_sent = TCP_ARP_REQUEST;
            tcp_start_retry_timer(p_struct);
         }
#else
         arp_prepare_request(dest_ip_address, ARP_REQUEST_IP);
         p_struct->tcp_last_frame_sent = TCP_ARP_REQUEST;
         tcp_start_retry_timer(p_struct);
#endif
#endif
         // --------------------------------------------------------------------------------
         // Set new value
         // --------------------------------------------------------------------------------
         //
         p_struct->tcp_socket_status =
            SOCK_READY_CONNECT | SOCK_ACTIV_OPEN | SOCK_DISCONNECTED;
         p_struct->tcp_port_local = source_port;
         p_struct->tcp_port_remote = dest_port;
         memcpy(p_struct->tcp_ip_remote, dest_ip_address, IP_ADR_SIZE);
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
         {
            printf_P(PSTR("TCP-O(%u,x) Active Open: %u "), Lva, dest_port);
            debug_print_ip(PSTR(""), dest_ip_address);
         }
#endif
         return Lva;
      }
      p_struct++;
   }
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      printf_P(PSTR("TCP-O(x,x) Active Open: %u ERROR\n"), dest_port);
#endif
   return TCP_WRONG_CONNECTIONS;
}


void tcp_close(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;

   if (tcp_get_struct(tcp_connection))
   {
      p_struct = &tcp_struct[tcp_connection];
      switch (p_struct->tcp_state_machine)
      {
      case LISTEN:
      case SYN_SENT:
         tcp_close_flags(tcp_connection);
         p_struct->tcp_socket_status = 0;
         break;
      case SYN_RCVD:
      case ESTABLISHED:
         p_struct->tcp_socket_status |= SOCK_DISCONNECTING;
         p_struct->tcp_flags |= TCP_FLAG_CLOSE_REQUESTED;
         break;
      default:
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
            printf_P(PSTR("TCP-T(%u,x) \"tcp_close\" wrong state: %S\n"),
                   tcp_connection,
                   pgm_read_word(&tcp_state_name[p_struct->tcp_state_machine]));
#endif
         break;
      }
   }
}


void tcp_abort(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;

   if (tcp_get_struct(tcp_connection))
   {
      p_struct = &tcp_struct[tcp_connection];
      switch (p_struct->tcp_state_machine)
      {
      case LISTEN:
      case SYN_SENT:
         tcp_close(tcp_connection);
         break;
      case SYN_RCVD:
      case ESTABLISHED:
         p_struct->tcp_socket_status |= SOCK_DISCONNECTING;
         p_struct->tcp_flags |= TCP_FLAG_ABORT;
         break;
      default:
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
            printf_P(PSTR("TCP-T(%u,x) \"tcp_abort\" wrong state: %S\n"),
                   tcp_connection,
                   pgm_read_word(&tcp_state_name[p_struct->tcp_state_machine]));
#endif
         break;
      }
   }
}


unsigned char tcp_get_error(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;
   unsigned char error = 0;

   if (tcp_get_struct(tcp_connection))
   {
      p_struct = &tcp_struct[tcp_connection];
      error = p_struct->tcp_socket_error;
#ifdef WEB_DEBUG_TCP
      p_struct->last_tcp_socket_error = p_struct->tcp_socket_error;
#endif
      p_struct->tcp_socket_error = SOCK_ERR_OK;
   }
   return error;
}

#ifdef TCP_ALIVE_ACTIV_ENABLE
void tcp_alive_set_time(unsigned char tcp_connection, unsigned int alive_time)
{
   TCP_STRUCTURE *p_struct;

   if (tcp_get_struct(tcp_connection))
   {
      p_struct = &tcp_struct[tcp_connection];
      if (alive_time == 0)
         alive_time = TCP_ALIVE_DEFAULT_TIME;
      if (alive_time < TCP_ALIVE_MIN_TIME)
         alive_time = TCP_ALIVE_MIN_TIME;
      if (alive_time > TCP_ALIVE_MAX_TIME)
         alive_time = TCP_ALIVE_MAX_TIME;
      p_struct->tcp_alive_time = alive_time;
#ifdef WEB_DEBUG_TCP
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
         printf_P(PSTR("TCP-A(%u,x) Alive time (%u0ms)\n"), tcp_connection,
                alive_time * 5);
#endif
   }
}


void tcp_alive_enable(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;

   if (tcp_get_struct(tcp_connection))
   {
      p_struct = &tcp_struct[tcp_connection];
      p_struct->tcp_alive_delay = p_struct->tcp_alive_time;
      p_struct->tcp_alive_timeout_delay =
         (unsigned int) (TCP_ALIVE_TIMEOUT_FACTOR * p_struct->tcp_alive_time);
      p_struct->tcp_alive_50ms_bit = 0;
      p_struct->tcp_alive_flag = TCP_ALIVE_FLAG_ENABLE;
      if (p_struct->tcp_state_machine == ESTABLISHED)
         p_struct->tcp_alive_flag |= TCP_ALIVE_FLAG_OPEN;
#ifdef WEB_DEBUG_TCP
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      {
         printf_P(PSTR("TCP-A(%u,x) Alive enable "), tcp_connection);
         if (p_struct->tcp_alive_flag & TCP_ALIVE_FLAG_OPEN)
            printf_P(PSTR("+ open "));
         printf_P(PSTR("(%u0ms; timeout:%u0ms)\n"), p_struct->tcp_alive_time * 5,
                p_struct->tcp_alive_timeout_delay * 5);
      }
#endif
   }
}


void tcp_alive_disable(unsigned char tcp_connection)
{
   if (tcp_get_struct(tcp_connection))
   {
      tcp_struct[tcp_connection].tcp_alive_flag = 0;
#ifdef WEB_DEBUG_TCP
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
         printf_P(PSTR("TCP-A(%u,x) Alive disable\n"), tcp_connection);
#endif
   }
}

void tcp_alive_main(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;

   p_struct = &tcp_struct[tcp_connection];
   if (p_struct->tcp_alive_flag & TCP_ALIVE_FLAG_OPEN)
   {
      if (p_struct->tcp_alive_50ms_bit)
      {
         p_struct->tcp_alive_50ms_bit = 0;
         if (p_struct->tcp_alive_timeout_delay)
            p_struct->tcp_alive_timeout_delay--;
         else
         {
#ifdef WEB_DEBUG_TCP
            if (web_debug_flag & WEB_DEBUG_TCP_BIT)
               printf_P(PSTR("TCP-A(%u,x) Alive Close\n"), tcp_connection);
#endif
            p_struct->tcp_alive_flag = 0;
            tcp_close(tcp_connection);
         }
         if (p_struct->tcp_alive_delay)
            p_struct->tcp_alive_delay--;
         else
         {
            tcp_prepare_frame(tcp_connection, TCP_CODE_ACK_ALIVE);
            p_struct->tcp_alive_delay = p_struct->tcp_alive_time;
         }
      }
   }
}
#endif

void tcp_rx_buffer_clear(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;

   p_struct = tcp_get_struct(tcp_connection);
   if (p_struct)
   {
      p_struct->tcp_socket_status &= ~SOCK_DATA_AVAILABLE;
      p_struct->tcp_rx.wtop = 0;
      p_struct->tcp_rx.rtop = 0;
      p_struct->tcp_rx.gap = TCP_RX_BUFFER_SIZE;
      p_struct->tcp_window_local = TCP_MAX_WINDOW_SIZE;
   }
}

unsigned char *tcp_rx_buffer_read(unsigned char tcp_connection,
                                  unsigned char *data_dest,
                                  unsigned int *tcp_data_count,
                                  TCP_RX_BUFFER_READ_MODE read_mode)
{
   TCP_STRUCTURE *p_struct;
   unsigned int temp;

   p_struct = tcp_get_struct(tcp_connection);
   if (p_struct)
   {
      if (p_struct->tcp_socket_status & SOCK_DATA_AVAILABLE)
      {
         if (tcp_data_count == 0x0)
            return 0x0;
         else if (read_mode == TCP_RX_BUFFER_GET_SIZE)
         {
            *tcp_data_count = (TCP_RX_BUFFER_SIZE - p_struct->tcp_rx.gap);
            return &p_struct->tcp_rx.data_frame[p_struct->tcp_rx.rtop];
         }
         else if ((read_mode == TCP_RX_BUFFER_READ_DATA)
                  || (read_mode == TCP_RX_BUFFER_SET_DATA))
         {
            if ((read_mode == TCP_RX_BUFFER_READ_DATA) && (data_dest == 0x0))
               return 0x0;
            if (*tcp_data_count > (TCP_RX_BUFFER_SIZE - p_struct->tcp_rx.gap))
               *tcp_data_count = (TCP_RX_BUFFER_SIZE - p_struct->tcp_rx.gap);
            temp = TCP_RX_BUFFER_SIZE - p_struct->tcp_rx.rtop;
            if (temp < *tcp_data_count) // found overflow
            {
               if (read_mode == TCP_RX_BUFFER_READ_DATA)
               {
                  memcpy(data_dest,
                         &p_struct->tcp_rx.data_frame[p_struct->tcp_rx.rtop],
                         temp);
                  data_dest += temp;
               }
               p_struct->tcp_rx.rtop = 0;
               temp = (*tcp_data_count - temp);
            }
            else
               temp = *tcp_data_count;  // set full size
            if (read_mode == TCP_RX_BUFFER_READ_DATA)
               memcpy(data_dest,
                      &p_struct->tcp_rx.data_frame[p_struct->tcp_rx.rtop],
                      temp);
            p_struct->tcp_rx.rtop += temp;
            p_struct->tcp_rx.gap += *tcp_data_count;
            if (p_struct->tcp_rx.wtop == p_struct->tcp_rx.rtop)
            {
               p_struct->tcp_rx.wtop = 0;
               p_struct->tcp_rx.rtop = 0;
               p_struct->tcp_rx.gap = TCP_RX_BUFFER_SIZE;
               p_struct->tcp_socket_status &= ~SOCK_DATA_AVAILABLE;
            }
            if (p_struct->tcp_rx.gap < TCP_MAX_WINDOW_SIZE)
               p_struct->tcp_window_local = p_struct->tcp_rx.gap;
            else if (p_struct->tcp_rx.gap >= TCP_MAX_WINDOW_SIZE)
               p_struct->tcp_window_local = TCP_MAX_WINDOW_SIZE;
            return &p_struct->tcp_rx.data_frame[p_struct->tcp_rx.rtop];
         }
      }
   }
   return 0x0;
}


unsigned int tcp_tx_buffer_write_internal(unsigned char tcp_connection,
                                          unsigned char *ram_data,
                                          PGM_P flash_data,
                                          unsigned int datalen,
                                          TCP_TX_BUFFER_WRITE_MODE write_mode)
{
   static unsigned char last_tx_buffer = 0;
   static unsigned char last_tx_age = 0;
   TCP_STRUCTURE *p_struct;
   unsigned char temp_count, Lva;
   unsigned int temp_datalen, block_datalen;
   TCP_TX_BUFFER_STRUCTURE *p_tx_buffer;

   if (tcp_get_struct(tcp_connection))
   {
      p_struct = &tcp_struct[tcp_connection];
      if ((p_struct->tcp_state_machine == ESTABLISHED) ||
          (p_struct->tcp_state_machine == CLOSE_WAIT))
      {
         temp_datalen = 0;
         for (temp_count = 0; temp_count < TCP_TX_BUFFER_COUNT; temp_count++)
         {
            p_tx_buffer = &p_struct->tcp_tx[temp_count];
            if ((p_tx_buffer->flags & TCP_TX_FLAG_FREE) ||
                (p_tx_buffer->flags & TCP_TX_FLAG_HALF_FULL))
            {
               if (!(p_tx_buffer->flags & TCP_TX_FLAG_HALF_FULL))
               {
                  p_tx_buffer->data_position =
                     &p_tx_buffer->data_frame[TCP_DATA_OFS];
                  p_tx_buffer->data_count = 0;
               }
               temp_datalen += TCP_TX_SEGMENT_SIZE - p_tx_buffer->data_count;
            }
         }
         if (write_mode == TCP_TX_BUFFER_GET_SIZE)
            return temp_datalen;
         else if ((write_mode == TCP_TX_BUFFER_WRITE_FLASH_DATA)
                  || (write_mode == TCP_TX_BUFFER_WRITE_RAM_DATA)
                  || (write_mode == TCP_TX_BUFFER_SET_DATA))
         {
            if (datalen == 0)
               return 0;
            if ((write_mode == TCP_TX_BUFFER_WRITE_FLASH_DATA)
                && (flash_data == 0x0))
            {
#ifdef WEB_DEBUG_TCP
               if (web_debug_flag & WEB_DEBUG_TCP_BIT)
                  printf_P(PSTR("TCP-D(%u,%u) buffer_prepare: no flash buffer\n"),
                         tcp_connection, tcp_rw_frame_count);
#endif
               return 0;
            }
            if ((write_mode == TCP_TX_BUFFER_WRITE_RAM_DATA)
                && (ram_data == 0x0))
            {
#ifdef WEB_DEBUG_TCP
               if (web_debug_flag & WEB_DEBUG_TCP_BIT)
                  printf_P(PSTR("TCP-D(%u,%u) buffer_prepare: no ram buffer\n"),
                         tcp_connection, tcp_rw_frame_count);
#endif
               return 0;
            }
            if (datalen > temp_datalen)
               datalen = temp_datalen;
            temp_datalen = datalen;
            temp_count = 0;
            while (temp_count < TCP_TX_BUFFER_COUNT)
            {
               if (last_tx_buffer >= TCP_TX_BUFFER_COUNT)
                  last_tx_buffer = 0;
               p_tx_buffer = &p_struct->tcp_tx[last_tx_buffer];
               if ((p_tx_buffer->flags & TCP_TX_FLAG_FREE) ||
                   (p_tx_buffer->flags & TCP_TX_FLAG_HALF_FULL))
               {
                  block_datalen = temp_datalen;
                  if (block_datalen >
                      (TCP_TX_SEGMENT_SIZE - p_tx_buffer->data_count))
                     block_datalen =
                        TCP_TX_SEGMENT_SIZE - p_tx_buffer->data_count;
                  if (write_mode == TCP_TX_BUFFER_WRITE_RAM_DATA)
                  {
                     memcpy(p_tx_buffer->data_position, ram_data,
                            block_datalen);
                     ram_data += block_datalen;
                  }
                  if (write_mode == TCP_TX_BUFFER_WRITE_FLASH_DATA)
                  {
                     memcpy_P(p_tx_buffer->data_position, flash_data,
                             block_datalen);
                     flash_data += block_datalen;
                  }
#ifdef WEB_DEBUG_TCP
                  if (web_debug_flag & WEB_DEBUG_TCP_BIT)
                     printf_P(PSTR("TCP-D(%u,%u) buffer_prepare: %uByte\n"),
                            tcp_connection, tcp_rw_frame_count,
                            block_datalen);
#endif
                  temp_datalen -= block_datalen;
                  p_tx_buffer->data_position += block_datalen;
                  p_tx_buffer->data_count += block_datalen;
                  if (p_tx_buffer->data_count & 0x01)
                     *p_tx_buffer->data_position = 0x0;
                  p_tx_buffer->flags &= ~TCP_TX_FLAG_FREE;
                  p_tx_buffer->flags |= TCP_TX_FLAG_FULL;
                  if (p_tx_buffer->data_count < TCP_TX_SEGMENT_SIZE)
                     p_tx_buffer->flags |= TCP_TX_FLAG_HALF_FULL;
                  else
                     p_tx_buffer->flags &= ~TCP_TX_FLAG_HALF_FULL;



                  if (last_tx_age == 0)
                  {
                     for (Lva = 0; Lva < TCP_TX_BUFFER_COUNT; Lva++)
                     {
                        if (p_struct->tcp_tx[Lva].flags & TCP_TX_FLAG_FULL)
                           p_struct->tcp_tx[Lva].age +=
                              TCP_TX_BUFFER_AGE_OVER;
                     }
                     last_tx_age = TCP_TX_BUFFER_AGE_OVER;
                  }
                  p_tx_buffer->age = last_tx_age;
                  last_tx_age++;
                  if (temp_datalen == 0)
                     return datalen;
               }
               last_tx_buffer++;
               temp_count++;
            }
         }
      }
   }
   return 0;
}


unsigned int tcp_tx_buffer_write(unsigned char tcp_connection,
                                 unsigned char *ram_data,
                                 unsigned int datalen,
                                 TCP_TX_BUFFER_WRITE_MODE write_mode)
{
   return tcp_tx_buffer_write_internal(tcp_connection, ram_data, 0x0, datalen,
                                       write_mode);
}


unsigned int tcp_tx_buffer_write_f(unsigned char tcp_connection,
                                   PGM_P flash_data,
                                   unsigned int datalen,
                                   TCP_TX_BUFFER_WRITE_MODE write_mode)
{
   return tcp_tx_buffer_write_internal(tcp_connection, 0x0, flash_data,
                                       datalen, write_mode);
}


#if defined(WEB_DEBUG_TCP)||defined(WEB_DEBUG_TCP_OFFLINE)
static PROGMEM char Cclsd[] = "CLOSED";
static PROGMEM char Clsn[]  = "LISTEN";
static PROGMEM char Csysn[] = "SYN_SENT";
static PROGMEM char Csyrv[] = "SYN_RCVD";
static PROGMEM char Cetab[] = "ESTABLISHED";
static PROGMEM char Cfwt1[] = "FIN_WAIT_1";
static PROGMEM char Cfwt2[] = "FIN_WAIT_2";
static PROGMEM char Cclwt[] = "CLOSE_WAIT";
static PROGMEM char Cclsg[] = "CLOSING";
static PROGMEM char Cltak[] = "LAST_ACK";
static PROGMEM char Ctmwt[] = "TIME_WAIT";
static PROGMEM char Cunrc[] = "UNREACHED";
PGM_P tcp_state_name[12] PROGMEM = {
   Cclsd, Clsn, Csysn, Csyrv,
   Cetab, Cfwt1, Cfwt2, Cclwt,
   Cclsg, Cltak, Ctmwt, Cunrc
};
#endif
void tcp_main(void)
{
#ifdef TCP_TX_DELAY_ENABLE
   static unsigned char tcp_data_delay = 0;
#endif
   static unsigned char tcp_connection = 0;
   static unsigned char tcp_tx_buffer_count = 0;
   unsigned char temp_count;
   TCP_STRUCTURE *p_struct;

   temp_count = 0;
   while (temp_count < TCP_MAX_CONNECTIONS)
   {
      tcp_connection++;
      if (tcp_connection >= TCP_MAX_CONNECTIONS)
         tcp_connection = 0;
      tcp_passive_reopen(tcp_connection);
      p_struct = &tcp_struct[tcp_connection];
      if (p_struct->tcp_flags & TCP_FLAG_OPEN)
         break;
      temp_count++;
   }
   if (temp_count >= TCP_MAX_CONNECTIONS)
      return;
   if (p_struct->tcp_flags & TCP_FLAG_OPEN)
   {
#ifdef TCP_TX_DELAY_ENABLE
      tcp_data_delay++;
      if (tcp_data_delay == TCP_TX_DELAY_COUNT)
      {
         tcp_data_delay = 0;
         tcp_prepare_data_frame(tcp_connection, TCP_DATA_FRAME_1ST_SEND,
                                TCP_TX_BUFFER_UNDEF);
      }
#else
      tcp_prepare_data_frame(tcp_connection, TCP_DATA_FRAME_1ST_SEND,
                             TCP_TX_BUFFER_UNDEF);
#endif
      // --------------------------------------------------------------------------------
      // check, if retansmit a CTRL-Frame
      // --------------------------------------------------------------------------------
      //
      if (p_struct->tcp_flags & TCP_FLAG_TIMER_CTRL_RETRANSMIT)
      {
         if (p_struct->tcp_timer >= TCP_RETRY_TIMEOUT)
         {
            p_struct->tcp_timer = 0;
            if (p_struct->tcp_retry_counter)
            {
               p_struct->tcp_retry_counter--;
               tcp_handle_retransmission(tcp_connection);
            }
            else
            {
               tcp_stop_timer(p_struct);
               tcp_close_flags(tcp_connection);
               tcp_handle_timeout(p_struct);
            }
         }
      }
      // --------------------------------------------------------------------------------
      // check, if retansmit a Data-Frame
      // --------------------------------------------------------------------------------
      //
      temp_count = 0;
      while (temp_count < TCP_TX_BUFFER_COUNT)
      {
         tcp_tx_buffer_count++;
         if (tcp_tx_buffer_count >= TCP_TX_BUFFER_COUNT)
            tcp_tx_buffer_count = 0;
         if (p_struct->tcp_tx[tcp_tx_buffer_count].
             flags & TCP_TX_FLAG_RETRANSMIT)
         {
            if (p_struct->tcp_tx[tcp_tx_buffer_count].timer >=
                TCP_RETRY_TIMEOUT)
            {
               p_struct->tcp_tx[tcp_tx_buffer_count].timer = 0;
               if (p_struct->tcp_tx[tcp_tx_buffer_count].retry_counter)
               {
                  if (tcp_prepare_data_frame
                      (tcp_connection, TCP_DATA_FRAME_RETRANSMIT,
                       tcp_tx_buffer_count))
                  {
#ifdef WEB_DEBUG_TCP
                     p_struct->tcp_tx_repeat_data += p_struct->tcp_tx[tcp_tx_buffer_count].data_count;
                     // count of repeated transmit data, see function tcp_process_frame()
#endif
                     p_struct->tcp_tx[tcp_tx_buffer_count].retry_counter--;
                  }
               }
               else
               {
                  p_struct->tcp_tx[tcp_tx_buffer_count].flags &=
                     ~TCP_TX_FLAG_RETRANSMIT;
                  tcp_close_flags(tcp_connection);
                  p_struct->tcp_socket_error |= SOCK_ERR_TCP_TIMEOUT;
               }
            }
         }
         temp_count++;
      }
      // --------------------------------------------------------------------------------
      // check, if Timeout of FIN-Frame arrive
      // --------------------------------------------------------------------------------
      //
      if (p_struct->tcp_flags & TCP_FLAG_TIMER_RUNNING)
      {
         if (p_struct->tcp_timer >= TCP_FIN_TIMEOUT)
         {
#ifdef WEB_DEBUG_TCP
            if (web_debug_flag & WEB_DEBUG_TCP_BIT)
               printf_P(PSTR("TCP-T(%u,%u) FIN Timeout (State: %S)\n"),
                      tcp_connection, tcp_rw_frame_count,
                      pgm_read_word(&tcp_state_name[p_struct->tcp_state_machine]));
#endif
            tcp_close_flags(tcp_connection);
            p_struct->tcp_socket_error |= SOCK_ERR_TCP_TIMEOUT;
         }
      }
      switch (p_struct->tcp_state_machine)
      {
      case CLOSED:
      case LISTEN:
         if ((p_struct->tcp_flags & TCP_FLAG_ACTIVE) &&
             (p_struct->tcp_flags & TCP_FLAG_IP_ADDR_RESOLVED))
         {
            tcp_generate_isn(p_struct, TCP_GENERATE_ISN_1ST);
            p_struct->tcp_sequence_ack = 0;
            tcp_prepare_frame(tcp_connection, TCP_CODE_SYN);
            p_struct->tcp_last_frame_sent = TCP_SYN_FRAME;
            tcp_start_retry_timer(p_struct);
            p_struct->tcp_state_machine = SYN_SENT;
         }
         break;
      case SYN_RCVD:
      case ESTABLISHED:
#ifdef TCP_ALIVE_ACTIV_ENABLE
         tcp_alive_main(tcp_connection);
#endif
         if ((p_struct->tcp_flags & TCP_FLAG_CLOSE_REQUESTED) &&
             (p_struct->tcp_sequence_next == p_struct->tcp_sequence_unack) &&
             tcp_tx_buffer_free(p_struct))
         {
            p_struct->tcp_sequence_unack++;
            tcp_prepare_frame(tcp_connection, TCP_CODE_FIN | TCP_CODE_ACK);
            p_struct->tcp_last_frame_sent = TCP_FIN_FRAME;
            tcp_start_retry_timer(p_struct);
            p_struct->tcp_state_machine = FIN_WAIT_1;
         }
         if ((p_struct->tcp_flags & TCP_FLAG_ABORT) &&
             (p_struct->tcp_sequence_next == p_struct->tcp_sequence_unack) &&
             tcp_tx_buffer_free(p_struct))
         {
#ifdef WEB_DEBUG_TCP
            if (web_debug_flag & WEB_DEBUG_TCP_BIT)
               printf_P(PSTR("TCP-X(%u,%u) Abort\n"), tcp_connection,
                      tcp_rw_frame_count);
#endif
            tcp_prepare_frame(tcp_connection, TCP_CODE_FIN);
            tcp_close_flags(tcp_connection);
         }
         break;
      case CLOSE_WAIT:
         if ((p_struct->tcp_sequence_next == p_struct->tcp_sequence_unack) &&
             tcp_tx_buffer_free(p_struct))
         {
            p_struct->tcp_sequence_unack++;
            tcp_prepare_frame(tcp_connection, TCP_CODE_FIN | TCP_CODE_ACK);
            p_struct->tcp_last_frame_sent = TCP_FIN_FRAME;
            tcp_start_retry_timer(p_struct);
            p_struct->tcp_state_machine = LAST_ACK;
         }
         break;
      default:
         break;
      }
#ifdef WEB_DEBUG_TCP
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      {
         if (p_struct->last_tcp_state_machine != p_struct->tcp_state_machine)
         {
            printf_P(PSTR("TCP-S(%u,%u) State-M: %S\n"), tcp_connection,
                   tcp_rw_frame_count,
                   pgm_read_word(&tcp_state_name[p_struct->tcp_state_machine]));
            p_struct->last_tcp_state_machine = p_struct->tcp_state_machine;
         }
      }
#endif
   }                            // if(p_struct->tcp_flags & TCP_FLAG_OPEN)
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
   {
      if (p_struct->tcp_socket_error != p_struct->last_tcp_socket_error)
      {
         printf_P(PSTR("TCP-E(%u,%u) Socket Error: "), tcp_connection,
                tcp_rw_frame_count);
#ifdef WEB_DEBUG_TCP_OFFLINE
         tcp_debug_display_socket_error(p_struct);
#else
         printf_P(NewLine);
         p_struct->last_tcp_socket_error = p_struct->tcp_socket_error;
#endif
      }
   }
#endif
}


void tcp_process_frame(void)
{
   unsigned int tcp_source_port;        // tcp source port
   unsigned int tcp_dest_port;
   unsigned long tcp_sequence;
   unsigned long tcp_ack;
   unsigned int tcp_code;
   unsigned char tcp_header_size;
   unsigned int tcp_data_count;
   unsigned char tcp_connection;
   TCP_STRUCTURE *p_struct;

#ifdef WEB_DEBUG_TCP
   unsigned char tcp_option_frame_buffer[40];
#endif

#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_TCP_S)
#endif
#ifdef WEB_DEBUG_TCP
      tcp_rw_frame_count++;
#endif
   tcp_source_port = DEVICE_READ_IP_FRAME_16BIT();
   tcp_dest_port = DEVICE_READ_IP_FRAME_16BIT();
   p_struct = &tcp_struct[0];
   for (tcp_connection = 0; tcp_connection < TCP_MAX_CONNECTIONS;
        tcp_connection++)
   {
      if (p_struct->tcp_flags & TCP_FLAG_OPEN)
      {
         if (p_struct->tcp_flags & TCP_FLAG_ACTIVE)
         {
            if ((tcp_dest_port == p_struct->tcp_port_local)
                && (tcp_source_port == p_struct->tcp_port_remote)
                &&
                (!memcmp
                 (ip_of_last_frame, p_struct->tcp_ip_remote, IP_ADR_SIZE)))
               break;
         }
         else if (p_struct->tcp_flags & TCP_FLAG_PASSIV)
         {
            if (tcp_dest_port == p_struct->tcp_port_local)
            {
//               if (!memcmp_P(p_struct->tcp_ip_remote, IP_FFFFFFFF, IP_ADR_SIZE))
               if(*(unsigned long *)&p_struct->tcp_ip_remote[0] == IP_FFFFFFFF)
               {
                  memcpy(p_struct->tcp_ip_remote, ip_of_last_frame,
                         IP_ADR_SIZE);
                  p_struct->tcp_port_remote = tcp_source_port;
#ifndef DEVICE_PPP
                  memcpy(p_struct->tcp_mac_remote, mac_of_last_frame, 6);
#endif
#ifdef WEB_DEBUG_TCP
                  if (web_debug_flag & WEB_DEBUG_TCP_BIT)
                     printf_P(PSTR("TCP-O(%u,%u) Open: %u\n"), tcp_connection,
                            tcp_rw_frame_count, p_struct->tcp_port_local);
#endif
               }
               if ((tcp_source_port == p_struct->tcp_port_remote) &&
                   (!memcmp
                    (ip_of_last_frame, p_struct->tcp_ip_remote, IP_ADR_SIZE)))
                  break;
            }
         }
      }
      p_struct++;
   }
   if (tcp_connection >= TCP_MAX_CONNECTIONS)
   {
#ifdef WEB_DEBUG_MEASURE
      MEASURE_SETTIME(MEA_TCP_E1)
#endif
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      {
         printf_P(PSTR("TCP-E(x,%u) %u.%u.%u.%u / %u / %u\n"), tcp_rw_frame_count,
                ip_of_last_frame[0], ip_of_last_frame[1], ip_of_last_frame[2],
                ip_of_last_frame[3], tcp_dest_port, tcp_source_port);
         p_struct = &tcp_struct[0];
         for (tcp_connection = 0; tcp_connection < TCP_MAX_CONNECTIONS;
              tcp_connection++)
         {
            printf_P(PSTR("TCP-E(%u,%u) %u.%u.%u.%u / %u / %u"), tcp_connection,
                   tcp_rw_frame_count, p_struct->tcp_ip_remote[0],
                   p_struct->tcp_ip_remote[1], p_struct->tcp_ip_remote[2],
                   p_struct->tcp_ip_remote[3], p_struct->tcp_port_local,
                   p_struct->tcp_port_remote);
#ifdef WEB_DEBUG_TCP_OFFLINE
            tcp_debug_display_flags(p_struct);
#else
            printf_P(NewLine);
#endif
            p_struct++;
         }
      }
#endif
      return;
   }
   tcp_sequence = DEVICE_READ_IP_FRAME_16BIT();
   tcp_sequence <<= 16;
   tcp_sequence |= DEVICE_READ_IP_FRAME_16BIT();
   tcp_ack = DEVICE_READ_IP_FRAME_16BIT();
   tcp_ack <<= 16;
   tcp_ack |= DEVICE_READ_IP_FRAME_16BIT();
   tcp_code = DEVICE_READ_IP_FRAME_16BIT();
   tcp_header_size = (unsigned char) ((tcp_code & DATA_OFS_MASK) >> 10);
   tcp_data_count = ip_header.length - IP_HEADER_SIZE - tcp_header_size;
   p_struct->tcp_window_remote = DEVICE_READ_IP_FRAME_16BIT();
   DEVICE_READ_IP_FRAME_DUMMY(4);
   if (tcp_data_count > p_struct->tcp_rx.gap)
   {
#ifdef WEB_DEBUG_MEASURE
      MEASURE_SETTIME(MEA_TCP_E2)
#endif
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
         printf_P(PSTR("TCP-E(%u,%u) Frame too long: %u\n"), tcp_connection,
                tcp_rw_frame_count, tcp_data_count);
#endif
      return;
   }
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
   {
      printf_P(PSTR("TCP-R(%u,%u) "), tcp_connection, tcp_rw_frame_count);
      if (tcp_code & TCP_CODE_FIN)
         printf_P(PSTR("FIN,"));
      if (tcp_code & TCP_CODE_SYN)
         printf_P(PSTR("SYN,"));
      if (tcp_code & TCP_CODE_RST)
         printf_P(PSTR("RST,"));
      if (tcp_code & TCP_CODE_PSH)
         printf_P(PSTR("PSH,"));
      if (tcp_code & TCP_CODE_ACK)
         printf_P(PSTR("ACK,"));
      if (tcp_code & TCP_CODE_URG)
         printf_P(PSTR("URG,"));
      printf_P(PSTR(" - %lu:%lu, %uByte\n"), tcp_sequence, tcp_ack, tcp_data_count);
   }
#endif
   if (tcp_header_size > TCP_HEADER_SIZE)
   {
      tcp_header_size -= TCP_HEADER_SIZE;
#ifndef WEB_DEBUG_TCP
      DEVICE_READ_IP_FRAME_DUMMY(tcp_header_size);
#else
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      {
         if (tcp_header_size > sizeof(tcp_option_frame_buffer))
            tcp_header_size = sizeof(tcp_option_frame_buffer);
         DEVICE_READ_IP_FRAME_DATA(tcp_option_frame_buffer, tcp_header_size);
         printf_P(PSTR("TCP-R(%u,%u) Option: %uByte  "), tcp_connection,
                tcp_rw_frame_count, tcp_header_size);
         debug_display_frame(tcp_option_frame_buffer, tcp_header_size, 0);
      }
      else
         DEVICE_READ_IP_FRAME_DUMMY(tcp_header_size);
#endif
   }
#ifdef TCP_ALIVE_ACTIV_ENABLE
   p_struct->tcp_alive_timeout_delay =
      (unsigned int) (TCP_ALIVE_TIMEOUT_FACTOR * p_struct->tcp_alive_time);
   p_struct->tcp_alive_delay = p_struct->tcp_alive_time;
#endif
   switch (p_struct->tcp_state_machine)
   {
   case CLOSED:
      if (!(tcp_code & TCP_CODE_RST))
      {
         if (tcp_code & TCP_CODE_ACK)
         {
            p_struct->tcp_sequence_next = tcp_ack;
            tcp_prepare_frame(tcp_connection, TCP_CODE_RST);
            p_struct->tcp_socket_status &= SOCK_DATA_AVAILABLE;
         }
         else
         {
            p_struct->tcp_sequence_next = 0;
            p_struct->tcp_sequence_ack = tcp_sequence + tcp_data_count;
            if (tcp_code & (TCP_CODE_SYN | TCP_CODE_FIN))
               p_struct->tcp_sequence_ack++;
            tcp_prepare_frame(tcp_connection, TCP_CODE_RST | TCP_CODE_ACK);
         }
      }
      break;
   case LISTEN:
      if (tcp_code & TCP_CODE_RST)
      {
         p_struct->tcp_socket_error |= SOCK_ERR_TCP;
         tcp_close_flags(tcp_connection);
      }
      else
      {
         if (tcp_code & TCP_CODE_ACK)
         {
            p_struct->tcp_socket_error |= SOCK_ERR_TCP;
            p_struct->tcp_sequence_next = tcp_ack;
            tcp_prepare_frame(tcp_connection, TCP_CODE_RST);
            tcp_close_flags(tcp_connection);
         }
         else if (tcp_code & TCP_CODE_SYN)
         {
            tcp_generate_isn(p_struct, TCP_GENERATE_ISN_1ST);
            p_struct->tcp_sequence_ack = tcp_sequence + 1;
            tcp_prepare_frame(tcp_connection, TCP_CODE_SYN | TCP_CODE_ACK);
            p_struct->tcp_last_frame_sent = TCP_SYN_ACK_FRAME;
            tcp_start_retry_timer(p_struct);
            p_struct->tcp_state_machine = SYN_RCVD;
         }
      }
      break;
   case SYN_SENT:
      if (tcp_code & TCP_CODE_ACK)
      {
         if (tcp_code & TCP_CODE_RST)
         {
            tcp_close_flags(tcp_connection);
            p_struct->tcp_socket_error |= SOCK_ERR_CONN_RESET;
            break;
         }
         else if (tcp_ack != p_struct->tcp_sequence_unack)
         {
            p_struct->tcp_socket_error |= SOCK_ERR_TCP_ACK;
#ifdef WEB_DEBUG_TCP
            if (web_debug_flag & WEB_DEBUG_TCP_BIT)
               printf_P(PSTR("TCP-E(%u,%u) wrong ACK:%lu (expected:%lu)\n"),
                      tcp_connection, tcp_rw_frame_count, tcp_ack,
                      p_struct->tcp_sequence_unack);
#endif
            tcp_sequence = p_struct->tcp_sequence_next; // save old seq_next
            p_struct->tcp_sequence_next = tcp_ack;
            tcp_prepare_frame(tcp_connection, TCP_CODE_RST);
            p_struct->tcp_sequence_next = tcp_sequence;
            break;
         }
      }
      if (tcp_code & TCP_CODE_SYN)
      {
         p_struct->tcp_sequence_ack = tcp_sequence;
#ifdef WEB_DEBUG_TCP
         p_struct->tcp_irs_number = tcp_sequence;
#endif
         p_struct->tcp_sequence_ack++;
         if (tcp_code & TCP_CODE_ACK)
         {
            tcp_stop_timer(p_struct);
            p_struct->tcp_sequence_next = p_struct->tcp_sequence_unack;
            tcp_prepare_frame(tcp_connection, TCP_CODE_ACK);
            tcp_connect_flags(tcp_connection);
         }
         else
         {
            tcp_stop_timer(p_struct);
            tcp_prepare_frame(tcp_connection, TCP_CODE_SYN | TCP_CODE_ACK);
            p_struct->tcp_last_frame_sent = TCP_SYN_ACK_FRAME;
            tcp_start_retry_timer(p_struct);
            p_struct->tcp_state_machine = SYN_RCVD;
         }
      }
      break;
   default:                    // not(CLOSED, LISTEN, SYN_SENT)
      if (tcp_code & TCP_CODE_RST)
      {
         tcp_close_flags(tcp_connection);
         p_struct->tcp_socket_error |= SOCK_ERR_CONN_RESET;
         break;
      }
      if (tcp_code & TCP_CODE_SYN)
      {
         tcp_prepare_frame(tcp_connection, TCP_CODE_RST);
         tcp_close_flags(tcp_connection);
         p_struct->tcp_socket_error |= SOCK_ERR_REMOTE;
         break;
      }
      if (!(tcp_code & TCP_CODE_ACK))
      {
         p_struct->tcp_socket_error |= SOCK_ERR_TCP;
         break;
      }
#ifdef TCP_ALIVE_PASSIV_ENABLE
      if ((tcp_data_count == 1)
          && ((tcp_sequence + 1) == p_struct->tcp_sequence_ack))
      {
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
            printf_P(PSTR("TCP-A(%u,%u) Passiv-Alive-ACK\n"), tcp_connection,
                   tcp_rw_frame_count);
#endif
         tcp_prepare_frame(tcp_connection, TCP_CODE_ACK);
         break;
      }
#endif
      if (tcp_sequence > p_struct->tcp_sequence_ack)
      {
         p_struct->tcp_socket_error |= SOCK_ERR_TCP_SEQ;
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
            printf_P(PSTR("TCP-E(%u,%u) frame lost SEQ:%lu (expected:%lu)\n"),
                   tcp_connection, tcp_rw_frame_count, tcp_sequence,
                   p_struct->tcp_sequence_ack);
#endif
      }
      if (tcp_ack > p_struct->tcp_sequence_unack)
      {
         p_struct->tcp_socket_error |= SOCK_ERR_TCP_ACK;
         tcp_prepare_frame(tcp_connection, TCP_CODE_ACK);
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
            printf_P(PSTR("TCP-E(%u,%u) wrong ACK:%lu (expected:%lu)\n"),
                   tcp_connection, tcp_rw_frame_count, tcp_ack,
                   p_struct->tcp_sequence_unack);
#endif
         break;
      }
      if (tcp_ack < p_struct->tcp_sequence_unack)
      {
         if (p_struct->tcp_state_machine == ESTABLISHED)
            tcp_tx_buffer_ack(p_struct, tcp_ack);
      }
      else if (tcp_ack == p_struct->tcp_sequence_unack)
      {
         tcp_stop_timer(p_struct);
         p_struct->tcp_sequence_next = p_struct->tcp_sequence_unack;
         switch (p_struct->tcp_state_machine)
         {
         case ESTABLISHED:
            tcp_tx_buffer_ack(p_struct, tcp_ack);
            break;
         case SYN_RCVD:
            tcp_connect_flags(tcp_connection);
            break;
         case FIN_WAIT_1:
            p_struct->tcp_state_machine = FIN_WAIT_2;
            break;
         case CLOSING:
            p_struct->tcp_state_machine = TIME_WAIT;
            break;
         case LAST_ACK:
            tcp_close_flags(tcp_connection);
            break;
         case TIME_WAIT:
            tcp_prepare_frame(tcp_connection, TCP_CODE_ACK);
            p_struct->tcp_timer = 0;
            break;
         default:
            break;
         }
      }
      if (((p_struct->tcp_state_machine == ESTABLISHED) ||
           (p_struct->tcp_state_machine == FIN_WAIT_1) ||
           (p_struct->tcp_state_machine == FIN_WAIT_2)) && (tcp_data_count))
      {
         tcp_rx_buffer_read_frame(tcp_connection, tcp_data_count,
                                  tcp_sequence);
      }
      if (tcp_code & TCP_CODE_FIN)
      {
         switch (p_struct->tcp_state_machine)
         {
         case SYN_RCVD:
         case ESTABLISHED:
            p_struct->tcp_state_machine = CLOSE_WAIT;
            break;
         case FIN_WAIT_1:
            p_struct->tcp_state_machine = CLOSING;
            p_struct->tcp_socket_status &= SOCK_DATA_AVAILABLE;
            p_struct->tcp_socket_status |= SOCK_DISCONNECTING;
            break;
         case FIN_WAIT_2:
            tcp_start_wait_timer(p_struct);
            p_struct->tcp_state_machine = TIME_WAIT;
            p_struct->tcp_socket_status &= SOCK_DATA_AVAILABLE;
            p_struct->tcp_socket_status |= SOCK_DISCONNECTING;
            break;
         case TIME_WAIT:
            p_struct->tcp_timer = 0;
            break;
         default:
            break;
         }
         p_struct->tcp_sequence_ack++;
         tcp_prepare_frame(tcp_connection, TCP_CODE_ACK);
      }
   }                            // switch(p_struct->tcp_state_machine)
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
   {
      if (p_struct->last_tcp_state_machine != p_struct->tcp_state_machine)
      {
         printf_P(PSTR("TCP-S(%u,%u) State-P: %S\n"), tcp_connection,
                tcp_rw_frame_count,
                pgm_read_word(&tcp_state_name[p_struct->tcp_state_machine]));
         p_struct->last_tcp_state_machine = p_struct->tcp_state_machine;
      }
      if (p_struct->tcp_socket_error != p_struct->last_tcp_socket_error)
      {
         printf_P(PSTR("TCP-E(%u,%u) Socket Error: "), tcp_connection,
                tcp_rw_frame_count);
#ifdef WEB_DEBUG_TCP_OFFLINE
         tcp_debug_display_socket_error(p_struct);
#else
         printf_P(NewLine);
         p_struct->last_tcp_socket_error = p_struct->tcp_socket_error;
#endif
      }
   }
#endif
#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_TCP_E3)
#endif
}


void tcp_tx_buffer_ack(TCP_STRUCTURE * p_struct, unsigned long tcp_ack)
{
   static unsigned long last_tcp_ack = 0;
   unsigned char temp_count, overflow = 0;

   temp_count = 0;
   if (last_tcp_ack > tcp_ack)
      overflow = 1;
   while (temp_count < TCP_TX_BUFFER_COUNT)
   {
      if (p_struct->tcp_tx[temp_count].flags & TCP_TX_FLAG_SENT)
      {
         if (((overflow == 0)
              && (tcp_ack >= p_struct->tcp_tx[temp_count].unack))
             || ((overflow)
                 && (tcp_ack <= p_struct->tcp_tx[temp_count].unack)))
         {
            p_struct->tcp_tx[temp_count].flags = TCP_TX_FLAG_FREE;
#ifdef WEB_DEBUG_TCP
            p_struct->tcp_tx_ok_data += p_struct->tcp_tx[temp_count].data_count;
            // count of all receive data, see function tcp_rx_buffer_read_frame()
#endif
         }
      }
      temp_count++;
   }                            // while(temp_count<TCP_TX_BUFFER_COUNT)
   last_tcp_ack = tcp_ack;
}


void tcp_rx_buffer_read_frame(unsigned char tcp_connection,
                              unsigned int tcp_data_count,
                              unsigned long tcp_sequence)
{
   TCP_STRUCTURE *p_struct;
   unsigned int temp;

   p_struct = &tcp_struct[tcp_connection];
   if (tcp_sequence < p_struct->tcp_sequence_ack)
   {
#ifdef WEB_DEBUG_TCP
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
         printf_P(PSTR("TCP-E(%u,%u) repeat ACK:%lu (expected:%lu)\n"),
                tcp_connection, tcp_rw_frame_count, tcp_sequence,
                p_struct->tcp_sequence_ack);
#endif
   }
   else if (tcp_sequence > p_struct->tcp_sequence_ack)
   {
#ifdef WEB_DEBUG_TCP
      p_struct->tcp_rx_wrong_data += tcp_data_count;    // count the wrong
                                                        // data
      p_struct->tcp_rx_lost_data +=
         (tcp_sequence - p_struct->tcp_sequence_ack);
#endif
   }
   else if (p_struct->tcp_rx.gap >= tcp_data_count)     // rx data-buffer
                                                        // empty?
   {
      p_struct->tcp_socket_status |= SOCK_DATA_AVAILABLE;
      p_struct->tcp_sequence_ack += tcp_data_count;
#ifdef WEB_DEBUG_TCP
      p_struct->tcp_rx_ok_data += tcp_data_count;
#endif
      temp = TCP_RX_BUFFER_SIZE - p_struct->tcp_rx.wtop;
      if (temp < tcp_data_count)        // found overflow
      {
         DEVICE_READ_IP_FRAME_DATA(&p_struct->tcp_rx.
                                   data_frame[p_struct->tcp_rx.wtop], temp);
         p_struct->tcp_rx.wtop = 0;
         p_struct->tcp_rx.gap -= temp;
         tcp_data_count -= temp;
#ifdef WEB_DEBUG_TCP
         if (web_debug_flag & WEB_DEBUG_TCP_BIT)
            printf_P(PSTR("TCP-R(%u,%u) RX-Buffer overflow:%u\n"), tcp_connection,
                   tcp_rw_frame_count, tcp_data_count);
#endif
      }
      DEVICE_READ_IP_FRAME_DATA(&p_struct->tcp_rx.data_frame[p_struct->tcp_rx.wtop], tcp_data_count);
      // read data
      p_struct->tcp_rx.wtop += tcp_data_count;
      p_struct->tcp_rx.gap -= tcp_data_count;
      if (p_struct->tcp_rx.gap < TCP_MAX_WINDOW_SIZE)
         p_struct->tcp_window_local = p_struct->tcp_rx.gap;
   }
   tcp_prepare_frame(tcp_connection, TCP_CODE_ACK);
}


void tcp_prepare_frame(unsigned char tcp_connection, unsigned int tcp_code)
{
   TCP_STRUCTURE *p_struct;
   unsigned char ip_offset;
   unsigned int tcp_tx_count;

   p_struct = &tcp_struct[tcp_connection];
#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_TCP_FRAME_S)
#endif
   if ((tcp_code == TCP_CODE_ACK) && (!tcp_tx_buffer_free(p_struct)))
   {
      if (tcp_prepare_data_frame
          (tcp_connection, TCP_DATA_FRAME_1ST_SEND, TCP_TX_BUFFER_UNDEF))
         return;
   }
   HOST_NETWORK_ENDIAN_16BIT(&p_struct->tcp_tx_ctrl_frame[TCP_SRCPORT_OFS],
                             &p_struct->tcp_port_local);
   HOST_NETWORK_ENDIAN_16BIT(&p_struct->tcp_tx_ctrl_frame[TCP_DESTPORT_OFS],
                             &p_struct->tcp_port_remote);
#ifdef TCP_ALIVE_ACTIV_ENABLE
   if (tcp_code == TCP_CODE_ACK_ALIVE)
      p_struct->tcp_sequence_next--;
#endif
   HOST_NETWORK_ENDIAN_32BIT(&p_struct->tcp_tx_ctrl_frame[TCP_SEQNR_OFS],
                             &p_struct->tcp_sequence_next);
#ifdef TCP_ALIVE_ACTIV_ENABLE
   if (tcp_code == TCP_CODE_ACK_ALIVE)
      p_struct->tcp_sequence_next++;
#endif
   HOST_NETWORK_ENDIAN_32BIT(&p_struct->tcp_tx_ctrl_frame[TCP_ACKNR_OFS],
                             &p_struct->tcp_sequence_ack);
   HOST_NETWORK_ENDIAN_16BIT(&p_struct->tcp_tx_ctrl_frame[TCP_WINDOW_OFS],
                             &p_struct->tcp_window_local);
   *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_CHKSUM_OFS] = 0;
   *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_URGENT_OFS] = 0;
#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_TCP_CHECKSUM_S2)
#endif
      if (tcp_code & TCP_CODE_SYN)
   {
      *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_DATA_CODE_OFS] =
         SWAPB(0x6000 | tcp_code);
      *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_DATA_OFS] =
         SWAPB(TCP_OPT_MSS);
      *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_DATA_OFS + 2] =
         SWAPB(TCP_RX_SEGMENT_SIZE);
      *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_CHKSUM_OFS] =
         ip_checksum(&p_struct->tcp_tx_ctrl_frame[TCP_SRCPORT_OFS],
                     TCP_HEADER_SIZE + TCP_OPT_MSS_SIZE,
                     p_struct->tcp_ip_remote);
      tcp_tx_count = IP_HEADER_SIZE + TCP_HEADER_SIZE + TCP_OPT_MSS_SIZE;
   }
   else
   {
      *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_DATA_CODE_OFS] =
         SWAPB(0x5000 | (0x0FFF & tcp_code));
#ifdef TCP_ALIVE_ACTIV_ENABLE
      if (tcp_code == TCP_CODE_ACK_ALIVE)
      {
         p_struct->tcp_tx_ctrl_frame[TCP_DATA_OFS] = 0xCC;
         *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_CHKSUM_OFS] =
            ip_checksum(&p_struct->tcp_tx_ctrl_frame[TCP_SRCPORT_OFS],
                        TCP_HEADER_SIZE + 1, p_struct->tcp_ip_remote);
         tcp_tx_count = IP_HEADER_SIZE + TCP_HEADER_SIZE + 1;
      }
      else
#endif
      {
         *(unsigned int *) &p_struct->tcp_tx_ctrl_frame[TCP_CHKSUM_OFS] =
            ip_checksum(&p_struct->tcp_tx_ctrl_frame[TCP_SRCPORT_OFS],
                        TCP_HEADER_SIZE, p_struct->tcp_ip_remote);
         tcp_tx_count = IP_HEADER_SIZE + TCP_HEADER_SIZE;
      }
   }
#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_TCP_CHECKSUM_E2)
#endif
   if (ip_header_write
          (p_struct->tcp_tx_ctrl_frame, &tcp_tx_count, &ip_offset,
           IP_TCP_PROT, ip_address, p_struct->tcp_ip_remote) == 0)
   {
#if defined(WEB_DEBUG_TCP)&&defined(IPSEC_ENABLE)
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      {
         printf_P(PSTR("TCP-E(%u,%u) IP-Error\n"), tcp_connection,
                tcp_rw_frame_count);
         debug_display_frame(p_struct->tcp_tx_ctrl_frame,
                             tcp_tx_count + IPSEC_OFFSET + ETH_HEADER_SIZE, 0);
      }
#endif
      return;
   }
#ifdef ETHERNET_ENABLE
   memcpy(&p_struct->
          tcp_tx_ctrl_frame[(unsigned char) (ETH_DA_OFS + ip_offset)],
          p_struct->tcp_mac_remote, 6);
   memcpy(&p_struct->
          tcp_tx_ctrl_frame[(unsigned char) (ETH_SA_OFS + ip_offset)],
          &mac_address[0], 6);
   *(unsigned int *) &p_struct->
      tcp_tx_ctrl_frame[(unsigned char) (ETH_TYPE_OFS + ip_offset)] =
      SWAPB(IP_FRAME);
#else
   memset(&p_struct->
          tcp_tx_ctrl_frame[(unsigned char) (ETH_DA_OFS + ip_offset)], 0x0,
          ETH_HEADER_SIZE);
#endif
   tcp_tx_count += ETH_HEADER_SIZE;
   if (buffer_write_command
       (p_struct->tcp_tx_ctrl_frame + ip_offset, tcp_tx_count))
   {
#ifdef WEB_DEBUG_MEASURE
      MEASURE_SETTIME(MEA_TCP_FRAME_E)
#endif
#ifdef WEB_DEBUG_TCP
         tcp_rw_frame_count++;
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
         printf_P(PSTR("TCP-W(%u,%u) Ctrl: "), tcp_connection, tcp_rw_frame_count);
#endif
   }
   else
   {
#ifdef WEB_DEBUG_TCP
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
         printf_P(PSTR("TCP-W(%u,%u) ERROR Ctrl: "), tcp_connection,
                tcp_rw_frame_count);
#endif
   }
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
   {
      if (tcp_code & TCP_CODE_FIN)
         printf_P(PSTR("FIN,"));
      if (tcp_code & TCP_CODE_SYN)
         printf_P(PSTR("SYN,"));
      if (tcp_code & TCP_CODE_RST)
         printf_P(PSTR("RST,"));
      if (tcp_code & TCP_CODE_PSH)
         printf_P(PSTR("PSH,"));
      if (tcp_code & TCP_CODE_ACK)
         printf_P(PSTR("ACK,"));
      if (tcp_code & TCP_CODE_URG)
         printf_P(PSTR("URG,"));
#ifdef TCP_ALIVE_ACTIV_ENABLE
      if (tcp_code == TCP_CODE_ACK_ALIVE)
      {
         printf_P(PSTR("ALIVE,"));
         printf_P(PSTR(" - %lu:%lu, 1Byte\n"), p_struct->tcp_sequence_next - 1,
                p_struct->tcp_sequence_ack);
      }
      else
#endif
         printf_P(PSTR(" - %lu:%lu\n"), p_struct->tcp_sequence_next,
                p_struct->tcp_sequence_ack);
   }
#endif
}


unsigned char tcp_prepare_data_frame(unsigned char tcp_connection,
                                     TCP_DATA_FRAME_MODE mode,
                                     unsigned char act_tx_buffer)
{
   TCP_STRUCTURE *p_struct;
   TCP_TX_BUFFER_STRUCTURE *p_tx_buffer;
   unsigned char ip_offset;
   unsigned int tcp_tx_data_all_count;
   unsigned char temp_count;
   unsigned char min_age = 0;

   p_struct = &tcp_struct[tcp_connection];
   if (act_tx_buffer == TCP_TX_BUFFER_UNDEF)
   {
      // min_age=0; // not nessecary because it start with
      // "(act_tx_buffer==TCP_TX_BUFFER_UNDEF)"
      for (temp_count = 0; temp_count < TCP_TX_BUFFER_COUNT; temp_count++)
      {
         if (p_struct->tcp_tx[temp_count].flags & TCP_TX_FLAG_FULL)
         {
            if ((p_struct->tcp_tx[temp_count].age < min_age)
                || (act_tx_buffer == TCP_TX_BUFFER_UNDEF))
            {
               min_age = p_struct->tcp_tx[temp_count].age;
               act_tx_buffer = temp_count;
            }
         }
      }
      if (act_tx_buffer == TCP_TX_BUFFER_UNDEF)
         return 0;
   }
   p_tx_buffer = &p_struct->tcp_tx[act_tx_buffer];
   if (p_tx_buffer->data_count > p_struct->tcp_window_remote)
      return 0;
#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_TCP_DATA_S)
#endif
   HOST_NETWORK_ENDIAN_16BIT(&p_tx_buffer->data_frame[TCP_SRCPORT_OFS],
                                &p_struct->tcp_port_local);
   HOST_NETWORK_ENDIAN_16BIT(&p_tx_buffer->data_frame[TCP_DESTPORT_OFS],
                             &p_struct->tcp_port_remote);
   if (mode == TCP_DATA_FRAME_1ST_SEND)
      p_tx_buffer->next = p_struct->tcp_sequence_next;
   HOST_NETWORK_ENDIAN_32BIT(&p_tx_buffer->data_frame[TCP_SEQNR_OFS],
                             &p_tx_buffer->next);
   HOST_NETWORK_ENDIAN_32BIT(&p_tx_buffer->data_frame[TCP_ACKNR_OFS],
                             &p_struct->tcp_sequence_ack);
#ifdef TCP_ADD_PSH_FLAG
   *(unsigned int *) &p_tx_buffer->data_frame[TCP_DATA_CODE_OFS] =
      SWAPB(0x5000 | TCP_CODE_ACK | TCP_CODE_PSH);
#else
   *(unsigned int *) &p_tx_buffer->data_frame[TCP_DATA_CODE_OFS] =
      SWAPB(0x5000 | TCP_CODE_ACK);
#endif
   HOST_NETWORK_ENDIAN_16BIT(&p_tx_buffer->data_frame[TCP_WINDOW_OFS],
                             &p_struct->tcp_window_local);
   *(unsigned int *) &p_tx_buffer->data_frame[TCP_CHKSUM_OFS] = 0;
   *(unsigned int *) &p_tx_buffer->data_frame[TCP_URGENT_OFS] = 0;
#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_TCP_CHECKSUM_S2)
#endif
   * (unsigned int *) &p_tx_buffer->data_frame[TCP_CHKSUM_OFS] =
                  ip_checksum(&p_tx_buffer->data_frame[TCP_SRCPORT_OFS],
                               TCP_HEADER_SIZE + p_tx_buffer->data_count,
                               p_struct->tcp_ip_remote);
#ifdef WEB_DEBUG_MEASURE
   MEASURE_SETTIME(MEA_TCP_CHECKSUM_E2)
#endif
   tcp_tx_data_all_count = IP_HEADER_SIZE+TCP_HEADER_SIZE+p_tx_buffer->data_count;
   if (ip_header_write
       (p_tx_buffer->data_frame, &tcp_tx_data_all_count, &ip_offset,
        IP_TCP_PROT, ip_address, p_struct->tcp_ip_remote) == 0)
   {
#if defined(WEB_DEBUG_TCP)&&defined(IPSEC_ENABLE)
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      {
         printf_P(PSTR("TCP-E(%u,%u) IP-Error\n"), tcp_connection,
                tcp_rw_frame_count);
         debug_display_frame(p_tx_buffer->data_frame,
                             tcp_tx_data_all_count + IPSEC_OFFSET +
                             ETH_HEADER_SIZE, 0);
      }
#endif
      return 0;
   }
#ifdef ETHERNET_ENABLE
   memcpy(&p_tx_buffer->data_frame[(unsigned char) (ETH_DA_OFS + ip_offset)],
          p_struct->tcp_mac_remote, 6);
   memcpy(&p_tx_buffer->data_frame[(unsigned char) (ETH_SA_OFS + ip_offset)],
          mac_address, 6);
   *(unsigned int *) &p_tx_buffer->
      data_frame[(unsigned char) (ETH_TYPE_OFS + ip_offset)] =
      SWAPB(IP_FRAME);
#else
   memset(&p_tx_buffer->data_frame[(unsigned char) (ETH_DA_OFS + ip_offset)],
          0x0, ETH_HEADER_SIZE);
#endif
   tcp_tx_data_all_count += ETH_HEADER_SIZE;
   if (buffer_write_command
       (p_tx_buffer->data_frame + ip_offset, tcp_tx_data_all_count))
   {
      if (mode == TCP_DATA_FRAME_1ST_SEND)
      {
         p_struct->tcp_sequence_unack += p_tx_buffer->data_count;       // advance
                                                                        // UNAcknowledge
                                                                        // Sequence-number
         p_struct->tcp_sequence_next = p_struct->tcp_sequence_unack;
         p_tx_buffer->unack = p_struct->tcp_sequence_unack;
         p_tx_buffer->flags &= ~(TCP_TX_FLAG_FULL | TCP_TX_FLAG_HALF_FULL);
         p_tx_buffer->flags |= (TCP_TX_FLAG_SENT | TCP_TX_FLAG_RETRANSMIT);
         p_tx_buffer->timer = 0;
         p_tx_buffer->retry_counter = TCP_MAX_RETRYS;
      }
#ifdef WEB_DEBUG_MEASURE
      MEASURE_SETTIME(MEA_TCP_DATA_E)
#endif
#ifdef WEB_DEBUG_TCP
      tcp_rw_frame_count++;
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      {
         printf_P(PSTR("TCP-W(%u,%u) "), tcp_connection, tcp_rw_frame_count);
         if (mode == TCP_DATA_FRAME_RETRANSMIT)
            printf_P(PSTR("Data-Retransmit"));
         else
         {
            printf_P(PSTR("Data, ACK"));
         }
         printf_P(PSTR(" - %lu:%lu, %uByte\n"), p_tx_buffer->next,
                p_struct->tcp_sequence_ack, p_tx_buffer->data_count);
      }
#endif
      return 1;
   }
   else
   {
#ifdef WEB_DEBUG_TCP
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      {
         printf_P(PSTR("TCP-W(%u,%u) ERROR Data - %lu:%lu, %uByte\n"),
                tcp_connection, tcp_rw_frame_count,
                p_struct->tcp_sequence_next, p_struct->tcp_sequence_ack,
                p_tx_buffer->data_count);
      }
#endif
   }
   return 0;
}


void tcp_arp_resolved(unsigned char *new_mac)
{
   unsigned char Lva;
   TCP_STRUCTURE *p_struct;

   p_struct = &tcp_struct[0];
   for (Lva = 0; Lva < TCP_MAX_CONNECTIONS; Lva++)
   {
      if ((p_struct->
           tcp_flags & (TCP_FLAG_ACTIVE | TCP_FLAG_IP_ADDR_RESOLVED)) ==
          TCP_FLAG_ACTIVE)
      {
         if (!memcmp(ip_of_last_frame, p_struct->tcp_ip_remote, IP_ADR_SIZE))
         {
            tcp_stop_timer(p_struct);
            p_struct->tcp_flags |= TCP_FLAG_IP_ADDR_RESOLVED;
            memcpy(p_struct->tcp_mac_remote, new_mac, 6);
         }
      }
      p_struct++;
   }
}

void tcp_start_retry_timer(TCP_STRUCTURE * p_struct)
{
   p_struct->tcp_timer = 0;
   p_struct->tcp_retry_counter = TCP_MAX_RETRYS;
   p_struct->tcp_flags |= (TCP_FLAG_TIMER_RUNNING | TCP_FLAG_TIMER_CTRL_RETRANSMIT);
}

void tcp_start_wait_timer(TCP_STRUCTURE * p_struct)
{
   p_struct->tcp_timer = 0;
   p_struct->tcp_flags |= TCP_FLAG_TIMER_RUNNING;
   p_struct->tcp_flags &= ~TCP_FLAG_TIMER_CTRL_RETRANSMIT;
}

void tcp_stop_timer(TCP_STRUCTURE * p_struct)
{
   p_struct->tcp_flags &= ~(TCP_FLAG_TIMER_RUNNING | TCP_FLAG_TIMER_CTRL_RETRANSMIT);
}

#if defined(WEB_DEBUG_TCP)||defined(WEB_DEBUG_TCP_OFFLINE)

static PROGMEM char Cnone[] = "NONE";
static PROGMEM char CarpR[] = "ARP_REQUEST";
static PROGMEM char CsynF[] = "SYN_FRAME";
static PROGMEM char CsynA[] = "SYN_ACK_FRAME";
static PROGMEM char CfinF[] = "FIN_FRAME";
PGM_P tcp_retransmit_name[5] PROGMEM = {
   Cnone, CarpR, CsynF, CsynA, CfinF
};
#endif
void tcp_handle_retransmission(unsigned char tcp_connection)
{
   TCP_STRUCTURE *p_struct;

   p_struct = &tcp_struct[tcp_connection];
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      printf_P(PSTR("TCP-W(%u,%u) Retransmit: %S\n"), tcp_connection,
             tcp_rw_frame_count,
             pgm_read_word(&tcp_retransmit_name[p_struct->tcp_last_frame_sent]));
#endif
   switch (p_struct->tcp_last_frame_sent)
   {
#ifdef ARP_ENABLE
   case TCP_ARP_REQUEST:
      arp_prepare_request(p_struct->tcp_ip_remote, ARP_REQUEST_IP);
      break;
#endif
   case TCP_SYN_FRAME:
      tcp_generate_isn(p_struct, TCP_GENERATE_ISN_RETRANSMIT);
      tcp_prepare_frame(tcp_connection, TCP_CODE_SYN);
      break;
   case TCP_SYN_ACK_FRAME:
      tcp_prepare_frame(tcp_connection, TCP_CODE_SYN | TCP_CODE_ACK);
      break;
   case TCP_FIN_FRAME:
      tcp_prepare_frame(tcp_connection, TCP_CODE_FIN | TCP_CODE_ACK);
      break;
   default:
      break;
   }
}

void tcp_handle_timeout(TCP_STRUCTURE * p_struct)
{
   if ((p_struct->tcp_flags
        & (TCP_FLAG_ACTIVE | TCP_FLAG_IP_ADDR_RESOLVED))
           == TCP_FLAG_ACTIVE)
      p_struct->tcp_socket_error |= SOCK_ERR_ARP_TIMEOUT;
   else
      p_struct->tcp_socket_error |= SOCK_ERR_TCP_TIMEOUT;
}


void tcp_clock_handler(void)
{
#if(TCP_MAX_CONNECTIONS>1)
   unsigned char Lva;
#endif
   unsigned char Lvb;

   tcp_isn_high_number++;
#if(TCP_MAX_CONNECTIONS==1)
   tcp_struct[0].tcp_timer++;
   for (Lvb = 0; Lvb < TCP_TX_BUFFER_COUNT; Lvb++)
      tcp_struct[0].tcp_tx[Lvb].timer++;
#ifdef TCP_ALIVE_ACTIV_ENABLE
   tcp_struct[0].tcp_alive_50ms_bit = 1;
#endif
#else
   for (Lva = 0; Lva < TCP_MAX_CONNECTIONS; Lva++)
   {
      tcp_struct[Lva].tcp_timer++;
      for (Lvb = 0; Lvb < TCP_TX_BUFFER_COUNT; Lvb++)
         tcp_struct[Lva].tcp_tx[Lvb].timer++;
#ifdef TCP_ALIVE_ACTIV_ENABLE
      tcp_struct[Lva].tcp_alive_50ms_bit = 1;
#endif
   }
#endif
}

void tcp_close_flags(unsigned char tcp_connection)
{
   unsigned char Lva;
   TCP_STRUCTURE *p_struct;

   p_struct = &tcp_struct[tcp_connection];
   p_struct->tcp_state_machine = CLOSED;
   p_struct->tcp_flags &= TCP_FLAG_RESET_MASK;
   p_struct->tcp_socket_status &= SOCK_DATA_AVAILABLE;
   p_struct->tcp_socket_status |= SOCK_DISCONNECTED;
   for (Lva = 0; Lva < TCP_TX_BUFFER_COUNT; Lva++)
      p_struct->tcp_tx[Lva].flags = TCP_TX_FLAG_FREE;
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      printf_P(PSTR("TCP-X(%u,%u) SOCK_DISCONNECT\n"), tcp_connection,
             tcp_rw_frame_count);
#endif
#ifdef TCP_ALIVE_ACTIV_ENABLE
   p_struct->tcp_alive_flag &= ~TCP_ALIVE_FLAG_OPEN;
#ifdef WEB_DEBUG_TCP
   if ((web_debug_flag & WEB_DEBUG_TCP_BIT)
       && (p_struct->tcp_alive_flag & TCP_ALIVE_FLAG_ENABLE))
      printf_P(PSTR("TCP-A(%u,%u) Alive close\n"), tcp_connection,
             tcp_rw_frame_count);
#endif
#endif
}


void tcp_connect_flags(unsigned char tcp_connection)
{
   unsigned char Lva;
   TCP_STRUCTURE *p_struct;

   p_struct = &tcp_struct[tcp_connection];
   p_struct->tcp_state_machine = ESTABLISHED;
   p_struct->tcp_socket_status &= ~(SOCK_READY_CONNECT | SOCK_DISCONNECTED);
   p_struct->tcp_socket_status |= SOCK_CONNECTED;
   for (Lva = 0; Lva < TCP_TX_BUFFER_COUNT; Lva++)
      p_struct->tcp_tx[Lva].flags = TCP_TX_FLAG_FREE;
#ifdef WEB_DEBUG_TCP
   if (web_debug_flag & WEB_DEBUG_TCP_BIT)
      printf_P(PSTR("TCP-X(%u,%u) SOCK_CONNECTED\n"), tcp_connection,
             tcp_rw_frame_count);
#endif
#ifdef TCP_ALIVE_ACTIV_ENABLE
   if (p_struct->tcp_alive_flag & TCP_ALIVE_FLAG_ENABLE)
   {
      p_struct->tcp_alive_flag |= TCP_ALIVE_FLAG_OPEN;
#ifdef WEB_DEBUG_TCP
      if (web_debug_flag & WEB_DEBUG_TCP_BIT)
         printf_P(PSTR("TCP-A(%u,%u) Alive open\n"), tcp_connection,
                tcp_rw_frame_count);
#endif
   }
#endif
}

unsigned char tcp_tx_buffer_free(TCP_STRUCTURE * p_struct)
{
   unsigned char Lva;

   for (Lva = 0; Lva < TCP_TX_BUFFER_COUNT; Lva++)
   {
      if (!(p_struct->tcp_tx[Lva].flags & TCP_TX_FLAG_FREE))
         return 0;
   }
   return 1;
}


void tcp_generate_isn(TCP_STRUCTURE * p_struct, TCP_GENERATE_ISN_MODE mode)
{
   if (mode == TCP_GENERATE_ISN_RETRANSMIT)
      tcp_isn_high_number += 0x4000;
   p_struct->tcp_sequence_next = tcp_isn_high_number;
   p_struct->tcp_sequence_next <<= 16;
   p_struct->tcp_sequence_unack = p_struct->tcp_sequence_next + 1;
#ifdef WEB_DEBUG_TCP
   p_struct->tcp_iss_number = p_struct->tcp_sequence_next;
#endif
}


#endif // #ifdef TCP_ENABLE
