/*
 * lcd.c
 *
 *  Created on: 30.07.2011
 *      Author: cyblord
 */

#include <avr/io.h>
#include <util/delay.h>
#include <stdlib.h>
#include <compat/deprecated.h>
#include "lcd_config.h"


#define INIT_DELAY() (_delay_ms(INIT_DELAY_MS))
#define STROBE_DELAY() (_delay_us(STROBE_DELAY_US))

#define commandMode() (cbi(LCD_CTRL,LCD_RS))
#define dataMode() (sbi(LCD_CTRL,LCD_RS))

//Display Commands


#define CMD_FUNCTION	32 	//Function Set
#define BIT_DL			16
#define BIT_N			8
#define BIT_F			4

#define CMD_ONOFF		8 //Display On/Off Command
#define BIT_D			4
#define BIT_C			2
#define BIT_B			1


//Untere 4 Bit von data auf den 4-Bit Datenbus ans Display legen
void setDataBus(unsigned char data) {
	if (data & 1) sbi(LCD_DATA,LCD_D4); else cbi(LCD_DATA,LCD_D4);
	if (data & 2) sbi(LCD_DATA,LCD_D5);	else cbi(LCD_DATA,LCD_D5);
	if (data & 4) sbi(LCD_DATA,LCD_D6);	else cbi(LCD_DATA,LCD_D6);
	if (data & 8) sbi(LCD_DATA,LCD_D7);	else cbi(LCD_DATA,LCD_D7);
}

void writeMode() {
#ifndef NOBUSY
	cbi(LCD_CTRL,LCD_RW);
#endif
}

#ifndef NOBUSY
void readMode() {
	sbi(LCD_CTRL,LCD_RW);
}
#endif


void strobe() {
	sbi(LCD_CTRL,LCD_E);
	STROBE_DELAY();
	cbi(LCD_CTRL,LCD_E);
}


void waitBusy() {
#ifdef NOBUSY
	_delay_us(50);
#else
	readMode();
	commandMode();

	//D7 =Eingang
	cbi(LCD_DATA_D,LCD_D7);
	cbi(LCD_DATA,LCD_D7);
	int busy;
	do {
		sbi(LCD_CTRL,LCD_E);
		STROBE_DELAY();
		busy=bit_is_set(LCD_INPUT,LCD_D7);
		cbi(LCD_CTRL,LCD_E);

		strobe();
	}while (busy);

	//D7 = Ausgang
	sbi(LCD_DATA_D,LCD_D7);
#endif
}



//Untere 4 Bit von nibble ans LCD bertragen
void sendNibble(unsigned char nibble) {
	setDataBus(nibble);
	strobe();
}

//Ein Byte ans LCD bertragen als 2x4 Bit, hheres nibble zuerst.
void sendByte(unsigned char data) {
	sendNibble(data>>4);
	sendNibble(data);
}

void sendCommand(unsigned char cmd) {
	waitBusy();
	writeMode();
	commandMode();
	sendByte(cmd);
}

void sendData(unsigned char data) {
	waitBusy();
	writeMode();
	dataMode();
	sendByte(data);
}



void lcd_init() {

	//Portrichtungen setzen
	sbi(LCD_DATA_D,LCD_D4);
	sbi(LCD_DATA_D,LCD_D5);
	sbi(LCD_DATA_D,LCD_D6);
	sbi(LCD_DATA_D,LCD_D7);

	sbi(LCD_CTRL_D,LCD_E);
	sbi(LCD_CTRL_D,LCD_RS);
#ifndef NOBUSY
	sbi(LCD_CTRL_D,LCD_RW);
#endif

	INIT_DELAY();

	//LCD Initalisierungssequenz
	writeMode();
	commandMode();

	sendNibble(0x3); //8-Bit Mode (DL=0)
	//Function Set: 0 0 1 DL N F  

#ifdef C080204
	sendNibble(0x3); //4-Bit Mode (DL=0)
		_delay_ms(10);
	sendNibble(0x2); //4-Bit Mode (DL=0)
	_delay_ms(10);
#endif

	unsigned char cmd=CMD_FUNCTION;
	if (LCD_LINES!=1) cmd |= BIT_N; // Anzahl Zeilen
	if (LCD_5X10) cmd |= BIT_F; //Zeichenhhe

	sendCommand(cmd); // 4-Bit Modus, Zeilen und Font setzen
	_delay_ms(10);
	//Display On/Off: 0 0 0 0 1 D C B
	sendCommand(0xC); //Display on, no cursor, no blink
	//Clear Display: 0 0 0 0 0 0 1
	sendCommand(0x1); //Clear Display and Cursor home

#ifdef NOBUSY
	_delay_ms(10);
#endif
}

void lcd_print(char* string) {
	while(*string) {
		sendData(*string);
		string++;
	}
}




void lcd_goto(int line,int pos) {
	if (line==2) sendCommand(128+LCD_SECONDLINE+pos);
	else sendCommand(128+LCD_FIRSTLINE+pos);
#ifdef NOBUSY
	_delay_ms(2);
#endif
}

void lcd_clear() {
	sendCommand(0x1);
#ifdef NOBUSY
	_delay_ms(20);
#endif
}

void lcd_home() {
	sendCommand(0x2);
#ifdef NOBUSY
	_delay_ms(2);
#endif
}

void lcd_cprint(char* s) {
	lcd_clear();
	lcd_print(s);
}

void lcd_printNum(const int num,char* s, const int radix) {
	ltoa(num,s,radix);
	lcd_print(s);
}
