#define LED_PORT_DDR        DDRB
#define LED_PORT_OUTPUT     PORTB
#define R_BIT            1
#define G_BIT            3
#define B_BIT            4

#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>  
#include <util/delay.h>     

#include <avr/pgmspace.h>   
extern "C"
{
	#include "usbdrv.h"
}
#include "oddebug.h"        
#include "requests.h"       

PROGMEM char usbHidReportDescriptor[22] = {    /* USB report descriptor */
    0x06, 0x00, 0xff,              // USAGE_PAGE (Generic Desktop)
    0x09, 0x01,                    // USAGE (Vendor Usage 1)
    0xa1, 0x01,                    // COLLECTION (Application)
    0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
    0x26, 0xff, 0x00,              //   LOGICAL_MAXIMUM (255)
    0x75, 0x08,                    //   REPORT_SIZE (8)
    0x95, 0x01,                    //   REPORT_COUNT (1)
    0x09, 0x00,                    //   USAGE (Undefined)
    0xb2, 0x02, 0x01,              //   FEATURE (Data,Var,Abs,Buf)
    0xc0                           // END_COLLECTION
};

static uint8_t count = 0;
static volatile uint8_t r = 0;
static volatile uint8_t g = 0;
static volatile uint8_t b = 0;
static uint8_t rb = 0;
static uint8_t gb = 0;
static uint8_t bb = 0;

extern "C" usbMsgLen_t usbFunctionSetup(unsigned char data[8])
{
	usbRequest_t *rq = (usbRequest_t *)data;

	if((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_VENDOR)
	{
		switch(rq->bRequest)
		{
		case CUSTOM_RQ_SET_RED:
			r = rq->wValue.bytes[0];
			break;	
		case CUSTOM_RQ_SET_GREEN:
			g = rq->wValue.bytes[0];
			break;	
		case CUSTOM_RQ_SET_BLUE:
			b = rq->wValue.bytes[0];
			break;	
		}
	}
	return 0;
}

//calibrazione fine dell'oscillatore interno finch la lunghezza misurata del 
//frame USB non conicide con quella aspettata
static void calibrateOscillator(void)
{
	unsigned char step = 128;
	unsigned char trialValue = 0, optimumValue;
	int x, optimumDev, targetValue = (unsigned)(1499 * (double)F_CPU / 10.5e6 + 0.5);
 
    do
	{
		OSCCAL = trialValue + step;
		x = usbMeasureFrameLength();
		if(x < targetValue)    
			trialValue += step;
		step >>= 1;
	}
	while(step > 0);

	optimumValue = trialValue;
	optimumDev = x; 
	for(OSCCAL = trialValue - 1; OSCCAL <= trialValue + 1; OSCCAL++)
	{
		x = usbMeasureFrameLength() - targetValue;
		if(x < 0) x = -x;
		if(x < optimumDev)
		{
			optimumDev = x;
			optimumValue = OSCCAL;
		}
	}
	OSCCAL = optimumValue;
}
 
extern "C" void usbEventResetReady(void)
{
	cli();
	calibrateOscillator();
	sei();
	eeprom_write_byte(0, OSCCAL);
}

void leds()
{
	if (++count == 0) 
	{
		rb = r;
		gb = g;
		bb = b;
		if (rb) PORTB |= (1 << R_BIT);
		if (gb) PORTB |= (1 << G_BIT);
		if (bb) PORTB |= (1 << B_BIT);
	}
	if (count == rb) PORTB &= ~(1 << R_BIT);
	if (count == gb) PORTB &= ~(1 << G_BIT);
	if (count == bb) PORTB &= ~(1 << B_BIT);
}

int main(void)
{
	unsigned char i;
	wdt_enable(WDTO_1S);
	
	LED_PORT_DDR |= _BV(R_BIT);
	LED_PORT_DDR |= _BV(G_BIT);
	LED_PORT_DDR |= _BV(B_BIT);
	
	//visualizza bianco finch non  terminata l'inizializzazione USB
	PORTB |= (1<<R_BIT);
	PORTB |= (1<<G_BIT);
	PORTB |= (1<<B_BIT);

	usbInit();
	usbDeviceDisconnect(); 
	i = 255;
	while(--i)
	{             
		wdt_reset();
		_delay_ms(1);
	}
	usbDeviceConnect();
	
	PORTB &= ~(1 << R_BIT);
	PORTB &= ~(1 << G_BIT);
	PORTB &= ~(1 << B_BIT);
    
	sei();

	while(1)
	{                
		leds();		
		wdt_reset();
		usbPoll();
    }
	return 0;
}