//////////////////////////////////////////////////////////////////
// 1-Wire Bus Funktionen
// Dateiname:   onewire.c
// Headerfile:  onewire.h
// Autor: Thomas Wiens th.wiens(at)gmx.de
// Datum: 6.11.2005
//////////////////////////////////////////////////////////////////
#include "main.h"

#ifndef W1DDR
#define W1DDR	DDRC					// Datenrichtungsregister fr 1-Wire
#define W1PORT	PORTC					// Port fr den 1-Wire Pin
#define W1TXD	PC0						// Pin fr 1-Wire Transmit
#define w1PIN	PINC					// Pin fr 1-Wire Receive
#endif

/* 1 Byte ber den 1-Wire Bus lesen
	Ein Read-Time-Slot hat eine Mindestdauer von 60s. Zwischen den einzelnen Slots muss einer
	Erholungszeit von mindestens 1s liegen.
	Der Slot wird gestartet, indem der Master den Bus fr mindestens 1s auf 0 zieht, und dann
	wieder vom Pull-Up auf 5V gehen lsst. 
	Der DS1820 bertrgt daraufhin sein Bit:
	Lesen einer 0:
		Der DS1820 zieht den Bus nach der fallenden Flanke vom Master auf 0.
		Das Signal muss innerhalb von 15s nach Beginn vom Master gesampelt werden.
	Lesen einer 1:
		Der Ds1820 lsst den Bus weiterhin auf 5V (vom Pull-Up hochgezogen)
		Das Signal muss innerhalb von 15s nach Beginn vom Master gesampelt werden.
*/
unsigned char wire_read_byte(void){
	unsigned char i;
	unsigned char rbyte = 0x00, mask = 0x01;
	cli();
	for (i = 0; i < 8; i++){
		W1DDR |= (1 << W1TXD);							// Pin als Ausgang schalten
		W1PORT &= ~(1 << W1TXD);						// Pin auf 0V
		_delay_us(1.0);									// Wartezeit 1s
		W1DDR &= ~(1 << W1TXD);						// Pin als Eingang schalten -> releasen
		_delay_us(14.0);								// Wartezeit 14s (gesamt 15s nach Slotbeginn)
		
	/* Version mit (1<<i) ist ineffizient, Verion mit Maskierung spart 20 Byte im Programmspeicher
        siehe auch Application Note AVR035: Efficient C Coding for AVR
		if ((w1PIN & (1 << W1TXD)) == 1)				// Zustand des Busses abfragen
			rbyte |= (1 << i);							// 1-Signal gelesen -> Bit an entsprechender Stelle setzen
	*/	
		if ((w1PIN & (1 << W1TXD)) == 1)				// Zustand des Busses abfragen
			rbyte |= mask;								// 1-Signal gelesen -> Bit an entsprechender Stelle setzen
		mask <<= 1;
		_delay_us(45.0);								// Wartezeit 45s		
	}
	sei();
	return rbyte;										// Gelesenes Byte zurckgeben
}

/* 1 Byte ber 1-Wire Bus senden
   Begonnen wird mit dem LSB, danach werden die einzelnen Bits bertragen
   Schreiben einer 0:
        Der Master zieht den Bus auf 0 und lsst ihn fr mindestens 60s
		und maximal 120s auf 0
	Schreiben einer 1:
		Der Master zieht den Bus auf 0 und lsst ihn innerhalb von 15s vom externen
		Pull-Up wiederstand auf 5V ziehen.
	Der DS1820 sampelt den Bus nach der ersten fallenden Flanke (Start des Write Time Slots)
	in einem Fenster von 15-60s.
	
	Zwischen den einzelnen Time Slots muss eine Recovery Time von mindestens 1s liegen.
*/

void wire_send_byte(unsigned char sbyte){
	unsigned char i;
	cli();
	for (i = 0; i < 8; i++){						// Einzelne Bits ausgeben
		W1DDR |= (1 << W1TXD);						// Pin als Ausgang schalten
		W1PORT &= ~(1 << W1TXD);					// Pin auf 0V
		_delay_us(1.0);								// 1s warten
		if (sbyte & (1 << i))						// Bit im Byte ist 1
			W1DDR &= ~(1 << W1TXD);				// Pin als Eingang schalten (TriState Z) -> Releasen
		_delay_us(59.0);							// Signal fr 59s anstehen lassen
		W1DDR &= ~(1 << W1TXD);					// Pin als Eingang schalten (TriState Z) -> Releasen
		_delay_us(1.5);								// Wartezeit zwischen den Timeslots 1.5s
	}	
	sei();
}

/* *** Initialisierung fr den 1-Wire Bus ***************************
   Master zieht den Bus fr eine Mindestdauer von 480s auf 0V.
   Dann Wird der Bus ber den Pullup-Widerstand auf 5V gezogen.
   Master geht dann in den Empfangsmodus. Der DS1820 erkennt die
   steigende Flanke, wartet 15-60s und zieht dann den Bus fr
   60-240s auf 0V. Nach dem Releasen muss der Master nochmal 
   fr min. 480s warten.
   Rckgabe: 0 bei Erfolg
             1 wenn keine Rckmeldung erfolgte
*/
unsigned char wire_master_reset(void){
	unsigned char fehler = 0;
	cli();
	W1DDR |= (1 << W1TXD);							// Pin als Ausgang schalten	
	W1PORT &= ~(1 << W1TXD);						// Pin auf 0V
	_delay_us(200.0);								// 500s warten
	_delay_us(200.0);								// Einzelne Verzgerungen mit mehr als 200s funktionieren nicht
	_delay_us(100.0);
	W1DDR &= ~(1 << W1TXD);						// Pin als Eingang schalten
	_delay_us(66.0);								// 66s warten
	fehler = w1PIN & (1 << W1TXD);					// Bus auf Null gezogen?
	_delay_us(200.0);								// 480-66s = 414s warten
	_delay_us(200.0);								// Einzelne Verzgerungen mit mehr als 200s funktionieren nicht
	_delay_us(14.0);
	sei();
	return fehler;
}

/* Auslesen des Rom-Codes (Nur mglich wenn nur ein Busteilnehmer angeschlossen)
Aufruf mit einem Zeiger auf ein 8 Byte (64 Bit) groes Char-Array.
In dem Array werden die empfangenen Daten geschrieben.
Taucht ein Fehler auf wird ein Wert <>0 zurckgegeben
*/	
int wire_read_rom(unsigned char *ptrData){
	unsigned char i;
	if (wire_master_reset())
		return (-1);								// Kein Presence erhalten
	wire_send_byte(READ_ROM);							// READ ROM[33h]
	for (i = 0;i < 8; i++){
		*ptrData = wire_read_byte();
		ptrData++;
	}	
	return 0;
}

/* Liest die Temperatur aus, wenn nur EIN DS1820 am Bus hngt.
   Stromversorgung extern. Parasite Power noch nicht 
   implementiert
*/

int wire_meas_one(void) {
	unsigned char ready = 0, i, data;
	char text[50];
	int temperature = 0;
	wire_master_reset();
	wire_send_byte(SKIP_ROM);							// Nur wenn ein DS1820 am Bus
	wire_send_byte(CONVERT_T);							// Temperaturmessung starten
	cli();
	// Read-Time Slot fragt ab, ob Konvertierung beendet
	do{
		W1DDR |= (1 << W1TXD);							// Pin als Ausgang schalten
		W1PORT &= ~(1 << W1TXD);						// Pin auf 0V
		_delay_us(1.0);									// Wartezeit 1s
		W1DDR &= ~(1 << W1TXD);						// Pin als Eingang schalten -> releasen
		_delay_us(14.0);								// Wartezeit 14s (gesamt 15s nach Slotbeginn)
		if ((w1PIN & (1 << W1TXD)) == 1)				// Zustand des Busses abfragen
			ready = 1;									// 1-Signal gelesen -> Wandlung fertig
		_delay_us(45.0);								// Wartezeit 45s		
	} while (!(ready));
	wire_master_reset();
	wire_send_byte(SKIP_ROM);							// Nur wenn ein DS1820 am Bus
	wire_send_byte(READ);								// Scratchpad auslesen
	for(i = 0;i < 8; i++)	{		
		data = wire_read_byte();
		if (i==0)
			temperature = data; // LSB
		if (i==1)
			temperature |= data << 8; // MSB
		sprintf(text, "Byte %d: Wert: %d\n\r", i, data);
		uputs(text);
	}
	sprintf(text, "Temperatur %2d.%1d C\n\r", temperature >> 1, ((temperature & 1) ? 5 : 0 ));
	uputs(text);
	sei();	
	return 0;
}

/* Liest die Temperatur aus dem DS1820 dessen Seriennummer bergeben wurde.
   Rckgabewert ist die Temperatur, Kommastelle ist um eine Stelle nach rechts verschoben
   Stromversorgung extern. Parasite Power noch nicht implementiert
*/
int wire_meas_serial(unsigned char *serial)
{
	unsigned char ready = 0, i, data;
	unsigned char *sicher;
	int temperature = 0;
	sicher = serial;									// Pointer sichern
	cli();
	wire_master_reset();
	wire_send_byte(MATCH_ROM);
	// Seriennummer ausgeben	
	for (i = 0; i < 8; i++) {
		wire_send_byte(*serial);
		serial++;
	}
	wire_send_byte(CONVERT_T);							// Temperaturmessung starten
	// Read-Time Slot fragt ab, ob Konvertierung beendet
	do{
		W1DDR |= (1 << W1TXD);							// Pin als Ausgang schalten
		W1PORT &= ~(1 << W1TXD);						// Pin auf 0V
		_delay_us(1.0);									// Wartezeit 1s
		W1DDR &= ~(1 << W1TXD);						// Pin als Eingang schalten -> releasen
		_delay_us(14.0);								// Wartezeit 14s (gesamt 15s nach Slotbeginn)
		if ((w1PIN & (1 << W1TXD)) == 1)				// Zustand des Busses abfragen
			ready = 1;									// 1-Signal gelesen -> Wandlung fertig
		_delay_us(45.0);								// Wartezeit 45s		
	} while (!(ready));	
	
	wire_master_reset();
	wire_send_byte(MATCH_ROM);
	// Seriennummer ausgeben
	for (i = 0; i < 8; i++) {
		wire_send_byte(*sicher);
		sicher++;
	}
	wire_send_byte(READ);								// Scratchpad auslesen
	for(i = 0;i < 8; i++)	{		
		data = wire_read_byte();
		if (i==0)
			temperature = data; // LSB
		if (i==1)
			temperature |= data << 8; // MSB
	}
	sei();	
	return (temperature * 5);
}
