

#include "hardware_t.h"
#include "tc_vars.h"
#include "flash_t.h"
#include "display_t.h"
#include "userif_t.h"
#include "commif_t.h"
#include "signal_t.h"

/***************************
  contents: implementation of flash routines
            for the AMD part.
*/

//#####################################################################################################################################################
// This routine actually takes about 3 us on a 33.333MHz Nios32.
// BF corrected value for actual nasys_clock_freq_1000 (12.5MHz)
void AMDFlash::wait_at_least_1_us(void)
{
  	volatile unsigned long iTimeout = nasys_clock_freq_1000 / 165;  //825; //8250
  	while (iTimeout--)
  	{
    	;
  	}
}
//#####################################################################################################################################################
// Unlock bypass mode, enabling fast writes.
void AMDFlash::unlock_bypass_mode(volatile unsigned char *flash_base)
{	// 065d
	flash_base[0xAAA] = 0xAA; // unlock bypass command - cycle 1
	flash_base[0x555] = 0x55; // unlock bypass command - cycle 2
	flash_base[0xAAA] = 0x20; // unlock bypass command - cycle 3
}
//#####################################################################################################################################################
// Turn bypass mode off, disabling fast writes and enabling normal function.
void AMDFlash::reset_bypass_mode(volatile unsigned char *flash_base)
{
  	*flash_base = 0x90; // exit unlock bypass reset command - cycle 1
  	*flash_base = 0x0;  // exit unlock bypass reset command - cycle 2
}
//#####################################################################################################################################################
// Read the given address until two successive reads yield the
// same value.
// Return value: 0 on success, -1 on timeout failure.
int AMDFlash::await_write_complete(volatile unsigned char *addr, const unsigned char correct_value)
{
        //
        // TPA 2/14/2003: The *Maximum* programming time is 150us.
        //                Waiting several times the maximum time
        //                seems like a much better idea than giving-up
        //                well before the published spec says we
        //                should.
        //
	unsigned long iTimeout = 1500; //1200			//BF #010
	int return_value = -1; // Pessimistic return value.

	while (iTimeout)
	{
		wait_at_least_1_us();
		//nr_delay(1);

    	// While the flash is working on program data, read accesses return
    	// "status" instead of data.  Status bit 7 is the complement of the
    	// data being written.  When the program operation is complete and
    	// successful, the written data is returned.  So, read the written
    	// address until it equals the data written.

		if (*addr == correct_value) break;
		iTimeout--;
	}

	if (iTimeout) return_value = 0;		//BF return with success
	
	return return_value;
}
//#####################################################################################################################################################
// Write val to the given flash address, in bypass mode (assumes
// that bypass mode has been enabled already).
// Return value: 0 on success, -1 on failure.
int AMDFlash::nr_flash_write_bypass(volatile unsigned char *flash_base, volatile unsigned char *addr, unsigned char val)
{
//BF del	unsigned char us1, us2;
	int lTimeout = 0;
	int result = -1;

	nm_dcache_invalidate_line(addr);
	nm_icache_invalidate_line(addr);
		
	while(result == -1)
	{
		*flash_base = 0xA0;   // unlock bypass program command - 1st cycle
		*addr = val;          // program address and data    - 2nd cycle
	
		result = await_write_complete(addr,val);
	
		lTimeout++;
		
		if (lTimeout == 1000){ result = -1; return result;}	//BF #010 leave with an error
	}

//BF del #010	us1 = *addr;

//BF del #010	if (us1 != val) result = -1;

	return result;							//BF #010 leave with success
}
//#####################################################################################################################################################
//BF #010 redesigned
// Erase the flash sector at sector_address.
// Return value: 0 on success, -1 on failure.

int AMDFlash::nr_flash_erase_sector(unsigned long *flash_base, unsigned long *sector_address)
{
	volatile unsigned char *fb  = (unsigned char *) flash_base;
	volatile unsigned char *sa  = (unsigned char *) sector_address;
	int result=-1;
	
//	printf("AMDFlash::nr_flash_erase_sector\r\n");	

//	Hardware::Disable_All_Interrupts();			//BF Disable the Interrupts is recommended by data sheet to avoid
								// errors in the command sequence -> AM29LV065d
								// -> causes problems with USTB-Mode -> timer 2

	*fb = 0xAA; 	// 1st cycle - unlock
	*fb = 0x55; 	// 2nd cycle - unlock
	*fb = 0x80; 	// 3rd cycle - set-up
	*fb = 0xAA; 	// 4th cycle - unlock
	*fb = 0x55; 	// 5th cycle - unlock

	*sa = 0x30; 	// 6th cycle - sector erase command
	
//	Hardware::Setup_Interrupts();				// setup interrupts to normal working mode needs about 10ns

//	nr_delay(140);						// wait at least 10ns + 140ns = 150ns -> this should be enough in most cases...
	
	//...but if not - wait here once more
	for (int i=1;i<1500 && *sa != 0xFF && sa[0xFFFE] != 0xFF;i++)		// check if memory contains 0xFF at end and beginning of sector
	{									// datasheet declares 0.9s typical for deleting one sector
		nr_delay(10);							// absolute maximum rating is 15s under worst case conditions
		//printf("\relapsed time since starting erasing sequence: %d ms",150+(i*10));
		//printf("\relapsed time since starting erasing sequence: %d ms",i*10);
	}
	//printf("\r\n");

	//check if middle of sector contains 0xFF
	if( sa[0x7777] == 0xFF)
	{result = 0;}
	else
	{printf("Erase check error - check value: %4x\r\n",sa[10]); result = -1;}

	if(result == -1){printf("Error while erasing flash sector\r\n");}


  	return result;
}
//#####################################################################################################################################################
// Write a buffer of data to the flash, using bypass mode.
// Return value: 0 on success, -1 on failure.
// Note: the integer "size" is given as a number of half-words to
// write.   How convenient.  We write this 8-bit-wide flash one byte
// at a time (of course).
int AMDFlash::nr_flash_write_buffer(unsigned long *flash_base, unsigned long *start_address, unsigned long *buffer, int size)
{
	volatile unsigned char *fb  = (unsigned char *) flash_base;
	unsigned char *sa  = (unsigned char *) start_address;
	unsigned char *buf = (unsigned char *) buffer;
	long num_bytes = size * 4;
	long i;
	int result = -1;
//printf("AMDFlash::nr_flash_write_buffer\n");
#ifdef nasys_main_flash
	if (-1 == (int)fb) fb = (unsigned char *) nasys_main_flash;
#endif // nasys_main_flash

	unlock_bypass_mode(fb);
	for (i = 0; i < num_bytes; ++i)
	{
		result = nr_flash_write_bypass(fb, sa + i, buf[i]);
		if(result == -1) break;					//BF #010 leave at failure
	}
	reset_bypass_mode(fb);

	if(result == -1){ printf("Error while writing buffer to flash\r\n"); }

	return result;
}
//#####################################################################################################################################################
/* BF not needed at this time
// Write val to the given flash address.
// Return value: 1 on success, 0 on failure.
int AMDFlash::amd29lv065d_flash_write_byte(unsigned char *flash_base, unsigned char *addr, unsigned char val)
{

    volatile unsigned char *fb = (unsigned char *) flash_base;
    volatile unsigned char *a = (unsigned char *) addr;
    unsigned char us1, us2;
    int result = 1;

//printf("AMDFlash::amd29lv065d_flash_write_byte\n");
    nm_dcache_invalidate_line(a);
    nm_icache_invalidate_line(a);

#ifdef nasys_main_flash
    if (-1 == (int)fb) fb = (unsigned char *) nasys_main_flash;
#endif // nasys_main_flash

    while (result)
    {
        fb[0x555] = 0xAA; // 1st cycle  addr = XXX, data = AA
        fb[0x333] = 0x55; // 2nd cycle  addr = XXX, data = 55
        fb[0x555] = 0xA0; // 3rd cycle  addr = XXX, data = A0

        *a = val;     // 4th cycle  addr = PA, data = PD

        result = await_write_complete(a,val);
    }
    //if(result) return result;

    us1 = *a;
    if (us1 != val) result = -1;

    return result;
}
*/
//#####################################################################################################################################################
/* BF not needed at this time
int AMDFlash::nr_flash_write(unsigned long *flash_base, unsigned long *addr, unsigned long val)
{
    unsigned char* fb       = (unsigned char *) flash_base;
    unsigned char* a        = (unsigned char *) addr;
    unsigned char  byte_val = val & 0xff;
    int result;
//printf("AMDFlash::nr_flash_write\n");
    result = amd29lv065d_flash_write_byte (fb, a,   byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 8) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+1, byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 16) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+2, byte_val);
     // Nonzero result means error.
    if (result) return result;

    byte_val = (val >> 24) & 0xff;
    result = amd29lv065d_flash_write_byte (fb, a+3, byte_val);

    return result;
}
*/
//#####################################################################################################################################################
/* BF not needed at this time
void AMDFlash::Clear_Data_FLASH(void)
{
	volatile unsigned long fb  = (unsigned long) nasys_main_flash;
	int result = 0;
//printf("AMDFlash::Clear_Data_FLASH\n");		
	unsigned long  Data_Index = 0x0000007F;//0x00000070;
	unsigned long  i;
	
	unsigned long  sectorOffset[] =  // AM29LV065d sector map
	{
		0x000000, 0x010000, 0x020000, 0x030000,
		0x040000, 0x050000, 0x060000, 0x070000,
		0x080000, 0x090000, 0x0A0000, 0x0B0000,
		0x0C0000, 0x0D0000, 0x0E0000, 0x0F0000,
		0x100000, 0x110000, 0x120000, 0x130000,
		0x140000, 0x150000, 0x160000, 0x170000,
		0x180000, 0x190000, 0x1A0000, 0x1B0000,
		0x1C0000, 0x1D0000, 0x1E0000, 0x1F0000,
		0x200000, 0x210000, 0x220000, 0x230000,
		0x240000, 0x250000, 0x260000, 0x270000,
		0x280000, 0x290000, 0x2A0000, 0x2B0000,
		0x2C0000, 0x2D0000, 0x2E0000, 0x2F0000,
		0x300000, 0x310000, 0x320000, 0x330000,
		0x340000, 0x350000, 0x360000, 0x370000,
		0x380000, 0x390000, 0x3A0000, 0x3B0000,
		0x3C0000, 0x3D0000, 0x3E0000, 0x3F0000,
		0x400000, 0x410000, 0x420000, 0x430000,
		0x440000, 0x450000, 0x460000, 0x470000,
		0x480000, 0x490000, 0x4A0000, 0x4B0000,
		0x4C0000, 0x4D0000, 0x4E0000, 0x4F0000,
		0x500000, 0x510000, 0x520000, 0x530000,
		0x540000, 0x550000, 0x560000, 0x570000,
		0x580000, 0x590000, 0x5A0000, 0x5B0000,
		0x5C0000, 0x5D0000, 0x5E0000, 0x5F0000,
		0x600000, 0x610000, 0x620000, 0x630000,
		0x640000, 0x650000, 0x660000, 0x670000,
		0x680000, 0x690000, 0x6A0000, 0x6B0000,
		0x6C0000, 0x6D0000, 0x6E0000, 0x6F0000,
		0x700000, 0x710000, 0x720000, 0x730000,
		0x740000, 0x750000, 0x760000, 0x770000,
		0x780000, 0x790000, 0x7A0000, 0x7B0000,
		0x7C0000, 0x7D0000, 0x7E0000, 0x7F0000,
	};
	
	for (i = Data_Index; i < sizeof(sectorOffset) / sizeof(*sectorOffset); ++i)
	{
#ifdef _Debug_	
		printf("Deleting Sector : %x\n", sectorOffset[i]);
#endif		
		volatile unsigned long *sector = (unsigned long *) sectorOffset[i];

		result = nr_flash_erase_sector((unsigned long *) nasys_main_flash, (unsigned long *) sector);		// Clear Sectors
		
		if(result) break;
	}
}
*/
//#####################################################################################################################################################
// BF Save/Recall -> save trace and all settings
// Return value: 0 on success, -1 on failure.
int AMDFlash::WriteTraceToFlash(unsigned long *flash_address)
{
	unsigned long cx;
	unsigned long buf[0x1000];
	unsigned long *buffer = (unsigned long *) buf;	
	int result = -1;
	int led_value = 0x0000;	

	//printf("AMDFlash::WriteTraceToFlash Address %x\n", flash_address);




	if(-1 == nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address))	// erase config area 
	{printf("Error while erasing flash sector\r\n");}

	
	//BF save signal 1
	for (cx = 0; cx < 0x1000; cx++)
	{
		buf[cx] = *(S1Ptr + (cx * 4) + 0) + (*(S1Ptr + (cx * 4) + 1) << 8) + (*(S1Ptr + (cx * 4) + 2) << 16) + (*(S1Ptr + (cx * 4) + 3) << 24);
	}
	result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address, buf, 0x1000);			// Flash Buffer	

	//BF save signal 2
	for (cx = 0; cx < 0x1000; cx++)
	{
		buf[cx] = *(S2Ptr + (cx * 4) + 0) + (*(S2Ptr + (cx * 4) + 1) << 8) + (*(S2Ptr + (cx * 4) + 2) << 16) + (*(S2Ptr + (cx * 4) + 3) << 24);
	}
	result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address + 0x1000, buf, 0x1000);	// Flash Buffer	

	if (NumberOfChannels == 4)
	{
		//BF save signal 3
		for (cx = 0; cx < 0x1000; cx++)
		{
			buf[cx] = *(S3Ptr + (cx * 4) + 0) + (*(S3Ptr + (cx * 4) + 1) << 8) + (*(S3Ptr + (cx * 4) + 2) << 16) + (*(S3Ptr + (cx * 4) + 3) << 24);
		}
		result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address + 0x2000, buf, 0x1000);	// Flash Buffer	
	
		//BF save signal 4
		for (cx = 0; cx < 0x1000; cx++)
		{
			buf[cx] = *(S4Ptr + (cx * 4) + 0) + (*(S4Ptr + (cx * 4) + 1) << 8) + (*(S4Ptr + (cx * 4) + 2) << 16) + (*(S4Ptr + (cx * 4) + 3) << 24);
			//BF del buf[cx] = *(SIGNAL4 + (cx * 4) + 0) + *(SIGNAL4 + (cx * 4) + 1) + *(SIGNAL4 + (cx * 4) + 2) + *(SIGNAL4 + (cx * 4) + 3);
		}
//	result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address + 0x3000, buf, 0x1000);	// Flash Buffer	
	}	

	// save the settings at the end of signal 4 memory area (300 * 4 byte = 1200 byte)
	if (LED_ON[0] == 1) led_value = led_value | 0x0001;		// Select Channel 1
	if (LED_ON[1] == 1) led_value = led_value | 0x0002;		// Select Channel 2
	if (LED_ON[2] == 1) led_value = led_value | 0x0004;		// Select Channel 3
	if (LED_ON[3] == 1) led_value = led_value | 0x0008;		// Select Channel 4
	if (LED_ON[4] == 1) led_value = led_value | 0x0010;		// Select Channel Math
	if (LED_ON[5] == 1) led_value = led_value | 0x0020;		// General
	if (LED_ON[6] == 1) led_value = led_value | 0x0040;		// Cursor						
	if (LED_ON[7] == 1) led_value = led_value | 0x0080;		// Quick Measure
	if (LED_ON[8] == 1) led_value = led_value | 0x0100;		// Edge
	if (LED_ON[9] == 1) led_value = led_value | 0x0200;		// Pattern - Not Used
	if (LED_ON[10] == 1) led_value = led_value | 0x0400;		// Pulse Width
	if (LED_ON[11] == 1) led_value = led_value | 0x0800;		// More - Not Used
	if (LED_ON[12] == 1) led_value = led_value | 0x1000;		// Run/Stop Green
	if (LED_ON[13] == 1) led_value = led_value | 0x2000;		// Run/Stop Red
	if (LED_ON[14] == 1) led_value = led_value | 0x4000;		// Single Red					
	if (LED_ON[15] == 1) led_value = led_value | 0x8000;		// Single Green	
		
	buf[0x1000 - 0] = (unsigned long) 0x00;
	buf[0x1000 - 1] = (unsigned long) 0xAACC5523;			// Configuration Start Sign
	buf[0x1000 - 2] = (unsigned long) 0x22334455;
	buf[0x1000 - 3] = (unsigned long) 0x00;		
	buf[0x1000 - 4] = (unsigned long) tc_version;			// Quattro Version
	buf[0x1000 - 5] = (unsigned long) tc_subversion;		// SubVerision
	buf[0x1000 - 6] = (unsigned long) tc_dev_version;		// Revision
	buf[0x1000 - 7] = (unsigned long) tc_dev_subversion;		// Compile Status
	buf[0x1000 - 8] = (unsigned long) 0x00;   	
	
	// Timebase
	buf[0x1000 - 10] = (unsigned long) MenuStatus[MENU_TIMEBASE][0];
	buf[0x1000 - 11] = (unsigned long) MenuStatus[MENU_TIMEBASE][1];
	buf[0x1000 - 12] = (unsigned long) MenuStatus[MENU_TIMEBASE][2];
	buf[0x1000 - 13] = (unsigned long) MenuStatus[MENU_TIMEBASE][3];
	buf[0x1000 - 14] = (unsigned long) MenuStatus[MENU_TIMEBASE][4];
	//buf[0x1000 - 15] = (unsigned long) MenuStatus[MENU_DISPLAY][2];	
	buf[0x1000 - 16] = (unsigned long) XY_Mode;
	buf[0x1000 - 17] = (unsigned long) MainTimebase;	
	buf[0x1000 - 18] = (unsigned long) USTB_Mode;
	buf[0x1000 - 19] = (unsigned long) USTB_Mode_bak;
	buf[0x1000 - 20] = (unsigned long) USTB_Dir;	
	buf[0x1000 - 21] = (unsigned long) USTB_Disp;
	buf[0x1000 - 22] = (unsigned long) DelayedTimebase;
//BF del #003	buf[0x1000 - 23] = (unsigned long) free;
	buf[0x1000 - 24] = (unsigned long) dmode_Window_Offset_Pos;
	buf[0x1000 - 25] = (unsigned long) MenuStatus[MENU_USTB][2];					// USTB buffer size
//	buf[0x1000 - 26] = (unsigned long) Cursor_Delayed_1;
//	buf[0x1000 - 27] = (unsigned long) Cursor_Delayed_2;
//	buf[0x1000 - 28] = (unsigned long) Cursor_Delayed_Size;
	buf[0x1000 - 29] = (unsigned long) MenuStatus[MENU_USTB][0];				     	// ultra slow timebase mode
	buf[0x1000 - 30] = (unsigned long) MenuStatus[MENU_USTB][1];
//    	BF buf[0x1000 - 31] = (unsigned long) ;		-> LED 1					// Trigger LED	
//  	BF buf[0x1000 - 32] = (unsigned long) ;		-> LED 2					// Trigger LED	
    	//buf[0x1000 - 33] = (unsigned long) free;
	buf[0x1000 - 34] = (unsigned long) SwitchesTB;	
	
	// Trigger
	buf[0x1000 - 35] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][0];				// Triggering - Auto
	buf[0x1000 - 36] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][1];				// Triggering - Coupling DC
	//buf[0x1000 - 37] = (unsigned long) free;				// 
	buf[0x1000 - 38] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][4];				// Triggering - External Probe = 1 : 1  ( muß noch als funktion integriert werden )
	buf[0x1000 - 39] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][3];				// Triggering - TV
	buf[0x1000 - 40] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][0];				// Triggering - positive edge
	buf[0x1000 - 41] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][1];				// Triggering - Source = channel 1
	buf[0x1000 - 42] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][2];				// Triggering - no externel triggering
	buf[0x1000 - 43] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][0];				// Triggering - Source Channel 1
	buf[0x1000 - 44] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][1];				// Triggering - Negative Pulse
	buf[0x1000 - 45] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][2];				// Triggering - bigger - smaller - then
	buf[0x1000 - 46] = (unsigned long) Selected_Trigger_Source;					// Selected Trigger Source is Channel 1
//BF del	buf[0x1000 - 47] = (unsigned long) SelectedEdgeExtern;          //BF not used  Trigger Edge by extern triggering
	buf[0x1000 - 48] = (unsigned long) PulseToggle;							// Pulse width button selection
	//buf[0x1000 - 49] = (unsigned long) free;	

	buf[0x1000 - 50] = (unsigned long) FFT_TriggerMode_bak;					     //BF Backup for Triggermode
	buf[0x1000 - 51] = (unsigned long) Trigger_Pos_CH1;                                          // Trigger Level Channel 1
	buf[0x1000 - 52] = (unsigned long) Trigger_Pos_CH2;                                          // Trigger Level Channel 2
	buf[0x1000 - 53] = (unsigned long) Trigger_Pos_CH3;                                          // Trigger Level Channel 3
	buf[0x1000 - 54] = (unsigned long) Trigger_Pos_CH4;                                          // Trigger Level Channel 4
	buf[0x1000 - 55] = (unsigned long) trg_val_reg;                                              // Trigger Value
	buf[0x1000 - 56] = (unsigned long) trg_hyst_reg;                                             // Trigger Hysterese
	buf[0x1000 - 57] = (unsigned long) trig_range_reg;                                           // Trigger Range
	buf[0x1000 - 58] = (unsigned long) trig_holdoff_reg;                                         // Trigger Holdoff
	buf[0x1000 - 59] = (unsigned long) trig_width_reg;                                           // Trigger Width

	buf[0x1000 - 60] = (unsigned long) ctrl_reg;                                                 // Control register
	buf[0x1000 - 61] = (unsigned long) adc_ctrl_reg;                                             // ADC Control register
	buf[0x1000 - 62] = (unsigned long) pre_reg;                                                  // PreTrigger Value
	//buf[0x1000 - 63] = (unsigned long) channel_Adr_add;                                          // Channel Add Value
	//buf[0x1000 - 64] = (unsigned long) channel_Adr_add2;                                           // ADC Change
	buf[0x1000 - 65] = (unsigned long) triggering;                                               // Triggering
	buf[0x1000 - 66] = (unsigned long) TriggerWay;                                               // Triggerway
	buf[0x1000 - 67] = (unsigned long) Trigger_Pos_CHE;                                          // Trigger extern value	
	
	//---------------------------------------------------
	// 		Channel 1
	//---------------------------------------------------
	buf[0x1000 - 68] = (unsigned long) MenuStatus[1][0];                                         // Channel - Coupling
	buf[0x1000 - 69] = (unsigned long) MenuStatus[1][1];                                         // Channel - BW Linit
	buf[0x1000 - 70] = (unsigned long) MenuStatus[1][2];                                         // Channel - Invert
	buf[0x1000 - 71] = (unsigned long) MenuStatus[1][3];                                         // Channel - Probe
	buf[0x1000 - 72] = (unsigned long) SwitchesCH1;                                              // Switches for Channel 1 and for the other the AC bit
	buf[0x1000 - 73] = (unsigned long) Selected_Voltage_CH1;                                     // Voltage 5V
	//BF del buf[0x1000 - 74] = (unsigned long) Trigger_Pos_CH1_Old;
	//BF del buf[0x1000 - 75] = (unsigned long) NoDisplay[0];;
	buf[0x1000 - 76] = (unsigned long) thres_low[0];						// QM thresholds
	buf[0x1000 - 77] = (unsigned long) thres_mdl[0];						// QM thresholds
	buf[0x1000 - 78] = (unsigned long) thres_up[0];							// QM thresholds
	buf[0x1000 - 79] = (unsigned long) thres_type[1];							// QM threshold type

	buf[0x1000 - 80] = (unsigned long) Selected_Voltage_CH1_bak; 
	buf[0x1000 - 81] = (unsigned long) Trigger_Pos_CH1_bak;
	buf[0x1000 - 82] = (unsigned long) CH1_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 83] = (unsigned long) ZeroLevelCH1;                                             // ZeroLevel
	buf[0x1000 - 84] = (unsigned long) Virtual_ZeroLevelCH1;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 85] = (unsigned long) alpha[0];                          			     // ADC-alpha correction -> BF changed
    	//buf[0x1000 - 85] = (unsigned long) CH1_Zero_Offset;                                          // Shift offset when stopped

	//---------------------------------------------------
	// 		Channel 2
	//---------------------------------------------------
	buf[0x1000 - 86] = (unsigned long) MenuStatus[2][0];                                         // Channel - Coupling
	buf[0x1000 - 87] = (unsigned long) MenuStatus[2][1];                                         // Channel - BW Linit
	buf[0x1000 - 88] = (unsigned long) MenuStatus[2][2];                                         // Channel - Invert
	buf[0x1000 - 89] = (unsigned long) MenuStatus[2][3];                                         // Channel - Probe

	buf[0x1000 - 90] = (unsigned long) SwitchesCH2;                                              // Switches for Channel
	buf[0x1000 - 91] = (unsigned long) Selected_Voltage_CH2;                                     // Voltage 5V
	//BF del buf[0x1000 - 92] = (unsigned long) Trigger_Pos_CH2_Old;
	//BF del buf[0x1000 - 93] = (unsigned long) NoDisplay[1];;
	buf[0x1000 - 94] = (unsigned long) thres_low[1];						// QM thresholds
	buf[0x1000 - 95] = (unsigned long) thres_mdl[1];						// QM thresholds
	buf[0x1000 - 96] = (unsigned long) thres_up[1];							// QM thresholds
	buf[0x1000 - 97] = (unsigned long) thres_type[1];							// QM threshold type
	
	buf[0x1000 - 98] = (unsigned long) Selected_Voltage_CH2_bak;
	buf[0x1000 - 99] = (unsigned long) Trigger_Pos_CH2_bak;
	buf[0x1000 - 100] = (unsigned long) CH2_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 101] = (unsigned long) ZeroLevelCH2;                                             // ZeroLevel
	buf[0x1000 - 102] = (unsigned long) Virtual_ZeroLevelCH2;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 103] = (unsigned long) alpha[1];                                                 // ADC-alpha correction -> BF changed
    	//buf[0x1000 - 103] = (unsigned long) CH2_Zero_Offset;                                          // Shift offset when stopped

	//---------------------------------------------------
	// 		Channel 3
	//---------------------------------------------------
	buf[0x1000 - 104] = (unsigned long) MenuStatus[3][0];                                         // Channel - Coupling
	buf[0x1000 - 105] = (unsigned long) MenuStatus[3][1];                                         // Channel - BW Linit
	buf[0x1000 - 106] = (unsigned long) MenuStatus[3][2];                                         // Channel - Invert
	buf[0x1000 - 107] = (unsigned long) MenuStatus[3][3];                                         // Channel - Probe
	buf[0x1000 - 108] = (unsigned long) SwitchesCH3;                                              // Switches for Channel
	buf[0x1000 - 109] = (unsigned long) Selected_Voltage_CH3;                                     // Voltage 5V

	//BF del buf[0x1000 - 110] = (unsigned long) Trigger_Pos_CH3_Old; 
	//BF del buf[0x1000 - 111] = (unsigned long) NoDisplay[2];;
	buf[0x1000 - 112] = (unsigned long) thres_low[2];						// QM thresholds
	buf[0x1000 - 113] = (unsigned long) thres_mdl[2];						// QM thresholds
	buf[0x1000 - 114] = (unsigned long) thres_up[2];						// QM thresholds
	buf[0x1000 - 115] = (unsigned long) thres_type[2];						// QM threshold type

	buf[0x1000 - 116] = (unsigned long) Selected_Voltage_CH3_bak;
	buf[0x1000 - 117] = (unsigned long) Trigger_Pos_CH3_bak;
	buf[0x1000 - 118] = (unsigned long) CH3_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 119] = (unsigned long) ZeroLevelCH3;                                             // ZeroLevel
	buf[0x1000 - 120] = (unsigned long) Virtual_ZeroLevelCH3;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 121] = (unsigned long) free;
    	//buf[0x1000 - 121] = (unsigned long) CH3_Zero_Offset;                                          // Shift offset when stopped

	//---------------------------------------------------
	// 		Channel 4
	//---------------------------------------------------
	buf[0x1000 - 122] = (unsigned long) MenuStatus[4][0];                                         // Channel - Coupling
	buf[0x1000 - 123] = (unsigned long) MenuStatus[4][1];                                         // Channel - BW Linit

	buf[0x1000 - 124] = (unsigned long) MenuStatus[4][2];                                         // Channel - Invert
	buf[0x1000 - 125] = (unsigned long) MenuStatus[4][3];                                         // Channel - Probe
	buf[0x1000 - 126] = (unsigned long) SwitchesCH4;                                              // Switches for Channel
	buf[0x1000 - 127] = (unsigned long) Selected_Voltage_CH4;                                     // Voltage 5V
	//BF del buf[0x1000 - 128] = (unsigned long) Trigger_Pos_CH4_Old;
	//BF del buf[0x1000 - 129] = (unsigned long) NoDisplay[3];;
	buf[0x1000 - 130] = (unsigned long) thres_low[3];						// QM thresholds
	buf[0x1000 - 131] = (unsigned long) thres_mdl[3];						// QM thresholds
	buf[0x1000 - 132] = (unsigned long) thres_up[3];						// QM thresholds
	buf[0x1000 - 133] = (unsigned long) thres_type[3];						// QM threshold type

	buf[0x1000 - 134] = (unsigned long) Selected_Voltage_CH4_bak;
	buf[0x1000 - 135] = (unsigned long) Trigger_Pos_CH4_bak;
	buf[0x1000 - 136] = (unsigned long) CH4_DAC_Offset;                                           // Channel DAC
	buf[0x1000 - 137] = (unsigned long) ZeroLevelCH4;                                             // ZeroLevel
	buf[0x1000 - 138] = (unsigned long) Virtual_ZeroLevelCH4;                                     // Virtual ZeroLevel
    	//buf[0x1000 - 139] = (unsigned long) free; 
    	//buf[0x1000 - 139] = (unsigned long) CH4_Zero_Offset;                                          // Shift offset when stopped                   	

	//Math Settings
	buf[0x1000 - 140] = (unsigned long) Channel_Math_Active;                                      // Math channel off
	buf[0x1000 - 141] = (unsigned long) MenuStatus[MENU_MATH][0];                                        // scale
	buf[0x1000 - 142] = (unsigned long) MenuStatus[MENU_MATH][2];                                        // standart math 1*2 of
	buf[0x1000 - 143] = (unsigned long) MenuStatus[MENU_MATH][3];                                        // standart math 1-2 off
	buf[0x1000 - 144] = (unsigned long) MenuStatus[MENU_MATH][4];                                        // standart math 1+2 off
	
	//FFT Settings
	buf[0x1000 - 145] = (unsigned long) FFT_Mode;
	buf[0x1000 - 146] = (unsigned long) FFT_Length;
	buf[0x1000 - 147] = (unsigned long) FFT_Level;
	buf[0x1000 - 148] = (unsigned long) FFT_Windex;
//BF not used	buf[0x1000 - 149] = (unsigned long) FFT_Scale;
	
	buf[0x1000 - 150] = (unsigned long) MenuStatus[MENU_FFT][0]; 
	buf[0x1000 - 151] = (unsigned long) MenuStatus[MENU_FFT][1];
	buf[0x1000 - 152] = (unsigned long) MenuStatus[MENU_FFT][2];
	buf[0x1000 - 153] = (unsigned long) MenuStatus[MENU_FFT][3];

    	// Cursors    	
	buf[0x1000 - 154] = (unsigned long) MenuStatus[16][0];                                        // Cursor source
	buf[0x1000 - 155] = (unsigned long) Cursor_Enabled;
	buf[0x1000 - 156] = (unsigned long) Cursor_Both_Selected;
	buf[0x1000 - 157] = (unsigned long) SelectedCursor;

	buf[0x1000 - 158] = (unsigned long) Cursor_Horizontal_Active;
	buf[0x1000 - 159] = (unsigned long) Cursor_Horizontal_Position_Real1;
	buf[0x1000 - 160] = (unsigned long) Cursor_Horizontal_Position_Real2;

	buf[0x1000 - 161] = (unsigned long) Cursor_Vertical_Active;
	buf[0x1000 - 162] = (unsigned long) Cursor_Vertical_Position_TY_1;
	buf[0x1000 - 163] = (unsigned long) Cursor_Vertical_Position_TY_2;
	
	// Quick Measure
	//buf[0x1000 - 164] = (unsigned long) QM_Enabled;
	buf[0x1000 - 165] = (unsigned long) QM_Type[0];
	buf[0x1000 - 166] = (unsigned long) QM_Type[1];
	buf[0x1000 - 167] = (unsigned long) QM_Type[2];
	buf[0x1000 - 168] = (unsigned long) QM_Channel[0];
	buf[0x1000 - 169] = (unsigned long) QM_Channel[1];
	
	buf[0x1000 - 170] = (unsigned long) QM_Channel[2];
	buf[0x1000 - 171] = (unsigned long) QM_Horizontal_Active;
	buf[0x1000 - 172] = (unsigned long) QM_Vertical_Active;
	buf[0x1000 - 173] = (unsigned long) Quick_Measure_Threshold_btn_select;
	
	// Display
	//buf[0x1000 - 174] = (unsigned long) GridColorArrayPos;                                // Normal Grid
	//buf[0x1000 - 175] = (unsigned long) GridColor_Val;
	//buf[0x1000 - 176] = (unsigned long) MenuStatus[MENU_DISPLAY][3];			// no display
	//buf[0x1000 - 177] = (unsigned long) dotted_grid;
	buf[0x1000 - 178] = (unsigned long) MenuStatus[MENU_DISPLAY][0];  			// Display - Persist
	//buf[0x1000 - 179] = (unsigned long) MenuStatus[MENU_DISPLAY][4];  			// Display - Draw Mode

	buf[0x1000 - 180] = (unsigned long) MenuStatus[MENU_ACQUIRE][0];          		// Acquire - Logic
        buf[0x1000 - 181] = (unsigned long) MenuStatus[MENU_ACQUIRE][1];        		// Acquire - Averaging
        buf[0x1000 - 183] = (unsigned long) MenuStatus[MENU_ACQUIRE][2];               		// Acquire - Filter Setting
	buf[0x1000 - 184] = (unsigned long) MenuStatus[MENU_ACQUIRE][3];        		// Acquire - Interpolation
	
	//buf[0x1000 - 185] = (unsigned long) MenuStatus[MENU_AUTOSCALE][1];          		// Autoscale - Slow TB

//	buf[0x1000 - 186] = (unsigned long) MenuStatus[MENU_HARDWARE][1];			//BF gain of input stage

	buf[0x1000 - 187] = (unsigned long) MenuPopupStatus[8][0];				//BF extern trigger setting
	buf[0x1000 - 188] = (unsigned long) MenuPopupStatus[8][1];
	buf[0x1000 - 189] = (unsigned long) MenuPopupStatus[8][2];
	buf[0x1000 - 190] = (unsigned long) MenuPopupStatus[11][0];				//BF TV trigger setting
	buf[0x1000 - 191] = (unsigned long) MenuPopupStatus[11][1];
	
	buf[0x1000 - 192] = (unsigned long) led_value;
	buf[0x1000 - 193] = (unsigned long) Channel_1_Active;
	buf[0x1000 - 194] = (unsigned long) Channel_2_Active;
	buf[0x1000 - 195] = (unsigned long) Channel_3_Active;
	buf[0x1000 - 196] = (unsigned long) Channel_4_Active;

	buf[0x1000 - 197] = (unsigned long) FFT_grid;	

	//buf[0x1000 - 198] = (unsigned long) free;
	
	buf[0x1000 - 199] = (unsigned long) HoldOff_Value;
	buf[0x1000 - 200] = (unsigned long) HoldOff_Expo;
	buf[0x1000 - 201] = (unsigned long) HoldOff_Steps;
	
	buf[0x1000 - 202] = (unsigned long) Pulse11_Value;
	buf[0x1000 - 203] = (unsigned long) Pulse11_Expo;
	buf[0x1000 - 204] = (unsigned long) Pulse11_Steps;
	
	buf[0x1000 - 205] = (unsigned long) Pulse12_Value;
	buf[0x1000 - 206] = (unsigned long) Pulse12_Expo;
	buf[0x1000 - 207] = (unsigned long) Pulse12_Steps;
	
	buf[0x1000 - 208] = (unsigned long) Pulse21_Value;
	buf[0x1000 - 209] = (unsigned long) Pulse21_Expo;
	buf[0x1000 - 210] = (unsigned long) Pulse21_Steps;
	
	buf[0x1000 - 211] = (unsigned long) Pulse22_Value;
	buf[0x1000 - 212] = (unsigned long) Pulse22_Expo;
	buf[0x1000 - 213] = (unsigned long) Pulse22_Steps;
	
//BF new Math values
	buf[0x1000 - 214] = (unsigned int) Math_Offset;
	buf[0x1000 - 215] = (unsigned int) Math_Mul_Offset;
	buf[0x1000 - 216] = (unsigned int) Math_Sub_Offset;
	buf[0x1000 - 217] = (unsigned int) Math_Add_Offset;
	buf[0x1000 - 218] = (unsigned int) math_mul_scale;
	buf[0x1000 - 219] = (unsigned int) math_mul_scale;
	buf[0x1000 - 220] = (unsigned int) math_sub_scale;
	buf[0x1000 - 221] = (unsigned int) math_add_scale;
//BF end

	buf[0x1000 - 222] = (unsigned long) MenuStatus[MENU_EXTUTILS][0];	//BF #001
	buf[0x1000 - 223] = (unsigned long) MenuStatus[MENU_EXTUTILS][1];	//BF #001
	buf[0x1000 - 224] = (unsigned long) MenuStatus[MENU_EXTUTILS][2];	//BF #001
	buf[0x1000 - 225] = (unsigned long) MenuStatus[MENU_EXTUTILS][3];	//BF #001
//	buf[0x1000 - 226] = (unsigned long) MenuStatus[MENU_HARDWARE][0];	//BF #002


	buf[0x1000 - 229] = (unsigned long) Trig_Pos_Mem;
	buf[0x1000 - 230] = (unsigned long) Trig_Pos_Display;
	//buf[0x1000 - 231] = (unsigned long) not used;
	buf[0x1000 - 232] = (unsigned long) MemWinStart;
	
	buf[0x1000 - 233] = (unsigned long) Trig_Pos_Mem_old;
	buf[0x1000 - 234] = (unsigned long) Trig_Pos_Display_old;
	//buf[0x1000 - 235] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][4];
	buf[0x1000 - 236] = (unsigned long) MemWinStart_old;
	
	buf[0x1000 - 237] = (unsigned long) ZoomFactor;
	buf[0x1000 - 238] = (unsigned long) ZoomFactor_old;
	buf[0x1000 - 239] = (unsigned long) ZoomFactorDel;

	//buf[0x1000 - 240] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][0];
	//buf[0x1000 - 241] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][1];
	//buf[0x1000 - 242] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][2];

	buf[0x1000 - 243] = (unsigned long) CH1_Del_Correct;
	buf[0x1000 - 244] = (unsigned long) CH2_Del_Correct;
	buf[0x1000 - 245] = (unsigned long) CH3_Del_Correct;
	buf[0x1000 - 246] = (unsigned long) CH4_Del_Correct;
	
	//buf[0x1000 - 247] = (unsigned long)   ;

// BF new zerolevels for XY
	buf[0x1000 - 248] = (unsigned int) Virtual_ZeroLevelXYCH1;
	buf[0x1000 - 249] = (unsigned int) Virtual_ZeroLevelXYCH2;
	buf[0x1000 - 250] = (unsigned int) Virtual_ZeroLevelXYCH3;
	buf[0x1000 - 251] = (unsigned int) Virtual_ZeroLevelXYCH4;
// BF end


	//BF backup values must be saved too
	buf[0x1000 - 252] = (unsigned long) ZeroLevelCH1_bak;
	buf[0x1000 - 253] = (unsigned long) ZeroLevelCH2_bak;
	buf[0x1000 - 254] = (unsigned long) ZeroLevelCH3_bak;
	buf[0x1000 - 255] = (unsigned long) ZeroLevelCH4_bak;

	buf[0x1000 - 256] = (unsigned long) Virtual_ZeroLevelCH1_bak;
	buf[0x1000 - 257] = (unsigned long) Virtual_ZeroLevelCH2_bak;
	buf[0x1000 - 258] = (unsigned long) Virtual_ZeroLevelCH3_bak;
	buf[0x1000 - 259] = (unsigned long) Virtual_ZeroLevelCH4_bak;

	buf[0x1000 - 260] = (unsigned long) TriggerMode_bak;

	buf[0x1000 - 261] = (unsigned long) Cursor_Vertical_Position_XY_1;
	buf[0x1000 - 262] = (unsigned long) Cursor_Vertical_Position_XY_2;
	buf[0x1000 - 263] = (unsigned long) Cursor_Vertical_Position_FFT_1;
	buf[0x1000 - 264] = (unsigned long) Cursor_Vertical_Position_FFT_2;
	//buf[0x1000 - 265] = (unsigned long) SavePointer;

	buf[0x1000 - 266] = (unsigned long) Channel_1_Active_bak;
	buf[0x1000 - 267] = (unsigned long) Channel_2_Active_bak;
	buf[0x1000 - 268] = (unsigned long) Channel_3_Active_bak;
	buf[0x1000 - 269] = (unsigned long) Channel_4_Active_bak;
	buf[0x1000 - 270] = (unsigned long) Channel_Math_Active_bak;
	
	//buf[0x1000 - 271] = (unsigned long) DrawMono;
	buf[0x1000 - 272] = (unsigned long) MainTimebase_bak;
	//buf[0x1000 - 273] = (unsigned long) free;
	//buf[0x1000 - 274] = (unsigned long) free;

	buf[0x1000 - 275] = (unsigned long) ZeroLevelCH1_Main_bak;
	buf[0x1000 - 276] = (unsigned long) ZeroLevelCH2_Main_bak;
	buf[0x1000 - 277] = (unsigned long) ZeroLevelCH3_Main_bak;
	buf[0x1000 - 278] = (unsigned long) ZeroLevelCH4_Main_bak;

	buf[0x1000 - 279] = (unsigned long) Virtual_ZeroLevelCH1_Main_bak;
	buf[0x1000 - 280] = (unsigned long) Virtual_ZeroLevelCH2_Main_bak;
	buf[0x1000 - 281] = (unsigned long) Virtual_ZeroLevelCH3_Main_bak;
	buf[0x1000 - 282] = (unsigned long) Virtual_ZeroLevelCH4_Main_bak;


	// BF store register values	#001
	buf[0x1000 - 283] = (unsigned long) channel_Adr_add12;
	buf[0x1000 - 284] = (unsigned long) channel_Adr_add34;
	buf[0x1000 - 285] = (unsigned long) adc_change12_reg;
	buf[0x1000 - 286] = (unsigned long) adc_change34_reg;
	buf[0x1000 - 287] = (unsigned long) adc_ctr12_reg;
	buf[0x1000 - 288] = (unsigned long) adc_ctr34_reg;

	//buf[0x1000 - 289] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][0];			// grid color
	//buf[0x1000 - 290] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][1];			// grid layout
	//buf[0x1000 - 291] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][2];			// status layout
	//buf[0x1000 - 292] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][3];			// not used
	//buf[0x1000 - 293] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][4];			// not used
	//buf[0x1000 - 294] = (unsigned long)MenuStatus[MENU_DISPL_SETUP][5];			// not used

	//buf[0x1000 - 295] = (unsigned long) thres_low[4];						// QM thresholds
	//buf[0x1000 - 296] = (unsigned long) thres_mdl[4];						// QM thresholds
	//buf[0x1000 - 297] = (unsigned long) thres_up[4];						// QM thresholds
	//buf[298] = (unsigned long) thres_type[4];					// QM threshold type

	buf[0x1000 - 299] = 0x23232323;	
	
	//write signal 4 and the settings to flash
	result = nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address + 0x3000, buf, 0x1000);	// Flash Buffer	

	return result;
}
//#####################################################################################################################################################
//BF Save/Recall -> load configuration of corresponding trace into memory
// Return value: 0 on success, -1 on failure.
int AMDFlash::LoadTraceSettings(unsigned long *flash_address)
{
	unsigned long lX;
	unsigned long buf[300];	
	int rcode = -1;

	for (lX = 0; lX < 300; lX++) buf[lX] = flash_address[0x4000 - lX];		
	
	if ((buf[4] == tc_version) && (buf[5] == tc_subversion) && (buf[6] == tc_dev_version) && (buf[1] == 0xAACC5523) && (buf[2] == 0x22334455) && (buf[299] == 0x23232323))
	{

		Hardware::Stop_Timer2();
		Hardware::Stop_Record();
		ADC_Data_Available = 0;	
		FFT_NewData = 0;
		ACQ_NewData = 0;


		// Timebase      	
		MenuStatus[MENU_TIMEBASE][0] = (short) buf[10];
		MenuStatus[MENU_TIMEBASE][1] = (short) buf[11];
		MenuStatus[MENU_TIMEBASE][2] = (short) buf[12];
		MenuStatus[MENU_TIMEBASE][3] = (short) buf[13];
		MenuStatus[MENU_TIMEBASE][4] = (short) buf[14];
		//MenuStatus[MENU_DISPLAY][2] = (short) buf[15];	//grid brightness
		XY_Mode = (unsigned char) buf[16];
		MainTimebase = (char) buf[17];
		USTB_Mode = (short) buf[18];
		USTB_Mode_bak = (short) buf[19];
		USTB_Dir = (short) buf[20];   
		USTB_Disp = (short) buf[21];
		DelayedTimebase = (char) buf[22];
//BF del #003			dmode_SIGNALFaktor_idx = (volatile short) buf[23];
		dmode_Window_Offset_Pos = (int) buf[24];
		MenuStatus[MENU_USTB][2] = (short) buf[25];
	
		// werden frisch berrechnet
		//Cursor_Delayed_1 = (int) buf[26];
		//Cursor_Delayed_2 = (int) buf[27];
		//Cursor_Delayed_Size = (int) buf[28];
		MenuStatus[MENU_USTB][0] = (short) buf[29];					//ultra slow timebase mode
		MenuStatus[MENU_USTB][1] = (short) buf[30];
	//  		 = (volatile char) buf[31];		-> LED 1			// Trigger LED	
	//      	= (volatile char) buf[32];		-> LED 2			// Trigger LED	
		//free = (int) buf[33];
		SwitchesTB = (unsigned short) buf[34];
		
		// Trigger	
		MenuStatus[MENU_TRIGGERMODE][0] = (short) buf[35];				// Triggering - Auto
		MenuStatus[MENU_TRIGGERMODE][1] = (short) buf[36];				// Triggering - Coupling
		// free = (short) buf[37];				// 
		MenuStatus[MENU_TRIGGERMODE][4] = (short) buf[38];				// Triggering - External Probe
		MenuStatus[MENU_TRIGGEREDGE][3] = (short) buf[39];				// Triggering - TV
		MenuStatus[MENU_TRIGGEREDGE][0] = (short) buf[40];				// Triggering - positive edge
		MenuStatus[MENU_TRIGGEREDGE][1] = (short) buf[41];				// Triggering - Source 
		MenuStatus[MENU_TRIGGEREDGE][2] = (short) buf[42];				// Triggering - no externel triggering
		MenuStatus[MENU_PULSEWIDTH][0] = (short) buf[43];				// Triggering - Source 
		//MenuStatus[MENU_PULSEWIDTH][1] = (short) buf[44];				// Triggering - Negative Pulse
		MenuStatus[MENU_PULSEWIDTH][2] = (short) buf[45];				// Triggering - bigger - smaller - then
		Selected_Trigger_Source = (char) buf[46];					// Selected Trigger Source is Channel 1
//BF del		SelectedEdgeExtern = (char) buf[47]; 	//BF not used  Trigger Edge by extern triggering
		PulseToggle = (char) buf[48];							// Pulse width button selection                                       //
		//free = (long) buf[49];						
		
		FFT_TriggerMode_bak = (short) buf[50];     					//BF backup trigger mode
		Trigger_Pos_CH1  = (int) buf[51];                                          // Trigger Level Channel 1
		Trigger_Pos_CH2  = (int) buf[52];                                          // Trigger Level Channel 2
		Trigger_Pos_CH3  = (int) buf[53];                                          // Trigger Level Channel 3
		Trigger_Pos_CH4  = (int) buf[54];                                          // Trigger Level Channel 4
		trg_val_reg      = (unsigned int) buf[55];                                              // Trigger Value
		trg_hyst_reg     = (unsigned int) buf[56];                                             // Trigger Hysterese
		trig_range_reg   = (unsigned int) buf[57];                                           // Trigger Range
		trig_holdoff_reg = (unsigned long) buf[58];                                         // Trigger Holdoff
		trig_width_reg   = (unsigned int) buf[59];                                           // Trigger Width
		
		ctrl_reg = (unsigned int) buf[60];                                                 // Control register
		adc_ctrl_reg = (unsigned int) buf[61];                                             // ADC Control register
		pre_reg = (unsigned int) buf[62];                                                  // PreTrigger Value
//        channel_Adr_add = (unsigned int) buf[63];                                          // Channel Add Value
//        adc_change_reg = (unsigned long) buf[64];                                           // ADC Change
		triggering = (unsigned int) buf[65];                                               // Triggering
		TriggerWay = (char) buf[66];                                               // Triggerway
		Trigger_Pos_CHE = (int) buf[67];                                                // Trigger extern value
		
		//---------------------------------------------------
		// 		Channel 1
		//---------------------------------------------------
		MenuStatus[1][0] = (short) buf[68];                                         // Channel - Coupling
		MenuStatus[1][1] = (short) buf[69];                                         // Channel - BW Linit
		MenuStatus[1][2] = (short) buf[70];                                         // Channel - Invert
		MenuStatus[1][3] = (short) buf[71];                                         // Channel - Probe
		SwitchesCH1 = (unsigned short) buf[72];                                              // Switches for Channel 1 and for the other the AC bit
		Selected_Voltage_CH1 = (volatile char) buf[73];                                     // Voltage
		//Trigger_Pos_CH1_Old = (int) buf[74];
		//NoDisplay[0] = (unsigned char) buf[75];
		thres_low[0] = (short) buf[76];						// QM thresholds;
		thres_mdl[0] = (short) buf[77];						// QM thresholds;
		thres_up[0]  = (short) buf[78];						// QM thresholds;
		thres_type[0] = (short) buf[79];					// QM threshold type
		
		Selected_Voltage_CH1_bak = (char) buf[80];
		Trigger_Pos_CH1_bak = (int) buf[81];
		CH1_DAC_Offset       = (int) buf[82];                                           // Channel DAC
		ZeroLevelCH1         = (int) buf[83];                                             // ZeroLevel
		Virtual_ZeroLevelCH1 = (int) buf[84];                                     // Virtual ZeroLevel
		//free             = (int) buf[85];
		//CH1_Zero_Offset    = (int) buf[85];                                          // Shift offset when stopped

		//---------------------------------------------------
		// 		Channel 2
		//---------------------------------------------------
		MenuStatus[2][0] = (short) buf[86];                                         // Channel - Coupling
		MenuStatus[2][1] = (short) buf[87];                                         // Channel - BW Linit
		MenuStatus[2][2] = (short) buf[88];                                         // Channel - Invert
		MenuStatus[2][3] = (short) buf[89];                                         // Channel - Probe
	
		SwitchesCH2 = (unsigned short) buf[90];                                              // Switches for Channel 2
		Selected_Voltage_CH2 = (volatile char) buf[91];                                     // Voltage
		//Trigger_Pos_CH2_Old = (int) buf[92];
		//NoDisplay[1] = (unsigned char) buf[93];
		thres_low[1] = (short) buf[94];						// QM thresholds;
		thres_mdl[1] = (short) buf[95];						// QM thresholds;
		thres_up[1]  = (short) buf[96];						// QM thresholds;
		thres_type[1] = (short) buf[97];					// QM threshold type

		Selected_Voltage_CH2_bak = (char) buf[98];
		Trigger_Pos_CH2_bak = (int) buf[99];

		CH2_DAC_Offset       = (int) buf[100];                                           // Channel DAC
		ZeroLevelCH2         = (int) buf[101];                                             // ZeroLevel
		Virtual_ZeroLevelCH2 = (int) buf[102];                                     // Virtual ZeroLevel
		//alpha[1]             = (int) buf[103];                                          // ADC-alpha correction -> BF changed
		//CH2_Zero_Offset    = (int) buf[103];                                          // Shift offset when stopped
	
		//---------------------------------------------------
		// 		Channel 3
		//---------------------------------------------------
		MenuStatus[3][0] = (short) buf[104];                                         // Channel - Coupling
		MenuStatus[3][1] = (short) buf[105];                                         // Channel - BW Linit
		MenuStatus[3][2] = (short) buf[106];                                         // Channel - Invert
		MenuStatus[3][3] = (short) buf[107];                                         // Channel - Probe
		SwitchesCH3 = (unsigned short) buf[108];                                              // Switches for Channel 3
		Selected_Voltage_CH3 = (volatile char) buf[109];                                     // Voltage
		//Trigger_Pos_CH3_Old = (int) buf[110];
		//NoDisplay[2] = (unsigned char) buf[111];
		thres_low[2] = (short) buf[112];						// QM thresholds;
		thres_mdl[2] = (short) buf[113];						// QM thresholds;
		thres_up[2]  = (short) buf[114];						// QM thresholds;
		thres_type[2] = (short) buf[115];						// QM threshold type
	
		Selected_Voltage_CH3_bak = (char) buf[116];
		Trigger_Pos_CH1_bak = (int) buf[117];
		CH3_DAC_Offset = (int) buf[118];                                           // Channel DAC
		ZeroLevelCH3 = (int) buf[119];                                             // ZeroLevel
		Virtual_ZeroLevelCH3 = (int) buf[120];                                     // Virtual ZeroLevel
		//alpha[2]             = (int) buf[121];                                          // ADC-alpha correction -> BF changed
		//CH3_Zero_Offset = (int) buf[121];                                          // Shift offset when stopped
	
		//---------------------------------------------------
		// 		Channel 4
		//---------------------------------------------------
		MenuStatus[4][0] = (short) buf[122];                                         // Channel - Coupling
		MenuStatus[4][1] = (short) buf[123];                                         // Channel - BW Linit
		MenuStatus[4][2] = (short) buf[124];                                         // Channel - Invert
		MenuStatus[4][3] = (short) buf[125];                                         // Channel - Probe
		SwitchesCH4 = (unsigned short) buf[126];                                              // Switches for Channel 4
		Selected_Voltage_CH4 = (volatile char) buf[127];                                     // Voltage
		//Trigger_Pos_CH4_Old = (int) buf[128];                       
		//NoDisplay[3] = (unsigned char) buf[129];                        
		thres_low[3] = (short) buf[130];						// QM thresholds;
		thres_mdl[3] = (short) buf[131];						// QM thresholds;
		thres_up[3]  = (short) buf[132];						// QM thresholds;
		thres_type[3] = (short) buf[133];						// QM threshold type

		Selected_Voltage_CH4_bak = (char) buf[134];
		Trigger_Pos_CH1_bak = (int) buf[135];
		CH4_DAC_Offset       = (int) buf[136];                                           // Channel DAC
		ZeroLevelCH4         = (int) buf[137];                                             // ZeroLevel
		Virtual_ZeroLevelCH4 = (int) buf[138];                                     // Virtual ZeroLevel
		//alpha[3]             = (int) buf[139];                                          // ADC-alpha correction -> BF changed
		//CH4_Zero_Offset = (int) buf[139];                                          	// Shift offset when stopped
		
        	Channel_Math_Active = (char) buf[140];                                      	// Math channel
        	MenuStatus[MENU_MATH][0] = (short) buf[141];                                        // scale
        	MenuStatus[MENU_MATH][2] = (short) buf[142];                                        // standart math 1*2 
        	MenuStatus[MENU_MATH][3] = (short) buf[143];                                        // standart math 1-2 
        	MenuStatus[MENU_MATH][4] = (short) buf[144];                                        // standart math 1+2 

    		//FFT Settings
     		FFT_Mode   = (int)buf[145];
    		FFT_Length = (int)buf[146];
    		FFT_Level  = (int)buf[147];
   		FFT_Windex = (int)buf[148];
    		//FFT_Scale  = (int)buf[149];

		MenuStatus[MENU_FFT][0] = (short)buf[150]; 
		MenuStatus[MENU_FFT][1] = (short)buf[151];
		MenuStatus[MENU_FFT][2] = (short)buf[152];
		MenuStatus[MENU_FFT][3] = (short)buf[153];

		// Cursors    	
/*		MenuStatus[16][0] = (short) buf[154];                                        // Cursor source
		//Cursor_Enabled = (char) buf[155];
	
		if (Cursor_Enabled) Cursor_Data_First_Draw = 1;
	
		Cursor_Both_Selected = (char) buf[156];
		SelectedCursor = (char) buf[157];

		Cursor_Horizontal_Active = (char) buf[158];
		Cursor_Horizontal_Position_Real1 = (long) buf[159];
		Cursor_Horizontal_Position_Real2 = (long) buf[160];

		Cursor_Vertical_Active = (char) buf[161];
		Cursor_Vertical_Position_TY_1 = (long) buf[162];
		Cursor_Vertical_Position_TY_2 = (long) buf[163];
	
		// Quick Measure
		//QM_Enabled = 0; //QM_Enabled = (char) buf[164];
	
		//if (QM_Enabled) Quick_Measure_First_Draw = 1;
	
		QM_Type[0] = (char) buf[165];
		QM_Type[1] = (char) buf[166];
		QM_Type[2] = (char) buf[167];
		QM_Channel[0] = (char) buf[168];
		QM_Channel[1] = (char) buf[169];
	
		QM_Channel[2] = (char) buf[170];
		QM_Horizontal_Active = (char) buf[171];
		QM_Vertical_Active = (char) buf[172];
		Quick_Measure_Threshold_btn_select = (char) buf[173];
*/	
		// Display
		//GridColorArrayPos = (unsigned char) buf[174];
		//GridColor_Val     = (unsigned char) buf[175];
		// MenuStatus[MENU_DISPLAY][3] = (unsigned char) buf[176];				// no display
		//dotted_grid       = (unsigned char) buf[177];
		MenuStatus[MENU_DISPLAY][0] = (short) buf[178];                                         // Display - Persist
		//MenuStatus[MENU_DISPLAY][4] = (short) buf[179];                                         // Display - Draw Mode
	
		MenuStatus[MENU_ACQUIRE][0] = (short) buf[180];                                         // Acquire - Logic
		MenuStatus[MENU_ACQUIRE][1] = (short) buf[181];                                         // Acquire - Averaging
		MenuStatus[MENU_ACQUIRE][2] = (short) buf[183];                                         // Acquire - Filter Setting
		MenuStatus[MENU_ACQUIRE][3] = (short) buf[184];                                         // Acquire - Interpolation
	
		//MenuStatus[MENU_AUTOSCALE][1] = (short) buf[185];                                        // Autoscale - Slow TB
//		MenuStatus[MENU_HARDWARE][1]  = (short) buf[186];
	
		MenuPopupStatus[8][0] = (short) buf[187];		                    //BF popup for external trigger
		MenuPopupStatus[8][1] = (short) buf[188];
		MenuPopupStatus[8][2] = (short) buf[189];
		MenuPopupStatus[11][0] = (short) buf[190];		                    //BF popup for TV trigger
		MenuPopupStatus[11][1] = (short) buf[191];

		// 192 BF -> reserved s.o. ledval

		Channel_1_Active = (char) buf[193];
		Channel_2_Active = (char) buf[194];
		Channel_3_Active = (char) buf[195];
		Channel_4_Active = (char) buf[196];

		FFT_grid = (unsigned char) buf[197];		

		// free = (volatile short) buf[198];
	
		HoldOff_Value = (int) buf[199];
		HoldOff_Expo  = (int) buf[200];
		HoldOff_Steps = (int) buf[201];
	
		Pulse11_Value = (int) buf[202];
		Pulse11_Expo  = (int) buf[203];
		Pulse11_Steps = (int) buf[204];
	
		Pulse12_Value = (int) buf[205];
		Pulse12_Expo  = (int) buf[206];
		Pulse12_Steps = (int) buf[207];
	
		Pulse21_Value = (int) buf[208];
		Pulse21_Expo  = (int) buf[209];
		Pulse21_Steps = (int) buf[210];
	
		Pulse22_Value = (int) buf[211];
		Pulse22_Expo  = (int) buf[212];
		Pulse22_Steps = (int) buf[213];
	
	//BF new Math values
		Math_Offset     = (int) buf[214]; 
		Math_Mul_Offset = (int) buf[215]; 
		Math_Sub_Offset = (int) buf[216];
		Math_Add_Offset = (int) buf[217];
		math_scale      = (int) buf[218];
		math_mul_scale  = (int) buf[219];
		math_sub_scale  = (int) buf[220];
		math_add_scale  = (int) buf[221];
	//BF end
	
		//MenuStatus[MENU_EXTUTILS][0] = (short) buf[222];	//BF #001
		//MenuStatus[MENU_EXTUTILS][1] = (short) buf[223];	//BF #001
		//MenuStatus[MENU_EXTUTILS][2] = (short) buf[224];	//BF #001
		//MenuStatus[MENU_EXTUTILS][3] = (short) buf[225];	//BF #001
//		MenuStatus[MENU_HARDWARE][0] = (short) buf[226];	//BF #002

		Trig_Pos_Mem     = (int) buf[229];
		Trig_Pos_Display = (int) buf[230];
		//not used    = (float) buf[231];
		MemWinStart      = (int) buf[232];
		
		Trig_Pos_Mem_old = (int) buf[233];
		Trig_Pos_Display_old = (int) buf[234];
		//MenuStatus[MENU_QUICKMEASURE][4] = (short) buf[235];
		MemWinStart_old = (int) buf[236];
		
		ZoomFactor        = (float) buf[237];
		ZoomFactor_old    = (float) buf[238];
		ZoomFactorDel     = (float) buf[239];

		//MenuStatus[MENU_QUICKMEASURE][0] = (short) buf[240];
		//MenuStatus[MENU_QUICKMEASURE][1] = (short) buf[241];
		//MenuStatus[MENU_QUICKMEASURE][2] = (short) buf[242];
		
		CH1_Del_Correct = (char) buf[243];	//BF #001
		CH2_Del_Correct = (char) buf[244];	//BF #001
		CH3_Del_Correct = (char) buf[245];	//BF #001
		CH4_Del_Correct = (char) buf[246];	//BF #001
	
		//  = (short) buf[247];	

	// BF new zerolevels for XY
		Virtual_ZeroLevelXYCH1 = (int)buf[248];
		Virtual_ZeroLevelXYCH2 = (int)buf[249];
		Virtual_ZeroLevelXYCH3 = (int)buf[250];
		Virtual_ZeroLevelXYCH4 = (int)buf[251];
	// BF end

		//BF backup values must be restored too
		ZeroLevelCH1_bak = (int)buf[252];
		ZeroLevelCH2_bak = (int)buf[253];
		ZeroLevelCH3_bak = (int)buf[254];
		ZeroLevelCH4_bak = (int)buf[255];

		Virtual_ZeroLevelCH1_bak = (int)buf[256];
		Virtual_ZeroLevelCH2_bak = (int)buf[257];
		Virtual_ZeroLevelCH3_bak = (int)buf[258];
		Virtual_ZeroLevelCH4_bak = (int)buf[259];

		TriggerMode_bak = (int)buf[260];
/*
		Cursor_Vertical_Position_XY_1 = (long)buf[261];
		Cursor_Vertical_Position_XY_2 = (long)buf[262];
		Cursor_Vertical_Position_FFT_1 = (long)buf[263];
		Cursor_Vertical_Position_FFT_2 = (long)buf[264];
		//SavePointer = (char)buf[265];
*/
		Channel_1_Active_bak = (char)buf[266];
		Channel_2_Active_bak = (char)buf[267];
		Channel_3_Active_bak = (char)buf[268];
		Channel_4_Active_bak = (char)buf[269];
		Channel_Math_Active_bak = (char)buf[270];

		// DrawMono  = (unsigned char)buf[271];
		MainTimebase_bak   = (char)buf[272];
		// free   = (short)buf[273];
		// free = (short)buf[274];

		ZeroLevelCH1_Main_bak = (int)buf[275];
		ZeroLevelCH2_Main_bak = (int)buf[276];
		ZeroLevelCH3_Main_bak = (int)buf[277];
		ZeroLevelCH4_Main_bak = (int)buf[278];

		Virtual_ZeroLevelCH1_Main_bak = (int)buf[279];
		Virtual_ZeroLevelCH2_Main_bak = (int)buf[280];
		Virtual_ZeroLevelCH3_Main_bak = (int)buf[281];
		Virtual_ZeroLevelCH4_Main_bak = (int)buf[282];

		
		// BF get ADC registervalues #002
		channel_Adr_add12 = buf[283];
		channel_Adr_add34 = buf[284];
		adc_change12_reg = buf[285];
		adc_change34_reg = buf[286];
		adc_ctr12_reg    = buf[287];
		adc_ctr34_reg    = buf[288];

		//MenuStatus[MENU_DISPL_SETUP][0] = (short) buf[289];			// grid color
		//MenuStatus[MENU_DISPL_SETUP][1] = (short) buf[290];			// grid layout
		//MenuStatus[MENU_DISPL_SETUP][2] = (short) buf[291];			// status layout
		//MenuStatus[MENU_DISPL_SETUP][3] = (short) buf[292];			// not used
		//MenuStatus[MENU_DISPL_SETUP][4] = (short) buf[293];			// not used
		//MenuStatus[MENU_DISPL_SETUP][5] = (short) buf[294];			// not used
	
		//thres_low[4] = (short) buf[295];					// QM thresholds;
		//thres_mdl[4] = (short) buf[296];					// QM thresholds;
		//thres_up[4]  = (short) buf[297];					// QM thresholds;
		//thres_type[4] = (short) buf[298];					// QM threshold type


		LED_ON[0] = 0;
		LED_ON[1] = 0;
		LED_ON[2] = 0;
		LED_ON[3] = 0;
		LED_ON[4] = 0;
		LED_ON[5] = 1;
		//LED_ON[6] = 0;
		//LED_ON[7] = 0;
		LED_ON[8] = 0;
		LED_ON[9] = 0;
		LED_ON[10] = 0;
		LED_ON[11] = 0;
		//LED_ON[12] = 0;
		//LED_ON[13] = 0;
		LED_ON[14] = 0;
		LED_ON[15] = 0;

		//decode LED settings
		if (((int) buf[192] & 0x0001) == 0x0001) LED_ON[0] = 1;
		if (((int) buf[192] & 0x0002) == 0x0002) LED_ON[1] = 1;
		if (((int) buf[192] & 0x0004) == 0x0004) LED_ON[2] = 1;
		if (((int) buf[192] & 0x0008) == 0x0008) LED_ON[3] = 1;
	        if (((int) buf[192] & 0x0010) == 0x0010) LED_ON[4] = 1;  // Math
		//if (((int) buf[192] & 0x0020) == 0x0020) LED_ON[5] = 1; // General
		//if (((int) buf[192] & 0x0040) == 0x0040) LED_ON[6] = 1;  // cursor
		//if (((int) buf[192] & 0x0080) == 0x0080) LED_ON[7] = 1; // Quick Measure
		if (((int) buf[192] & 0x0100) == 0x0100) LED_ON[8] = 1;
		if (((int) buf[192] & 0x0200) == 0x0200) LED_ON[9] = 1;
		if (((int) buf[192] & 0x0400) == 0x0400) LED_ON[10] = 1;
		if (((int) buf[192] & 0x0800) == 0x0800) LED_ON[11] = 1;

		UserIF::Buttonhandler(BTN_STOP);			// switch to stop mode	

		switch(Pulse11_Expo)
		{
			case 1:
			{
				Pulse11.Write_Value((float) Pulse11_Value / (float) 1000000000);
				break;
			}
			case 2:
			{
				Pulse11.Write_Value((float) Pulse11_Value / (float) 1000000);
				break;
			}
			case 3:
			{
				Pulse11.Write_Value((float) Pulse11_Value / (float) 1000);
				break;
			}
		}
		Pulse11.Write_StepSize(Pulse11_Steps);
		Pulse11.RenderText();

		switch(Pulse12_Expo)
		{
			case 1:
			{
				Pulse12.Write_Value((float) Pulse12_Value / (float) 1000000000);
				break;
			}
			case 2:
			{
				Pulse12.Write_Value((float) Pulse12_Value / (float) 1000000);
				break;
			}
			case 3:
			{
				Pulse12.Write_Value((float) Pulse12_Value / (float) 1000);
				break;
			}
		}
		Pulse12.Write_StepSize(Pulse12_Steps);
		Pulse12.RenderText();
	
		switch(Pulse21_Expo)
		{
			case 1:
			{
				Pulse21.Write_Value((float) Pulse21_Value / (float) 1000000000);
				break;
			}
			case 2:
			{
				Pulse21.Write_Value((float) Pulse21_Value / (float) 1000000);
				break;
			}
			case 3:
			{
				Pulse21.Write_Value((float) Pulse21_Value / (float) 1000);
				break;
			}
		}
		Pulse21.Write_StepSize(Pulse21_Steps);
	        Pulse21.RenderText();

		switch(Pulse22_Expo)
		{
			case 1:
			{
				Pulse22.Write_Value((float) Pulse22_Value / (float) 1000000000);
				break;
			}
			case 2:
			{
				Pulse22.Write_Value((float) Pulse22_Value / (float) 1000000);
				break;
			}
			case 3:
			{
				Pulse22.Write_Value((float) Pulse22_Value / (float) 1000);
				break;
			}
		}
		Pulse22.Write_StepSize(Pulse22_Steps);
		Pulse22.RenderText();


//BF changed
		//Math_Offset_mul.Write_StepSize(1);
	//	Math_Offset_mul.Write_Value((float)(Math_Mul_Offset / 50000 ));
	//	Math_Offset_mul.RenderText();
	
	
		//Math_Offset_sub.Write_StepSize(1);
	//	Math_Offset_sub.Write_Value((float)(Math_Sub_Offset / 50000 ));
	//	Math_Offset_sub.RenderText();
	
	
		//Math_Offset_add.Write_StepSize(0.001);
	//	Math_Offset_add.Write_Value((float)(Math_Add_Offset / 50000 ));
	//	Math_Offset_add.RenderText();
//BF end	

		UserIF::UpdateChannel(1, Channel_1_Active);
		UserIF::UpdateChannel(2, Channel_2_Active);
					
		VoltageChangedCh1 = 1;
		VoltageChangedCh2 = 1;

		Rotary_Steps = 0;
		UserIF::ON_Zero_Channel_1();
		Rotary_Steps = 0;
		UserIF::ON_Zero_Channel_2();
	
		Hardware::SetSwitches(1, Selected_Voltage_CH1);
		Hardware::SetSwitches(2, Selected_Voltage_CH2);

		if (NumberOfChannels > 2)
		{
	
			UserIF::UpdateChannel(3, Channel_3_Active);
			UserIF::UpdateChannel(4, Channel_4_Active);

			VoltageChangedCh3 = 1;
			VoltageChangedCh4 = 1;

			Rotary_Steps = 0;
			UserIF::ON_Zero_Channel_3();
			Rotary_Steps = 0;
			UserIF::ON_Zero_Channel_4();
		
			Hardware::SetSwitches(3, Selected_Voltage_CH3);
			Hardware::SetSwitches(4, Selected_Voltage_CH4);
	
		}
		else
		{
			UserIF::UpdateChannel(3, 0);
			UserIF::UpdateChannel(4, 0);
		}


		TriggerLevelChanged = 1;
		TriggerModeChanged = 1;	
		TriggerWayChanged = 1;
		TimebaseChanged = 1;
		TimeOffsetChanged = 1;
		//MenuStatusChanged = 1;
		MenuAktive = 1;
	
		//Cursor_Data_First_Draw = 0;
		//Quick_Measure_First_Draw = 0;
	
		Display::RecalcTimeParameters();
		Hardware::SetupTrigger();
		Display::CALCPRETRIGGER();

//		GainIdx = MenuStatus[MENU_HARDWARE][1] - 230;	//BF #004
//		Signal::Build_ScaleLookupTable();

		ACQ_Average = (unsigned char)(MenuStatus[MENU_ACQUIRE][1] - AVRG_OFF);

		if (Memory_Window_visible) 
		Display::DRAWMEMORY(1, 0, 0);		//close memory window
	
		//Display::DrawInitialScreen();
		//Display::RefreshScreen();
	
		if(FFT_Mode != FFT_OFF)
		{
			Display::FFT_CalcCursorData();
			Signal::FFT_buildScaleLookupTable();
		}
		else
		Display::CALCCURSORDATA();
	
		Hardware::Send_LED();		

		if (USTB_Mode != USTB_OFF)		// if USTB mode was active bevor prepare buffers and pointerrs
		Signal::USTB_clearBuffer();
		else
		{
			S1Ptr = SIGNAL1;		// adjust signal pointers to 16k signal buffers
			S2Ptr = SIGNAL2;
			S3Ptr = SIGNAL3;
			S4Ptr = SIGNAL4;
			SMPtr = SIGNALM;
		}
		
		rcode = 0;	//success
	}

	return rcode;
}
//#####################################################################################################################################################
//BF Save/Recall -> load sample data from flash
void AMDFlash::LoadTracePoints(unsigned long *flash_address)
{
	unsigned long lX, bufdata;
	unsigned long buf[300];	

	for (lX = 0; lX < 300; lX++) buf[lX] = flash_address[0x4000 - lX];		
	
	if ((buf[4] == tc_version) && (buf[5] == tc_subversion) && (buf[6] == tc_dev_version) && (buf[1] == 0xAACC5523) && (buf[2] == 0x22334455) && (buf[299] == 0x23232323))
	{

		for (lX = 0; lX < 0x1000; lX++)
		{
			bufdata = *(flash_address + lX + 0x0000);
			S1Ptr[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
			S1Ptr[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
			S1Ptr[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
			S1Ptr[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);
		
			bufdata = *(flash_address + lX + 0x1000);
			S2Ptr[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
			S2Ptr[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
			S2Ptr[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
			S2Ptr[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);

		}	

		if (NumberOfChannels == 4)
		{
			for (lX = 0; lX < 0x1000; lX++)
			{
				bufdata = *(flash_address + lX + 0x2000);
				S3Ptr[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
				S3Ptr[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
				S3Ptr[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
				S3Ptr[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);
			
				bufdata = *(flash_address + lX + 0x3000);
				S4Ptr[(lX * 4) + 0] = (unsigned char) ((bufdata & 0x000000FF) >> 0);	
				S4Ptr[(lX * 4) + 1] = (unsigned char) ((bufdata & 0x0000FF00) >> 8);
				S4Ptr[(lX * 4) + 2] = (unsigned char) ((bufdata & 0x00FF0000) >> 16);
				S4Ptr[(lX * 4) + 3] = (unsigned char) ((bufdata & 0xFF000000) >> 24);
			}	

			//delete last 1200 byte which are used for the settings
			for(int i=0;i < 1200;i++) S4Ptr[0x4000-i] = ADC_ZERO + int((float)Virtual_ZeroLevelCH4 / ScaleFactor[Selected_Voltage_CH4][GainIdx]);
		}


	}
}
//#####################################################################################################################################################
//BF Save/Recall -> load trace
void AMDFlash::ReadTraceFromFlash(unsigned long *flash_address)
{
		if (LoadTraceSettings(flash_address) == 0)	// load trace points if load of settings where successful
		LoadTracePoints(flash_address);
}
//#####################################################################################################################################################
/* BF not needed at this time
void AMDFlash::copy_flash_to_planes(void)
{
	unsigned long lX;
#ifdef _Debug_	
	printf("Restore the planes \n");
#endif

//printf("AMDFlash::copy_flash_to_planes\n");	
	for (lX = 0; lX < 9600; lX++)
	{	
		UI_Plane1[lX] = *(UI_Plane1_Flash + lX);	
		Buffer_UI2Plane[lX] = *(UI_Plane2_Flash + lX);	
		UI_Plane3[lX] = *(UI_Plane3_Flash + lX);	
		UI_Plane4[lX] = *(UI_Plane4_Flash + lX);	
		UI_Plane5[lX] = *(UI_Plane5_Flash + lX);	
		Channel_Plane1[lX] = *(Channel_Plane1_Flash + lX);	
		Channel_Plane2[lX] = *(Channel_Plane2_Flash + lX);	
		Channel_Plane3[lX] = *(Channel_Plane3_Flash + lX);	
		Channel_Plane4[lX] = *(Channel_Plane4_Flash + lX);	
		Channel_Math_Plane[lX] = *(Channel_Math_Plane_Flash + lX);	
		Buffer_Plane1[lX] = *(Channel_Plane1_Flash + lX);	
		Buffer_Plane2[lX] = *(Channel_Plane2_Flash + lX);	
		Buffer_Plane3[lX] = *(Channel_Plane3_Flash + lX);	
		Buffer_Plane4[lX] = *(Channel_Plane4_Flash + lX);	
		Buffer_PlaneM[lX] = *(Channel_Math_Plane_Flash + lX);			
		Memory_Plane1[lX] = *(Memory_Plane1_Flash + lX);	
		Memory_Plane2[lX] = *(Memory_Plane2_Flash + lX);	
		Memory_Plane3[lX] = *(Memory_Plane3_Flash + lX);	
		Marker_Plane1[lX] = *(Marker_Plane1_Flash + lX);	
		Marker_Plane2[lX] = *(Marker_Plane2_Flash + lX);	
		Grid_Plane[lX] = *(Grid_Plane_Flash + lX);	
	}
	UpdateMenuTextPlane = true;
}
*/
//#####################################################################################################################################################
// BF needed for copying splash logo to screen
void AMDFlash::copy_flash_to_plane(unsigned long *flash_address, unsigned long *plane)
{
	unsigned long lX;
//printf("AMDFlash::copy_flash_to_plane\n");		
	for (lX = 0; lX < 9600; lX++)
	//for (lX = TOP_PLANE_MAX; lX < 9600; lX++)
	{	
		plane[lX] = flash_address[lX];	
	}
}
//#####################################################################################################################################################
// save actual configuration to flash
void AMDFlash::Write_Config_Flash(void)
{
	unsigned long *SectorAddress;
	unsigned long buf[300];
	int led_value = 0x0000;
	char lWriteError=0;

	if (config_changed == 0) return;
	//if (AS_request){ config_changed = 0; return; }			//don't write config if autoscale is active

	printf("AMDFlash::Write_Config_Flash\r\n");

	//backup LED setting
	if (LED_ON[0] == 1) led_value = led_value | 0x0001;		// Select Channel 1
	if (LED_ON[1] == 1) led_value = led_value | 0x0002;		// Select Channel 2
	if (LED_ON[2] == 1) led_value = led_value | 0x0004;		// Select Channel 3
	if (LED_ON[3] == 1) led_value = led_value | 0x0008;		// Select Channel 4
	if (LED_ON[4] == 1) led_value = led_value | 0x0010;		// Select Channel Math
	if (LED_ON[5] == 1) led_value = led_value | 0x0020;		// General
	if (LED_ON[6] == 1) led_value = led_value | 0x0040;		// Cursor						
	if (LED_ON[7] == 1) led_value = led_value | 0x0080;		// Quick Measure
	if (LED_ON[8] == 1) led_value = led_value | 0x0100;		// Edge
	if (LED_ON[9] == 1) led_value = led_value | 0x0200;		// BF Trigger armed
	if (LED_ON[10] == 1) led_value = led_value | 0x0400;		// Pulse Width
	if (LED_ON[11] == 1) led_value = led_value | 0x0800;		// BF Trigger Event
	if (LED_ON[12] == 1) led_value = led_value | 0x1000;		// Run/Stop Green
	if (LED_ON[13] == 1) led_value = led_value | 0x2000;		// Run/Stop Red
	if (LED_ON[14] == 1) led_value = led_value | 0x4000;		// Single Red					
	if (LED_ON[15] == 1) led_value = led_value | 0x8000;		// Single Green	

	SectorAddress = (unsigned long) nasys_main_flash + Config_Flash;			// get config start address

	if (-1 == nr_flash_erase_sector((unsigned long *) nasys_main_flash, SectorAddress))		// erase config area 
	{ lWriteError = true; }

	buf[0] = (unsigned long) 0x00;
	buf[1] = (unsigned long) 0xAACC5523;		// Configuration Start Sign
	buf[2] = (unsigned long) 0x22334455;
	buf[3] = (unsigned long) 0x00;		
	buf[4] = (unsigned long) tc_version;		// Version
	buf[5] = (unsigned long) tc_subversion;		// SubVersion
	buf[6] = (unsigned long) tc_dev_version;	// Revision
	buf[7] = (unsigned long) tc_dev_subversion;	// Compile Status
	buf[8] = (unsigned long) 0x00;   	

	buf[9] = (unsigned long) FlashPreset;		//BF preset flag

	// Timebase
	buf[10] = (unsigned long) MenuStatus[MENU_TIMEBASE][0];		//Main
	buf[11] = (unsigned long) MenuStatus[MENU_TIMEBASE][1];		//Delayed
	buf[12] = (unsigned long) MenuStatus[MENU_TIMEBASE][2];		//XY mode
	buf[13] = (unsigned long) MenuStatus[MENU_TIMEBASE][3];		
	buf[14] = (unsigned long) MenuStatus[MENU_TIMEBASE][4];		
	buf[15] = (unsigned long) MenuStatus[MENU_DISPLAY][2];	
	buf[16] = (unsigned long) XY_Mode;
	buf[17] = (unsigned long) MainTimebase;	
	buf[18] = (unsigned long) USTB_Mode;
	buf[19] = (unsigned long) USTB_Mode_bak;
	buf[20] = (unsigned long) USTB_Dir;	
	buf[21] = (unsigned long) USTB_Disp;	
	buf[22] = (unsigned long) DelayedTimebase;

	buf[23] = (unsigned long) MenuStatus[MENU_UTILITY][3];				//calibration set
	buf[24] = (unsigned long) dmode_Window_Offset_Pos;
	buf[25] = (unsigned long) MenuStatus[MENU_USTB][2];				// USTB buffer size

	buf[26] = (unsigned long) New_Menu;						//BF added
	buf[27] = (unsigned long) Active_Menu;						//BF added
	buf[28] = (unsigned long) MenuAktive;						//BF added

	buf[29] = (unsigned long) MenuStatus[MENU_USTB][0];				//USTB mode			
	buf[30] = (unsigned long) MenuStatus[MENU_USTB][1];				//USTB display
	buf[31] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][0];				// LED 1
	buf[32] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][1];				// LED 2
    	//buf[33] = (unsigned long) free;
	buf[34] = (unsigned long) SwitchesTB;	
	
	// Trigger
	buf[35] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][0];				// Triggering - Auto / Normal / Combi
	buf[36] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][1];				// Triggering - Coupling DC/AC
	buf[37] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][2];				// Trigger sub menu LED setting
	buf[38] = (unsigned long) MenuStatus[MENU_TRIGGERMODE][4];				// Triggering - External Probe
	buf[39] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][3];				// Triggering - TV

	buf[40] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][0];    				// Triggering - positive edge
	buf[41] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][1]; 				// Triggering - Edge Source 
	buf[42] = (unsigned long) MenuStatus[MENU_TRIGGEREDGE][2];				// Triggering - externel triggering
	buf[43] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][0];				// Triggering - Source Channel 1
	buf[44] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][1];				// Triggering - Negative Pulse
	buf[45] = (unsigned long) MenuStatus[MENU_PULSEWIDTH][2];				// Triggering - bigger - smaller - then
	buf[46] = (unsigned long) Selected_Trigger_Source;					// Selected Trigger Source is Channel 1
	buf[47] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][3];				// Trigger sub menu Single Trigger setting;
	buf[48] = (unsigned long) PulseToggle;                                              // Pulse width button selection
	//buf[49] = (unsigned long) free;
	buf[50] = (unsigned long) FFT_TriggerMode_bak;					//BF backup the trigger mode
	buf[51] = (unsigned long) Trigger_Pos_CH1;                                          // Trigger Level Channel 1
	buf[52] = (unsigned long) Trigger_Pos_CH2;                                          // Trigger Level Channel 2
	buf[53] = (unsigned long) Trigger_Pos_CH3;                                          // Trigger Level Channel 3
	buf[54] = (unsigned long) Trigger_Pos_CH4;                                          // Trigger Level Channel 4
	buf[55] = (unsigned long) trg_val_reg;                                              // Trigger Value
	buf[56] = (unsigned long) trg_hyst_reg;                                             // Trigger Hysterese
	buf[57] = (unsigned long) trig_range_reg;                                           // Trigger Range
	buf[58] = (unsigned long) trig_holdoff_reg;                                         // Trigger Holdoff
	buf[59] = (unsigned long) trig_width_reg;                                           // Trigger Width

	buf[60] = (unsigned long) ctrl_reg;                                                 // Control register
	buf[61] = (unsigned long) adc_ctrl_reg;                                             // ADC Control register
	buf[62] = (unsigned long) pre_reg;                                                  // PreTrigger Value
//	buf[63] = (unsigned long) channel_Adr_add;                                          // Channel Add Value
	buf[64] = (unsigned long) MenuStatus[MENU_TRIGGERSUB][4];			// Trigger sub menu Auto Pre Trigger setting;
	buf[65] = (unsigned long) triggering;                                               // Triggering
	buf[66] = (unsigned long) TriggerWay;                                               // Triggerway
	buf[67] = (unsigned long) Trigger_Pos_CHE;                                          // Trigger extern value
	
	//---------------------------------------------------
	// 		Channel 1
	//---------------------------------------------------
	buf[68] = (unsigned long) MenuStatus[1][0];                                         // Channel - Coupling
	buf[69] = (unsigned long) MenuStatus[1][1];                                         // Channel - BW Linit
	buf[70] = (unsigned long) MenuStatus[1][2];                                         // Channel - Invert
	buf[71] = (unsigned long) MenuStatus[1][3];                                         // Channel - Probe
	
	buf[72] = (unsigned long) SwitchesCH1;                                              // Switches for Channel 1 and for the other the AC bit
	buf[73] = (unsigned long) Selected_Voltage_CH1;                                     // Voltage 5V
	buf[74] = (unsigned long) Trigger_Pos_CH1_Old;
	buf[75] = (unsigned long) NoDisplay[0];
	buf[76] = (unsigned long) thres_low[0];						// QM thresholds
	buf[77] = (unsigned long) thres_mdl[0];						// QM thresholds
	buf[78] = (unsigned long) thres_up[0];						// QM thresholds
	buf[79] = (unsigned long) thres_type[0];					// QM threshold type
	buf[80] = (unsigned long) Selected_Voltage_CH1_bak;
	buf[81] = (unsigned long) Trigger_Pos_CH1_bak; 
	buf[82] = (unsigned long) CH1_DAC_Offset;                                           // Channel DAC
	buf[83] = (unsigned long) ZeroLevelCH1;                                             // ZeroLevel
	buf[84] = (unsigned long) Virtual_ZeroLevelCH1;                                     // Virtual ZeroLevel
        //buf[85] = (unsigned long) alpha[0];                                               // ADC-alpha correction -> BF changed
    	//buf[85] = (unsigned long) CH1_Zero_Offset;                                          // Shift offset when stopped

	//---------------------------------------------------
	// 		Channel 2
	//---------------------------------------------------
	buf[86] = (unsigned long) MenuStatus[2][0];                                         // Channel - Coupling
	buf[87] = (unsigned long) MenuStatus[2][1];                                         // Channel - BW Linit
	buf[88] = (unsigned long) MenuStatus[2][2];                                         // Channel - Invert
	buf[89] = (unsigned long) MenuStatus[2][3];                                         // Channel - Probe

	buf[90] = (unsigned long) SwitchesCH2;                                              // Switches for Channel
	buf[91] = (unsigned long) Selected_Voltage_CH2;                                     // Voltage 5V
	buf[92] = (unsigned long) Trigger_Pos_CH2_Old;
	buf[93] = (unsigned long) NoDisplay[1];
	buf[94] = (unsigned long) thres_low[1];						// QM thresholds
	buf[95] = (unsigned long) thres_mdl[1];						// QM thresholds
	buf[96] = (unsigned long) thres_up[1];						// QM thresholds
	buf[97] = (unsigned long) thres_type[1];					// QM threshold type
	buf[98] = (unsigned long) Selected_Voltage_CH2_bak;
	buf[99] = (unsigned long) Trigger_Pos_CH2_bak; 
	buf[100] = (unsigned long) CH2_DAC_Offset;                                           // Channel DAC
	buf[101] = (unsigned long) ZeroLevelCH2;                                             // ZeroLevel
	buf[102] = (unsigned long) Virtual_ZeroLevelCH2;                                     // Virtual ZeroLevel
        //buf[103] = (unsigned long) alpha[1];                                               // ADC-alpha correction -> BF changed
    //buf[103] = (unsigned long) CH2_Zero_Offset;                                          // Shift offset when stopped

	//---------------------------------------------------
	// 		Channel 3
	//---------------------------------------------------
	buf[104] = (unsigned long) MenuStatus[3][0];                                         // Channel - Coupling
	buf[105] = (unsigned long) MenuStatus[3][1];                                         // Channel - BW Linit
	buf[106] = (unsigned long) MenuStatus[3][2];                                         // Channel - Invert
	buf[107] = (unsigned long) MenuStatus[3][3];                                         // Channel - Probe
	buf[108] = (unsigned long) SwitchesCH3;                                              // Switches for Channel
	buf[109] = (unsigned long) Selected_Voltage_CH3;                                     // Voltage 5V
	buf[110] = (unsigned long) Trigger_Pos_CH3_Old;
	buf[111] = (unsigned long) NoDisplay[2];;
	buf[112] = (unsigned long) thres_low[2];					// QM thresholds
	buf[113] = (unsigned long) thres_mdl[2];					// QM thresholds
	buf[114] = (unsigned long) thres_up[2];						// QM thresholds
	buf[115] = (unsigned long) thres_type[2];					// QM threshold type
	buf[116] = (unsigned long) Selected_Voltage_CH3_bak;
	buf[117] = (unsigned long) Trigger_Pos_CH3_bak; 
	buf[118] = (unsigned long) CH3_DAC_Offset;                                           // Channel DAC
	buf[119] = (unsigned long) ZeroLevelCH3;                                             // ZeroLevel
	buf[120] = (unsigned long) Virtual_ZeroLevelCH3;                                     // Virtual ZeroLevel
       // buf[121] = (unsigned long) alpha[2];                                               // ADC-alpha correction -> BF changed
    //buf[121] = (unsigned long) CH3_Zero_Offset;                                          // Shift offset when stopped

	//---------------------------------------------------
	// 		Channel 4
	//---------------------------------------------------
	buf[122] = (unsigned long) MenuStatus[4][0];                                         // Channel - Coupling
	buf[123] = (unsigned long) MenuStatus[4][1];                                         // Channel - BW Linit

	buf[124] = (unsigned long) MenuStatus[4][2];                                         // Channel - Invert
	buf[125] = (unsigned long) MenuStatus[4][3];                                         // Channel - Probe
	buf[126] = (unsigned long) SwitchesCH4;                                              // Switches for Channel
	buf[127] = (unsigned long) Selected_Voltage_CH4;                                     // Voltage 5V
	buf[128] = (unsigned long) Trigger_Pos_CH4_Old;
	buf[129] = (unsigned long) NoDisplay[3];;
	buf[130] = (unsigned long) thres_low[3];					// QM thresholds;
	buf[131] = (unsigned long) thres_mdl[3];					// QM thresholds;
	buf[132] = (unsigned long) thres_up[3];						// QM thresholds;
	buf[133] = (unsigned long) thres_type[3];					// QM threshold type
	buf[134] = (unsigned long) Selected_Voltage_CH4_bak;
	buf[135] = (unsigned long) Trigger_Pos_CH4_bak; 
	buf[136] = (unsigned long) CH4_DAC_Offset;                                           // Channel DAC
	buf[137] = (unsigned long) ZeroLevelCH4;                                             // ZeroLevel
	buf[138] = (unsigned long) Virtual_ZeroLevelCH4;                                     // Virtual ZeroLevel
       // buf[139] = (unsigned long) alpha[3];                                               // ADC-alpha correction -> BF changed

 
	buf[139] = (unsigned long) FFT_Layout;                                          // Layout in FFT function

	//Math Settings
	buf[140] = (unsigned long) Channel_Math_Active;                                      // Math channel
	//buf[141] = (unsigned long) MenuStatus[MENU_MATH][0];                                 // scale
	buf[142] = (unsigned long) MenuStatus[MENU_MATH][2];                                 // math 1*2 
	buf[143] = (unsigned long) MenuStatus[MENU_MATH][3];                                 // math 1-2 
	buf[144] = (unsigned long) MenuStatus[MENU_MATH][4];                                 // math 1+2 
	
	//FFT Settings
	buf[145] = (unsigned long) FFT_Mode;
	buf[146] = (unsigned long) FFT_Length;
	buf[147] = (unsigned long) FFT_Level;
	buf[148] = (unsigned long) FFT_Windex;
//BF not used	buf[149] = (unsigned long) FFT_Scale;
	
	buf[150] = (unsigned long) MenuStatus[MENU_FFT][0]; 
	buf[151] = (unsigned long) MenuStatus[MENU_FFT][1];
	buf[152] = (unsigned long) MenuStatus[MENU_FFT][2];
	buf[153] = (unsigned long) MenuStatus[MENU_FFT][3];
	
	// Cursors    	
	buf[154] = (unsigned long) MenuStatus[MENU_CURSOR][0];                                        // Cursor soruce
	buf[155] = (unsigned long) Cursor_Enabled;
	buf[156] = (unsigned long) Cursor_Both_Selected;
	buf[157] = (unsigned long) SelectedCursor;

	buf[158] = (unsigned long) Cursor_Horizontal_Active;
	buf[159] = (unsigned long) Cursor_Horizontal_Position_Real1;
	buf[160] = (unsigned long) Cursor_Horizontal_Position_Real2;

	buf[161] = (unsigned long) Cursor_Vertical_Active;
	buf[162] = (unsigned long) Cursor_Vertical_Position_TY_1;
	buf[163] = (unsigned long) Cursor_Vertical_Position_TY_2;

	// Quick Measure
	buf[164] = (unsigned long) QM_Enabled;
	buf[165] = (unsigned long) QM_Type[0];
	buf[166] = (unsigned long) QM_Type[1];
	buf[167] = (unsigned long) QM_Type[2];
	buf[168] = (unsigned long) QM_Channel[0];
	buf[169] = (unsigned long) QM_Channel[1];
	buf[170] = (unsigned long) QM_Channel[2];
	buf[171] = (unsigned long) QM_Horizontal_Active;
	buf[172] = (unsigned long) QM_Vertical_Active;
	buf[173] = (unsigned long) Quick_Measure_Threshold_btn_select;
	
	// Display
	buf[174] = (unsigned long) GridColorArrayPos;                                        // Normal Grid
	buf[175] = (unsigned long) GridColor_Val;
	buf[176] = (unsigned long) MenuStatus[MENU_DISPLAY][3];						// no display
	buf[177] = (unsigned long) dotted_grid;
	buf[178] = (unsigned long) MenuStatus[MENU_DISPLAY][0];                                         // Display - Persist
	buf[179] = (unsigned long) MenuStatus[MENU_DISPLAY][4];                                         // Display - Draw Mode
	
	buf[180] = (unsigned long) MenuStatus[MENU_ACQUIRE][0];                                         // Acquire - Logic
	buf[181] = (unsigned long) MenuStatus[MENU_ACQUIRE][1];                                         // Acquire - Averaging
	buf[183] = (unsigned long) MenuStatus[MENU_ACQUIRE][2];                                         // Acquire - Filter Setting
	buf[184] = (unsigned long) MenuStatus[MENU_ACQUIRE][3];                                         // Acquire - Interpolation
	
	buf[185] = (unsigned long) MenuStatus[MENU_AUTOSCALE][3];                                       // Autoscale - Setup TB search
	buf[186] = (unsigned long) MenuStatus[MENU_HARDWARE][1];			//BF gain for the input stage	
	
	buf[187] = (unsigned long) MenuPopupStatus[8][0];
	buf[188] = (unsigned long) MenuPopupStatus[8][1];
	buf[189] = (unsigned long) MenuPopupStatus[8][2];  				//BF popup external trigger
	buf[190] = (unsigned long) MenuPopupStatus[11][0]; 				//BF popup TV trigger
	buf[191] = (unsigned long) MenuPopupStatus[11][1];

	buf[192] = (unsigned long)led_value;						// BF reserved for LED setting -> see at top

	buf[193] = (unsigned long) Channel_1_Active;
	buf[194] = (unsigned long) Channel_2_Active;
	buf[195] = (unsigned long) Channel_3_Active;
	buf[196] = (unsigned long) Channel_4_Active;

	buf[197] = (unsigned long) FFT_grid;				//FFT Grid division

 	buf[198] = (unsigned long) ColorPalette;
	
	
	buf[199] = (unsigned long) HoldOff_Value;
	buf[200] = (unsigned long) HoldOff_Expo;
	buf[201] = (unsigned long) HoldOff_Steps;
	
	buf[202] = (unsigned long) Pulse11_Value;
	buf[203] = (unsigned long) Pulse11_Expo;
	buf[204] = (unsigned long) Pulse11_Steps;
	
	buf[205] = (unsigned long) Pulse12_Value;
	buf[206] = (unsigned long) Pulse12_Expo;
	buf[207] = (unsigned long) Pulse12_Steps;
	
	buf[208] = (unsigned long) Pulse21_Value;
	buf[209] = (unsigned long) Pulse21_Expo;
	buf[210] = (unsigned long) Pulse21_Steps;
	
	buf[211] = (unsigned long) Pulse22_Value;
	buf[212] = (unsigned long) Pulse22_Expo;
	buf[213] = (unsigned long) Pulse22_Steps;

//BF new Math values
	buf[214] = (unsigned int) Math_Offset;
	buf[215] = (unsigned int) Math_Mul_Offset;
	buf[216] = (unsigned int) Math_Sub_Offset;
	buf[217] = (unsigned int) Math_Add_Offset;
	buf[218] = (unsigned int) math_mul_scale;
	buf[219] = (unsigned int) math_mul_scale;
	buf[220] = (unsigned int) math_sub_scale;
	buf[221] = (unsigned int) math_add_scale;
//BF end

	buf[222] = (unsigned long) MenuStatus[MENU_EXTUTILS][0];	//BF #001
	buf[223] = (unsigned long) MenuStatus[MENU_EXTUTILS][1];	//BF #001
	buf[224] = (unsigned long) MenuStatus[MENU_EXTUTILS][2];	//BF #001
	buf[225] = (unsigned long) MenuStatus[MENU_EXTUTILS][3];	//BF #001
	buf[226] = (unsigned long) MenuStatus[MENU_HARDWARE][0];	//BF #002


	buf[229] = (unsigned long) Trig_Pos_Mem;
	buf[230] = (unsigned long) Trig_Pos_Display;
	//buf[231] = (unsigned long) not used;
	buf[232] = (unsigned long) MemWinStart;
	
	buf[233] = (unsigned long) Trig_Pos_Mem_old;
	buf[234] = (unsigned long) Trig_Pos_Display_old;
	buf[235] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][4];
	buf[236] = (unsigned long) MemWinStart_old;
	
	buf[237] = (unsigned long) ZoomFactor;
	buf[238] = (unsigned long) ZoomFactor_old;
	buf[239] = (unsigned long) ZoomFactorDel;

	buf[240] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][0];
	buf[241] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][1];
	buf[242] = (unsigned long) MenuStatus[MENU_QUICKMEASURE][2];

	buf[243] = (unsigned long) CH1_Del_Correct;	//BF #001
	buf[244] = (unsigned long) CH2_Del_Correct;	//BF #001
	buf[245] = (unsigned long) CH3_Del_Correct;	//BF #001
	buf[246] = (unsigned long) CH4_Del_Correct;	//BF #001

	buf[247] = (unsigned long) QP_mode;		//BF #006

// BF new zerolevels for XY
	buf[248] = (unsigned long) Virtual_ZeroLevelXYCH1;
	buf[249] = (unsigned long) Virtual_ZeroLevelXYCH2;
	buf[250] = (unsigned long) Virtual_ZeroLevelXYCH3;
	buf[251] = (unsigned long) Virtual_ZeroLevelXYCH4;
// BF end

	//BF backup values must be saved too
	buf[252] = (unsigned long) ZeroLevelCH1_bak;
	buf[253] = (unsigned long) ZeroLevelCH2_bak;
	buf[254] = (unsigned long) ZeroLevelCH3_bak;
	buf[255] = (unsigned long) ZeroLevelCH4_bak;

	buf[256] = (unsigned long) Virtual_ZeroLevelCH1_bak;
	buf[257] = (unsigned long) Virtual_ZeroLevelCH2_bak;
	buf[258] = (unsigned long) Virtual_ZeroLevelCH3_bak;
	buf[259] = (unsigned long) Virtual_ZeroLevelCH4_bak;

	buf[260] = (unsigned long) TriggerMode_bak;


	buf[261] = (unsigned long) Cursor_Vertical_Position_XY_1;
	buf[262] = (unsigned long) Cursor_Vertical_Position_XY_2;
	buf[263] = (unsigned long) Cursor_Vertical_Position_FFT_1;
	buf[264] = (unsigned long) Cursor_Vertical_Position_FFT_2;
	buf[265] = (unsigned long) SavePointer;


	buf[266] = (unsigned long) Channel_1_Active_bak;
	buf[267] = (unsigned long) Channel_2_Active_bak;
	buf[268] = (unsigned long) Channel_3_Active_bak;
	buf[269] = (unsigned long) Channel_4_Active_bak;
	buf[270] = (unsigned long) Channel_Math_Active_bak;
	
	buf[271] = (unsigned long) DrawMono;
	buf[272] = (unsigned long) MainTimebase_bak;
	buf[273] = (unsigned long) MenuStatus[MENU_QMTHRESHOLDS][0];
	//buf[274] = free

	buf[275] = (unsigned long) ZeroLevelCH1_Main_bak;
	buf[276] = (unsigned long) ZeroLevelCH2_Main_bak;
	buf[277] = (unsigned long) ZeroLevelCH3_Main_bak;
	buf[278] = (unsigned long) ZeroLevelCH4_Main_bak;

	buf[279] = (unsigned long) Virtual_ZeroLevelCH1_Main_bak;
	buf[280] = (unsigned long) Virtual_ZeroLevelCH2_Main_bak;
	buf[281] = (unsigned long) Virtual_ZeroLevelCH3_Main_bak;
	buf[282] = (unsigned long) Virtual_ZeroLevelCH4_Main_bak;

	// BF store registervalues	#001
	buf[283] = (unsigned long) channel_Adr_add12;
	buf[284] = (unsigned long) channel_Adr_add34;
	buf[285] = (unsigned long) adc_change12_reg;
	buf[286] = (unsigned long) adc_change34_reg;
	buf[287] = (unsigned long) adc_ctr12_reg;
	buf[288] = (unsigned long) adc_ctr34_reg;

	buf[289] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][0];			// grid color
	buf[290] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][1];			// grid layout
	buf[291] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][2];			// status layout
	//buf[292] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][3];			// not used
	//buf[293] = (unsigned long) MenuStatus[MENU_DISPL_SETUP][4];			// not used
	//buf[294] = (unsigned long)MenuStatus[MENU_DISPL_SETUP][5];			// not used

	buf[295] = (unsigned long) thres_low[4];					// QM thresholds;
	buf[296] = (unsigned long) thres_mdl[4];					// QM thresholds;
	buf[297] = (unsigned long) thres_up[4];						// QM thresholds;
	buf[298] = (unsigned long) thres_type[4];					// QM threshold type

	buf[299] = 0x23232323;

//BF #010{
	if (-1 == nr_flash_write_buffer((unsigned long *) nasys_main_flash, SectorAddress, buf, 300))		// Flash Buffer
	{ lWriteError = true; }

	//Check values
	if ((SectorAddress[1] == 0xAACC5523) && (SectorAddress[2] == 0x22334455) && (SectorAddress[299] == 0x23232323))
	{
		//printf("Configuration written to flash config sector\r\n");
	}
	else
	{ printf("Flash checksum error\r\n"); lWriteError = true;}


	if(config_changed == 2 || lWriteError == true)
	{	
		//write backup
		SectorAddress = (unsigned long) nasys_main_flash + ConfBackup_Flash;				// get config backup start address
		
		nr_flash_erase_sector((unsigned long *) nasys_main_flash, SectorAddress);			// erase config area 
		
		nr_flash_write_buffer((unsigned long *) nasys_main_flash, SectorAddress, buf, 300);		// Flash Buffer
	
		//Check values
		if ((SectorAddress[1] == 0xAACC5523) && (SectorAddress[2] == 0x22334455) && (SectorAddress[299] == 0x23232323))
		{
			//printf("Configuration written to flash backup sector\r\n");
		}
		else
		printf("Flash checksum error\r\n");
	}


	config_changed = 0;
	

}
//#####################################################################################################################################################
// load configuration from flash
void AMDFlash::Read_Config_Flash(void)
{
//	int x;
	unsigned long *SectorAddress;
	unsigned long buf[300];
	
	for(int lbackup=0;lbackup<2;lbackup++)	//try only two times
	{
		if (lbackup)
		{SectorAddress = (unsigned long) nasys_main_flash + ConfBackup_Flash;}		// get config backup start address
		else
		{SectorAddress = (unsigned long) nasys_main_flash + Config_Flash;}		// get config start address
		
		for (int x = 0; x < 300; x++) buf[x] = SectorAddress[x];
		
		if ((buf[4] == tc_version) && (buf[5] == tc_subversion) && (buf[6] == tc_dev_version) && (buf[1] == 0xAACC5523) && (buf[2] == 0x22334455) && (buf[299] == 0x23232323))
		{
	
			//printf("AMDFlash::Read_Config_Flash\r\n");
	
			FlashPreset = (unsigned char) buf[9];
	
			// Timebase      	
			MenuStatus[MENU_TIMEBASE][0] = (short) buf[10];
			MenuStatus[MENU_TIMEBASE][1] = (short) buf[11];
			MenuStatus[MENU_TIMEBASE][2] = (short) buf[12];
			MenuStatus[MENU_TIMEBASE][3] = (short) buf[13];
			MenuStatus[MENU_TIMEBASE][4] = (short) buf[14];
	
			MenuStatus[MENU_DISPLAY][2] = (short) buf[15];
	
			XY_Mode = (unsigned char) buf[16];
			MainTimebase = (char) buf[17];
			USTB_Mode = (short) buf[18];
			USTB_Mode_bak = (short) buf[19];
			USTB_Dir  = (short) buf[20];
			USTB_Disp = (short) buf[21];
		
			DelayedTimebase = (char) buf[22];
			
			MenuStatus[MENU_UTILITY][3] = (short) buf[23];					// calibration set
			CalSet = MenuStatus[MENU_UTILITY][3] - 264;

			dmode_Window_Offset_Pos = (int) buf[24];
			MenuStatus[MENU_USTB][2] = (short) buf[25];					//USTB buffer size
	
			New_Menu    = (char) buf[26];
			Active_Menu = (char) buf[27];
			MenuAktive  = (char) buf[28];
	
			MenuStatus[MENU_USTB][0] = (short) buf[29];					//ultra slow timebase mode
			MenuStatus[MENU_USTB][1] = (short) buf[30];
	      		MenuStatus[MENU_TRIGGERSUB][0] = (short) buf[31];				// Trigger LED 1
	      		MenuStatus[MENU_TRIGGERSUB][1] = (short) buf[32];				// Trigger LED 2
			//free = (int) buf[33];
			SwitchesTB = (unsigned short) buf[34];
		
			// Trigger	
			MenuStatus[MENU_TRIGGERMODE][0] = (short) buf[35];				// Triggering - Auto
			MenuStatus[MENU_TRIGGERMODE][1] = (short) buf[36];				// Triggering - Coupling
			MenuStatus[MENU_TRIGGERSUB][2]  = (short) buf[37];				// Trigger sub menu LED setting Ch 3+4
			MenuStatus[MENU_TRIGGERMODE][4] = (short) buf[38];				// Triggering - External Probe = 1 : 1 
			MenuStatus[MENU_TRIGGEREDGE][3] = (short) buf[39];				// Triggering - TV
			MenuStatus[MENU_TRIGGEREDGE][0] = (short) buf[40];				// Triggering - positive edge
			MenuStatus[MENU_TRIGGEREDGE][1] = (short) buf[41];				// Triggering - Source = channel 1
			MenuStatus[MENU_TRIGGEREDGE][2] = (short) buf[42];				// Triggering - no externel triggering
			MenuStatus[MENU_PULSEWIDTH][0] = (short) buf[43];				// Triggering - Source Channel 1
			MenuStatus[MENU_PULSEWIDTH][1] = (short) buf[44];				// Triggering - Negative Pulse
			MenuStatus[MENU_PULSEWIDTH][2] = (short) buf[45];				// Triggering - bigger - smaller - then
			Selected_Trigger_Source = (char) buf[46];					// Selected Trigger Source is Channel 1
			MenuStatus[MENU_TRIGGERSUB][3] = (short) buf[47];  				// Trigger sub menu Single Trigger setting;
			PulseToggle = (char) buf[48];                                              	// Pulse width button selection
			//free = (long) buf[49];
			
			FFT_TriggerMode_bak = (short) buf[50]; 						//BF backup the trigger mode
			Trigger_Pos_CH1 = (int) buf[51];                                          	// Trigger Level Channel 1
			Trigger_Pos_CH2 = (int) buf[52];                                          	// Trigger Level Channel 2
			Trigger_Pos_CH3 = (int) buf[53];                                          	// Trigger Level Channel 3
			Trigger_Pos_CH4 = (int) buf[54];                                          	// Trigger Level Channel 4
			trg_val_reg = (unsigned int) buf[55];                               		// Trigger Value
			trg_hyst_reg = (unsigned int) buf[56];                                          // Trigger Hysterese
			trig_range_reg = (unsigned int) buf[57];                                        // Trigger Range
			trig_holdoff_reg = (unsigned long) buf[58];                                     // Trigger Holdoff
			trig_width_reg = (unsigned int) buf[59];                                        // Trigger Width
			
			ctrl_reg = (unsigned int) buf[60];                                              // Control register
			adc_ctrl_reg = (unsigned int) buf[61];                                          // ADC Control register
			pre_reg = (unsigned int) buf[62];                                               // PreTrigger Value
		//       channel_Adr_add = (unsigned int) buf[63];                                      // Channel Add Value
			MenuStatus[MENU_TRIGGERSUB][4] = (short) buf[64];				// Trigger sub menu auto pre trigger setting
			triggering = (unsigned int) buf[65];                                            // Triggering
			TriggerWay = (char) buf[66];                                               	// Triggerway
			Trigger_Pos_CHE = (int) buf[67];                                                // Trigger extern value
			
			//---------------------------------------------------
			// 		Channel 1
			//---------------------------------------------------
			MenuStatus[1][0] = (short) buf[68];                                         // Channel - Coupling
			MenuStatus[1][1] = (short) buf[69];                                         // Channel - BW Linit
			MenuStatus[1][2] = (short) buf[70];                                         // Channel - Invert
			MenuStatus[1][3] = (short) buf[71];                                         // Channel - Probe
			SwitchesCH1 = (unsigned short) buf[72];                                              // Switches for Channel 1 and for the other the AC bit
			Selected_Voltage_CH1 = (volatile char) buf[73];                                     // Voltage
			Trigger_Pos_CH1_Old = (int) buf[74];
			NoDisplay[0] = (unsigned char) buf[75];
			thres_low[0] = (short) buf[76];					// QM thresholds;
			thres_mdl[0] = (short) buf[77];					// QM thresholds;
			thres_up[0]  = (short) buf[78];					// QM thresholds;
			thres_type[0] = (short) buf[79];				// QM threshold type
			Selected_Voltage_CH1_bak  = (char) buf[80];
			Trigger_Pos_CH1_bak = (int) buf[81]; 
			CH1_DAC_Offset 	     = (int) buf[82];                                           // Channel DAC
			ZeroLevelCH1         = (int) buf[83];                                             // ZeroLevel
			Virtual_ZeroLevelCH1 = (int) buf[84];                                     // Virtual ZeroLevel
			//alpha[0]             = (int) buf[85];                                          // ADC-alpha correction -> BF changed
			//CH1_Zero_Offset = (int) buf[85];                                          // Shift offset when stopped
			
			//---------------------------------------------------
			// 		Channel 2
			//---------------------------------------------------
			MenuStatus[2][0] = (short) buf[86];                                         // Channel - Coupling
			MenuStatus[2][1] = (short) buf[87];                                         // Channel - BW Linit
			MenuStatus[2][2] = (short) buf[88];                                         // Channel - Invert
			MenuStatus[2][3] = (short) buf[89];                                         // Channel - Probe
		
			SwitchesCH2 = (unsigned short) buf[90];                                              // Switches for Channel 2
			Selected_Voltage_CH2 = (volatile char) buf[91];                                     // Voltage
			Trigger_Pos_CH2_Old = (int) buf[92];
			NoDisplay[1] = (unsigned char) buf[93];
			thres_low[1] = (short) buf[94];					// QM thresholds;
			thres_mdl[1] = (short) buf[95];					// QM thresholds;
			thres_up[1]  = (short) buf[96];					// QM thresholds;
			thres_type[1] = (short) buf[97];				// QM threshold type
			Selected_Voltage_CH2_bak  = (char) buf[98];
			Trigger_Pos_CH2_bak = (int) buf[99]; 
			CH2_DAC_Offset       = (int) buf[100];                                           // Channel DAC
			ZeroLevelCH2         = (int) buf[101];                                             // ZeroLevel
			Virtual_ZeroLevelCH2 = (int) buf[102];                                     // Virtual ZeroLevel
			//alpha[1]             = (int) buf[103];                                          // ADC-alpha correction -> BF changed
			//CH2_Zero_Offset = (int) buf[103];                                          // Shift offset when stopped
		
			//---------------------------------------------------
			// 		Channel 3
			//---------------------------------------------------
			MenuStatus[3][0] = (short) buf[104];                                         // Channel - Coupling
			MenuStatus[3][1] = (short) buf[105];                                         // Channel - BW Linit
			MenuStatus[3][2] = (short) buf[106];                                         // Channel - Invert
			MenuStatus[3][3] = (short) buf[107];                                         // Channel - Probe
			SwitchesCH3 = (unsigned short) buf[108];                                              // Switches for Channel 3
			Selected_Voltage_CH3 = (volatile char) buf[109];                                     // Voltage
			Trigger_Pos_CH3_Old = (int) buf[110];
			NoDisplay[2] = (unsigned char)buf[111];
			thres_low[2] = (short) buf[112];					// QM thresholds;
			thres_mdl[2] = (short) buf[113];					// QM thresholds;
			thres_up[2]  = (short) buf[114];					// QM thresholds;
			thres_type[2] = (short) buf[115];					// QM threshold type
			Selected_Voltage_CH3_bak  = (char) buf[116];
			Trigger_Pos_CH3_bak = (int) buf[117];
			CH3_DAC_Offset       = (int) buf[118];                                           // Channel DAC
			ZeroLevelCH3         = (int) buf[119];                                             // ZeroLevel
			Virtual_ZeroLevelCH3 = (int) buf[120];                                     // Virtual ZeroLevel
			//alpha[2]             = (int) buf[121];                                          // ADC-alpha correction -> BF changed
			//CH3_Zero_Offset = (int) buf[121];                                          // Shift offset when stopped
		
			//---------------------------------------------------
			// 		Channel 4
			//---------------------------------------------------
			MenuStatus[4][0] = (short) buf[122];                                         // Channel - Coupling
			MenuStatus[4][1] = (short) buf[123];                                         // Channel - BW Linit
			MenuStatus[4][2] = (short) buf[124];                                         // Channel - Invert
			MenuStatus[4][3] = (short) buf[125];                                         // Channel - Probe
			SwitchesCH4 = (unsigned short) buf[126];                                              // Switches for Channel 4
			Selected_Voltage_CH4 = (volatile char) buf[127];                                     // Voltage
			Trigger_Pos_CH4_Old = (int) buf[128];
			NoDisplay[3] = (unsigned char) buf[129]; 
			thres_low[3] = (short) buf[130];					// QM thresholds;
			thres_mdl[3] = (short) buf[131];					// QM thresholds;
			thres_up[3]  = (short) buf[132];					// QM thresholds;
			thres_type[3] = (short) buf[133];					// QM threshold type
			Selected_Voltage_CH4_bak  = (char) buf[134];
			Trigger_Pos_CH4_bak = (int) buf[135]; 
			CH4_DAC_Offset       = (int) buf[136];                                           // Channel DAC
			ZeroLevelCH4         = (int) buf[137];                                             // ZeroLevel
			Virtual_ZeroLevelCH4 = (int) buf[138];                                     // Virtual ZeroLevel
	
	
			FFT_Layout = (int) buf[139];                                          // FFT layout
	
			Channel_Math_Active = (char) buf[140];                                      	// Math channel
			//MenuStatus[MENU_MATH][0] = (short) buf[141];                                        // scale
			MenuStatus[MENU_MATH][2] = (short) buf[142];                                        // math 1*2
			MenuStatus[MENU_MATH][3] = (short) buf[143];                                        // math 1-2
			MenuStatus[MENU_MATH][4] = (short) buf[144];                                        // math 1+2
	
			//FFT Settings
			FFT_Mode   = (int)buf[145];
			FFT_Length = (int)buf[146];
			FFT_Level  = (int)buf[147];
			FFT_Windex = (int)buf[148];
//BF not used			FFT_Scale  = (int)buf[149];
	
			MenuStatus[MENU_FFT][0] = (short)buf[150]; 
			MenuStatus[MENU_FFT][1] = (short)buf[151];
			MenuStatus[MENU_FFT][2] = (short)buf[152];
			MenuStatus[MENU_FFT][3] = (short)buf[153];
	
			// Cursors    	
			MenuStatus[16][0] = (short) buf[154];                                        // Cursor source
			Cursor_Enabled = (char) buf[155];
		
			if (Cursor_Enabled) Cursor_Data_First_Draw = 1;
		
			Cursor_Both_Selected = (char) buf[156];
			SelectedCursor = (char) buf[157];

			Cursor_Horizontal_Active = (char) buf[158];
			Cursor_Horizontal_Position_Real1 = (long) buf[159];
			Cursor_Horizontal_Position_Real2 = (long) buf[160];

			Cursor_Vertical_Active = (char) buf[161];
			Cursor_Vertical_Position_TY_1 = (long) buf[162];
			Cursor_Vertical_Position_TY_2 = (long) buf[163];
		
			// Quick Measure
			QM_Enabled = (char) buf[164];
		
			if (QM_Enabled) Quick_Measure_First_Draw = 1;
		
			QM_Type[0] = (char) buf[165];
			QM_Type[1] = (char) buf[166];
			QM_Type[2] = (char) buf[167];
			QM_Channel[0] = (char) buf[168];
			QM_Channel[1] = (char) buf[169];
			QM_Channel[2] = (char) buf[170];
			QM_Horizontal_Active = (char) buf[171];
			QM_Vertical_Active = (char) buf[172];
			Quick_Measure_Threshold_btn_select = (char) buf[173];
		
			// Display
			GridColorArrayPos = (unsigned char) buf[174];
			GridColor_Val     = (unsigned char) buf[175];
			MenuStatus[MENU_DISPLAY][3] = (short) buf[176];				// no display
			dotted_grid       = (unsigned char) buf[177];
			MenuStatus[MENU_DISPLAY][0] = (short) buf[178];                              // Display - Persist
			MenuStatus[MENU_DISPLAY][4] = (short) buf[179];                              // Display - Draw Mode
		
			MenuStatus[MENU_ACQUIRE][0] = (short) buf[180];                        	// Acquire - Logic
			MenuStatus[MENU_ACQUIRE][1] = (short) buf[181];                     	// Acquire - Averaging
			MenuStatus[MENU_ACQUIRE][2] = (short) buf[183];                   	// Acquire - Filter Setting
			MenuStatus[MENU_ACQUIRE][3] = (short) buf[184];                   	// Acquire - Interpolation
		
			MenuStatus[MENU_AUTOSCALE][3] = (short) buf[185];			// Autoscale - Setup TB search
			MenuStatus[MENU_HARDWARE][1] = (short) buf[186]; 			//BF gain for the input stage
		
			MenuPopupStatus[8][0] = (short) buf[187];				//BF popup for external trigger
			MenuPopupStatus[8][1] = (short) buf[188];
			MenuPopupStatus[8][2] = (short) buf[189];
	
			MenuPopupStatus[11][0] = (short) buf[190];				//BF popup for TV trigger
			MenuPopupStatus[11][1] = (short) buf[191];
	
			//(unsigned char) buf[192];						//BF reserved for LED settings -> see below
	
			Channel_1_Active = (char) buf[193];
			Channel_2_Active = (char) buf[194];
			Channel_3_Active = (char) buf[195];
			Channel_4_Active = (char) buf[196];
	
			FFT_grid = (unsigned char) buf[197];
	
			ColorPalette = (volatile short) buf[198];
		
			HoldOff_Value = (int) buf[199];
			HoldOff_Expo = (int) buf[200];
			HoldOff_Steps = (int) buf[201];
		
			Pulse11_Value = (int) buf[202];
			Pulse11_Expo = (int) buf[203];
			Pulse11_Steps = (int) buf[204];
		
			Pulse12_Value = (int) buf[205];
			Pulse12_Expo = (int) buf[206];
			Pulse12_Steps = (int) buf[207];
		
			Pulse21_Value = (int) buf[208];
			Pulse21_Expo = (int) buf[209];
			Pulse21_Steps = (int) buf[210];
		
			Pulse22_Value = (int) buf[211];
			Pulse22_Expo = (int) buf[212];
			Pulse22_Steps = (int) buf[213];
	
	//BF new Math values
			Math_Offset     = (int) buf[214]; 
			Math_Mul_Offset = (int) buf[215]; 
			Math_Sub_Offset = (int) buf[216];
			Math_Add_Offset = (int) buf[217];
			math_scale      = (int) buf[218];
			math_mul_scale  = (int) buf[219];
			math_sub_scale  = (int) buf[220];
			math_add_scale  = (int) buf[221];
	//BF end
	
	
			MenuStatus[MENU_EXTUTILS][0] = (short) buf[222];	//BF #001
			MenuStatus[MENU_EXTUTILS][1] = (short) buf[223];	//BF #001
			MenuStatus[MENU_EXTUTILS][2] = (short) buf[224];	//BF #001
			MenuStatus[MENU_EXTUTILS][3] = (short) buf[225];	//BF #001
			MenuStatus[MENU_HARDWARE][0] = (short) buf[226];	//BF #002
	
			Trig_Pos_Mem = (int) buf[229];
			Trig_Pos_Display = (int) buf[230];
			//not used = (float) buf[231];
			MemWinStart = (int) buf[232];
		
			Trig_Pos_Mem_old = (int) buf[233];
			Trig_Pos_Display_old = (int) buf[234];
			MenuStatus[MENU_QUICKMEASURE][4] = (short) buf[235];	
			MemWinStart_old = (int) buf[236];
		
			ZoomFactor        = (float) buf[237];
			ZoomFactor_old    = (float) buf[238];
			ZoomFactorDel     = (float) buf[239];

			MenuStatus[MENU_QUICKMEASURE][0] = (short) buf[240];	
			MenuStatus[MENU_QUICKMEASURE][1] = (short) buf[241];
			MenuStatus[MENU_QUICKMEASURE][2] = (short) buf[242];
		
			CH1_Del_Correct = (char) buf[243];	// BF #001
			CH2_Del_Correct = (char) buf[244];	// BF #001
			CH3_Del_Correct = (char) buf[245];	// BF #001
			CH4_Del_Correct = (char) buf[246];	// BF #001
		
			QP_mode = (unsigned char) buf[247];	//BF #006
	
			Virtual_ZeroLevelXYCH1 = (int)buf[248];
			Virtual_ZeroLevelXYCH2 = (int)buf[249];
			Virtual_ZeroLevelXYCH3 = (int)buf[250];
			Virtual_ZeroLevelXYCH4 = (int)buf[251];
	
	
			//BF backup values must be restored too
			ZeroLevelCH1_bak = (int)buf[252];
			ZeroLevelCH2_bak = (int)buf[253];
			ZeroLevelCH3_bak = (int)buf[254];
			ZeroLevelCH4_bak = (int)buf[255];
	
			Virtual_ZeroLevelCH1_bak = (int)buf[256];
			Virtual_ZeroLevelCH2_bak = (int)buf[257];
			Virtual_ZeroLevelCH3_bak = (int)buf[258];
			Virtual_ZeroLevelCH4_bak = (int)buf[259];
	
			TriggerMode_bak = (int)buf[260];
	
			Cursor_Vertical_Position_XY_1 = (long)buf[261];
			Cursor_Vertical_Position_XY_2 = (long)buf[262];
			Cursor_Vertical_Position_FFT_1 = (long)buf[263];
			Cursor_Vertical_Position_FFT_2 = (long)buf[264];
			SavePointer = (char)buf[265];

			Channel_1_Active_bak = (char)buf[266];
			Channel_2_Active_bak = (char)buf[267];
			Channel_3_Active_bak = (char)buf[268];
			Channel_4_Active_bak = (char)buf[269];
			Channel_Math_Active_bak = (char)buf[270];
	
			DrawMono = (unsigned char)buf[271];
			MainTimebase_bak   = (char)buf[272];
			MenuStatus[MENU_QMTHRESHOLDS][0] = (short)buf[273];
			MenuStatus[MENU_QMTHRESHOLDS][1] = (short)buf[274];
	
			ZeroLevelCH1_Main_bak = (int)buf[275];
			ZeroLevelCH2_Main_bak = (int)buf[276];
			ZeroLevelCH3_Main_bak = (int)buf[277];
			ZeroLevelCH4_Main_bak = (int)buf[278];
	
			Virtual_ZeroLevelCH1_Main_bak = (int)buf[279];
			Virtual_ZeroLevelCH2_Main_bak = (int)buf[280];
			Virtual_ZeroLevelCH3_Main_bak = (int)buf[281];
			Virtual_ZeroLevelCH4_Main_bak = (int)buf[282];
	
			// BF get ADC registervalues #002
			channel_Adr_add12 = buf[283];
			channel_Adr_add34 = buf[284];
			adc_change12_reg = buf[285];
			adc_change34_reg = buf[286];
			adc_ctr12_reg    = buf[287];
			adc_ctr34_reg    = buf[288];

			MenuStatus[MENU_DISPL_SETUP][0] = (short) buf[289];			// grid color
			MenuStatus[MENU_DISPL_SETUP][1] = (short) buf[290];			// grid layout
			MenuStatus[MENU_DISPL_SETUP][2] = (short) buf[291];			// status layout
			//MenuStatus[MENU_DISPL_SETUP][3] = (short) buf[292];			// not used
			//MenuStatus[MENU_DISPL_SETUP][4] = (short) buf[293];			// not used
			//MenuStatus[MENU_DISPL_SETUP][5] = (short) buf[294];			// not used

			thres_low[4] = (short) buf[295];					// QM thresholds;
			thres_mdl[4] = (short) buf[296];					// QM thresholds;
			thres_up[4]  = (short) buf[297];					// QM thresholds;
			thres_type[4] = (short) buf[298];					// QM thresholds;
		
			LED_ON[0] = 0;
			LED_ON[1] = 0;
			LED_ON[2] = 0;
			LED_ON[3] = 0;
			LED_ON[4] = 0;
			LED_ON[5] = 0;
			LED_ON[6] = 0;
			LED_ON[7] = 0;
			LED_ON[8] = 0;
			LED_ON[9] = 0;
			LED_ON[10] = 0;
			LED_ON[11] = 0;
			LED_ON[12] = 1;		// Run/Stop Green
			LED_ON[13] = 0;
			LED_ON[14] = 0;
			LED_ON[15] = 0;
		
			if (((int) buf[192] & 0x0001) == 0x0001) LED_ON[0] = 1;
			if (((int) buf[192] & 0x0002) == 0x0002) LED_ON[1] = 1;
			if (((int) buf[192] & 0x0004) == 0x0004) LED_ON[2] = 1;
			if (((int) buf[192] & 0x0008) == 0x0008) LED_ON[3] = 1;
			if (((int) buf[192] & 0x0010) == 0x0010) LED_ON[4] = 1;  	// Math
			if (((int) buf[192] & 0x0020) == 0x0020) LED_ON[5] = 1; 	// Main Wheel
			if (((int) buf[192] & 0x0040) == 0x0040) LED_ON[6] = 1;
			if (((int) buf[192] & 0x0080) == 0x0080) LED_ON[7] = 1; 	// Quick Measure
			if (((int) buf[192] & 0x0100) == 0x0100) LED_ON[8] = 1;
			if (((int) buf[192] & 0x0200) == 0x0200) LED_ON[9] = 1;
			if (((int) buf[192] & 0x0400) == 0x0400) LED_ON[10] = 1;
			if (((int) buf[192] & 0x0800) == 0x0800) LED_ON[11] = 1;
			//if (((int) buf[192] & 0x1000) == 0x1000) LED_ON[12] = 1; 	// Run/Stop Green
			//if (((int) buf[192] & 0x2000) == 0x2000) LED_ON[13] = 1; 	// Run/Stop Red
			//if (((int) buf[192] & 0x4000) == 0x4000) LED_ON[14] = 1;
			//if (((int) buf[192] & 0x8000) == 0x8000) LED_ON[15] = 1;
		
	//BF changed		
			//Math_Offset_mul.Write_StepSize(1);
			Math_Offset_mul.Write_Value(float((float)Math_Mul_Offset /  (float)50000));
			Math_Offset_mul.RenderText();
	
			//Math_Offset_mul.Write_StepSize(1);
			Math_Offset_sub.Write_Value(float((float)Math_Sub_Offset /  (float)50000));
			Math_Offset_sub.RenderText();
	
			//Math_Offset_mul.Write_StepSize(1);
			Math_Offset_add.Write_Value(float((float)Math_Add_Offset /  (float)50000));
			Math_Offset_add.RenderText();
	//BF end
		
			GainIdx = MenuStatus[MENU_HARDWARE][1] - 230;	//BF #004
			ACQ_Average = MenuStatus[MENU_ACQUIRE][1] - 254;	

			config_loaded = true;
/*
			if (lbackup)
			printf("Config loaded from backup sector\r\n");
			else
			printf("Config loaded successfully\r\n");
*/
			break;
	
		}
		else
		{
			if (lbackup)
			{
				config_loaded = false;
				printf("Error - loading config from Flash failed\r\n");
			}
		}
	
	}

	Signal::buildScaleLookupTable();

	if (FFT_Mode != FFT_OFF)
	Signal::FFT_buildScaleLookupTable();


}
//#####################################################################################################################################################
/*BF only needed for test purposes
void AMDFlash::Read_TestData_Flash(void)
{
	int x, cnt;
//printf("AMDFlash::Read_TestData_Flash\n");	
	unsigned long *flash_address;
	unsigned long buf[256];
	unsigned char buf2[1024];
    	
	flash_address = (unsigned long) nasys_main_flash + TestData_Flash;
	
	for (x = 0; x < 256; x++) buf[x] = flash_address[x];
	
	for (x = 0; x < 256; x++)
    {
        buf2[(x * 4) + 0] = (unsigned char) buf[x] & 0x000000FF;
        buf2[(x * 4) + 1] = (unsigned char) ((buf[x] & 0x0000FF00) >> 8);
        buf2[(x * 4) + 2] = (unsigned char) ((buf[x] & 0x00FF0000) >> 16);
        buf2[(x * 4) + 3] = (unsigned char) ((buf[x] & 0xFF000000) >> 24);
    }                                                       	
	
#ifdef _Debug_
    cnt = 0;
    for (x = 0; x < 48; x++)
    {
        printf("%3d %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x %2x \n", cnt, buf2[(cnt * 21) + 0], buf2[(cnt * 21) + 1], buf2[(cnt * 21) + 2], buf2[(cnt * 21) + 3], buf2[(cnt * 21) + 4], buf2[(cnt * 21) + 5], buf2[(cnt * 21) + 6], buf2[(cnt * 21) + 7], buf2[(cnt * 21) + 8], buf2[(cnt * 21) + 9], buf2[(cnt * 21) + 10], buf2[(cnt * 21) + 11], buf2[(cnt * 21) + 12], buf2[(cnt * 21) + 13], buf2[(cnt * 21) + 14], buf2[(cnt * 21) + 15], buf2[(cnt * 21) + 16], buf2[(cnt * 21) + 17], buf2[(cnt * 21) + 18], buf2[(cnt * 21) + 19], buf2[(cnt * 21) + 20]);

        cnt++;
        nr_delay(10);
    }      	
#endif
}

void AMDFlash::Write_TestData_Flash(void)
{
    int x, cnt;
	unsigned long *flash_address;
	unsigned long buf[300];
//printf("AMDFlash::Write_TestData_Flash\n");	
	flash_address = (unsigned long) nasys_main_flash + TestData_Flash;
#ifdef _Debug_	
	printf("Flashadr : %x = %x\n", flash_address, flash_address[0]);
#endif	
	for (x = 0; x < 300; x++) buf[x] = flash_address[x];
	
	nr_flash_erase_sector((unsigned long *) nasys_main_flash, flash_address);

	buf[32] = 0x12345678;
	
	nr_flash_write_buffer((unsigned long *) nasys_main_flash, flash_address, buf, 34);
#ifdef _Debug_	
	if (Debug_Mode) printf("Flash TestData written\n");
#endif	
}
*/

//#####################################################################################################################################################
// BF used in CommIF::InterpretUART() -> USB programming interface
int AMDFlash::WriteData_Byte_Flash(unsigned long *flash_address, unsigned char add, unsigned char data)
{
    int result = 0;
    int errors = 100;
    unsigned char* fb       = (unsigned char *) nasys_main_flash;
    unsigned char* a        = (unsigned char *) flash_address;
//printf("AMDFlash::WriteData_Byte_Flash\n");

//    result = amd29lv065d_flash_write_byte(fb, a + add, data);

    while (errors)
    {
	   unlock_bypass_mode(fb);
	   result = nr_flash_write_bypass(fb, a + add, data);
	   reset_bypass_mode(fb);
	
	   if (result != -1) errors = 0;
    }
	
    return result;	
}

//#####################################################################################################################################################
// BF -> used in CommIF::InterpretUART()
void AMDFlash::ReadData_Prepare_Flash(unsigned long *flash_address)
{
	int x, cnt;
	
	unsigned long *flash_adr;
	unsigned char result = 0;
	//printf("AMDFlash::ReadData_Prepare_Flash\n");    	
	flash_adr = (unsigned long) nasys_main_flash + flash_address;
	
	for (x = 0; x < 256; x++)
	{
		Flash_Read_Buffer[(x * 4) + 0] = (unsigned char) flash_adr[x] & 0x000000FF;
		Flash_Read_Buffer[(x * 4) + 1] = (unsigned char) ((flash_adr[x] & 0x0000FF00) >> 8);
		Flash_Read_Buffer[(x * 4) + 2] = (unsigned char) ((flash_adr[x] & 0x00FF0000) >> 16);
		Flash_Read_Buffer[(x * 4) + 3] = (unsigned char) ((flash_adr[x] & 0xFF000000) >> 24);
	//        nr_delay(5);
	}
}

// BF -> used in CommIF::Interpret UART()
unsigned char AMDFlash::ReadData_Byte_Flash(unsigned int add)
{
	unsigned char result = 0;
//printf("AMDFlash::ReadData_Byte_Flash\n");
    result = Flash_Read_Buffer[add];

    return result;
}
//#####################################################################################################################################################
void AMDFlash::Read_Protected_Flash(void)
{
	int x;
	
	unsigned long *SectorAddress;
 	
	//printf("AMDFlash::Read_Protected_Flash\r\n");   	

	for(int lbackup=0;lbackup<2;lbackup++)	//try only two times
	{
		if (lbackup)
		{SectorAddress = (unsigned long) nasys_main_flash + ProtBackup_Flash;}			// get config backup start address
		else
		{SectorAddress = (unsigned long) nasys_main_flash + Protected_Config_Flash;}		// get config start address
		
		// read flash to global buffer
		for (x = 0; x < 256; x++)
		{ Flash_Protect_Buffer[x] = SectorAddress[x]; }
		
		if (Flash_Protect_Buffer[0] == 0xFF2332FF)
		{
			tc_model           = Flash_Protect_Buffer[1];
			tc_serial          = Flash_Protect_Buffer[2];
	
			tc_production_lot1 = Flash_Protect_Buffer[3];
			tc_production_lot2 = Flash_Protect_Buffer[4];
			tc_shipment_date   = Flash_Protect_Buffer[5];

			//Standard Set
			DAC_Correction[0][0][0] = (int)Flash_Protect_Buffer[9]; 	//channel 1
			DAC_Correction[0][0][1] = (int)Flash_Protect_Buffer[10]; 
			DAC_Correction[0][0][2] = (int)Flash_Protect_Buffer[11]; 
	
			DAC_Correction[0][1][0] = (int)Flash_Protect_Buffer[12]; 	//channel 2
			DAC_Correction[0][1][1] = (int)Flash_Protect_Buffer[13]; 
			DAC_Correction[0][1][2] = (int)Flash_Protect_Buffer[14]; 
	
			DAC_Correction[0][2][0] = (int)Flash_Protect_Buffer[15]; 	//channel 3
			DAC_Correction[0][2][1] = (int)Flash_Protect_Buffer[16]; 
			DAC_Correction[0][2][2] = (int)Flash_Protect_Buffer[17]; 
	
			DAC_Correction[0][3][0] = (int)Flash_Protect_Buffer[18]; 	//channel 4
			DAC_Correction[0][3][1] = (int)Flash_Protect_Buffer[19]; 
			DAC_Correction[0][3][2] = (int)Flash_Protect_Buffer[20]; 

			//Active Probe Set
			DAC_Correction[1][0][0] = (int)Flash_Protect_Buffer[21]; 	//channel 1
			DAC_Correction[1][0][1] = (int)Flash_Protect_Buffer[22]; 
			DAC_Correction[1][0][2] = (int)Flash_Protect_Buffer[23]; 
	
			DAC_Correction[1][1][0] = (int)Flash_Protect_Buffer[24]; 	//channel 2
			DAC_Correction[1][1][1] = (int)Flash_Protect_Buffer[25]; 
			DAC_Correction[1][1][2] = (int)Flash_Protect_Buffer[26]; 
	
			DAC_Correction[1][2][0] = (int)Flash_Protect_Buffer[27]; 	//channel 3
			DAC_Correction[1][2][1] = (int)Flash_Protect_Buffer[28]; 
			DAC_Correction[1][2][2] = (int)Flash_Protect_Buffer[29]; 
	
			DAC_Correction[1][3][0] = (int)Flash_Protect_Buffer[30]; 	//channel 4
			DAC_Correction[1][3][1] = (int)Flash_Protect_Buffer[31]; 
			DAC_Correction[1][3][2] = (int)Flash_Protect_Buffer[32]; 

			//Set3
			DAC_Correction[2][0][0] = (int)Flash_Protect_Buffer[33]; 	//channel 1
			DAC_Correction[2][0][1] = (int)Flash_Protect_Buffer[34]; 
			DAC_Correction[2][0][2] = (int)Flash_Protect_Buffer[35]; 
	
			DAC_Correction[2][1][0] = (int)Flash_Protect_Buffer[36]; 	//channel 2
			DAC_Correction[2][1][1] = (int)Flash_Protect_Buffer[37]; 
			DAC_Correction[2][1][2] = (int)Flash_Protect_Buffer[38]; 
	
			DAC_Correction[2][2][0] = (int)Flash_Protect_Buffer[39]; 	//channel 3
			DAC_Correction[2][2][1] = (int)Flash_Protect_Buffer[40]; 
			DAC_Correction[2][2][2] = (int)Flash_Protect_Buffer[41]; 
	
			DAC_Correction[2][3][0] = (int)Flash_Protect_Buffer[42]; 	//channel 4
			DAC_Correction[2][3][1] = (int)Flash_Protect_Buffer[43]; 
			DAC_Correction[2][3][2] = (int)Flash_Protect_Buffer[44]; 

			//Set4
			DAC_Correction[3][0][0] = (int)Flash_Protect_Buffer[45]; 	//channel 1
			DAC_Correction[3][0][1] = (int)Flash_Protect_Buffer[46]; 
			DAC_Correction[3][0][2] = (int)Flash_Protect_Buffer[47]; 
	
			DAC_Correction[3][1][0] = (int)Flash_Protect_Buffer[48]; 	//channel 2
			DAC_Correction[3][1][1] = (int)Flash_Protect_Buffer[49]; 
			DAC_Correction[3][1][2] = (int)Flash_Protect_Buffer[50]; 
	
			DAC_Correction[3][2][0] = (int)Flash_Protect_Buffer[51]; 	//channel 3
			DAC_Correction[3][2][1] = (int)Flash_Protect_Buffer[52]; 
			DAC_Correction[3][2][2] = (int)Flash_Protect_Buffer[53]; 
	
			DAC_Correction[3][3][0] = (int)Flash_Protect_Buffer[54]; 	//channel 4
			DAC_Correction[3][3][1] = (int)Flash_Protect_Buffer[55]; 
			DAC_Correction[3][3][2] = (int)Flash_Protect_Buffer[56]; 
	
	
	
	
			ADC_Offset[0][0] = (int) Flash_Protect_Buffer[57];
			ADC_Offset[0][1] = (int) Flash_Protect_Buffer[58];
			ADC_Offset[0][2] = (int) Flash_Protect_Buffer[59];
			ADC_Offset[0][3] = (int) Flash_Protect_Buffer[60];	
			
			ADC_Offset[1][0] = (int) Flash_Protect_Buffer[61];
			ADC_Offset[1][1] = (int) Flash_Protect_Buffer[62];
			ADC_Offset[1][2] = (int) Flash_Protect_Buffer[63];
			ADC_Offset[1][3] = (int) Flash_Protect_Buffer[64];
		
			ADC_Offset[2][0] = (int) Flash_Protect_Buffer[65];
			ADC_Offset[2][1] = (int) Flash_Protect_Buffer[66];
			ADC_Offset[2][2] = (int) Flash_Protect_Buffer[67];
			ADC_Offset[2][3] = (int) Flash_Protect_Buffer[68];
		
			ADC_Offset[3][0] = (int) Flash_Protect_Buffer[69];
			ADC_Offset[3][1] = (int) Flash_Protect_Buffer[70];
			ADC_Offset[3][2] = (int) Flash_Protect_Buffer[71];
			ADC_Offset[3][3] = (int) Flash_Protect_Buffer[72];

// 73 - 77 is reserved memory for factory register settings don't use!!!
		
	//BF del		adc_change12_reg = Flash_Protect_Buffer[73];			//BF Value is 0x02020800
	//BF del		adc_ctr12_reg    = 0;	//Flash_Protect_Buffer[74];			//BF Value is 0x00000000
	//BF del		adc_change34_reg = Flash_Protect_Buffer[75];			//BF Value is 0x00000000	
	//BF del		channel_Adr_add  = 0x5F0A;	//Flash_Protect_Buffer[76];		//BF Value is 0x5F0A
	//BF del		channel_Adr_add2 = 0;		//Flash_Protect_Buffer[77];		//BF Value is 0x0000
	
	/*
			printf("\r\n");
			printf("----------------------------------------------------------------\r\n");
			printf("- protected register settings -\r\n");
			printf("adc_change12_reg : %x \r\n", adc_change12_reg);
			printf("adc_change34_reg : %x \r\n", adc_change34_reg);
			printf("adc_ctr12_reg    : %x \r\n", adc_ctr12_reg);
			printf("channel_Adr_add12  : %x \r\n", channel_Adr_add12);
			printf("channel_Adr_add34 : %x \r\n", channel_Adr_add34);
			printf("-----------------------------------------------------------------\r\n");
			printf("\r\n");
	*/
	
/*	
			if (lbackup)
			printf("protected config loaded from backup sector\r\n");
			else
			printf("protected config loaded successfully\r\n");
*/	
			break;
	
		}
		else
		{
			if (lbackup)
			{ printf("Error - loading protected config from flash failed\r\n"); }
		}
	
		
	}

}
//#####################################################################################################################################################
void AMDFlash::Write_Protected_Flash(void)
{
   	int x, cnt;
	unsigned long *SectorAddress;
	
	//printf("AMDFlash::Write_Protected_Flash()\r\n");	
	
	SectorAddress = (unsigned long) nasys_main_flash + Protected_Config_Flash;		// get config start address
	
	nr_flash_erase_sector((unsigned long *) nasys_main_flash, SectorAddress);	// erase config area 

	Flash_Protect_Buffer[0] = 0xFF2332FF;              // Kennung
	//Flash_Protect_Buffer[1] = tc_model;                // model 			BF del
	//Flash_Protect_Buffer[2] = tc_serial;               // serial nr		BF del
	Flash_Protect_Buffer[3] = tc_production_lot1;      // production lot	
	Flash_Protect_Buffer[4] = tc_production_lot2;      // production lot	
	Flash_Protect_Buffer[5] = tc_shipment_date;        // shipment date

	//Standard Set
	Flash_Protect_Buffer[9] = (unsigned long) DAC_Correction[0][0][0]; 	//channel 1
	Flash_Protect_Buffer[10] = (unsigned long) DAC_Correction[0][0][1]; 
	Flash_Protect_Buffer[11] = (unsigned long) DAC_Correction[0][0][2]; 

	Flash_Protect_Buffer[12] = (unsigned long) DAC_Correction[0][1][0]; 	//channel 2
	Flash_Protect_Buffer[13] = (unsigned long) DAC_Correction[0][1][1]; 
	Flash_Protect_Buffer[14] = (unsigned long) DAC_Correction[0][1][2]; 

	Flash_Protect_Buffer[15] = (unsigned long) DAC_Correction[0][2][0]; 	//channel 3
	Flash_Protect_Buffer[16] = (unsigned long) DAC_Correction[0][2][1]; 
	Flash_Protect_Buffer[17] = (unsigned long) DAC_Correction[0][2][2]; 

	Flash_Protect_Buffer[18] = (unsigned long) DAC_Correction[0][3][0]; 	//channel 4
	Flash_Protect_Buffer[19] = (unsigned long) DAC_Correction[0][3][1]; 
	Flash_Protect_Buffer[20] = (unsigned long) DAC_Correction[0][3][2]; 

	//Active Probe Set
	Flash_Protect_Buffer[21] = (unsigned long) DAC_Correction[1][0][0]; 	//channel 1
	Flash_Protect_Buffer[22] = (unsigned long) DAC_Correction[1][0][1]; 
	Flash_Protect_Buffer[23] = (unsigned long) DAC_Correction[1][0][2]; 

	Flash_Protect_Buffer[24] = (unsigned long) DAC_Correction[1][1][0]; 	//channel 2
	Flash_Protect_Buffer[25] = (unsigned long) DAC_Correction[1][1][1]; 
	Flash_Protect_Buffer[26] = (unsigned long) DAC_Correction[1][1][2]; 

	Flash_Protect_Buffer[27] = (unsigned long) DAC_Correction[1][2][0]; 	//channel 3
	Flash_Protect_Buffer[28] = (unsigned long) DAC_Correction[1][2][1]; 
	Flash_Protect_Buffer[29] = (unsigned long) DAC_Correction[1][2][2]; 

	Flash_Protect_Buffer[30] = (unsigned long) DAC_Correction[1][3][0]; 	//channel 4
	Flash_Protect_Buffer[31] = (unsigned long) DAC_Correction[1][3][1]; 
	Flash_Protect_Buffer[32] = (unsigned long) DAC_Correction[1][3][2]; 

	//Set 3
	Flash_Protect_Buffer[33] = (unsigned long) DAC_Correction[2][0][0]; 	//channel 1
	Flash_Protect_Buffer[34] = (unsigned long) DAC_Correction[2][0][1]; 
	Flash_Protect_Buffer[35] = (unsigned long) DAC_Correction[2][0][2]; 

	Flash_Protect_Buffer[36] = (unsigned long) DAC_Correction[2][1][0]; 	//channel 2
	Flash_Protect_Buffer[37] = (unsigned long) DAC_Correction[2][1][1]; 
	Flash_Protect_Buffer[38] = (unsigned long) DAC_Correction[2][1][2]; 

	Flash_Protect_Buffer[39] = (unsigned long) DAC_Correction[2][2][0]; 	//channel 3
	Flash_Protect_Buffer[40] = (unsigned long) DAC_Correction[2][2][1]; 
	Flash_Protect_Buffer[41] = (unsigned long) DAC_Correction[2][2][2]; 

	Flash_Protect_Buffer[42] = (unsigned long) DAC_Correction[2][3][0]; 	//channel 4
	Flash_Protect_Buffer[43] = (unsigned long) DAC_Correction[2][3][1]; 
	Flash_Protect_Buffer[44] = (unsigned long) DAC_Correction[2][3][2]; 

	//Set 4
	Flash_Protect_Buffer[45] = (unsigned long) DAC_Correction[3][0][0]; 	//channel 1
	Flash_Protect_Buffer[46] = (unsigned long) DAC_Correction[3][0][1]; 
	Flash_Protect_Buffer[47] = (unsigned long) DAC_Correction[3][0][2]; 

	Flash_Protect_Buffer[48] = (unsigned long) DAC_Correction[3][1][0]; 	//channel 2
	Flash_Protect_Buffer[49] = (unsigned long) DAC_Correction[3][1][1]; 
	Flash_Protect_Buffer[50] = (unsigned long) DAC_Correction[3][1][2]; 

	Flash_Protect_Buffer[51] = (unsigned long) DAC_Correction[3][2][0]; 	//channel 3
	Flash_Protect_Buffer[52] = (unsigned long) DAC_Correction[3][2][1]; 
	Flash_Protect_Buffer[53] = (unsigned long) DAC_Correction[3][2][2]; 

	Flash_Protect_Buffer[54] = (unsigned long) DAC_Correction[3][3][0]; 	//channel 4
	Flash_Protect_Buffer[55] = (unsigned long) DAC_Correction[3][3][1]; 
	Flash_Protect_Buffer[56] = (unsigned long) DAC_Correction[3][3][2]; 



	Flash_Protect_Buffer[57] = (unsigned long) ADC_Offset[0][0];		//channel 1
	Flash_Protect_Buffer[58] = (unsigned long) ADC_Offset[0][1];
	Flash_Protect_Buffer[59] = (unsigned long) ADC_Offset[0][2];
	Flash_Protect_Buffer[60] = (unsigned long) ADC_Offset[0][3];	
	
	Flash_Protect_Buffer[61] = (unsigned long) ADC_Offset[1][0];		//channel 2
	Flash_Protect_Buffer[62] = (unsigned long) ADC_Offset[1][1];
	Flash_Protect_Buffer[63] = (unsigned long) ADC_Offset[1][2];
	Flash_Protect_Buffer[64] = (unsigned long) ADC_Offset[1][3];	

	Flash_Protect_Buffer[65] = (unsigned long) ADC_Offset[2][0];		//channel 3
	Flash_Protect_Buffer[66] = (unsigned long) ADC_Offset[2][1];
	Flash_Protect_Buffer[67] = (unsigned long) ADC_Offset[2][2];
	Flash_Protect_Buffer[68] = (unsigned long) ADC_Offset[2][3];	

	Flash_Protect_Buffer[69] = (unsigned long) ADC_Offset[3][0];		//channel 4
	Flash_Protect_Buffer[70] = (unsigned long) ADC_Offset[3][1];
	Flash_Protect_Buffer[71] = (unsigned long) ADC_Offset[3][2];
	Flash_Protect_Buffer[72] = (unsigned long) ADC_Offset[3][3];	

// 73 - 77 is reserved memory for factory register settings don't use!!!

/* BF buffer must be written unchanged to flash to prevent loosing factory setting!
	Flash_Protect_Buffer[73] = (unsigned long) 0x02020F00;	//0x02020800;	//adc_change12_reg;  	
	Flash_Protect_Buffer[74] = (unsigned long) 0x0000;			//adc_ctr12_reg;
	Flash_Protect_Buffer[75] = (unsigned long) 0x0200000A;	//0x02020800;	//adc_change34_reg;
	Flash_Protect_Buffer[76] = (unsigned long) 0x5F0A;			//channel_Adr_add;
	Flash_Protect_Buffer[77] = (unsigned long) 0x5F5F			//channel_Adr_add2;
*/

	nr_flash_write_buffer((unsigned long *) nasys_main_flash, SectorAddress, Flash_Protect_Buffer, 80);	// Flash Buffer

	// check values
	if (SectorAddress[0] == 0xFF2332FF)
	printf("protected config written to flash\r\n");
	else
	printf("flash checksum error\r\n");


	//Write backup
	SectorAddress = (unsigned long) nasys_main_flash + ProtBackup_Flash;						// get config backup start address

	nr_flash_erase_sector((unsigned long *) nasys_main_flash, SectorAddress);					// erase backup area 

	nr_flash_write_buffer((unsigned long *) nasys_main_flash, SectorAddress, Flash_Protect_Buffer, 80);		// Flash Buffer

	// check values
	if (SectorAddress[0] == 0xFF2332FF)
	printf("protected config written to flash backup sector\r\n");
	else
	printf("backup flash checksum error\r\n");


}

//#####################################################################################################################################################
//BF #002 read protected register values to get factory setting
void AMDFlash::ReadProtReg(void)
{
	int x;
	unsigned long *SectorAddress;

	//Check if protected buffer is already loaded	
	if (Flash_Protect_Buffer[0] == 0xFF2332FF)
	{
		adc_change12_reg  = Flash_Protect_Buffer[73];
		adc_ctr12_reg     = Flash_Protect_Buffer[74];
		adc_change34_reg  = Flash_Protect_Buffer[75];	
		channel_Adr_add12 = Flash_Protect_Buffer[76];
		channel_Adr_add34 = Flash_Protect_Buffer[77];

		adc_ctr34_reg = adc_ctr12_reg;

	}
	else
	{
		//if not - get values from flash
		SectorAddress = (unsigned long) nasys_main_flash + Protected_Config_Flash;
	
		if (SectorAddress[0] == 0xFF2332FF)
		{
			adc_change12_reg  = SectorAddress[73];
			adc_ctr12_reg     = SectorAddress[74];
			adc_change34_reg  = SectorAddress[75];	
			channel_Adr_add12 = SectorAddress[76];
			channel_Adr_add34 = SectorAddress[77];
	
			adc_ctr34_reg = adc_ctr12_reg;
	
		}
		else
		{ printf("Loading Protected Registers failed\r\n");}


	}


	printf("\r\n");
	printf("----------------------------------------------------------------\r\n");
	printf("- protected register settings -\r\n");
	printf("adc_change12_reg  : %x \r\n", adc_change12_reg);
	printf("adc_change34_reg  : %x \r\n", adc_change34_reg);
	printf("channel_Adr_add12 : %8x \r\n", channel_Adr_add12);
	printf("channel_Adr_add34 : %8x \r\n", channel_Adr_add34);
	printf("adc_ctr12_reg     : %x \r\n", adc_ctr12_reg);
	printf("adc_ctr34_reg     : %x \r\n", adc_ctr34_reg);
	printf("-----------------------------------------------------------------\r\n");
	printf("\r\n");



	
}

//#####################################################################################################################################################
//BF #016 load selected trace from flash
void AMDFlash::RecallTrace(void)
{

	if (RC_overlay) return;	

	if (!RC_request)
	{
		// save actual configuration
		Write_Config_Flash();
	}
	RC_request = 1;
	UI_request = 0;

	VirtualTimebase	= 0;
	Run = 1;	//initialize operating mode
	
	switch(SavePointer)
	{
		case 1 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A01); break;
		case 2 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A02); break;
		case 3 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A03); break;
		case 4 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A04); break;
		case 5 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A05); break;
		case 6 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A06); break;
		case 7 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A07); break;
		case 8 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A08); break;
		case 9 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A09); break;
		case 10 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A10); break;
		case 11 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A11); break;
		case 12 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A12); break;
		case 13 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A13); break;
		case 14 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A14); break;
		case 15 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A15); break;
		case 16 : ReadTraceFromFlash(SAVE_SIGNAL_NR_A16); break;
	
		case 17 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B01); break;
		case 18 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B02); break;
		case 19 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B03); break;
		case 20 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B04); break;
		case 21 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B05); break;
		case 22 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B06); break;
		case 23 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B07); break;
		case 24 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B08); break;
		case 25 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B09); break;
		case 26 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B10); break;
		case 27 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B11); break;
		case 28 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B12); break;
		case 29 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B13); break;
		case 30 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B14); break;
		case 31 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B15); break;
		case 32 : ReadTraceFromFlash(SAVE_SIGNAL_NR_B16); break;
	
		case 33 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C01); break;
		case 34 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C02); break;
		case 35 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C03); break;
		case 36 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C04); break;
		case 37 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C05); break;
		case 38 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C06); break;
		case 39 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C07); break;
		case 40 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C08); break;
		case 41 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C09); break;
		case 42 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C10); break;
		case 43 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C11); break;
		case 44 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C12); break;
		case 45 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C13); break;
		case 46 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C14); break;
		case 47 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C15); break;
		case 48 : ReadTraceFromFlash(SAVE_SIGNAL_NR_C16); break;
	
		case 49 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D01); break;
		case 50 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D02); break;
		case 51 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D03); break;
		case 52 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D04); break;
		case 53 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D05); break;
		case 54 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D06); break;
		case 55 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D07); break;
		case 56 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D08); break;
		case 57 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D09); break;
		case 58 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D10); break;
		case 59 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D11); break;
		case 60 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D12); break;
		case 61 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D13); break;
		case 62 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D14); break;
		case 63 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D15); break;
		case 64 : ReadTraceFromFlash(SAVE_SIGNAL_NR_D16); break;
	
		case 65 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E01); break;
		case 66 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E02); break;
		case 67 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E03); break;
		case 68 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E04); break;
		case 69 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E05); break;
		case 70 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E06); break;
		case 71 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E07); break;
		case 72 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E08); break;
		case 73 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E09); break;
		case 74 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E10); break;
		case 75 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E11); break;
		case 76 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E12); break;
		case 77 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E13); break;
		case 78 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E14); break;
		case 79 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E15); break;
		case 80 : ReadTraceFromFlash(SAVE_SIGNAL_NR_E16); break;
	}

	if (Run)
	{
		RC_request = 0;
	 	return; 	//no data found -> leave
	}

	Display::MenuInit();
	Display::RefreshScreen();
	Display::DRAW_ALL_CURSOR();

	// check if logic processor has to be started
	LA_active = 0;

	if (MenuStatus[MENU_ACQUIRE][0] != LOGA_OFF)
	{
		// restore voltage ranges
		Selected_Voltage_CH1 = Selected_Voltage_CH1_bak;
		Selected_Voltage_CH2 = Selected_Voltage_CH2_bak;
		Selected_Voltage_CH3 = Selected_Voltage_CH3_bak;
		Selected_Voltage_CH4 = Selected_Voltage_CH4_bak;

		// restore trigger settings
		Trigger_Pos_CH1 = Trigger_Pos_CH1_bak;
		Trigger_Pos_CH2 = Trigger_Pos_CH2_bak;
		Trigger_Pos_CH3 = Trigger_Pos_CH3_bak;
		Trigger_Pos_CH4 = Trigger_Pos_CH4_bak;

		Signal::LOGA_LogicProcessor();
	}


}


//#####################################################################################################################################################
//BF #016 save selected trace to flash
void AMDFlash::SaveTrace(void)
{
	if (RC_request)			//don't save signal which is recalled from Memory
	{ RC_request = 0; return; }

	if (RC_overlay)			//don't save signal which is recalled from Memory
	{ 
		printf("Terminating Overlay Mode\r\n");
		RC_overlay = 0; 
		return;		
	}


	//&~np_timercontrol_cont_mask;	// Stop Timer
	timer_rotary_busy->np_timercontrol = ((timer_rotary_busy->np_timercontrol & 3) + np_timercontrol_stop_mask);
	timer_rotary_busy->np_timerstatus = 0;

	Display::DRAWROUNDBUTTON(250, 200, 140, 40, 0, 0);
	Display::TEXTOUTxvbig("Saving Signals", 260, 212, 1, UI_Plane2);

	switch(SavePointer)
	{
		case 1 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A01); break;
		case 2 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A02); break;
		case 3 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A03); break;
		case 4 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A04); break;
		case 5 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A05); break;
		case 6 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A06); break;
		case 7 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A07); break;
		case 8 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A08); break;
		case 9 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A09); break;
		case 10 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A10); break;
		case 11 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A11); break;
		case 12 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A12); break;
		case 13 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A13); break;
		case 14 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A14); break;
		case 15 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A15); break;
		case 16 :	WriteTraceToFlash(SAVE_SIGNAL_NR_A16); break;

		case 17 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B01); break;
		case 18 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B02); break;
		case 19 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B03); break;
		case 20 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B04); break;
		case 21 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B05); break;
		case 22 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B06); break;
		case 23 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B07); break;
		case 24 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B08); break;
		case 25 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B09); break;
		case 26 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B10); break;
		case 27 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B11); break;
		case 28 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B12); break;
		case 29 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B13); break;
		case 30 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B14); break;
		case 31 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B15); break;
		case 32 :	WriteTraceToFlash(SAVE_SIGNAL_NR_B16); break;

		case 33 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C01); break;
		case 34 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C02); break;
		case 35 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C03); break;
		case 36 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C04); break;
		case 37 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C05); break;
		case 38 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C06); break;
		case 39 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C07); break;
		case 40 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C08); break;
		case 41 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C09); break;
		case 42 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C10); break;
		case 43 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C11); break;
		case 44 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C12); break;
		case 45 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C13); break;
		case 46 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C14); break;
		case 47 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C15); break;
		case 48 :	WriteTraceToFlash(SAVE_SIGNAL_NR_C16); break;

		case 49 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D01); break;
		case 50 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D02); break;
		case 51 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D03); break;
		case 52 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D04); break;
		case 53 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D05); break;
		case 54 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D06); break;
		case 55 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D07); break;
		case 56 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D08); break;
		case 57 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D09); break;
		case 58 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D10); break;
		case 59 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D11); break;
		case 60 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D12); break;
		case 61 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D13); break;
		case 62 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D14); break;
		case 63 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D15); break;
		case 64 :	WriteTraceToFlash(SAVE_SIGNAL_NR_D16); break;

		case 65 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E01); break;
		case 66 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E02); break;
		case 67 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E03); break;
		case 68 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E04); break;
		case 69 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E05); break;
		case 70 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E06); break;
		case 71 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E07); break;
		case 72 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E08); break;
		case 73 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E09); break;
		case 74 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E10); break;
		case 75 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E11); break;
		case 76 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E12); break;
		case 77 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E13); break;
		case 78 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E14); break;
		case 79 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E15); break;
		case 80 :	WriteTraceToFlash(SAVE_SIGNAL_NR_E16); break;

	}

	Display::DRAWROUNDBUTTON(250, 200, 140, 40, 0, 1);
	Display::TEXTOUTxvbig("Saving Signals", 260, 212, 0, UI_Plane2); 



}


//##########################################################################################################################################################
//BF new Recall Overlay function #013
void AMDFlash::RecallOverlay(void)
{
	if (FFT_Mode != FFT_OFF) return;
	if (USTB_Mode != USTB_OFF) return;

	if (RC_overlay) return;		//prevent recursive call and return to normal 
	if (RC_request) return;		//prevent recursive call and return to normal 

	unsigned long *lp_TracePointer = NULL;	
	unsigned long *lp_PlanePointer = UI_Plane3;               //UI_Plane1;	

	unsigned long *lp_BackupPointer1 = Buffer_Plane1;
	unsigned long *lp_BackupPointer2 = Buffer_Plane2;
	unsigned long *lp_BackupPointer3 = Buffer_Plane3;
	unsigned long *lp_BackupPointer4 = Buffer_Plane4;



/*
unsigned long *Buffer_PlaneM = (unsigned long *) 0x00930C00;
unsigned long *Buffer_Plane1 = (unsigned long *) 0x0093A200;
unsigned long *Buffer_Plane2 = (unsigned long *) 0x00943800;
unsigned long *Buffer_Plane3 = (unsigned long *) 0x0094CE00;
unsigned long *Buffer_Plane4 = (unsigned long *) 0x00956400;



unsigned long *UI_Plane1 = (unsigned long *) 0x0095FA00;		// white/grey
unsigned long *UI_Plane2 = (unsigned long *) 0x00969000;		// black
unsigned long *UI_Plane3 = (unsigned long *) 0x00972600;		// light yellow
unsigned long *UI_Plane4 = (unsigned long *) 0x0097BC00;		// yellow
unsigned long *UI_Plane5 = (unsigned long *) 0x00985200;		// dark grey

unsigned long *Channel_Plane1 = (unsigned long *) 0x0098E800;		// neon yellow
unsigned long *Channel_Plane2 = (unsigned long *) 0x00997E00;		// neon green
unsigned long *Channel_Plane3 = (unsigned long *) 0x009A1400;		// neon blue
unsigned long *Channel_Plane4 = (unsigned long *) 0x009AAA00;		// neon red
unsigned long *Channel_Math_Plane = (unsigned long *) 0x009B4000;	// neon pink

unsigned long *Marker_Plane1 = (unsigned long *) 0x009D9800;		// light red
unsigned long *Marker_Plane2 = (unsigned long *) 0x009E2E00;		// light red

unsigned long *Grid_Plane = (unsigned long *) 0x009EC400;		// grey
*/


	
	VirtualTimebase	   = 0;
	//printf("Starting Overlay Mode\r\n");

	// save actual configuration
	config_changed = 2;
	Write_Config_Flash();

	Run = 1;	//initialize operating mode

	//lets point the trace pointer to the choosen trace in the flash memory
	switch(SavePointer)
	{
		case 1 : lp_TracePointer = SAVE_SIGNAL_NR_A01; break;
		case 2 : lp_TracePointer = SAVE_SIGNAL_NR_A02; break;
		case 3 : lp_TracePointer = SAVE_SIGNAL_NR_A03; break;
		case 4 : lp_TracePointer = SAVE_SIGNAL_NR_A04; break;
		case 5 : lp_TracePointer = SAVE_SIGNAL_NR_A05; break;
		case 6 : lp_TracePointer = SAVE_SIGNAL_NR_A06; break;
		case 7 : lp_TracePointer = SAVE_SIGNAL_NR_A07; break;
		case 8 : lp_TracePointer = SAVE_SIGNAL_NR_A08; break;
		case 9 : lp_TracePointer = SAVE_SIGNAL_NR_A09; break;
		case 10 : lp_TracePointer = SAVE_SIGNAL_NR_A10; break;
		case 11 : lp_TracePointer = SAVE_SIGNAL_NR_A11; break;
		case 12 : lp_TracePointer = SAVE_SIGNAL_NR_A12; break;
		case 13 : lp_TracePointer = SAVE_SIGNAL_NR_A13; break;
		case 14 : lp_TracePointer = SAVE_SIGNAL_NR_A14; break;
		case 15 : lp_TracePointer = SAVE_SIGNAL_NR_A15; break;
		case 16 : lp_TracePointer = SAVE_SIGNAL_NR_A16; break;
	
		case 17 : lp_TracePointer = SAVE_SIGNAL_NR_B01; break;
		case 18 : lp_TracePointer = SAVE_SIGNAL_NR_B02; break;
		case 19 : lp_TracePointer = SAVE_SIGNAL_NR_B03; break;
		case 20 : lp_TracePointer = SAVE_SIGNAL_NR_B04; break;
		case 21 : lp_TracePointer = SAVE_SIGNAL_NR_B05; break;
		case 22 : lp_TracePointer = SAVE_SIGNAL_NR_B06; break;
		case 23 : lp_TracePointer = SAVE_SIGNAL_NR_B07; break;
		case 24 : lp_TracePointer = SAVE_SIGNAL_NR_B08; break;
		case 25 : lp_TracePointer = SAVE_SIGNAL_NR_B09; break;
		case 26 : lp_TracePointer = SAVE_SIGNAL_NR_B10; break;
		case 27 : lp_TracePointer = SAVE_SIGNAL_NR_B11; break;
		case 28 : lp_TracePointer = SAVE_SIGNAL_NR_B12; break;
		case 29 : lp_TracePointer = SAVE_SIGNAL_NR_B13; break;
		case 30 : lp_TracePointer = SAVE_SIGNAL_NR_B14; break;
		case 31 : lp_TracePointer = SAVE_SIGNAL_NR_B15; break;
		case 32 : lp_TracePointer = SAVE_SIGNAL_NR_B16; break;
	
		case 33 : lp_TracePointer = SAVE_SIGNAL_NR_C01; break;
		case 34 : lp_TracePointer = SAVE_SIGNAL_NR_C02; break;
		case 35 : lp_TracePointer = SAVE_SIGNAL_NR_C03; break;
		case 36 : lp_TracePointer = SAVE_SIGNAL_NR_C04; break;
		case 37 : lp_TracePointer = SAVE_SIGNAL_NR_C05; break;
		case 38 : lp_TracePointer = SAVE_SIGNAL_NR_C06; break;
		case 39 : lp_TracePointer = SAVE_SIGNAL_NR_C07; break;
		case 40 : lp_TracePointer = SAVE_SIGNAL_NR_C08; break;
		case 41 : lp_TracePointer = SAVE_SIGNAL_NR_C09; break;
		case 42 : lp_TracePointer = SAVE_SIGNAL_NR_C10; break;
		case 43 : lp_TracePointer = SAVE_SIGNAL_NR_C11; break;
		case 44 : lp_TracePointer = SAVE_SIGNAL_NR_C12; break;
		case 45 : lp_TracePointer = SAVE_SIGNAL_NR_C13; break;
		case 46 : lp_TracePointer = SAVE_SIGNAL_NR_C14; break;
		case 47 : lp_TracePointer = SAVE_SIGNAL_NR_C15; break;
		case 48 : lp_TracePointer = SAVE_SIGNAL_NR_C16; break;
	
		case 49 : lp_TracePointer = SAVE_SIGNAL_NR_D01; break;
		case 50 : lp_TracePointer = SAVE_SIGNAL_NR_D02; break;
		case 51 : lp_TracePointer = SAVE_SIGNAL_NR_D03; break;
		case 52 : lp_TracePointer = SAVE_SIGNAL_NR_D04; break;
		case 53 : lp_TracePointer = SAVE_SIGNAL_NR_D05; break;
		case 54 : lp_TracePointer = SAVE_SIGNAL_NR_D06; break;
		case 55 : lp_TracePointer = SAVE_SIGNAL_NR_D07; break;
		case 56 : lp_TracePointer = SAVE_SIGNAL_NR_D08; break;
		case 57 : lp_TracePointer = SAVE_SIGNAL_NR_D09; break;
		case 58 : lp_TracePointer = SAVE_SIGNAL_NR_D10; break;
		case 59 : lp_TracePointer = SAVE_SIGNAL_NR_D11; break;
		case 60 : lp_TracePointer = SAVE_SIGNAL_NR_D12; break;
		case 61 : lp_TracePointer = SAVE_SIGNAL_NR_D13; break;
		case 62 : lp_TracePointer = SAVE_SIGNAL_NR_D14; break;
		case 63 : lp_TracePointer = SAVE_SIGNAL_NR_D15; break;
		case 64 : lp_TracePointer = SAVE_SIGNAL_NR_D16; break;
	
		case 65 : lp_TracePointer = SAVE_SIGNAL_NR_E01; break;
		case 66 : lp_TracePointer = SAVE_SIGNAL_NR_E02; break;
		case 67 : lp_TracePointer = SAVE_SIGNAL_NR_E03; break;
		case 68 : lp_TracePointer = SAVE_SIGNAL_NR_E04; break;
		case 69 : lp_TracePointer = SAVE_SIGNAL_NR_E05; break;
		case 70 : lp_TracePointer = SAVE_SIGNAL_NR_E06; break;
		case 71 : lp_TracePointer = SAVE_SIGNAL_NR_E07; break;
		case 72 : lp_TracePointer = SAVE_SIGNAL_NR_E08; break;
		case 73 : lp_TracePointer = SAVE_SIGNAL_NR_E09; break;
		case 74 : lp_TracePointer = SAVE_SIGNAL_NR_E10; break;
		case 75 : lp_TracePointer = SAVE_SIGNAL_NR_E11; break;
		case 76 : lp_TracePointer = SAVE_SIGNAL_NR_E12; break;
		case 77 : lp_TracePointer = SAVE_SIGNAL_NR_E13; break;
		case 78 : lp_TracePointer = SAVE_SIGNAL_NR_E14; break;
		case 79 : lp_TracePointer = SAVE_SIGNAL_NR_E15; break;
		case 80 : lp_TracePointer = SAVE_SIGNAL_NR_E16; break;
	}


	//Load trace settings from flash
	if (LoadTraceSettings(lp_TracePointer) != 0) return;

	if (USTB_Mode != USTB_OFF)			// stored trace was USTB -> leave overlay
	{ Hardware::Restore_From_Flash(); return; }

	Hardware::Start_Record();

	Display::RefreshScreen();

	//wait for ADC
	for(int timeout = 0;timeout < 100 && !ADC_Data_Available;timeout++)
	nr_delay(20);

	//get live trace with loaded settings
	Hardware::Handle_ADC();

	// copy signals to buffer
	for (int i=0;i<0x4000;i++)
	{
		SIGNAL1_Buffer[i] = SIGNAL1[i];
		SIGNAL2_Buffer[i] = SIGNAL2[i];
		SIGNAL3_Buffer[i] = SIGNAL3[i];
		SIGNAL4_Buffer[i] = SIGNAL4[i];
	}	

	Signal::processSignal();

	//bend plane pointers to change signal colour 
	Buffer_Plane1 = lp_PlanePointer;
	Buffer_Plane2 = lp_PlanePointer;
	Buffer_Plane3 = lp_PlanePointer;
	Buffer_Plane4 = lp_PlanePointer;

	//clear grid area of plane 
	for (int i = GRID_PLANE_MIN; i < GRID_PLANE_MAX; i++) { lp_PlanePointer[i] = 0; }

	//draw actual signal with loaded settings
	//printf("Draw live trace....");
	Display::DrawSignals();		


	//restore plane pointers
	Buffer_Plane1 = lp_BackupPointer1;
	Buffer_Plane2 = lp_BackupPointer2;
	Buffer_Plane3 = lp_BackupPointer3;
	Buffer_Plane4 = lp_BackupPointer4;

	//load the choosen trace from flash
	//printf("Load trace....");
	Run = 0;
	LoadTracePoints(lp_TracePointer);

	//???Signal::ProcessSignalData();

	//draw loaded trace
	//printf("Draw loaded trace\r\n");
	Display::DrawSignals();		

	Hardware::TransferPlanes();
	//CommIF::Transfer_USB_Data();

	RC_request = 1;
	UI_request = 0;
	RC_user_if = 0;

	
	// overlay loop
	for (RC_overlay=1;RC_overlay;)
	{
		//handle user interface
		UserIF::Rotary_Interface();
		UserIF::Keyboard_Interface();

		//redraw the traces	
		if(RC_user_if && RC_overlay)
		{
			Hardware::ClearPlanes();

			//Run  = 1;
			RC_request  = 0;
			ACQ_NewData = 1;
			UI_request  = 0;

			// restore live signals from buffer
			for (int i=0;i<0x4000;i++)
			{
				SIGNAL1[i] = SIGNAL1_Buffer[i];
				SIGNAL2[i] = SIGNAL2_Buffer[i];
				SIGNAL3[i] = SIGNAL3_Buffer[i];
				SIGNAL4[i] = SIGNAL4_Buffer[i];
			}	
			
			Signal::processSignal();
		
			//change signal colour
			Buffer_Plane1 = lp_PlanePointer;
			Buffer_Plane2 = lp_PlanePointer;
			Buffer_Plane3 = lp_PlanePointer;
			Buffer_Plane4 = lp_PlanePointer;
		
			//clear grid area of plane 
			for (int i = GRID_PLANE_MIN; i < GRID_PLANE_MAX; i++) { lp_PlanePointer[i] = 0; }


			//draw actual signal with loaded settings
			//printf("Draw live trace....");
			Display::DrawSignals();		

			//UpdateMenuTextPlane = true;
			//Hardware::TransferPlanes();
		

			//restore plane pointers
			Buffer_Plane1 = lp_BackupPointer1;
			Buffer_Plane2 = lp_BackupPointer2;
			Buffer_Plane3 = lp_BackupPointer3;
			Buffer_Plane4 = lp_BackupPointer4;


			//load the choosen trace from flash
			//printf("Load trace....");
			Run=0;
			LoadTracePoints(lp_TracePointer);
			//Run = 0;
			RC_request  = 0;
			ACQ_NewData = 1;
			UI_request  = 0;
			Signal::processSignal();
			
			//draw loaded trace
			//printf("Draw loaded trace\r\n");
			Display::DrawSignals();

			UpdateMenuTextPlane = true;
			
			RC_request = 1;
			RC_user_if = 0;
		}

		Hardware::TransferPlanes();

	}

	//clean up and leave the overlay routine 
	for (int i = GRID_PLANE_MIN; i < GRID_PLANE_MAX; i++) { lp_PlanePointer[i] = 0; }
	RC_user_if = 0;

}



