
/* Includes ------------------------------------------------------------------*/
#include "dogm_128.h"
#include "font/SystemFont.h"
#include "font/HeaderFont.h"
#include "font/BigFont.h"

#define  RPRINTF_FLOAT
#define FALSE 0
#define false 0
#define TRUE  1
#define true  1

#define hexchar(x)	((((x)&0x0F)>9)?((x)+'A'-10):((x)+'0'))

/**
  * @brief  Initializes the low level interface used to drive the DOGM128
  * @param  None
  * @retval None
  */

TLCD lcd;

void spi_init(void)
{

  GPIO_InitTypeDef GPIO_InitStructure;
  SPI_InitTypeDef  SPI_InitStructure;

  /* Enable the SPI periph */
  RCC_APB1PeriphClockCmd(DOGM128_SPI_CLK, ENABLE);

  /* Enable SCK, MOSI and MISO GPIO clocks */
  RCC_AHB1PeriphClockCmd(DOGM128_SPI_SCK_GPIO_CLK | DOGM128_SPI_MISO_GPIO_CLK | DOGM128_SPI_MOSI_GPIO_CLK, ENABLE);

  /* Enable CS  GPIO clock */
  RCC_AHB1PeriphClockCmd(DOGM128_SPI_CS_GPIO_CLK, ENABLE);

  /* Enable INT1 GPIO clock */
  RCC_AHB1PeriphClockCmd(DOGM128_SPI_A0_GPIO_CLK, ENABLE);

  /* Enable INT2 GPIO clock */
  RCC_AHB1PeriphClockCmd(DOGM128_SPI_RST_GPIO_CLK, ENABLE);

  GPIO_PinAFConfig(DOGM128_SPI_SCK_GPIO_PORT, DOGM128_SPI_SCK_SOURCE, DOGM128_SPI_SCK_AF);
  GPIO_PinAFConfig(DOGM128_SPI_MISO_GPIO_PORT, DOGM128_SPI_MISO_SOURCE , DOGM128_SPI_MISO_AF);
  GPIO_PinAFConfig(DOGM128_SPI_MOSI_GPIO_PORT, DOGM128_SPI_MOSI_SOURCE, DOGM128_SPI_MOSI_AF );

  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_DOWN;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;

  /* SPI SCK pin configuration */
  GPIO_InitStructure.GPIO_Pin = DOGM128_SPI_SCK_PIN  ;
  GPIO_Init(DOGM128_SPI_SCK_GPIO_PORT, &GPIO_InitStructure);

  /* SPI  MOSI pin configuration */
  GPIO_InitStructure.GPIO_Pin =  DOGM128_SPI_MOSI_PIN ;
  GPIO_Init(DOGM128_SPI_MOSI_GPIO_PORT, &GPIO_InitStructure);


  /* SPI MISO pin configuration */
  GPIO_InitStructure.GPIO_Pin = DOGM128_SPI_MISO_PIN;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_Init(DOGM128_SPI_MISO_GPIO_PORT, &GPIO_InitStructure);

  /* SPI configuration -------------------------------------------------------*/
  SPI_I2S_DeInit(DOGM128_SPI);
  SPI_InitStructure.SPI_Direction = SPI_Direction_2Lines_FullDuplex;
  SPI_InitStructure.SPI_DataSize = SPI_DataSize_8b;
  SPI_InitStructure.SPI_CPOL = SPI_CPOL_Low;
  SPI_InitStructure.SPI_CPHA = SPI_CPHA_1Edge;
  SPI_InitStructure.SPI_NSS = SPI_NSS_Soft;
  SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_4;
  SPI_InitStructure.SPI_FirstBit = SPI_FirstBit_MSB;
  SPI_InitStructure.SPI_CRCPolynomial = 7;
  SPI_InitStructure.SPI_Mode = SPI_Mode_Master;
  SPI_Init(DOGM128_SPI, &SPI_InitStructure);

  /* Enable SPI1  */
  SPI_Cmd(DOGM128_SPI, ENABLE);

  /* Configure GPIO PIN for DOGM128 Chip select */
  GPIO_InitStructure.GPIO_Pin = DOGM128_SPI_CS_PIN;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(DOGM128_SPI_CS_GPIO_PORT, &GPIO_InitStructure);

  /* Deselect : Chip Select high */

  // GPIO_SetBits(DOGM128_SPI_CS_GPIO_PORT, DOGM128_SPI_CS_PIN);
  // nur wenn kein 10K Widerstand am Display eingeltet wurde
  /* Configure GPIO PINs to detect Interrupts */
  GPIO_InitStructure.GPIO_Pin = DOGM128_SPI_A0_PIN;
  GPIO_Init(DOGM128_SPI_A0_GPIO_PORT, &GPIO_InitStructure);

  /* Configure GPIO PINs to detect Interrupts */
  GPIO_InitStructure.GPIO_Pin = DOGM128_SPI_RST_PIN;
  GPIO_Init(DOGM128_SPI_RST_GPIO_PORT, &GPIO_InitStructure);
  
 }

void Lcd_Init(void)
{

	DOGM128_RST_LOW();

	Delay(0xfffff);
	Delay(0xfffff);

	DOGM128_RST_HIGH();		/* reset high for dogm128 */

	Delay(0xfffff);

	DOGM128_A0_LOW();	/* A0 low for init */

	/* tell the display that we want to start */
	DOGM128_CS_LOW();	//CS low


	/* init sequence */
	dogm128_send_command(DOGM128_DISPLAY_START_ADDRESS_BASE + 0);
	dogm128_send_command(DOGM128_ADC_REVERSE);
	dogm128_send_command(DOGM128_COM_OUTPUT_SCAN_NORMAL);
	dogm128_send_command(DOGM128_DISPLAY_NORMAL);
	dogm128_send_command(DOGM128_BIAS_19);
	dogm128_send_command(DOGM128_POWER_CONTROL_BASE + 0x07);
	dogm128_send_command(DOGM128_BOOSTER_RATIO_SET);
	dogm128_send_command(0x00); /* Booster x4 */
	dogm128_send_command(DOGM128_V0_OUTPUT_RESISTOR_BASE + 0x07);
	dogm128_send_command(DOGM128_ELECTRONIC_VOLUME_MODE_SET);
	dogm128_send_command(0x16); /* Contrast */
	dogm128_send_command(DOGM128_STATIC_INDICATOR_OFF);
	dogm128_send_command(0x00); /* Flashing OFF */
	dogm128_send_command(DOGM128_DISPLAY_ON);

	/* end transfer */
	DOGM128_CS_HIGH();	//CS high

}


/**
  * @brief  Sends a Byte through the SPI interface and return the Byte received 
  *         from the SPI bus.
  * @param  Byte : Byte send.
  * @retval The received byte value
  */
uint8_t DOGM128_SendByte(uint8_t byte)
{
	/* Loop while DR register in not emplty */
	while (SPI_I2S_GetFlagStatus(DOGM128_SPI, SPI_I2S_FLAG_TXE) == RESET){};
	/* Send a Byte through the SPI peripheral */
	SPI_I2S_SendData(DOGM128_SPI, byte);
	/* Wait to receive a Byte */
	while (SPI_I2S_GetFlagStatus(DOGM128_SPI, SPI_I2S_FLAG_RXNE) == RESET){};
	/* Return the Byte read from the SPI bus */
	return (uint8_t) SPI_I2S_ReceiveData(DOGM128_SPI);
}



void dogm128_send_command(uint8_t command)
{
     DOGM128_A0_LOW();/* A0 low for commands */
     DOGM128_SendByte(command);
}

void dogm128_send_data(uint8_t data)
{
     DOGM128_A0_HIGH();	 /* A0 high for data */
     DOGM128_SendByte(data);

}


//*----------------------------------------------------------------------------
// Display aus Mirror-Buffer anzeigen
//*----------------------------------------------------------------------------
void Lcd_Show(void)
{
	volatile uint8_t x, y;

	DOGM128_CS_LOW();	//CS low;       // Display selektieren
	for(y = 0; y < 8; y++) {
    dogm128_send_command(y + 0xB0); // Row select
    dogm128_send_command(0x10);     // Column select high
    dogm128_send_command(0x00);     // Column select low
    for(x = 0; x < 128; x++) { dogm128_send_data(lcd.Mirror[x][y]); }
	}
	DOGM128_CS_HIGH();	//CS low();     // Display deselektieren
}
//*----------------------------------------------------------------------------
// Display (Mirror) l�schen
// danach unbedingt Lcd_Show() aufrufen
//*----------------------------------------------------------------------------
void Lcd_Clear(void)
{
	uint16_t i;
	uint8_t  color = 0x00;
	uint8_t  *ptr = (uint8_t *)&lcd.Mirror;

	if(lcd.invers) { color = 0xFF; }
	for(i = 0; i < sizeof(lcd.Mirror); i++) { *ptr++ = color; }
}
//*----------------------------------------------------------------------------
// Pixel setzen
// x = 0..128, y = 0..64
//*----------------------------------------------------------------------------
void Lcd_SetPixel(uint8_t x, uint8_t y)
{
	if((y >= LCD_HEIGHT) || (x >= LCD_WIDTH)) return;
	if(lcd.invers) {
    lcd.Mirror[x][y / 8] &= ~(1 << (y & 0x07));
	} else {
    lcd.Mirror[x][y / 8] |= (1 << (y & 0x07));
	}
}
//*----------------------------------------------------------------------------
// Pixel l�schen
// x = 0..128, y = 0..64
//*----------------------------------------------------------------------------
void Lcd_ClrPixel(uint8_t x, uint8_t y)
{
	if((y >= LCD_HEIGHT) || (x >= LCD_WIDTH)) return;
	if(lcd.invers) {
    lcd.Mirror[x][y / 8] |= (1 << (y & 0x07));
	} else {
    lcd.Mirror[x][y / 8] &= ~(1 << (y & 0x07));
	}
}
//*----------------------------------------------------------------------------
// Horizontale Linie zeichnen
// x = 0..128, y = 0..64, w = L�nge
//*----------------------------------------------------------------------------
void Lcd_Hline(uint8_t x, uint8_t y, uint8_t w)
{
	uint8_t xp;

	for(xp = x; xp <= x + w - 1; xp++) { Lcd_SetPixel(xp, y); }
}
//*----------------------------------------------------------------------------
// Vertikale Linie zeichnen
// x = 0..128, y = 0..64, h = H�he
//*----------------------------------------------------------------------------
void Lcd_Vline(uint8_t x, uint8_t y, uint8_t h)
{
	uint8_t yp;

	for(yp = y; yp <= y + h - 1; yp++) { Lcd_SetPixel(x, yp); }
}
//*----------------------------------------------------------------------------
// Bereich l�schen
//*----------------------------------------------------------------------------
void Lcd_Erase(uint8_t x, uint8_t y, uint8_t w, uint8_t h)
{
	uint8_t xp, yp;

	for(xp = x; xp <= x + w - 1; xp++) {
		for(yp = y; yp <= y + h - 1; yp++) {
			Lcd_ClrPixel(xp, yp);
    }
  }
}
//*----------------------------------------------------------------------------
// DISP-Rectangle
// x = 0..128, y = 0..64, w = Breite, h = H�he, mode = FRAMED/FILLED
//*----------------------------------------------------------------------------
void Lcd_Rectangle(uint8_t x, uint8_t y, uint8_t w, uint8_t h, uint8_t mode)
{
	uint8_t xp, yp;

  if(mode == FRAMED) {
    for(xp = x; xp <= x + w - 1; xp++) {
      Lcd_SetPixel(xp, y);
      Lcd_SetPixel(xp, y + h - 1);
    }
    for(yp = y + 1; yp < y + h - 1; yp++) {
      Lcd_SetPixel(x, yp);
      Lcd_SetPixel(x + w - 1, yp);
    }
  } else {
    for(xp = x; xp <= x + w - 1; xp++) {
      for(yp = y; yp <= y + h - 1; yp++) {
        Lcd_SetPixel(xp, yp);
      }
    }
  }
}
//*----------------------------------------------------------------------------
// Round Rectangle
//*----------------------------------------------------------------------------
void Lcd_RoundRectangle(uint8_t x, uint8_t y, uint8_t w, uint8_t h, uint8_t mode)
{
  uint8_t xp, yp;

  if(mode == FRAMED) {
    for(xp = x + 2; xp <= x + w - 3; xp++) {
      Lcd_SetPixel(xp, y);
      Lcd_SetPixel(xp, y + h - 1);
    }
    Lcd_SetPixel(x + 1,     y + 1);
    Lcd_SetPixel(x + w - 2, y + 1);
    Lcd_SetPixel(x + 1,     y + h - 2);
    Lcd_SetPixel(x + w - 2, y + h - 2);
    for(yp = y + 2; yp <= y + h - 3; yp++) {
      Lcd_SetPixel(x, yp);
      Lcd_SetPixel(x + w - 1, yp);
    }
  } else {
    Lcd_Hline(x + 2, y,     w - 4);
    Lcd_Hline(x + 1, y + 1, w - 2);
    for(yp = y + 2; yp <= y + h - 3; yp++) {
      Lcd_Hline(x, yp, w);
    }
    Lcd_Hline(x + 1, yp,     w - 2);
    Lcd_Hline(x + 2, yp + 1, w - 4);
  }
}
//*----------------------------------------------------------------------------
// Fortschrittsbalken anzeigen
//*----------------------------------------------------------------------------
void Lcd_Bar(uint8_t x, uint8_t y, uint8_t w, uint8_t max, uint8_t pos)
{
  uint16_t len = ((uint16_t)(w - 4) * (uint16_t)pos) / (uint16_t)max;

  Lcd_RoundRectangle(x, y, w, 7, 0);
  lcd.invers = TRUE;
  Lcd_Rectangle(x + 2, y + 2, w - 4, 3, FILLED);
  lcd.invers = FALSE;
  Lcd_Rectangle(x + 2, y + 2, len, 3, FILLED);
}
//*----------------------------------------------------------------------------

//*----------------------------------------------------------------------------
// FONT-ROUTINEN
//*----------------------------------------------------------------------------
//*----------------------------------------------------------------------------
// Font Init
// Setzt die Font-Headerdaten
//*----------------------------------------------------------------------------
void Font_Init(uint8_t *ptr)
{
  lcd.Char_First  = (((TFONTHEAD *)ptr)->Font_FirstChar);
  lcd.Char_Last   = (((TFONTHEAD *)ptr)->Font_LastChar);
  lcd.Font_Height = (((TFONTHEAD *)ptr)->Font_Height);
  lcd.fontwidth =  ptr + sizeof(TFONTHEAD);
  lcd.fontdata =   lcd.fontwidth + (lcd.Char_Last - lcd.Char_First + 1);

  lcd.X_Pos = 0;
  lcd.X_Max = LCD_WIDTH - 1;    // (131)
  lcd.Y_Pos = 0;
}
//*----------------------------------------------------------------------------
// Position f�r Textausgabe setzen
//*----------------------------------------------------------------------------
void Font_SetPos(uint8_t x, uint8_t y)
{
  lcd.X_Pos = x;
  lcd.Y_Pos = y;
}
//*----------------------------------------------------------------------------
// Fordert die Breite in pixeln den ge�nschten Zeichens an (0 = Error)
//*----------------------------------------------------------------------------
uint8_t Font_CharWidth(char c)
{
  if(c < lcd.Char_First) { return 0; }
  if(c > lcd.Char_Last)  { return 0; }
  return lcd.fontwidth[c - lcd.Char_First];
}
//*----------------------------------------------------------------------------
// Breite des Textes in Pixeln errechnen
//*----------------------------------------------------------------------------
uint16_t Font_TextWidth(char *ptr, uint8_t mode)
{
  uint16_t width = 0;
  char c;

  while(TRUE) {
    if(mode & TXT_MODE_FLASH) { c = ptr[0]; }
                         else { c = *ptr; }
    if(c == 0) { break; }
    width += Font_CharWidth(c) + 1;
    ptr++;
  }
  return width - 1;
}
//*----------------------------------------------------------------------------
// Zeichen anzeigen mit dem aktuellen Zeichensatz (aus Font_init)
//*----------------------------------------------------------------------------
void Font_DrawChar(char c)
{
  uint8_t x, y, i = 0;
  uint16_t idx = 0;
  uint8_t BitCount;
  uint8_t width = Font_CharWidth(c);
  uint8_t FData;

  if(width == 0) { return; }
  if((lcd.X_Pos + width) > lcd.X_Max) { return; }

  c -= lcd.Char_First;
  while(i < c) { idx += lcd.fontwidth[i++]; }
  idx = idx * lcd.Font_Height;
  BitCount = idx & 0x07;
  idx >>= 3;
  FData = lcd.fontdata[idx++] >> BitCount;

  for(x = 0; x < width; x++) {
    for(y = 0; y < lcd.Font_Height; y++) {
      if(BitCount > 7) {
        FData = lcd.fontdata[idx++];
        BitCount = 0;
      }
      if(FData & 0x01) { Lcd_SetPixel(lcd.X_Pos + x, lcd.Y_Pos + y); }
                  else { Lcd_ClrPixel(lcd.X_Pos + x, lcd.Y_Pos + y); }
      FData >>= 1;
      BitCount++;
    }
  }
  lcd.X_Pos += width + 1;
	//Lcd_Show(d);
}
//*----------------------------------------------------------------------------
// Text ausgeben an x,y
//*----------------------------------------------------------------------------

#ifdef RPRINTF_FLOAT
//*----------------------------------------------------------------------------
// *** rprintfNum ***
// special printf for numbers only
// see formatting information below
//	Print the number "n" in the given "base"
//	using exactly "numDigits"
//	print +/- if signed flag "isSigned" is TRUE
//	use the character specified in "padchar" to pad extra characters
//
//	Examples:
//	rprintfNum(10, 6,  TRUE, ' ',   1234);  -->  " +1234"
//	rprintfNum(10, 6, FALSE, '0',   1234);  -->  "001234"
//	rprintfNum(16, 6, FALSE, '.', 0x5AA5);  -->  "..5AA5"
//*----------------------------------------------------------------------------
void rprintfNum(char base, char numDigits, char isSigned, char padchar, long n)
{

	char *p, buf[32];
	unsigned long x;
	unsigned char count;

	// prepare negative number
	if( isSigned && (n < 0) )
	{
		x = -n;
	}
	else
	{
	 	x = n;
	}

	// setup little string buffer
	count = (numDigits-1)-(isSigned?1:0);
  	p = buf + sizeof (buf);
  	*--p = '\0';

	// force calculation of first digit
	// (to prevent zero from not printing at all!!!)
	*--p = hexchar(x%base); x /= base;
	// calculate remaining digits
	while(count--)
	{
		if(x != 0)
		{
			// calculate next digit
			*--p = hexchar(x%base); x /= base;
		}
		else
		{
			// no more digits left, pad out to desired length
			*--p = padchar;
		}
	}

	// apply signed notation if requested
	if( isSigned )
	{
		if(n < 0)
		{
   			*--p = '-';
		}
		else if(n > 0)
		{
	   		*--p = '+';
		}
		else
		{
	   		*--p = ' ';
		}
	}

	// print the string right-justified
	count = numDigits;
	while(count--)
	{
		Font_DrawChar(*p++);
	}
}


//*----------------------------------------------------------------------------
//*** rprintfFloat ***floating-point print
//*----------------------------------------------------------------------------


void rprintfFloat(char numDigits, double x)
{
	unsigned char firstplace = FALSE;
	unsigned char negative;
	unsigned char i, digit;
	double place = 1.0;

	// save sign
	negative = (x<0);
	// convert to absolute value
	x = (x>0)?(x):(-x);

	// find starting digit place
	for(i=0; i<15; i++)
	{
		if((x/place) < 10.0)
			break;
		else
			place *= 10.0;
	}
	// print polarity character
	if(negative)
		Font_DrawChar('-');
	else
		Font_DrawChar('+');

	// print digits
	for(i=0; i<numDigits; i++)
	{
		digit = (x/place);

		if(digit | firstplace | (place == 1.0))
		{
			firstplace = TRUE;
			Font_DrawChar(digit+0x30);
		}
		else
			Font_DrawChar(' ');

		if(place == 1.0)
		{
			Font_DrawChar('.');
		}

		x -= (digit*place);
		place /= 10.0;
	}
}
#endif

void rprintfStr(char str[])
{
	// send a string stored in RAM
	// check to make sure we have a good pointer
	if (!str) return;

	// print the string until a null-terminator
	while (*str)
		Font_DrawChar(*str++);
}
void rprintfStrLen(char str[], unsigned int start, unsigned int len)
{
	register int i=0;

	// check to make sure we have a good pointer
	if (!str) return;
	// spin through characters up to requested start
	// keep going as long as there's no null
	while((i++<start) && (*str++));
//	for(i=0; i<start; i++)
//	{
//		// keep steping through string as long as there's no null
//		if(*str) str++;
//	}

	// then print exactly len characters
	for(i=0; i<len; i++)
	{
		// print data out of the string as long as we haven't reached a null yet
		// at the null, start printing spaces
		if(*str)
			Font_DrawChar(*str++);
		else
			Font_DrawChar(' ');
	}

}


//*----------------------------------------------------------------------------
// Fonts Init
//*----------------------------------------------------------------------------
void Font_SystemFont(void) { Font_Init((uint8_t *)&SystemFont); }
void Font_HeaderFont(void) { Font_Init((uint8_t *)&HeaderFont); }
void Font_BigFont(void)    { Font_Init((uint8_t *)&BigFont); }
//*----------------------------------------------------------------------------

