//
// menu - complete handling of menu structure
//

#include "menu.h"

#include <stdbool.h>
#include <stdint.h>
#include <avr/pgmspace.h>

#include "encoder.h"
#include "m50530.h"
#include "config.h"

static struct
{
  int8_t Level;
  int8_t Position[4];
  bool   Changed;
} gMenuInfo = {-1, {0, 0, 0, 0}, false};

typedef enum { asSelect, asIncrement, asDecrement } NActionSelect;

  // user defined character definition
const uint8_t coUDF[] /*PROGMEM*/ = { 0x04, 0x0E, 0x0E, 0x0E, 0x1F, 0x1F, 0x04, 0x00,  // 0xF4 Alarm aktiv
                                  0x04, 0x0A, 0x0A, 0x0A, 0x11, 0x1B, 0x04, 0x00,  // 0xF5 Alarm inaktiv
                                  0x02, 0x09, 0x05, 0x15, 0x05, 0x09, 0x02, 0x00,  // 0xF6 Ton an
                                  0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,  // 0xF7 Ton aus
                                  0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x00, 0x0C, 0x00,  // 0xF8 Warnsymbol
                                  0x08, 0x0C, 0x0E, 0x0F, 0x0E, 0x0C, 0x08, 0x00,  // 0xF9 Menüauswahl
                                  0x1F, 0x02, 0x04, 0x08, 0x04, 0x02, 0x1F, 0x00}; // 0xFA Heizung
//                                  0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00,  // 0xFB Volume 1
//                                  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,  // 0xFC Volume 2
//                                  0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C, 0x00,  // 0xFD Volume 3
//                                  0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x1E, 0x00}; // 0xFE Volume 4


void menu_init(void)
{
  gMenuInfo.Level = -1;
  gMenuInfo.Changed = false;  
  
    // define characters
  //LCDWriteBufferPGM(0xA0, &coUDF[0], 56);
  LCDWriteBuffer(0xA0, &coUDF[0], 56);
  LCDWriteLoop(0x10, 8); // 0xFB Volume 1
  LCDWriteLoop(0x18, 8); // 0xFC Volume 2
  LCDWriteLoop(0x1C, 8); // 0xFD Volume 3
  LCDWriteLoop(0x1E, 8); // 0xFE Volume 4
}

  // main menu
  // >
const char * gMenuStr[3] = { "Aktion", "Einstellungen", "Ende" };
  // > Aktion >
const char * gMenu0Str[4] = { "Deckel", "Regelung", "Alarm", "zur\x5\x63k" };
  // > Aktion > Deckel >
const char * gMenu00Str[2] = { "Offen", "Geschlossen" };
  // > Aktion > Regelung >
  // > Aktion > Alarm >
const char * gMenu0xStr[2] = { "Einschalten", "Ausschalten" };
  // > Einstellungen >
const char * gMenu1Str[5] = { "Temperatur", "Luftfeuchtigkeit", "Alarm", "Version", "zur\x5\x63k" };
  // > Einstellungen > Temperatur >
const char * gMenu10Str[4] = { "Soll", "Hysterese", "Timeout", "zur\x5\x63k" };
  // > Einstellungen > Luftfeuchtigkeit >
const char * gMenu11Str[3] = { "Alarm Min", "Alarm Max", "zur\x5\x63k" };
  // > Einstellungen > Alarm >
const char * gMenu12Str[3] = { "Stumm", "Lautst�rke", "zur\x5\x63k" };

  // level 0 (main menu)
int8_t getPositionCountLevel0(void)
{
  return 3;
}

  // level 1 (Aktion/Einstellungen)
int8_t getPositionCountLevel1(const int8_t aLevel0)
{
  int8_t lValue = 0;

  switch ( aLevel0 )
  {
    case 0: // Aktion
      lValue = 4;
      break;

    case 1: // Einstellungen
      lValue = 5;
      break;
  }

  return lValue;
}

  // level 2 (Aktion/Einstellungen)
int8_t getPositionCountLevel2(const int8_t aLevel0, const int8_t aLevel1)
{
  int8_t lValue = 0;

  switch ( aLevel0 )
  {
    case 0: // Aktion
      switch ( aLevel1 )
      {
        case 0: // Deckel
        case 1: // Regelung
        case 2: // Alarm
          lValue = 2;
          break;
      }
      break;

    case 1: // Einstellungen
      switch ( aLevel1 )
      {
        case 0: // Temperatur
          lValue = 4;
          break;

        case 1: // Luftfeuchtigkeit
        case 2: // Alarm
          lValue = 3;
          break;
      }
      break;
  }
  
  return lValue;
}

  // level 3 (Temperatur/Luftfeuchtigkeit/Alarm)
int8_t getPositionCountLevel3(const int8_t aLevel0, const int8_t aLevel1, const int8_t aLevel2)
{
  int8_t lValue = 0;

  if ( 1 == aLevel0 )
  {
    switch ( aLevel1 )
    {
      case 0: // Temperatur
        switch ( aLevel2 )
        {
          case 0: // Soll
          case 1: // Hysterese
          case 2: // Timeout
            lValue = 1;
            break;
        }
        break;

      case 1: // Luftfeuchtigkeit
        switch ( aLevel2 )
        {
          case 0: // Alarm Min
          case 1: // Alarm Max
            lValue = 1;
            break;
        }
        break;

      case 2: // Alarm
        switch ( aLevel1 )
        {
          case 0: // Stumm
          case 1: // Lautstärke
            lValue = 1;
            break;
        }
        break;
    }
  }
  
  return lValue;
}

void doAction(const NActionSelect aAction, const int8_t aLevel0, const int8_t aLevel1, const int8_t aLevel2, const int8_t aLevel3)
{
  switch ( aLevel0 )
  {
    case 0: // Aktion
      switch ( aLevel1 )
      {
        case 0: // Deckel
          switch ( aLevel2 )
          {
            case 0: // offen
              gConfig.CaseOpen = true;
              break;

            case 1: // geschlossen
              gConfig.CaseOpen = false;
              break;

            default:
              return;
          }

          gMenuInfo.Level--;
          gMenuInfo.Changed = true;
          break;

        case 1: // Regelung
          switch ( aLevel2 )
          {
            case 0: // einschalten
              gConfig.Regulate = true;
              break;

            case 1: // ausschalten
              gConfig.Regulate = false;
              break;

            default:
              return;
          }

          gMenuInfo.Level--;
          gMenuInfo.Changed = true;
          break;

        case 2: // Alarm
          switch ( aLevel2 )
          {
            case 0: // einschalten
              gConfig.AlarmEnabled = true;
              break;

            case 1: // ausschalten
              gConfig.AlarmEnabled = false;
              break;

            default:
              return;
          }

          gMenuInfo.Level--;
          gMenuInfo.Changed = true;
          break;
      }
      break;

    case 1: // Einstellungen
      switch ( aLevel1 )
      {
        case 0: // Temperatur
          switch ( aLevel2 )
          {
            case 0: // Soll
              ;
              break;

            case 1: // Hysterese
              ;
              break;

            case 2: // Timeout
              ;
              break;

            default:
              return;
          }

          gMenuInfo.Level--;
          gMenuInfo.Changed = true;
          break;

        case 1: // Luftfeuchtigkeit
          switch ( aLevel2 )
          {
            case 0: // einschalten
              gConfig.Regulate = true;
              break;

            case 1: // ausschalten
              gConfig.Regulate = false;
              break;

            default:
              return;
          }

          gMenuInfo.Level--;
          gMenuInfo.Changed = true;
          break;

        case 2: // Alarm
          switch ( aLevel2 )
          {
            case 0: // Stumm
              gConfig.AlarmEnabled = true;
              break;

            case 1: // Lautstärke
              
              gConfig.AlarmEnabled = false;
              break;

            default:
              return;
          }

          gMenuInfo.Level--;
          gMenuInfo.Changed = true;
          break;

        case 3: // Version
          gMenuInfo.Level = 3;
          gMenuInfo.Changed = true;
          break;

      }
      break;
  }
}

void menu_cycle(void)
{
  bool lKey = encode_readKey();
  int8_t lEnc = encode_read4(); 

    // any action?
  if ( lKey || lEnc )
  {
      // enable menu?
    if ( -1 == gMenuInfo.Level )
    {
      gMenuInfo.Level = 0;
      gMenuInfo.Position[0] = 0; 
    }
    else if ( lEnc )
    {
        // increment
      if ( 0 < lEnc )
      {
        int8_t lPosCount = 0;

        switch ( gMenuInfo.Level )
        {
          case 0: // 1st level (main menu)
            gMenuInfo.Position[0] = (gMenuInfo.Position[0] + 1) % getPositionCountLevel0();
            break;

          case 1: // 2nd level (Aktion/Einstellungen)
            lPosCount = getPositionCountLevel1(gMenuInfo.Position[0]);

            if ( lPosCount )
              gMenuInfo.Position[1] = (gMenuInfo.Position[1] + 1) % lPosCount;
            else
              doAction(asIncrement, gMenuInfo.Position[0], gMenuInfo.Position[1], -1, -1);
            break;

          case 2: // 3rd level
            lPosCount = getPositionCountLevel2(gMenuInfo.Position[0], gMenuInfo.Position[1]);

            if ( lPosCount )
              gMenuInfo.Position[2] = (gMenuInfo.Position[2] + 1) % lPosCount;
            else
              doAction(asIncrement, gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2], -1);
            break;

          case 3: // 4th level
            lPosCount = getPositionCountLevel3(gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2]);

            if ( lPosCount )
              gMenuInfo.Position[3] = (gMenuInfo.Position[3] + 1) % lPosCount;
            else
              doAction(asIncrement, gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2], gMenuInfo.Position[3]);
            break;
        }
      }
        // decrement
      else
      {
        int8_t lPosCount = 0;

        switch ( gMenuInfo.Level )
        {
          case 0: // 1st level (main menu)
            --gMenuInfo.Position[0];
            if ( 0 > gMenuInfo.Position[0] )
              gMenuInfo.Position[0] = getPositionCountLevel0()-1;
            break;

          case 1: // 2nd level (Aktion/Einstellungen)
            lPosCount = getPositionCountLevel1(gMenuInfo.Position[0]);
            
            if ( lPosCount )
            {
              --gMenuInfo.Position[1];
              if ( 0 > gMenuInfo.Position[1] )
                gMenuInfo.Position[1] = lPosCount-1;
            }
            else
              doAction(asDecrement, gMenuInfo.Position[0], gMenuInfo.Position[1], -1, -1);
            break;

          case 2: // 3rd level
            lPosCount = getPositionCountLevel2(gMenuInfo.Position[0], gMenuInfo.Position[1]);
            
            if ( lPosCount )
            {
              --gMenuInfo.Position[2];
              if ( 0 > gMenuInfo.Position[2] )
                gMenuInfo.Position[2] = lPosCount-1;
            }
            else
              doAction(asDecrement, gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2], -1);
            break;

          case 3: // 4th level
            lPosCount = getPositionCountLevel3(gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2]);
            
            if ( lPosCount )
            {
              --gMenuInfo.Position[3];
              if ( 0 > gMenuInfo.Position[3] )
                gMenuInfo.Position[3] = lPosCount-1;
            }
            else
              doAction(asDecrement, gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2], gMenuInfo.Position[3]);
            break;
        }
      }
    }
    else
    {
      // key pressed
      switch ( gMenuInfo.Level )
      {
        case 0: // 1st level (main menu)
          if ( gMenuInfo.Position[0] < (getPositionCountLevel0()-1) )
          {
            gMenuInfo.Level++;
            gMenuInfo.Position[gMenuInfo.Position[0]] = 0;
          }
          else // Ende
            gMenuInfo.Level--;
          break;

        case 1: // 2nd level (Aktion/Einstellungen)
          if ( gMenuInfo.Position[1] < (getPositionCountLevel1(gMenuInfo.Position[0])-1) )
          {
              // 3rd level available?
            if ( getPositionCountLevel2(gMenuInfo.Position[0], gMenuInfo.Position[1]) )
            {
              gMenuInfo.Level++;
              gMenuInfo.Position[2] = 0;
            }
            else
            {
              doAction(asSelect, gMenuInfo.Position[0], gMenuInfo.Position[1], -1, -1);
            }
          }
          else // zurück
            gMenuInfo.Level--;
          break;

        case 2: // 3rd level
          if ( gMenuInfo.Position[2] < (getPositionCountLevel2(gMenuInfo.Position[0], gMenuInfo.Position[1])-1) )
          {
              // 4th level available?
            if ( getPositionCountLevel3(gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2]) )
            {
              gMenuInfo.Level++;
              gMenuInfo.Position[3] = 0;  // mit aktuellem Wert vorbelegen
            }
            else
            {
              doAction(asSelect, gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2], -1);
            }            
          }
          else
            gMenuInfo.Level--;
          break;

        case 3: // 4th level
          if ( gMenuInfo.Position[3] < (getPositionCountLevel3(gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2])-1) )
          {
            //doAction(asSelect, gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2], gMenuInfo.Position[3]);
        	doAction(asSelect, gMenuInfo.Position[0], gMenuInfo.Position[1], gMenuInfo.Position[2], -1);
          }
          else
            gMenuInfo.Level--;
          break;
      }
    }

    gMenuInfo.Changed = true;
  }
}

void LCDOutDec(int8_t aNumber);

void LCDClearLine(const uint8_t aLine)
{
  LCDSetCursorPos(aLine, 0);
  LCDWriteLoop(' ', 15);
}

void menu_output(void)
{
  if ( gMenuInfo.Changed )
  {
    LCDClearLine(2);
    LCDClearLine(3);

      // first menu line
    LCDSetCursorPos(2, 0);
    switch ( gMenuInfo.Level )
    {
      case 1:
        LCDWrite(gMenuStr[gMenuInfo.Position[0]]);
        break;

      case 2:
        switch ( gMenuInfo.Position[0] )
        {
          case 0: // Aktion
            LCDWrite(gMenu0Str[gMenuInfo.Position[1]]);
            break;

          case 1: // Einstellungen
            LCDWrite(gMenu1Str[gMenuInfo.Position[1]]);
            break;
        }
        break;

      case 3:
        if ( 1 == gMenuInfo.Position[0] )
        {
          switch ( gMenuInfo.Position[1] )
          {
            case 0: // Temperatur
              LCDWrite(gMenu10Str[gMenuInfo.Position[2]]);
              break;

            case 1: // Luftfeuchtigkeit
              LCDWrite(gMenu11Str[gMenuInfo.Position[2]]);
              break;

            case 2: // Alarm
              LCDWrite(gMenu12Str[gMenuInfo.Position[2]]);
              break;
          }
        }
        break;
    }
    
      // second menu line
    LCDSetCursorPos(3, 0);
    switch ( gMenuInfo.Level )
    {
      case 0:
        LCDWrite(gMenuStr[gMenuInfo.Position[0]]);
        break;

      case 1:
        switch ( gMenuInfo.Position[0] )
        {
          case 0: // Aktion
            LCDWrite(gMenu0Str[gMenuInfo.Position[1]]);
            break;

          case 1: // Einstellungen
            LCDWrite(gMenu1Str[gMenuInfo.Position[1]]);
            break;
        }
        break;

      case 2:
        switch ( gMenuInfo.Position[0] )
        {
          case 0: // Aktion
            switch ( gMenuInfo.Position[1] )
            {
              case 0: // Deckel
                //LCDWrite(gMenu00Str[gMenuInfo.Position[2]]);
                LCDWrite(gMenu00Str[gMenuInfo.Position[2]]);
                break;

              case 1: // Regelung
                LCDWrite(gMenu0xStr[gMenuInfo.Position[2]]);
                break;

              case 2: // Alarm
                LCDWrite(gMenu0xStr[gMenuInfo.Position[2]]);
                break;
            }
            break;

          case 1: // Einstellungen
            switch ( gMenuInfo.Position[1] )
            {
              case 0: // Temperatur
                LCDWrite(gMenu10Str[gMenuInfo.Position[2]]);
                break;

              case 1: // Luftfeuchtigkeit
                LCDWrite(gMenu11Str[gMenuInfo.Position[2]]);
                break;

              case 2: // Timeout
                LCDWrite(gMenu12Str[gMenuInfo.Position[2]]);
                break;
            }
            break;
        }
        break;

      case 3:
        {
            // Einstellungen
          if ( 1 == gMenuInfo.Position[0] )
          {
            switch ( gMenuInfo.Position[1] )
            {
              case 0: // Temperatur
                switch ( gMenuInfo.Position[2] )
                {
                  case 0: // Soll
                    //LCDSetCursorPos(3, );
                    LCDWrite("T (soll) xx.x �C");
                    LCDSetCursorPos(3, 9);
                    LCDOutDec(gConfig.SollTemp);
                    break;

                  case 1: // Hysterese
                    LCDSetCursorPos(3, 10);
                    LCDOutDec(gConfig.Hysterese);
                    LCDWrite(" K");
                    break;

                  case 2: // Timeout
                    LCDWrite("xxx s");
                    break;

                }
                break;

              case 1: // Luftfeuchtigkeit
                switch ( gMenuInfo.Position[2] )
                {
                  case 0: // Alarm Min
                    LCDWrite("xx rel %");
                    break;

                  case 1: // Alarm Max
                    LCDWrite("xx rel %");
                    break;
                }
                break;

              case 2: // Alarm
                switch ( gMenuInfo.Position[2] )
                {
                  case 0: // Stumm
                    LCDWrite("aus");
                    break;

                  case 1: // Lautstärke
                    LCDWrite("min ******** max");
                    break;
                }
                break;
            } // switch ( level 1 )
          } // if 1 == level0
        }
        break;
    }

    gMenuInfo.Changed = false;

/*
    LCDSetCursorPos(3, 9);
    LCDOutDec(gMenuInfo.Position[0]);
    LCDWriteChar('/');
    LCDOutDec(gMenuInfo.Position[1]);
    LCDWriteChar('/');
    LCDOutDec(gMenuInfo.Position[2]);
    LCDWriteChar('/');
    LCDOutDec(gMenuInfo.Level);
*/
  }
}

