/*
,-----------------------------------------------------------------------------------------.
| this file implements the driver for the microchip enc28j60 ethernet controller
| - some ideas are based on the enc28j60 driver of the procyon avrlib ;)
|
| BUGS:
| - sometimes enc28j60 is not correctly initialised (not receiving any packets)
|
| Author   : Simon Schulz / avr{AT}auctionant.de
| this version 2007'0819 improved by eProfi (PHLCON 3472, skip select_bank for regs 1b..1f,
|   enc28j60_select_bank  inlined into  enc28j60_read_address  and  enc28j60_write_address)
|-----------------------------------------------------------------------------------------
| License:
| This program is free software; you can redistribute it and/or modify it under
| the terms of the GNU General Public License as published by the Free Software
| Foundation; either version 2 of the License, or (at your option) any later
| version.
| This program is distributed in the hope that it will be useful, but
|
| WITHOUT ANY WARRANTY;
|
| without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
| PURPOSE. See the GNU General Public License for more details.
|
| You should have received a copy of the GNU General Public License along with
| this program; if not, write to the Free Software Foundation, Inc., 51
| Franklin St, Fifth Floor, Boston, MA 02110, USA
|
| http://www.gnu.de/gpl-ger.html
`-----------------------------------------------------------------------------------------*/
#include "enc28j60.h"
#include "httpd.h"

//-----------------------------------------------------------------------------

unsigned char mymac[6];
unsigned char enc_revid = 0;

//-----------------------------------------------------------------------------

static volatile unsigned char enc_cur_bank = 0;
static volatile unsigned int  enc_next_packet_ptr = 0;

static unsigned char enc_configdata[] PROGMEM = {

	// enc registers

	// tx buffer
	ENC_REG_ETXSTL, LO8(ENC_TX_BUFFER_START),
	ENC_REG_ETXSTH, HI8(ENC_TX_BUFFER_START),
	ENC_REG_ETXNDL, LO8(ENC_TX_BUFFER_END),
	ENC_REG_ETXNDH, HI8(ENC_TX_BUFFER_END),

	// rx buffer
	ENC_REG_ERXSTL, LO8(ENC_RX_BUFFER_START),
	ENC_REG_ERXSTH, HI8(ENC_RX_BUFFER_START),
	ENC_REG_ERXNDL, LO8(ENC_RX_BUFFER_END),
	ENC_REG_ERXNDH, HI8(ENC_RX_BUFFER_END),

	// push mac out of reset
	ENC_REG_MACON2, 0x00,

	// mac receive enable, rx and tx pause control frames enable
	ENC_REG_MACON1, (1<<ENC_BIT_MARXEN) | (1<<ENC_BIT_RXPAUS) | (1<<ENC_BIT_TXPAUS),

	#ifdef FULL_DUPLEX
	// mac auto padding of small packets, add crc, frame length check, full duplex
	ENC_REG_MACON3, (1<<ENC_BIT_PADCFG0) | (1<<ENC_BIT_TXCRCEN)
			 | (1<<ENC_BIT_FRMLNEN) | (1<<ENC_BIT_FULDPX),
	#else
	// mac auto padding of small packets, add crc, frame length check, half duplex
	ENC_REG_MACON3, (1<<ENC_BIT_PADCFG0) | (1<<ENC_BIT_TXCRCEN)
			 | (1<<ENC_BIT_FRMLNEN),
	#endif

	// max framelength 1518
	ENC_REG_MAMXFLL, LO8(1518),
	ENC_REG_MAMXFLH, HI8(1518),

	#ifdef FULL_DUPLEX
	// back-to-back inter packet gap delay time (0x15 for full duplex)
	ENC_REG_MABBIPG, 0x15,
	#else
	// back-to-back inter packet gap delay time (0x12 for half duplex)
	ENC_REG_MABBIPG, 0x12,
	#endif

	// non back-to-back inter packet gap delay time (should be 0x12)
	ENC_REG_MAIPGL, 0x12,

	#ifndef FULL_DUPLEX
	// non back-to-back inter packet gap delay time high (should be 0x0C for half duplex)
	ENC_REG_MAIPGH, 0x0C,
	#endif

	// our mac address
	ENC_REG_MAADR5, MYMAC1,
	ENC_REG_MAADR4, MYMAC2,
	ENC_REG_MAADR3, MYMAC3,
	ENC_REG_MAADR2, MYMAC4,
	ENC_REG_MAADR1, MYMAC5,
	ENC_REG_MAADR0, MYMAC6,

	// disable CLKOUT pin
	ENC_REG_ECOCON, 0x00,

	// end of enc registers marker
	0xFF, 0xFF,

	// now the phy registers (with 2 bytes data each)

	#ifdef FULL_DUPLEX
	// set the PDPXMD full duplex mode bit on the phy
	#define ENC_REG_PHCON1_VALUE (0x0000 | (1 << ENC_BIT_PDPXMD))
	ENC_REG_PHCON1, HI8(ENC_REG_PHCON1_VALUE), LO8(ENC_REG_PHCON1_VALUE),
	#endif

	#ifndef FULL_DUPLEX
	// in half duplex do not loop back transmitted data
	#define ENC_REG_PHCON2_VALUE (0x0000 | (1 << ENC_BIT_HDLDIS))
	ENC_REG_PHCON2, HI8(ENC_REG_PHCON2_VALUE), LO8(ENC_REG_PHCON2_VALUE),
	#endif

	// leds: leda (yellow) rx and tx activity, stretch to 40ms
	//       ledb (green)  link status
	#define ENC_REG_PHCON_VALUE (0x0000 | (1 << ENC_BIT_STRCH) \
			     | (7 << ENC_BIT_LACFG0)               \
			     | (4 << ENC_BIT_LBCFG0))
	ENC_REG_PHLCON, HI8(ENC_REG_PHCON_VALUE), LO8(ENC_REG_PHCON_VALUE),

	// end of config data marker
	0xFF, 0xFF
};

//-----------------------------------------------------------------------------

static void usdelay( unsigned int us )
{
	while( us-- ) {
		// 4 times * 4 cycles gives 16 cyles = 1 us with 16 MHz clocking
		unsigned char i=4;
		// this while loop executes with exact 4 cycles:
		while( i-- ) { asm volatile("nop"); }
	}
}

//-----------------------------------------------------------------------------

static inline void spi_init(void)
{
	// configure pins MOSI, SCK as output
	SPI_DDR |= (1<<SPI_MOSI) | (1<<SPI_SCK);
	// pull SCK high
	SPI_PORT |= (1<<SPI_SCK);

	// configure pin MISO as input
	SPI_DDR &= ~(1<<SPI_MISO);
	SPI_DDR |= (1<<SPI_SS);

	//SPI: enable, master, positive clock phase, msb first, SPI speed fosc/2
	SPCR = (1<<SPE) | (1<<MSTR);
	SPSR = (1<<SPI2X);

	usdelay(10000);
}

static inline void spi_put( unsigned char value )
{
	SPDR = value;
	while( !(SPSR & (1<<SPIF)) ) ;
}

static inline unsigned char spi_get(void)
{
	unsigned char value = SPDR;
	return value;
}

//-----------------------------------------------------------------------------

static inline void enc_reset(void)
{
	enc_select();
	spi_put( ENC_SPI_OP_SC );
	enc_deselect();

	// errata #2: wait for at least 300 us
	usdelay( 1000 );
}

static void enc_clrbits_reg( unsigned char reg, unsigned char bits )
{
	// no automatic bank switching in this function !!!

	unsigned char addr = reg & ENC_REG_ADDR_MASK;

	enc_select();
	spi_put( ENC_SPI_OP_BFC | addr );
	spi_put( bits );
	enc_deselect();
}

static void enc_setbits_reg( unsigned char reg, unsigned char bits )
{
	// no automatic bank switching in this function !!!

	unsigned char addr = reg & ENC_REG_ADDR_MASK;

	enc_select();
	spi_put( ENC_SPI_OP_BFS | addr );
	spi_put( bits );
	enc_deselect();
}

static unsigned char enc_read_reg( unsigned char reg )
{
	unsigned char value;
	unsigned char addr = reg & ENC_REG_ADDR_MASK;

	if( addr < 0x1A ) {
		unsigned char bank = (reg & ENC_REG_BANK_MASK) >> ENC_REG_BANK_SHIFT;
		if( bank != enc_cur_bank ) {
			// need to switch bank first
			enc_clrbits_reg( ENC_REG_ECON1, 0x03 << ENC_BIT_BSEL0 );
			if( bank ) {
				enc_setbits_reg( ENC_REG_ECON1, bank << ENC_BIT_BSEL0 );
			}
			enc_cur_bank = bank;
		}
	}

	enc_select();
	spi_put( ENC_SPI_OP_RCR | addr );
	spi_put( 0x00 );
	if( reg & ENC_REG_WAIT_MASK ) spi_put( 0x00 );
	value = spi_get();
	enc_deselect();

	return value;
}

static void enc_write_reg( unsigned char reg, unsigned char value )
{
	unsigned char addr = reg & ENC_REG_ADDR_MASK;

	if( addr < 0x1A ) {
		unsigned char bank = (reg & ENC_REG_BANK_MASK) >> ENC_REG_BANK_SHIFT;
		if( bank != enc_cur_bank ) {
			// need to switch bank first
			enc_clrbits_reg( ENC_REG_ECON1, 0x03 << ENC_BIT_BSEL0 );
			if( bank ) {
				enc_setbits_reg( ENC_REG_ECON1, bank << ENC_BIT_BSEL0 );
			}
			enc_cur_bank = bank;
		}
	}

	enc_select();
	spi_put( ENC_SPI_OP_WCR | addr );
	spi_put( value );
	enc_deselect();
}

#if 0
static unsigned int enc_read_phyreg( unsigned char phyreg )
{
	unsigned int value;

	enc_write_reg( ENC_REG_MIREGADR, phyreg );
	enc_write_reg( ENC_REG_MICMD, (1<<ENC_BIT_MIIRD) );
	usdelay(10);
	while( enc_read_reg( ENC_REG_MISTAT ) & (1<<ENC_BIT_BUSY) ) ;
	enc_write_reg( ENC_REG_MICMD, 0x00 );
	value = (((unsigned int) enc_read_reg( ENC_REG_MIRDH )) << 8);
	value |= ((unsigned int) enc_read_reg( ENC_REG_MIRDL ));

	return value;
}
#endif

static void enc_write_phyreg( unsigned char phyreg, unsigned int value )
{
	enc_write_reg( ENC_REG_MIREGADR, phyreg );
	enc_write_reg( ENC_REG_MIWRL, LO8(value) );
	enc_write_reg( ENC_REG_MIWRH, HI8(value) );
	usdelay(10);
	while( enc_read_reg( ENC_REG_MISTAT ) & (1<<ENC_BIT_BUSY) ) ;
}

static void enc_read_buf( unsigned char *buf, unsigned int len )
{
	enc_select();
	spi_put( ENC_SPI_OP_RBM );
	for(; len > 0; len--, buf++ ) {
		spi_put( 0x00 );
		*buf = spi_get();
	}
	enc_deselect();
}

static void enc_write_buf( unsigned char *buf, unsigned int len )
{
	enc_select();
	spi_put( ENC_SPI_OP_WBM );
	for(; len > 0; len--, buf++ ) {
		spi_put( *buf );
	}
	enc_deselect();
}

//-----------------------------------------------------------------------------

void enc_send_packet( unsigned int len, unsigned char *buf )
{
	unsigned char ctrl = 0;
	unsigned int ms = 100;

	// wait up to 100 ms for the previos tx to finish
	while( ms-- ) 
	{
		if( !(enc_read_reg( ENC_REG_ECON1 ) & (1<<ENC_BIT_TXRTS)) ) break;
		usdelay( 1000 );
	}

	#ifdef FULL_DUPLEX
	// reset tx logic if TXRTS bit is still on
	if( enc_read_reg( ENC_REG_ECON1 ) & (1<<ENC_BIT_TXRTS) ) 
	{
	#else
	// errata #12: reset tx logic
	if( 1 )
	{
	#endif
		enc_setbits_reg( ENC_REG_ECON1, (1<<ENC_BIT_TXRST) );
		enc_clrbits_reg( ENC_REG_ECON1, (1<<ENC_BIT_TXRST) );
	}

	// setup write pointer
	enc_write_reg( ENC_REG_EWRPTL, LO8(ENC_TX_BUFFER_START) );
	enc_write_reg( ENC_REG_EWRPTH, HI8(ENC_TX_BUFFER_START) );

	// end pointer (points to last byte) to start + len
	enc_write_reg( ENC_REG_ETXNDL, LO8(ENC_TX_BUFFER_START+len) );
	enc_write_reg( ENC_REG_ETXNDH, HI8(ENC_TX_BUFFER_START+len) );

	// enc requires 1 control byte before the package
	enc_write_buf( &ctrl, 1 );

	// copy packet to enc buffer
	enc_write_buf( buf, len );

	// clear TXIF flag
	enc_clrbits_reg( ENC_REG_EIR, (1<<ENC_BIT_TXIF) );

	// start transmission by setting the TXRTS bit
	enc_setbits_reg( ENC_REG_ECON1, (1<<ENC_BIT_TXRTS) );
}

unsigned int enc_receive_packet( unsigned int bufsize, unsigned char *buf )
{
	unsigned char rxheader[6];
	unsigned int len, status;
	unsigned char u;

	// check rx packet counter
	u = enc_read_reg( ENC_REG_EPKTCNT );
	
	if( u == 0 ) 
	{
		// packetcounter is 0, there is nothing to receive, go back
		return 0;
	}

	//set read pointer to next packet
	enc_write_reg( ENC_REG_ERDPTL, LO8(enc_next_packet_ptr) );
	enc_write_reg( ENC_REG_ERDPTH, HI8(enc_next_packet_ptr) );

	// read enc rx packet header
	enc_read_buf( rxheader, sizeof(rxheader) );
	enc_next_packet_ptr  =             rxheader[0];
	enc_next_packet_ptr |= ((unsigned)(rxheader[1])) << 8;
	len                  =             rxheader[2];
	len                 |= ((unsigned)(rxheader[3])) << 8;
	status               =             rxheader[4];
	status              |= ((unsigned)(rxheader[5])) << 8;

	// added by Sjors: reset the ENC when needed
	// If the receive OK bit is not 1 or the zero bit is not zero, or the packet is larger then the buffer, reset the enc chip and SPI
	if ((!(status & (1<<7))) || (status & 0x8000) || (len > bufsize))
	{ 
		enc_init();
	}
	
	// skip the checksum (4 bytes) at the end of the buffer
	len -= 4;

	// if the application buffer is to small, we just truncate
	if( len > bufsize ) len = bufsize;

	// now read the packet data into buffer
	enc_read_buf( buf, len );

	// adjust the ERXRDPT pointer (= free this packet in rx buffer)
	if(enc_next_packet_ptr-1 > ENC_RX_BUFFER_END || enc_next_packet_ptr-1 < ENC_RX_BUFFER_START ) 
	{
		enc_write_reg( ENC_REG_ERXRDPTL, LO8(ENC_RX_BUFFER_END) );
		enc_write_reg( ENC_REG_ERXRDPTH, HI8(ENC_RX_BUFFER_END) );
	}
	else
	{
		enc_write_reg( ENC_REG_ERXRDPTL, LO8(enc_next_packet_ptr-1) );
		enc_write_reg( ENC_REG_ERXRDPTH, HI8(enc_next_packet_ptr-1) );
	}

	// trigger a decrement of the rx packet counter
	// this will clear PKTIF if EPKTCNT reaches 0
	enc_setbits_reg( ENC_REG_ECON2, (1<<ENC_BIT_PKTDEC) );

	// return number of bytes written to the buffer
	return len;
}

//-----------------------------------------------------------------------------

void enc_init(void)
{
	int i=0;
	unsigned int u;
	unsigned char r, d;

	// config enc chip select as output and deselect enc
	ENC_DDR  |= (1<<ENC_CS);
	ENC_PORT |= (1<<ENC_CS);

	// init spi
	spi_init();

	// send a reset command via spi to the enc
	enc_reset();

	// wait for the CLKRDY bit
	while( !(enc_read_reg( ENC_REG_ESTAT ) & (1<<ENC_BIT_CLKRDY)) ) ;

	// get enc revision id
	enc_revid = enc_read_reg( ENC_REG_EREVID );

	// setup mymac variable
	mymac[0] = MYMAC1;
	mymac[1] = MYMAC2;
	mymac[2] = MYMAC3;
	mymac[3] = MYMAC4;
	mymac[4] = MYMAC5;
	mymac[5] = MYMAC6;

	// setup enc registers according to the enc_configdata struct
	while(1) {
		r = pgm_read_byte( &enc_configdata[i++] );
		d = pgm_read_byte( &enc_configdata[i++] );
		if( r == 0xFF && d == 0xFF ) break;
		enc_write_reg( r, d );
	}
	// now the phy registers
	while(1) {
		r = pgm_read_byte( &enc_configdata[i++] );
		d = pgm_read_byte( &enc_configdata[i++] );
		if( r == 0xFF && d == 0xFF ) break;
		u = (((unsigned int)d) << 8);
		d = pgm_read_byte( &enc_configdata[i++] );
		u |= d;
		enc_write_phyreg( r, u );
	}

	// setup receive next packet pointer
	enc_next_packet_ptr = ENC_RX_BUFFER_START;

	// configure the enc interrupt sources
	enc_write_reg( ENC_REG_EIE, (1 << ENC_BIT_INTIE)  | (1 << ENC_BIT_PKTIE)
				  | (0 << ENC_BIT_DMAIE)  | (0 << ENC_BIT_LINKIE)
				  | (0 << ENC_BIT_TXIE)   | (0 << ENC_BIT_WOLIE)
				  | (0 << ENC_BIT_TXERIE) | (0 << ENC_BIT_RXERIE));

	// enable receive
	enc_setbits_reg( ENC_REG_ECON1, (1<<ENC_BIT_RXEN) );

	// the enc interrupt on the atmega is still disabled
	// needs to get enabled with ETH_INT_ENABLE;
}
