#include <stdint.h>
#include <stddef.h>
#include <util/delay.h>
#include "pictiva128x64.h"
#include "LucidaTypewriter.h"
#include "LucidaTypewriter_7.h"
#include "ArialBlack_16.h"

#define OLED_USE_SPI 0

#if OLED_USE_SPI
static void writeToOLED(uint8_t data)
{
	CLRPIN(OLED_CS);
	SPDR = data;
	while (!(SPSR & _BV(SPIF)));
	SETPIN(OLED_CS);
}
#else
static void writeToOLED(uint8_t data)
{
	PORTC = data;
	CLRPIN(OLED_CS);
	SETPIN(OLED_E);
	CLRPIN(OLED_E);
	SETPIN(OLED_CS);
}
#endif

static void writeCommand(uint8_t cmd)
{
	CLRPIN(OLED_DC);
	writeToOLED(cmd);
}

static void writeData(uint8_t data)
{
	SETPIN(OLED_DC);
	writeToOLED(data);
}

void oled_init()
{
	uint8_t n;

#if OLED_USE_SPI
	SPCR = _BV(SPE) | _BV(MSTR);
#else
	CLRPIN(OLED_RW);
	CLRPIN(OLED_E);
#endif

	CLRPIN(OLED_RESET);
	_delay_ms(5);
	SETPIN(OLED_RESET);

	_delay_ms(25);
	_delay_ms(25);
	_delay_ms(25);
	_delay_ms(25);
	SETPIN(OLED_VCC);

	writeCommand(0x15);
	writeCommand(0x00);
	writeCommand(0x3F);

	writeCommand(0x75);
	writeCommand(0x00);
	writeCommand(0x3F);

	writeCommand(0x81);
	writeCommand(0x33);

	writeCommand(0x86);

	writeCommand(0xA0);
	writeCommand(0x56); // 0x52

	writeCommand(0xA1);
	writeCommand(0x00);

	writeCommand(0xA2);
	writeCommand(0x4C); // 0x40

	writeCommand(0xA4);

	writeCommand(0xA8);
	writeCommand(0x3F);

	writeCommand(0xB1);
	writeCommand(0x22);

	writeCommand(0xB2);
	writeCommand(0x46);

	writeCommand(0xB3);
	writeCommand(0x41);

	writeCommand(0xBF);
	writeCommand(0x0D);

	writeCommand(0xBE);
	writeCommand(0x10);

	writeCommand(0xBC);
	writeCommand(0x0B);

#if 1
	writeCommand(0xB8);
	writeCommand(0x01);
	writeCommand(0x11);
	writeCommand(0x22);
	writeCommand(0x32);
	writeCommand(0x43);
	writeCommand(0x54);
	writeCommand(0x65);
	writeCommand(0x76);
#endif

	writeCommand(0xAD);
	writeCommand(0x02);

	/* Clear data RAM */
#if 1
	for (n = 0; n < 64; n++) {
		uint8_t i;

		for (i = 0; i < 64; i++) {
			writeData(0);
		}
	}
#endif
	
	/* Display on */
	writeCommand(0xAF);

#if 0
	{
		uint8_t i, n, x;
		int8_t dir;

		_delay_ms(10);
#if 0
		for (n = 1; n <= 64; n++) {
			for (i = 0; i < 64; i++) {
				if (n % 8 == 0)
					writeData(0b10100000);
				else
					writeData(0);
			}
		}
#endif

		x = 0;
		dir = 1;
		while (1) {
			writeCommand(0x75);
			writeCommand(0);
			writeCommand(63);

			writeCommand(0x15);
			writeCommand(0);
			writeCommand(63);

			for (n = 0; n < 64; n++) {
				for (i = 0; i < 64; i++) {
					writeData(0xa0);
				}
			}
			while(1);

			//_delay_ms(1);
			
			if (dir == 1) {
				x++;
				if (x == 14) {
					x=14;
					dir = -1;
				}
			}
			else {
				if (x == 0) {
					x = 1;
					dir = 1;
				}
				else x--;
			}
		}
	}
#endif
}

static uint8_t su8GlblFontHeight;
static uint8_t su8GlblFontFirstChar;
static uint8_t su8GlblFontCharCount;
static void* spGlblFont;

struct FontDef {
	uint16_t   fontSize;
	uint8_t    fontWidth;
	uint8_t    fontHeight;
 	uint8_t    fontFirstChar;
 	uint8_t    fontCharCount;
 	uint8_t    fontCharWidths[1];
	uint8_t    font_data[];
};

static void setFont(void* data)
{
	spGlblFont = data;
	su8GlblFontHeight = pgm_read_byte_near(data + offsetof(struct FontDef, fontHeight));
	su8GlblFontFirstChar = pgm_read_byte_near(data + offsetof(struct FontDef, fontFirstChar));
	su8GlblFontCharCount = pgm_read_byte_near(data + offsetof(struct FontDef, fontCharCount));
}

uint8_t drawText(uint8_t xPos, uint8_t yPos, uint8_t brightness, uint8_t ch)
{
	int8_t n, i;
	uint8_t bytes;
	uint8_t fontWidth;
	uint8_t col1, col2;
	uint8_t firstBlank;
	void* addr;

	ch 			= ch - su8GlblFontFirstChar;
	fontWidth	= pgm_read_byte_near(spGlblFont + offsetof(struct FontDef, fontCharWidths) + ch);
	bytes		= su8GlblFontHeight >> 3;
	if ((bytes << 3) < su8GlblFontHeight) bytes++;
	n 			= xPos >> 1;
	firstBlank	= ((n << 1) < xPos) ? 1 : 0;
	xPos		= n;

	addr = spGlblFont + offsetof(struct FontDef, fontCharWidths) + su8GlblFontCharCount;

	for (i = 0; i < ch; i++) {
		addr += pgm_read_byte_near(spGlblFont + offsetof(struct FontDef, fontCharWidths) + i) * bytes;
	}

	for (n = 0; n < bytes; n++) {
		uint8_t bits;

		addr += n * fontWidth;
		bits  = su8GlblFontHeight - 8*n;
		if (bits > 8) bits = 8;

		writeCommand(0x15);
		writeCommand(xPos);
		writeCommand(0x3F);
		writeCommand(0x75);
		writeCommand(yPos);
		writeCommand(yPos + bits - 1);

		for (i = 0; i < fontWidth; i++) {
			uint8_t j;
			
			if (i == 0 && firstBlank) {
				col1 = 0;
				i--;
			}
			else {
				col1 = pgm_read_byte_near(addr + i);
			}

			if (i + 1 < fontWidth) {
				i++;
				col2 = pgm_read_byte_near(addr + i);
			}
			else col2 = 0;

			col1 >>= 8 - bits;
			col2 >>= 8 - bits;

			for (j = 0; j < bits; j++) {
				uint8_t ram;

				ram  = (col1 & 0x01) ? (brightness << 4) : 0;
				ram |= (col2 & 0x01) ? brightness : 0;
				writeData(ram);

				col1 >>= 1;
				col2 >>= 1;
			}
		}
		
		yPos += bits;
	}

	return fontWidth;
}


void writeText(uint8_t x, uint8_t y, uint8_t brightness, const char* text)
{
	uint8_t i=0;

	while (1) {
		uint8_t ch;

		ch = text[i];
		if (ch == 0) {
			break;
		}

		x += drawText(x, y, brightness, ch) + 1;
		i++;
	}
}

void selectFont(uint8_t nr)
{
	switch (nr) {
		case 0:
			setFont((void*)LucidaTypewriter_7);
			break;
		case 1:
			setFont((void*)lucidaTypewriter_12_Font);
			break;
		case 2:
			setFont((void*)ArialBlack_16);
			break;
	}
}
