#include "capture.h"
#include "main.h"
#include "ui.h"

#include <fcntl.h>
#include <errno.h>
#include <libv4l2.h>
#include <sys/stat.h>


#include <QMessageBox>
#include <QMutexLocker>

#include <QDebug>

static int xioctl (int fd, int request, void * arg)
{
    int r;
    do r = v4l2_ioctl (fd, request, arg);
    while (-1 == r && EINTR == errno);
    return r;
}

Capture::Capture (QWidget *p) :
     _parent (p),
     _video_device (-1),
    _video_buffer (0),
    _buffer_size  (0),
    _abort    (false),
    _cinit    (false),
    _width  (1),
    _height (1)
{
    _dialog.setLayout (&_grid);
    _reset.setText ("set default values");

    // redirect action fron run() to captureEvent()
    connect (this, SIGNAL(captureResult (int)), this, SLOT(captureEvent (int)));
}

#define CAPTURE_FRAME   0
#define CAPTURE_ERROR   1
#define CAPTURE_WAIT    2

void Capture::captureEvent (int r)
{
    switch (r)
    {
    case CAPTURE_FRAME:
        emit buffer_read (static_cast <uchar*> (_video_buffer), _width, _height);
        break;
    case CAPTURE_ERROR:
            QMessageBox::warning (0, "ERROR", "data lost\nDevice will be closed.");
            closeDevice();
        break;
    default:
        break;
    }
    // wake up run(), which is in _condition.wait()
    _condition.wakeAll();
}

int Capture::openDeviceDialog ()
{
    OpenDeviceDialog dialog (0);
    if (dialog.exec())
        return openDevice (dialog.devicename());
    else
        return 0;
}

int Capture::openDeviceFromSettings ()
{
    QString name = globalsettings.value("device", "/dev/video0").toString();
    return openDevice (name);
}

int Capture::openDevice (const QString& devicename)
{
    closeDevice();

    QMutexLocker locker (&_mutex);

    // open the video device:
    struct stat st;                             // file type, defined in sys/stat.h

    if ((stat (devicename.toAscii(), &st) == -1) ||
        (!S_ISCHR (st.st_mode)) ||
        ((_video_device = v4l2_open(devicename.toAscii(), O_RDWR | O_NONBLOCK)) == -1))
    {
        QMessageBox::warning (_parent, "Warning", QString ("cannot open device %1").arg(devicename));
        return false;
    }

    // initialize the video device:

    struct v4l2_format fmt;
    memset (&fmt, 0, sizeof (fmt));

    fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
    fmt.fmt.pix.width =  640;
    fmt.fmt.pix.height = 480;
    fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_RGB24;
    fmt.fmt.pix.field = V4L2_FIELD_INTERLACED;

    if (-1 == xioctl (_video_device, VIDIOC_S_FMT, &fmt))
    {
        QMessageBox::warning (_parent, "Warning", QString ("cannot talk to device %1").arg(devicename));
        locker.unlock();    // otherwise closeDevice() can't proceed
        closeDevice();
        return false;
    }

    // VIDIOC_S_FMT may change width and height
    _width  = fmt.fmt.pix.width;
    _height = fmt.fmt.pix.height;
    _video_buffer = malloc (fmt.fmt.pix.sizeimage);
    _buffer_size = fmt.fmt.pix.sizeimage;

    if (!_video_buffer) exit (EXIT_FAILURE);

    globalsettings.setValue ("device", devicename);

    return true;
}

void Capture::closeDevice()
{
    QMutexLocker locker (&_mutex);

    if (_video_buffer) free (_video_buffer);
    if (_video_device) v4l2_close (_video_device);

    _video_buffer = 0;
    _buffer_size  = 0;
    _video_device = -1;
}


Capture::~Capture ()
{
    _abort = true;
    closeDevice();
    for (int i=0; i<_controls.size(); i++) delete _controls[i];
    _condition.wakeAll();
    wait();
}


void Capture::run ()
{
    int result;

    for (int i=0;;i++)
    {

        _mutex.lock();

        // check, if thread shall exit (_abort is set by ~Capture)
        if (_abort == true)
        {
            _mutex.unlock();
            return;
        }

        // read in a new video frame
        if (_video_device == -1)
        {
            // if the video device is not opened, wait 1/4 second and try again
            result = CAPTURE_WAIT;
            msleep (250);
        }
        else
        {
            // since read() returns immediately with a frame, use select() and wait for a new
            // frame - this dramatically reduces the processor load
            fd_set fds;
            struct timeval tv;
            FD_ZERO(&fds);
            FD_SET(_video_device, &fds);
            tv.tv_sec = 2;
            tv.tv_usec = 0;
            select(_video_device+1, &fds, NULL, NULL, &tv);

            if ((v4l2_read (_video_device, _video_buffer, _buffer_size) == -1) && (errno != EAGAIN))
            {
                result = CAPTURE_ERROR;
                msleep (250);
            }
            else
            {
                result = CAPTURE_FRAME;
            }
        }


        // signal result to main thread and wait
        emit (captureResult (result));
        _condition.wait (&_mutex);
        _mutex.unlock();
    }
}

void Capture::hideControls ()
{
    _dialog.hide();
}

void Capture::showControls ()
{
    if (!_cinit)
    {
        struct v4l2_queryctrl queryctrl;
        struct v4l2_querymenu querymenu;
        struct v4l2_control   cntrl;

        int row=0;
        int col=0;

        for (int i=V4L2_CID_BASE; i<V4L2_CID_LASTP1; i++)
        {
            memset (&queryctrl, 0, sizeof (queryctrl));
            memset (&cntrl, 0, sizeof (cntrl));

            queryctrl.id = i;
            cntrl.id = i;

            if (xioctl (_video_device, VIDIOC_QUERYCTRL, &queryctrl) == 0)
            {
                if (!(queryctrl.flags & V4L2_CTRL_FLAG_DISABLED))
                {
                    if (xioctl (_video_device, VIDIOC_G_CTRL, &cntrl) == 0)
                    {
                        Control* c = new Control (queryctrl);
                        _controls.append (c);

                        switch (queryctrl.type)
                        {
                        case V4L2_CTRL_TYPE_INTEGER:
                        case V4L2_CTRL_TYPE_INTEGER64:
                            _grid.addWidget (c->label(), row, col++, 1, 1);
                            _grid.addWidget (c->widget().spin, row++, col--, 1, 1);
                            connect (c, SIGNAL (valueChanged(int, int)), this, SLOT (writeControl (int, int)));
                            break;
                        case V4L2_CTRL_TYPE_BOOLEAN:
                            _grid.addWidget (c->label(), row, col++, 1, 1);
                            _grid.addWidget (c->widget().check, row++, col--, 1, 1);
                            connect (c, SIGNAL (valueChanged(int, int)), this, SLOT (writeControl (int, int)));
                            break;
                        case V4L2_CTRL_TYPE_MENU:
                            _grid.addWidget (c->label(), row, col++, 1, 1);
                            _grid.addWidget (c->widget().combo, row++, col--, 1, 1);

                            // get items for this menu type control
                            memset (&querymenu, 0, sizeof (querymenu));
                            querymenu.id = queryctrl.id;
                            for (int m=queryctrl.minimum; m<=queryctrl.maximum; m++)
                            {
                                querymenu.index = m;
                                if (xioctl (_video_device, VIDIOC_QUERYMENU, &querymenu) == 0)
                                    c->addItem (reinterpret_cast <char*>(querymenu.name));
                            }
                            connect (c, SIGNAL (valueChanged(int, int)), this, SLOT (writeControl (int, int)));
                            break;
                        default:
                            fprintf (stderr, "unknown control type: %u\n", queryctrl.type);
                        }
                    }

                }
            }
        }
        _grid.addWidget (&_reset, row, 1);
        connect (&_reset, SIGNAL(pressed()), this, SLOT(resetControls()));
    }
    _cinit = true;
    _dialog.show();
}

void Capture::resetControls ()
{
    for (int i=0; i<_controls.size(); i++) _controls[i]->setDefault();
}


void Capture::writeControl (int id, int value)
{
    struct v4l2_control cntrl;
    cntrl.id = id;
    cntrl.value = value;
    xioctl (_video_device, VIDIOC_S_CTRL, &cntrl);
}

