#include "ui.h"

#include "main.h"
#include "ui_mainwindow.h"
#include "ui_opendialog.h"

#include <QImage>
#include <QPixmap>
#include <QPainter>
#include <QMouseEvent>
#include <QKeyEvent>

// Screen is a subclassed QLabel
// QLabel's ability to display QPixmaps is used to print the video frame
// a modified paintEvent draws additional items (like text) into the video

Screen::Screen (QWidget *p)
: _move_target (false)
{
    setFocusPolicy (Qt::ClickFocus);
}

void Screen::mousePressEvent (QMouseEvent* event)
{
    if (global.target_lock) return;

    global.target_x = event->pos().x();
    global.target_y = event->pos().y();
    _move_target = true;
}

void Screen::mouseReleaseEvent (QMouseEvent* event)
{
    if (!_move_target) return;

    global.target_x = event->pos().x();
    global.target_y = event->pos().y();
    _move_target = false;
}

void Screen::mouseMoveEvent (QMouseEvent* event)
{
    if (!_move_target) return;

    global.target_x = event->pos().x();
    global.target_y = event->pos().y();
}

Screen::~Screen () {}

void Screen::paintEvent (QPaintEvent * event)
{
    QLabel::paintEvent (event);
    QPainter painter(this);
    painter.setPen(Qt::red);

    QRect view = painter.viewport();

    for (int i=1; i <=global.target_count; i++)
    {
        QRect circle (global.target_x - global.target_inc*i,
                      global.target_y - global.target_inc*i,
                      global.target_inc*2*i, global.target_inc*2*i);

        painter.drawEllipse (circle);
    }
    painter.drawLine (global.target_x, 0, global.target_x, view.height());
    painter.drawLine (0, global.target_y, view.width(), global.target_y);
}

// MainWindow displays the UI, connects buttons and starts the capture-thread
// the SLOT show_buffer gets called by Capture::buffer_read and updates
// the image of Screen, which in turn is drawn by Screen's paintEvent



MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{

    ui->setupUi(this);


    global.target_x = globalsettings.value ("target/x", 320).toInt();
    global.target_y = globalsettings.value ("target/y", 240).toInt();
    global.target_inc = globalsettings.value ("target/inc", 30).toInt();
    global.target_count = globalsettings.value ("target/count", 3).toInt();
    global.target_lock = true;

    ui->_lockBox->setChecked (true);
    ui->_incBox->setValue (global.target_inc);
    ui->_countBox->setValue (global.target_count);

    capture = new Capture (this);
    capture->openDeviceFromSettings ();

    // signal, that tells the main thread to read out the video buffer
    connect (capture, SIGNAL(buffer_read (uchar *, int, int)), this, SLOT(show_buffer (uchar *, int, int)));

    // buttons and widgets
    connect (ui->_controlButton, SIGNAL(pressed ()), capture, SLOT(showControls ()));
    connect (ui->_deviceButton, SIGNAL(pressed ()),  capture, SLOT(openDeviceDialog ()));

    connect (ui->_lockBox, SIGNAL(stateChanged (int)),  this, SLOT(lockTarget (int)));
    connect (ui->_incBox,  SIGNAL(valueChanged (int)),  this, SLOT(incTarget (int)));
    connect (ui->_countBox,  SIGNAL(valueChanged (int)),  this, SLOT(countTarget (int)));

    // start the capture-thread
    capture->start();

}

MainWindow::~MainWindow()
{
    globalsettings.setValue("target/x", global.target_x);
    globalsettings.setValue("target/y", global.target_y);
    globalsettings.setValue("target/inc", global.target_inc);
    globalsettings.setValue("target/count", global.target_count);


    delete ui;
    delete capture;
}

void MainWindow::lockTarget (int b)
{
    if (b == Qt::Checked) global.target_lock = true;
    else                  global.target_lock = false;
}

void MainWindow::incTarget (int i)
{
    global.target_inc = i;
}

void MainWindow::countTarget (int c)
{
    global.target_count = c;
}

void MainWindow::closeEvent (QCloseEvent* event)
{
    capture->hideControls();
    QMainWindow::closeEvent (event);
}

void MainWindow::keyPressEvent (QKeyEvent *event)
{
    if (global.target_lock) return;

    if      (event->key() == Qt::Key_Right) global.target_x++;
    else if (event->key() == Qt::Key_Left)  global.target_x--;
    else if (event->key() == Qt::Key_Up)    global.target_y--;
    else if (event->key() == Qt::Key_Down)  global.target_y++;
}

void MainWindow::show_buffer (uchar *buffer, int width, int height)
{
    #define FORMAT QImage::Format_RGB888
//    #define FORMAT QImage::Format_ARGB8555_Premultiplied
//    #define FORMAT QImage::Format_ARGB6666_Premultiplied
//    #define FORMAT QImage::Format_RGB666
//    #define FORMAT QImage::Format_ARGB8565_Premultiplied

    QImage image (buffer, width, height, FORMAT);
    ui->_screenFrame->setPixmap(QPixmap::fromImage(image));
}

OpenDeviceDialog::OpenDeviceDialog (QWidget *p)
    : QDialog (p),
    ui(new Ui::OpenDeviceDialog)
{
    ui->setupUi(this);
}

OpenDeviceDialog::~OpenDeviceDialog ()
{
    delete ui;
}

QString OpenDeviceDialog::devicename ()
{
    return QString ("/dev/%1").arg(ui->_deviceName->text());
}

