/* ---------------------------------------------
Timer_V1

Timer fr UV-Belichter
SN, 2005-08-29

Funktion:
- Vorgabezeit wird im 7-Segmentdisplay angezeigt
- Button drehen: Zeitvorgabe in Sekundenschritten
- Button gedrckt halten und drehen: Zeitvorgabe in Minutenschritten
- Button kurz drcken: gespeicherte Zeit wiederholen
- Button 1..3 s gedrckt: Start beim loslassen
- Button >3s gedrckt: Speichern der eingestellen Zeit in EEPROM,
  Anzeige zeigt noch 1s lang 'SAFE'
- Button gedrckt whrend Timer luft: sofortiger Stopp

--------------------------------------------- */

#include <inttypes.h>
#include <stdlib.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/delay.h>
#include <avr/eeprom.h>

#define EEPROM __attribute__ ((section (".eeprom")))

#define LIGHTOFF  	(_BV(PB4))
#define LIGHTON  	0
#define PushBTN		(_BV(PB5))
#define ENC_A		(PINB & _BV(PINB6))
#define ENC_B		(PINB & _BV(PINB7))

typedef struct
{
	uint8_t bitmask;
	void (*fnActionRise)(void);
	void (*fnActionFall)(void);
} ACTION;

/* --------------------------------------------- */

const unsigned short digtab[10] = {0x7e, 0x06, 0x6d, 0x4f, 0x17, 0x5b, 0x7b, 0x0e, 0x7f, 0x5f};

/* --------------------------------------------- */

int8_t eePreset_time_s EEPROM = 30;
int8_t eePreset_time_m EEPROM = 1;

/* --------------------------------------------- */


volatile int	enc_delta;		// -128 ... 127
uint8_t PB_old;
uint8_t divider;
uint8_t divider_s;
uint8_t actdigit;  
uint8_t timer_running;
uint8_t digits[4];
uint16_t btnIsDown;
uint8_t displaySafe;
uint8_t OutStatus;

int8_t preset_time_s;
int8_t preset_time_m;
char enc_last = 0x01;

void Btn0Down(void)
{
	if (timer_running)
	{
		timer_running=0;
		OutStatus = LIGHTOFF;
	}
	else
		btnIsDown=1;
}

void Btn0Up(void)
{
	{
		if (btnIsDown > 300)  // >3s : Zeit in EEPROM speichern
		{
			cli();
			eeprom_write_byte(&eePreset_time_s, preset_time_s);
			eeprom_write_byte(&eePreset_time_m, preset_time_m);
			sei();
			displaySafe=100;	// Safe Meldung 1s stehen lassen
		}
		else
		if (btnIsDown > 50)  // >0,5s : Timer starten
		{
			timer_running=1;
			OutStatus=LIGHTON;
		}
		else
		if (btnIsDown > 2)  // >0,02s : letzte Zeit aus EEPROM holen
		{
			cli();
			preset_time_s=eeprom_read_byte(&eePreset_time_s);
			preset_time_m=eeprom_read_byte(&eePreset_time_m);
			sei();
		}
	}
		
	btnIsDown=0;
}

const ACTION ActionTab[]  =
{
	{ PushBTN, Btn0Up, Btn0Down },
	{ 0, NULL, NULL }
};

void CheckIO(void)
{
	ACTION* p;
	p=ActionTab;
	while(p->bitmask)
	{
		if ((PINB ^ PB_old) & PB_old & p->bitmask)
			if  (p->fnActionFall!=NULL)
				p->fnActionFall();

		if ((PINB ^ PB_old) & PINB & p->bitmask)
			if  (p->fnActionRise!=NULL)
				p->fnActionRise();
		p++;
	}
	PB_old=PINB;
}

/* --------------------------------------------- */

void ioinit (void)
{
    /* set PortB 0..4 as output */
    DDRB = 0x1f;
	PORTB = _BV(5) | LIGHTOFF; //actdigit;

	
	PB_old=PINB;

	int i=0;
	if( ENC_A )
		i = 1;

	if( ENC_B )
		i ^= 3;				// convert gray to binary
	enc_last = i;

	DDRD = 0xff;	// Segment Ausgnge
	// PORTD = 0xff;	// test: alle Segmente an
	PORTD = 0x01;


    TCCR1B = _BV(WGM12) | _BV(CS10);	// int.Clock, no Prescaler, CTC Mode
	OCR1A = 7373;  						// 2 ms
	TCNT1 = 0;

    timer_enable_int (_BV(OCIE1A));
	sei();
}

/* --------------------------------------------- */

SIGNAL (SIG_TIMER1_COMPA)
{
	divider++;

	actdigit++;
	if (actdigit>3)
		actdigit=0;

	int i = 0;

	if( ENC_A )
		i = 1;

	if( ENC_B )
		i ^= 3;				// convert gray to binary

	i -= enc_last;			// difference new - last

	if( i & 1 )				// bit 0 = value (1)
	{
		enc_last += i;			// store new as next last

		enc_delta += (i & 2) - 1;		// bit 1 = direction (+/-)
	}

	if (PINB & PushBTN)
	{
		if (enc_delta>0)
		{
			preset_time_s += enc_delta;
			while (preset_time_s > 59)
			{
				preset_time_s -= 60;
				preset_time_m++;
			}
		}
		
		if (enc_delta<0 && (preset_time_s>0 || preset_time_m>0))
		{
			preset_time_s += enc_delta;
			while (preset_time_s < 0)
			{
				preset_time_s += 60;
				if (preset_time_m > 0)
					preset_time_m--;
			}
		}
	}
	else
	{
		if (enc_delta>0)
		{
			preset_time_m += enc_delta;
			if (preset_time_m > 59)
			{
				preset_time_m = 59;
			}
			btnIsDown=0;
		}
		
		if (enc_delta<0)
		{
			preset_time_m += enc_delta;
			if (preset_time_m < 0)
			{
				preset_time_m = 0;
			}
			btnIsDown=0;
		}
	}
	
	enc_delta = 0;
		
	if (divider==5)	// 10 ms
	{
		if (btnIsDown)
			btnIsDown++;
			
		// Segmentbytes setzen
		if (displaySafe || btnIsDown>300)
		{
			digits[3] = 0x79;
			digits[2] = 0x39;
			digits[1] = 0x3f;
			digits[0] = 0x5b;
			if (displaySafe)
				displaySafe--;
		}
		else
		{
			digits[3] = digtab[preset_time_s % 10];
			digits[2] = digtab[(preset_time_s/10) % 10];
			digits[1] = digtab[preset_time_m % 10];
			digits[0] = digtab[(preset_time_m/10) % 10];
		}
		
		divider = 0;
		
		// Sekundentaktung
		divider_s++;
		if (divider_s==100)
		{
			if (timer_running)
			{
				if (preset_time_s > 0)
				{
					preset_time_s--;
				}
				else 
				{
					if (preset_time_m > 0)
					{
						preset_time_m--;
						preset_time_s=59;
					}
				}
				
				if (preset_time_m==0 && preset_time_s==0)
				{
					timer_running=0;
					OutStatus=LIGHTOFF;
				}
			}
			divider_s=0;
		}
	}
	
	// Ausgnge setzen, ein Digit anwhlen und zugehrige Segmente setzen
	PORTB = _BV(5) | _BV(actdigit) | OutStatus;
	PORTD = digits[actdigit];
	
}



int main (void)
{
	divider=0;
	divider_s=0;
	actdigit=0;
	timer_running=0;
	displaySafe=0;
	OutStatus=LIGHTOFF;

	// letzte Zeit aus EEPROM holen
	preset_time_s=eeprom_read_byte(&eePreset_time_s);
	preset_time_m=eeprom_read_byte(&eePreset_time_m);
	
	
    ioinit ();

    /* loop forever */
    for (;;)
	{
	    CheckIO();
		_delay_ms(20.0);
    }
}
