
#include <stdio.h>
#include "usart.h"
#include "stm32f4xx_usart.h"
#include "stm32f4xx_gpio.h"
#include "stm32f4xx.h"
#include "stm32f4xx_rcc.h"
#include "misc.h"

extern void PrintChar(char c);

// Buffersizes must be 2^n
#define TBUFSIZE   256
#define RBUFSIZE   256                // Probably enough
#define TMASK     (TBUFSIZE-1)
#define RMASK     (RBUFSIZE-1)
#define RBUFLEN   (uint8_t)(r_in - r_out)
#define TBUFLEN   (uint8_t)(t_in - t_out)

//-------------------------------------------
// Declaration of local variables
//-------------------------------------------

volatile uint8_t tbuf[TBUFSIZE];
volatile uint8_t rbuf[RBUFSIZE];
volatile uint8_t rmsg  = 0;
volatile uint8_t t_in  = 0;
volatile uint8_t t_out = 0;
volatile uint8_t r_in  = 0;
volatile uint8_t r_out = 0;
volatile uint8_t echo  = 1;

#ifdef __GNUC__
  /* With GCC/RAISONANCE, small printf (option LD Linker->Libraries->Small printf
     set to 'Yes') calls __io_putchar() */
  #define PUTCHAR_PROTOTYPE int __io_putchar(int ch)
#else
  #define PUTCHAR_PROTOTYPE int fputc(int ch, FILE *f)
#endif /* __GNUC__ */

void USART_NVIC_Config(void);

/*******************************************************************************
* Function Name  : USART_Configuration
* Description    : Configure Open_USART 
* Input          : None
* Output         : None
* Return         : None
* Attention		 : None
*******************************************************************************/
void USART_Configuration(void)
{ 												
  GPIO_InitTypeDef GPIO_InitStructure;
  USART_InitTypeDef USART_InitStructure; 

  RCC_AHB1PeriphClockCmd(Open_USART_TX_GPIO_CLK,ENABLE);
  RCC_AHB1PeriphClockCmd(Open_USART_RX_GPIO_CLK,ENABLE);

  if ((Open_USART_CLK==RCC_APB2Periph_USART1)||(Open_USART_CLK==RCC_APB2Periph_USART6))
  {
    RCC_APB2PeriphClockCmd(Open_USART_CLK,ENABLE);
  }
  else
  {
    RCC_APB1PeriphClockCmd(Open_USART_CLK,ENABLE);
  }

  GPIO_PinAFConfig(Open_USART_TX_GPIO_PORT, Open_USART_TX_SOURCE, Open_USART_TX_AF);
  GPIO_PinAFConfig(Open_USART_RX_GPIO_PORT, Open_USART_RX_SOURCE, Open_USART_RX_AF);

  /*
  *  Open_USART_TX -> PA9 , Open_USART_RX -PA10
  */
  GPIO_InitStructure.GPIO_Pin = Open_USART_TX_PIN;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;

  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_UP;
  GPIO_Init(Open_USART_TX_GPIO_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = Open_USART_RX_PIN;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_OD;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_Init(Open_USART_RX_GPIO_PORT, &GPIO_InitStructure);

/*
   		 USARTx configured as follow:
         - BaudRate = 115200 baud  
		 - Word Length = 8 Bits
         - One Stop Bit
         - No parity
         - Hardware flow control disabled (RTS and CTS signals)
         - Receive and transmit    
 */

  USART_InitStructure.USART_BaudRate = 115200;
  USART_InitStructure.USART_WordLength = USART_WordLength_8b;
  USART_InitStructure.USART_StopBits = USART_StopBits_1;
  USART_InitStructure.USART_Parity = USART_Parity_No;
  USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
  USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
  USART_Init(Open_USART, &USART_InitStructure);
  /* Enable the Open_USART Transmit interrupt: this interrupt is generated when the 
     Open_USART transmit data register is empty */
  USART_ITConfig(Open_USART,USART_IT_RXNE,ENABLE);

  USART_Cmd(Open_USART, ENABLE);
  USART_NVIC_Config();

}

void USART_NVIC_Config(void)
{
  NVIC_InitTypeDef NVIC_InitStructure;

  /* Enable the USARTx Interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = Open_USART_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 1;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
}

int USARTx_GetLine(char *str)
//-------------------------------------------------------
// Check if a CR-terminated line is received on UART
// if 'YES'
// then copy line to *str and return(StringLength)
// else return(StringLength=0)
//-------------------------------------------------------
{
  uint8_t i=0;
  if(rmsg)                               // if message in buffer available
  {
    rmsg--;                              // decrease message buffer
    str[i]=rbuf [r_out++ & RMASK];       // get first char
    while(str[i])                        // while not stringend
    {
      str[++i]=rbuf [r_out++ & RMASK];   // get next char
    }
    if (i==0)
    {
      str[0]=0x0D;                        // insert CR
      str[1]=0x00;                        // and insert stringend
      i=1;                                // adjust string length
    }
  }
  return(i);                              // return(StringLength)
}


void USARTx_IRQHANDLER(void)
{
  char ch;
  if(USART_GetITStatus(Open_USART, USART_IT_RXNE) != RESET)
  {
    ch = USART_ReceiveData(Open_USART);       // get received character
    if (echo)
    {
      while (USART_GetFlagStatus(Open_USART, USART_FLAG_TXE) == RESET);
      USART_SendData(Open_USART, (uint16_t)ch);
    }
    if (ch=='\r')
    {
      rbuf[r_in++ & RMASK] = 0;      // add stringend to buffer
      if (echo)
      {
        while (USART_GetFlagStatus(Open_USART, USART_FLAG_TXE) == RESET);
        USART_SendData(Open_USART, '\n');
      }
      rmsg++;                        // increment message counter
    }
    if (ch==0x08)                    // if backspace
    {
      if (rbuf[(r_in-1) & RMASK])    // if there is a char, left from cursor
      {
        rbuf[(--r_in) & RMASK]=0;    // then delete it  and go one char back.
        if (echo)
        {
          while (USART_GetFlagStatus(Open_USART, USART_FLAG_TXE) == RESET);
          USART_SendData(Open_USART, ' ');   // echo SPACE to delete char on Terminal
          while (USART_GetFlagStatus(Open_USART, USART_FLAG_TXE) == RESET);
          USART_SendData(Open_USART, 0x08);  // Back-SPACE again
        }
      }
    }
    if (ch > 0x19)                  // if printable char
    {
      rbuf [r_in++ & RMASK] = ch;   // add char to buffer
    }
  }
}
/* Use no semihosting */
#if 0
#pragma import(__use_no_semihosting)
struct __FILE
{  
	int handle;
};
FILE __stdout;

int _sys_exit(int x)
{
	x = x;
	return(0);
}
#endif

/**
  * @brief  Retargets the C library printf function to the USART.
  * @param  None
  * @retval None
  */
PUTCHAR_PROTOTYPE
{
  /* Place your implementation of fputc here */
  /* e.g. write a character to the USART */
  USART_SendData(Open_USART, (uint8_t) ch);

  /* Loop until the end of transmission */
  while (USART_GetFlagStatus(Open_USART, USART_FLAG_TC) == RESET)
  {}

  return ch;
}
