/*
Simple four channel signal plot. Version 1.0
    Copyright (C) 2006  by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/io.h>
#include <unistd.h>
#include <GL/glut.h>
#include <argp.h>

#include "simplers232.c"


static error_t parse_opt(int key, char *arg, struct argp_state *state);
int main(int argc, char **argv);
void printabout(void);
void printhelp(void);
void update_stats(int foo);
void visibility_update(int foo);
void info(void);
void window_size_update(int width, int height);
void getandcompress(int foo);
void drawstring(char *text);
void printinfo(void);
void drawgraph(int val, float posxe);
void input_key_cursor (int key, int x, int y);
void input_key_key (unsigned char key, int x, int y);


int windowid_main;		//Id for main window
int windowid_about;		//ID for about window
int windowid_help;		//Id for help window
float posxc = -1.0;		//Current window drawing position
float scale = 0.0001;		//Scale for dawing
int windowwidth = 1000;		//Main window width
int windowheight = 400;		//Main window height

float view_offset = 0;		//Move graph to left or right side

int show_about = 0;		//show/ hinde about window
int show_help = 1;		//show/hide help

//trigger
int prevpins;			//Previous pin state
int triggermask = 0x0f;		//Which pins can activate the trigger
int triggerdelay = 0;		//After the full window had been drawn -> wait

//some statistics
int datarate_c = 0;		//current recived bytes
int datarate_l = 0;		//last recived bytes
int samples_c = 0;		//current recived samples
int samples_l = 0;		//last recived samples
int gsamples_l = 0;		//global last samples
int triggerrate_c = 0;		//current trigger rate
int triggerrate_l = 0;		//last trigger rate
int fifo_max_c = 0;		//current maximum data got from FIFO
int fifo_max_l = 0;		//last maximum data got from FIFO

//command line parser
const char *argp_program_version = "Simple signal plot 1.0";
const char *argp_program_bug_address = "<m.marwedelATonlinehomeDOTde>";
static char doc[] = "Simple signal plot, reads data from the serial port and shows them as graphs";
static char args_doc[] = "";
static struct argp_option options_arg[] = {
       {"device",   'd', "DEVICE", 0,
        "Sets an input device, otherwise common input devices were probed" },
       { 0 }
};

char *input_device = NULL;

static error_t parse_opt(int key, char *arg, struct argp_state *state)  {
//react on the parsed options
switch (key) {
  case 'd':
           input_device = arg;
           break;
  case ARGP_KEY_ARG:
           if (state->arg_num >= 2)	//Too many arguments
             argp_usage(state);
           break;
  case ARGP_KEY_END:			//Zero arguments are ok
           break;
  default:
           return ARGP_ERR_UNKNOWN;
}
return 0;
}

static struct argp argp = { options_arg, parse_opt, args_doc, doc };

int main(int argc, char **argv) {
argp_parse (&argp, argc, argv, 0, 0, 0);	//parse arguments
info();				//Console info
//Serial Init
if (serial_open(input_device) < 0) {
  return 1;			//Something went wrong opening the port
}
//GLUT Init
glutInit(&argc, argv);
glutInitDisplayMode(GLUT_RGBA);
glutInitWindowSize(windowwidth,windowheight); 	//Size of main window
glutInitWindowPosition(10,100);		//Position of main window
windowid_main = glutCreateWindow("simple signal plot"); //Create main window
glClearColor(1.0,1.0,1.0,1.0);		//Color for clearing the screen
glutSpecialFunc(input_key_cursor);	//React on cursor keys
glutKeyboardFunc(input_key_key);	//React on normal keys
glutReshapeFunc(window_size_update);	//Window size modyfied by user
glutVisibilityFunc(visibility_update);	//Window ist overlapped or minimized etc
glutTimerFunc(1000,getandcompress, 0);	//Read serial port and compress data
glutTimerFunc(2000,update_stats, 0);	//Update stats
glClear(GL_COLOR_BUFFER_BIT);		//Clear screen now
//create about window
windowid_about = glutCreateSubWindow(windowid_main,windowwidth/2-380 , 25, 760,
                 windowheight-50);	//Create about window
glutSetWindow(windowid_about);
glutDisplayFunc(printabout);
glutKeyboardFunc(input_key_key);	//React on normal keys
glutHideWindow();
//create help window
windowid_help = glutCreateSubWindow(windowid_main, windowwidth/2-200, 40, 400,
                 windowheight-80);	//Create about window
glutSetWindow(windowid_help);
glutDisplayFunc(printhelp);
glutKeyboardFunc(input_key_key);	//React on normal keys
glutShowWindow();
glutSetWindow(windowid_main);
printinfo();				//Window info
glutMainLoop();
}

void printabout(void) {
static char label[200];
glutSetWindow(windowid_about);
glViewport(0, 0, 760, windowheight-50);
glMatrixMode(GL_PROJECTION);
glLoadIdentity();
gluOrtho2D(0.0, 1.0, 0.0, 1.0);
glClearColor(0.4, 0.55, 0.85, 1.0);
glClear(GL_COLOR_BUFFER_BIT);		//Clear screen now
glColor3f(0.5, 0.5 , 0.5);
glBegin(GL_LINE_LOOP);
glVertex2f(0.0, 0.0);
glVertex2f(0.0, .999);
glVertex2f(0.999, 0.999);
glVertex2f(0.999, 0.0);
glEnd();
glColor3f(0.0, 0.0, 0.0);
glRasterPos2f (0.3, 0.9);
sprintf (label, "Simple signal plot 1.0 (c) 2006 by Malte Marwedel");
drawstring(label);
glRasterPos2f (0.02, 0.8);
sprintf (label, "This program is free software; you can redistribute it and/or modify it under the terms of");
drawstring(label);
glRasterPos2f (0.02, 0.72);
sprintf (label, "the GNU General Public License as published by the Free Software Foundation; either");
drawstring(label);
glRasterPos2f (0.02, 0.64);
sprintf (label, "version 2 of the License, or (at your option) any later version.");
drawstring(label);
glRasterPos2f (0.02, 0.52);
sprintf (label, "This program is distributed in the hope that it will be useful, but WITHOUT ANY");
drawstring(label);
glRasterPos2f (0.02, 0.44);
sprintf (label, "WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS");
drawstring(label);
glRasterPos2f (0.02, 0.36);
sprintf (label, "FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.");
drawstring(label);
glRasterPos2f (0.02, 0.24);
sprintf (label, "You should have received a copy of the GNU General Public License  along with this");
drawstring(label);
glRasterPos2f (0.02, 0.16);
sprintf (label, "program. If not, write to ");
drawstring(label);
glRasterPos2f (0.02, 0.08);
sprintf (label, "The Free Software Foundation, Inc. 59 Temple Place, Suite 330  Boston, MA 02111, USA.");
drawstring(label);
glutSetWindow(windowid_main);
}

void printhelp(void) {
static char label[200];
glutSetWindow(windowid_help);
glViewport(0, 0, 400, windowheight-80);
glMatrixMode(GL_PROJECTION);
glLoadIdentity();
gluOrtho2D(0.0, 1.0, 0.0, 1.0);
glClearColor(0.3, 0.85, 0.55, 1.0);
glClear(GL_COLOR_BUFFER_BIT);		//Clear screen now
glColor3f(0.5, 0.5 , 0.5);
glBegin(GL_LINE_LOOP);
glVertex2f(0.0, 0.0);
glVertex2f(0.0, .999);
glVertex2f(0.999, 0.999);
glVertex2f(0.999, 0.0);
glEnd();
glColor3f(0.0, 0.0, 0.0);
glRasterPos2f (0.1, 0.9);
sprintf (label, "h -> show/hide this screen");
drawstring(label);
glRasterPos2f (0.1, 0.82);
sprintf (label, "l -> show/hide copyright and license");
drawstring(label);
glRasterPos2f (0.1, 0.74);
sprintf (label, "+ -> zoom in");
drawstring(label);
glRasterPos2f (0.1, 0.66);
sprintf (label, "- -> soom out");
drawstring(label);
glRasterPos2f (0.1, 0.58);
sprintf (label, "c -> clear screen");
drawstring(label);
glRasterPos2f (0.1, 0.50);
sprintf (label, "D -> increase trigger delay by 25%s","%");
drawstring(label);
glRasterPos2f (0.1, 0.42);
sprintf (label, "d -> decrease trigger delay by 25%s","%");
drawstring(label);
glRasterPos2f (0.1, 0.34);
sprintf (label, "right -> shift graph");
drawstring(label);
glRasterPos2f (0.1, 0.26);
sprintf (label, "left -> shift graph ");
drawstring(label);
glRasterPos2f (0.1, 0.18);
sprintf (label, "1,2,3,4 -> enable/disable trigger for input");
drawstring(label);
glRasterPos2f (0.1, 0.10);
sprintf (label, "q -> close port and quit");
drawstring(label);
glRasterPos2f (0.1, 0.02);
sprintf (label, "parameter -d specify the serial device");
drawstring(label);
glutSetWindow(windowid_main);
}

void update_stats(int foo) {
datarate_l = datarate_c;
datarate_c = 0;
samples_l = samples_c;
samples_c = 0;
if (samples_l > 0) {
  if (gsamples_l == 0) {
    gsamples_l = samples_l;
  } else {
    gsamples_l = (samples_l + gsamples_l*5)/6;
  }
}
triggerrate_l = triggerrate_c;
triggerrate_c = 0;
fifo_max_l = fifo_max_c *100/254;
fifo_max_c = 0;
glutTimerFunc(1000,update_stats, 0);	//Recall this function after 1s
printinfo();
}

void visibility_update(int foo) {
printinfo();
}


void info(void) {			//Print usage information at beginning
printf("info: Simple signal plot (c) 2006 by Malte Marwedel\n");
printf("info: press 'l' in window to see license\n");
if (input_device == NULL) {
  printf("info: By default /dev/USB0 and /dev/ttyS0 were tested\n");
  printf("info: use parameter -d to specify an other device. Example -d/dev/ttyS2\n");
}
}

void window_size_update(int width, int height) {	//Update window geometry
windowwidth = width;
windowheight = height;

glViewport (0, 0, width, height);
glutSetWindow(windowid_about);
glutReshapeWindow (760, windowheight-50);
glutPositionWindow (windowwidth/2-380, 25);
glutPostRedisplay();
glutSetWindow(windowid_help);
glutReshapeWindow (400, windowheight-80);
glutPositionWindow (windowwidth/2-200, 40);
glutPostRedisplay();
glutSetWindow(windowid_main);
glClear(GL_COLOR_BUFFER_BIT);
printinfo();
}

void getandcompress(int foo) {		//Get data from serial port and compress
int no,nun;
int newval, newr;
int oldval, oldr;
float posxe;

no = serial_get();			//Get values
if (no >= 0) {				//count for stats
  datarate_c += no;
  if ( fifo_max_c < no) {		//some more stats
    fifo_max_c = no;
  }
}
if (no > 0) {				//We got values to draw
  oldval = rs232recbuf[0] & 0x000f;	//The value
  oldr = (rs232recbuf[0] & 0x00f0)>>4;	//Times of value
  samples_c += oldr;			//count for stats
  for (nun = 1; nun < no; nun++) {	//Loop to compress and draw values
    newval = rs232recbuf[nun] & 0x000f;	//The value
    newr = (rs232recbuf[nun] & 0x00f0)>>4;//Times of value
    samples_c += newr;			//count for stats
    //Compress the values to reduce line drawing calls
    if (newval == oldval) {
      oldr += newr;			//Just add the times
    } else {
      posxe = posxc+(float)oldr*scale;
      drawgraph(oldval,posxe);		//Draw old values
      posxc = posxe;
      oldval = newval;
      oldr = newr;
    }
    if (triggermask) {			//At least one Pin used as trigger
      if (posxc > (float)(1.0+(float)((float)triggerdelay/50.0))) {
        if ((oldval & triggermask) != (prevpins & triggermask)) {
          triggerrate_c++;		//count for stats
          posxc = -1.0-view_offset;
        }
      }
      prevpins = oldval;
    } else { 				//No trigger
      if (posxc > 1.0) {
        posxc = -1.0;
      }
    }
  }
  posxe = posxc+(float)oldr*scale;	//Update drawing position
  drawgraph(oldval,posxe);		//Draw reaming values
  posxc = posxe;			//Byte done
  glutPostRedisplay();			//Update screen
}
glutTimerFunc(1,getandcompress, 0);	//Recall this function after 1ms
}

void drawstring(char *text) {		//Draws a string with glut
int nun;
for (nun = 0; nun < strlen(text); nun++) {
  glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, *(text+nun));
}
}

void printinfo(void) {
int nun;
float samples_per_pix;
static char label[200], labelb[200];
//Clear old text above
glPushMatrix();
glBegin(GL_QUADS);
glColor3f(1.0, 1.0, 1.0);
glVertex2f(-1.0, 1.0);
glVertex2f(-1.0, 0.86);
glVertex2f(1.0, 0.86);
glVertex2f(1.0, 1.0);
glEnd();
glPopMatrix();
//Write scale
glColor3f (0.0, 0.0, 0.0);
glRasterPos2f (-0.9, 0.9);
samples_per_pix = 1/(scale*windowwidth/2.0);
sprintf(label, "Samples per pixel: %f", samples_per_pix);
drawstring(label);
//Write trigger
glColor3f(0.0, 0.0, 0.0);
glRasterPos2f(-0.3, 0.9);
if (triggermask) {
  sprintf(label,"");
  sprintf(labelb,"");
  for (nun = 0; nun < 4; nun++) {
    if (triggermask & (0x01<<nun)) {
      sprintf(label, "%s %i",labelb,(nun+1));
    }
    sprintf(labelb,"%s", label);
  }
  sprintf(label,
         "Trigger set on inputs%s, pause at end: %i %s, start offset: %i %s",
          labelb,triggerdelay,"%",(int)(view_offset*50.0), "%");
} else {
  sprintf(label, "Trigger is off");
}
drawstring(label);
//Clear old text below
glPushMatrix();
glBegin(GL_QUADS);
glColor3f(1.0, 1.0, 1.0);
glVertex2f(-1.0, -1.0);
glVertex2f(-1.0, -0.755);
glVertex2f(1.0, -0.755);
glVertex2f(1.0, -1.0);
glEnd();
glPopMatrix();
//Write stats
glColor3f(0.0, 0.0, 0.0);
glRasterPos2f(-0.9, -0.9);
sprintf(label, "Data rate: %i byte/s Sample rate: %i 1/s Trigger rate: %i 1/s",
        datarate_l, samples_l, triggerrate_l);
drawstring(label);
//FIFO load
if (fifo_max_l >= 95) {
  glColor3f (1.0, 0.0, 0.0);
} else {
  glColor3f (0.0, 0.0, 0.0);
}
glRasterPos2f(0.5, -0.9);
sprintf(label, "max FIFO usage: %i %s",fifo_max_l,"%");
drawstring(label);
//Clear old text in middle
glPushMatrix();
glBegin(GL_QUADS);
glColor3f(1.0, 1.0, 1.0);
glVertex2f(-1.0, 0.11);
glVertex2f(-1.0, -0.11);
glVertex2f(1.0, -0.11);
glVertex2f(1.0, 0.11);
glEnd();
glPopMatrix();
//Write stats
glColor3f(0.0, 0.0, 0.0);
glRasterPos2f(-0.2, -0.05);
sprintf(label, "Screen shows %f ms",
        samples_per_pix*windowwidth*1000/gsamples_l);
drawstring(label);
glutPostRedisplay();

}


void drawgraph(int val, float posxe) {
float peg;
//Remove old drawing
glPushMatrix();
glBegin(GL_QUADS);		//Begin drawing a quadrilateral
glColor3f(1.0, 1.0, 1.0);	//white color
glVertex2f(posxc, 0.1);
glVertex2f(posxe+0.05, 0.1);
glVertex2f(posxe+0.05, 0.87);
glVertex2f(posxc, 0.87);
glEnd();
glPopMatrix();
glPushMatrix();
glBegin(GL_QUADS);		//Begin drawing a quadrilateral
glColor3f(1.0, 1.0, 1.0);	//white color
glVertex2f(posxc, -0.76);
glVertex2f(posxe+0.05, -0.76);
glVertex2f(posxe+0.05, -0.1);
glVertex2f(posxc, -0.1);
glEnd();
glPopMatrix();
//draw the value1
glPushMatrix();
glColor3f(1.0,0.3,0.3);
glBegin(GL_LINES);
peg = ((float)(val & 0x1))*(0.1);
glVertex2f(posxc, +0.75+peg);
glVertex2f(posxe, +0.75+peg);
glEnd();
glPopMatrix();
//draw the value2
glPushMatrix();
glColor3f(0.2,1.0,0.2);
glBegin(GL_LINES);
peg = ((float)((val>>1) & 0x1))*(0.1);
glVertex2f(posxc, +0.25+peg);
glVertex2f(posxe, +0.25+peg);
glEnd();
glPopMatrix();
//draw the value3
glPushMatrix();
glColor3f(0.1,0.1,1.0);
glBegin(GL_LINES);
peg = ((float)((val>>2) & 0x1))*(0.1);
glVertex2f(posxc, -0.25+peg);
glVertex2f(posxe, -0.25+peg);
glEnd();
glPopMatrix();
//draw the value4
glPushMatrix();
glColor3f(0.8,0.5,0.0);
glBegin(GL_LINES);
peg = ((float)((val>>3) & 0x1))*(0.1);
glVertex2f(posxc, -0.75+peg);
glVertex2f(posxe, -0.75+peg);
glEnd();
glPopMatrix();
}

void input_key_cursor (int key, int x, int y) {
if ((key == GLUT_KEY_LEFT) && (view_offset >= 0.1)) {
  view_offset -= 0.1;
}
if (key == GLUT_KEY_RIGHT) {
  view_offset += 0.1;
}
printinfo();
}

void input_key_key (unsigned char key, int x, int y) {
if (key == 'q') {
  serial_close();
  exit(0);
}
if (key == '+') {
  scale *= 2.0;
  glClear(GL_COLOR_BUFFER_BIT);
}
if (key == '-') {
  scale /= 2.0;
  glClear(GL_COLOR_BUFFER_BIT);
}
if (key == 'c') {
  glClear(GL_COLOR_BUFFER_BIT);
}
if (key == '1') {  //toggels bit 0
  triggermask = (triggermask & ~0x01) | (((~triggermask) & 0x01));
}
if (key == '2') {  //toggels bit 1
  triggermask = (triggermask & ~0x02) | (((~triggermask) & 0x02));
}
if (key == '3') {  //toggels bit 2
  triggermask = (triggermask & ~0x04) | (((~triggermask) & 0x04));
}
if (key == '4') {  //toggels bit 3
  triggermask = (triggermask & ~0x08) | (((~triggermask) & 0x08));
}
if (key == 'D') {
  triggerdelay += 25;
}
if ((key == 'd') && (triggerdelay >= 25)) {
  triggerdelay -= 25;
}
if (key == 'l') {
  if (show_about) {	//then hinde window
    show_about = 0;
    glutSetWindow(windowid_about);
    glutHideWindow();
    glutSetWindow(windowid_main);
    glClear(GL_COLOR_BUFFER_BIT);
  } else {		//otherwise show it
    show_about = 1;
    glutSetWindow(windowid_about);
    glutShowWindow();
    glutSetWindow(windowid_main);
  }
}
if (key == 'h') {
  if (show_help) {	//then hinde window
    show_help = 0;
    glutSetWindow(windowid_help);
    glutHideWindow();
    glutSetWindow(windowid_main);
    glClear(GL_COLOR_BUFFER_BIT);
  } else {		//otherwise show it
    show_help = 1;
    glutSetWindow(windowid_help);
    glutShowWindow();
    glutSetWindow(windowid_main);
  }
}

printinfo();
}
