/*
Simple four channel signal sampler version 1.0
    Copyright (C) 2006  by Malte Marwedel

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <inttypes.h>

typedef unsigned char  u08;
typedef unsigned short u16;


register unsigned char valold asm("r2");	//Stores previous sample
register unsigned char readno asm("r3");	//How often the sample was equal

#define fifo_size 980
u08 volatile fifo_tx[fifo_size+1];		//The FIFO
u16 volatile fifo_wp = 1;			//FIFO write pointer
u16 fifo_rp = 1;				//FIFO read pointer


//If you use an version of avr-libc before 1.4, you need this:
#define ISR(TIMER2_COMP_vect) SIGNAL(SIG_OUTPUT_COMPARE2)
//otherwise comment it out.

ISR(TIMER2_COMP_vect) {			//Samples the input pins
u08 val;
u16 fifo_wpl;

val = PINC & 0x0f;			//Reads the input ports
fifo_wpl = fifo_wp;			//Local version in registers
if (val == valold) {			//If this sample and previous are equal
  if (readno >= 0x0f) {			//If 15 times the same data
    if (fifo_tx[fifo_wpl] == 0) {	//If FIFO is free
      fifo_tx[fifo_wpl] = val | 0xf0;	//Put data into FIFO
      fifo_wpl--;			//Next FIFO adress
      if (fifo_wpl == 0) {		//If FIFO adress poiter at bottom
        fifo_wpl = fifo_size;		//Reset FIFO adress pointer
      }
    }
    readno = 0;				//None new reads so far
  }
} else {				//If there were different data now
  if (fifo_tx[fifo_wpl] == 0) {		//If FIFO is free
    fifo_tx[fifo_wpl] = (u08)(valold | (u08)(readno<<4)); //Put data into FIFO
    fifo_wpl--;				//Next FIFO adress
    if (fifo_wpl == 0) {		//If FIFO adress poiter at bottom
      fifo_wpl = fifo_size;		//Reset FIFO adress pointer
    }
  }
  readno = 0;				//None new reads so far
}
valold = val;		//The current value will be the previous value next time
readno++;				//We read one value in this interrupt
fifo_wp = fifo_wpl;			//Write local value back into global one
}


int main(void) {
u16 nun;
//Uart Init
UBRRL = 2;		//stands for 230,4kBaud at 11,0592MHZ
UCSRB |= 1<<TXEN;	//Activates den UART transmitter
UCSRB |= 1<<RXEN;	//Activates UART reciver
//Timer2 init
TCNT2 = 0;		//Timer2 reset
TCCR2 = (1 << CS20);	//Prescaler = 1
TCCR2 |= (1 << WGM21);	//Clear timer on compare match
TIMSK = 1 << OCIE2;	//Compare match int active
OCR2 = 221;		//50,04KHZ sample rate at 11.0592MHZ
//Initialize register variables
valold = 0x0f;
readno = 0;
//Initialize FIFO as void
for (nun = 0; nun <= fifo_size; nun++) {
  fifo_tx[nun] = 0;	//Zero has the meaning of empty in this FIFO
}
sei();			//Interrupts enabled
for (;;) {		//Ednless loop which frees FIFO
  while (fifo_tx[fifo_rp] == 0);	//Wait until FIFO contains data
  while (!(UCSRA & (1<<UDRE)));	//Wait until UART is ready
  UDR = fifo_tx[fifo_rp];	//Send data
  fifo_tx[fifo_rp] = 0;		//Mark memory as free
  fifo_rp--;			//Decrease reding pointer
  if (fifo_rp == 0) {		//If reading pointer at minimum
    fifo_rp = fifo_size;	//Restart at highest address
  }
}
}
