#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/pgmspace.h>
#include <avr/delay.h>
#include <stdlib.h> 
#include "titlescreen.h"
#include "font.h"

#define KeyCount 5

struct
{
	uint8_t *port;
	uint8_t mask;	
	uint8_t down;	
	uint16_t delay;
} Keys[KeyCount]={{&PIND,1<<4,0,0},{&PIND,1<<3,0,0},{&PIND,1<<1,0,0},{&PIND,1<<0,0,0},{&PIND,1<<2,0,0}};
#define KEY_UP 0
#define KEY_LEFT 1
#define KEY_DOWN 2
#define KEY_RIGHT 3
#define KEY_ENTER 4

char seed=0;

void HandleKeys(void)
{
	seed++;
			
	for (uint8_t i=0;i<KeyCount;i++)
	{
		Keys[i].delay=0;
	}
};

uint8_t CheckKey(uint8_t index)
{
	if (!(*Keys[index].port&Keys[index].mask)&&(Keys[index].down==0))
	{
		Keys[index].down=1;
		Keys[index].delay=1;
	}
	
	if ((Keys[index].delay==0)&&(*Keys[index].port&Keys[index].mask)&&Keys[index].down)
	{
			srand(seed);
			Keys[index].down=0;
			return 1;
			
						
	};

	return 0;
};

uint8_t CheckKeyHeld(uint8_t index)
{
	if ((Keys[index].delay==0)&&(Keys[index].down))
		return 1;		
	else
		return 0;
};

SIGNAL(SIG_OVERFLOW2)
{
	HandleKeys();
};


#define LCD_E PC5
#define LCD_RW PC4
#define LCD_A0 PC3
#define LCD_CS PC2
#define LCD_RES PC1


inline void LCD_Write(char isData,char data)
{
	PORTB=data;
	if (isData)
		PORTC|=(1<<LCD_A0);
	else
		PORTC&=~(1<<LCD_A0);
		
	PORTC|=(1<<LCD_E);
//	asm volatile("nop"::); 		
	PORTC&=~(1<<LCD_E);

};

void LCD_Enable(void)
{
	//Enable backlight
	PORTD|=(1<<5);
	
	//Perform reset
	PORTC&=~(1<<LCD_RES);
	asm volatile("nop"::);
	PORTC|=(1<<LCD_RES);
	
	//Start Display line=0
	LCD_Write(0, 0x40); 	
	//ADC normal (turned)
	LCD_Write(0, 0xA0);
	//Driving Bias 1/7
	LCD_Write(0, 0xA3);
	//Output mode: Not mirrored
	LCD_Write(0, 0xC0);
	//Enable Voltage regulator circuits
	LCD_Write(0, 0x2F);
	//V5 Regulator Ratio=0
	LCD_Write(0, 0x20);
	//Disable static indicator
	LCD_Write(0, 0xAC);
	//Enable writing to contrast register
	LCD_Write(0, 0x81);
	//Write contrast (medium)
	LCD_Write(0, 0x20);
	//Disable display test
	LCD_Write(0, 0xA4);
	//Enable Display
	LCD_Write(0, 0xAF);
	//Normal mode (not reversed)
	LCD_Write(0, 0xA6);	
};

void LCD_Clear(void)
{
	for (char y=0;y<8;y++)
	{
		//Move to current line
		LCD_Write(0, 0xB0|y);
		
		//Move to first Column		
		LCD_Write(0, 0x10);
		LCD_Write(0, 0x00);
		
		for (char x=0;x<128;x++)
			LCD_Write(1,0x00);		
	};	
}

void LCD_Showtitle(char offset)
{
	char temp;

	//Write titlescreen:
	for (char y=0;y<8;y++)
	{
		//Move to current line
		LCD_Write(0, 0xB0|y);		
		
		//Move to first Column		
		LCD_Write(0, 0x10|(offset>>4));
		LCD_Write(0, 0x00|(offset&0xF));
		
		for (char x=0;x<128-offset;x++)
		{
			memcpy_P(&temp, &title[(y<<7)+x], 1);
			LCD_Write(1,temp);
		};
		
	};	
}

void LCD_Drawtext(char* text, signed short px, char py)
{	
	
	char i=0;
	char charx;
	char temp;

	//Move to current line
	LCD_Write(0, 0xB0|py);		

	if (px>127)
		return;
		
	if (px<=0)
	{
		//Move to left column
		LCD_Write(0, 0x10);
		LCD_Write(0, 0x00);		
	}
	else
	{
		//Move to first column of text
		LCD_Write(0, 0x10|((px)>>4));
		LCD_Write(0, 0x00|((px)&0xF));					
	};
	
	while (text[i]!=0)
	{
		charx=255;
		if ((text[i]>='a') && (text[i]<='z'))
			charx=text[i]-'a';
		else if ((text[i]>='A') && (text[i]<='Z'))
			charx=text[i]-'A';
		else if ((text[i]>='0') && (text[i]<='9'))
			charx=text[i]-'0'+26;
		else if (text[i]=='.')
			charx=36;
		else if (text[i]==':')
			charx=37;
		else if (text[i]=='-')
			charx=38;
		else if (text[i]=='>')
			charx=39;
			
		if (charx!=255)
		{
			for (char x=0;x<5;x++)
				if (px+x>=0)
				{
					memcpy_P(&temp, &font[charx*5+x], 1);
					LCD_Write(1,temp);					
				};				
		}
		else
		{
			for (char x=0;x<5;x++)
				if (px+x>=0)
					LCD_Write(1,0x00);
		}		
		
		i++;
		px+=6;
		
		//Write dummy data for character spacing
		if (px>0)
			LCD_Write(1,0x00);					
		
	};
};

void LCD_Drawmenumarker(char item)
{
	for (char i=0;i<3;i++)
	{
		if (item==i)
			LCD_Drawtext(">", 28, i+2);
		else
			LCD_Drawtext(" ", 28, i+2);		
	}
		
	
}

void LCD_Scrolllogo(void)
{
	
	LCD_Showtitle(0);
	char line=0;

	for (unsigned long i=0;i<40000000;i++);
	{}
		
	while (1)
	{	
			
		LCD_Write(0, 0x40|line);
		line++;
		
		if (line>63)
		{
			for (unsigned long i=0;i<20000;i++)
			{
//				HandleKeys();
				if (CheckKey(KEY_ENTER))
				{
					LCD_Write(0, 0x40);					
					return;
				};
			}
			line=0;			
		};
			
		for (unsigned long i=0;i<1000;i++)
		{
//				HandleKeys();
				if (CheckKey(KEY_ENTER))
				{
					LCD_Write(0, 0x40);					
					return;
				};
		}
	}	

};

void LCD_Moveto(char x, char y)
{
	//Move to current line
	LCD_Write(0, 0xB0|y);

	//Move to column
	LCD_Write(0, 0x10|((x)>>4));
	LCD_Write(0, 0x00|((x)&0xF));	
}

void LCD_Drawblock(char x, char y, char data)
{
	LCD_Moveto(x, y);
	LCD_Write(1, data);
};

void LCD_Drawvertline(char x)
{
	for (char y=0; y<8; y++)
	{
		LCD_Drawblock(x, y, 0xFF);
	};
};




volatile unsigned short sound_count=0;
volatile unsigned short sound_fcount=0;
volatile unsigned char sound_state=0;
volatile unsigned short sound_length=0;
volatile unsigned short sound_freq=0;

void Sound_Beep(unsigned short freq, unsigned short length)
{
	sound_length=(F_CPU/256/8*length)/1000;
	sound_freq=(F_CPU/256/8/2/freq);
	sound_state=0;
	sound_count=0;
	sound_fcount=0;
	
	//Enable timer
	TCNT0=0;
	TCCR0=(1<<CS01);
};

void Sound_BeepST(unsigned short freq, unsigned short length)
{
	Sound_Beep(freq, length);
	for (unsigned long i=0;i<((F_CPU/4000)*length)/0xFFFF;i++)
		_delay_loop_2(0xFFFF);
}

SIGNAL(SIG_OVERFLOW0)
{
	sound_count++;
	if (sound_count>sound_length)
	{
		PORTD&=~(1<<6);
		TCCR0=0;		
	};
	sound_fcount++;
	if (sound_fcount>sound_freq)
	{
		sound_fcount=0;
		if (sound_state==0)
		{
			sound_state=1;
			PORTD|=(1<<6);			
		}
		else
		{
			sound_state=0;
			PORTD&=~(1<<6);
		};		
	};	
}

void Tetris_Drawfield(unsigned short* field)
{
	char data;
	for (char y=0; y<8; y++)
	{
		for (char x=0; x<10; x++)
		{
			if (field[x]&(1<<(y<<1)))
				data=(1<<2)|(1<<1)|(1<<0);
			else
				data=0;
			if (field[x]&(1<<((y<<1)+1)))
				data|=(1<<6)|(1<<5)|(1<<4);

			if (y==7)
				data|=(1<<7);
			
			LCD_Moveto(44+x*4,y);
			for (char x1=0; x1<3; x1++)
				LCD_Write(1, data);			
				
			//Dummy write
			if (y==7)
				LCD_Write(1, 1<<7);
			else
				LCD_Write(1, 0);			
		};		
	};	
};
volatile char tTimer=0;

SIGNAL(SIG_OUTPUT_COMPARE1A)
{
	tTimer=1;		
	TCNT1=0;
};

#define tBlockCount 7
char tBlocks[tBlockCount][5]={{0x3,0x3,0,0,2}, {0xF,0,0,0,1},      {0x6,0x3,0,0,2},   {0x1,0x3,0x1,0,3},{0x4,0x7,0,0,2},   {0x3,0x1,0,0,2}, {0x7, 0, 0, 0, 1}};
char tBlocksR[tBlockCount][5]={{0x3,0x3,0,0,2},{0x1,0x1,0x1,0x1,4},{0x1,0x3,0x02,0,3},{0x2,0x7,0,0,2},  {0x3,0x2,0x2,0,3}, {0x1,0x3,0,0,2}, {0x1, 0x1, 0x1, 0, 3}};

char Tetris_Checkcollision(unsigned short * field, char X, char Y, char R,  char M, char W, char H, char Block)
{
	char x1;
	char y1;
	
	for (char y=0; y<H; y++)
		for (char x=0; x<W; x++)
		{
			if (M==0)
			{
				x1=x;
				y1=y;
			}
			else
			{
				x1=W-1-x;
				y1=H-1-y;
			};
					
			if (((R==0)&&(tBlocks[Block][x1]&(1<<y1))) ||
				((R==1)&&(tBlocksR[Block][x1]&(1<<y1))))
			{
				if (field[x+X]&(1<<(y+Y)))
					return 1;					
			};					
		};
	return 0;
}

void Tetris_Drawscore(unsigned short score, char multi)
{
	char text[2];
	text[1]=0;
	char temp;
	unsigned short divide;	
	divide=10000;
	
	for (char x=0; x<5; x++)
	{
		temp=(score/divide)%10;
		text[0]='0'+temp;

		LCD_Drawtext(&text[0],x*6,0);			
		divide/=10;
	}; 

	divide=100;
	char x=1;
	LCD_Drawtext("X", 0, 1);
	for (char i=0; i<3; i++)
	{
		temp=(multi/divide)%10;
		
		if (temp>0)
		{
			text[0]='0'+temp;
			LCD_Drawtext(&text[0],x*6,1);			
			x++;			
		};
		divide/=10;				
	};	
};

void Tetris_Drawbg(void)	
{
	LCD_Clear();
	//Draw playing field borders:
	LCD_Drawvertline(42);
	LCD_Drawvertline(84);
	LCD_Drawblock(43, 7, (1<<7));
	LCD_Drawblock(83, 7, (1<<7));
	
	Tetris_Drawscore(0,1);
	
};


void Tetris(void)
{
	unsigned short tSpeed=9000;
	
	OCR1A=tSpeed;
	TCCR1B=(1<<CS12)|(1<<CS10);
	TCNT1=0;
	TIMSK|=(1<<OCIE1A);
	
	
	Tetris_Drawbg();
	


	char tBlock=rand()%tBlockCount;
	char tBlockR=0;
	char tBlockM=0;
	char tBlockW=tBlocks[tBlock][4];
	char tBlockH=tBlocksR[tBlock][4];	
	char tBlockX=5-tBlockW/2;
	char tBlockY=0;
	char tFailed=0;

	char tBlockOR;
	char tBlockOM;
	char tBlockOW;
	char tBlockOH;	
	char tBlockOX;
	
	char tBlockMove=0;
	char tBlockAllowMove=0;

	char tCollided=0;
	
	unsigned short tScore=0;
	char tMulti=1;
	
	unsigned short tField[10];
	unsigned short tField1[10];
		
	for (char i=0; i<10; i++)
		tField[i]=0;
		
	Tetris_Drawfield(&tField[0]);
	
	while (1)
	{	
		if ((tFailed) || ((tCollided) && (tBlockY==0)))
		{
			TCCR1B=0;
			LCD_Clear();
			LCD_Drawtext("Game over",37,3);					

			char text[2];
			text[1]=0;
			char temp;
			unsigned short divide;	
			divide=10000;
	
			for (char x=0; x<5; x++)
			{
				temp=(tScore/divide)%10;
				text[0]='0'+temp;

				LCD_Drawtext(&text[0],49+x*6,4);			
				divide/=10;
			}; 			
			
			LCD_Drawtext("Press Enter",31,7);					
				
			Sound_BeepST(3220, 100);
			Sound_BeepST(880, 100);
			Sound_BeepST(440, 100);
			
			while (1)
			{
				if (CheckKey(KEY_ENTER))
					return;						
			};					
		}
				
		
		if (tTimer==1)
		{
			if (CheckKeyHeld(KEY_DOWN))
				Sound_BeepST(880, 10);

			tBlockY++;		
			tTimer=0;			
			
			if (tCollided==1)
			{
				//Fill real field
				for (char i=0; i<10; i++)
					tField[i]=tField1[i];
			
				tBlock=rand()%tBlockCount;

				tBlockR=rand()%2;
				tBlockM=rand()%2;
					
				if (tBlockR==0)			
				{
					tBlockW=tBlocks[tBlock][4];
					tBlockH=tBlocksR[tBlock][4];
				}
				else
				{
					tBlockW=tBlocksR[tBlock][4];
					tBlockH=tBlocks[tBlock][4];
				};

				tBlockX=5-tBlockW/2;
				tBlockY=0;	
				
			
				TCCR1B=(1<<CS12)|(1<<CS10);
				TCNT1=0;				
			}
			
						
		}
		else
		{
			if ((tBlockMove==1) && (tBlockAllowMove))
				tBlockX--;

			if ((tBlockMove==2) && (tBlockAllowMove))
				tBlockX++;			
				
			tBlockMove=0;			
			
		};
		
		tBlockAllowMove=1;

		if (CheckKey(KEY_LEFT))
		{			
			if (tBlockX>0)
			{
				tBlockMove=1;
				Sound_Beep(1000, 50);			
			}
			else
				Sound_Beep(500, 50);
			
		}
		else if (CheckKey(KEY_RIGHT))
		{
			if (tBlockX<10-tBlockW)
			{
				tBlockMove=2;
				Sound_Beep(1000, 50);			
			}
			else
				Sound_Beep(500, 50);
		}
		else if (CheckKey(KEY_UP))
		{
			Sound_Beep(1000, 50);
			
			tBlockOX=tBlockX;
			tBlockOM=tBlockM;
			tBlockOR=tBlockR;			
			tBlockOW=tBlockW;			
			tBlockOH=tBlockH;			
			
			if (tBlockR==0)
			{
				tBlockR=1;
				
				tBlockW=tBlocksR[tBlock][4];
				tBlockH=tBlocks[tBlock][4];
			}
			else
			{
				tBlockR=0;
				if (tBlockM==0)
					tBlockM=1;
				else
					tBlockM=0;
					
				tBlockW=tBlocks[tBlock][4];
				tBlockH=tBlocksR[tBlock][4];
			}					
			if ((tBlockX+tBlockW)>9)
				tBlockX=10-tBlockW;
				
			if (Tetris_Checkcollision(&tField[0], tBlockX, tBlockY, tBlockR, tBlockM, tBlockW, tBlockH, tBlock))
			{
				Sound_Beep(500, 50);
				tBlockX=tBlockOX;
				tBlockM=tBlockOM;
				tBlockR=tBlockOR;
				tBlockW=tBlockOW;
				tBlockH=tBlockOH;				
			};
			
		}
		else if (CheckKey(KEY_DOWN))
		{
			TCCR1B=(1<<CS12)|(1<<CS10);
		}		
		else if (CheckKeyHeld(KEY_DOWN))
		{
			TCCR1B=(1<<CS11)|(1<<CS10);
		}		
		else if (CheckKey(KEY_ENTER))
		{
			Sound_Beep(1000, 50);
			
			TCNT1=0;
			
			LCD_Clear();
			LCD_Drawtext("Press again to quit", 7, 3);
			LCD_Drawtext("or up to resume", 19, 4);
						
			
			while (!CheckKey(KEY_UP))
			{
				if (CheckKey(KEY_ENTER))
				{
					Sound_BeepST(3220, 100);
					Sound_BeepST(880, 100);
					Sound_BeepST(440, 100);
										
					return;					
				};
			};			
			
			Tetris_Drawbg();			
			Tetris_Drawfield(&tField1[0]);
			Tetris_Drawscore(tScore,tMulti);
			
			Sound_Beep(1000, 50);			
		}		
		
			
		char x1,y1;
			
		tCollided=0;
		//Fill dummy field
		for (char i=0; i<10; i++)
			tField1[i]=tField[i];
			
		//Insert block into field
		for (char y=0; y<tBlockH; y++)
			for (char x=0; x<tBlockW; x++)
			{
				if (tBlockM==0)
				{
					x1=x;
					y1=y;
				}
				else
				{
					x1=tBlockW-1-x;
					y1=tBlockH-1-y;
				};
					
				if (((tBlockR==0)&&(tBlocks[tBlock][x1]&(1<<y1))) ||
					((tBlockR==1)&&(tBlocksR[tBlock][x1]&(1<<y1))))
				{
					tField1[x+tBlockX]|=(1<<(y+tBlockY));
					if (y+tBlockY==15)
						tCollided=1;
					else if (tField[x+tBlockX]&(1<<(y+tBlockY+1)))
						tCollided=1;
					else if (tField[x+tBlockX]&(1<<(y+tBlockY)))
						tFailed=1;
					
					if (tBlockMove==1)
						if (tField[x+tBlockX-1]&(1<<(y+tBlockY)))
							tBlockAllowMove=0;
					if (tBlockMove==2)
						if (tField[x+tBlockX+1]&(1<<(y+tBlockY)))
							tBlockAllowMove=0;						
				};					
			};


		Tetris_Drawfield(&tField1[0]);
		
		char linecount=0;
		
		for (signed char y=15; y>=0; y--)
		{
			char empty=0;
			char x=0;
			
			while ((x<10) && (empty==0))
			{
				if (!(tField[x]&(1<<y)))
					empty=1;
				x++;				
			}
			
			if (empty==0)
			{
				Sound_BeepST(440, 100);
				Sound_BeepST(880, 100);
				Sound_BeepST(3220, 100);
				
				tSpeed/=1.1;
				OCR1A=tSpeed;				
				TCNT1=0;
				
				tScore+=10*tMulti;
				Tetris_Drawscore(tScore,tMulti);
				
				linecount++;				
												
				unsigned short mask=0;
				
				for (char y1=0; y1<=y; y1++)
					mask|=(1<<y1);
				
				for (x=0; x<10; x++)
				{
					tField[x]=((tField[x]&(~mask))|((tField[x]<<1)&mask));
				};
				
				y++;				
			};
				
		}
		if (linecount>1)
		{
			for (char i=0;i<linecount;i++)
				Sound_BeepST(880, 40);			
			
			tMulti=tMulti*(1<<(linecount-1));
			Tetris_Drawscore(tScore, tMulti);
		};
		
		
	};	
};


SIGNAL(SIG_INTERRUPT0)
{
	//MCUCR&=~((1<<SM2)|(1<<SM1)|(1<<SM0));		
	GICR&=~(1<<INT0);
}

int main(void)
{		
	OSCCAL=0x9C;

	WDTCR=0;
	UCSRA=0;
	UCSRB=0;
	ADMUX=0;
	ADCSRA=0;
	ACSR=(1<<ACD);
	
	DDRB=0xFF;
	DDRC=0xFF;
	DDRD=(1<<7)|(1<<6)|(1<<5);
	PORTD|=(1<<4)|(1<<3)|(1<<2)|(1<<1)|(1<<0);
		
	//Enable 8-Bit Counter0 overflow Interrupt
	TIMSK|=(1<<TOIE0);
	
	//Enable 8-Bit Counter2
	TCCR2=(1<<CS22)|(1<<CS21);
	TIMSK|=(1<<TOIE2);
	
	sei();
	
	
	
	//Enable INT0
	GICR|=(1<<INT0);
	
	//Enable Power down mode
	MCUCR|=(1<<SE)|(1<<SM1);
	
	asm volatile("sleep"::);
		
	
	LCD_Enable();
	
	LCD_Showtitle(0);
	
	Sound_BeepST(440, 100);
	Sound_BeepST(880, 100);
	Sound_BeepST(3220, 100);

	for (unsigned long i=0;i<100000000;i++);
		asm volatile("nop"::);


		
	for (char x=1;x<129;x++)
	{
		LCD_Showtitle(x);
			
		//Clear left column:
		for (char y=0;y<8;y++)
		{
		
			//Move to current line
			LCD_Write(0, 0xB0|y);		
		
			//Move to left Column		
			LCD_Write(0, 0x10|((x-1)>>4));
			LCD_Write(0, 0x00|((x-1)&0xF));
			LCD_Write(1, 0x00);
		};
		
		LCD_Drawtext(" Choose a game:", x-106, 0);

		LCD_Drawtext(" Tetris", x-82, 2);	
		LCD_Drawtext(" Logoscroll", x-94, 3);
		LCD_Drawtext(" Shutdown", x-88, 4);
		
		LCD_Drawtext(">", x-100, 2);
		
		
		for (unsigned long i=0;i<500000;i++);
			asm volatile("nop"::);		
	};
	char item=0;
	
	while (1)
	{
		LCD_Clear();
		LCD_Drawtext(" Choose a game:", 22, 0);

		LCD_Drawtext(" Tetris", 46, 2);	
		LCD_Drawtext(" Logoscroll", 34, 3);
		LCD_Drawtext(" Shutdown", 40, 4);
		
		LCD_Drawmenumarker(item);
		
		char runmenu=0;
	
		while (runmenu==0)
		{
//			HandleKeys();	
			if (CheckKey(KEY_UP))
			{
				Sound_Beep(1000, 50);	
				
				if (item>0)
					item--;
				
				LCD_Drawmenumarker(item);						
			}
			else if (CheckKey(KEY_DOWN))
			{
				Sound_Beep(1000, 50);	
				
				if (item<2)
					item++;
				
				LCD_Drawmenumarker(item);						
			}
			else if (CheckKey(KEY_ENTER))
			{
				Sound_BeepST(2000, 150);	
				
				if (item==0)
					runmenu=1;				
				else if (item==1)
					runmenu=2;				
				else if (item==2)
				{
					Sound_BeepST(3220, 100);
					Sound_BeepST(880, 100);
					Sound_BeepST(440, 100);
					
					WDTCR=(1<<WDE);						
				}
			};		
		};	
		
		if (runmenu==1)
			Tetris();
		else if (runmenu==2)
			LCD_Scrolllogo();
	};

	
	
	return 0;	
}

