// Mr.MidiPlayer v1.0
// Plays all Midi-Files on formatted SD/MMC-Card (in the order they were copied to the card)
// Mr.MidiPlayer simply ignores FAT16/FAT32, so you MUST format your card, then copy your MIDI files on it. Use FAT16 then.
// NEVER delete and then copy files (just adding new files is OK, though) -> Reason is that fragmented files will most probably occur and Mr.MidiPlayer cannot handle such files.

// Hardware:
// ATmega8 (or ATmega32 if you don't have an ATmega8 lying around)
// No crystal required (8MHz internal clock is sufficient)
// Uses UART-Tx for MIDI-Data output
// SPI-IO plus extra Pin (CS) for SD-card
// Control buttons on PIND2-7

// Circuit Description:
/*
Take the circuit from Mr.Midi (the original Player/Rekorder) available on www.mikrocontroller.net!
Using 3.3V power supply only:
- ATmega8-L with SD-card without the voltage dividers:
	Pins 1,7,20 - 3.3V
	Pins 8,21,22 - GND
	Pin 16 (PORTB2) - MMC Pin 1 (CS)
	Pin 17 (PORTB3) - MMC Pin 2 (DI)
	Pin 18 (PORTB4) - MMC Pin 7 (DO)
	Pin 19 (PORTB5) - MMC Pin 5 (CLK)
*/

// Software:
// MMC-Driver from alp_mp3 - Open Source Atmel AVR based MP3 Player (modified, though)
// My own MIDI-File-Parser (Format 0 only)
// Compiles with WinAVR plus AVRStudio 4

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

// Recorder feature untested!!
//#define REKORDER

#ifndef F_CPU
	#define F_CPU 8000000
#endif

#define KEY_MASK 0x3F

#define KEY_FFW 0x01
#define KEY_PAUSE 0x02
#define KEY_PLAY 0x04
#define KEY_NEXT 0x08
//#define KEY_RESTART 0x10
#define KEY_BEGINNING 0x10

#define UART_BAUD_RATE 31250

#define START_SECTOR 528UL

#ifdef __AVR_ATmega32__
	#define LEDON(a) PORTC &= ~(1<<(a));
	#define LEDOFF(a) PORTC |= (1<<(a));
#else
	#define LEDON(a)
	#define LEDOFF(a)
#endif

char flags=0, dspd=0;
unsigned int old_tempo;
unsigned long time=0, delta_time=0, real_time=0;
unsigned long trk_len;


void send_all_off();


// MCU initialisieren
void hw_init(void) {
	// Ports
	DDRB = 0x00;
	PORTB = 0x3F;

	DDRD = 0x02;	// Input-Port, auer TxD
	PORTD = 0xFF;	// Pull-Ups aktivieren

	// Timer
	TCCR0 = 0;
	TCNT0 = 0;

	TCNT1 = 0;
	OCR1A = 2058;
	TCCR1A = 0;
	TCCR1B = 0;//8;	// Vorteiler 8 -> 1s Auflsung und CTC mode

	// Interrupts
	// mega8
	TIMSK = 0x51;	// Compare Match A Int. on, Overflow Timer 0 on
	TIFR = 0x51;	// Flag lschen

	// UART
	// Hilfsmakro zur UBRR-Berechnung ("Formel" laut Datenblatt)
	#define UART_UBRR_CALC(BAUD_,FREQ_) ((FREQ_)/((BAUD_)*16L)-1)
	UBRRH = (uint8_t)(UART_UBRR_CALC(UART_BAUD_RATE,F_CPU)>>8);
	UBRRL = (uint8_t)UART_UBRR_CALC(UART_BAUD_RATE,F_CPU);
	UCSRB = (1<<RXEN)|(1<<TXEN);//|(1<<RXCIE);//|(1<<UDRIE);

	//uart_init();
	
	//ffnet einen kanal fr printf
	//uart_connect_stdio();

	//Initialisierung der MMC/SD-Karte
	//printf ("System OK\n\n");	

	send_all_off();

	// give mmc time
	//delay_ms(100);
	// set the MMC pins
	LEDON(1);
	MMC_hw_init();
	spi_init();
	sei();
	delay_ms(10);
	MMC_init();
	LEDON(2);

	//Initialisierung der MMC/SD-Karte
	//printf ("Card OK\n\n");	
	delay_ms(10);

	//unsigned long b = mmc_find_buf("MThd", 4);
	//mmc_fill_buf(520UL);//520UL);//1946UL);
}


void sendbyte(unsigned char byte) {
	while (!(UCSRA & (1<<UDRE)));
	UDR = byte;
}


// Sends to all channels
void send_all_ch(unsigned char data) {
    unsigned char i;
    for (i=0; i<16; i++) {
        sendbyte(0xB0+i);
        sendbyte(data);
        sendbyte(0x00);
    }
}


// All midi off
void send_all_off(void) {
    send_all_ch(0x7B);
    send_all_ch(0x79);
}


unsigned char fetchbyte(void) {
	trk_len--;
	return mmc_fetch_buf();
}


#ifdef REKORDER
unsigned char getdata(void) {
	unsigned char b;
	do{
		while (!(UCSRA & (1<<RXC)));
		b = UDR;
	}while (b == 0xf8 || b == 0xfe);
	return b;
}


void writebyte(unsigned char b) {
	trk_len++;
	mmc_write_byte(b);
}


void writestring(char *buf, unsigned char len) {
	unsigned char cnt=0;
	for (cnt=0; cnt<len; cnt++)
		writebyte(buf[cnt]);
}
#endif


void checkbyte(unsigned char b) {
	if (b != fetchbyte())
	{
		LEDON(7);
		for(;1;);
	}
}


char checkstring(char *buf, unsigned char len) {
	unsigned char b, cnt=0;

	for (cnt=0; cnt<len; cnt++) {
		b = fetchbyte();
		if (b != (unsigned char)buf[cnt])
			return 0;
	}
	return 1;
}


#ifdef REKORDER
void write_varlen(unsigned long value) {
	unsigned long buffer = value & 0x7f;
	while ((value >>= 7) > 0) {
		buffer <<= 8;
		buffer |= 0x80;
		buffer += (value & 0x7f);
	}
	while (1) {
		writebyte((unsigned char)buffer);
		if (buffer & 0x80)
			buffer >>= 8;
		else
			break;
	}
}


void write_time(void) {
	write_varlen(time);
	time = 0;
}
#endif


/*
unsigned long fetch_varlen(void) {
	unsigned long value;
	unsigned char c;

	if ((value = fetchbyte()) & 0x80) {
		value &= 0x7f;
		do {
			value = (value << 7) + ((c = fetchbyte())) & 0x7f);
		} while (c & 0x80);
	}
	return value;
}
*/

unsigned long fetch_varlen(void) {
	unsigned long v;
	unsigned char c;

	c = fetchbyte();
	v = c;
	if ((v & 0x80) > 0) {
		v &= 0x7F;
		do {
			c = fetchbyte();
			v = (v << 7) + (c & 0x7F);
		} while (c & 0x80);
	}
  	return v;
}


void key_detect(void) {
	static unsigned char ok=0;
	unsigned char k = ((~PIND)>>2) & KEY_MASK;

	if (k != ok) {
		ok = k;
		if (ok != 0) {
			if (ok & KEY_FFW) {
				old_tempo = OCR1A;
				OCR1A = old_tempo>>1;
				TCNT1 = 0;
				dspd = 1;
			}
			else if (ok & KEY_NEXT) {
				send_all_off();
				trk_len = 0;
				mmc_complete_read();
			}
			else if (ok & KEY_BEGINNING) {
				send_all_off();
				trk_len = 0;
				mmc_complete_read();
				mmc_setsect(START_SECTOR-1UL);
			}
			else if (ok & KEY_PLAY) {
				TCNT1 = 0;
				TCCR1B = 8+2;	// Start Time
			}
			/*else if (ok & KEY_RESTART) {
				send_all_off();
				trk_len = 0;
				mmc_complete_read();
				mmc_setsect(sect_current);
			}*/
			else if (ok & KEY_PAUSE) {
				send_all_off();
				TCCR1B = 0;	// Stopp
			}
		}
		else if (dspd) {
			dspd = 0;
			OCR1A = old_tempo;
			TCNT1 = 0;
		}
	}
		
}


void wait_for_deltatime(void) {
	unsigned long dt;

	dt = fetch_varlen();
	if (time <= dt)
		delta_time = dt - time;
	else {
		delta_time = 0;
		time -= dt;
	}
	if (delta_time != 0) {
		flags = 2;
		while (flags & 2)
			key_detect();
	}
}


void main(void) {
	unsigned char ev, len, adp;
	unsigned int gl_timeset=96;
	unsigned long temp;

#ifdef __AVR_ATmega32__
	PORTC = 0xff;
	DDRC = 0xff;
#endif
	LEDON(0);
	hw_init();
	LEDON(3);

#ifndef REKORDER
	mmc_load_start(START_SECTOR);

	while (1) {
		LEDOFF(4);
		LEDOFF(5);
		LEDOFF(6);
		LEDOFF(7);
		//mmc_find_buf("MThd", 4);

		while (!checkstring("MThd", 4)) {
			mmc_complete_read();
		}
		checkbyte(0);
		checkbyte(0);
		checkbyte(0);
		checkbyte(6);
		checkbyte(0);
		checkbyte(0);
		checkbyte(0);
		checkbyte(1);
		gl_timeset = fetchbyte() << 8;
		gl_timeset |= fetchbyte();

		TCNT1 = 0;
		if (gl_timeset < 0x8000)	// Delta-Ticks
			OCR1A = 500000/gl_timeset;
		else				// SMPTE-Timeformat
			OCR1A = 1000000/(-(char)(gl_timeset>>8))/(unsigned char)gl_timeset;
	
		checkstring("MTrk", 4);
	
		temp = (unsigned long)fetchbyte() << 24;
		temp |= (unsigned long)fetchbyte() << 16;
		temp |= (unsigned long)fetchbyte() << 8;
		temp |= (unsigned long)fetchbyte();
		trk_len = temp;
		time = 0;
	
		TCCR1B = 8+2;	// Start Time
	
		LEDON(4);
		while (trk_len > 0) {// && filepos < 0x70) {
			//key_detect();
			wait_for_deltatime();
			if (trk_len == 0)
				break;
			ev = fetchbyte();
			if (ev == 0xFF) {	// META-EVENT
				ev = fetchbyte();
				switch (ev) {
					case 0x2f:	// End Of Track
						LEDON(5);
						checkbyte(0);
						send_all_off();
						trk_len = 0;
						mmc_complete_read();
						break;
					case 0x51:	// Tempo Change
						checkbyte(3);
						temp = (unsigned long)fetchbyte() << 16;
						temp |= (unsigned long)fetchbyte() << 8;
						temp |= (unsigned long)fetchbyte();
						TCNT1 = 0;
						old_tempo = temp/gl_timeset;
						if (dspd)
							OCR1A = old_tempo>>1;
						else
							OCR1A = old_tempo;						
						break;
					default:
						temp = fetch_varlen();
						while (temp>0) {
							temp--;
							fetchbyte();
						}
				}
			}
			else if (ev == 0xf0) {	// SYSEX
				sendbyte(0xf0);
				temp = fetch_varlen();
				while (temp>0) {
					temp--;
					ev = fetchbyte();
					sendbyte(ev);
				}
			}
			else if (ev >= 0x80) {	// MIDI-EVENT
				sendbyte(ev);
				switch (ev) {
					case 0xf2:
						len = 2;
						break;
					case 0xf3:
						len = 1;
						break;
					default:
						switch (ev & 0xf0) {
							case 0xf0:
								len = 0;
								break;
							case 0xc0:	// ProgramChange
							case 0xd0:	// ChannelAftertouch
								len = 1;
								break;
							case 0x80:	// NoteOff
							case 0x90:	// NoteOn
							case 0xa0:	// PolyphonicAftertouch
							case 0xb0:	// ControlModeChange
							case 0xe0:	// PitchWheel
								len = 2;
								break;
							default:
								for (;1;);
						}
				}
				adp = len;
				while (len > 0) {
					sendbyte(fetchbyte());
					len--;
				}
			}
			else {			// Running Status
				sendbyte(ev);
				len = adp-1;
				while (len > 0) {
					sendbyte(fetchbyte());
					len--;
				}
			}
		}
	}
#endif
#ifdef REKORDER
	// Rekorder-Task
	unsigned char cnt=0, dp[4];
	
	mmc_write_start(START_SECTOR);

	writestring("MThd", 4);
	writebyte(0);
	writebyte(0);
	writebyte(0);
	writebyte(6);
	writebyte(0);
	writebyte(0);
	writebyte(0);
	writebyte(1);
	writebyte(0);
	writebyte(240);
	writestring("MTrk", 4);
	writebyte(255);
	writebyte(255);
	writebyte(255);
	writebyte(255);
	OCR1A = 500000/240;
	trk_len = 0;
	time = 0;
	while (1) {
		ev = getdata();
		write_time();
		TCCR1B = 8+2;	// Start Time
		writebyte(ev);

		if (ev == 0xf0) {	// SysEx
			do {
				ev = getdata();
				writebyte(ev);
			} while (ev != 0xf7);
		}
		else if (ev < 0x80) {	// Running status
			len = adp-1;
			while (len > 0) {
				ev = getdata();
				writebyte(ev);
				len--;
			}
		}
		else {
			// MidiEvent
			switch (ev) {
			case 0xf2:
				len = 2;
				break;
			case 0xf3:
				len = 1;
				break;
			default:
				switch (ev & 0xf0) {
				case 0xf0:
					len = 0;
					break;
				case 0xc0:	// ProgramChange
				case 0xd0:	// ChannelAftertouch
					len = 1;
					break;
				case 0x80:	// NoteOff
				case 0x90:	// NoteOn
				case 0xa0:	// PolyphonicAftertouch
				case 0xb0:	// ControlModeChange
				case 0xe0:	// PitchWheel
					len = 2;
					break;
				default:
					len = 0;
				}
			}
			adp = len;
			while (len > 0) {
				ev = getdata();
				writebyte(ev);
				len--;
			}
		}
	}
	writebyte(0xff);
	writebyte(0x2f);
	writebyte(0);
	writebyte((unsigned char)(trk_len>>24));
	writebyte((unsigned char)(trk_len>>16));
	writebyte((unsigned char)(trk_len>>8));
	writebyte((unsigned char)trk_len);
	while(1);
#endif
}


ISR(SIG_OUTPUT_COMPARE1A) {
	time++;
	if ((flags & 2) && time>=delta_time) {
		flags = 1;
		time = 0;
	}
	real_time += OCR1A;
}
