#include <inttypes.h>
#include <math.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <avr/io.h>
#include <avr/eeprom.h>
#include <avr/pgmspace.h>
//#include <avr/signal.h>
//#include <avr/interrupt.h>
#include <util/delay.h>

#include "lcd.h"
#include "i2cmaster.h"

#ifndef sbi				// good old macros
#define BV(bit)			(1<<(bit))
#define cbi(addr,bit)	addr &= ~_BV(bit)
#define sbi(addr,bit)	addr |= _BV(bit)
#define outp(data,addr)	addr = (data)
#define inp(addr)		(addr)
#define outb(addr,data)	addr = (data)
#define inb(addr)		(addr)
#define PRG_RDB(addr)	(pgm_read_byte(addr))
#define	nop()	asm volatile("nop"::)
#endif


//I2C device address of  CS8416
#define CS8416_ADDRESS 0x20
//AD0 = 0
//AD1 = 0
//AD2 = 0
//BASE = 0010xxx

#define TAS5508_ADDRESS 0x36


// 1/3Oct Equalizer Frequencies
const uint16_t EQFreqs[32] = {0,20,25,32,40,50,63,80,
                        100,125,160,200,250,315,400,500,
                        630,800,1000,1250,1600,2000,2500,3150,
                        4000,5000,6300,8000,10000,12500,16000,20000};

// Strings to show
const char EQFreqStr[32][8] = {"None","20Hz","25Hz","32Hz",
                              "40Hz","50Hz","63Hz","80Hz",
                              "100Hz","125Hz","160Hz","200Hz",
                              "250Hz","315Hz","400Hz","500Hz",
                              "630Hz","800Hz","1.0kHz","1.25kHz",
                              "1.6kHz","2.0kHz","2.5kHz","3.15kHz",
                              "4.0kHz","5.0kHz","6.3kHz","8.0kHz",
                              "10kHz","12.5kHz","16kHz","20kHz"};

//string for Q each index (actual Q = Q^2/100)
const char EQQStr[32][8] = {"0.000","0.010","0.040","0.090",
							"0.160","0.250","0.360","0.490",
							"0.640","0.810","1.000","1.210",
							"1.440","1.690","1.960","2.250",
							"2.560","2.890","3.240","3.610",
							"4.000","4.410","4.840","5.290",
							"5.760","6.250","6.760","7.290",
							"7.840","8.410","9.000","9.610"};

const char EQTypeStr[4][12] = {"None","LShelf","HShelf","Peak"};

// Crossover Filter Point32_ts
const uint16_t FilterFreqs[32] =
                     {0,50,65,80,100,120,150,180,220,260,300,350,
                      400,500,650,800,1000,1200,1500,1800,2200,2600,3000,
                      3500,4000,5000,6500,8000,10000,12000,15000,18000};

const char FilterFreqStr[32][8] =
				{"None","50Hz","65Hz","80Hz",
				 "100Hz","120Hz","150Hz","180Hz",
				 "220Hz","260Hz","300Hz","350Hz",
				 "400Hz","500Hz","650Hz","800Hz",
				 "1kHz","1.2kHz","1.5kHz","1.8kHz",
				 "2.2kHz","2.6kHz","3.0kHz","3.5kHz",
				 "4.0kHz","5.0kHz","6.5kHz","8.0kHz",
				 "10kHz","12kHz","15kHz","18kHz"};

// CS8416 SPDIF Frequencies
const uint32_t SPDIFFreqs[4] = {44100,48000,88200,96000};
const char currentfsstr[4][8] = {"44k","48k","88k","96k"};


// Global Values
uint8_t currentfs;			//Fs (44.1/48/88.2/96) current state
uint8_t currentFilterMenu;	//Filter Menu Number
uint8_t FilterChanged;		//1: There were data change

uint8_t currentEQMenu;	//EQ Menu Number
uint8_t EQChanged;		//1: there were data change

uint8_t	KeyMode;	//0:none, 1: First Key, 2: Repeat Key
uint8_t PrevKey;	//remember previous key state
uint8_t	NewKey;		//get new key entry

// Parameters to store Flash
struct EQParam {
   uint8_t   EQType;   //0:None, 1:LowShelf, 2:HighShelf, 3:Peaking
   uint8_t   EQFreq;   //0 - 31
   uint8_t   EQQ;            //0 - 31
   int8_t   EQGain;   //-15,... 0,... 15
   };	//4 bytes

struct FilterParam {
   uint8_t LPF;   //0:No LPF
   uint8_t HPF;   //0:No HPF
   int8_t OFFSET; //0:mid
   uint8_t MUTE;  //0:normal, 1:mute
   };	//4 bytes

struct Params {
	int16_t Initialized;	//If not EEPROM initialized, this is 0xFF
	int16_t currentvol;
	int8_t currentch;
	int8_t currentwin;

	struct FilterParam FilterParams[4];   //CH12,34,56,78

	struct EQParam EQParams[4][3];   // CH12,34,56,78 - EQ5,6,7

} ;
struct Params DevParam;	//4 + 16 + 48 = 68 bytes
struct Params DevParam_EEPROM __attribute__((section(".eeprom")));

//Utility Functions

void msDelay(uint16_t delay)
{
	while (delay--) 
	{
	_delay_ms(1);
	} 
}

//Read Flash
void readDevParam()
{
	eeprom_busy_wait();
	eeprom_read_block(&DevParam, &DevParam_EEPROM,sizeof(DevParam_EEPROM));		

}

//Write Flash
void writeDevParam(int mode)
{
	eeprom_busy_wait();
	switch (mode)
		{
		case 0:	//Write All
			eeprom_write_block(&DevParam, &DevParam_EEPROM, sizeof(DevParam_EEPROM));
			break;
		case 1:	//Volume Only
			eeprom_write_word((uint16_t *)&(DevParam_EEPROM.currentvol), DevParam.currentvol);
			break;
		case 2:	//Channel Only
			eeprom_write_byte((uint8_t *)&(DevParam_EEPROM.currentch), DevParam.currentch);
			break;
		case 3:	//Window Only
			eeprom_write_byte( (uint8_t *)&(DevParam_EEPROM.currentwin), DevParam.currentwin);
			break;
		default:
			eeprom_write_block(&DevParam, &DevParam_EEPROM, sizeof(DevParam_EEPROM));
			break;
		}
	eeprom_busy_wait();
}




//LCD Function
void SendString(char * SendStr, uint8_t Line)
{
   int StrLen;
   char Packet[12];
   int i;

	lcd_gotoxy(0,0);
	msDelay(1);

	if (Line == 2)
	{
		lcd_gotoxy(0,1);
	}
	msDelay(1);

   StrLen = strlen(SendStr);
   memset(Packet, 0, sizeof(Packet));
   if (StrLen < 10)
   {
	   strncpy(Packet, SendStr, StrLen);
	   for (i = StrLen; i <= 10; i++)
	   {
	      Packet[i] = 0x20;
	   }
   }
   else
   {
	   strncpy(Packet, SendStr, 10);
   }

    lcd_puts(Packet);
	lcd_gotoxy(0,0);
}

//LCD Function
void ClearLCD()
{
	lcd_clrscr();
}

//KEY Function
uint8_t PollKey()
{

	if ( bit_is_clear(PINA, PINA0))	//PA0 = UP
		return 3;

	if ( bit_is_clear(PINA, PINA4))	//PA4 = DOWN
		return 4;

	if ( bit_is_clear(PINA, PINA1))	//PA1 = Left
		return 5;

	if ( bit_is_clear(PINA, PINA3))	//PA3 = Right
		return 6;

	if ( bit_is_clear(PINA, PINA2))	//PA2 = Enter
		return 1;
/*
         return 1;   //ENTER
         return 2;   //cancel
         return 3;   //up
         return 4;   //Down
         return 5;   //Left
         return 6;   //right
*/
   return 0;
}

//I2C Function
/* I2CWrite
write data to slave.
Sends START, slave address, index and then data, STOP
Waits for slave to respond. 
PARAM1: uint8_t address - slave device address
PARAM2: uint8_t index - subaddress of register/memory to write 
PARAM3: char * writestring - write data buffer
PARAM4: uint8_t len  - length of data to write   
RETURN: 0 or negative error code (see code)
*/  
int8_t I2CWrite(uint8_t address, uint8_t subaddress, char * writestring, uint8_t len)
{
	uint8_t  cnt;
	int8_t err;
	
	i2c_start(address + I2C_WRITE);

	err = i2c_write(subaddress);
	if (err == 1)
	{
		i2c_stop();
		return -20; // write failed
	}
	for (cnt = 0; cnt < len; cnt++)
	{
		i2c_write(writestring[cnt]);
    }
    i2c_stop();
	return 0;
}


/* I2CRead 
PARAM1: uint8_taddress - slave address
PARAM2: uint8_t subaddress - subaddress in in EEPROM to read from 
PARAM3: char *retstring          - buffer for data
PARAM4: uint8_t len   - size of data buffer 
          
RETURN: 0 or error code*/  
int8_t I2CRead(uint8_t address, uint8_t subaddress, char * retstring, uint8_t len)
{
	uint8_t cnt;
	int8_t err;

	i2c_start(address + I2C_WRITE);

	err = i2c_write(subaddress);
	if (err == 1)
	{
		i2c_stop();
		return -20; // access failed
	}

	i2c_rep_start(address + I2C_READ); 

	for (cnt = 0; cnt < len; cnt++)
	{
		if (cnt==(len-1))
		{
			retstring[cnt] = i2c_readNak();	//nak
		}
		else
        {
			retstring[cnt] = i2c_readAck();
		}
	}
	i2c_stop();
	return 0;
	
}

//I2C Functions
uint8_t I2C_GetFs()
{
   //returns data rate.
   //  0 : Error
   //  1 : invalid
   //  2 : 32 kHz
   //  3 : 38 kHz
   //  4 : 44.1kHz
   //  5 : 48 kHz
   //  6 : 88.2 kHz
   //  7 : 96 kHz
   //  8 : 176.4kHz
   //  9 : 192 kHz
   char Clk_Mode;
   uint8_t return_code;
   return_code = I2CRead(TAS5508_ADDRESS, 0x00, &Clk_Mode, 1);
   if (return_code != 0) {      return 0;   }
   //if ( (Clk_Mode & 0x01) == 0){  return 1; }
   //upper 3 bit shows mode
   return ( ((Clk_Mode & 0xE0) >> 5) + 2  );
}

// TAS5518 and PSVC Volume control
// PSVC is 4 bit programmable, 16dB
void I2C_SetVolume()
{
   char test_string[4];
   uint8_t return_code;
   int16_t TAS5508Volume;
   uint8_t	ControlVal;

   //OCR2 is Voltage controller.
   if (DevParam.currentvol < 0x48)   //+x.xx dB
   {
      TAS5508Volume = DevParam.currentvol;
	  //Voltage Control = 14V (18V supply)
	  OCR2 = 0xB3;
   }
   else
   {
      if (DevParam.currentvol < 0x89)  //0 - -15dB
      {
         TAS5508Volume = 0x48;	// keep 0dB
		//Ref Voltage is between 0.2 - 2.5V
		ControlVal = DevParam.currentvol / 4;
		switch (ControlVal)	// log calculation is too long.
			{
			case 18:
				OCR2 = 0xB3;	// result 14.00V
				break;
			case 19:
				OCR2 = 0x9F;
				break;
			case 20:
				OCR2 = 0x8E;
				break;
			case 21:
				OCR2 = 0x7E;
				break;
			case 22:
				OCR2 = 0x71;
				break;
			case 23:
				OCR2 = 0x64;
				break;
			case 24:
				OCR2 = 0x59;
				break;
			case 25:
				OCR2 = 0x50;
				break;
			case 26:
				OCR2 = 0x47;
				break;
			case 27:
				OCR2 = 0x3F;
				break;
			case 28:
				OCR2 = 0x38;
				break;
			case 29:
				OCR2 = 0x32;
				break;
			case 30:
				OCR2 = 0x2D;
				break;
			case 31:
				OCR2 = 0x28;
				break;
			case 32:
				OCR2 = 0x24;
				break;
			case 33:
				OCR2 = 0x20;
				break;
			}				
      }
      else
      {
         TAS5508Volume = DevParam.currentvol - 0x48;	//Offset PSVC dB
		 //Voltage control = 0.28V
		 OCR2 = 0x1C;
      }
   }

   //set volume
   test_string[0] = 0;
   test_string[1] = 0;
   test_string[2] = 0;
   test_string[3] = TAS5508Volume & 0xff;
   return_code = I2CWrite(TAS5508_ADDRESS, 0xD9, test_string, 4);

}


// CS8416 Rx Selector
void I2C_SetCH()
{
   char WriteData;
   WriteData = 0x80 + DevParam.currentch * 8 + DevParam.currentch;
   I2CWrite(CS8416_ADDRESS, 0x04, &WriteData, 1);
}

//Make biquad Filter parameter
void MakeEQ(int32_t Fs, int32_t f0, int32_t mode, double Q, int32_t gain,  char * EQ)
{
   double w0;
   double cosw0;
   double sinw0;
   double alpha;
   double A;
   double sqrtA;
   double twosqrtAalpha;
   double b0;
   double b1;
   double b2;
   double a0;
   double a1;
   double a2;
   int32_t Qb0;
   int32_t Qb1;
   int32_t Qb2;
   int32_t Qa1;
   int32_t Qa2;

   b0 = 0.0;
   b1 = 1.0;
   b2 = 0.0;
   a0 = 1.0;
   a1 = 0.0;
   a2 = 0.0;

   w0 = 2.000 * 3.141516 * f0 / Fs;
   cosw0 = cos(w0);
   sinw0 = sin(w0);

   alpha = sinw0 / (2.00 * Q);
   A =  pow( 10.00, ((double)(gain)) /40.0 );
   sqrtA = sqrt(A);
   twosqrtAalpha = 2*sqrtA*alpha;

   if ( mode == 0) //HPF
   {
      a0 = 1.0 + alpha;
      b1 = -1.0 * (1.0 + cosw0) / a0 ;
      b0 = -1.0 * b1 / 2.00;
      b2 = b0;
      a1 = -2.0 * cosw0 / a0 ;
      a2 = (1.0 - alpha) / a0;
   }
   if ( mode == 1) //LPF
   {
      a0 = 1.0 + alpha;
      b1 = (1.0 - cosw0) / a0 ;
      b0 = b1 / 2.00;
      b2 = b0;
      a1 = -2.0 * cosw0 / a0 ;
      a2 = (1.0 - alpha) / a0;
   }
   if ( mode == 2) //LowShelf
   {
      a0 = (A+1) + (A-1)*cosw0 + twosqrtAalpha;
      b0 = A*((A+1) - (A-1)*cosw0 + twosqrtAalpha);
      b1 = 2*A*((A-1) - (A+1)*cosw0) ;
      b2 = A*((A+1)-(A-1)*cosw0-twosqrtAalpha);
      a1 = -2.0*((A-1)+(A+1)*cosw0) ;
      a2 = (A+1)+(A-1)*cosw0-twosqrtAalpha;
   }
   if ( mode == 3) //HighShelf
   {
      a0 = (A+1)-(A-1)*cosw0+twosqrtAalpha;
      b0 = A*((A+1) + (A-1)*cosw0 + twosqrtAalpha);
      b1 = -2*A*((A-1) + (A+1)*cosw0) ;
      b2 = A*((A+1)+(A-1)*cosw0-twosqrtAalpha);
      a1 = 2.0*((A-1)-(A+1)*cosw0) ;
      a2 = (A+1)-(A-1)*cosw0-twosqrtAalpha;
   }
   if ( mode == 4) //Peaking
   {
      a0 = 1.0 + alpha / A;
      b0 = 1.0 + alpha * A;
      b1 = -2.0 * cosw0 ;
      b2 = 1.0 - alpha * A;
      a1 = -2.0 * cosw0 ;
      a2 = 1.0 - alpha / A;
   }
   if ( mode > 1)
   {
      b0 = b0 / a0;
      b1 = b1 / a0;
      b2 = b2 / a0;
      a1 = a1 / a0;
      a2 = a2 / a0;
   }
   //quantize
   Qb0 = (int32_t)(b0 *  8388608L);
   Qb1 = (int32_t)(b1 *  8388608L);
   Qb2 = (int32_t)(b2 *  8388608L);
   Qa1 = (int32_t)(a1 * -8388608L);
   Qa2 = (int32_t)(a2 * -8388608L);

   //Modify sign bit
   if (Qb0 < 0) Qb0 = Qb0 + 268435456L;
   if (Qb1 < 0) Qb1 = Qb1 + 268435456L;
   if (Qb2 < 0) Qb2 = Qb2 + 268435456L;
   if (Qa1 < 0) Qa1 = Qa1 + 268435456L;
   if (Qa2 < 0) Qa2 = Qa2 + 268435456L;

   //get each byte
   *(EQ   ) = (uint8_t)((Qb0>>24) & 0xFF);
   *(EQ+1 ) = (uint8_t)((Qb0>>16) & 0xFF);
   *(EQ+2 ) = (uint8_t)((Qb0>>8)  & 0xFF);
   *(EQ+3 ) = (uint8_t)( Qb0      & 0xFF);
   *(EQ+4 ) = (uint8_t)((Qb1>>24) & 0xFF);
   *(EQ+5 ) = (uint8_t)((Qb1>>16) & 0xFF);
   *(EQ+6 ) = (uint8_t)((Qb1>>8)  & 0xFF);
   *(EQ+7 ) = (uint8_t)( Qb1      & 0xFF);
   *(EQ+8 ) = (uint8_t)((Qb2>>24) & 0xFF);
   *(EQ+9 ) = (uint8_t)((Qb2>>16) & 0xFF);
   *(EQ+10) = (uint8_t)((Qb2>>8)  & 0xFF);
   *(EQ+11) = (uint8_t)( Qb2      & 0xFF);
   *(EQ+12) = (uint8_t)((Qa1>>24) & 0xFF);
   *(EQ+13) = (uint8_t)((Qa1>>16) & 0xFF);
   *(EQ+14) = (uint8_t)((Qa1>>8)  & 0xFF);
   *(EQ+15) = (uint8_t)( Qa1      & 0xFF);
   *(EQ+16) = (uint8_t)((Qa2>>24) & 0xFF);
   *(EQ+17) = (uint8_t)((Qa2>>16) & 0xFF);
   *(EQ+18) = (uint8_t)((Qa2>>8)  & 0xFF);
   *(EQ+19) = (uint8_t)( Qa2      & 0xFF);
}

void ClearEQ(char * EQ)
{
   //get each byte
   memset(EQ, 0,20);
   *(EQ+1 ) = 0x80;
}

// convert Q value to string
double EQQNumToValue(int32_t EQQ)
{
   double retVal;
   if ((EQQ < 0) || (EQQ > 31))
   {
      retVal = 0;
   }
   else
   {
      retVal = (EQQ * EQQ) / 100.00 ;
   }
   return retVal;
}

// convert EQ frequency value to string
int32_t FreqFromEQSetting(int32_t EQSetting)
{
   if ((EQSetting < 0) || (EQSetting > 31))
      return 0;

   return EQFreqs[EQSetting];
}

// convert Filter frequency value to string
int32_t FreqFromFilterSetting(int32_t FilterSetting)
{
   if ((FilterSetting < 0) || (FilterSetting > 31))
      return 0;

   return FilterFreqs[FilterSetting];
}

// Calculate Equalizers and apply
void applyEQ(int32_t FS)
{
   char EQ[20];
   int32_t return_code;
   int32_t Freq;
   double Q;

   int32_t i;
   int32_t j;

   //Loop for CH12,34,56,78
   for (j = 0; j <= 3; j++)
   {
      //Loop for EQ5,6,7 in each CH
      for (i = 0; i <= 2; i++)
      {
         Freq = FreqFromEQSetting(DevParam.EQParams[j][i].EQFreq);
         if ((Freq > 0) && (DevParam.EQParams[j][i].EQType > 0))
         {
            Q = EQQNumToValue(DevParam.EQParams[j][i].EQQ);
            MakeEQ(FS, Freq, DevParam.EQParams[j][i].EQType + 1, Q ,
             DevParam.EQParams[j][i].EQGain, EQ);
         }
         else
         {
            ClearEQ(EQ);
         }
         return_code = I2CWrite(TAS5508_ADDRESS, 0x55 + i + j*7*2,EQ,20);
         return_code = I2CWrite(TAS5508_ADDRESS, 0x5C + i + j*7*2,EQ,20);
      }
   }
}

// Calculate Filter and apply
void applyfilter(int32_t FS)
{
   char EQ[20];
   int32_t return_code;
   int32_t Freq;
   char Param[4];
   char MutePattern;
   int32_t i;

   for (i = 0; i <= 3; i++)
   {
      //CHxx EQ1,2 (LPF)
      Freq = FreqFromFilterSetting(DevParam.FilterParams[i].LPF);
      if (Freq > 0)
      {
         MakeEQ(FS, Freq, 1, 0.71, 0, EQ);     //Q=0.71 LPF
      }
      else
      {
         ClearEQ(EQ);
      }
      return_code = I2CWrite(TAS5508_ADDRESS, 0x51+i*14,EQ,20);
      return_code = I2CWrite(TAS5508_ADDRESS, 0x52+i*14,EQ,20);
      return_code = I2CWrite(TAS5508_ADDRESS, 0x58+i*14,EQ,20);
      return_code = I2CWrite(TAS5508_ADDRESS, 0x59+i*14,EQ,20);

      //CHxx EQ3,4 (HPF)
      Freq = FreqFromFilterSetting(DevParam.FilterParams[i].HPF);
      if (Freq > 0)
      {
         MakeEQ(FS, Freq, 0,  0.71, 0, EQ);     //Q=0.71, HPF
      }
      else
      {
         ClearEQ(EQ);
      }
      return_code = I2CWrite(TAS5508_ADDRESS, 0x53+i*14,EQ,20);
      return_code = I2CWrite(TAS5508_ADDRESS, 0x54+i*14,EQ,20);
      return_code = I2CWrite(TAS5508_ADDRESS, 0x5A+i*14,EQ,20);
      return_code = I2CWrite(TAS5508_ADDRESS, 0x5B+i*14,EQ,20);
   }

   Param[0] = 0;
   Param[1] = 0;
   Param[2] = 0;

   for (i = 0; i <= 3; i++)
   {
      //CH Offset
      Param[3] = (uint8_t)(0x48 - DevParam.FilterParams[i].OFFSET * 4);
      return_code = I2CWrite(TAS5508_ADDRESS, 0xD1+i*2,Param,4);
      return_code = I2CWrite(TAS5508_ADDRESS, 0xD2+i*2,Param,4);
   }

   //CH Mute
   MutePattern = (char)
                (DevParam.FilterParams[0].MUTE * 0x03 +
                 DevParam.FilterParams[1].MUTE * 0x0C +
                 DevParam.FilterParams[2].MUTE * 0x30 +
                 DevParam.FilterParams[3].MUTE * 0xC0);
   return_code = I2CWrite(TAS5508_ADDRESS, 0x0F, &MutePattern,1);
}


//Filter setting entry
void Menu01(int32_t NewKeyData)
{
   char Line1[20];
   char Line2[20];
   memset(Line1, 0, sizeof(Line1));
   memset(Line2, 0, sizeof(Line2));

      switch (NewKeyData)
      {
      case 1:  //Enter
      case 4:    //Down
         currentFilterMenu = 0;
         FilterChanged = 0;
         DevParam.currentwin = 3;
		 writeDevParam(3);
         return;
         break;
      case 2:  //Cancel
      case 3:    //Up
         DevParam.currentwin = 0;
		 writeDevParam(3);
		 return;
         break;
      case 5:    //Left
      case 6:   //Right
         DevParam.currentwin = 2;
		 writeDevParam(3);
         return;
         break;
      default:
         break;
      }

      //Create Screen String
      strcpy(Line1, "TAS4i");
      strcpy(Line2, "Filter?");
      //Send LCD
      SendString(Line1,1);
      SendString(Line2,2);
}

//EQ Setting Entry
void Menu02(int32_t NewKeyData)
{
   char Line1[20];
   char Line2[20];
   memset(Line1, 0, sizeof(Line1));
   memset(Line2, 0, sizeof(Line2));

      switch (NewKeyData)
      {
      case 1:  //Enter
      case 4:    //Down
		currentEQMenu = 0;
		EQChanged = 0;
         DevParam.currentwin = 4;
		 writeDevParam(3);
         return;
         break;
      case 2:  //cancel
      case 3:    //Up
         DevParam.currentwin = 0;
		 writeDevParam(3);
         return;
         break;
      case 5:    //Left
      case 6:   //Right
         DevParam.currentwin = 1;
		 writeDevParam(3);
         return;
         break;
      default:
         break;
      }

      //Create Screen String
      strcpy(Line1, "TAS4i");
      strcpy(Line2, "EQ Menu ?");
      //Send LCD
      SendString(Line1,1);
      SendString(Line2,2);
}

// Convert Filter value to string
void FilterNumToStr(int32_t FilterNum, char * FilterStr)
{

   if ((FilterNum < 0) || (FilterNum > 31))
      strcpy(FilterStr, "--");
   else
      strcpy(FilterStr, FilterFreqStr[FilterNum]);

}

void MuteToStr(int32_t MuteValue, char * MuteStr)
{
   if (MuteValue == 0)
   {
      strcpy(MuteStr, "In Use");
   }
   else
   {
      strcpy(MuteStr, "Mute");
   }
}

//Filter Menu Screen
void FilterMenu(int32_t NewKeyData)
{
   //Control EQ0,1(reserved for LPF) ,2,3 (for HPF)
   char Line1[20];
   char Line2[20];

   int32_t CurrentCH;
   int32_t CurrentFilterSetting;
   CurrentCH = (int32_t)(currentFilterMenu / 4);
   CurrentFilterSetting = currentFilterMenu - CurrentCH * 4;

   memset(Line1, 0, sizeof(Line1));
   memset(Line2, 0, sizeof(Line2));


      switch (NewKeyData)
      {
      case 1:  //Enter
         //save changes
         FilterChanged = 0;
         DevParam.currentwin = 3;
         applyfilter(SPDIFFreqs[currentfs - 4]);
           writeDevParam(0);
         return;
         break;
      case 2:   //Cancel
         //discard changes
         FilterChanged = 0;
           readDevParam();
         DevParam.currentwin = 1;
           writeDevParam(0);
         return;
         break;
      case 3:    //Up
         if (currentFilterMenu > 0)
            currentFilterMenu -= 1;
         else
		 {
		 	if (currentFilterMenu == 0)
			{
		         DevParam.currentwin = 1;
		 		 writeDevParam(3);
		         return;
			}
		 }
         break;
      case 4:    //Down
         if (currentFilterMenu < 15)
            currentFilterMenu += 1;
         else
            currentFilterMenu = 15;
         break;
      case 5:    //Left
         FilterChanged = 1;
         switch (CurrentFilterSetting)
         {
            case 0:   //LPF
               if (DevParam.FilterParams[CurrentCH].LPF > 0)
                   DevParam.FilterParams[CurrentCH].LPF -= 1;
               else
                   DevParam.FilterParams[CurrentCH].LPF = 0;
               break;
            case 1:   //HPF
               if (DevParam.FilterParams[CurrentCH].HPF > 0)
                   DevParam.FilterParams[CurrentCH].HPF -= 1;
               else
                   DevParam.FilterParams[CurrentCH].HPF = 0;
               break;
            case 2:   //OFFSET
               if (DevParam.FilterParams[CurrentCH].OFFSET > -15)
                   DevParam.FilterParams[CurrentCH].OFFSET -= 1;
               else
                   DevParam.FilterParams[CurrentCH].OFFSET = -15;
               break;
            case 3:   //MUTE
               if (DevParam.FilterParams[CurrentCH].MUTE == 1)
                   DevParam.FilterParams[CurrentCH].MUTE = 0;
               else
                   DevParam.FilterParams[CurrentCH].MUTE = 1;
               break;
         }
         break;
      case 6:   //Right
         FilterChanged = 1;
         switch (CurrentFilterSetting)
         {
            case 0:   //LPF
               if (DevParam.FilterParams[CurrentCH].LPF < 31)
                   DevParam.FilterParams[CurrentCH].LPF += 1;
               else
                   DevParam.FilterParams[CurrentCH].LPF = 31;
               break;
            case 1:   //HPF
               if (DevParam.FilterParams[CurrentCH].HPF < 31)
                   DevParam.FilterParams[CurrentCH].HPF += 1;
               else
                   DevParam.FilterParams[CurrentCH].HPF = 0;
               break;
            case 2:   //OFFSET
               if (DevParam.FilterParams[CurrentCH].OFFSET < 15)
                   DevParam.FilterParams[CurrentCH].OFFSET += 1;
               else
                   DevParam.FilterParams[CurrentCH].OFFSET = 15;
               break;
            case 3:   //MUTE
               if (DevParam.FilterParams[CurrentCH].MUTE == 0)
                   DevParam.FilterParams[CurrentCH].MUTE = 1;
               else
                   DevParam.FilterParams[CurrentCH].MUTE = 0;
               break;
         }
         break;
      default:
         //SendString("No Key",2);
         break;
      }

      //Create Screen String
         switch (CurrentFilterSetting)
         {
            case 0:   //LPF
               sprintf(Line1, "C%i%i LPF",(int8_t)CurrentCH*2+1,(int8_t)CurrentCH*2+2);
               FilterNumToStr(DevParam.FilterParams[CurrentCH].LPF,Line2);
               break;
            case 1:   //HPF
               sprintf(Line1, "C%i%i HPF",(int8_t)CurrentCH*2+1,(int8_t)CurrentCH*2+2);
               FilterNumToStr(DevParam.FilterParams[CurrentCH].HPF,Line2);
               break;
            case 2:   //OFFSET
               sprintf(Line1, "C%i%i OFS",(int8_t)CurrentCH*2+1,(int8_t)CurrentCH*2+2);
               sprintf(Line2, "%ddB", DevParam.FilterParams[CurrentCH].OFFSET);
               break;
            case 3:   //MUTE
               sprintf(Line1, "C%i%i MUT",(int8_t)CurrentCH*2+1,(int8_t)CurrentCH*2+2);
               MuteToStr(DevParam.FilterParams[CurrentCH].MUTE, Line2);
               break;
         }
      //Send LCD
      if (FilterChanged == 1)
      {
         strcat(Line1, "*");
      }
      SendString(Line1,1);
      SendString(Line2,2);
}

void EQTypeToStr(int32_t EQTypeNum, char * RetStr)
{
   if ((EQTypeNum < 0) || (EQTypeNum > 3))
      strcpy(RetStr, "--");
   else
      strcpy(RetStr, EQTypeStr[EQTypeNum]);
}

void EQFreqNumToStr(int32_t EQFreqNum, char * EQFreqRetStr)
{
   if ((EQFreqNum < 0) || (EQFreqNum > 31))
      strcpy(EQFreqRetStr, "--");
   else
      strcpy(EQFreqRetStr, EQFreqStr[EQFreqNum]);
}

void EQQNumToStr(int32_t EQQNum, char * EQQRetStr)
{
   if ((EQQNum < 0) || (EQQNum > 31))
      strcpy(EQQRetStr, "--");
   else
      strcpy(EQQRetStr, EQQStr[EQQNum]);
}

//EQ Menu Screen
void EQMenu(int32_t NewKeyData)
{
   //Control EQ5,6,7
   char Line1[20];
   char Line2[20];
  //double EQQ;
   int32_t CurrentCH; //0,1,2,3 = CH12,34,56,78
   int32_t CurrentEQ; //0,1,2 = EQ5,6,7
   int32_t CurrentSetting; //0,1,2,3 = EQType, EQFreq, EQQ, EQGain

   CurrentCH = (int32_t)(currentEQMenu / 12);
   CurrentEQ = (int32_t)((currentEQMenu - CurrentCH * 12) / 4);
   CurrentSetting = currentEQMenu - CurrentCH * 12 - CurrentEQ * 4;

   memset(Line1, 0, sizeof(Line1));
   memset(Line2, 0, sizeof(Line2));

      switch (NewKeyData)
      {
      case 1:  //Enter
         //save changes
         EQChanged = 0;
         DevParam.currentwin = 4;
         applyEQ(SPDIFFreqs[currentfs - 4]);
           writeDevParam(0);
         return;
         break;
      case 2: //Cancel
         //discard changes
         EQChanged = 0;
         readDevParam();
         DevParam.currentwin = 2;
           writeDevParam(0);
         return;
         break;
      case 3:    //Up
         if (currentEQMenu > 0)
            currentEQMenu -= 1;
         else
		 {
		 	if (currentEQMenu == 0)
		 	{
         		DevParam.currentwin = 2;
		        writeDevParam(3);
				return;
			}
		 }	
         break;
      case 4:    //Down
         if (currentEQMenu < 47)
            currentEQMenu += 1;
         else
		 	currentEQMenu = 47;
         break;
      case 5:    //Left
         EQChanged = 1;
         switch (CurrentSetting)
         {
            case 0:
               if ( DevParam.EQParams[CurrentCH][CurrentEQ].EQType > 0 )  //0:None, 1:LowShelf, 2:HighShelf, 3:Peaking
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQType -= 1;
               else
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQType = 0;
               break;
            case 1:
               if ( DevParam.EQParams[CurrentCH][CurrentEQ].EQFreq > 0 )   //0 - 31
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQFreq -= 1;
               else
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQFreq = 0;
               break;
            case 2:
               if ( DevParam.EQParams[CurrentCH][CurrentEQ].EQQ > 0 )     //0 - 31
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQQ -= 1;
               else
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQQ = 0;
               break;
            case 3:
               if ( DevParam.EQParams[CurrentCH][CurrentEQ].EQGain > -15 )   //-15,... 0,... 15
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQGain -= 1;
               else
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQGain = -15;
               break;
            default:
               break;
         }
         break;
      case 6:   //Right
         EQChanged = 1;
         switch (CurrentSetting)
         {
            case 0:
               if ( DevParam.EQParams[CurrentCH][CurrentEQ].EQType < 3 )  //0:None, 1:LowShelf, 2:HighShelf, 3:Peaking
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQType += 1;
               else
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQType = 3;
               break;
            case 1:
               if ( DevParam.EQParams[CurrentCH][CurrentEQ].EQFreq < 31 )   //0 - 31
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQFreq += 1;
               else
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQFreq = 31;
               break;
            case 2:
               if ( DevParam.EQParams[CurrentCH][CurrentEQ].EQQ < 31 )     //0 - 31
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQQ += 1;
               else
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQQ = 31;
               break;
            case 3:
               if ( DevParam.EQParams[CurrentCH][CurrentEQ].EQGain < 15 )   //-15,... 0,... 15
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQGain += 1;
               else
                  DevParam.EQParams[CurrentCH][CurrentEQ].EQGain = 15;
               break;
            default:
               break;
         }
         break;
      default:
         //SendString("No Key",2);
         break;
      }

      //Create Screen String
      switch (CurrentSetting)
      {
         case 0:   //EQType
            sprintf(Line1, "C%d%dE%dT",(int8_t)CurrentCH*2+1,(int8_t)CurrentCH*2+2,(int8_t)CurrentEQ+5 );
            EQTypeToStr(DevParam.EQParams[CurrentCH][CurrentEQ].EQType ,Line2);
            break;
         case 1:   //EQFreq
            sprintf(Line1, "C%d%dE%dF",(int8_t)CurrentCH*2+1,(int8_t)CurrentCH*2+2,(int8_t)CurrentEQ+5);
            EQFreqNumToStr(DevParam.EQParams[CurrentCH][CurrentEQ].EQFreq, Line2);
            break;
         case 2:   //EQQ
            sprintf(Line1, "C%d%dE%dQ",(int8_t)CurrentCH*2+1,(int8_t)CurrentCH*2+2,(int8_t)CurrentEQ+5);
            EQQNumToStr(DevParam.EQParams[CurrentCH][CurrentEQ].EQQ, Line2);
            break;
         case 3:
            sprintf(Line1, "C%d%dE%dG",(int8_t)CurrentCH*2+1,(int8_t)CurrentCH*2+2,(int8_t)CurrentEQ+5);
            sprintf(Line2, "%ddB", DevParam.EQParams[CurrentCH][CurrentEQ].EQGain);
            break;
         default:
            break;
      }
      //Send LCD
      if (EQChanged == 1)
      {
         strcat(Line1, "*");
      }

      SendString(Line1,1);
      SendString(Line2,2);
}

// Main Screen to show Volume, Channel
void mainScreen(int32_t NewKeyData)
{
   char Line1[20];
   char Line2[20];
   int Volume;

   Volume = (int32_t)(((DevParam.currentvol - 0x48) * -1) / 4);
   //writeDevParam(3);
   memset(Line1, 0, sizeof(Line1));
   memset(Line2, 0, sizeof(Line2));


      switch (NewKeyData)
      {
      case 1:  //Enter
         //SendString("Enter Pressed",2);
         DevParam.currentwin = 1;
           writeDevParam(0);
         return;
         break;
      case 2:
         //SendString("Cancel Pressed",2);
         break;
      case 3:    //Up
         if (Volume < 16)
         {
            Volume += 1;
            DevParam.currentvol = ((Volume * 4) * -1) + 0x48;
            writeDevParam(1);
            I2C_SetVolume();
         }
         else
            Volume = 16;
         break;
      case 4:    //Down
         if (Volume > -100)
         {
            Volume -= 1;
            DevParam.currentvol = ((Volume * 4) * -1) + 0x48;
            writeDevParam(1);
            I2C_SetVolume();
         }
         else
            Volume = -100;
         break;
      case 5:    //Left
         if (DevParam.currentch > 0)
         {
            DevParam.currentch -= 1;
            writeDevParam(2);
            I2C_SetCH();
         }
         else
            DevParam.currentch = 0;
         break;
      case 6:   //Right
         if (DevParam.currentch < 7)
         {
            DevParam.currentch += 1;
            writeDevParam(2);
            I2C_SetCH();
         }
         else
            DevParam.currentch = 7;
         break;
      default:
         break;
      }

      //Create Screen String
	  strcpy(Line1, "TAS4i D");
	  Line1[7] = 0x30 + DevParam.currentch;
//      sprintf(Line1, "TAS4i D%d", DevParam.currentch);

	  sprintf(Line2, "%2d", Volume);
      if ((currentfs < 4) || (currentfs > 7))
	  	strcat(Line2, " --k");
      else
	  	strcat(Line2, " ");
		strcat(Line2, currentfsstr[currentfs - 4]);
//         sprintf(Line2, "%2d %s", Volume, currentfsstr[currentfs - 4]);

      //Send LCD
      SendString(Line1,1);
      SendString(Line2,2);

}

// Get Key / pass screen
void ProcessKey(int32_t NewKeyData)
{
   switch (DevParam.currentwin)
   {
      case 0:
         mainScreen(NewKeyData);
         break;
      case 1:
         Menu01(NewKeyData);
         break;
      case 2:
         Menu02(NewKeyData);
         break;
      case 3:
         FilterMenu(NewKeyData);
         break;
      case 4:
         EQMenu(NewKeyData);
         break;
      default:
         mainScreen(NewKeyData);
         break;
   }
}


// Init
void initializeBoard()
{
   char WriteData;
   char LCDLine[20];
 
 	//PortC is used for LCD output
	DDRC=0xF7;
	PORTC=0x00;

	//PortA is for Key btn input
	DDRA = 0xE0;	//PortA, 7,6,5 is out, other in
	PORTA = 0xBF;	//LED Off

	//PortA is used for I2C
	DDRE=0xFF;	//PE is output
	PORTE = 0xC8;	//Reset High

    msDelay(10);

   lcd_init(LCD_DISP_ON);
   lcd_home();

  i2c_init();
   currentfs = 0;

	//Initialize key state
	KeyMode = 0;
	PrevKey = 0;
	NewKey = 0;

   readDevParam();
   //Once for Initial.
   	if (DevParam.Initialized != 0)
	{
		memset(LCDLine, 0, sizeof(LCDLine));
		strcpy(LCDLine, "Init");
	    SendString(LCDLine,1);
		memset(LCDLine, 0, sizeof(LCDLine));
		strcpy(LCDLine, "EEPROM.");
	    SendString(LCDLine,2);
    	msDelay(1500);
	   	memset(&DevParam, 0, sizeof(DevParam));
		DevParam.currentvol = 0xE0;
		DevParam.FilterParams[0].MUTE = 1;
		DevParam.FilterParams[1].MUTE = 1;
		DevParam.FilterParams[2].MUTE = 1;
		DevParam.FilterParams[3].MUTE = 1;
		writeDevParam(0);
	}	
   ClearLCD();

	memset(LCDLine, 0, sizeof(LCDLine));
	strcat(LCDLine, "K.D.A.");
    SendString(LCDLine,1);
	memset(LCDLine, 0, sizeof(LCDLine));
	strcat(LCDLine, "TAS-4i2");
    SendString(LCDLine,2);
    msDelay(1500);


   //Serial Audio Format = 1 0 00 0 1 0 1   (I2S 24bit)
   WriteData = 0x85;   //I2S
   I2CWrite(CS8416_ADDRESS, 0x05, &WriteData, 1);

   //Control3 = 10110000
   WriteData = 0xB0;   //1011 GPIO is TX passthrough
   I2CWrite(CS8416_ADDRESS, 0x03, &WriteData, 1);
   WriteData = 0x80;
   I2CWrite(CS8416_ADDRESS, 0x04, &WriteData, 1);

   I2C_SetCH();

	//DevParam.currentvol = 0xF0;
    //set volume
    I2C_SetVolume();

	ClearLCD();
}

// Main Loop
int main()
{
   int32_t return_code;
   int32_t newfs;
   char MUTEMode;

	//OCR2 is written from CPU, stores Compare value
	OCR2 = 0x05;	//initial value
	//TCNT2 is Timer/Counter, counts up by TCR2 
	TCNT2 = 0x00;	//Clear
	//TCCR2 is control register.
	//FOC2 = 0, WGM20 = 1, COM21 = 1, COM20 = 0, WGM21 = 1, CS = 001
	TCCR2 = 0x69;

	//PortB is used for Volume Control.
	DDRB = 0xF0;	//PB7,6,5,4 is output
	PORTB = 0x00;

   msDelay(100);

   initializeBoard();

   MUTEMode = 0x00;       //Unmute all
   return_code = I2CWrite(TAS5508_ADDRESS, 0x0F, &MUTEMode, 1);

   while (1)
   {
		switch(KeyMode)
		{
		case 0:
		  msDelay(50);
		  break;
		case 1:
			msDelay(400);
			break;
		case 2:
			msDelay(200);
			break;
		default:
			msDelay(100);
			break;
		}
     {
         //task for check fs
         newfs = I2C_GetFs();
         if (currentfs != newfs)
         {
            if ((newfs < 4) || (newfs > 7)) //non support
            {
                //mute all
                MUTEMode = 0xFF;
                I2CWrite(TAS5508_ADDRESS, 0x0f, &MUTEMode, 1);
            }
            else
            {
                applyfilter(SPDIFFreqs[newfs-4]);
                applyEQ(SPDIFFreqs[newfs-4]);
            }
            currentfs = newfs;
         }
     }
     //Task Key Process
     {
         NewKey = PollKey();
			switch(KeyMode)
			{
			case 0:	//Initial Key State, None Key detected
				if (NewKey > 0)
				{
					KeyMode = 1;	//to state 1.
					if (NewKey != PrevKey)	{	ProcessKey(NewKey);	}
					PrevKey = NewKey;
				}
				else	{ ProcessKey(0); }
			  break;
			case 1:	//Some key detected. wait 400 ms to another input.
				if (NewKey > 0)
				{
					if (NewKey != PrevKey) 	{	KeyMode = 1;	}	//different key detected.
					else  					{	KeyMode = 2;	}	//Same key detected.
		         	ProcessKey(NewKey);
					PrevKey = NewKey;
				}
				else	//no key pressed. return to state 0.
				{
					KeyMode = 0;
					PrevKey = NewKey;
					ProcessKey(0);
				}
				break;
			case 2:	//If same key still pressed, run same routine per 100ms.
				if (NewKey > 0)
				{
					if (NewKey != PrevKey)	{	KeyMode = 1; 	}	//different key detected.
					else					{	KeyMode = 2;	}	//repeat same key operation
		         	ProcessKey(NewKey);
					PrevKey = NewKey;
				}
				else
				{
					KeyMode = 0;
					PrevKey = NewKey;
					ProcessKey(0);
				}
				break;
			default:
				break;
			}	//switch
	     }	//Task
    }	//While

	return 0;
}


