/******************************************************************************\
* Projekt:                                                                     *
* Ziel-System: LPC17xx                                                         *
* Autor: Marco Kaminski                                                        *
* Version: v0.a.002                                                            *
* Datum: 01.12.2012                                                            *
*                                                                              *
* Beschreibung:                                                                *
* OHCI-Stack fuer LPC17xx-Prozessoren                                          *
*                                                                              *
* Changelog:                                                                   *
* v0.a.002 - Umstellung von Bitschubsen auf structs.                           *
* v0.a.001 - Erste Version                                                     *
*                                                                              *
* Lizenz:                                                                      *
* Dieses Programm ist freie Software. Sie können es unter den Bedingungen      *
* der GNU General Public License, wie von der Free Software Foundation         *
* veröffentlicht, weitergeben und/oder modifizieren, entweder gemäß Version    *
* 2 der Lizenz oder (nach Ihrer Option) jeder späteren Version.                *
*                                                                              *
* Die Veröffentlichung dieses Programms erfolgt in der Hoffnung, daß es        *
* Ihnen von Nutzen sein wird, aber OHNE IRGENDEINE GARANTIE, sogar ohne die    *
* implizite Garantie der MARKTREIFE oder der VERWENDBARKEIT FÜR EINEN          *
* BESTIMMTEN ZWECK. Details finden Sie in der GNU General Public License.      *
*                                                                              *
* Sie sollten eine Kopie der GNU General Public License zusammen mit diesem    *
* Programm erhalten haben.                                                     *
* Falls nicht, schreiben Sie an die Free Software Foundation,                  *
* Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.                *
\******************************************************************************/
#ifndef OHCI_H_
#define OHCI_H_

// Missing Defines in LPC17xx.h
#define	 AHB_SRAM0					0x2007C000			//< Address of AHB SRAM bank0
#define	 AHB_SRAM1					0x20080000			//< Address of AHB SRAM bank1

// Memory-Locations
#define MEM_HC						(LPC_USB_BASE)
#define MEM_HCRH					(&(LPC_USB->HcRhDescriptorA))
#define MEM_HCCTRL					(&(LPC_USB->HcControl))
#define MEM_HCCMDST					(&(LPC_USB->HcCommandStatus))
#define	 MEM_HCCA					(AHB_SRAM1 + 0x000)
#define	 MEM_TDHEAD					(AHB_SRAM1 + 0x100)
#define	 MEM_TDTAIL					(AHB_SRAM1 + 0x110)
#define	 MEM_EDCTRL					(AHB_SRAM1 + 0x120)
#define	 MEM_EDBULKIN				(AHB_SRAM1 + 0x130)
#define	 MEM_EDBULKOUT				(AHB_SRAM1 + 0x140)
#define	 MEM_TDBUFFER				(AHB_SRAM1 + 0x150)

// USB-Pin Function
#define	 USB_UP_LED					1					//< 1[18] Function 1 USB_UP_LED
#define	 USB_PPWR					2					//< 1[19] Function 2 /USB_PPWR
#define	 USB_PWRD					2					//< 1[22] Function 2 USB_PWRD
#define	 USB_OVRCR					2					//< 1[27] Function 2 /USB_OVRCR
#define	 USB_DP						1					//< 0[29] Function 1 USB_D+
#define	 USB_DM						1					//< 0[30] Function 1 USB_D-

// Power Control for Peripherals register (PCONP)
#define	 PCUSB 						31					//< USB interface power/clock control bit

// USB/OTG Clock Control Register (USBClkCtrl/OTGClkCtrl)
// USB/OTG Clock Status Register (USBClkSt/OTGClkSt)
#define	 HOST_CLK					0					//< Host clock enable bit
#define	 DEV_CLK					1					//< Host clock enable bit
#define	 I2C_CLK					2					//< Host clock enable bit
#define	 OTG_CLK					3					//< Host clock enable bit
#define	 AHB_CLK					4					//< Host clock enable bit

// HcRevision Register
#define HcRevision_REV				0x000000FF

// HcInterruptStatus Register
// HcInterruptEnable Register
// HcInterruptDisable Register
#define HcInterrupt_SO				0x00000001
#define HcInterrupt_WDH				0x00000002
#define HcInterrupt_SF				0x00000004
#define HcInterrupt_RD				0x00000008
#define HcInterrupt_UE				0x00000010
#define HcInterrupt_FNO				0x00000020
#define HcInterrupt_RHSC			0x00000040
#define HcInterrupt_OC				0x40000000
#define HcInterrupt_MIE				0x80000000

// 4.4.1 Host Controller Communications Area Format Seite 48 in OHCI-Spec
typedef struct hcca {									//< Host Controller Communication Area
	uint32_t IntTable[32];								//< Interrupt Table
	uint16_t FrameNumber;								//< Frame Number
	uint16_t Pad1;
	uint32_t DoneHead;									//< Done Head
	uint8_t  Reserved[116];								//< Reserved for future use
} HCCA;

// 7.1.2 HcControl Register Seite 123 in OHCI-Spec
typedef struct hcCtrl {
	uint32_t CBSR		:2;								//< ControlBulkServiceRatio
	uint32_t PLE		:1;								//< PeriodicListEnable
	uint32_t IE			:1;								//< IsochronousEnable
	uint32_t CLE		:1;								//< ControlListEnable
	uint32_t BLE		:1;								//< BulkListEnable
	uint32_t HCFS		:2;								//< HostControllerFunctionalState for USB
	uint32_t IR			:1;								//< InterruptRouting
	uint32_t RWC		:1;								//< RemoteWakeupConnected
	uint32_t RWE		:1;								//< RemoteWakeupEnable
	uint32_t			:21;
} HCCTRL;

#define CBSR_1						0
#define CBSR_2						1
#define CBSR_3						2
#define CBSR_4						3
#define HCFS_USBRESET				0
#define HCFS_USBRESUME				1
#define HCFS_USBOPERATIONAL			2
#define HCFS_USBSUSPEND				3

// 7.1.3 HcCommandStatus Register Seite 126 in OHCI-Spec
typedef struct hcCmdSt {
	uint32_t HCR		:1;								//< HostControllerReset
	uint32_t CLF		:1;								//< ControlListFilled
	uint32_t BLF		:1;								//< BulkListFilled
	uint32_t OCR		:1;								//< OwnershipChangeRequest
	uint32_t			:12;
	uint32_t SOC		:2;								//< SchedulingOverrunCount
	uint32_t			:14;
} HCCMDST;

// 7.4 Root Hub Partition Seite 137 in OHCI-Spec
typedef struct hcRh{
	// LPC_USB->HcRhDescriptorA Register
	struct {
		uint8_t  NDP;									//< NumberDownstreamPorts
		uint32_t PSM	:1;								//< PowerSwitchingMode
		uint32_t NPS	:1;								//< NoPowerSwitching
		uint32_t DT		:1;								//< DeviceType
		uint32_t OCPM	:1;								//< OverCurrentProtectionMode
		uint32_t NOCP	:1;								//< NoOverCurrentProtection
		uint32_t		:11;
		uint8_t  POTPG;									//< PowerOnToPowerGoodTime
	} DescriptorA;

	// LPC_USB->HcRhDescriptorB Register
	struct {
		uint16_t DR;									//< DeviceRemovable
		uint16_t PPCM;									//< PortPowerControlMask
	} DescriptorB;

	// LPC_USB->HcRhStatus Register
	struct {
		uint32_t LPS	:1;								//< (read)  LocalPowerStatus
														//  (write) ClearGlobalPower
		uint32_t OCI	:1;								//< OverCurrentIndicator
		uint32_t		:13;
		uint32_t DRWE	:1;								//< (read)  DeviceRemoteWakeupEnable
														//  (write) SetRemoteWakeupEnable
		uint32_t LPSC	:1;								//< (read)  LocalPowerStatusChange
														//  (write) SetGlobalPower
		uint32_t OCIC	:1;								//< OverCurrentIndicatorChange
		uint32_t		:13;
		uint32_t CRWE	:1;								//< ClearRemoteWakeupEnable
	} Status;

	// LPC_USB->HcRhPortStatus[1:NDP] Register
	struct {
		uint32_t CCS	:1;								//< (read)  CurrentConnectStatus
														//  (write) ClearPortEnable
		uint32_t PES	:1;								//< (read)  PortEnableStatus
														//  (write) SetPortEnable
		uint32_t PSS	:1;								//< (read)  PortSuspendStatus
														//  (write) SetPortSuspend
		uint32_t POCI	:1;								//< (read)  PortOverCurrentIndicator
														//  (write) ClearSuspendStatus
		uint32_t PRS	:1;								//< (read)  PortResetStatus
														//  (write) SetPortReset
		uint32_t		:3;
		uint32_t PPS	:1;								//< (read)  PortPowerStatus
														//  (write) SetPortPower
		uint32_t LSDA	:1;								//< (read)  LowSpeedDeviceAttached
														//  (write) ClearPortPower
		uint32_t		:6;
		uint32_t CSC	:1;								//< ConnectStatusChange
		uint32_t PESC	:1;								//< PortEnableStatusChange
		uint32_t PSSC	:1;								//< PortSuspendStatusChange
		uint32_t OCIC	:1;								//< PortOverCurrentIndicatorChange
		uint32_t PRSC	:1;								//< PortResetStatusChange
		uint32_t 		:11;
	} PortStatus[15];
} HCRH;

// 4.2.1 Endpoint Descriptor Format Seite 30 in OHCI-Spec
typedef struct hcEd {
	// DWORD0 - Control
	struct {
		uint32_t FA		:7;								//< FunctionAddress
		uint32_t EN		:4;								//< EndpointNumber
		uint32_t D		:2;								//< Direction
		uint32_t S		:1;								//< Speed
		uint32_t K		:1;								//< sKip
		uint32_t F		:1;								//< Format
		uint32_t MPS	:11;							//< MaximumPacketSize
		uint32_t		:5;
	} Control;

	// DWORD1 - TailTd
	uint32_t TailTD;									//< TDQueueTailPointer

	// DWORD2 - HeadTd
	struct {
		uint32_t H		:1;								//< Halted
		uint32_t C		:1;								//< toggleCarry
		uint32_t res0	:2;								//< reserved set to 0
		uint32_t HeadP	:28;							//< TDQueueHeadPointer
	} HeadTD;

	// DWORD3 - Next
	uint32_t NextTD;									//< NextED
} HCED;

// 4.3.1.2 General Transfer Descriptor Field Definitions Seite 34 in OHCI-Spec
typedef struct hcGtd {									//< HostController Transfer Descriptor
	// DWORD0 - Control
	struct {
		uint32_t		:18;
		uint32_t R		:1;								//< Buffer Rounding
		uint32_t DP		:2;								//< Direction/PID
		uint32_t DI		:3;								//< DelayInterrupt
		uint32_t T		:2;								//< DataToggle
		uint32_t EC		:2;								//< ErrorCount
		uint32_t CC		:4;								//< ConditionCode
	} Control;

	// DWORD1 - CurrBuffer
	uint32_t CurrBufPtr;								//< Physical address of current buffer pointer

	// DWORD2 - Next TD
	uint32_t NextTD;									//< Physical pointer to next Transfer Descriptor

	// DWORD3 - Buffer End
	uint32_t BuffEnd;									//< Physical address of end of buffer
} HCGTD;

#define TD_SETUP					0					//< Direction of Setup Packet
#define TD_OUT						1					//< Direction Out
#define TD_IN						2					//< Direction In
#define TD_TOGGLE_0					2					//< Toggle 0
#define TD_TOGGLE_1					3					//< Toggle 1

typedef struct setup {
	uint8_t  rescipient	:5;								//<	Recipient
	uint8_t  type		:2;								//< Command Type
	uint8_t  direction	:1;								//< Data Direction
	uint8_t  request;									//< Specific Request
	uint16_t value;										//< Use varies according to request
	uint16_t index;										//< Use varies according to request
	uint16_t length;									//< Number of bytes to transfer if there is a data stage
} SETUP;

// Control RequestType Fields
#define RECIPIENT_DEVICE			0
#define RECIPIENT_INTERFACE			1
#define REQUEST_TYPE_STANDARD		0
#define REQUEST_TYPE_CLASS			1
#define REQUEST_TYPE_VENDOR			2
#define HOST_TO_DEVICE				0
#define DEVICE_TO_HOST				1

// Descriptor Types
#define DEVICE_DESCRIPTOR			1
#define CONFIGURATION_DESCRIPTOR	2
#define STRING_DESCRIPTOR			3
#define INTERFACE_DESCRIPTOR		4
#define ENDPOINT_DESCRIPTOR			5
#define DEVICE_QUALIFIER			6
#define OTHER_SPED_CONFIGURATION	7
#define INTERFACE_POWER				8
#define ON_THE_GO					9

// USB Standard Requests
#define GET_STATUS					0
#define CLEAR_FEATURE				1
#define SET_FEATURE					3
#define SET_ADDRESS					5
#define GET_DESCRIPTOR				6
#define SET_DESCRIPTOR				7
#define GET_CONFIGURATION			8
#define SET_CONFIGURATION			9
#define GET_INTERFACE				10
#define SET_INTERFACE				11
#define SYNCH_FRAME					12

void	init_Addresses		(void);
void	init_USBHost		(void);
void	initStruct_HCTD	(volatile HCGTD *td);
void	initStruct_HCED	(volatile HCED  *ed);
void	initStruct_HCCA	(volatile HCCA  *hcca);
int32_t processDoneQueue	(void);
int32_t	enumerateDevice	(void);
uint8_t assignAddress		(void);
void	releaseAddress		(uint8_t a);
int16_t selectConfig		(USBConfig_t *configlist, uint8_t numConfigs);

int32_t ControlTransfer	(uint8_t direction, uint8_t type, uint8_t recipient, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, uint16_t wLength, volatile uint8_t *buffer);

void	USB_IRQHandler		(void);

#endif
