/******************************************************************************\
* Projekt:                                                                     *
* Ziel-System: LPC17xx                                                         *
* Autor: Marco Kaminski                                                        *
* Version: v0.a.002                                                            *
* Datum: 01.12.2012                                                            *
*                                                                              *
* Beschreibung:                                                                *
* OHCI-Stack fuer LPC17xx-Prozessoren                                          *
*                                                                              *
* Changelog:                                                                   *
* v0.a.002 - Umstellung von Bitschubsen auf structs.                           *
* v0.a.001 - Erste Version                                                     *
*                                                                              *
* Lizenz:                                                                      *
* Dieses Programm ist freie Software. Sie können es unter den Bedingungen      *
* der GNU General Public License, wie von der Free Software Foundation         *
* veröffentlicht, weitergeben und/oder modifizieren, entweder gemäß Version    *
* 2 der Lizenz oder (nach Ihrer Option) jeder späteren Version.                *
*                                                                              *
* Die Veröffentlichung dieses Programms erfolgt in der Hoffnung, daß es        *
* Ihnen von Nutzen sein wird, aber OHNE IRGENDEINE GARANTIE, sogar ohne die    *
* implizite Garantie der MARKTREIFE oder der VERWENDBARKEIT FÜR EINEN          *
* BESTIMMTEN ZWECK. Details finden Sie in der GNU General Public License.      *
*                                                                              *
* Sie sollten eine Kopie der GNU General Public License zusammen mit diesem    *
* Programm erhalten haben.                                                     *
* Falls nicht, schreiben Sie an die Free Software Foundation,                  *
* Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.                *
\******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef __USE_CMSIS
#include "LPC17xx.h"
#endif

#ifdef __USE_DRIVER
#include "lpc_types.h"
#include "lpc17xx_pinsel.h"
#endif

#include "global_inc.h"
#include "usart.h"
#include "usb.h"
#include "ohci.h"
#include "utils/delay.h"

USBDevice_t DeviceListHead, *DeviceListTail;
uint8_t		Addresses[4] = {0UL, 0UL,0UL, 0UL}, lastAddress = 1;

volatile  uint32_t	RhScIntr = 0;										//< Root Hub Status Change interrupt
volatile  uint32_t	WdhIntr  = 0;										//< Semaphore to wait until the TD is submitted

volatile  HCRH		*HcRh		= (volatile  HCRH    *)MEM_HCRH;		//< RootHub Registers
volatile  HCCTRL	*HcCtrl		= (volatile  HCCTRL  *)MEM_HCCTRL;		//< Host Controller Control Register
volatile  HCCMDST	*HcCmdSt	= (volatile  HCCMDST *)MEM_HCCMDST;	//< Host Controller Command Status Register
volatile  HCCA		*Hcca		= (volatile  HCCA    *)MEM_HCCA;		//< Host Controller Communications Area structure
volatile  HCGTD	*TDHead		= (volatile  HCGTD   *)MEM_TDHEAD;		//< Head transfer descriptor structure
volatile  HCGTD	*TDTail		= (volatile  HCGTD   *)MEM_TDTAIL;		//< Tail transfer descriptor structure
volatile  HCED		*EDCtrl		= (volatile  HCED    *)MEM_EDCTRL;		//< Control endpoint descriptor structure
volatile  HCED		*EDBulkIn	= (volatile  HCED    *)MEM_EDBULKIN;	//< BulkIn endpoint descriptor structure
volatile  HCED		*EDBulkOut	= (volatile  HCED    *)MEM_EDBULKOUT;	//< BulkOut endpoint descriptor structure
volatile  uint8_t	*TDBuffer	= (volatile  uint8_t *)MEM_TDBUFFER;	//< Current Buffer Pointer of transfer descriptor

void init_Addresses() {
	uint8_t i;
	for (i = 0;i < 4;i++) {
		Addresses[i] = 0UL;
	}
}

void initStruct_HCCA (volatile HCCA *hcca) {
    uint32_t  i;

    for (i = 0; i < 32; i++) {
        hcca->IntTable[i] = 0;
    }
	hcca->FrameNumber = 0;
	hcca->DoneHead    = 0;
}

void initStruct_HCED (volatile HCED *ed) {
	// Die hcEd-Struktur enthält weitere zusammengesetzte Datentypen.
	// Diese Datentypen sind immer 32-Bit breit und koennen daher auf uint32_t gecastet werden.
	// So lässt sich das komplette struct initilisieren.
    *((uint32_t*)&(ed->Control))	= 0;
    *((uint32_t*)&(ed->TailTD))		= 0;
    *((uint32_t*)&(ed->HeadTD))		= 0;
    *((uint32_t*)&(ed->NextTD))		= 0;
}

void initStruct_HCTD (volatile HCGTD *td) {
	// Die hcGtd-Struktur enthält weitere zusammengesetzte Datentypen.
	// Diese Datentypen sind immer 32-Bit breit und koennen daher auf uint32_t gecastet werden.
	// So lässt sich das komplette struct initilisieren.
    *((uint32_t*)&(td->Control))	= 0;
    *((uint32_t*)&(td->CurrBufPtr))	= 0;
    *((uint32_t*)&(td->NextTD))		= 0;
    *((uint32_t*)&(td->BuffEnd))	= 0;
}

void  init_USBHost (void) {
	PINSEL_CFG_Type pinCfg;

	// USB-Peripherie einschalten
	LPC_SC->PCONP       |= (1<<PCUSB);
	// Host-Clock, OTG-Clock und AHB-Clock einschalten
	LPC_USB->USBClkCtrl	 = (1<<HOST_CLK) | (1<<OTG_CLK) | (1<<AHB_CLK);

	// warten bis Host-/AHB-Clock stabil sind
	while (!(LPC_USB->USBClkSt & ((1<<HOST_CLK) | (1<<OTG_CLK) | (1<<AHB_CLK))));

	// USB-Controller als OHCI-Host verwenden
	LPC_USB->OTGStCtrl = 0x3;

	// Ausgänge für USB-Host konfigurieren
	pinCfg.OpenDrain = PINSEL_PINMODE_NORMAL;
	pinCfg.Pinmode   = PINSEL_PINMODE_PULLUP;

    pinCfg.Portnum   = 0;
    pinCfg.Pinnum    = 29;
    pinCfg.Funcnum   = USB_DP;
    PINSEL_ConfigPin(&pinCfg);

    pinCfg.Portnum   = 0;
    pinCfg.Pinnum    = 30;
    pinCfg.Funcnum   = USB_DM;
    PINSEL_ConfigPin(&pinCfg);

    // Die Ausgänge für USB_UP_LED, USB_PPWR, USB_PWRD und USB_OVRCR sind am LPCxpresso Base Board nicht verbunden.
#if (defined(BOARD) && (BOARD != LPCXPRESSO))
	pinCfg.Portnum   = 1;
    pinCfg.Pinnum    = 18;
    pinCfg.Funcnum   = USB_UP_LED;
    PINSEL_ConfigPin(&pinCfg);

	pinCfg.Portnum   = 1;
    pinCfg.Pinnum    = 19;
    pinCfg.Funcnum   = USB_PPWR;
    PINSEL_ConfigPin(&pinCfg);

    pinCfg.Portnum   = 1;
    pinCfg.Pinnum    = 22;
    pinCfg.Funcnum   = USB_PWRD;
    PINSEL_ConfigPin(&pinCfg);

    pinCfg.Portnum   = 1;
    pinCfg.Pinnum    = 27;
    pinCfg.Funcnum   = USB_OVRCR;
    PINSEL_ConfigPin(&pinCfg);
#endif

	printf("Initialisiere Host Stack\r\n");
	printf("Speicheradresse HCCA:       %lx\r\n", Hcca);
	printf("Speicheradresse TDHead:     %lx\r\n", TDHead);
	printf("Speicheradresse TDTail:     %lx\r\n", TDTail);
	printf("Speicheradresse EDCtrl:     %lx\r\n", EDCtrl);
	printf("Speicheradresse EDBulkIn:   %lx\r\n", EDBulkIn);
	printf("Speicheradresse EDBulkOut:  %lx\r\n", EDBulkOut);
	printf("Speicheradresse TDBuffer:   %lx\r\n", TDBuffer);
	printf("\r\n");

	// TD, ED und HCCA initialisieren
	initStruct_HCED(EDCtrl);
    initStruct_HCED(EDBulkIn);
    initStruct_HCED(EDBulkOut);
    initStruct_HCTD(TDHead);
    initStruct_HCTD(TDTail);
    initStruct_HCCA(Hcca);

    // Host resetten
    // 50ms vor dem Reset warten
    _delay_ms(50.0f);
    // HARDWARE RESET
    LPC_USB->HcControl			= 0;
    // Controllist-Head auf 0 setzen
    LPC_USB->HcControlHeadED	= 0;
    // Bulklist-Head auf 0 setzen
    LPC_USB->HcBulkHeadED		= 0;
    // SOFTWARE RESET
    HcCmdSt->HCR				= 1;
    // Warten bis der Reset durchgefuehrt wurde.
    while(HcCmdSt->HCR) { }
    // Frame Interval setzen (12000 Bits pro Frame)
    LPC_USB->HcFmInterval		= ((((6 * (19999UL - 210)) / 7) << 16) | 19999UL);
    
    // Host starten
    HcCtrl->HCFS				= HCFS_USBOPERATIONAL;
    // Spannungsversorgung fuer USB auf dem BaseBoard nicht schaltbar
    HcRh->DescriptorA.NPS		= 1;
    // Keine Stromüberwachung fuer USB auf dem BaseBoard moeglich.
    HcRh->DescriptorA.NOCP		= 1;
    // Spannungsversorgung aller Ports einschalten
    // Wird dass benoetigt wenn die Spannungsversorgung nicht schaltbar ist?
    HcRh->Status.LPSC			= 1;
    // Speicherbereich für Host Controller Communication Area
    LPC_USB->HcHCCA				= (uint32_t)Hcca;
    // Interrupt-Flags lösche
    LPC_USB->HcInterruptStatus |= LPC_USB->HcInterruptStatus;
    // Enable WritebackDoneHead, RootHubChanged and MasterInterrupt
    LPC_USB->HcInterruptEnable  = HcInterrupt_MIE | HcInterrupt_WDH | HcInterrupt_RHSC;

    // ARM USB-Interrupts aktivieren, höchste Priorität
	NVIC_EnableIRQ(USB_IRQn);
	NVIC_SetPriority(USB_IRQn, 0);

	// Keine USB-Adressen vergeben
	init_Addresses();

	// verkettete Liste zurück setzen
	DeviceListTail				= &DeviceListHead;
	DeviceListTail->next		= NULL;

	// Ausgabe der Host-Eogenschaften.
	printf("Host-Details\r\n");
	printf("OHCI Revision: %x\r\n", (uint8_t)LPC_USB->HcRevision);
	printf("Number of Ports: %d\r\n", HcRh->DescriptorA.NDP);
	printf("Host Initialized\r\n");
}

int32_t processDoneQueue() {
	// Transport Desriptoren werden an 16-Byte-Grenzen ausgerichtet
	// Das LSb signalisiert das eine weitere Liste im Host Controller angefangen wurde.
	HCGTD *TD = (HCGTD *)(Hcca->DoneHead & 0xFFFFFFF0);
    // WDH-Interrupt-Flag löschen und dem HC signalisieren das die Done Queue abgearbeitet wurde.
	LPC_USB->HcInterruptStatus = HcInterrupt_WDH;
	// WDH-Interrupt wieder aktivieren
	LPC_USB->HcInterruptEnable = HcInterrupt_WDH;
	// Completion Codes des TDs zurueck geben
	return TD->Control.CC;
}

int32_t enumerateDevice (void) {
    int32_t  rc;
    uint8_t  c,i,e,offset,address;
    uint16_t descIndex, descLength, configNum;

    if (!HcRh->PortStatus[0].CCS) {
    	printf("Device Disconnected!\r\n");
        RhScIntr = 0;
        lastAddress = 0;
    	return OK;
    }

    // mindestens 50ms vor dem Port-Reset warten (USB 2.0 spec)
    // PowerOnToPowerGoodTime aus RhDescriptorA ?
    // In der LPCUSBlib (Host_LPC.c Zeile 81-85) ist noch folgende Sequenz zu finden.
	//   USB_Host_VBUS_Manual_Off();
	//   USB_OTGPAD_On();
	//   USB_Host_VBUS_Auto_Enable();
	//   USB_Host_VBUS_Auto_On();
	_delay_ms(100.0f);

    // Port 1 reseten
	HcRh->PortStatus[0].PRS = 1;
    // warten bis reset durchgefuehrt wurde
	while (HcRh->PortStatus[0].PRS) { }
    // Reset-Flag zuruecksetzen
	HcRh->PortStatus[0].PRSC = 1;
    // 100ms nach dem Reset warten
	// Wird der Status HOST_STATE_Powered_WaitForDeviceSettle aus der LPCUSBlib (Host_LPC.c sein
	// Der Wert fuer HOST_DEVICE_SETTLE_DELAY_MS ist nicht zu finden, evtl. PowerOnToPowerGoodTime aus RhDescriptorA ?
	_delay_ms(200.0f);

    // Device-Adresse 0, Endpoint 0
	EDCtrl->Control.FA = 0;
	EDCtrl->Control.EN = 0;
    // maximale Paketlänge für Endpunkt 0 = 8 (64 bei HighSpeed ?)
	EDCtrl->Control.MPS = 8;

    // Die ersten 8 Bytes des Device-Descriptors lesen
	descIndex = 0; descLength = 8;
	rc = ControlTransfer(DEVICE_TO_HOST, REQUEST_TYPE_STANDARD, RECIPIENT_DEVICE, GET_DESCRIPTOR, (DEVICE_DESCRIPTOR << 8) | (descIndex), 0, descLength, TDBuffer);
    if (rc != OK) {
        printf("ERROR: In %s at Line %u - rc = %li\r\n", __FUNCTION__, __LINE__, rc);
        return (rc);
    }

    // Port 1 reseten
	HcRh->PortStatus[0].PRS = 1;
    // warten bis reset durchgeführt wurde
	while (HcRh->PortStatus[0].PRS) { }
    // Reset-Flag zuruecksetzen
	HcRh->PortStatus[0].PRSC = 1;
    // 100ms nach dem Reset warten
	// Quelle ?
	_delay_ms(200.0f);

	// maximale Paketlänge für Endpunkt 0 setzen, Wert aus Device-Descriptor
    EDCtrl->Control.MPS = TDBuffer[7];

    // Dem Device seine neue Adresse bekannt geben.
    address = assignAddress();
    rc = ControlTransfer(HOST_TO_DEVICE, REQUEST_TYPE_STANDARD, RECIPIENT_DEVICE, SET_ADDRESS, address, 0, 0, NULL);
    if (rc != OK) {
        printf("ERROR: In %s at Line %u - address = %i, rc = %li\r\n", __FUNCTION__, __LINE__, address, rc);
        releaseAddress(address);
        return (rc);
    }
    _delay_ms(2.0f);

    // Der ControlPipe die neue Adresse geben.
    EDCtrl->Control.FA = address;
    DeviceListTail->Address = address;

    // Kompletten Device-Descriptor einlesen.
	descIndex = 0; descLength = 18;
	rc = ControlTransfer(DEVICE_TO_HOST, REQUEST_TYPE_STANDARD, RECIPIENT_DEVICE, GET_DESCRIPTOR, (DEVICE_DESCRIPTOR << 8) | (descIndex), 0, descLength, TDBuffer);
    if (rc != OK) {
        printf("ERROR: In %s at Line %u - rc = %li\r\n", __FUNCTION__, __LINE__, rc);
        releaseAddress(address);
        return (rc);
    }
    memcpy(&(DeviceListTail->Descriptor), (const void *)TDBuffer, 18);

	// Alle Configuration-Descriptoren einlesen.
    DeviceListTail->ConfigureList = (USBConfig_t *)malloc(sizeof(USBConfig_t) * DeviceListTail->Descriptor.bNumConfigurations);
    for (c = 0; c < DeviceListTail->Descriptor.bNumConfigurations; c++) {
    	// Die ersten 9 Bytes der des Configuration-Descriptors einlesen um die Laenge des gesmaten Descriptors zu erhalten.
    	descIndex = c; descLength = 9;
    	rc = ControlTransfer(DEVICE_TO_HOST, REQUEST_TYPE_STANDARD, RECIPIENT_DEVICE, GET_DESCRIPTOR, (CONFIGURATION_DESCRIPTOR << 8) | (descIndex), 0, descLength, TDBuffer);
		if (rc != OK) {
	        printf("ERROR: In %s at Line %u - c = %i rc = %li\r\n", __FUNCTION__, __LINE__, c, rc);
	        releaseAddress(address);
			return (rc);
		}
        memcpy(&(DeviceListTail->ConfigureList[c].Descriptor), (const void *)TDBuffer, 9);

        // Alle Interface und Endpoint-Descritoren einlesen.
    	descIndex = c; descLength = DeviceListTail->ConfigureList[c].Descriptor.wTotalLength;
    	rc = ControlTransfer(DEVICE_TO_HOST, REQUEST_TYPE_STANDARD, RECIPIENT_DEVICE, GET_DESCRIPTOR, (CONFIGURATION_DESCRIPTOR << 8) | (descIndex), 0, descLength, TDBuffer);
		if (rc != OK) {
	        printf("ERROR: In %s at Line %u - rc = %li\r\n", __FUNCTION__, __LINE__, rc);
	        releaseAddress(address);
			return (rc);
		}
		DeviceListTail->ConfigureList[c].InterfaceList = (USBInterface_t *)malloc(sizeof(USBInterface_t) * DeviceListTail->ConfigureList[c].Descriptor.bNumInterfaces);
        for(i = 0, offset = 9; i < DeviceListTail->ConfigureList[c].Descriptor.bNumInterfaces; i++) {
			memcpy(&(DeviceListTail->ConfigureList[c].InterfaceList[i].Descriptor), (const void *)&(TDBuffer[offset]), 9);
			offset += 9;
			DeviceListTail->ConfigureList[c].InterfaceList[i].EndpointList = (struct EndpointDescr *)malloc(sizeof(struct EndpointDescr) * DeviceListHead.ConfigureList[c].InterfaceList[i].Descriptor.bNumEndpoints);
			for(e = 0; e < DeviceListTail->ConfigureList[c].InterfaceList[i].Descriptor.bNumEndpoints; e++) {
				memcpy(&(DeviceListTail->ConfigureList[c].InterfaceList[i].EndpointList[e]), (const void *)&(TDBuffer[offset]), 7);
				offset += 7;
			}
        }
    }

    // Devicekonfiguration auswählen.
    configNum = selectConfig(DeviceListTail->ConfigureList, DeviceListTail->Descriptor.bNumConfigurations);
    ControlTransfer(HOST_TO_DEVICE, REQUEST_TYPE_STANDARD, RECIPIENT_DEVICE, SET_CONFIGURATION, configNum, 0, 0, NULL);
    if (rc != OK) {
        releaseAddress(address);
        printf("ERROR: In %s at Line %u - rc = %li\r\n", __FUNCTION__, __LINE__, rc);
        return rc;
    }

    // Ein paar Devices benötigen noch ein kleines delay
    // Quelle ?
    _delay_ms(100.0f);

    DeviceListTail->next = (USBDevice_t *)malloc(sizeof(USBDevice_t));
    if (DeviceListTail->next != NULL) {
		DeviceListTail = DeviceListTail->next;
		DeviceListTail->next = NULL;
    }

    lastAddress = address;
    RhScIntr = 0;
    return (rc);
}

uint8_t assignAddress() {
	uint8_t a;

	for(a=1; a < 128;a++) {
		if ((Addresses[a/32] & (1<<(a%32))) == 0) {
			Addresses[a/32] |= (1<<(a%32));
			return a;
		}
	}

	return 0;
}

void releaseAddress(uint8_t a) {
	Addresses[a/32] &= ~(1<<(a%32));
}

int16_t selectConfig(USBConfig_t *configlist, uint8_t numConfigs) {
	return 1;
}

int32_t ControlTransfer (uint8_t direction, uint8_t type, uint8_t recipient, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, uint16_t wLength, volatile uint8_t *buffer) {
    int32_t  rc;

    // Setup-Packet zusammenstellen.
	((SETUP *)TDBuffer)->rescipient = recipient;
	((SETUP *)TDBuffer)->type       = type;
	((SETUP *)TDBuffer)->direction  = direction;
	((SETUP *)TDBuffer)->request    = bRequest;
	((SETUP *)TDBuffer)->value      = wValue;
	((SETUP *)TDBuffer)->index      = wIndex;
	((SETUP *)TDBuffer)->length     = wLength;
//	printf("%2x %2x %2x %2x %2x %2x %2x %2x\r\n", TDBuffer[0], TDBuffer[1], TDBuffer[2], TDBuffer[3], TDBuffer[4], TDBuffer[5], TDBuffer[6], TDBuffer[7]);

    // Setup-Packet an den Transport-Deskriptor haengen
    initStruct_HCTD(TDHead);
    initStruct_HCTD(TDTail);
	// Setup-Pakete haben ein feste Laenge von 8 Bytes
    ((HCGTD *)TDHead)->BuffEnd		= (uint32_t) (TDBuffer + (8 - 1));
    ((HCGTD *)TDHead)->Control.R	= 1;
    ((HCGTD *)TDHead)->Control.DP	= TD_SETUP;
    ((HCGTD *)TDHead)->Control.DI	= 0;
    ((HCGTD *)TDHead)->Control.T	= TD_TOGGLE_0;
    ((HCGTD *)TDHead)->Control.CC	= 0xF;
    ((HCGTD *)TDHead)->CurrBufPtr	= (uint32_t) TDBuffer;
    ((HCGTD *)TDHead)->NextTD		= (uint32_t) TDTail;

	// Transport-Descriptor an denm Endpoint-Descriptor haengen
    // initStruct_HCED(EDCtrl);		// EDCtrl wurde bereits mit Adresse des Devices/Endpoint beschrieben
	((HCED *) EDCtrl)->TailTD		= (uint32_t) TDTail;
	((HCED *) EDCtrl)->HeadTD.C		= 1;
	((HCED *) EDCtrl)->HeadTD.H		= 0;
	((HCED *) EDCtrl)->HeadTD.res0	= 0;
	((HCED *) EDCtrl)->HeadTD.HeadP	= (uint32_t) TDHead >> 4;
	((HCED *) EDCtrl)->NextTD		= 0;

	// Endpoint-Descriptor an den Host-Controller uebergeben
	LPC_USB->HcControlHeadED		= (uint32_t) EDCtrl;
	// Host-Controller signalisieren, dass die Control-Liste nicht leer ist.
	HcCmdSt->CLF					= 1;
	// Host-Controller soll die Control-Liste verarbeiten
	HcCtrl->CLE						= 1;

	// Warten bis der Host-Controller das Paket abgearbeitet hat.
	while (!WdhIntr) { asm("nop"); }
	WdhIntr = 0;
	// Antwort vom Host-Controller auswerten
	rc = processDoneQueue();

    if (rc != OK) {
    	printf("ERROR: In %s at Line %u - rc = %li\r\n", __FUNCTION__, __LINE__, rc);
    	return rc;
    }

    // Datewn emfangen/senden
    if (wLength) {
        initStruct_HCTD(TDHead);
        initStruct_HCTD(TDTail);
        ((HCGTD *)TDHead)->BuffEnd		= (uint32_t) (TDBuffer + (wLength - 1));
        ((HCGTD *)TDHead)->Control.R	= 1;
        ((HCGTD *)TDHead)->Control.DP	= (direction == HOST_TO_DEVICE) ? TD_OUT : TD_IN;
        ((HCGTD *)TDHead)->Control.DI	= 0;
        ((HCGTD *)TDHead)->Control.T	= TD_TOGGLE_1;
        ((HCGTD *)TDHead)->Control.CC	= 0xF;
        ((HCGTD *)TDHead)->CurrBufPtr	= (uint32_t) TDBuffer;
        ((HCGTD *)TDHead)->NextTD		= (uint32_t) TDTail;

    	// Transport-Descriptor an denm Endpoint-Descriptor haengen
        // initStruct_HCED(EDCtrl);		// EDCtrl wurde bereits mit Adresse des Devices/Endpoint beschrieben
    	((HCED *) EDCtrl)->TailTD		= (uint32_t) TDTail;
    	((HCED *) EDCtrl)->HeadTD.C		= 1;
    	((HCED *) EDCtrl)->HeadTD.H		= 0;
    	((HCED *) EDCtrl)->HeadTD.res0	= 0;
    	((HCED *) EDCtrl)->HeadTD.HeadP	= (uint32_t) TDHead >> 4;
    	((HCED *) EDCtrl)->NextTD		= 0;

    	// Endpoint-Descriptor an den Host-Controller uebergeben
    	LPC_USB->HcControlHeadED		= (uint32_t) EDCtrl;
    	// Host-Controller signalisieren, dass die Control-Liste nicht leer ist.
    	HcCmdSt->CLF					= 1;
    	// Host-Controller soll die Control-Liste verarbeiten
    	HcCtrl->CLE						= 1;

    	// Warten bis der Host-Controller das Paket abgearbeitet hat.
    	while (!WdhIntr) { asm("nop"); }
    	WdhIntr = 0;
    	// Antwort vom Host-Controller auswerten
    	rc = processDoneQueue();
	}

    if (rc != OK) {
    	printf("ERROR: In %s at Line %u - rc = %li\r\n", __FUNCTION__, __LINE__, rc);
    	return rc;
    }

    initStruct_HCTD(TDHead);
    initStruct_HCTD(TDTail);
    ((HCGTD *)TDHead)->BuffEnd		= (uint32_t) NULL;
    ((HCGTD *)TDHead)->Control.R	= 1;
    ((HCGTD *)TDHead)->Control.DP	= (direction == HOST_TO_DEVICE) ? TD_IN : TD_OUT;
    ((HCGTD *)TDHead)->Control.DI	= 0;
    ((HCGTD *)TDHead)->Control.T	= TD_TOGGLE_1;
    ((HCGTD *)TDHead)->Control.CC	= 0xF;
    ((HCGTD *)TDHead)->CurrBufPtr	= (uint32_t) NULL;
    ((HCGTD *)TDHead)->NextTD		= (uint32_t) TDTail;

	// Transport-Descriptor an denm Endpoint-Descriptor haengen
    // initStruct_HCED(EDCtrl);		// EDCtrl wurde bereits mit Adresse des Devices/Endpoint beschrieben
	((HCED *) EDCtrl)->TailTD		= (uint32_t) TDTail;
	((HCED *) EDCtrl)->HeadTD.C		= 1;
	((HCED *) EDCtrl)->HeadTD.H		= 0;
	((HCED *) EDCtrl)->HeadTD.res0	= 0;
	((HCED *) EDCtrl)->HeadTD.HeadP	= (uint32_t) TDHead >> 4;
	((HCED *) EDCtrl)->NextTD		= 0;

	// Endpoint-Descriptor an den Host-Controller uebergeben
	LPC_USB->HcControlHeadED		= (uint32_t) EDCtrl;
	// Host-Controller signalisieren, dass die Control-Liste nicht leer ist.
	HcCmdSt->CLF					= 1;
	// Host-Controller soll die Control-Liste verarbeiten
	HcCtrl->CLE						= 1;

	// Warten bis der Host-Controller das Paket abgearbeitet hat.
	while (!WdhIntr) { asm("nop"); }
	WdhIntr = 0;
	// Antwort vom Host-Controller auswerten
	rc = processDoneQueue();

	return rc;
}

void USB_IRQHandler (void) {
	uint32_t int_status;

    // aktive Interrupts zwischenspeichern
    int_status  = LPC_USB->HcInterruptStatus;

	// RootHubStatusChange Interrupt
	if (int_status & HcInterrupt_RHSC) {
//		printf("%lx - ", *((uint32_t *)&HcRh->PortStatus[0]));
		// TODO: Bei Prozessoren mit mehr als einem Port die anderen Ports ueberpruefen
		// ConnectStatusChange
		// Device (Dis)Connected, SetPortReset, SetPortEnable, SetPortSuspend
		if (HcRh->PortStatus[0].CSC) {
			// CurrentConnectStatus
			if (HcRh->PortStatus[0].CCS) {
				// Neues Device
				RhScIntr = 1;
			} else {
				// Device entfernt
				RhScIntr = 1;
			}
			// Flag ConnectStatusChange zurueck setzen
			HcRh->PortStatus[0].CSC = 1;
		}
//		printf("%lx\r\n", *((uint32_t *)&HcRh->PortStatus[0]));
	}

	// WritebackDoneHead Interrupt
	if (int_status & HcInterrupt_WDH) {
		// WDH-Status nicht löschen, um das auslesen der DoneQeue zu ermöglichen.
		int_status &= ~HcInterrupt_WDH;
		// WDH-Interrupt ausschalten sonst gibt es weitere Interrupts.
		LPC_USB->HcInterruptDisable = HcInterrupt_WDH;
		WdhIntr = 1;
	}

	//InterruptStatus-Register zurueck setzen
	LPC_USB->HcInterruptStatus = int_status;
    return;
}
