/*************************************************************************

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

*************************************************************************/

#include <avr/io.h>
#include <util/delay.h>
#include <stdio.h>
#include <avr/interrupt.h>
#include "i2c.h"
#include <util/atomic.h>



/* Position of the LED on MMR-70 */
#define LED_PIN	 PD7
#define LED_DDR  DDRD
#define LED_PORT PORTD

//                        "----------------------------64----------------------------------"
char radiotext_text[64] = "Radiotext ist cool! * * *                                       ";

/* intialization data for register 0x00:0x16 */
uint8_t init_data[] =
{
	0x02, 0x83, 0x0A, 0x00,
	0x00, 0x00, 0x00, 0x7E,
	0x0E, 0x08, 0x3F, 0x2A,
	0x0C, 0xE6, 0x3F, 0x70,
	0x0A, 0xE4, 0x00, 0x42,
	0xC0, 0x41, 0xF4
};



/* 4 RDS 0A Groups containing Program service name */
uint8_t station_frames[20][3] =
{
	{0x03, 0xE0, 0x00},
	{0x05, 0x01, 0x24},
	{0x05, 0xE0, 0xCD},
	{0x05, ' ' , ' ' },	// 1  2

	{0x03, 0xE0, 0x00},
	{0x05, 0x01, 0x21},
	{0x05, 0xE0, 0xCD},
	{0x05, ' ' , ' ' }, // 3  4

	{0x03, 0xE0, 0x00},
	{0x05, 0x01, 0x22},
	{0x05, 0xE0, 0xCD},
	{0x05, ' ' , ' ' }, // 5  6

	{0x03, 0xE0, 0x00},
	{0x05, 0x01, 0x27},
	{0x05, 0xE0, 0xCD},
	{0x05, ' ' , ' ' }, // 7  8

	// Radiotext
/* 1 RDS 2A Group containing Radiotext chars */
	{0x03, 0xE0, 0x00},
	{0x05, 0x21, 0x20},
	{0x05, ' ' , ' ' },	// 1  2
	{0x05, ' ' , ' ' },	// 3  4

};



/************************************************************************ /
 * Inserts text in Radiotext string
 *************************************************************************/
void ns741_rds_set_radiotext(char *text)
{
	uint8_t i = 0, isend = 0;
	for (i=0; i<64; i++)
	{
		if (text[i]==0)
			isend = 1;
		
		if (!isend)
			radiotext_text[i] = text[i];
		else
			radiotext_text[i] = ' ';
	}
	
	//strcpy( radiotext_text, text );
}


/************************************************************************ /
 * Inserts text in RDS frames
 *************************************************************************/
void ns741_rds_set_progname(char *text)
{
	station_frames[3][1] = text[0];
	station_frames[3][2] = text[1];
	station_frames[7][1] = text[2];
	station_frames[7][2] = text[3];
	station_frames[11][1]= text[4];
	station_frames[11][2]= text[5];
	station_frames[15][1]= text[6];
	station_frames[15][2]= text[7];
}



/************************************************************************ /
 * Used to write a register of NS741
 *************************************************************************/
uint8_t TWI_send(uint8_t addr, uint8_t data)
{
	ATOMIC_BLOCK(ATOMIC_FORCEON)
	{
		i2c_start((0x66<<1) | I2C_WRITE);
		i2c_write(addr);
		i2c_write(data);
		i2c_stop();
	}
	return 0;
}



/************************************************************************ /
 * Used to write 2 Registers of NS741
 * Do not call it outside INT0 because it is not atomic
 *************************************************************************/
uint8_t TWI_send_word(uint8_t addr, uint8_t data0, uint8_t data1)
{
	i2c_start((0x66<<1) | I2C_WRITE);
	i2c_write(addr);
	i2c_write(data0);
	i2c_write(data1);
	i2c_stop();
	printf("~");
	return 0;
}



/************************************************************************ /
 * Sets FM-Transmitter frequency
 * Must be given in kHz
 *
 * Thanks to Carsten Sch. (dg3ycs) for reverse engineering
 *************************************************************************/
void ns741_set_frequency(uint32_t f_khz)
{
	/* calculate frequency in 8.192kHz steps*/
	uint16_t val = (uint16_t)((uint32_t)f_khz*1000ULL/8192ULL);
	TWI_send(0x0A,		val);
	TWI_send(0x0B, val >> 8);
	printf("\nSet frequency to  %lu(%lu) kHz \n", f_khz, val*8192UL/1000UL);
}



/************************************************************************ /
 * Write initial Data to NS741
 *
 * Thanks to Silvan König for init-sequence
 *************************************************************************/
void ns741_init(void)
{
	uint8_t i;

	/* init LED */
	LED_DDR |= (1<<LED_PIN);

	/* init i2c */
	i2c_init();

	/* sync i2c bus */
	TWI_send(0,0);

	/* write init_data[] */
	i2c_start((0x66<<1)| I2C_WRITE);	  // Start writing from
	i2c_write(0);						  // Register Address 0x00
	for (i=0; i<sizeof(init_data); i++)   // Write whole array
	{
		i2c_write(init_data[i]);
	}
	i2c_stop();							  // Stop Transmission


	// start
	TWI_send(0x02, 0x0b);
	TWI_send(0x15, 0x11);
	TWI_send(0x10, 0xE0);


	// Configuration
	TWI_send(0x02, 0x0B);
	TWI_send(0x07, 0x7E);
	TWI_send(0x08, 0x0E);
	TWI_send(0x02, 0xCA);	//0x0A Sendeleistung
	TWI_send(0x01, 0x81);
}



/************************************************************************ /
 *  Starts RDS Transmission
 *************************************************************************/
void ns741_rds_start(void)
{
	MCUCR |= (1<<ISC01);  // falling edge INT0
	// GIFR  &= (1<<INTF0);  // Clear flag to prevent too early interrupt
	GIMSK |= (1<<INT0);   // activate INT0

	_delay_ms(10);		  // without this delay RDS hangs up

	/* Send first RDS Block to trigger RDINT */
	TWI_send_word(station_frames[0][0],
			      station_frames[0][1],
			      station_frames[0][2]
			     );


}



/************************************************************************ /
 *  Stops RDS Transmission
 *************************************************************************/
void ns741_rds_stop(void)
{
	 GIMSK &= ~(1<<INT0);       // INT0 deactivate
}



/************************************************************************ /
 *  Switches FM-Transmission On / Off
 *
 *  Thanks to Silvan König for reverse engineering
 *************************************************************************/
void ns741_power(uint8_t on)
{
	if (on == 1 )
		TWI_send(0x00, 0x03);	// module active
	else
		TWI_send(0x00, 0x02);	// modul deactivate
}



/************************************************************************ /
 * RDINT_Interrupt
 * Triggered by FM-Module
 *************************************************************************/
ISR(INT0_vect)
{
	static uint8_t frame_counter = 0;
	static uint8_t radiotext_counter = 0;
	uint8_t i = 0;
	
	if (frame_counter == 0)
	{
		//Text segment address code
		station_frames[17][2] = 0x20 | radiotext_counter;
		i = radiotext_counter<<2; // *4
		station_frames[18][1] = radiotext_text[i];
		station_frames[18][2] = radiotext_text[i+1];
		station_frames[19][1] = radiotext_text[i+2];
		station_frames[19][2] = radiotext_text[i+3];
		
		radiotext_counter = (radiotext_counter + 1) % 16;
	}
	LED_PORT ^= (1<<LED_PIN); // Indicator - RDS runnnin
	printf("*");			  // ...

	/* Send next i2c-Frame */
	TWI_send_word(station_frames[frame_counter][0],
						  station_frames[frame_counter][1],
					      station_frames[frame_counter][2]
					      );
	/* only 4+1 RDS-groups a 4+1 i2c-frames*/
	frame_counter = (frame_counter + 1) % 20;
}
