
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <avr/eeprom.h>

#include <avr/pgmspace.h>
#include "usbdrv.h"
#include "oddebug.h"
#include "zacwire.h"

zacstate state = ZAC_READING_START_BIT_LOW;

/* ------------------------------------------------------------------------- */
/* ----------------------------- USB interface ----------------------------- */
/* ------------------------------------------------------------------------- */

PROGMEM const char usbHidReportDescriptor[22] = { /* USB report descriptor */
0x06, 0x00, 0xff, // USAGE_PAGE (Generic Desktop)
		0x09, 0x01, // USAGE (Vendor Usage 1)
		0xa1, 0x01, // COLLECTION (Application)
		0x15, 0x00, //   LOGICAL_MINIMUM (0)
		0x26, 0xff, 0x00, //   LOGICAL_MAXIMUM (255)
		0x75, 0x08, //   REPORT_SIZE (8)
		0x95, 0x80, //   REPORT_COUNT (128)
		0x09, 0x00, //   USAGE (Undefined)
		0xb2, 0x02, 0x01, //   FEATURE (Data,Var,Abs,Buf)
		0xc0 // END_COLLECTION
		};
/* Since we define only one feature report, we don't use report-IDs (which
 * would be the first byte of the report). The entire report consists of 128
 * opaque data bytes.
 */

/* The following variables store the status of the current data transfer */
static uchar currentAddress;
// remaining bytes of a bus transfer
static uchar bytesRemaining;

unsigned char buffer[129];

// half the value of tstrobe
volatile uint16_t strobe = 0;
// high byte of a zacwire paxcket
volatile uchar zac_high_byte = 0;
// low byte of a zacwire paxcket
volatile uchar zac_low_byte = 0;
// the bit of the byte currently being read including parity
volatile uchar zac_current_bit = 19;

volatile uchar edge = 0;

volatile uint16_t timer_start = 0;
volatile uint16_t timer_end = 0;

/**
 * Start timer1 in normal mode
 */
void timer1_start_normal();
/**
 * Stop timer1
 */
void timer1_stop();
/**
 * Stop timer1 from ctc mode
 */
void timer1_stop_ctc();
/**
 * Start timer1 in ctc mode and load the output compare register (OCR1A)
 */
void timer1_start_ctc(uint16_t cmp);
/**
 * Select falling edge for INT1 pin
 */
void int1_falling_edge(void);
/**
 * Select rising edge for INT1 pin
 */
void int1_rising_edge(void);
/**
 * Turn INT1 off
 */
void int1_enable(void);
/**
 * Turn INT1 on
 */
void int1_disable(void);

/* ------------------------------------------------------------------------- */

/* usbFunctionRead() is called when the host requests a chunk of data from
 * the device. For more information see the documentation in usbdrv/usbdrv.h.
 */uchar usbFunctionRead(uchar *data, uchar len) {
	if (len > bytesRemaining)
		len = bytesRemaining;

	// eeprom_read_block(data, (uchar *)0 + currentAddress, len);

	int i;

	// copy outgoing data from buffer to host
	for (i = 0; i < len; i++) {
		data[i] = buffer[i + currentAddress];
	}

	currentAddress += len;
	bytesRemaining -= len;
	return len;
}

/* usbFunctionWrite() is called when the host sends a chunk of data to the
 * device. For more information see the documentation in usbdrv/usbdrv.h.
 */uchar usbFunctionWrite(uchar *data, uchar len) {
	if (bytesRemaining == 0)
		return 1; /* end of transfer */
	if (len > bytesRemaining)
		len = bytesRemaining;

	// eeprom_write_block(data, (uchar *)0 + currentAddress, len);

	int i;

	// copy incoming data from host to device
	for (i = 0; i < len; i++) {
		buffer[i + currentAddress] = data[i];
	}

	currentAddress += len;
	bytesRemaining -= len;

	return bytesRemaining == 0; /* return 1 if this was the last chunk */
}

/* ------------------------------------------------------------------------- */

usbMsgLen_t usbFunctionSetup(uchar data[8]) {
	usbRequest_t *rq = (void *) data;

	if ((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS) { /* HID class request */
		if (rq->bRequest == USBRQ_HID_GET_REPORT) { /* wValue: ReportType (highbyte), ReportID (lowbyte) */
			/* since we have only one report type, we can ignore the report-ID */
			bytesRemaining = 128;
			currentAddress = 0;
			return USB_NO_MSG; /* use usbFunctionRead() to obtain data */
		} else if (rq->bRequest == USBRQ_HID_SET_REPORT) {
			/* since we have only one report type, we can ignore the report-ID */
			bytesRemaining = 128;
			currentAddress = 0;
			return USB_NO_MSG; /* use usbFunctionWrite() to receive data from host */
		}
	} else {
		/* ignore vendor type requests, we don't use any */
	}
	return 0;
}

ISR (TIMER1_COMPA_vect) {

     PORTC ^= ( 1 << PC3 );

	 // stop the counter and reset timer count
	 timer1_stop_ctc();
	 TCNT1 = 0;

	// reading the HIGH byte of a zacwire packet
	if (state == ZAC_READING_HIGH_BYTE) {
		// read bits ignoring the last bit (parity)
		// actually reading 9 bits in total
		if (PIND & (1<<PIND3)) {
			if (zac_current_bit > 1) {
				zac_high_byte |= (1 << (zac_current_bit - 1));
			}
		}
		// decrement current bit every trigger
		if (zac_current_bit > 0) {
			zac_current_bit--;
		}
		// have read all bits from HIGH byte, switch
		// to LOW byte
		else {
			state = ZAC_READING_LOW_BYTE;
			// now we have one bit more since there comes another stop bit
			zac_current_bit = 9;
		}

	}

	// after reading the parity bit of the first byte, we can omit
	// the stop bit, since the next falling edge does not
	// happen until the next start bit (LOW byte)

	// reading the LOW byte of a zacwire packet
	else if (state == ZAC_READING_LOW_BYTE) {
		// read bits ignoring the first bit (start bit) and the last bit (parity)
		// actually reading 10 bits in total
		if (PIND & (1<<PIND3)) {
			if (zac_current_bit > 1 && zac_current_bit < 9) {
				zac_low_byte |= (1 << (zac_current_bit - 2));
				PORTC |= ( 1 << PC3 );
			}
		}
		// decrement current bit every trigger
		if (zac_current_bit > 0) {
			zac_current_bit--;
		}
		// all bits read => back to the beginning
		else {
			zac_current_bit = 8;
			state = ZAC_READING_START_BIT_LOW;
		}
	}


}

ISR (INT1_vect) {

	/**
	 * Beginning of zacwire packet, first falling edge
	 * start measuring tstrobe time
	 */
	if (state == ZAC_READING_START_BIT_LOW) {

		/*
		 *  if there was on measurement cycle, store data
		 *  from zacwire in usb transfer buffer
		 */
		if (zac_high_byte > 0)
			buffer[0] = zac_high_byte;
		if (zac_low_byte > 0)
			buffer[1] = zac_low_byte;
		// reset measured values
		zac_high_byte = 0;
		zac_low_byte = 0;

		// start timer1 and use rising edge for int1
		// if the next rising edge happens this is the second half of the start bit
		timer1_start_normal();
		int1_rising_edge();

		state = ZAC_READING_START_BIT_HIGH;
		// toggle pin for debugging purposes
		// PORTC ^= ( 1 << PC3 );


	}
	else if (state == ZAC_READING_START_BIT_HIGH) {
		// rising edge of the start bit
		// at this stage the first rising edge of the start bit happened
		// we now have the strobe time acquired
		timer1_stop();
		strobe = TCNT1;
		// select falling edge, thus the startbit is complete
		// on next falling edge
		int1_falling_edge();
		// next stage
		state = ZAC_ACQUIRING_TSTROBE;
	}
	else if (state == ZAC_ACQUIRING_TSTROBE) {

		// falling edge of start bit occured
		// stop standard timer, reset timer count

		TCNT1 = 0;

		// start timer 1 in ctc mode, timer interrupt then occurs
		// each tstrobe/2 (50% duty cycle)

		timer1_start_ctc(strobe);

		// we now start reading the first byte
		state = ZAC_READING_HIGH_BYTE;

	}
	if (state == ZAC_READING_HIGH_BYTE || state == ZAC_READING_LOW_BYTE) {
		TCNT1 = 0;
		timer1_start_ctc(strobe);
	}

}

void timer1_start_normal() {
	TCCR1B = (0 << CS12) | (0 << CS11) | (1 << CS10); // no prescaler
	// TIMSK |= (1 << TOIE1); // timer overflow interrupt enable
}
void timer1_stop() {
	TCCR1B = (0 << CS12) | (0 << CS11) | (0 << CS10); // stop timer
	// TIMSK &= ~(1 << TOIE1); // stop overflow interrupt
	// TIFR |= (1 << TOV1); // clear timer overflow flag
}

void timer1_stop_ctc() {
	TIMSK &= ~(1 << OCIE1A); // stop ctc interrupt
	// TIFR |= (1 << TOV1); // clear timer overflow flag
}

void timer1_start_ctc(uint16_t cmp) {
	OCR1A = cmp; // set value to output compare register
	TCCR1B = (1 << WGM12) | (0 << CS12) | (0 << CS11) | (1 << CS10); // ctc, no prescaler
	//Enable the Output Compare A interrupt
	TIMSK |= (1 << OCIE1A);
}

void int1_falling_edge(void) {
	MCUCR &= ~(1 << ISC10);
	GIFR |= (1 << INTF1); // clear external interrupt flag
}

void int1_rising_edge(void) {
	MCUCR |= (1 << ISC10);
	GIFR |= (1 << INTF1); // clear external interrupt flag
}

void int1_enable(void) {
	GICR |= 1 << INT1;
}

void int1_disable(void) {
	GICR &= ~(1 << INT1);
}



int main(void) {

	uchar i;

	wdt_enable(WDTO_1S);
	/* Even if you don't use the watchdog, turn it off here. On newer devices,
	 * the status of the watchdog (on/off, period) is PRESERVED OVER RESET!
	 */
	/* RESET status: all port bits are inputs without pull-up.
	 * That's the way we need D+ and D-. Therefore we don't need any
	 * additional hardware initialization.
	 */
	odDebugInit(); DBG1(0x00, 0, 0); /* debug output: main starts */
	usbInit();
	usbDeviceDisconnect(); /* enforce re-enumeration, do this while interrupts are disabled! */
	i = 0;
	while (--i) { /* fake USB disconnect for > 250 ms */
		wdt_reset();
		_delay_ms(1);
	}
	usbDeviceConnect();

	DDRC = 0xFF;

	// enable INT0 and INT1
	GICR = 1 << INT0 | 1 << INT1;
	// VUSB defaults for INT0 and trigger on falling edge for INT1
	MCUCR =  (1 << ISC11) | (0 << ISC10) | (1 << ISC00) | (1 << ISC01);


	TCNT1 = 0;

	// turn TSIC on
	zac_on();
	_delay_us(125);

	// enable interrupts
	sei();

	DBG1(0x01, 0, 0); /* debug output: main loop starts */

	for (;;) { /* main event loop */
		DBG1(0x02, 0, 0); /* debug output: main loop iterates */
		wdt_reset();
		usbPoll();
	}

	return 0;
}
