/* LCD2_4BP.C: 
*
* Accessing a LC-Display which is directly connected to a Port (here P1)

* This sourcecode is the interface between the LCD-Driver and the I2C-Bus Module.
*
* The schematics for the display is included as PDF file. 
* Software assumes a 4x20 Display, but will work wit smaller sized too..
*
* (C) 2003 www.Wickenhaeuser.com
*
* Pin assignment:             
*  LCD HD44078A (4.bit mode)  			CPU
*  --------------------------------------------------------------------------
*     1  GND                                      GND                     
*     2  Vcc                                      Vcc                     
*     3  V0  (contrast, norm. ca. 0-2V)
*     4  RS  (register selection)                 P1.0                    
*     5  RW  (read/write)                         P1.1                    
*     6  EN  (chip enable)                        P1.2                    
*     7  DB0 (data bit 0)                         not connected           
*     8  DB1 (data bit 1)                         not connected           
*     9  DB2 (data bit 2)                         not connected           
*     10 DB3 (data bit 3)                         not connected           
*     11 DB4 (data bit 4)                         P1.4                    
*     12 DB5 (data bit 5)                         P1.5                    
*     13 DB6 (data bit 6)                         P1.6                    
*     14 DB7 (data bit 7)                         P1.7                    
*     15 BACKLIGHT (norm. +)			  
*     16 BACKLIGHT (norm. -)
*
*     optional LCD Vcc-Switch (low active) 	  P1.3
*/

#include <stdio.h>
#include <reg51.h>
#include "lcd_io.h"

#define LCD_PORT P1	// Used Port


/******************** lcd_port() *****************/
uchar lcd_port(uchar val) reentrant{	/* Send 4 bits */
	LCD_PORT = val | 4;	// E 1->0 Clock
	LCD_PORT = val;	
	return 0;	// OK
}

/******************** lcd_busy() *****************/
void lcd_busy(void) reentrant{
	uint i;
	uchar busy;
	LCD_PORT = 0xF2;
	for(i=0;i<2000;i++){	// about max 20 msec on a 20MHz machine...
		LCD_PORT = 0xF6;	// RW=1;
		busy=LCD_PORT&0x80;	// Read High-Nibble
		LCD_PORT = 0xF2;
		if(!busy) break;	
		LCD_PORT = 0xF6;	// Read Low-Nibble (Dummy)
		LCD_PORT = 0xF2;
	}
	LCD_PORT = 0xF0;
	return;		// Return even if there is NO Display after a certain time...
}

/******************** lcd_databyte(): Data-Byte (RS=1) *************/
void lcd_databyte(uchar date) reentrant {

	lcd_busy();
	lcd_port((date&0xF0)|1);	/* Highbyte, RS=1 */
	lcd_port(((date<<4))|1);	/* Lowbyte, RS=1 */


}

/******************** lcd_combyte(): Command-Byte (RS=0) *************/
void lcd_combyte(uchar date) reentrant {

	lcd_busy();
	lcd_port((date&0xF0));	/* Highbyte, RS=0 */
	lcd_port(((date<<4)));	/* Lowbyte, RS=0 */


}

/***************** lcd_initport(): Initialize Port, Optionally reset LCD and setz Access-Type **********/
uchar lcd_initport(void){

	/* Bit 3=HIGH disables LCD power on Wickenhaeuser boards in parallel 4 Bit Mode */
	lcd_port(255);
	_wait_ms(200);		/* Wait abt. 200 ms for RESET */
	lcd_port(247);
	_wait_ms(200);		/* Wait abt. 200 ms for POWER ON */

	lcd_port(0x30);
	_wait_ms(5);
	lcd_port(0x30);
	_wait_ms(1);
	lcd_port(0x30);
	_wait_ms(1);
	lcd_port(0x20);		/* Switch to 4 Bit... */
	_wait_ms(1);
	lcd_port(0x20);		/* Now: 4 Bit, Small Font, >=2 lines */
	_wait_ms(1);
	lcd_port(0x80);
	_wait_ms(1);
	return 0;
}



// EOF
