/* LCD_DRV.C */


/* The driver's layer */

#include <stdio.h>
#include <stdarg.h>
#include <reg51.h>

#include "lcd_io.h"
#include "lcd_drv.h"

// This definition is not to be seen by default, because it needs <stdarg.h>
// Declaring the byte-handler (iofunk()) as 'uint' is simply to prevent a compiler warning, 
// 'void' would be correct!
extern int _doprnt( void (*iofunk)(uchar) reentrant, far char* pfmt, va_list (fap));


/******************* internal Track XY cursor for NLs ***********************/
static signed char near lcd_pos_x, lcd_pos_y;



/************************ lcd_setxy **************************/
static void lcd_setxy(void) reentrant {	// local function!
	uchar date;
	if(lcd_pos_x>19){	// Ensure that all is inside the visible rectangle
		lcd_pos_x=-1;
		lcd_pos_y++;
	}
	if(lcd_pos_y>3){
		lcd_pos_y=0;
		lcd_pos_x=-1;
	}else if(lcd_pos_y<0){
		lcd_pos_y=3;
		lcd_pos_x=-1;
	}

	switch(lcd_pos_y){	// Here for a 4x40 Display
		case 1: date=64; break;
		case 2: date=20; break;
		case 3: date=84; break;
		default: date=0;
	}
	date|=128;
	if(lcd_pos_x>0) date+=lcd_pos_x;
	lcd_combyte(date);
}

/*********************** lcd_gotoxy() *************************/
void lcd_gotoxy(char nx,char ny) reentrant{
	lcd_pos_x=nx;
	lcd_pos_y=ny;
	lcd_setxy();
}


/***************************************************** 
* lcd_putc(): send datebyte
*****************************************************/
void lcd_putc(uchar date) reentrant{
	uchar i;
	if(date==8){			/* Backspace */
		if(lcd_pos_x>0) lcd_pos_x--;
		lcd_setxy();
	}else if(date!='\n'){			/* ASCII char */
		if(lcd_pos_x<0){
			for(i=0;i<20;i++){	/* Clear line */
				lcd_databyte(' ');
			}
			lcd_pos_x=0;
			lcd_setxy();
		}
		lcd_databyte(date);	/* Highbyte */
		lcd_pos_x++;
	}
	if(lcd_pos_x>19 || date=='\n'){ /* Newline */
		lcd_pos_x=-1;
		lcd_pos_y++;
		lcd_setxy();
	}
}


// Character-Graphics for 2-line Display thin lines
code unsigned char segment2[]={ 
         31,1,1,1,1,1,1,1,
         1,1,1,1,1,1,1,31,
         1,1,1,1,1,1,1,1,
         0,0,0,0,0,0,0,1,
         31,1,1,1,1,1,1,31,
         0,0,0,0,0,0,0,31,
         31,0,0,0,0,0,0,31,
	 1,0,0,0,0,0,0,0
	};

// List of segment-positions for 2-line Display
static code uchar dig2_tab[]={
	2,1,2,0,	// 0
	32,2,32,2,	// 1
	2,5,3,4,	// 2
	32,1,32,4,	// 3
	32,2,2,1,	// 4
	32,1,2,6,	// 5
	2,1,2,6,	// 6
	32,2,32,0,	// 7
	2,1,2,4,	// 8
	32,1,2,4,	// 9
	32,32,32,32,	// Space
	32,'-',32,32,	// -
	};



/******************** lcd_cgchars() ******************/
void lcd_cgchars(uchar *cdat){
	uchar i,c;
	lcd_combyte(64);		/* Set CG RAM addr. 0 */
	for(i=0;i<64;i++){
		c=*cdat++;
		lcd_databyte(c);	/* Write Bytes */
	}	
	lcd_setxy();	/* Restore old cursor position */
}

/******************** lcd_clear() ****************/
void lcd_clear(void){
	lcd_combyte(1);
	lcd_pos_x=0;
	lcd_pos_y=0;
}

/******************** lcd_cursor() ****************/
void lcd_cursor(uchar cur){
	lcd_combyte(cur+12);
}

/******************** lcd_init() *****************/
uchar lcd_init(void){		/* Return 1: ERROR, 0: OK */

	if(lcd_initport()) return 1; // Failed: NO DISPLAY?

	/* Now visible initialisation */
	lcd_combyte(15);	/* Display ON, Cursor on & blink */
	lcd_clear();
	lcd_cursor(LCD_CURSOR_BLINK);
	lcd_combyte(6);		/* Entry-Mode: Cursor Shift Right */
	lcd_cgchars(segment2); /* Required for LARGE 7-SEGMENT DIGITS */
	return 0;		/* OK */
}

/**************** print a large DIGIT 2 Lines. Must be in LINE >0! ************/
static void lcd_digit2(uchar dig) reentrant{
	uchar x,y;
	dig*=4;
	lcd_putc(dig2_tab[dig++]);
	x=lcd_pos_x;	// Take after 1.st char, might be after NL..
	y=lcd_pos_y;
	lcd_putc(dig2_tab[dig++]);
	lcd_gotoxy(x-1,y-1);
	lcd_putc(dig2_tab[dig++]);
	lcd_putc(dig2_tab[dig]);
	lcd_gotoxy(x+1,y);

}

/************** lcd_printf() ********************/
int lcd_printf(far char* pfmt, ...){
        va_list(ap);
        va_start(ap,pfmt);
        return _doprnt(lcd_putc, pfmt, ap);
}

/***************************************************** 
* lcd_d2_putc(): send datebyte. If digit: print large!
*****************************************************/
void lcd_d2_putc(uchar date) reentrant{
	if(date>='0' && date<='9'){
		lcd_digit2(date-'0');		
	}else if(date==' '){
		lcd_digit2(10);
	}else if(date=='-'){
		lcd_digit2(11);
	}else{
		lcd_putc(date);
		lcd_gotoxy(lcd_pos_x-1,lcd_pos_y-1);
		lcd_putc(' ');
		lcd_gotoxy(lcd_pos_x,lcd_pos_y+1);
	}
}


/************** lcd_d2_printf() ********************/
int lcd_d2_printf(far char* pfmt, ...){
        va_list(ap);
        va_start(ap,pfmt);
        return _doprnt(lcd_d2_putc, pfmt, ap);
}


 // EOF