/* LCD2_I2C.C: 
*
* Accessing a LC-Display by I2C-bus, uses our software emulation of the I2C-bus. 
* This C sourcefile uses an external assembler module for accessing devices on the I2C-bus. 
*
* This sourcecode is the interface between the LCD-Driver and the I2C-Bus Module.
*
* The schematics for the display is included as PDF file. 
* Software assumes a 4x20 Display, connected to I2C by a PCF8574 with Adr. 66.
*
* (C) 2003 www.Wickenhaeuser.com
*
*/

#include <stdio.h>
#include "lcd_io.h"

// Re-declare Low level routines as REENTRANT for use in lcd_printf()
extern uint i2c_selectslave(uchar adr) reentrant ;
extern uint i2c_sendstop(void) reentrant;  	
extern uint i2c_setbyte(uchar wert) reentrant;
extern uint i2c_reset(void);


#define LCD_ADDR 112	// Display's address A0=0, A1=0, A2=0, using PCF8574 (not -A!)

/******************** lcd_port() *****************/
uchar lcd_port(uchar val) reentrant{	/* Send a 2-byte stream... */
 	if(i2c_selectslave(LCD_ADDR)) return 1;	/* fast version with selectslave */
	i2c_setbyte(val|64);		/* E: 1->0: Clock */
	i2c_setbyte(val);
	i2c_sendstop();
	return 0;	// OK	
}

/******************** lcd_databyte(): Data-Byte (RS=1) *************/
void lcd_databyte(uchar date) reentrant {
	lcd_port((date>>4)|32);	/* Highbyte */
	lcd_port((date&15)|32);	/* Lowbyte */
}

/******************** lcd_combyte(): Command-Byte (RS=0) *************/
void lcd_combyte(uchar date) reentrant {
	lcd_port(date>>4);	/* Highbyte */
	lcd_port(date&15);	/* Lowbyte */
	if(date==1) _wait_ms(5); /* Clear/Home: Wait a short time, Don't check Busy on I2C... */

}

/***************** lcd_initport(): Initialize Port, Optionally reset LCD and setz Access-Type **********/
uchar lcd_initport(void){
	i2c_reset();             /* Reset Bus (it's safer) */
	/* Bit 7=HIGH disables LCD power on Wickenhaeuser boards */
	if(lcd_port(255)) return 1;	/* failed: NO DISPLAY??? */
	_wait_ms(200);		/* Wait abt. 200 ms for RESET */
	lcd_port(127);
	_wait_ms(200);		/* Wait abt. 200 ms for POWER ON */

	lcd_port(3);
	_wait_ms(5);
	lcd_port(3);
	_wait_ms(5);
	lcd_port(3);
	lcd_port(2);		/* Switch to 4 Bit */
	lcd_port(2);		/* Now: 4 Bit, Small Font, >=2 lines */
	lcd_port(8);
	return 0;
}



// EOF
