package gui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;

import javax.swing.JComponent;
import javax.swing.SwingUtilities;

import util.FontChar;
import util.LCDFont;

public class CharEditor extends JComponent implements MouseListener,MouseMotionListener, MouseWheelListener, ImageObserver{
	
	private Pixel[][] pixel;
	private Pixel[][] overlay;
	
	private double zoom=16;

	private MainFrame mainFrame;
	
	private int w;
	private int h;
	
	private int xOffset;
	private int yOffset;
	
	private Dimension fontMetrics;
	
	private Point mouseDown;
	private Point lastPixel = new Point();
	private Point actualPixel = new Point();
	private BufferedImage offScreenImg;
	
	public CharEditor(LCDFont font,MainFrame mainFrame){
		this.mainFrame = mainFrame;
		this.fontMetrics = font.getFontMetrics();
		
		w = fontMetrics.width*font.getBigFontX();
		h = fontMetrics.height*font.getBigFontY();
		
		pixel = new Pixel[w][h];
		overlay = new Pixel[w][h];
		for (int i = 0; i < w; i++) {
			for (int j = 0; j < h; j++) {
				pixel[i][j] = new Pixel();
				overlay[i][j] = new Pixel();
				overlay[i][j].setColor(new Color(255,255,255,0));
			}
		}
		this.addMouseWheelListener(this);
		this.addMouseMotionListener(this);
		this.addMouseListener(this);
		this.setPreferredSize(new Dimension((int)(w*zoom),(int)(h*zoom)));
		offScreenImg = new BufferedImage(w*(int)zoom+1,h*(int)zoom+1,BufferedImage.TYPE_BYTE_INDEXED);
		offPaint();
	}
	public void setFontChar(FontChar fontChar){
		pixel = fontChar.getPixelArray();
		offPaint();
	}
	
	public void paint(Graphics g){
		super.paint(g);	
		
		xOffset = (int) ((this.getWidth()/2)-(int)(zoom*w/2));
		yOffset = (int) ((this.getHeight()/2)-(int)(zoom*h/2));
		
		g.drawImage(offScreenImg,xOffset,yOffset,this);
	}

	public void mouseDragged(MouseEvent e) {
		actualPixel = new Point((int)((e.getX()-xOffset)/zoom),(int)((e.getY()-yOffset)/zoom));
		if (!lastPixel.equals(actualPixel)) {
			switch (MainFrame.tool) {
			case MainFrame.TOOL_PEN:
				Color color = Color.BLACK;
				if (SwingUtilities.isLeftMouseButton(e)) {
					color = Color.BLACK;
				}else if (SwingUtilities.isRightMouseButton(e)) {
					color = Color.WHITE;
				}
				int x = (e.getX()-xOffset)/(int)zoom;
				int y = (e.getY()-yOffset)/(int)zoom;
				mainFrame.setStatusLabelText("X:"+x+" - Y:"+y);
				setPixel(x,y,color);
				offPaint();
				break;
				
			case MainFrame.TOOL_LINE:
				for (int i = 0; i < w; i++) {
					for (int j = 0; j < h; j++) {
						overlay[i][j].setColor(new Color(255,255,255,0));
					}
				}
				int x_start = (mouseDown.x-xOffset)/(int)zoom;
				int y_start = (mouseDown.y-yOffset)/(int)zoom;
				int x_end = (e.getX()-xOffset)/(int)zoom;
				int y_end = (e.getY()-yOffset)/(int)zoom;
				mainFrame.setStatusLabelText("Linie von X:"+x_start+" - Y:"+y_start+"  bis  X:"+x_end+" - Y:"+y_end);
				drawLine(x_start,y_start,x_end,y_end,new Color(200,200,255,127));
				break;
				
			case MainFrame.TOOL_RECT:
				for (int i = 0; i < w; i++) {
					for (int j = 0; j < h; j++) {
						overlay[i][j].setColor(new Color(255,255,255,0));
					}
				}
				x_start= (mouseDown.x-xOffset)/(int)zoom;
				y_start= (mouseDown.y-yOffset)/(int)zoom;
				int width = (e.getX()-xOffset)/(int)zoom-(mouseDown.x-xOffset)/(int)zoom;
				int height = (e.getY()-yOffset)/(int)zoom-(mouseDown.y-yOffset)/(int)zoom;
				drawRect(x_start,y_start,width,height,new Color(200,200,255,127),1);
				mainFrame.setStatusLabelText("Rechteck X:"+x_start+" - Y:"+y_start+"    Breite:"+width+" - Hhe:"+height);
				break;  
			case MainFrame.TOOL_RECT_FILLED:
				for (int i = 0; i < w; i++) {
					for (int j = 0; j < h; j++) {
						overlay[i][j].setColor(new Color(255,255,255,0));
					}
				}
				x_start= (mouseDown.x-xOffset)/(int)zoom;
				y_start= (mouseDown.y-yOffset)/(int)zoom;
				width = (e.getX()-xOffset)/(int)zoom-(mouseDown.x-xOffset)/(int)zoom;
				height = (e.getY()-yOffset)/(int)zoom-(mouseDown.y-yOffset)/(int)zoom;
				drawFilledRect(x_start,y_start,width,height,new Color(200,200,255,127),1);
				mainFrame.setStatusLabelText("Rechteck X:"+x_start+" - Y:"+y_start+"    Breite:"+width+" - Hhe:"+height);
				break;
			case MainFrame.TOOL_ELIPSE:
				for (int i = 0; i < w; i++) {
					for (int j = 0; j < h; j++) {
						overlay[i][j].setColor(new Color(255,255,255,0));
					}
				}
				int cx,cy,xrad,yrad;
				xrad = (int)((e.getX()-mouseDown.x)/(2.0*zoom));
				yrad = (int)((e.getY()-mouseDown.y)/(2.0*zoom));
				cx = (int)(((((e.getX()-xOffset)-(mouseDown.x-xOffset))/2.0)+mouseDown.x-xOffset)/zoom);
				cy = (int)(((((e.getY()-xOffset)-(mouseDown.y-yOffset))/2.0)+mouseDown.y-yOffset)/zoom);
				drawEllipse(cx,cy,xrad,yrad,new Color(200,200,255,127),1);
				break;
				
			default:
				break;
			}
			lastPixel=actualPixel;
		}
	}

	public void mouseMoved(MouseEvent e) {
		/*actualPixel = new Point((int)(e.getX()/zoom),(int)(e.getY()/zoom));
		if (!lastPixel.equals(actualPixel)) {
			for (int i = 0; i < w; i++) {
				for (int j = 0; j < h; j++) {
					overlay[i][j].setColor(new Color(255,255,255,0));
				}
			}
			setOverlayPixel(x,y,new Color(200,200,255,127));
			offPaint();
			lastPixel=actualPixel;
		}*/
		int x = (e.getX()-xOffset)/(int)zoom;
		int y = (e.getY()-yOffset)/(int)zoom;
		mainFrame.setStatusLabelText("X:"+x+" - Y:"+y);
	}

	public void mouseClicked(MouseEvent e) {
		//rasterCircle(10,10,5);
		
	}

	public void mousePressed(MouseEvent e) {
		lastPixel = new Point((int)((e.getX()-xOffset)/zoom),(int)((e.getY()-yOffset)/zoom));
		mouseDown = e.getPoint();
		if (MainFrame.tool==MainFrame.TOOL_PEN) {
			if (SwingUtilities.isLeftMouseButton(e)) {
				int x = (e.getX()-xOffset)/(int)zoom;
				int y = (e.getY()-yOffset)/(int)zoom;
				setPixel(x,y,Color.BLACK);
			}else if (SwingUtilities.isRightMouseButton(e)) {
				int x = (e.getX()-xOffset)/(int)zoom;
				int y = (e.getY()-yOffset)/(int)zoom;
				setPixel(x,y,Color.WHITE);
			}
			offPaint();
		}
	}

	public void mouseReleased(MouseEvent e) {
		Color color = Color.BLACK;
		if (SwingUtilities.isLeftMouseButton(e)) {
			color = Color.BLACK;
		}else if (SwingUtilities.isRightMouseButton(e)) {
			color = Color.WHITE;
		}
		for (int i = 0; i < w; i++) {
			for (int j = 0; j < h; j++) {
				Color overlayColor = overlay[i][j].getColor();
				overlay[i][j].setColor(new Color(255,255,255,0));
				if (!overlayColor.equals(new Color(255,255,255,0))) {
					pixel[i][j].setColor(color);
				}
				
			}
		}
		offPaint();
	}

	public void mouseEntered(MouseEvent e) {

	}

	public void mouseExited(MouseEvent e) {

	}

	public void mouseWheelMoved(MouseWheelEvent e) {
		if (e.getWheelRotation()<0) {
			if (zoom<=50) {
				zoom = Math.round(zoom+zoom/8+0.5);
			}
		}else{
			if (zoom>=2) {
				zoom=(int)(zoom-zoom/8);
			}
		}
		this.setSize(new Dimension((int)(w*zoom),(int)(h*zoom)));
		this.setPreferredSize(new Dimension((int)(w*zoom),(int)(h*zoom)));
		if (zoom>10) {
			offScreenImg = new BufferedImage(w*(int)zoom+1,h*(int)zoom+1,BufferedImage.TYPE_BYTE_INDEXED);
		}else{
			offScreenImg = new BufferedImage(w*(int)zoom,h*(int)zoom,BufferedImage.TYPE_BYTE_INDEXED);
		}
		
		offPaint();
	}
	
	void offPaint(){
		Graphics graphics = offScreenImg.getGraphics();
		for (int i = 0; i < w; i++) {
			for (int j = 0; j < h; j++) {
								
				graphics.setColor(pixel[i][j].getColor());
				graphics.fillRect( ((i*(int)zoom)) , ((j*(int)zoom)) , (int)zoom , (int)zoom );
				
				
				if (overlay[i][j].getColor().getAlpha()!=0) {
					graphics.setColor(overlay[i][j].getColor());
					graphics.fillRect( ((i*(int)zoom)) , ((j*(int)zoom)) , (int)zoom , (int)zoom );
				}
				if (zoom>10) {
					graphics.setColor(Color.LIGHT_GRAY);
					graphics.drawRect( (i*(int)zoom), (j*(int)zoom) , (int)zoom , (int)zoom );
				}
			}
		}
		repaint();
	}
	
	private void drawRect(int x, int y, int w, int h, Color color,int level) {
		drawLine(x, y, x+w, y,color);		// oben
		drawLine(x, y, x, y+h,color);		// links
		drawLine(x, y+h, x+w, y+h,color);	// unten
		drawLine(x+w, y, x+w, y+h,color);	// rechts
		offPaint();
	}
	
	void setOverlayPixel(int x, int y, Color color){
		if (x<pixel.length&&y<pixel[0].length&&x>=0&&y>=0) {
			overlay[x][y].setColor(color);
		}
	}
	
	void setPixel(int x, int y,Color color){
		if (x<pixel.length&&y<pixel[0].length&&x>=0&&y>=0) {
			pixel[x][y].setColor(color);
		}
		
	}
	
	/*void drawEllipse(int x0,int y0,int rx,int ry) {
			  Color color = new Color(200,200,255);
			  final int rx2 = (rx*rx);
			  final int ry2 = (ry*ry);
			  int F = (int)((ry2-rx2*ry+0.25*rx2));
			  int ddF_x = (0);
			  int ddF_y = (2*rx2*ry);
			  int x = (0);
			  int y = (ry);

			  setPixel(x+x0, y+y0, color);
			    // while ( 2*ry2*x < 2*rx2*y ) {  we can use ddF_x and ddF_y
			  while( ddF_x < ddF_y ) {
			    if(F >= 0) {
			      y     -= 1;        // south
			      ddF_y -= 2*rx2;
			      F     -= ddF_y;
			    }
			    x     += 1;          // east
			    ddF_x += 2*ry2;
			    F     += ddF_x + ry2;
			    setPixel(x+x0, y+y0, color);
			  }
			  F = (int)(ry2*(x+0.5)*(x+0.5) + rx2*(y-1)*(y-1) - rx2*ry2);
			  while( y > 0 ) {
			    if(F <= 0) {
			      x     += 1;        // east
			      ddF_x += 2*ry2;
			      F     += ddF_x;
			    }
			    y     -=1;           // south
			    ddF_y -= 2*rx2;
			    F     += rx2 - ddF_y;
			    setPixel(x+x0, y+y0, color);
			  }
			}*/
	
	void rasterCircle(int x0, int y0, int radius)
	 {
	   int f = 1 - radius;
	   int ddF_x = 0;
	   int ddF_y = -2 * radius;
	   int x = 0;
	   int y = radius;
	 
	   setPixel(x0, y0 + radius,Color.BLUE);
	   setPixel(x0, y0 - radius,Color.BLUE);
	   setPixel(x0 + radius, y0,Color.BLUE);
	   setPixel(x0 - radius, y0,Color.BLUE);
	 
	   while(x < y) 
	   {
	     if(f >= 0) 
	     {
	       y--;
	       ddF_y += 2;
	       f += ddF_y;
	     }
	     x++;
	     ddF_x += 2;
	     f += ddF_x + 1;
	 
	     setPixel(x0 + x, y0 + y,Color.BLUE);
	     setPixel(x0 - x, y0 + y,Color.BLUE);
	     setPixel(x0 + x, y0 - y,Color.BLUE);
	     setPixel(x0 - x, y0 - y,Color.BLUE);
	     setPixel(x0 + y, y0 + x,Color.BLUE);
	     setPixel(x0 - y, y0 + x,Color.BLUE);
	     setPixel(x0 + y, y0 - x,Color.BLUE);
	     setPixel(x0 - y, y0 - x,Color.BLUE);
	   }
	   offPaint();
	 }

	void drawLine( int d1x, int d1y, int d2x, int  d2y, Color color){
	 	// Nach http://turing.fh-landshut.de/~jamann/bresenham.html   
	    boolean Z = true;     // zur Bestimmung der Bewegunsrichtung
	    int[] m;             // Bewegungsrichtung/Vektoren
	    // delta a, delta b und Bewegungsrichtung bestimmen
	    
	    int dx = Math.abs(d2x) - Math.abs(d1x);
	    int dy = Math.abs(d2y) - Math.abs(d1y);

	    int da = Math.abs(dx), db = Math.abs(dy);
	    

	    if(da-db<0){
	    	int tmp = da;
	    	da = db;
	    	db = tmp;
	    	Z = false;
	    	}

	    boolean X = (dx >= 0);
	    boolean Y = (dy >= 0);

	    int array[][] ={{ 0,-1,-1,-1 },{ 0,-1,1,-1 },{ 0,1,-1,1 },{ 0,1,1,1 },       
	    				{ -1,0,-1,-1 },{ 1,0,1,-1 },{ -1,0,-1,1 },{ 1,0,1,1 }};
	    
	    m = array[(X?1:0) + (Y?2:0) + (Z?4:0)];
	    
	    int gradient = 2 * db - da;
	    
	    int x = d1x;
	    int y = d1y;
	    
	    setOverlayPixel(x, y,color);

	    for(int i = 0; i < da; i++){
	    	if(gradient >= 0){
	    		x = x + m[2];
	    		y = y + m[3];

	    		gradient = gradient + 2 * db - 2 * da;
	    	}else{
	    		x = x + m[0];
	    		y = y + m[1];
	    		
	    		gradient = gradient + 2 * db;
	    	}
	    	setOverlayPixel(x, y,color);
	     }
	   offPaint();
	 }
	
	void drawEllipse(int CX, int CY, int XRadius, int YRadius, Color color, int level) { 
		if (!(XRadius<=0||YRadius<=0)) {
			int X,Y; 
		      int XChange, YChange; 
		      int EllipseError; 
		      int TwoASquare, TwoBSquare; 
		      int StoppingX, StoppingY; 
		  
		      TwoASquare = 2*XRadius*XRadius; 
		      TwoBSquare = 2*YRadius*YRadius; 
		      X = XRadius; 
		      Y = 0; 
		      XChange = YRadius*YRadius*(1-2*XRadius); 
		      YChange =  XRadius*XRadius; 
		      EllipseError = 0; 
		      StoppingX = TwoBSquare*XRadius; 
		      StoppingY = 0; 
		  
		      // algorithm continues on the next page  
		  
		      while (StoppingX >= StoppingY) //1st set of points, y > 1 
		      { 
		    	  setOverlayPixel(CX+X, CY+Y,color);
		    	  setOverlayPixel(CX-X, CY+Y,color);
		    	  setOverlayPixel(CX-X, CY-Y,color);
		    	  setOverlayPixel(CX+X, CY-Y,color);          
		           Y += 1; 
		           StoppingY += TwoASquare; 
		           EllipseError += YChange; 
		           YChange += TwoASquare; 
		  
		           if ((2*EllipseError + XChange) > 0 )  
		           { 
		                X -= 1; 
		                StoppingX -= TwoBSquare; 
		                EllipseError += XChange; 
		                XChange += TwoBSquare; 
		           } 
		      } 
		  
		      // 1st point set is done; start the 2nd set of points      
		  
		      X =  0; 
		      Y = YRadius; 
		      XChange = YRadius*YRadius; 
		      YChange = XRadius*XRadius*(1-2*YRadius); 
		      EllipseError = 0; 
		      StoppingX = 0; 
		      StoppingY = TwoASquare*YRadius; 
		      while ( StoppingX <= StoppingY ) //2nd set of points, y < 1 
		      { 
		    	  setOverlayPixel(CX+X, CY+Y,color);
		    	  setOverlayPixel(CX-X, CY+Y,color);
		    	  setOverlayPixel(CX-X, CY-Y,color);
		    	  setOverlayPixel(CX+X, CY-Y,color); 
		           X += 1; 
		           StoppingX += TwoBSquare; 
		           EllipseError += XChange; 
		           XChange += TwoBSquare; 
		           if ((2*EllipseError + YChange) > 0 ) 
		           { 
		                Y -= 1; 
		                StoppingY -= TwoASquare; 
		                   EllipseError += YChange; 
		                   YChange += TwoASquare; 
		           } 
		      } 
		}
		offPaint();
	 }

	void drawFilledRect(int x, int y, int w, int h,Color color,int level){
		int x_start = Math.min(x,w+x);
		int x_end = Math.max(x,x+w);
		int y_start = Math.min(y,y+h);
		int y_end = Math.max(y,y+h);
		for (int i = x_start; i <= x_end; i++) {
			for (int j = y_start; j <= y_end; j++) {
				setOverlayPixel(i,j,color);
			}
		}
		offPaint();
	}
}
