//*----------------------------------------------------------------------------
// eve.c
// Routinen fr die Ansteuerung des FT800 v1.0
// 10/2013 Erwin Reuss
// http://www.mc32.de
//*----------------------------------------------------------------------------
#include "eve.h"
#include "spi.h"
//*----------------------------------------------------------------------------
static uint32_t dli;   // Display-List Index
static uint32_t cmdi;  // Command-List Index
//*----------------------------------------------------------------------------
// BOOTUP-CODE
//*----------------------------------------------------------------------------
const uint32_t bootup[] = {
  CLEAR_COLOR_RGB(0,0,0),
  CLEAR(1,1,1),
  DISPLAY()
}; 
//*----------------------------------------------------------------------------
// TRANSFER START/END
//*----------------------------------------------------------------------------
static void EVE_TransferStart(uint32_t adr, T_TRANSFER_RW rw)
{
  EVE_Select();
  if(rw == T_READ) { 
    SPIO((adr >> 16) & 0x3F);   // COM00 - ADR 21:16
    SPIO((adr >>  8) & 0xFF);   // ADR 15:8
    SPIO((adr      ) & 0xFF);   // ADR 7:0
    SPIO(0xFF);                 // Dummy
  } else {
    SPIO(((adr >> 16) & 0x3F) | 0x80);   // COM10 - ADR 21:16
    SPIO((adr >>  8) & 0xFF);   // ADR 15:8
    SPIO((adr      ) & 0xFF);   // ADR 7:0
  }    
}
//*----------------------------------------------------------------------------
__INLINE void EVE_TransferEnd(void) 
{
  EVE_Deselect(); 
}  
//*----------------------------------------------------------------------------
// BOOTUP
//*----------------------------------------------------------------------------
void EVE_Init(void)
{
  dl_ini();       // Display-List Reset
  cmd_ini();      // CMD-List Reset

  host_command(CORE_RESET);  
  rd8(0x00000);

  wrmem(RAM_DL, (uint8_t *)&bootup, sizeof(bootup));
  wr8(REG_DLSWAP, DLSWAP_FRAME);
        
  wr32(REG_GPIO_DIR, 0x02);
  wr32(REG_GPIO,     0x82);
  wr32(REG_PWM_DUTY, 0x60);
  wr32(REG_PCLK_POL, 1);
  wr32(REG_PCLK,     5);
}
//*----------------------------------------------------------------------------
// HOST COMMAND
//*----------------------------------------------------------------------------
void host_command(uint8_t cmd)
{
  EVE_Select(); 
  SPIO((cmd & 0x3F) | 0x40);
  SPIO(0x00);
  EVE_TransferEnd();
}
//*----------------------------------------------------------------------------
// MEMEORY READ
//*----------------------------------------------------------------------------
void rdmem(uint32_t adr, uint8_t *buf, uint16_t cnt)
{
  EVE_TransferStart(adr, T_READ);
  while(cnt) {
    *buf++ = SPIO(0xFF); 
    cnt--;
  }
  EVE_TransferEnd();
}
//*----------------------------------------------------------------------------
// MEMEORY WRITE
//*----------------------------------------------------------------------------
void wrmem(uint32_t adr, uint8_t *buf, uint16_t cnt)
{
  EVE_TransferStart(adr, T_WRITE);
  while(cnt) {
    SPIO(*buf++); 
    cnt--;
  }
  EVE_TransferEnd();
}
//*----------------------------------------------------------------------------
// WRITE 8-BIT
//*----------------------------------------------------------------------------
void wr8(uint32_t adr, uint8_t val)
{
  EVE_TransferStart(adr, T_WRITE);
  SPIO(val);
  EVE_TransferEnd();
}
//*----------------------------------------------------------------------------
// WRITE 16-BIT
//*----------------------------------------------------------------------------
void wr16(uint32_t adr, uint16_t val)
{
  EVE_TransferStart(adr, T_WRITE);
  SPIO(val        & 0xFF);
  SPIO((val >> 8) & 0xFF);
  EVE_TransferEnd();
}
//*----------------------------------------------------------------------------
// WRITE 32-BIT
//*----------------------------------------------------------------------------
void wr32(uint32_t adr, uint32_t val)
{
  EVE_TransferStart(adr, T_WRITE);
  SPIO(val         & 0xFF);
  SPIO((val >> 8)  & 0xFF);
  SPIO((val >> 16) & 0xFF);
  SPIO((val >> 24) & 0xFF);
  EVE_TransferEnd();
}
//*----------------------------------------------------------------------------
// READ 8-BIT
//*----------------------------------------------------------------------------
uint8_t rd8(uint32_t adr)
{
  EVE_TransferStart(adr, T_READ);
  uint8_t res = SPIO(0xFF);
  EVE_TransferEnd();
  return res;
}
//*----------------------------------------------------------------------------
// READ 16-BIT
//*----------------------------------------------------------------------------
uint16_t rd16(uint32_t adr)
{
  EVE_TransferStart(adr, T_READ);
  uint16_t res = SPIO(0xFF);
  res    |= SPIO(0xFF) << 8;
  EVE_TransferEnd();
  return res;
}
//*----------------------------------------------------------------------------
// READ 32-BIT
//*----------------------------------------------------------------------------
uint32_t rd32(uint32_t adr)
{
  EVE_TransferStart(adr, T_READ);
  uint32_t res = SPIO(0xFF);
  res    |= SPIO(0xFF) << 8;
  res    |= SPIO(0xFF) << 16;
  res    |= SPIO(0xFF) << 24;
  EVE_TransferEnd();
  return res;
}
//*----------------------------------------------------------------------------
// DISPLAY-LIST RESET
//*----------------------------------------------------------------------------
void dl_ini(void)
{ 
  dli = 0; 
}
//*----------------------------------------------------------------------------
// DISPLAY-LIST COMMAND
//*----------------------------------------------------------------------------
void dl(uint32_t cmd) 
{ 
  wr32(RAM_DL + dli, cmd); 
  dli += 4; 
}
//*----------------------------------------------------------------------------
// CMD-FIFO RESET
//*----------------------------------------------------------------------------
void cmd_ini(void)
{ 
  cmdi = 0; 
}
//*----------------------------------------------------------------------------
// CMD GET FREE BUFFER
//*----------------------------------------------------------------------------
static uint16_t cmd_fifo_free(void)
{
  uint16_t full = (cmdi - rd16(REG_CMD_READ)) & 0xFFF;
  return (FT_CMD_FIFO_SIZE - 4) - full;
}
//*----------------------------------------------------------------------------
// UPDATE CMD FIFO
//*----------------------------------------------------------------------------
static void cmd_update_fifo(uint16_t cnt)
{
  cmdi = (cmdi + cnt) & 0xFFF;
  cmdi = (cmdi + 3) & 0xFFC;
  wr16(REG_CMD_WRITE, cmdi);
}
//*----------------------------------------------------------------------------
// CHECK FREE
//*----------------------------------------------------------------------------
static void cmd_wait_free(uint16_t cnt)
{
  uint16_t free;
  do { 
    free = cmd_fifo_free();
  } while(free < cnt);  
}
//*----------------------------------------------------------------------------
// WRITE CMD32
//*----------------------------------------------------------------------------
void cmd(uint32_t cmd)
{
  cmd_wait_free(4);
  wr32(RAM_CMD + cmdi, cmd);
  cmd_update_fifo(4);  
}
//*----------------------------------------------------------------------------
// WAIT FOR CMD FIFO EMPTY
//*----------------------------------------------------------------------------
void cmd_wait_ready(void)
{
  while(rd16(REG_CMD_READ) != rd16(REG_CMD_WRITE));
  cmdi = rd16(REG_CMD_WRITE);  
} 
//*----------------------------------------------------------------------------
// WRITE CMD BUFFER
//*----------------------------------------------------------------------------
void cmd_buf(uint8_t *buf, uint16_t cnt)
{
  uint32_t len; 
  uint32_t txcnt; 

  while(cnt > 0) {
    len = cnt;
    if(len > cmd_fifo_free()) { len = cmd_fifo_free(); }
    cmd_wait_free(len);
    
    EVE_TransferStart(RAM_CMD + cmdi, T_WRITE);
    txcnt = 0;
    while(len--) {
      SPIO(*buf++);
      txcnt++;
    }
    len = txcnt;
    EVE_TransferEnd();
    cmd_update_fifo(len);
    cmd_wait_ready();
    cnt -= len;         
  };  
}
//*----------------------------------------------------------------------------
// CMD-FUNKTIONEN
//*----------------------------------------------------------------------------
void cmd_memset(uint32_t ptr, uint32_t value, uint32_t num)
{
  cmd(CMD_MEMSET);
  cmd(ptr); 
  cmd(value); 
  cmd(num); 
}
//*----------------------------------------------------------------------------
void cmd_bgcolor(uint32_t c)
{
  cmd(CMD_BGCOLOR);
  cmd(c);
}
//*----------------------------------------------------------------------------
void cmd_fgcolor(uint32_t c)
{
  cmd(CMD_FGCOLOR);
  cmd(c);
}
//*----------------------------------------------------------------------------
void cmd_append(uint32_t ptr, uint32_t num)
{
  cmd(CMD_APPEND);
  cmd(ptr); 
  cmd(num); 
}  
//*----------------------------------------------------------------------------
void cmd_getptr(uint32_t result)
{
  cmd(CMD_GETPTR);
  cmd(result); 
}  
//*----------------------------------------------------------------------------
void cmd_setfont(uint32_t font, uint32_t ptr)
{
  cmd(CMD_SETFONT);
  cmd(font);
  cmd(ptr);
}
//*----------------------------------------------------------------------------
void cmd_text(uint16_t x, uint16_t y, uint16_t font, uint16_t options, const char * s)
{
  cmd(CMD_TEXT);
  cmd(y << 16 | x);
  cmd(options << 16 | font);
  cmd_buf((uint8_t *)s, StrLen(s) + 1); 
}
//*----------------------------------------------------------------------------
void cmd_number(uint16_t x, uint16_t y, uint16_t font, uint16_t options, uint32_t n)
{
  cmd(CMD_NUMBER);
  cmd(y << 16 | x);
  cmd(options << 16 | font);
  cmd(n); 
}
//*----------------------------------------------------------------------------
void cmd_spinner(uint16_t x, uint16_t y, uint16_t style, uint16_t scale)
{
  cmd(CMD_SPINNER);
  cmd(y << 16 | x);
  cmd(scale << 16 | style);
}
//*----------------------------------------------------------------------------
void cmd_gauge(uint16_t x, uint16_t y, uint16_t r, uint16_t options, uint16_t major, uint16_t minor, uint16_t val, uint16_t range)
{
  cmd(CMD_GAUGE);
  cmd(y << 16 | x);
  cmd(options << 16 | r);
  cmd(minor << 16 | major);
  cmd(range << 16 | val);
}
//*----------------------------------------------------------------------------
void cmd_clock(uint16_t x, uint16_t y, uint16_t r, uint16_t options, uint16_t h, uint16_t m, uint16_t s, uint16_t ms)
{
  cmd(CMD_CLOCK);
  cmd(y << 16 | x);
  cmd(options << 16 | r);
  cmd(m << 16 | h);
  cmd(ms << 16 | s);
}
//*----------------------------------------------------------------------------
void cmd_gradient(uint16_t x0, uint16_t y0, uint32_t rgb0, uint16_t x1, uint16_t y1, uint32_t rgb1)
{
  cmd(CMD_GRADIENT);
  cmd(y0 << 16 | x0);
  cmd(rgb0);
  cmd(y1 << 16 | x1);
  cmd(rgb1);
}
//*----------------------------------------------------------------------------
void cmd_track(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t tag)
{
  cmd(CMD_TRACK);
  cmd(y << 16 | x);
  cmd(h << 16 | w);
  cmd(tag);
}
//*----------------------------------------------------------------------------
void cmd_slider(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t options, uint16_t val, uint16_t range)
{
  cmd(CMD_SLIDER);
  cmd(y << 16 | x);
  cmd(h << 16 | w);
  cmd(val << 16 | options);
  cmd(range);
}
//*----------------------------------------------------------------------------
void cmd_button(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t font, uint16_t options, const char *s)
{
  cmd(CMD_BUTTON);
  cmd(y << 16 | x);
  cmd(h << 16 | w);
  cmd(y << 16 | font);
  cmd_buf((uint8_t *)s, StrLen(s) + 1); 
}
//*----------------------------------------------------------------------------
void cmd_keys(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t font, uint16_t options, const char *s)
{
  cmd(CMD_KEYS);
  cmd(y << 16 | x);
  cmd(h << 16 | w);
  cmd(y << 16 | font);
  cmd_buf((uint8_t *)s, StrLen(s) + 1); 
}
//*----------------------------------------------------------------------------
