//*----------------------------------------------------------------------------
// eve.h
//*----------------------------------------------------------------------------
#ifndef __EVE_H
#define __EVE_H
//*----------------------------------------------------------------------------
#include "stdint.h"
//*----------------------------------------------------------------------------
#define POWER_STANDBY         0x41          // Put FT800 core to standby mode. Clock gate off, PLL and Oscillator remain on (default).
#define POWER_SLEEP           0x42          // Put FT800 core to sleep mode. Clock gate off, PLL and Oscillator off.
#define POWER_DOWN            0x50          // Switch off 1.2V internal regulator. Clock gate off, PLL and Oscillator off.
//*----------------------------------------------------------------------------
#define CLOCK_INT             0x48          // Select PLL input from Internal relaxation oscillator (default).
#define CLOCK_EXT             0x44          // Select PLL input from Crystal oscillator or external input clock.
#define CLOCK_48M             0x62          // Switch PLL output clock to 48MHz (default).
#define CLOCK_36M             0x61          // Switch PLL output clock to 36MHz.
#define CLOCK_24M             0x64          // For Internal tests.
//*----------------------------------------------------------------------------
#define CORE_RESET            0x68          // Send reset pulse to FT800 core.
//*----------------------------------------------------------------------------
#define FT_DL_SIZE            (0x2000)      // 8KB Display List buffer size
#define FT_CMD_FIFO_SIZE      (0x1000)      // 4KB coprocessor Fifo size
#define FT_CMD_SIZE           (4)           // 4 byte per coprocessor command of EVE
//*----------------------------------------------------------------------------
#define FT_DispWidth          480           // Display Width in pixel
#define FT_DispHeight         272           // Display Height in pixel
//*----------------------------------------------------------------------------
#define CMD_DLSTART           0xFFFFFF00    // Start a new display list
#define CMD_SWAP              0xFFFFFF01    // Swap the current display list
#define CMD_INTERRUPT         0xFFFFFF02    // Trigger interrupt INT_CMDFLAG
#define CMD_CRC               0xFFFFFF03    //
#define CMD_HAMMERAUX         0xFFFFFF04    //
#define CMD_MARCH             0xFFFFFF05    //
#define CMD_IDCT              0xFFFFFF06    //
#define CMD_EXECUTE           0xFFFFFF07    //
#define CMD_GETPOINT          0xFFFFFF08    //
#define CMD_BGCOLOR           0xFFFFFF09    // Set the background color
#define CMD_FGCOLOR           0xFFFFFF0A    // Set the foreground color
#define CMD_GRADIENT          0xFFFFFF0B    // Draw a smooth color gradient
#define CMD_TEXT              0xFFFFFF0C    // Draw text
#define CMD_BUTTON            0xFFFFFF0D    // Draw a button
#define CMD_KEYS              0xFFFFFF0E    // Draw a row of keys
#define CMD_PROGRESS          0xFFFFFF0F    // Draw a progress bar
#define CMD_SLIDER            0xFFFFFF10    // Draw a slider
#define CMD_SCROLLBAR         0xFFFFFF11    // Draw a scrollbar
#define CMD_TOGGLE            0xFFFFFF12    // Draw a toggle switch
#define CMD_GAUGE             0xFFFFFF13    // Draw a gauge
#define CMD_CLOCK             0xFFFFFF14    // Draw an analog clock
#define CMD_CALIBRATE         0xFFFFFF15    // Execute the touch screen calibration routine
#define CMD_SPINNER           0xFFFFFF16    // Start an animated spinner
#define CMD_STOP              0xFFFFFF17    // Stop any spinner, screensaver or sketch
#define CMD_MEMCRC            0xFFFFFF18    // Compute a CRC-32 for memory
#define CMD_REGREAD           0xFFFFFF19    // Read a register value
#define CMD_MEMWRITE          0xFFFFFF1A    // Write bytes into memory
#define CMD_MEMSET            0xFFFFFF1B    // Fill memory with a byte value
#define CMD_MEMZERO           0xFFFFFF1C    // Write zero to a block of memory
#define CMD_MEMCPY            0xFFFFFF1D    // Copy a block of memory
#define CMD_APPEND            0xFFFFFF1E    // Append memory to display list
#define CMD_SNAPSHOT          0xFFFFFF1F    // Take a snapshot of the current screen
#define CMD_TOUCH_TRANSFORM   0xFFFFFF20    //
#define CMD_BITMAP_TRANSFORM  0xFFFFFF21    // 
#define CMD_INFLATE           0xFFFFFF22    // Decompress data into memory
#define CMD_GETPTR            0xFFFFFF23    // Get the end memory address of inflated data
#define CMD_LOADIMAGE         0xFFFFFF24    // Load a JPEG image
#define CMD_GETPROPS          0xFFFFFF25    //
#define CMD_LOADIDENTITY      0xFFFFFF26    // Set the current matrix to the identity matrix
#define CMD_TRANSLATE         0xFFFFFF27    // Apply a translation to the current matrix
#define CMD_SCALE             0xFFFFFF28    // Apply a scale to the current matrix
#define CMD_ROTATE            0xFFFFFF29    // Apply a rotation to the current matrix
#define CMD_SETMATRIX         0xFFFFFF2A    // Write the current matrix to the display list
#define CMD_SETFONT           0xFFFFFF2B    // Set up a custom font
#define CMD_TRACK             0xFFFFFF2C    // Track touches for a graphic object
#define CMD_DIAL              0xFFFFFF2D    // Draw a rotary dial control
#define CMD_NUMBER            0xFFFFFF2E    // Draw a decimal number
#define CMD_SCREENSAVER       0xFFFFFF2F    // Start an animated screensaver
#define CMD_SKETCH            0xFFFFFF30    // Start a continous sketch update
#define CMD_LOGO              0xFFFFFF31    // Play FTDI logo animation
#define CMD_COLDSTART         0xFFFFFF32    // Set co-processor engine state to default values
#define CMD_GETMATRIX         0xFFFFFF33    // Retrieves the current matrix coefficients
#define CMD_GRADCOLOR         0xFFFFFF34    // Set the 3D button highlight color
//*----------------------------------------------------------------------------
// Graphics primitives list
//*----------------------------------------------------------------------------
#define BITMAPS               1
#define FTPOINTS              2
#define POINTS                2
#define LINES                 3
#define LINE_STRIP            4
#define EDGE_STRIP_R          5
#define EDGE_STRIP_L          6
#define EDGE_STRIP_A          7
#define EDGE_STRIP_B          8
#define RECTS                 9
//*----------------------------------------------------------------------------
// Graphocs bitmap format table
//*----------------------------------------------------------------------------
#define ARGB1555              0
#define L1                    1
#define L4                    2
#define L8                    3
#define RGB332                4
#define ARGB2                 5
#define ARGB4                 6
#define RGB565                7
#define PALETTED              8
#define TEXT8X8               9
#define TEXTVGA               10
#define BARGRAPH              11
//*----------------------------------------------------------------------------
// Options
//*----------------------------------------------------------------------------
#define OPT_MONO              0x0001
#define OPT_NODL              0x0002
#define OPT_FLAT              0x0100
#define OPT_SIGNED            0x0100
#define OPT_CENTERX           0x0200
#define OPT_CENTERY           0x0400
#define OPT_CENTER            0x0600
#define OPT_RIGHTX            0x0800
#define OPT_NOBACK            0x1000
#define OPT_NOTICKS           0x2000
#define OPT_NOHM              0x4000
#define OPT_NOPOINTER         0x4000
#define OPT_NOSECS            0x8000
#define OPT_NOHANDS           0xC000
//*----------------------------------------------------------------------------
// Alpha-Func
//*----------------------------------------------------------------------------
#define NEVER                 0
#define LESS                  1
#define LEQUAL                2
#define GREATER               3
#define GEQUAL                4
#define EQUAL                 5
#define NOTEQUAL              6
#define ALWAYS                7
//*----------------------------------------------------------------------------
// Bitmap Wrap
//*----------------------------------------------------------------------------
#define BORDER                0
#define REPEAT                1
//*----------------------------------------------------------------------------
// Bitmap Filter
//*----------------------------------------------------------------------------
#define NEAREST               0
#define BILINEAR              1
//*----------------------------------------------------------------------------
// Blend_func constant value definition
//*----------------------------------------------------------------------------
#define ZERO                  0
#define ONE                   1
#define SRC_ALPHA             2
#define DST_ALPHA             3
#define ONE_MINUS_SRC_ALPHA   4
#define ONE_MINUS_DST_ALPHA   5
//*----------------------------------------------------------------------------
// Stencil test passes
//*----------------------------------------------------------------------------
//#define ZERO                0
#define KEEP                  1
#define REPLACE               2
#define INCR                  3
#define DECR                  4
#define INVERT                5
#define INCR_WRAP             6
#define DECR_WRAP             7
//*----------------------------------------------------------------------------
// Touchmode
//*----------------------------------------------------------------------------
#define TOUCHMODE_OFF         0
#define TOUCHMODE_ONESHOT     1
#define TOUCHMODE_FRAME       2
#define TOUCHMODE_CONTINUOUS  3
//*----------------------------------------------------------------------------
// Dlswap options
//*----------------------------------------------------------------------------
#define DLSWAP_DONE           0
#define DLSWAP_FRAME          2
#define DLSWAP_LINE           1
//*----------------------------------------------------------------------------
#define ADC_SINGLE_ENDED      0
#define ADC_DIFFERENTIAL      1
//*----------------------------------------------------------------------------
#define LINEAR_SAMPLES        0
#define ULAW_SAMPLES          1
#define ADPCM_SAMPLES         2
//*----------------------------------------------------------------------------
// Interrupt-Flags
//*----------------------------------------------------------------------------
#define INT_SWAP              0x01
#define INT_TOUCH             0x02
#define INT_TAG               0x04
#define INT_SOUND             0x08
#define INT_PLAYBACK          0x10
#define INT_CMDEMPTY          0x20
#define INT_CMDFLAG           0x40
#define INT_CONVCOMPLETE      0x80
//*----------------------------------------------------------------------------
#define RAM_G                 0x000000      // -03FFFF = 256k - Main Graphics RAM
#define RAM_DL                0x100000      // -101FFF = 8k   - Display List RAM
#define RAM_PAL               0x102000      // -1023FF = 1k   - Palette RAM
#define RAM_CMD               0x108000      // -108FFF = 4k   - Graphics Engine Command Buffer
//*----------------------------------------------------------------------------
#define ROM_CHIPID            0x0C0000      // FT800 chip identification and revision information = 08000100
#define ROM_FONT              0x0BB23C      // -0FFFFB = 275k - Font Table and Bitmap
#define ROM_FONT_ADDR         0x0FFFFC      // -0FFFFF = 4b   - Font Table Pointer Address
//*----------------------------------------------------------------------------
#define REG_ID                0x102400      // Identification register, always reads as 7Ch
#define REG_FRAMES            0x102404      // Frame counter, since reset
#define REG_CLOCK             0x102408      // Clock cycles, since reset
#define REG_FREQUENCY         0x10240C      // Main clock frequency
#define REG_RENDERMODE        0x102410      // Rendering mode: 0 = normal, 1 = single-line
#define REG_SNAPY             0x102414      // Scan line select for RENDERMODE 1
#define REG_SNAPSHOT          0x102418      // trigger for RENDERMODE 1
#define REG_CPURESET          0x10241C      // Coprocessor reset control
#define REG_TAP_CRC           0x102420      // Live video tap crc
#define REG_TAP_MASK          0x102424      // Live video tap mask
#define REG_HCYCLE            0x102428      // Horizontal total cycle count
#define REG_HOFFSET           0x10242C      // Horizontal display start offset
#define REG_HSIZE             0x102430      // Horizontal display pixel count
#define REG_HSYNC0            0x102434      // Horizontal sync fall offset
#define REG_HSYNC1            0x102438      // Horizontal sync rise offset
#define REG_VCYCLE            0x10243C      // Vertical total cycle count
#define REG_VOFFSET           0x102440      // Vertical display start offset
#define REG_VSIZE             0x102444      // Vertical display line count
#define REG_VSYNC0            0x102448      // Vertical sync fall offset
#define REG_VSYNC1            0x10244C      // Vertical sync rise offset
#define REG_DLSWAP            0x102450      // Display list swap control
#define REG_ROTATE            0x102454      // Screen 180 degree rotate
#define REG_OUTBITS           0x102458      // Output bit resolution, 3x3x3 bits
#define REG_DITHER            0x10245C      // Output dither enable
#define REG_SWIZZLE           0x102460      // Output RGB signal swizzle
#define REG_CSPREAD           0x102464      // Output clock spreading enable
#define REG_PCLK_POL          0x102468      // PCLK polarity, 0 = output on PCLK rising edge, 1 = output on PCLK falling edge
#define REG_PCLK              0x10246C      // PCLK frequency divider, 0 = disable
#define REG_TAG_X             0x102470      // Tag query X coordinate
#define REG_TAG_Y             0x102474      // Tag query Y coordinate
#define REG_TAG               0x102478      // Tag query result
#define REG_VOL_PB            0x10247C      // Volume for playback
#define REG_VOL_SOUND         0x102480      // Volume for synthesizer sound
#define REG_SOUND             0x102484      // Sound effect select
#define REG_PLAY              0x102488      // Start effect playback
#define REG_GPIO_DIR          0x10248C      // GPIO pin direction, 1 = output, 0 = input
#define REG_GPIO              0x102490      // GPIO read/write
#define REG_INT_FLAGS         0x102498      // Interrupt flags
#define REG_INT_EN            0x10249C      // Global interrupt enable
#define REG_INT_MASK          0x1024A0      // Interrupt enable mask
#define REG_PLAYBACK_START    0x1024A4      // Audio playback RAM start address
#define REG_PLAYBACK_LENGTH   0x1024A8      // Audio playback sample length (bytes)
#define REG_PLAYBACK_READPTR  0x1024AC      // Audio playback current read pointer
#define REG_PLAYBACK_FREQ     0x1024B0      // Audio playback frequency (Hz)
#define REG_PLAYBACK_FORMAT   0x1024B4      // Audio playback format
#define REG_PLAYBACK_LOOP     0x1024B8      // Audio playback loop enable
#define REG_PLAYBACK_PLAY     0x1024BC      // Start audio playback
#define REG_PWM_HZ            0x1024C0      // PWM output frequency (Hz)
#define REG_PWM_DUTY          0x1024C4      // PWM output duty cycle 0=0%, 128=100%
#define REG_MACRO0            0x1024C8      // Display list macro command 0
#define REG_MACRO1            0x1024CC      // Display list macro command 1
#define REG_CMD_READ          0x1024E4      // Command buffer read pointer
#define REG_CMD_WRITE         0x1024E8      // Command buffer write pointer
#define REG_CMD_DL            0x1024EC      // Command DL offset
#define REG_TOUCH_MODE        0x1024F0      // Touch-screen sampling mode
#define REG_TOUCH_ADC_MODE    0x1024F4      // Select single ended (low power) or differential (accurate) sampling
#define REG_TOUCH_CHARGE      0x1024F8      // Touch-screen charge time, units of 6 clocks
#define REG_TOUCH_SETTLE      0x1024FC      // Touch-screen settle time, units of 6 clocks
#define REG_TOUCH_OVERSAMPLE  0x102500      // Touch-screen oversample factor
#define REG_TOUCH_RZTHRESH    0x102504      // Touch-screen resistance threshold
#define REG_TOUCH_RAW_XY      0x102508      // Touch-screen raw (x; y) (16, 16)
#define REG_TOUCH_RZ          0x10250C      // Touch-screen resistance
#define REG_TOUCH_SCREEN_XY   0x102510      // Touch-screen screen (x; y) (16, 16)
#define REG_TOUCH_TAG_XY      0x102514      // Touch-screen screen (x; y) used for tag lookup
#define REG_TOUCH_TAG         0x102518      // Touch-screen tag result
#define REG_TOUCH_TRANSFORM_A 0x10251C      // Touch-screen transform coefficient (s15.16)
#define REG_TOUCH_TRANSFORM_B 0x102520      // Touch-screen transform coefficient (s15.16)
#define REG_TOUCH_TRANSFORM_C 0x102524      // Touch-screen transform coefficient (s15.16)
#define REG_TOUCH_TRANSFORM_D 0x102528      // Touch-screen transform coefficient (s15.16)
#define REG_TOUCH_TRANSFORM_E 0x10252C      // Touch-screen transform coefficient (s15.16)
#define REG_TOUCH_TRANSFORM_F 0x102530      // Touch-screen transform coefficient (s15.16)
#define REG_TOUCH_DIRECT_XY   0x102574      // Touch screen direct (x; y) conversions
#define REG_TOUCH_DIRECT_Z1Z2 0x102578      // Touch screen direct (z1; z2) conversions
#define REG_TRACKER           0x109000      // Read Tracking Value
//*----------------------------------------------------------------------------
#define VERTEX2F(x,y)           (0x40000000 | (((x) & 0x7FFF) << 15) | ((y) & 0x7FFF))
#define VERTEX2II(x,y,handle,cell) \
                                (0x80000000 | (((x) & 0x1FF) << 21) | (((y) & 0x1FF) << 12) | (((handle) & 0x1F) << 7) | ((cell) & 0x7F))
//*----------------------------------------------------------------------------
#define DISPLAY()               (0x00000000)
#define BITMAP_SOURCE(addr)     (0x01000000 | ((addr) & 0xFFFFF))
#define CLEAR_COLOR_RGB(r,g,b)  (0x02000000 | (((r) & 0xFF) << 16) | (((g) & 0xFF) << 8) | ((b) & 0xFF))
#define TAG(s)                  (0x03000000 | ((s) & 0xFF))
#define COLOR_RGB(r,g,b)        (0x04000000 | (((r) & 0xFF) << 16) | (((g) & 0xFF) << 8) | ((b) & 0xFF))
#define BITMAP_HANDLE(handle)   (0x05000000 | ((handle) & 0x1F))
#define CELL(cell)              (0x06000000 | ((cell) & 0x7F))
#define BITMAP_LAYOUT(format,linestride,height) \
                                (0x07000000 | (((format) & 0x1F) << 19) | (((linestride) & 0x3FF) << 9) | ((height) & 0x1FF))
#define BITMAP_SIZE(filter,wrapx,wrapy,width,height) \
                                (0x08000000 | (((filter) & 0x01) << 20) | (((wrapx) & 0x01) << 19) | (((wrapy) & 0x01) << 18) | (((width) & 0x1FF) << 9) | ((height) & 0x1FF))
#define ALPHA_FUNC(func,ref)    (0x09000000 | (((func) & 0x07) << 8) | ((ref) & 0xFF))
#define STENCIL_FUNC(func,ref,mask) \
                                (0x0A000000 | (((func) & 0x07) << 16) | (((ref) & 0xFF) << 8) | ((mask) & 0xFF))
#define BLEND_FUNC(src,dst)     (0x0B000000 | (((src) & 0x07) << 3) | ((dst) & 0x07))
#define STENCIL_OP(sfail,spass) (0x0C000000 | (((sfail) & 0x07) << 3) | ((spass) & 0x07))
#define POINT_SIZE(size)        (0x0D000000 | ((size) & 0x1FFF))
#define LINE_WIDTH(width)       (0x0E000000 | ((width) & 0xFFF))
#define CLEAR_COLOR_A(alpha)    (0x0F000000 | ((alpha) & 0xFF))
#define COLOR_A(alpha)          (0x10000000 | ((alpha) & 0xFF))
#define CLEAR_STENCIL(s)        (0x11000000 | ((s) & 0xFF))
#define CLEAR_TAG(s)            (0x12000000 | ((s) & 0xFF))
#define STENCIL_MASK(mask)      (0x13000000 | ((mask) & 0xFF))
#define TAG_MASK(mask)          (0x14000000 | ((mask) & 0x01))
#define BITMAP_TRANSFORM_A(a)   (0x15000000 | ((a) & 0x1FFFF))
#define BITMAP_TRANSFORM_B(b)   (0x16000000 | ((b) & 0x1FFFF))
#define BITMAP_TRANSFORM_C(c)   (0x17000000 | ((c) & 0xFFFFFF))
#define BITMAP_TRANSFORM_D(d)   (0x18000000 | ((d) & 0x1FFFF))
#define BITMAP_TRANSFORM_E(e)   (0x19000000 | ((e) & 0x1FFFF))
#define BITMAP_TRANSFORM_F(f)   (0x1A000000 | ((f) & 0xFFFFFF))
#define SCISSOR_XY(x,y)         (0x1B000000 | (((x) & 0x1FF) << 9) | ((y) & 0x1FF))
#define SCISSOR_SIZE(width,height) \
                                (0x1C000000 | (((width) & 0x3FF) << 10) | ((height) & 0x3FF))
#define CALL(dest)              (0x1D000000 | ((dest) & 0xFFFF))
#define JUMP(dest)              (0x1E000000 | ((dest) & 0xFFFF))
#define BEGIN(prim)             (0x1F000000 | ((prim) & 0x0F))
#define COLOR_MASK(r,g,b,a)     (0x20000000 | (((r) & 0x01) << 3) | (((g) & 0x01) << 2) | (((b) & 0x01) << 1) | ((a) & 0x01))
#define END()                   (0x21000000)
#define SAVE_CONTEXT()          (0x22000000)
#define RESTORE_CONTEXT()       (0x23000000)
#define RETURN()                (0x24000000)
#define MACRO(m)                (0x25000000 | ((m) & 0x01))
#define CLEAR(c,s,t)            (0x26000000 | (((c) & 0x01) << 2) | (((s) & 0x01) << 1) | ((t) & 0x01))
//*----------------------------------------------------------------------------
typedef enum {
  T_READ  = 0,
  T_WRITE = 0x80
} T_TRANSFER_RW;
//*----------------------------------------------------------------------------
// FT800 font table structure 
// Font table address in ROM can be found by reading the address from 0xFFFFC location. 
// 16 font tables are present at the address read from location 0xFFFFC 
//*----------------------------------------------------------------------------
#define FONT_TABLE_ADR            0xFFFFC
#define FT_GPU_NUMCHAR_PERFONT    (128)
#define FT_GPU_FONT_TABLE_SIZE    (148)
//*----------------------------------------------------------------------------
typedef struct FT_Gpu_Fonts
{
	uint8_t	  FontWidth[FT_GPU_NUMCHAR_PERFONT];  // Width of each character font from 0 to 127
	uint32_t	FontBitmapFormat;   	              // Bitmap format of font - L1, L4, L8
	uint32_t	FontLineStride;      	              // Font line stride in FT800 ROM
	uint32_t	FontWidthInPixels;                  // Font width in pixels
	uint32_t	FontHeightInPixels;	                // Font height in pixels
	uint32_t	PointerToFontGraphicsData;  	      // Pointer to font graphics raw data
} FT_Gpu_Fonts_t;
//*----------------------------------------------------------------------------
// FUNKTIONSPROTOTYPEN
//*----------------------------------------------------------------------------
extern void EVE_Init(void);
extern void EVE_Bootup(void);
extern void host_command(uint8_t cmd);
//*----------------------------------------------------------------------------
extern void wr8(uint32_t adr, uint8_t val);
extern void wr16(uint32_t adr, uint16_t val);
extern void wr32(uint32_t adr, uint32_t val);
extern void wrmem(uint32_t adr, uint8_t *buf, uint16_t cnt);
//*----------------------------------------------------------------------------
extern uint8_t   rd8(uint32_t adr);
extern uint16_t  rd16(uint32_t adr);
extern uint32_t  rd32(uint32_t adr);
extern void      rdmem(uint32_t adr, uint8_t *buf, uint16_t cnt);
//*----------------------------------------------------------------------------
extern void dl_ini(void);
extern void dl(uint32_t cmd);
extern void dl_buf(uint8_t *buf, uint16_t cnt);
//*----------------------------------------------------------------------------
extern void cmd_ini(void);
extern void cmd(uint32_t cmd);
extern void cmd_buf(uint8_t *buf, uint16_t cnt);
extern void cmd_wait_ready(void);
//*----------------------------------------------------------------------------
extern void cmd_memset(uint32_t ptr, uint32_t value, uint32_t num);
extern void cmd_bgcolor(uint32_t c);
extern void cmd_fgcolor(uint32_t c);
extern void cmd_append(uint32_t ptr, uint32_t num);
extern void cmd_getptr(uint32_t result);
extern void cmd_setfont(uint32_t font, uint32_t ptr);
extern void cmd_text(uint16_t x, uint16_t y, uint16_t font, uint16_t options, const char * s);
extern void cmd_number(uint16_t x, uint16_t y, uint16_t font, uint16_t options, uint32_t n);
extern void cmd_spinner(uint16_t x, uint16_t y, uint16_t style, uint16_t scale);
extern void cmd_gauge(uint16_t x, uint16_t y, uint16_t r, uint16_t options, uint16_t major, uint16_t minor, uint16_t val, uint16_t range);
extern void cmd_clock(uint16_t x, uint16_t y, uint16_t r, uint16_t options, uint16_t h, uint16_t m, uint16_t s, uint16_t ms);
extern void cmd_gradient(uint16_t x0, uint16_t y0, uint32_t rgb0, uint16_t x1, uint16_t y1, uint32_t rgb1);
extern void cmd_track(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t tag);
extern void cmd_slider(uint16_t x, uint16_t y, uint16_t h, uint16_t w, uint16_t options, uint16_t val, uint16_t range);
extern void cmd_button(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t font, uint16_t options, const char *s);
extern void cmd_keys(uint16_t x, uint16_t y, uint16_t w, uint16_t h, uint16_t font, uint16_t options, const char *s);
//*----------------------------------------------------------------------------
#endif
