/*
 * Name: main.c
 * Project: vuc-demo
 * License: GNU GPL v2 (see License.txt)
 */

#define VUC_DBG_ON	0

#define VUC_IS_PC	0

#include <config.h>
#include <demo.h>

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <util/delay.h>

#include <stdint.h>
#include <string.h>
#include <stdarg.h>

#include "vuc_avr.h"
#include "../src/vuc_proto.h"
//#include "../src/vuc_types.h"

#include "main.h"


#ifdef COM_UART
#  include "uart.h"
#endif
#ifdef COM_UNIX
#  include "unix.h"
#endif



#define VUC_WAIT_DEL	5000

#define DELVAL		15000
#define LEDVAL		6

#define BUF_SIZE	16

//#define SC_STATE_WAIT	0
#define SC_STATE_REC	1
#define SC_STATE_OUT	2
#define SC_STATE_SENT	3

#define SCOPE_DATA_ADR	0

//#define SCOPE_PKT_LEN	50	// Packet size (if packet-mode)


u16 re_cnt= 0;
u16 wait_cnt= 0;
#if (VUC_DBG_ON)
u8 in_dbgf= 0;
u16 nested_dbgf= 0;
#endif
u8 deaf= 0;
//u8 send_dis= 0;

#if (VUC_PKT_SEQ)
u8 vuc_is_warm= 0;
u8 vuc_in_pkt_no;
u8 vuc_out_pkt_no;
#else
#  define vuc_in_pkt_no		0
#  define vuc_out_pkt_no	0
#endif

vuc_adr_t adr;
typedef i8 (* vuc_pkt_f_t) (u8 blk_no, void * data);

#define VUC_PKT_CHANNELS	3

#if (VUC_PACKET)
typedef struct {
  u8 state;
  u8 blk_no;
  vuc_pkt_f_t func;
  void * data;
} vuc_pkt_ch_t;
vuc_pkt_ch_t vuc_pkt_channels[VUC_PKT_CHANNELS];
u8 vuc_pkt_ch_no= 0;
u8 vuc_out_pkt_wait= 0;		// sendq waiting for pkt-code -> DO NOT SEND
#endif
#if (VUC_PKT_CHK)
u8 vuc_in_pkt_code= 0;		// ack or nak coming in from PC
u8 vuc_out_pkt_code= 0;
#endif
#if (VUC_PKT_CHK)
u8 vuc_chk;
#endif
#if (VUC_TEST_PROTO==1)
u8 diag_cnt= 0;
#endif



u8 scope[SCOPE_SIZE], sc_i= 0, sc_state= 0, sc_last= 0;

vuc_put_data_t vuc_put_data[VUC_UC_PUT_ADRS]= {
  { 4, gen_put_freq },
  { 1, gen_put_power },
  { 1, scope_put_tb },
};

vuc_get_data_t vuc_get_data[VUC_UC_GET_ADRS]= {
  { scope_get },
};


#ifndef SCOPE_PKT_LEN
#  define SCOPE_PKT_LEN		SCOPE_SIZE
#endif
#if (!VUC_PACKET)
#  undef SCOPE_PKT_LEN
#  define SCOPE_PKT_LEN		SCOPE_SIZE
#endif
#define SCOPE_PACKETS \
  ( SCOPE_SIZE/SCOPE_PKT_LEN + (SCOPE_SIZE%SCOPE_PKT_LEN ? 1 : 0) )


#define errf(...)		


void vuc_in_pkt_start ();
void vuc_in_send_hs (i8 rep);

void vuc_pkt_beg (u8 len);
void vuc_pkt_end (void);

void vuc_poll (void);
void main_poll (void);
i8 vuc_pkt_q_send (u8 channel_no, u8 wait, vuc_pkt_f_t func, void * data);


/* ------------------------------------------------------------------------- */

void vuc_send_8 (u8 data)
{
  com_putc(data);
}

void vuc_send_8_8 (u8 data1, u8 data2)
{
  com_putc(data1);
  com_putc(data2);
}

char hex04 (u8 n)
{
  n&= 0x0f;
  return (n<=9 ? n+'0' : n-10+'A');
}

#if (VUC_PKT_CHK)

void vuc_c_send_8 (const char c)
{
  vuc_chk= VUC_CHK(vuc_chk, c);
  vuc_send_8(c);
}

void vuc_c_send_8_8 (const char c1, const char c2)
{
  vuc_chk= VUC_CHK(vuc_chk, c1);
  vuc_chk= VUC_CHK(vuc_chk, c2);
  vuc_send_8_8(c1, c2);
}

#else /* VUC_PKT_CHK */

void vuc_c_send_8 (const char c)
{
  vuc_send_8(c);
}

void vuc_c_send_8_8 (const char c1, const char c2)
{
  vuc_send_8_8(c1, c2);
}

#endif /* VUC_PKT_CHK */

void vuc_c_send_s (const char * s)
{
  while (*s)
    vuc_c_send_8(*s++);
}

void send_hex04 (u8 n)
{
  vuc_c_send_8(hex04(n));
}

void send_hex08 (u8 n)
{
  send_hex04(n>>4);
  send_hex04(n&0x0f);
}

void send_hex16 (u16 n)
{
  send_hex08(n>>8);
  send_hex08(n&0xff);
}

void send_hex32 (u32 n)
{
  send_hex16(n>>16);
  send_hex16(n&0xffff);
}


void vuc_put_beg (u8 adr)
{
  vuc_c_send_8(VUC_CMD_PUT|adr);
}


static u8 vuc_vsendf_cnt (const char * str, va_list args)
{
  u8 len= 0;

  for ( ; *str; str++) {
    if (*str!='%' || !*(str+1)) {
      len++;
    } else {
      if (*(str+1)=='%') {
        len++; str++;
      } else {
        if (*(str+1)=='s') {
          char * s= va_arg(args, char *);
          len+= strlen(s); str++;
        } else
        if (*(str+1)=='x') {
          va_arg(args, int);
          len+= 2; str++;
        }
        if (*(str+1)=='X') {
          va_arg(args, int);
          len+= 4; str++;
        }
      }
    }
  }
  return len;
}

static void vuc_vsendf_out (const char * str, va_list args)
{
  for ( ; *str; str++) {
    if (*str!='%' || !*(str+1)) {
      vuc_c_send_8(*str);
    } else {
      if (*(str+1)=='%') {
        vuc_c_send_8('%');
        str++;
      } else {
        if (*(str+1)=='s') {
          vuc_c_send_s(va_arg(args, char *));
          str++;
        } else
        if (*(str+1)=='x') {
          send_hex08(va_arg(args, int));
          str++;
        }
        if (*(str+1)=='X') {
          send_hex16(va_arg(args, int));
          str++;
        }
      }
    }
  }
}


void vuc_vsendf (u8 mode, u8 cmd, const char * fmt, va_list args)
{
  va_list args2;
  u8 len= 0;

  // unfortunately, we need the length in advance, but we don't want a buffer
  // so, we gotta loop over args twice.
  va_copy(args2, args);
  len= vuc_vsendf_cnt(fmt, args2);
  va_end(args);

#if (VUC_PACKET)
  if (mode&0x01)
    vuc_pkt_beg(len+1+1);
#endif
  vuc_c_send_8_8(cmd, len);

  va_copy(args2, args);
  vuc_vsendf_out(fmt, args2);
  va_end(args);

#if (VUC_PACKET)
  if (mode&0x01)
    vuc_pkt_end();
#endif
}

void vuc_sendf (u8 mode, u8 cmd, const char * fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  vuc_vsendf(mode, cmd, fmt, args);
  va_end(args);
}


#if (VUC_DBG_ON)

void dbgf (const char * fmt, ...)
{
  va_list args;

  if (in_dbgf) {
    nested_dbgf++;
    return;
  }
  in_dbgf= 1;
  va_start(args, fmt);
  vuc_vsendf(1, VUC_CMD_META|VUC_META_DBG, fmt, args);
  va_end(args);
  in_dbgf= 0;
}

#else /* VUC_DBG_ON */

#  define dbgf(str, ...) {}
#  define vuc_dbg_beg(len) {}
#  define vuc_dbg_end() {}
#  define vuc_dbgs(str) {}
#  define vuc_dbgn4(n) {}
#  define vuc_dbgn8(n) {}
#  define vuc_dbgn16(n) {}

#endif /* VUC_DBG_ON */


#if (VUC_PACKET)
// we have received a handshake-"meta"...
void vuc_in_handshake (u8 subcmd, u8 len, u8 * data)
{
  if (!vuc_out_pkt_wait && subcmd!=VUC_META_PKT_WTF)
#if (VUC_PKT_SEQ)
    { dbgf("err! redundant handshake %x in #%x.", subcmd, *data); return; }
#else
    { dbgf("err! redundant handshake %x.", subcmd); return; }
#endif
  VUC_OPT(PKT_SEQ,
    if (*data!=vuc_out_pkt_no)
      { dbgf("err! wrong handshake #. Got %x, want %x.", *data, vuc_out_pkt_no); return; }
  )
  if (subcmd==VUC_META_PKT_OK || subcmd==VUC_META_PKT_BAD) {
#if (VUC_PKT_SEQ)
    if (vuc_out_pkt_no!=*data)
      subcmd= VUC_META_PKT_BAD;
    else
    if (subcmd==VUC_META_PKT_OK)
      vuc_out_pkt_no++;
#endif
    vuc_in_pkt_code= subcmd;
    return;
  }
  dbgf("META %x", subcmd);
  return;
}
#endif

// we have received a "meta"-cmd...
u8 vuc_in_meta (u8 subcmd, u8 len, u8 * data)
{
  dbgf("META %x", subcmd);
  return 1;
}

void vuc_in_err (u8 code, u16 data)
{
  dbgf("ERR %x %X", code, data);
}


/* ------------------------------------------------------------------------- */

i8 gen_put_freq (u8 * data)
{
  u16 * buf16= (void *)data;

  // TODO: Wait for overflow or match
  cli();
  PWM_ICR= buf16[0];
  PWM_OCR= buf16[1];
  sei();

  dbgf("timer_freq: %X %X", buf16[0], buf16[1]);

  return 1;
}

i8 gen_put_power (u8 * data)
{
  PWM_TCCR_A= ((*data&1)<<PWM_COM_1);

  dbgf("gen_power: %x", *data);

  return 1;
}

i8 scope_put_tb (u8 * data)
{
  SCOPE_OCR= *data;

  dbgf("scope_tb: %x", *data);

  return 1;
}

i8 scope_get (u8 * data)
{
  dbgf(__FUNCTION__);

  return 1;
}

/* ------------------------------------------------------------------------- */

u16 int_cnt= 0;

ISR(ADC_vect)
{
  u8 b= ADCH;
  SCOPE_TIFR|= (1<<SCOPE_OCF);
  if (sc_state==0) {		// Wait for trigger
    if (sc_last<SCOPE_TRIGGER && b>=SCOPE_TRIGGER) {
      scope[sc_i++]= sc_last;
      scope[sc_i++]= b;
      sc_state++;		// SC_STATE_REC
    }
    sc_last= b;
  } else
  if (sc_state==SC_STATE_REC) { // Record
    scope[sc_i++]= b;
    if (sc_i>=SCOPE_SIZE) {
      sc_i= 0;
      sc_state++;		// SC_STATE_OUT = Ready
    }
  }
  int_cnt++;
}


i8 send_dbg (u8 pkt_no, void * data)
{
#if (VUC_TEST_PROTO==1)
  vuc_sendf(1, VUC_CMD_META|VUC_META_MSG, "resent:%X wait:%X  sc_state:%x diag_cnt:%X", re_cnt, wait_cnt, sc_state, diag_cnt);
#else
  vuc_sendf(1, VUC_CMD_META|VUC_META_MSG, "resent:%X wait:%X  sc_state:%x", re_cnt, wait_cnt, sc_state);
#endif
  return VUC_ST_PKG_WAIT_ACK;
}

void main_poll (void)
{
  static u32 del= DELVAL;
  static u8 blink= LEDVAL;

  if (!(del--)) {
    del= DELVAL;
    vuc_send_8(VUC_CMD_META|VUC_META_ALIVE);
    if (!(blink--)) {
      blink= LEDVAL;
      LED0_PORT_ADR^= (1<<LED0_BIT_NO);		// Toggle LED
      if (LED0_PORT_ADR&(1<<LED0_BIT_NO)) {
        vuc_pkt_q_send(2, 0, send_dbg, NULL);
//        dbgf("state:%x cnt:%X", sc_state, int_cnt);
      }
    }
  }
}


/* ------------------------------------------------------------------------- */

#define VUC_PUT_ADRS VUC_UC_PUT_ADRS
#define VUC_GET_ADRS VUC_UC_GET_ADRS
#include "../src/vuc_input.h"

void vuc_pkt_beg (u8 len)
{
#if (VUC_PACKET)
//  vuc_pkt_code= 
  vuc_chk= 0;
  vuc_send_8_8(VUC_CMD_META|VUC_META_PKT, len);
  VUC_OPT(PKT_SEQ, vuc_send_8(vuc_out_pkt_no);)
#endif
}

void vuc_pkt_end (void)
{
  VUC_OPT(PKT_CHK, vuc_send_8(vuc_chk);)
}


void vuc_in_pkt_start ()
{}

// send handshake to PC
void vuc_in_send_hs (i8 rep)
{
  if (rep) {
    vuc_out_pkt_code= rep<0 ? VUC_META_PKT_BAD : VUC_META_PKT_OK;
    vuc_send_8(VUC_CMD_META|vuc_out_pkt_code);
    VUC_OPT(PKT_SEQ, vuc_send_8(vuc_in_pkt_no);)
  }
}


i8 vuc_sendq_0 (u8 pkt_no, void * data)
{
  return VUC_ST_PKG_WAIT_ACK;
}

void vuc_poll ()
{
  u16 u;
#if (VUC_PACKET)
  vuc_pkt_ch_t * ch;
  ch= &vuc_pkt_channels[vuc_pkt_ch_no];
#endif

  main_poll();

  if (!deaf) {
    u= com_getc();
    if (!(u&0xff00))
      vuc_input(u&0xff);
  }

#if (VUC_PACKET)
  switch (ch->state) {
    case VUC_ST_PKG_CONTINUE:
    case VUC_ST_PKG_WAIT_ACK:		// wait for handshake
      if (!vuc_in_pkt_code) {
        if (!--wait_cnt) {
          wait_cnt= VUC_WAIT_DEL;
          vuc_send_8(VUC_CMD_META|VUC_META_PKT_WTF);
          VUC_OPT(PKT_SEQ, vuc_send_8(vuc_out_pkt_no);)
        }
        return;
      }
      if (vuc_in_pkt_code==VUC_META_PKT_OK) {
        ch->blk_no++;
      } else {				// must be VUC_META_PKT_BAD
        ch->state= 1;			// -> resend
#if (VUC_DBG_ON)
        re_cnt++;
#endif
      }
      vuc_out_pkt_wait= vuc_in_pkt_code= 0;
      ch->state= (ch->state+1)&3;	// next packet or end
      break;
    case 2:				// send packet
      if ((ch->state= ch->func(ch->blk_no, ch->data))) {
        vuc_out_pkt_wait= 1;
        wait_cnt= VUC_WAIT_DEL;
      }
      return;
  }
  if (++vuc_pkt_ch_no>=VUC_PKT_CHANNELS)
    vuc_pkt_ch_no= 0;
#endif
}

i8 vuc_pkt_q_send (u8 channel_no, u8 wait, vuc_pkt_f_t func, void * data)
{
#if (VUC_PACKET)
  vuc_pkt_ch_t * ch= &vuc_pkt_channels[channel_no];

  while (ch->state)
    if (wait)
      vuc_poll();		// wait for channel free
    else
      return -1;		// *so* sorry, pal...
  ch->state= 2;
  ch->blk_no= 0; //vuc_in_pkt_code= 0;
  ch->func= func;
  ch->data= data;
//  vuc_out_pkt_wait= 1;
#else
  func(0);
#endif
  return 0;
}


i8 vuc_send_hello (u8 pkt_no, void * data)
{
  vuc_pkt_beg(2); vuc_c_send_8_8(VUC_CMD_META|VUC_META_HELLO, VUC_PROTO_MAJOR<<4|VUC_PROTO_MINOR); vuc_pkt_end();
  return VUC_ST_PKG_WAIT_ACK;
}

void vuc_init ()
{
#if (VUC_PACKET)
  memset(vuc_pkt_channels, 0, sizeof(vuc_pkt_channels));
  vuc_pkt_channels[0].func= vuc_sendq_0;
#endif

  vuc_pkt_q_send(1, 1, vuc_send_hello, NULL);
}


i8 scope_send_pkt (u8 pkt_no, void * data)
{
  u8 v, x, x0= pkt_no*SCOPE_PKT_LEN;

  if (pkt_no>=SCOPE_PACKETS) {
#if (VUC_TEST_PROTO==1)
    diag_cnt++;
#endif
    sc_state= 0;			// all blocks sent and received - next round...
    return VUC_ST_PKG_DONE;
  }
  if (!pkt_no) {
    vuc_pkt_beg(1+SCOPE_SIZE-(u8)(pkt_no*SCOPE_PKT_LEN));
    vuc_put_beg(SCOPE_DATA_ADR);
  } else {
    vuc_pkt_beg(SCOPE_SIZE-(u8)(pkt_no*SCOPE_PKT_LEN));
  }
  for (x=x0; x<SCOPE_SIZE && x<(x0+SCOPE_PKT_LEN); x++) {
#if (VUC_TEST_PROTO==1)
    v= x ? x : diag_cnt;
#else
    v= scope[x];
#endif
    vuc_c_send_8(v);
  }
  vuc_pkt_end();
  return VUC_ST_PKG_CONTINUE;
}

void scope_poll ()
{
  if (sc_state<SC_STATE_OUT)
    return;				// wait for samples
  if (sc_state<SC_STATE_SENT) {
    if (vuc_pkt_q_send(1, 0, scope_send_pkt, NULL)>=0)
      sc_state++;
  }
}


char * id= "vuc-demo";
char * ver= "v" PACKAGE_VERSION;
char * proto= "p" VUC_PROTO_VER;

i8 send_id (u8 pkt_no, void * data)
{
  vuc_sendf(1, VUC_CMD_META|VUC_META_ID, id);
  return VUC_ST_PKG_WAIT_ACK;
}

i8 send_greeting (u8 pkt_no, void * data)
{
  vuc_sendf(1, VUC_CMD_META|VUC_META_MSG, "%s %s %s", id, ver, proto);
  return VUC_ST_PKG_WAIT_ACK;
}

int main (void)
{
  u16 w;

  PORTB= PORTC= PORTD= 0xff;			// activate all pull-ups
  SCOPE_PORT_ADR&= ~(1<<SCOPE_BIT_NO);		// except on ADC-Input

  DDRB= DDRC= DDRD= 0x00;			// all pins input
  LED0_DDR_ADR|= (1<<LED0_BIT_NO);		// LED-pin is output

  w= F_CPU/8/2/PWM_DEFAULT;
  PWM_ICR= w;
  PWM_OCR= w/4;					// 25% PW
  // Phase and Frequency Correct PWM, TOP=ICR
  PWM_TCCR_A= (1<<PWM_COM_1);			// Clear on Match when upcounting. Set on Match when downcounting.
  PWM_TCCR_B= (1<<PWM_WGM_3)|(1<<PWM_CS_1);	// Prescaler 8
  PWM_DDR_ADR|= (1<<PWM_BIT_NO);		// PWM-pin is output

  // ADC-trigger-timer
  SCOPE_TCCR_A= (1<<SCOPE_WGM_1);		// CTC Mode
  SCOPE_TCCR_B= SCOPE_PRESC_BITS;		// Set Prescaler
  SCOPE_OCR= F_CPU/SCOPE_PRESC_VAL/SCOPE_FREQ;	// Set Samplerate

  ADMUX= ADC_REF_BITS|ADC_IN_BITS|(1<<ADLAR);	// Ref select, input select, Result left-adjusted
  ADCSRA= ADC_PRESC_BITS|(1<<ADEN)|(1<<ADATE)|(1<<ADIE);// Enable, Auto Trigger Enable, Int when ready
  ADCSRB= (1<<ADTS1)|(1<<ADTS0);		// Auto-trigger on timer 0 compare A
  DIDR0= (1<<ADC_IN_BITS)&0x3f;			// Disable Digital Input

#ifdef COM_UART
  uart_init(UART_BAUD_SELECT_DOUBLE_SPEED(BAUDRATE, F_CPU));
#endif
#ifdef COM_UNIX
  unix_init(UNIX_SERVER_PATH);
#endif

  sei();

  vuc_init();
  vuc_pkt_q_send(1, 1, send_id, NULL);
  vuc_pkt_q_send(1, 1, send_greeting, NULL);

loop:
  vuc_poll();
  scope_poll();
  main_poll();
  goto loop;

  uart_exit();
  return 0;
}

// eof
