/*
* eia-232
* by themroc 2013-11-13
* this version always returns -1 in case of error
* fork of rs232 by:
*
***************************************************************************
*
* Author: Teunis van Beelen
*
* Copyright (C) 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013 Teunis van Beelen
*
* teuniz@gmail.com
*
***************************************************************************
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation version 2 of the License.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
***************************************************************************
*
* This version of GPL is at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
*
***************************************************************************
*/

/* last revision: February 1, 2013 */

/* For more info and how to use this libray, visit: http://www.teuniz.net/RS-232/ */

#include "eia-232.h"


#ifdef __linux__   /* Linux */

int error;

/*
 com_name:
   something like "/dev/ttyS0", "/dev/ttyUSB0" or "/dev/ttyAMA0", "/dev/rfcomm0"
   find out with
     grep serial /proc/tty/drivers
 */
int EIA232_open (char * com_name, int baudrate, EIA232_HANDLE * com_handle)
{
  int baudr, status;
  struct termios port_settings;
  EIA232_HANDLE my_handle;

  switch(baudrate)
  {
    case      50 : baudr = B50;
                   break;
    case      75 : baudr = B75;
                   break;
    case     110 : baudr = B110;
                   break;
    case     134 : baudr = B134;
                   break;
    case     150 : baudr = B150;
                   break;
    case     200 : baudr = B200;
                   break;
    case     300 : baudr = B300;
                   break;
    case     600 : baudr = B600;
                   break;
    case    1200 : baudr = B1200;
                   break;
    case    1800 : baudr = B1800;
                   break;
    case    2400 : baudr = B2400;
                   break;
    case    4800 : baudr = B4800;
                   break;
    case    9600 : baudr = B9600;
                   break;
    case   19200 : baudr = B19200;
                   break;
    case   38400 : baudr = B38400;
                   break;
    case   57600 : baudr = B57600;
                   break;
    case  115200 : baudr = B115200;
                   break;
    case  230400 : baudr = B230400;
                   break;
    case  460800 : baudr = B460800;
                   break;
    case  500000 : baudr = B500000;
                   break;
    case  576000 : baudr = B576000;
                   break;
    case  921600 : baudr = B921600;
                   break;
    case 1000000 : baudr = B1000000;
                   break;
    case 1152000 : baudr = B1152000;
                   break;
    case 1500000 : baudr = B1500000;
                   break;
    case 2000000 : baudr = B2000000;
                   break;
    case 2500000 : baudr = B2500000;
                   break;
    case 3000000 : baudr = B3000000;
                   break;
    case 3500000 : baudr = B3500000;
                   break;
    case 4000000 : baudr = B4000000;
                   break;
    default      : printf("invalid baudrate\n");
                   return(-1);
  }

  my_handle = open(com_name, O_RDWR | O_NOCTTY | O_NDELAY);
  if(my_handle<0) {
    perror("unable to open comport");
    return(-1);
  }

  memset(&port_settings, 0, sizeof(port_settings));  /* clear the new struct */

  port_settings.c_cflag = baudr | CS8 | CLOCAL | CREAD;
  port_settings.c_iflag = IGNPAR;
  port_settings.c_oflag = 0;
  port_settings.c_lflag = 0;
  port_settings.c_cc[VMIN] = 0;      /* block untill n bytes are received */
  port_settings.c_cc[VTIME] = 0;     /* block untill a timer expires (n * 100 mSec.) */
  error = tcsetattr(my_handle, TCSANOW, &port_settings);
  if(error==-1) {
    close(my_handle);
    perror("unable to adjust portsettings ");
    return(-1);
  }

  if(ioctl(my_handle, TIOCMGET, &status) == -1) {
    perror("unable to get portstatus");
    return(-1);
  }

  status |= TIOCM_DTR;    /* turn on DTR */
  status |= TIOCM_RTS;    /* turn on RTS */

  if(ioctl(my_handle, TIOCMSET, &status) == -1) {
    perror("unable to set portstatus");
    return(-1);
  }

  *com_handle= my_handle;
  return 0;
}

void EIA232_close (EIA232_HANDLE com_handle)
{
  int status;

  if(ioctl(com_handle, TIOCMGET, &status) == -1)
    perror("unable to get portstatus");

  status &= ~TIOCM_DTR;    /* turn off DTR */
  status &= ~TIOCM_RTS;    /* turn off RTS */

  if(ioctl(com_handle, TIOCMSET, &status) == -1)
    perror("unable to set portstatus");

  close(com_handle);
}


int EIA232_poll (EIA232_HANDLE com_handle, void * buf, int size)
{
#ifndef __STRICT_ANSI__                       /* __STRICT_ANSI__ is defined when the -ansi option is used for gcc */
  if(size>SSIZE_MAX)
    size = (int)SSIZE_MAX;  /* SSIZE_MAX is defined in limits.h */
#else
  if(size>4096)
    size = 4096;
#endif

  return read(com_handle, buf, size);
}


int EIA232_send_byte (EIA232_HANDLE com_handle, char byte)
{
  return write(com_handle, &byte, 1);
}


int EIA232_send_buf (EIA232_HANDLE com_handle, void * buf, int size)
{
  return write(com_handle, buf, size);
}


/*
Constant  Description
TIOCM_LE  DSR (data set ready/line enable)
TIOCM_DTR DTR (data terminal ready)
TIOCM_RTS RTS (request to send)
TIOCM_ST  Secondary TXD (transmit)
TIOCM_SR  Secondary RXD (receive)
TIOCM_CTS CTS (clear to send)
TIOCM_CAR DCD (data carrier detect)
TIOCM_CD  Synonym for TIOCM_CAR
TIOCM_RNG RNG (ring)
TIOCM_RI  Synonym for TIOCM_RNG
TIOCM_DSR DSR (data set ready)
*/

int EIA232_get_CTS (EIA232_HANDLE com_handle)
{
  int status;

  ioctl(com_handle, TIOCMGET, &status);

  if(status&TIOCM_CTS)
    return(1);
  else
    return(0);
}

int EIA232_get_DSR (EIA232_HANDLE com_handle)
{
  int status;

  ioctl(com_handle, TIOCMGET, &status);

  if(status&TIOCM_DSR)
    return(1);
  else
    return(0);
}

void EIA232_set_DTR (EIA232_HANDLE com_handle, int on)
{
  int status;

  if(ioctl(com_handle, TIOCMGET, &status) == -1)
    perror("unable to get portstatus");

  status= on
    ? status | TIOCM_DTR     /* turn on DTR */
    : status & ~TIOCM_DTR;   /* turn off DTR */

  if(ioctl(com_handle, TIOCMSET, &status) == -1)
    perror("unable to set portstatus");
}

void EIA232_set_RTS (EIA232_HANDLE com_handle, int on)
{
  int status;

  if(ioctl(com_handle, TIOCMGET, &status) == -1)
    perror("unable to get portstatus");

  status= on
    ? status | TIOCM_RTS     /* turn on RTS */
    : status & ~TIOCM_RTS;   /* turn off RTS */

  if(ioctl(com_handle, TIOCMSET, &status) == -1)
    perror("unable to set portstatus");
}


#else         /* windows */

/*
 com_name:
   something like "\\\\.\\COM1" or "\\\\.\\COM16"
 */
int EIA232_open (char * com_name, int baudrate, EIA232_HANDLE * com_handle)
{
  EIA232_HANDLE my_handle;
  char baudr[64];

  snprintf(baudr, 64, "baud=%d data=8 parity=N stop=1 dtr=on rts=on", baudrate);
  my_handle = CreateFileA(com_name,
                      GENERIC_READ|GENERIC_WRITE,
                      0,                          /* no share  */
                      NULL,                       /* no security */
                      OPEN_EXISTING,
                      0,                          /* no threads */
                      NULL);                      /* no templates */

  if(my_handle==INVALID_HANDLE_VALUE) {
    printf("unable to open comport\n");
    return(-1);
  }

  DCB port_settings;
  memset(&port_settings, 0, sizeof(port_settings));  /* clear the new struct  */
  port_settings.DCBlength = sizeof(port_settings);

  if(!BuildCommDCBA(baudr, &port_settings)) {
    printf("unable to set comport dcb settings\n");
    CloseHandle(my_handle);
    return(-1);
  }

  if(!SetCommState(my_handle, &port_settings)) {
    printf("unable to set comport cfg settings\n");
    CloseHandle(my_handle);
    return(-1);
  }

  COMMTIMEOUTS Cptimeouts;

  Cptimeouts.ReadIntervalTimeout         = MAXDWORD;
  Cptimeouts.ReadTotalTimeoutMultiplier  = 0;
  Cptimeouts.ReadTotalTimeoutConstant    = 0;
  Cptimeouts.WriteTotalTimeoutMultiplier = 0;
  Cptimeouts.WriteTotalTimeoutConstant   = 0;

  if(!SetCommTimeouts(my_handle, &Cptimeouts))
  {
    printf("unable to set comport time-out settings\n");
    CloseHandle(my_handle);
    return(-1);
  }

  *com_handle= my_handle;
  return 0;
}

void EIA232_close (EIA232_HANDLE com_handle)
{
  CloseHandle(com_handle);
}


int EIA232_poll (EIA232_HANDLE com_handle, void * buf, int size)
{
  int n;

  if(size>4096)
    size = 4096;

/* added the void pointer cast, otherwise gcc will complain about */
/* "warning: dereferencing type-punned pointer will break strict aliasing rules" */

  ReadFile(com_handle, buf, size, (LPDWORD)((void *)&n), NULL);

  return(n);
}


int EIA232_send_byte (EIA232_HANDLE com_handle, char byte)
{
  int n;

  WriteFile(com_handle, &byte, 1, (LPDWORD)((void *)&n), NULL);

  return n;
}

int EIA232_send_buf (EIA232_HANDLE com_handle, void * buf, int size)
{
  int n;

  WriteFile(com_handle, buf, size, (LPDWORD)((void *)&n), NULL);

  return(n);
}


int EIA232_get_CTS (EIA232_HANDLE com_handle)
{
  int status;

  GetCommModemStatus(com_handle, (LPDWORD)((void *)&status));

  if(status&MS_CTS_ON)
    return(1);
  else
    return(0);
}

int EIA232_get_DSR (EIA232_HANDLE com_handle)
{
  int status;

  GetCommModemStatus(com_handle, (LPDWORD)((void *)&status));

  if(status&MS_DSR_ON)
    return(1);
  else
    return(0);
}


void EIA232_set_DTR (EIA232_HANDLE com_handle, int on)
{
  if (on)
    EscapeCommFunction(com_handle, SETDTR);
  else
    EscapeCommFunction(com_handle, CLRDTR);
}

void EIA232_set_RTS (EIA232_HANDLE com_handle, int on)
{
  if (on)
    EscapeCommFunction(com_handle, SETRTS);
  else
    EscapeCommFunction(com_handle, CLRRTS);
}


#endif


void EIA232_send_string (EIA232_HANDLE com_handle, const char *text)  /* sends a string to serial port */
{
  while(*text != 0)
    EIA232_send_byte(com_handle, *(text++));
}

