
#define DBG_GUI		0

#define VUC_DBG_ON	1
#define VUC_DBG_PROTO	1
#define VUC_DBG_IN	1
#define VUC_DBG_OUT	1

#define SHOW_TIMESTAMP	0

#define VUC_IS_HOST	1


#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>

#ifdef _WIN32

#include <windows.h>

#else /* _WIN32 */

#include <unistd.h>

#endif /* _WIN32 */

#include "anygtk.h"
//#include "host.h"
//#include "vuc.h"
//#include "vuc_types.h"

#include "libvuc.h"

#if (COM_SER)
#  include "eia-232.h"
#endif
#if (COM_UDP)
#  include "udp.h"
#endif
#if (COM_UNIX)
#  include "unix.h"
#endif


guint vuc_proto_opts= 0xffff;		// default: use all compiled-in options
#define VUC_OPT_CHK_PACKET	(vuc_proto_opts & VUC_OPT_FLG_PACKET)
#define VUC_OPT_CHK_PKT_CHK	(vuc_proto_opts & VUC_OPT_FLG_PKT_CHK)
#define VUC_OPT_CHK_PKT_SEQ	(vuc_proto_opts & VUC_OPT_FLG_PKT_SEQ)


static gint verbosity= 2;

static gint vuc_err_cnt[256];
static gint vuc_alive_cnt= 0;
#if (VUC_PACKET)
static gint vuc_in_chk_cnt= 0;
static gint vuc_out_chk_cnt= 0;
static gint vuc_line_err_cnt= 0;
#endif
static gint vuc_proto_err_cnt= 0;

#if (VUC_PKT_SEQ)
static gint vuc_is_warm= 0;		// 0=Let uC initialize vuc_in_pkt_no
u8 vuc_in_pkt_no= 0;			// Expected packet no
static u8 vuc_in_pkt_hs_no= 0;		// Last packet no
static u8 vuc_out_pkt_no;
#else
#  define vuc_in_pkt_no		0
#  define vuc_out_pkt_no	0
#endif
#if (VUC_PACKET)
static u8 vuc_in_pkt_hs_code= VUC_META_PKT_BAD;// Last handshake sent
static u8 vuc_out_pkt_wait= 0;			// Packet sent - waiting for handshake
#endif

static GAsyncQueue * send_q;


static GTimeVal start;

static gint vuc_put_cnt;
static vuc_put_data_t * vuc_put_data;

static gint vuc_get_max;
static vuc_get_data_t * vuc_get_data;


#if (VUC_DBG_ON)
// debugging generally on
#  define dbgf(lvl, ...)	msgf(10+lvl, __VA_ARGS__)
#else
#  define dbgf(lvl, ...)	
#endif

#if (VUC_DBG_ON)
// serious stuff
#  define dbgf0(lvl, ...)	dbgf(lvl, __VA_ARGS__)
#else
#  define dbgf0(lvl, ...)	
#endif

#if (VUC_DBG_IN)
// basic i/o
#  define dbgf1(lvl, ...)	dbgf(lvl, __VA_ARGS__)
#else
#  define dbgf1(lvl, ...)	
#endif

#if (VUC_DBG_OUT)
// basic i/o
#  define dbgf2(lvl, ...)	dbgf(lvl, __VA_ARGS__)
#else
#  define dbgf2(lvl, ...)	
#endif

#if (VUC_DBG_PROTO)
// protocol
#  define dbgf3(lvl, ...)	dbgf(lvl, __VA_ARGS__)
#  define VUC_PROTO_DBGF(...)	dbgf(2, __VA_ARGS__)
#else
#  define dbgf3(lvl, ...)	
#  define VUC_PROTO_DBGF(...)	
#endif



static GtkBuilder * builder;

static GtkEntry * gui_alive_cnt= NULL;
#if (VUC_PACKET)
static GtkEntry * gui_in_chk_cnt= NULL;
static GtkEntry * gui_out_chk_cnt= NULL;
static GtkEntry * gui_line_err_cnt= NULL;
#endif
static GtkEntry * gui_proto_err_cnt= NULL;
static GtkEntry * gui_status_msg= NULL;


static gboolean vuc_io_timeout (gpointer data);


/*****************************************************************************/

static void vuc_in_pkt_start();
// send handshake to uC
static void vuc_in_send_hs (i8 rep);


static int msgfv (gint lvl, const char * fmt, va_list args)
{
  if (lvl>verbosity)
    return;

#if (SHOW_TIMESTAMP)
  i32 now= my_time();
  printf("%d.%03d ", now/1000, now%1000);
#endif

  vprintf(fmt, args);
  printf("\n");
}

static int msgf (gint lvl, const char * fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  msgfv(lvl, fmt, args);
  va_end(args);
}

static int infof (const char * fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  msgfv(1, fmt, args);
  va_end(args);
}

static int errf (const char * fmt, ...)
{
  va_list args;

#if (SHOW_TIMESTAMP)
  i32 now= my_time();
  printf("%d.%03d ", now/1000, now%1000);
#endif

  printf("ERROR! ");

  va_start(args, fmt);
  vprintf(fmt, args);
  printf("\n");
  va_end(args);
}

// returns no of milliseconds this program is running (assuming it was started
// when tv_usec was 0. If not, there's a constant offset below 1 second)
static i32 my_time ()
{
  GTimeVal now;

  g_get_current_time(&now);
  return (now.tv_sec-start.tv_sec)*1000 + (now.tv_usec/1000);
}


#if (COM_SER)

EIA232_HANDLE ser_handle;

static gint if_ser_init (gchar * com_name, gint baudrate)
{
  if (EIA232_open(com_name, baudrate, &ser_handle)<0) {
    return 0;
  }
  g_timeout_add(5, vuc_io_timeout, NULL);
  return 1;
}

static gsize if_ser_receive (gpointer buffer, gint buf_size)
{
  return EIA232_poll(ser_handle, buffer, buf_size);
}

static gint if_ser_send_8 (u8 data)
{
  EIA232_send_byte(ser_handle, data);
  return 1;
}

static gint if_ser_send_s (u8 * data, int len)
{
  EIA232_send_buf(ser_handle, data, len);
  return 1;
}

#endif /* COM_SER */


#if (COM_UDP)

static gsize if_udp_receive (gpointer buffer, gint buf_size)
{
  return udp_gets(buffer, buf_size);
}

static gint if_udp_send_8 (u8 data)
{
  udp_putc(data);
  return 1;
}

static gint if_udp_send_s (u8 * data, int len)
{
  udp_puts(data, len);
  return 1;
}

#endif /* COM_UDP */


#if (COM_UNIX)

static gsize if_unix_receive (gpointer buffer, gint buf_size)
{
  return unix_gets(buffer, buf_size);
}

static gint if_unix_send_8 (u8 data)
{
  unix_putc(data);
  return 1;
}

static gint if_unix_send_s (u8 * data, int len)
{
  unix_puts(data, len);
  return 1;
}

#endif /* COM_UNIX */


typedef gsize (* receive_f)(gpointer buffer, gint buf_size);
typedef gint (* send_8_f)(u8 data);
typedef gint (* send_s_f)(u8 * data, int len);

receive_f if_receive;
send_8_f if_send_8;
send_s_f if_send_s;

#if (COM_UDP)
#  define COM_UDP_OPT_STR	", 'udp' for internet"
#else
#  define COM_UDP_OPT_STR	""
#endif

#if (COM_UNIX)
#  define COM_UNIX_OPT_STR	", 'unix' for unix domain socket"
#else
#  define COM_UNIX_OPT_STR	""
#endif

/*
 return: -1=exit 0=ok 1..10=error
*/
static void if_init
  (gint default_rate, gsize xml_len, const gchar * xml_data, int argc, char * argv[])
{
  gchar ** remain= NULL;
  gchar * if_type= NULL;
  gboolean show_config= FALSE;
  gboolean dump_xml= FALSE;
  GOptionEntry entries[] = {
    { "show-cfg", 's', 0, G_OPTION_ARG_NONE, &show_config, "Show configuration", NULL },
    { "dump-xml", 'x', 0, G_OPTION_ARG_NONE, &dump_xml, "Print glade xml", NULL },
    { "if-type", 't', 0, G_OPTION_ARG_STRING, &if_type, "Interface type: 'ser' for RS232 (default)" COM_UDP_OPT_STR COM_UNIX_OPT_STR, "TYPE" },
    { "verbosity", 'V', 0, G_OPTION_ARG_INT, &verbosity, "Set verbosity (0:silent .. 2:normal .. 10+:debug)", "LEVEL" },
    { G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_STRING_ARRAY, &remain, NULL, NULL },
    { NULL }
  };
  GError * error= NULL;
  GOptionContext * context;

  context= g_option_context_new("[ [--] interface parameters]");
  g_option_context_add_main_entries(context, entries, NULL);
  g_option_context_add_group(context, gtk_get_option_group(TRUE));
  if (!g_option_context_parse (context, &argc, &argv, &error)) {
    infof("option parsing failed: %s\n", error->message);
    exit(99);
  }
  g_option_context_free(context);

  if (show_config) {
    infof("Configuration:");
    infof("Option	 compiled in  active");
    infof("----------------------------");
    infof("VUC_PACKET	%d	%d", VUC_PACKET, VUC_OPT_CHK_PACKET ? 1 : 0);
    infof("VUC_PKT_CHK	%d	%d", VUC_PKT_CHK, VUC_OPT_CHK_PKT_CHK ? 1 : 0);
    infof("VUC_PKT_SEQ	%d	%d", VUC_PKT_SEQ, VUC_OPT_CHK_PKT_SEQ ? 1 : 0);
    exit(0);
  }

  if (dump_xml) {
    infof(xml_data);
    exit(0);
  }

  if (!if_type || strcasecmp(if_type, "ser")==0) {
    gint bd= 0;
    if (!remain || !remain[0]) {
      errf("Need serial port name.");
      exit(3);
    }
    if (remain[1])
      bd= atoi(remain[1]);
    if (!bd)
      bd= default_rate;
    infof("Opening serial port '%s' with %d baud.", remain[0], bd);
    if (if_ser_init(remain[0], bd)>0) {
      if_receive= if_ser_receive;
      if_send_8= if_ser_send_8;
      if_send_s= if_ser_send_s;
      return;
    } else
      exit(4);
  }

#if (COM_UDP)
  if (strcasecmp(if_type, "udp")==0) {
    if (!remain || !remain[0]) {
      errf("Need hostname or IP.");
      exit(3);
    }
    gchar * server= remain[0];
    gchar * port= remain[1] ? remain[1] : G_STRINGIFY(UDP_DEFAULT_PORT);
    infof("Opening server '%s' port '%s'.", server, port);
    if (udp_init(server, port)>0) {
      if_receive= if_udp_receive;
      if_send_8= if_udp_send_8;
      if_send_s= if_udp_send_s;
      g_timeout_add(5, vuc_io_timeout, NULL);
      return;
    } else
      exit(4);
  }
#endif

#if (COM_UNIX)
  if (strcasecmp(if_type, "unix")==0) {
    gchar * socket= remain[0] ? remain[0] : UNIX_SERVER_PATH;
    infof("Opening socket '%s'.", socket);
    if (unix_init(socket)>0) {
      if_receive= if_unix_receive;
      if_send_8= if_unix_send_8;
      if_send_s= if_unix_send_s;
      g_timeout_add(5, vuc_io_timeout, NULL);
      return;
    } else
      exit(4);
  }
#endif

  infof("Bad port type.");
  exit(2);
}

static gint if_send_16 (u16 data)
{
  u16 ve_data= VUC_U16_TO_VE(data);

//  if_send_8(ve_data&0xff); if_send_8(ve_data>>8);
  if_send_s((void *)&ve_data, 2);
  return 1;
}


/*** VUC *********************************************************************/

typedef struct {
  gint pkt, start, end;
  u8 buf[];
} buf_send;


static void vuc_send_buf (buf_send * b);


static void vuc_send_8 (u8 data)
{
  if_send_8(data);
}

static void vuc_send_8_8 (u8 data1, u8 data2)
{
  u16 d= data1 | (data2<<8);

//  if_send_8(data1); if_send_8(data2);
  if_send_s((void *)&d, 2);
}


#if (VUC_PACKET)
static int vuc_meta_hs (u8 hs, u8 len, u8 * data)
{
}
#endif

static u8 vuc_in_handshake (u8 subcmd, u8 len, u8 * data)
{
  buf_send * b;

  dbgf3(1,
#if (VUC_PKT_SEQ)
    "meta pkt #%x %s", *data,
#else
    "meta pkt %s",
#endif
    subcmd==VUC_META_PKT_WTF ? "wtf" : subcmd==VUC_META_PKT_OK ? "ok" : "bad"
  );
  if (subcmd==VUC_META_PKT_WTF) {
#if (VUC_PKT_SEQ)
    if (*data!=vuc_in_pkt_hs_no) {
      // Some serious error occured
      // TODO: look into int, reset or do something clever
      // ... but for now just continue with the pkt.no. the uC thinks is right
      vuc_in_pkt_hs_code= VUC_META_PKT_BAD;
      vuc_in_pkt_hs_no= vuc_in_pkt_no= *data;
      dbgf3(0, "pkt # forced to %x", vuc_in_pkt_no);
    }
#endif
    // uC missed last handshake - just repeat it
#if (VUC_PKT_SEQ)
    vuc_send_8_8(VUC_CMD_META|vuc_in_pkt_hs_code, vuc_in_pkt_hs_no);
#else
    vuc_send_8(VUC_CMD_META|vuc_in_pkt_hs_code);
#endif
    return;
  }
  if (subcmd!=VUC_META_PKT_WTF && !vuc_out_pkt_wait)
#if (VUC_PKT_SEQ)
    return errf("redundant handshake %x in #%x.", subcmd, *data);
#else
    return errf("redundant handshake %x.", subcmd);
#endif
  VUC_OPT(PKT_SEQ,
    if (subcmd!=VUC_META_PKT_WTF && *data!=vuc_out_pkt_no)
      return errf("wrong handshake #. Got %x, want %x.", *data, vuc_out_pkt_no);
  )
  if (subcmd==VUC_META_PKT_OK) {
    vuc_out_pkt_wait= 0;
    VUC_OPT(PKT_SEQ, vuc_out_pkt_no++;)
//    dbgf3(1, "Next out-pkt #%x", vuc_out_pkt_no);
    if ((b= g_async_queue_try_pop(send_q)))
      vuc_send_buf(b);
  }
#if (VUC_PKT_SEQ)
  else
    vuc_out_chk_cnt++;
#endif

  return 0;
}

static void print_meta (gint lvl, char * type, u8 len, u8 * data)
{
  char * buf= g_alloca(len+1);

  g_strlcpy(buf, data, len+1);
  msgf(lvl, "meta %s: %s", type, buf);
//  fwrite(data, len, 1, stdout);
//  dbgf3("\n");
}

static u8 vuc_in_meta (u8 subcmd, u8 len, u8 * data)
{
  if (subcmd==VUC_META_ALIVE)
    { vuc_alive_cnt++; return 0; }

  else if (subcmd==VUC_META_EXT)
    print_meta(2, "ext", len, data);
  else if (subcmd==VUC_META_MSG) {
    print_meta(1, "msg", len, data);
  }
  else if (subcmd==VUC_META_DBG)
    { print_meta(10, "dbg", len, data); return 0; }

  else if (subcmd==VUC_META_HELLO)
    msgf(1, "meta hello: %d.%d", data[0]>>4, data[0]&0x0f);
  else if (subcmd==VUC_META_ID)
    print_meta(1, "id", len, data);

  else
    print_meta(0, "???", len, data);

  return 1;
}

static void vuc_in_err (u8 code, u16 data)
{
  gchar text[60];

  vuc_err_cnt[code]++;

  if (code==VUC_ERR_PKT_HDR || code==VUC_ERR_PKG_BAD) {
    vuc_line_err_cnt++;
  } else {
    vuc_proto_err_cnt++;
  }

  if (code==VUC_ERR_PKT_HDR) {
    dbgf0(0, "err pkt header: %x", data&0xff);
  }
}


#define VUC_IN_DATA_SIZE 256

#define VUC_PUT_ADRS vuc_put_cnt
#define VUC_GET_ADRS vuc_get_max

u8 vuc_out_on= 1;
#include "vuc_input.h"


// a packet-header just came in
static void vuc_in_pkt_start()
{
//  pkt_start= my_time();
}

// send handshake to uC
static void vuc_in_send_hs (i8 rep)
{
  if (rep) {
    vuc_in_pkt_hs_code= rep<0 ? VUC_META_PKT_BAD : VUC_META_PKT_OK;
    vuc_in_pkt_hs_no= vuc_in_pkt_no;
#if (VUC_PKT_SEQ)
    vuc_send_8_8(VUC_CMD_META|vuc_in_pkt_hs_code, vuc_in_pkt_hs_no);
#else
    vuc_send_8(VUC_CMD_META|vuc_in_pkt_hs_code);
#endif
#if (VUC_DBG_OUT)
#  if (VUC_PKT_SEQ)
    dbgf2(3, "OUT: %02x %02x", VUC_CMD_META|vuc_in_pkt_hs_code, vuc_in_pkt_no);
#  else
    dbgf2(3, "OUT: %02x", VUC_CMD_META|vuc_in_pkt_hs_code);
#  endif
#endif
#if (VUC_PKT_CHK)
    if (rep<0)
      vuc_in_chk_cnt++;
#endif
  }
}


static void vuc_send_buf (buf_send * b)
{
#if (VUC_PACKET)
  gint i, j= 0;
#if (VUC_PKT_CHK)
  u8 chk= 0;
#endif
#if (VUC_DBG_OUT)
  GString * dbg_hex= g_string_new(NULL);
#endif

  b->pkt= 1;
  VUC_OPT(PACKET,
    vuc_out_pkt_wait= 1;
    b->buf[j++]= VUC_CMD_META|VUC_META_PKT;
    b->buf[j++]= b->end-b->start;
  )
  VUC_OPT(PKT_SEQ, b->buf[j++]= vuc_out_pkt_no;)
#if (VUC_PKT_CHK)
  for (i=b->start; i<b->end; i++)
    chk+= b->buf[i];
  b->buf[b->end++]= chk;
#endif
  b->start= 0;
#endif

  if_send_s(&b->buf[b->start], b->end-b->start);
  for (i=b->start; i<b->end; i++) {
//    if_send_8(b->buf[i]);
#if (VUC_DBG_OUT)
    g_string_append_printf(dbg_hex, " %02x", b->buf[i]);
#endif
  }
#if (VUC_DBG_OUT)
  dbgf2(3, "OUT:%s", dbg_hex->str); g_string_free(dbg_hex, 1);
#endif
}

static void vuc_send_pkt (buf_send * b)
{
  if (vuc_out_pkt_wait)
    g_async_queue_push(send_q, b);
  else
    vuc_send_buf(b);
}

/* Send some 8-bit-values to uC */
void vuc_put_8 (guint channel, guint count, ...)
{
  va_list vals;
  gint v, i, j= 2;
  buf_send * b= g_malloc(sizeof(buf_send)+4+1+count);

  VUC_OPT(PKT_SEQ, j++;)
  b->pkt= 0; b->start= j; b->end= j+1+count;
#if (VUC_DBG_OUT)
  GString * dbg_hex= g_string_new(NULL);
#endif

  // TODO: channel-numbers above 64
  b->buf[j++]= VUC_CMD_PUT|channel;
#if (VUC_DBG_OUT)
  g_string_append_printf(dbg_hex, "%02x =", VUC_CMD_PUT|channel);
#endif
  va_start(vals, count);
  for (i=0; i<count; i++) {
    v= va_arg(vals, int);
    b->buf[j++]= v;
#if (VUC_DBG_OUT)
    g_string_append_printf(dbg_hex, " %02x", (u8)v);
#endif
  }
  va_end(vals);

#if (VUC_DBG_OUT)
  dbgf3(0, "PUT: %s", dbg_hex->str); g_string_free(dbg_hex, 1);
#endif
  vuc_send_pkt(b);
}

/* Send some 16-bit-values to uC */
void vuc_put_16 (guint channel, guint count, ...)
{
  va_list vals;
  gint v, i, j= 2;
  buf_send * b= g_malloc(sizeof(buf_send)+4+1+count*2);

  VUC_OPT(PKT_SEQ, j++;)
  b->pkt= FALSE; b->start= j; b->end= j+1+count*2;
#if (VUC_DBG_OUT)
  GString * dbg_hex= g_string_new(NULL);
#endif

  // TODO: channel-numbers above 64
  b->buf[j++]= VUC_CMD_PUT|channel;
#if (VUC_DBG_OUT)
  g_string_append_printf(dbg_hex, "%02x =", VUC_CMD_PUT|channel);
#endif
  va_start(vals, count);
  for (i=0; i<count; i++) {
    v= va_arg(vals, int);
    b->buf[j++]= v&0xff;
    b->buf[j++]= v>>8;
#if (VUC_DBG_OUT)
    g_string_append_printf(dbg_hex, " %04x", v);
#endif
  }
  va_end(vals);

#if (VUC_DBG_OUT)
  dbgf3(0, "PUT: %s", dbg_hex->str); g_string_free(dbg_hex, 1);
#endif
  vuc_send_pkt(b);
}

#define IO_IN_BUF_SIZE 1024

static gint vuc_io_poll ()
{
  gint size, read= 0;
  gchar io_in_buf[IO_IN_BUF_SIZE];
  guchar c;
#if (VUC_DBG_IN)
  GString * dbg_hex= g_string_new(NULL);
  GString * dbg_asc= g_string_new(NULL);
#endif

//  GIOStatus
  size= if_receive(io_in_buf, IO_IN_BUF_SIZE);
  if (size<=0)
    return;
  while (read<size) {
    c= io_in_buf[read];
    vuc_input(io_in_buf[read++]);
#if (VUC_DBG_IN)
    g_string_append_printf(dbg_hex, " %02x", c);
    g_string_append_printf(dbg_asc, "%c", (c<0x20 || c>=0x80) ? '.' : c);
#endif
  }
#if (VUC_DBG_IN)
  dbgf1(3, "IN: %s  \t%s", dbg_hex->str, dbg_asc->str);
  g_string_free(dbg_hex, 1); g_string_free(dbg_asc, 1);
#endif
}

static gboolean vuc_io_timeout (gpointer data)
{
  vuc_io_poll();
  return 1;
}


/* Send "meta"-cmd w/o parameter */
static void vuc_meta_0 (u8 meta)
{
  meta|= VUC_CMD_META;
  if_send_8(meta);

#if (VUC_DBG_OUT)
  dbgf2(3, "OUT: %02x", meta);
#endif
}



/*** Scope *******************************************************************/


/*** /Scope ******************************************************************/


/*** GUI *********************************************************************/

void on_window1_destroy (GtkWidget * widget, gpointer udata)
{
  gtk_main_quit();
}


static void show_counter (gint * cnt, gint * last, gint max, gchar * fmt, GtkEntry * entry)
{
  gchar text[10];

  if (entry && *last!=*cnt) {
    if (*cnt>max)
      *cnt= 0;
    snprintf(text, 10, fmt, *cnt);
    gtk_entry_set_text(entry, text);
    *last= *cnt;
  }
}


static gboolean alive_bg (gpointer data)
{
  vuc_send_8(VUC_META_ALIVE);
}

static gboolean counter_bg (gpointer data)
{
  static gint last_alive= 0;
  static gint last_line_err= 0;
  static gint last_proto_err= 0;
  static gint last_scope_diag_ok= 0;
  static gint last_scope_diag_err= 0;
  gchar text[32];

  show_counter(&vuc_alive_cnt, &last_alive, 999, "%03d", gui_alive_cnt);
#if (VUC_PACKET)
  show_counter(&vuc_in_chk_cnt, &last_line_err, 999, "%03d", gui_in_chk_cnt);
  show_counter(&vuc_out_chk_cnt, &last_line_err, 999, "%03d", gui_out_chk_cnt);
  show_counter(&vuc_line_err_cnt, &last_line_err, 9999, "%04d", gui_line_err_cnt);
#endif
  show_counter(&vuc_proto_err_cnt, &last_proto_err, 9999, "%04d", gui_proto_err_cnt);

/*
  gpointer p= g_async_queue_try_pop(console_q);
  if (p) {
    fprintf(stderr, "%s\n", (char *)p);
    g_free(p);
  }
*/
  return 1;
}

GTimeVal start_time;

GtkWindow * window1;

/*
 return: -1=exit 0=ok 1..10=error
*/
GtkBuilder * vuc_init (
  gint default_rate, 
  gsize xml_len, const gchar * xml_data,
  int argc, char * argv[]
) {
  gint i;
  GError * error;

  GtkScrolledWindow * dbg_scr;
  GtkAdjustment * dbg_adj;
  GtkTextView * dbg_buf;

  GThread * send_th;

  GtkStyle * style;

  g_get_current_time(&start);

  gtk_init(&argc, &argv);

  if_init(default_rate, xml_len, xml_data, argc, argv);

  builder= gtk_builder_new();
  if (!gtk_builder_add_from_string(builder, xml_data, xml_len, NULL)) {
    infof("Need input!");
    exit(1);
  }

  vuc_put_cnt= 0; vuc_put_data= NULL;
  vuc_get_max= 0; vuc_get_data= NULL;

  memset(vuc_err_cnt, 0, sizeof(gint)*256);

  send_q= g_async_queue_new();

  window1= GTK_WINDOW(gtk_builder_get_object(builder, "window1"));
  gtk_signal_connect(GTK_OBJECT(window1), "destroy", GTK_SIGNAL_FUNC(on_window1_destroy), NULL);

  gui_alive_cnt= GTK_ENTRY(gtk_builder_get_object(builder, "alive_cnt"));
  if (gui_alive_cnt) {
    gui_in_chk_cnt= GTK_ENTRY(gtk_builder_get_object(builder, "in_chk_cnt"));
    gui_out_chk_cnt= GTK_ENTRY(gtk_builder_get_object(builder, "out_chk_cnt"));
    gui_line_err_cnt= GTK_ENTRY(gtk_builder_get_object(builder, "line_err_cnt"));
    gui_proto_err_cnt= GTK_ENTRY(gtk_builder_get_object(builder, "proto_err_cnt"));
    gui_status_msg= GTK_ENTRY(gtk_builder_get_object(builder, "status_msg"));

    style= gtk_style_copy(gtk_widget_get_style(GTK_WIDGET(gui_alive_cnt)));
    memcpy(&style->text[GTK_STATE_INSENSITIVE], &style->text[GTK_STATE_NORMAL], sizeof(GdkColor));
    memcpy(&style->base[GTK_STATE_INSENSITIVE], &style->base[GTK_STATE_NORMAL], sizeof(GdkColor));

    gtk_widget_set_style(GTK_WIDGET(gui_alive_cnt), style);
    gtk_widget_set_style(GTK_WIDGET(gui_in_chk_cnt), style);
    gtk_widget_set_style(GTK_WIDGET(gui_out_chk_cnt), style);
    gtk_widget_set_style(GTK_WIDGET(gui_line_err_cnt), style);
    gtk_widget_set_style(GTK_WIDGET(gui_proto_err_cnt), style);
    gtk_widget_set_style(GTK_WIDGET(gui_status_msg), style);

    g_timeout_add(10, counter_bg, NULL);
  }

  dbg_scr= GTK_SCROLLED_WINDOW(gtk_builder_get_object(builder, "scrolledwindow1"));
  if (dbg_scr) {
    dbg_buf= GTK_TEXT_VIEW(gtk_builder_get_object(builder, "textview1"));
  }

  g_timeout_add(500, alive_bg, NULL);

  return builder;
}

void vuc_in (gint addr, u16 size, vuc_put_f func, gpointer user)
{
  if (addr>=vuc_put_cnt) {
    vuc_put_cnt= addr+1;
    vuc_put_data= g_realloc(vuc_put_data, vuc_put_cnt*sizeof(vuc_put_data_t));	// lets hope there are no uninitialised elements...
  }
  vuc_put_data[addr].no_of_bytes= size;
  vuc_put_data[addr].func= func;
  vuc_put_data[addr].user= user;
}

int vuc_go ()
{
  gtk_widget_show(GTK_WIDGET(window1));

  gtk_main();
}

int vuc_exit ()
{
}

// eof
