/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <config.h>

#include <math.h>
#include <stdio.h>

#if (GTK_MAJOR_VERSION<2)
#  include <gdk/gdk.h>
#endif
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkprivate.h>

#include "anygtk.h"
#include "vuc_host.h"
#include "vuc_proto.h"

#include "vuc_scope.h"


enum {
  PROP_0,

  PROP_VUC_SIZE,
  PROP_VUC_ADDR,
  PROP_VUC_INIT,
  PROP_VUC_RUN,

  PROP_BG_COLOR,
  PROP_GRID_COLOR,
  PROP_BEAM_COLOR,
};

/* Forward declarations */

static void vuc_scope_class_init (VucScopeClass * class);
static void vuc_scope_init (VucScope * sc);
#if (GTK_2)
static void vuc_scope_finalize (GObject * g_obj);
#endif /* GTK_2 */
static void vuc_scope_realize (GtkWidget * widget);
static void vuc_scope_set_property (GObject * g_obj, guint prop_id, const GValue * value, GParamSpec * pspec);
static void vuc_scope_get_property (GObject * g_obj, guint prop_id, GValue * value, GParamSpec * pspec);
static void vuc_scope_size_request (GtkWidget * widget, GtkRequisition * requisition);
static void vuc_scope_size_allocate (GtkWidget * widget, GtkAllocation * allocation);
static gint vuc_scope_expose (GtkWidget * widget, GdkEventExpose * event);

static void draw_pixmap (VucScope * sc);
static void draw_screen (VucScope * sc);

/* Local data */

static GtkWidgetClass * vuc_scope_parent_class= NULL;

guint vuc_scope_get_type ()
{
  static guint type= 0;

  if (!type)
    {
      GtkTypeInfo info=
      {
	"VucScope",
	sizeof(VucScope),
	sizeof(VucScopeClass),
	(GtkClassInitFunc)vuc_scope_class_init,
	(GtkObjectInitFunc)vuc_scope_init,
	/*(GtkArgSetFunc)*/ NULL,
	/*(GtkArgGetFunc)*/ NULL,
      };

      type= gtk_type_unique(gtk_widget_get_type(), &info);
    }

  return type;
}

static void vuc_scope_class_init (VucScopeClass * class)
{
  GtkObjectClass * object_class;
  GtkWidgetClass * widget_class;

  object_class= (GtkObjectClass*)class;
  widget_class= (GtkWidgetClass*)class;

  vuc_scope_parent_class= gtk_type_class(gtk_widget_get_type());

  widget_class->realize= vuc_scope_realize;
  widget_class->expose_event= vuc_scope_expose;
  widget_class->size_request= vuc_scope_size_request;
  widget_class->size_allocate= vuc_scope_size_allocate;

  object_class->agtk_set_prop_func= vuc_scope_set_property;
  object_class->agtk_get_prop_func= vuc_scope_get_property;
#if (GTK_2)
  object_class->finalize= vuc_scope_finalize;
#endif
//  object_class->destroy= vuc_scope_destroy;

  agtk_add_prop_int(
    object_class, "VucScope", PROP_VUC_SIZE, GTK_PARAM_READWRITE,
    "vuc-size", NULL, NULL, 0, 0xffff, 0
  );
  agtk_add_prop_int(
    object_class, "VucScope", PROP_VUC_ADDR, GTK_PARAM_READWRITE,
    "vuc-addr", NULL, NULL, -1, VUC_ADDR_MAX, -1
  );
  agtk_add_prop_string(
    object_class, "VucButton", PROP_VUC_INIT, GTK_PARAM_WRITABLE,
    "vuc-init", NULL, NULL, ""
  );
  agtk_add_prop_string(
    object_class, "VucButton", PROP_VUC_RUN, GTK_PARAM_WRITABLE,
    "vuc-run", NULL, NULL, ""
  );
  agtk_add_prop_string(
    object_class, "VucScope", PROP_BG_COLOR, GTK_PARAM_READWRITE,
    "bg-color", NULL, NULL, "#000"
  );
  agtk_add_prop_string(
    object_class, "VucScope", PROP_GRID_COLOR, GTK_PARAM_READWRITE,
    "grid-color", NULL, NULL, "#333"
  );
  agtk_add_prop_string(
    object_class, "VucScope", PROP_BEAM_COLOR, GTK_PARAM_READWRITE,
    "beam-color", NULL, NULL, "#6fa"
  );

}

static void vuc_scope_init (VucScope * obj)
{
  gtk_widget_set_has_window(GTK_WIDGET(obj), TRUE);

  obj->vuc_size= 0; obj->vuc_addr= -1;

  obj->points= g_malloc0(sizeof(GdkPoint)*2*2);
  obj->points[0].x= 0;              obj->points[0].y= VUC_SCOPE_NORM/2;
  obj->points[1].x= VUC_SCOPE_NORM; obj->points[1].y= VUC_SCOPE_NORM/2;
  obj->p_cnt= obj->p_max= 2;
#if (GTK_3)
  obj->csurf= NULL;
#else
  obj->pixmap= NULL;
  obj->gc= NULL;
#endif
}

GtkWidget * vuc_scope_new (void)
{
  VucScope * sc;

  sc= gtk_type_new(vuc_scope_get_type());

  return GTK_WIDGET(sc);
}

#if (GTK_2)

static void vuc_scope_finalize (GObject * obj)
{
  VucScope * sc= VUC_SCOPE(obj);

//  _vuc_scope_clear_layout (scale);
//  vuc_scope_clear_marks (scale);
  G_OBJECT_CLASS(vuc_scope_parent_class)->finalize(obj);
}

#endif /* GTK_2 */


static void parse_color (GdkColor * col, gchar * str, gint def)
{
  if (!gdk_color_parse(str, col)) {
//    msgf("Could not parse '%s' - using default.", str);
    col->red= col->green= col->blue= def;
  }
}

static void new_string (gchar ** str, const gchar * val)
{
  if (*str)
    g_free(*str);
  *str= g_strdup(val);
}

static void new_color (gchar ** str, GdkColor * col, const gchar * val, gint def)
{
  new_string(str, val);
  parse_color(col, *str, def);
}

static void vuc_scope_set_property
  (GObject * g_obj, guint prop_id, const GValue * value, GParamSpec * pspec)
{
  VucScope * sc= VUC_SCOPE(g_obj);

  switch (prop_id) {
    case PROP_VUC_SIZE: sc->vuc_size= g_value_get_int(value); break;
    case PROP_VUC_ADDR: sc->vuc_addr= g_value_get_int(value); break;
    case PROP_VUC_INIT: case PROP_VUC_RUN: /*vuc_prop_hdl_string(dir, value, obj->vuc_code);*/ break;
    case PROP_BG_COLOR:
      new_color(&sc->bg_col_str, &sc->bg_color, g_value_get_string(value), 0); break;
    case PROP_GRID_COLOR:
      new_color(&sc->grid_col_str, &sc->grid_color, g_value_get_string(value), 0x30); break;
    case PROP_BEAM_COLOR:
      new_color(&sc->beam_col_str, &sc->beam_color, g_value_get_string(value), 0xff); break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(g_obj, prop_id, pspec);
  }
}

static void vuc_scope_get_property
  (GObject * g_obj, guint prop_id, GValue * value, GParamSpec * pspec)
{
  VucScope * sc= VUC_SCOPE(g_obj);

  switch (prop_id) {
    case PROP_VUC_SIZE: g_value_set_int(value, sc->vuc_size); break;
    case PROP_VUC_ADDR: g_value_set_int(value, sc->vuc_addr); break;
    case PROP_VUC_INIT: case PROP_VUC_RUN: /*vuc_prop_hdl_string(dir, value, obj->vuc_code);*/ break;

    case PROP_BG_COLOR: g_value_set_string(value, sc->bg_col_str); break;
    case PROP_GRID_COLOR: g_value_set_string(value, sc->grid_col_str); break;
    case PROP_BEAM_COLOR: g_value_set_string(value, sc->beam_col_str); break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(g_obj, prop_id, pspec);
  }
}

static void vuc_scope_realize (GtkWidget * widget)
{
  VucScope * sc;
  GdkWindowAttr attributes;
  GdkColor green, amber, red, peak;
  gint attributes_mask;

  g_return_if_fail(widget != NULL);
  g_return_if_fail(VUC_IS_SCOPE(widget));

  GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
  sc= VUC_SCOPE(widget);

  attributes.x= widget->allocation.x;
  attributes.y= widget->allocation.y;
  attributes.width= widget->allocation.width;
  attributes.height= widget->allocation.height;
  attributes.wclass= GDK_INPUT_OUTPUT;
  attributes.window_type= GDK_WINDOW_CHILD;
  attributes.event_mask= gtk_widget_get_events(widget) | GDK_EXPOSURE_MASK;
  attributes.visual= gtk_widget_get_visual(widget);
  attributes.colormap= gtk_widget_get_colormap(widget);

  attributes_mask= GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window= gdk_window_new(widget->parent->window, &attributes, attributes_mask);

  widget->style= gtk_style_attach(widget->style, widget->window);

  gdk_window_set_user_data(widget->window, widget);

  gtk_style_set_background(widget->style, widget->window, GTK_STATE_ACTIVE);
  new_color(&sc->bg_col_str, &sc->bg_color, VUC_SCOPE_BG_COLOR, 0);
  new_color(&sc->grid_col_str, &sc->grid_color, VUC_SCOPE_GRID_COLOR, 0x30);
  new_color(&sc->beam_col_str, &sc->beam_color, VUC_SCOPE_BEAM_COLOR, 0xff);
}

static void vuc_scope_size_request (GtkWidget * widget, GtkRequisition * requisition)
{
  VucScope * sc= VUC_SCOPE(widget);

  requisition->width= VUC_SCOPE_WIDTH;
  requisition->height= VUC_SCOPE_HEIGHT;
}

static void vuc_scope_size_allocate (GtkWidget * widget, GtkAllocation * allocation)
{
  VucScope * sc;

  g_return_if_fail(widget != NULL);
  g_return_if_fail(VUC_IS_SCOPE(widget));
  g_return_if_fail(allocation != NULL);

  widget->allocation= *allocation;
  sc= VUC_SCOPE(widget);

  if (GTK_WIDGET_REALIZED(widget)) {
    draw_pixmap(sc);
    gdk_window_move_resize(widget->window,
			   allocation->x, allocation->y,
			   allocation->width, allocation->height);
    draw_screen(sc);
  }
}

static gint vuc_scope_expose (GtkWidget * widget, GdkEventExpose * event)
{
  VucScope * sc;

  g_return_val_if_fail(widget != NULL, FALSE);
  g_return_val_if_fail(VUC_IS_SCOPE(widget), FALSE);
  g_return_val_if_fail(event != NULL, FALSE);

  if (event->count > 0)
    return FALSE;

  sc= VUC_SCOPE(widget);
  draw_screen(sc);

  return FALSE;
}


#if (GTK_3)

static void draw_pixmap (VucScope * sc)
{
  gint i, o, x, ox, oy, mx, my;
  GdkWindow * win= gtk_widget_get_window(GTK_WIDGET(sc));
  GdkPoint * points= g_alloca(sizeof(GdkPoint)*sc->p_cnt*2);
  cairo_t * cr;
  GtkAllocation alloc;

  gtk_widget_get_allocation(GTK_WIDGET(sc), &alloc);

  if (sc->csurf)
    cairo_surface_destroy(csurf);
  csurf= cairo_image_surface_create(CAIRO_FORMAT_ARGB32, alloc.width, alloc.height);
  cr= cairo_create(csurf);
  cairo_set_source_rgb(cr, 0, 0, 0);
  cairo_rectangle(cr, 0, 0, alloc.width, alloc.height);
  cairo_fill(cr);

  ox= alloc.width*6/100;  mx= alloc.width-(2*ox);
  oy= alloc.height*6/100; my= alloc.height-(2*oy);

  cairo_set_source_rgb(cr, (double)div_color.red/0xffff, (double)div_color.green/0xffff, (double)div_color.blue/0xffff);
  cairo_set_line_width(cr, 1);
  for (i=0; i<=8; i++) {
    cairo_move_to(cr, ox-0.5,             oy+i*my/8-0.5);
    cairo_line_to(cr, alloc.width-ox-0.5, oy+i*my/8-0.5);
  }
  for (i=0; i<=8; i++) {
    cairo_move_to(cr, ox+i*mx/8-0.5, oy-0.5);
    cairo_line_to(cr, ox+i*mx/8-0.5, alloc.height-oy-0.5);
  }
  for (i=0; i<=5*8; i++) {
    cairo_move_to(cr, alloc.width/2-ox/4-0.5, oy+i*my/5/8-0.5);
    cairo_line_to(cr, alloc.width/2+ox/4-0.5, oy+i*my/5/8-0.5);
  }
  for (i=0; i<=5*8; i++) {
    cairo_move_to(cr, ox+i*mx/5/8-0.5, alloc.height/2-oy/4-0.5);
    cairo_line_to(cr, ox+i*mx/5/8-0.5, alloc.height/2+oy/4-0.5);
  }
  cairo_stroke(cr);

  cairo_set_source_rgb(cr, (double)beam_color.red/0xffff, (double)beam_color.green/0xffff, (double)beam_color.blue/0xffff);
  cairo_move_to(cr, ox+sc_points[0].x*mx/VUC_SCOPE_NORM-0.5, oy+sc_points[0].y*my/VUC_SCOPE_NORM-0.5);
  for (i=1; i<sc->p_cnt; i++)
    cairo_line_to(cr, ox+sc_points[i].x*mx/VUC_SCOPE_NORM-0.5, oy+sc_points[i].y*my/VUC_SCOPE_NORM-0.5);
  cairo_stroke(cr);

  cairo_destroy(cr);
}

static void draw_screen (VucScope * sc)
{
  GdkWindow * win= gtk_widget_get_window(GTK_WIDGET(u_scope_area));
  cairo_t * cr= gdk_cairo_create(win);

  if (!sc->csurf)
    draw_pixma(sc);

  cairo_set_source_surface(cr, csurf, 0, 0);
  cairo_paint(cr);
  cairo_destroy(cr);
}

#else /* GTK_3 */

static void draw_pixmap (VucScope * sc)
{
  gint i, o, x, ox, oy, mx, my;
  GdkColormap * colormap= NULL;
  GdkWindow * win= gtk_widget_get_window(GTK_WIDGET(sc));
  GdkPoint * pa= g_alloca(sizeof(GdkPoint)*sc->p_cnt*2);
  GtkAllocation alloc;

  gtk_widget_get_allocation(GTK_WIDGET(sc), &alloc);

  if (sc->pixmap) {
    gdk_gc_unref(sc->gc);
    gdk_pixmap_unref(sc->pixmap);
  }
  sc->pixmap= gdk_pixmap_new(win, alloc.width, alloc.height, -1);
  gdk_draw_rectangle(sc->pixmap, GTK_WIDGET(sc)->style->black_gc, TRUE, 0, 0, alloc.width, alloc.height);
  sc->gc= gdk_gc_new(sc->pixmap);
  colormap= gdk_gc_get_colormap(sc->gc);

  ox= alloc.width*5/100;  mx= alloc.width-(2*ox);
  oy= alloc.height*5/100; my= alloc.height-(2*oy);

  gdk_colormap_alloc_color(colormap, &sc->grid_color, TRUE, TRUE);
  gdk_gc_set_foreground(sc->gc, &sc->grid_color);
  for (i=0; i<=8; i++) {
    pa[0].x= ox;             pa[0].y= oy+i*my/8;
    pa[1].x= alloc.width-ox; pa[1].y= oy+i*my/8;
    gdk_draw_lines(sc->pixmap, sc->gc, pa, 2);
  }
  for (i=0; i<=8; i++) {
    pa[0].x= ox+i*mx/8; pa[0].y= oy;
    pa[1].x= ox+i*mx/8; pa[1].y= alloc.height-oy;
    gdk_draw_lines(sc->pixmap, sc->gc, pa, 2);
  }
  for (i=0; i<=5*8; i++) {
    pa[0].x= alloc.width/2-ox/4; pa[0].y= oy+i*my/5/8;
    pa[1].x= alloc.width/2+ox/4; pa[1].y= oy+i*my/5/8;
    gdk_draw_lines(sc->pixmap, sc->gc, pa, 2);
  }
  for (i=0; i<=5*8; i++) {
    pa[0].x= ox+i*mx/5/8; pa[0].y= alloc.height/2-oy/4;
    pa[1].x= ox+i*mx/5/8; pa[1].y= alloc.height/2+oy/4;
    gdk_draw_lines(sc->pixmap, sc->gc, pa, 2);
  }

  gdk_colormap_alloc_color(colormap, &sc->beam_color, TRUE, TRUE);
  gdk_gc_set_foreground(sc->gc, &sc->beam_color);
  for (x=0; x<sc->p_cnt; x++) {
    pa[x].x= ox+sc->points[x].x*mx/VUC_SCOPE_NORM;
    pa[x].y= oy+sc->points[x].y*my/VUC_SCOPE_NORM;
  }
  gdk_draw_lines(sc->pixmap, sc->gc, pa, sc->p_cnt);
}

static void draw_screen (VucScope * sc)
{
  GdkDrawable * drawable= GTK_WIDGET(sc)->window;
  GtkAllocation alloc;

  gtk_widget_get_allocation(GTK_WIDGET(sc), &alloc);

  if (!sc->pixmap)
    draw_pixmap(sc);

  gdk_draw_drawable(drawable, sc->gc, sc->pixmap,
		    0, 0,  0, 0,  alloc.width, alloc.height);
}

#endif /* GTK_3 */

static gint get_sample (gint size_d, gpointer data)
{
  return
    size_d==1 ? (*(guint8 *)data) :
    size_d==2 ? (*(guint16 *)data) :
    size_d==4 ? (*(guint32 *)data) :
    0;
}

i8 vuc_scope_in (u8 * data, void * user)
{
  VucScope * sc= VUC_SCOPE(user);
  gint size_x= sc->vuc_size;
  gint size_y= 256; gint size_d= 1;	// TODO: get that somewhere else...
  u8 x, v;
  gint c, lc= (size_y-1)-get_sample(size_d, data);

  if (sc->p_max<=size_x) {
    sc->p_max= size_x+1;
    sc->points= g_realloc(sc->points, sc->p_max*sizeof(GdkPoint)*2);
  }
  sc->p_cnt= 0;
  for (x=0; x<size_x; x++) {
    c= (size_y-1)-get_sample(size_d, data);
    data+= size_d;
    if (x==0 || c!=lc) {
      sc->points[sc->p_cnt].x= x*VUC_SCOPE_NORM/size_x; sc->points[sc->p_cnt].y= lc*VUC_SCOPE_NORM/size_y; sc->p_cnt++;
      sc->points[sc->p_cnt].x= x*VUC_SCOPE_NORM/size_x; sc->points[sc->p_cnt].y= c*VUC_SCOPE_NORM/size_y; sc->p_cnt++;
    }
    lc= c;
  }
  sc->points[sc->p_cnt].x= x*VUC_SCOPE_NORM/size_x; sc->points[sc->p_cnt].y= c*VUC_SCOPE_NORM/size_y; sc->p_cnt++;
  draw_pixmap(sc);
  draw_screen(sc);
  return 1;
}

