// ---[bc.c] ------------------------------------------------------------------
/*
Grundlage war die Applicationnote AVR450 von Atmel, auf der dann aber frei
variiert wurde:
Timer0 gibt den Takt fr die Messungen und die Zeitzhlung vor, 
es werden 8 Regelvorgnge je Sekunde ausgefhrt,
2x je Sekunde wird die Anzeige der Messwerte aktualisiert

Timer1 arbeitet im Modus Fast-PWM 9-Bit zur Ansteuerung des Reglers.

Der Ablauf der Ladung erfolgt in 4 Abschnitten:
0. Testen, ob die Umgebung zum Laden geeignet ist (Uref ok, Akku vorhanden)
1. Fast charge - Current gesteuerte Phase
2. Fast charge - Voltage gesteuerte Phase
3. Trickle charge 
4. Ende

Die 5 Phasen werden nacheinander aus main() heraus aufgerufen.
Aus den Phasen heraus werden Routinen zur Messung (U, I, Zeit) aufgerufen
und ausgewertet.
Mit den Messungen wird die geladene Kapazitt berechnet und angezeigt.

Die berprfung der Batterietemperatur kann nachgerstet werden.

Die Zeitzhlung beginnt mit dem Start des Ladevorganges, die Teilzeiten fr
die drei Ladephasen werden getrennt aufgezeichnet und ausgegeben.

Das Programm ist getestet und funktioniert.

*/
// ----------------------------------------------------------------------------
#define EXTERN
//#define DELAY

#include <avr/interrupt.h>
#include <stdlib.h>
#include <util/delay.h>
#include <avr/wdt.h>
#include "global.h"
#include "li_ion.h"
#include "lcd.h"

#define T0_PRESCALE  (1<<CS02)			// Prescale = 256
//#define T0_PRELOAD (256 - 156)   		// 4.000.000 / (256 * 156) = 100.16 -> 100 Aufrufe / Sekunde
#define T0_PRELOAD (256 - 195)   		// 4.000.000 / (256 * 195) =  80.13 ->  80 Aufrufe / Sekunde
// 4.000.000 / (256 * 80) = 195.3125   - bei Preload von 195 fehlen 0.3125 Zhler pro Runde
// bei 100 Runden 31.25 - die 0.25 vergessen wir. Also nach 100 Runden Reload um 31 erhhen !
#define T0_PRELOAD1 (256 - 195 - 31)

// ----------------------------------------------------------------------------
void wait_1sec(void)
{
uint8_t i;
for (i = 0; i < 250 ; i++)
	{
	_delay_ms(4);
	}
}

// ----------------------------------------------------------------------------
// Stoppt den PWM und setzt den Pin.B1 (OC1A) auf low
void Stop_PWM(void)                         
{
TCCR1B 	= 0;									// den Counter1/CTC stoppen
TCCR1A 	= 0; 									// OC1A abkoppeln
PORTB	&= (~(1<<PB1));         			// PORTB.1 (OC1A) auf low
DDRB 	&= (~(1<<PB1));         			// PORTB.1 (OC1A) als Eingang
}

// ---------------------------------------------------------------------------
// Startet den PWM auf OC1A (wieder)
void Start_PWM(void)
{
TCNT1  = 0;										// Counter auf 0
PORTB &= (~(1<<PB1));						// OC1A auf low
DDRB 	|= (1<<PB1);     						// PORTB.1 (OC1A) als output
// clear OC1A (PB.1) on Compare Match when upcounting, set OC1A when downcounting, Fast PWM 9 Bit
TCCR1A = (1<<COM1A1 | 0<<COM1A0 |1<<WGM11);
TCCR1B = (1<<WGM12 | 1<<CS10);			// Fast PWM 9 Bit, Prescale 1
}

// ----------------------------------------------------------------------------
// gibt einen Warnton auf OC1B aus
// dazu den Timer1 umprogrammieren fr Tonausgabe an OC1B
// der Timer0 luft noch und setzt Bits in FLAG
// der Ton hlt 1/8 Sekunde an, dann 7/8 Sekunde Pause
// der Parameter (signals) gibt die Anzahl der auszugebenden Tne an
void err_sound(uint8_t signals)
{
uint16_t t_OCR1A;						// OCR1A/B werden gesichert
//uint16_t t_OCR1B;
uint8_t  t_COUNTER10;
uint8_t  t_COUNTER80;

t_OCR1A = OCR1A;
//t_OCR1B = OCR1B;
t_COUNTER10 = COUNTER10;
t_COUNTER80 = COUNTER80;

COUNTER10 = 1;							// Wird vom Timer0 genutzt
COUNTER80 = 1;							// fhrt nach 1/8 Sekunde zum berlauf

// TIMER1 bestimmt mit seinem Compare-Wert die Frequenz
OCR1A  = TON;							// OCR1A legt TOP fr den Counter fest
//OCR1B  = TON;							// beide OCR1B schaltet den Pin OC1B !

TCNT1H = 0;								// Counter1 auf 0 setzen
TCNT1L = 0;

TCCR1A = (1<<COM1B0);				// Toggle OC1B on Compare match
TCCR1B = (1<<WGM12 | 1<<CS10);	// Timer1 fr Tonsteuerung starten, CTC-Mode, Prescale 1

// jetzt erfolgt die Signalausgabe an OC1B, wobei OCR1A den Schaltpunkt bestimmt
// der Timer setzt ist Abstnden Bits in FLAG, z.B Bit.3 2x je Sekunde
while(signals)
	{
	if (FLAG & 0x02)					// Sekundenflag - TON an
		{
		FLAG &= (~(0x03));			// Flag1/0 wieder lschen (da beide gleichzeitig gesetzt werden)
		DDRB |= (1<<PB2);				// Ton ausgeben
		}
	else if (FLAG & 0x01)			// 1/8 Sekundenflag - TON aus
		{
		FLAG &= (~(0x01));			// Flag wieder lschen
		if (DDRB & (1<<PB2))			// Nur wenn DDRB2 als Ausgang geschaltet ist
			{
			DDRB &= (~(1<<PB2));		// Tonausgabe beenden
			signals--;					// und den Zhler fr die Anzahl der Signal decrementieren
			}
		}
	}
PORTB &= (~(1<<PB2));				// PinB2 auf low
DDRB  &= (~(1<<PB2));				// PinB.2 als Eingang / abschalten
TCCR1B = 0;								// PWM abschalten

OCR1A = t_OCR1A;
//OCR1B = t_OCR1B;

COUNTER10 = t_COUNTER10;
COUNTER80 = t_COUNTER80;

}
// ----------------------------------------------------------------------------
// fhrt eine Einzelmessung auf einem bereits eingestellten Port durch
// zur einfachen Umstellung vom Polling- auf Interrupt-Betrieb.
// Smtliche Spannungsmessung laufen ber diese Funktion !!
void messung(void)
{
ADCSR |= ((1<<ADSC) | (1<<ADIF));   	// Start a new A/D conversion, clear ADIF
while (!(ADCSR & (1<<ADIF)));       	// warten bis ADIF gesetzt ist
}

// ----------------------------------------------------------------------------
// fhrt eine Messreihe auf dem angebenen Kanal aus und gibt den Mittelwert zurck
// Der Takt des ADC betrgt F_CPU / 32, pro Messung werden 13 Takte bentigt
// (32 * 13) / 4.000.000 -> ca. 75 us * 32 Messungen -> 2.4 ms
uint16_t messreihe(uint8_t channel)
{
uint16_t mittelwert = 0;
uint8_t i;
ADMUX = channel;								// Port fr Messung festlegen

for(i = 0; i < 32; i++)
	{
	messung();									// fhrt die Messung aus, das Ergebnis steht in ADC
	mittelwert += ADC;
	//_delay_us(80);							// max 260us/MHz
	}

mittelwert /= 32;								// Mittelwert durch Anzahl der Messungen teilen
return(mittelwert);
}

// ----------------------------------------------------------------------------
// fhrt eine einzelne Messung der Referenzspannung aus und liefert nur TRUE oder FALSE zurck
// umgestellt auf Messreihe
uint8_t messung_uref(void)
{
uint16_t u_ref;

u_ref = messreihe(U_REF);					// fhrt die Messung aus, das Ergebnis steht in ADC
													// Messwert sollte 2500mV sein = 500
if ((u_ref > (500 - 3)) && (u_ref < (500 + 3))) return(1);


return(0);
}


// ----------------------------------------------------------------------------
// wird 80x pro Sekunde aufgerufen, incrementiert zwei Zhler
// und setzt das Flag auf 1 (nach 10 Takten) bzw. 2 (nach 80 Takten = 1 Sekunde)
// zustzlich noch Bit.2 fr refresh der Spannungs-Stromanzeige im 0.5 Sekundentakt
ISR(TIMER0_OVF_vect)
{
static uint8_t q = 100;

if (q == 0)										// Preload um ganze Sekunden zu zhlen
	{
	TCNT0 = T0_PRELOAD1;						// Preload zur Korrektur alle 100 Runden
	q = 100;										// um 31 vergrern = 0.31 pro Runde
	}
else
	{
	TCNT0 = T0_PRELOAD;						// Preload nach Overflow neu laden
	q--;
	}


if (!(--COUNTER10))							// decrementieren, wenn dann == 0
	{
	COUNTER10 = 10;							// zurck zum Startwert
	FLAG |= 0x01;								// Flag.0 setzen
	}
if (COUNTER80 == 20) FLAG |= 0x04;		// Flag.2 setzen
if (COUNTER80 == 60) FLAG |= 0x04;		// Flag.2 setzen
													// wird benutzt, um Teile der Anzeige 
													// alle 1/2 Sekunde neu zu schreiben
if (!(--COUNTER80))							// decrementieren, wenn dann == 0
	{
	COUNTER80 = 80;							// zurck zum Startwert
	FLAG |= 0x02;								// Flag.1 setzen 
	}
}



// ----------------------------------------------------------------------------
void err_msg(uint8_t error)
{
Stop_PWM();										//  PWM abschalten !!!	
wdt_disable();

lcd_goto_xy(4,1);
if 	  (error == ERR_UREF)	   lcd_puts("Err: Uref off limit");
else if (error == ERR_TIMEOUT)   lcd_puts("Err: TimeOverFlow !");
else if (error == ERR_NO_AKKU)   lcd_puts("Err: no Akku !");
else if (error == ERR_TEMP) 	   lcd_puts("Err: Temperatur !");
else if (error == ERR_FULL)      lcd_puts("Err: Akku voll !");
else if (error == ERR_AKKU_LOST) lcd_puts("Err: Akku lost !");
else if (error == ERR_READY)     lcd_puts("OK: Akku geladen !");
else if (error == ERR_WDT)       lcd_puts("Err: Watchdog reset");

if (error == ERR_READY) err_sound(3); 	// 3x im Sekundentakt
else err_sound(24);							// bei allen anderen Fehler 24x
while(1); 										// Endlosschleife
}

// ----------------------------------------------------------------------------
int main(void)
{
Stop_PWM();									// fhrt die beiden folgenden Einstellungen aus
												// damit der PWM garantiert auf low steht.
sei();               					// interrupt enable
												// TIMER0 - Taktgeber und Zeitzhler
												// F_CPU 4.000.000
TCCR0 = (1<<CS02);						// Prescale = 256 -> 4.000.000 / (256 * PRELOAD) 
TCNT0 = T0_PRELOAD;
TIMSK |= (1<<TOIE0);						// Timer0_overflow aktivieren
COUNTER10 	= 10;							// Counter, die von der ISR decrementiert werden
COUNTER80 	= 80;

// TIMER1 - PWM
// wird von Start_PWM() / Stop_PWM() bernommen !

												// PORTD.7 fr Start-Taste
PORTD |= (1<<PD7);						// Pullup setzen

//ADC enablen, Single A/D conversion, F_CPU/32
ADCSRA 	= (1<<ADEN | 1<<ADPS2 | 1<<ADPS0);

lcd_init(LCD_DISP_ON);

if (MCUCSR & 1<<WDRF)					// dann hat der Watchdog zugeschlagen
	{
	lcd_goto_xy(4,1);
	MCUCSR &= (~(1<<WDRF));				// Flag fr WDT zurcksetzen
	err_msg(ERR_WDT);
	}

// #### PHASE 0 ###############################################################

// Zuerst die Referenzspannung testen, dazu den Messwert am LM385 2,5 testen,
// er muss bei 500 (*5mv) +- 2 liegen, Meldung ausgeben
lcd_goto_xy(1,1);
if (messung_uref())
	{
	lcd_puts("URef ok. (");
	lcd_puts(utoa(ADC , msg, 10));
	lcd_putc(')');		
	}
else
	{
	err_msg(ERR_UREF);					// err_msg gibt Meldung aus und stoppt die Programmausfhrung
	}

// dann die Spannung am Akku messen:
// liegt sie unterhalb 7,0 Volt, dann Warnung ausgeben
// liegt sie oberhalb  8,4 Volt, dann ist der Akku bereits geladen
// Meldung ausgeben
lcd_goto_xy(2,1);
ADMUX = U_BATT;							// fr die nchste Einzelmessung
messung();									// Einzelmessung auf vorher eingestelltem Port
status[U] = ADC;							// Messwert einlesen
status[I] = 0;								// ein Strom kann noch nicht gemessen werden !

//	temp16 *= 10;							// je Bit 5mV * 2 (Spannungsteiler)
lcd_puts("Akkuspannung: ");
lcd_puts(ltoa(status[U], msg, 10));	// Ausgabe der dreistellige Ziffer
lcd_puts("0mV");							// anstelle der Multiplikation wird nur eine "0" nachgestellt

if (status[U] < 700)						// wenn weniger als 7000mV gemessen wurden .....
	{
	err_msg(ERR_NO_AKKU);				// err_msg gibt Meldung aus und stoppt die Programmausfhrung
	}
if (status[U] >= U_TRICKLE)			// dann ist der Akku bereits geladen
	{
	err_msg(ERR_FULL);					// err_msg gibt Meldung aus und stoppt die Programmausfhrung
	}

// wenn kein Fehler aufgetreten ist, dann auf Tastendruck warten und den Ladevorgang starten
lcd_goto_xy(4,1);
lcd_puts("press key to start");
err_sound(3);								// Achtung - der Timer0 muss bereits laufen, sei() ausgefhrt sein.
												// die Einstellungen fr den TIMER1/PWM mssen anschlieend erfolgen !!!
												
												// Der Pullup muss rechtzeitig vorher akiviert sein
while (PIND & (1<<PD7));         	// warten solange wie PD.7 high ist !!!!!!

//lcd_command(LCD_HOME);				// LCD lschen
lcd_command(1<<LCD_CLR);

time.sec 		= 0x00;
time.min 		= 0x00;
time.hour 		= 0x00;
charge.fast_C 	= 0xFF;					// Damit keine Ausgabe der Werte im Display erfolgt
charge.fast_U 	= 0xFF;
charge.trickle	= 0xFF;
												// TIMER0 - Taktgeber und Zeitzhler zurckstellen
												// F_CPU 4.000.000
TCCR0 = (1<<CS02);						// Prescale = 256 -> 4.000.000 / (256 * PRELOAD) 
TCNT0 = T0_PRELOAD;
TIMSK |= (1<<TOIE0);						// Timer0_overflow aktivieren
COUNTER10 	= 10;							// Counter, die von der ISR decrementiert werden
COUNTER80 	= 80;

OCR1A 		= 0;							// damit die PWM garantiert von 0 beginnt
charge_time = 0;							// damit die Zeit auf jeden Fall angezeigt wird

// #### PHASE 1 ###############################################################
//
print_zeile1();							// Anzeige Spannung, Strom und OCR1A -> alles Null !
print_zeile3();							// Zeile 3 des Displays 
lcd_goto_xy(4,1);							// Meldung fr die nchste Ladephase ausgeben
lcd_puts("> Phase 1 ...");

wdt_enable(WDTO_500MS);

Start_PWM();								// PWM starten
Fast_C();									// Laden mit maximalem Strom bis zu Ladeschlussspannung

charge.fast_C = charge_time;			// Ladezeit der Phase (00:00) speichern
charge_time = 0;							// und Zeitzhlung neu starten
Stop_PWM();
wdt_disable();

ADMUX = U_BATT;
messung();									// liest neuen Wert in ADC ein

if (ADC > U_FAST_C + 10)				// wenn die Spannung hher als Max-Wert
	{											// dann ist der Akku entnommen worden
	err_msg(ERR_AKKU_LOST);
	}

// #### PHASE 3 ###############################################################
//

err_sound(1);								// Signalton ausgeben (1)

print_zeile3();							// Zeile 3 des Displays aktualisieren
lcd_goto_xy(4,1);							// Meldung fr die nchste Ladephase ausgeben
lcd_puts("> Phase 2 ...");

wdt_enable(WDTO_500MS);
Start_PWM();								// PWM wieder starten

Fast_U();									// Halten der Ladeschlussspannung bis zu min. Strom

charge.fast_U = charge_time;			// Ladezeit der Phase (00:00) speichern
//charge_time = 0;						// und die Zeitzhlung neu starten - 
												// nach unten hinter die Wartezeit verschoben !!
Stop_PWM();
wdt_disable();

ADMUX = U_BATT;
messung();									// liest neuen Wert in ADC ein

if (ADC > U_FAST_U + 10)				// wenn die Spannung hher als Max-Wert
	{											// dann ist der Akku entnommen worden
	err_msg(ERR_AKKU_LOST);
	}

// #### PHASE 4 ###############################################################
//

print_zeile3();							// Zeile 3 des Displays aktualisieren
err_sound(2);								// Signalton ausgeben (2) 

lcd_goto_xy(4,1);							// Meldung fr die nchste Ladephase ausgeben
lcd_puts("> Phase 3 ...");
wait_1sec();								// kurze Wartezeit einlegen, damit die Spannung abfallen kann

ADMUX = U_BATT;							// Batteriespannung messen
do 
{
if (FLAG & 0x02)							// eine neue Sekunde ist angebrochen
	{
	FLAG &= (~(0x02));					// FLAG.1 lschen
	new_second();							// current wird von new_second() ausgewertet
	messung();								// liest neuen Wert in ADC ein
	}
}
while (ADC > (U_TRICKLE - 2));		// Batteriespannung im Leerlauf etwas abklingen lassen

wdt_enable(WDTO_500MS);
												// die Spannung steht noch auf Maximalwert und soll abfallen knnen
charge_time = 0;							// und die Zeitzhlung neu starten - die Wartezeit nicht zhlen !!
Trickle_U();								// Laden mit geringem Strom bis Ladeschlussspannung

charge.trickle = charge_time;			// Ladezeit der Phase (00:00) speichen
wdt_disable();	

// #### PHASE 5 ###############################################################
print_zeile1();							// alle Zeilen aktualisieren
print_zeile2();
print_zeile3();

ADMUX = U_BATT;
messung();									// liest neuen Wert in ADC ein

if (ADC > U_TRICKLE + 10)				// wenn die Spannung hher als Max-Wert
	{											// dann ist der Akku entnommen worden
	err_msg(ERR_AKKU_LOST);
	}
/*else if (ADC < U_TRICKLE -10)
	{
	err_msg(); */
else							
	{
	err_msg(ERR_READY);					// und beenden
	}
while(1);

}

// --- [ eof ] ----------------------------------------------------------------

