// --- [li_ion.c] -------------------------------------------------------------

#include <stdlib.h>
#include <stdint.h>
#include <avr/wdt.h>
#include <util/delay.h>
#include "li_ion.h"
#include "global.h"
#include "lcd.h"

uint32_t ladung;								// die Summe aller Strme in mA*Sekunde/2.5
uint16_t current;								// der Strom der letzten Sekunde

// ----------------------------------------------------------------------------
// druckt eine 16Bit-Zahl an der aktuellen Cursorposition mit fhrenden Leerstellen
void lcd_print16(uint16_t zahl)
{
if (zahl < 1000) lcd_putc('.');
if (zahl < 100 ) lcd_putc('.');
if (zahl < 10  ) lcd_putc('.');
lcd_puts(ultoa(zahl, msg, 10));
}

// ----------------------------------------------------------------------------
// druckt eine 8Bit-Zahl an der aktuellen Cursorposition mit fhrenden Leerstellen
void lcd_print8(uint8_t zahl)
{
if (zahl < 100 ) lcd_putc('.');
if (zahl < 10  ) lcd_putc('.');
lcd_puts(utoa(zahl, msg, 10));
}

// ----------------------------------------------------------------------------
// formatiert eine 16-Bit Zahl in Minuten und Sekunden
void lcd_print_ms(uint16_t zeit)
{
uint8_t temp;

if (zeit == 0xFF)								// wenn zeit == 0
	{
	lcd_puts("--:--");
	return;
	}

temp = (uint8_t) (zeit / 60);				// die Minuten

if (temp < 10) lcd_putc('0');
lcd_puts(utoa(temp, msg, 10));

zeit -= (temp * 60);
lcd_putc(':');

if (zeit < 10) lcd_putc('0');				// die Sekunden
lcd_puts(utoa(zeit, msg, 10));
}

// ----------------------------------------------------------------------------
// gibt die Zeile 1 des Displays aus mit Spannung, Strom und OCR1A
void print_zeile1(void)
{
//uint8_t temp1;
uint16_t temp;
lcd_goto_xy(1,1);								// Ausgabe der Spannung in Volt

//temp1  = status[U] / 100;				// die Vorkommastellen ausgeben
//if (temp1 < 10) lcd_putc('.');
//lcd_puts(utoa(temp1, msg, 10));
//lcd_putc(',');
//temp1 = (uint8_t) status[U] % 100;
//lcd_puts(utoa(temp1, msg, 10));
lcd_print16(status[U]);
lcd_puts("0mV  ");
//lcd_puts("V   ");

temp = status[I] * 5;						// Multiplikation mit 2.5
temp /= 2;										// 1 Digit entspricht 2.5mA

lcd_print16(temp);							// Ausgabe des Stromes
lcd_puts("mA ");

lcd_print16(OCR1A);							// Ausgabe des OCR1A (duty-cycle)
}

// ----------------------------------------------------------------------------
// gibt die Zeile 2 des Displays aus mit Dauer und geladenem Strom
void print_zeile2(void)
{
uint16_t temp;

lcd_goto_xy(2,1);
lcd_putc('0');									// fhrende Null der Stunden
lcd_putc(time.hour + 0x30);				// die Stunden werden wohl nie grer als 9 !!!
lcd_putc(':');
if (time.min < 10) lcd_putc('0');		// Minuten
lcd_puts(utoa(time.min, msg, 10));
lcd_putc(':');
if (time.sec < 10) lcd_putc('0');		// Sekunden
lcd_puts(utoa(time.sec, msg, 10));
													// 1000mA <> 400 Digits
temp = (uint16_t) (ladung / 1440);		// das ist die Ladung in mAh (ladung / (3600 / 2.5))
lcd_goto_xy(2,14);
lcd_print16(temp);
lcd_puts("mAh");

}

// ----------------------------------------------------------------------------
// gibt die Zeile 3 des Displays mit den Ladezeiten der Phasen aus
void print_zeile3(void)
{
lcd_goto_xy(3,1);								// Ladezeit Phase 1
lcd_print_ms(charge.fast_C);

lcd_goto_xy(3,9);								// Ladezeit Phase 2
lcd_print_ms(charge.fast_U);

lcd_goto_xy(3,16);							// Ladezeit Phase 3
lcd_print_ms(charge.trickle);
}

// ----------------------------------------------------------------------------
// eine neue Sekunde ist angebrochen - die Zeitzhlung wird aktualisiert.
// Die Zeitanzeige am LCD wird neu geschrieben,
// die Ladung ist neu zu berechnen - hier wird auf 2 lokale Variablen zurckgegriffen
void new_second(void)
{
charge_time++;									// die Ladezeit der Phasen1..3 in Sekunden
// zuerst die allgemeine Zeitzhlung
if ((++time.sec) == 60)
	{
	time.sec = 0;
	if ((++time.min) == 60)					// die Ladezeit in Minuten
		{
		time.min = 0;
		time.hour++;							// die Stunden nicht weiter prfen
		}
	}

// vom Akku aufgenommenen Strom in lokalen Variablen aufaddieren und anzeigen
current /= 8;									// den Mittelwert der 8 Strommessungen bilden
ladung += current;							// Ladung gemessen in mA * Sekunden
current = 0;									// fr die nchste Sekunde wieder zurcksetzen

// dann die Zeitanzeige und Ladungsanzeige aktualisieren
print_zeile2();

if (!(messung_uref()))						// Referenzspannung prfen
	{
	err_msg(ERR_UREF);
	}

if (!(temperatur_check()))					// die Akku-Temperatur prfen
	{
	err_msg(ERR_TEMP);
	}
}

// ----------------------------------------------------------------------------
// dummy-Funktion zum Nachrsten - berprft die Temperatur des Akkus 
uint8_t temperatur_check(void)
{
// Temperatur messen
// mit Grenzwerten vergleichen und Ergebnis zurckliefern

return(1);
}

// ----------------------------------------------------------------------------
// Ladephase 1 - Regelung nach maximalem Strom bis zum Erreichen der Endspannung.
// Der Ladetrom wird in Richtung Sollwert erhht - und dann die Spannung gemesssen.
// Sobald die Ladeendspannung errreicht ist, wird diese Phase beendet
void Fast_C(void)
{
current = 0;									// Summe der Strme innerhalb einer Sekunde
do
   {
	if (FLAG & 0x01)							// Messung in Intervallen ausfhren,
		{											// die durch Timer0 vorgegeben werden
		FLAG &= (~(0x01));					// FLAG.0 lschen
		
		status[U] = messreihe(U_BATT);	// nun die aktuelle Spannung messen
													// -> Regelgre
		status[I] = messreihe(I_BATT);	// Strom messen und speichern
		current += status[I];				// und Summe ber 1 Sekunde bilden

													// hier wird um den Sollwert geregelt
		if      ((status[I] < I_FAST_C) && (OCR1A < PWM_TOP)) OCR1A++;
		else if ((status[I] > I_FAST_C) && (OCR1A > 0x00)) OCR1A--;
		wdt_reset();
		}

	if (FLAG & 0x02)							// eine neue Sekunde ist angebrochen
		{
		FLAG &= (~(0x02));					// FLAG.1 lschen
		new_second();							// current wird von new_second() ausgewertet

		if (charge_time > FAST_C_TIMEOUT)// wenn der Grenzwert fr die Ladedauer in Phase 2 berschritten ist
			{
			err_msg(ERR_TIMEOUT);
			}
		}

	if (FLAG & 0x04)							// Anzeige von Spannung, Strom und OCR1A aktualisieren
		{
		FLAG &= (~(0x04));					// Flag lschen
		print_zeile1();
		}

	// die Schleife wird beendet, sobald die Spannung hher als der Grenzwert ist.
	}
while (status[U] <= U_FAST_C);        	// Grenzwert der Akkuspannung fr Phase 1 ist berschritten
Stop_PWM();
}

// ----------------------------------------------------------------------------
// Ladephase 2 - Regelung des Lasestromes - um die Ladeschlussspannung zu halten.
// Die Spannung wird gemessen und um die Ladeschlussspannung geregelt.
// Sobald der Ladestrom eine untere Grenze erreicht hat, wird diese Phase beendet
void Fast_U(void)
{
current = 0;
do
   {
	if (FLAG & 0x01)							// Messung in Intervallen ausfhren,
		{											// die durch Timer0 vorgegeben werden
		FLAG &= (~(0x01));					// FLAG.0 lschen
		
		status[I] = messreihe(I_BATT);	// Strom messen
		current += status[I];				// und Summe ber 1 Sekunde bilden		
													// -> Regelgre
		status[U] = messreihe(U_BATT);	// nun die aktuelle Spannung messen
		
													// falls die Spannung zu hoch wird, die Schleife abbrechen
//		if (status[U] > (U_FAST_U + 10)) break; // der Wert 5 fhrt gelegentlich zu Abbrchen
		
												// hier wird um den Sollwert geregelt
	
	// das Aufwrtsregeln fhrt zu Schwingungen - und macht auch eigentlich keinen Sinn,
	// da der Strom ja stetig abfallen muss
	// im Gegensatz zu FAST_C wird hier nur ABWRTS geregelt
	//	if ((status[U] < U_FAST_U) && (temp < PWM_TOP)) temp++; // ohne ist das Regelverhalten stetiger !!
		if ((status[U] > U_FAST_U) && (OCR1A > 0x00)) OCR1A--;      

		wdt_reset();
		}

	if (FLAG & 0x02)							// eine neue Sekunde ist angebrochen
		{
		FLAG &= (~(0x02));					// FLAG.1 lschen
		new_second();							// current wird von new_second() ausgewertet


		if (charge_time > FAST_U_TIMEOUT)// wenn der Grenzwert fr die Ladedauer in Phase 2 berschritten ist
			{
			err_msg(ERR_TIMEOUT);
			}
		}

	if (FLAG & 0x04)							// Anzeige von Spannung, Strom und OCR1A aktualisieren
		{
		FLAG &= (~(0x04));					// Flag lschen
		print_zeile1();
		}

	// die Schleife wird beendet, sobald der Ladestrom kleiner als der Grenzwert ist.
	}
while (status[I] > I_FAST_U);       	// unterer Grenzwert des Ladestroms fr Phase 2 ist berschritten


}

// ----------------------------------------------------------------------------
// Ladephase 3 - Regelung nach Trickle-Strom bis zum Erreichen der Endspannung.
// Der Ladetrom wird in Richtung Sollwert erhht - und dann die Spannung gemesssen.
// Sobald die Ladeendspannung errreicht ist, wird diese Phase beendet
void Trickle_U(void)
{
current= 0;
OCR1A  = 0;										// mit 0 mA neu beginnen
Start_PWM();									// Ladestrom wieder anfahren

do
   {
	if (FLAG & 0x01)							// Messung in Intervallen ausfhren,
		{											// die durch Timer0 vorgegeben werden
		FLAG &= (~(0x01));					// FLAG.0 lschen
		status[U] = messreihe(U_BATT);	// nun die aktuelle Spannung messen
													// -> Regelgre
		status[I] = messreihe(I_BATT);	// Strom messen und speichern
		current += status[I];				// und Summe ber 1 Sekunde bilden

													// hier wird um den Sollwert geregelt
		if      ((status[I] < I_TRICKLE) && (OCR1A < PWM_TOP)) OCR1A++;
		else if ((status[I] > I_TRICKLE) && (OCR1A > 0x00)) OCR1A--;
		wdt_reset();
		}

	if (FLAG & 0x02)							// eine neue Sekunde ist angebrochen
		{
		FLAG &= (~(0x02));					// FLAG.1 lschen
		new_second();							// current wird von new_second() ausgewertet

		if (charge_time > TRICKLE_TIMEOUT)// wenn der Grenzwert fr die Ladedauer in Phase 3 berschritten ist
			{
			err_msg(ERR_TIMEOUT);
			}
		}

	if (FLAG & 0x04)							// Anzeige von Spannung, Strom und OCR1A aktualisieren
		{
		FLAG &= (~(0x04));					// Flag lschen
		print_zeile1();
		}

	// die Schleife wird beendet, sobald die Spannung hher als der Grenzwert ist.
	}
while (status[U] <= U_TRICKLE);        // Grenzwert der Akkuspannung fr Phase 3 ist berschritten
Stop_PWM();										// Ladung stoppen
}

// --- [ eof ] ----------------------------------------------------------------