/*
 * PCB Exposure Timer, main.c
 *
 * (c) Axel (XL) Schwenke, axel.schwenke@gmx.net
 *
 * $Id: main.c 209 2012-11-29 17:17:18Z schwenke $
 *
 */

/* configurables */

#define DEFAULT_TIME 120    /*  2:00 min */
#define MAXIMUM_TIME 30*60  /* 30:00 min */
#define NUMBER_BEEPS 120    /* number alert beeps */

/* encoder acceleration characteristics */
#define ACCEL 24            /* acceleration */
#define SCALE 32            /* speed scale, power of 2 recommended */
#define VMAX  30            /* max speed */

/* end configurables */


#if MAXIMUM_TIME > 99*60+59
#error MAXIMUM_TIME exceeds 99:59 min
#endif

#if SCALE*VMAX > 32767
#error possible overflow in encoder acceleration, decrease SCALE or VMAX
#endif


#include <avr/eeprom.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

#include <util/atomic.h>
#include <util/delay.h>

#include "hal.h"


/* global variables */
struct {
    int16_t time;      /* seconds */
    int16_t last_time;
    uint8_t alert;     /* 0=disable, 1=enable acoustic alert */
    uint8_t state;     /* enum, see below */
} glob;

/* possible values for glob.state */
#define READY   0
#define RUNNING 1
#define DONE    2

/* display buffer, 5*8 bits, first 35 bits used */
uint8_t display[5];

/* 7 segment decoding table
 * 1=a, 2=b, 4=c ... 0x40=g */
const uint8_t n_to_7[10] =
{
    0x3F, 0x06, 0x5B, 0x4F, 0x66, /* 0-4 */
    0x6D, 0x7D, 0x07, 0x7F, 0x6F  /* 5-9 */
};

/* rotary encoder decoding table */
const int8_t enc_table[16] =
{
    0,  0, -1, 0,
    0,  0,  0, 1,
    1,  0,  0, 0,
    0, -1,  0, 0
};


/* clock tick, incremented every 1 ms */
volatile int16_t vtick;

/* encoder status */
volatile int8_t  enc_delta;  /* -128..+127 */
volatile uint8_t enc_button; /* enum, see below */
#define pressed_short 1
#define pressed_long  2

/* defaults in EEPROM */
EEMEM uint16_t def_time = DEFAULT_TIME;
EEMEM uint8_t def_alert = 1;


/* function prototypes */
void debounce_button(void);
void debounce_encoder(void);
void hardware_init(void);
void refresh(void);
void restart_timer(void);
void setup_display(void);
void show_alert_state(void);


/* here we go! */
int __attribute__((OS_main)) main (void)
{
    hardware_init();
    glob.time = eeprom_read_word(&def_time);
    glob.last_time = glob.time;
    glob.alert = eeprom_read_byte(&def_alert);
    glob.state = READY;
    setup_display();
    refresh();

    uint16_t tick = 0;
    uint16_t last_tick = 0;
    uint8_t  beeps = 0;
    uint8_t  need_refresh = 0;
    uint8_t  button = 0;
    int8_t   delta = 0;

    while (1) {

        /* wait for next timer interrupt */
        sleep_cpu();

        ATOMIC_BLOCK(ATOMIC_RESTORESTATE)
        {
            tick = vtick;
        }

        /* new tick? */
        if (tick == last_tick) {
            continue;
        }
        last_tick = tick;

        /* read volatile variables */
        ATOMIC_BLOCK(ATOMIC_RESTORESTATE)
        {
            button = enc_button; enc_button = 0;
            delta = enc_delta; enc_delta = 0;
        }

        switch (glob.state) {

        case READY:

            if (delta) {
                glob.time += delta;
                if (glob.time < 0) {
                    glob.time = 0;
                }
                if (glob.time > MAXIMUM_TIME) {
                    glob.time = MAXIMUM_TIME;
                }
                need_refresh = 1;
            }

            if (button == pressed_short) {
                if (glob.time) {
                    /* start timer */
                    glob.last_time = glob.time;
                    glob.state = RUNNING;
                    restart_timer();
                    bulb_on();
                    need_refresh = 1;
                } else {
                    show_alert_state();
                    glob.state = DONE;
                }
            }

            if (button == pressed_long) {
                if (glob.time) {
                    /* save time in EEPROM */
                    eeprom_write_word(&def_time, glob.time);
                    need_refresh = 1;
                } else {
                    /* toggle alert status */
                    glob.alert = (glob.alert ? 0 : 1);
                    eeprom_write_byte(&def_alert, glob.alert);
                    show_alert_state();
                }
                /* beep once */
                restart_timer();
                beeps = 1;
                glob.state = DONE;
            }

            break;


        case RUNNING:

            if (tick == 0) {
                glob.time--;
                need_refresh = 1;
                if (glob.alert) {
                    PORTD ^= (_BV(buzzer_pin_a) | _BV(buzzer_pin_b));
                }
            }

            /* button press or fisnished */
            if (glob.time == 0 || button) {
                bulb_off();
                glob.state = DONE;
                need_refresh = 1;
                if (glob.alert) {
                    if (button) {
                        /* beep once at abort */
                        restart_timer();
                        beeps = 1;
                    } else {
                        beeps = NUMBER_BEEPS;
                    }
                }
            }

            break;


        case DONE:

            if (beeps && tick==1) {
                buzzer_on();
                beeps--;
            }
            if (tick==50) {
                buzzer_off();
            }

            if (button) {
                if (glob.time == 0) {
                    glob.time = glob.last_time;
                }
                glob.state = READY;
                need_refresh = 1;
                buzzer_off();
                beeps = 0;
            }

            break;

        } /* end of switch (glob.state) */

        if (need_refresh) {
            setup_display();
            refresh();
            need_refresh = 0;
        }
    }
}


/********************************************************************
 * Sub Routines
 ********************************************************************/

/* scan and debounce the push button
 * we detect short press (at time of button release)
 * and long press (after ~1 sec) */
void debounce_button(void)
{
    static uint8_t bst;  /* button status */
    static uint8_t cnt;  /* button debounce counter */

    /* check if the button status has changed */
    uint8_t i = (bst ^ sample_button) & 1;

    if (i) {
        bst ^= i; /* remember current status  */
        cnt = 25; /* set debounce time = 25ms */
    }
    else {
        /* decrement the debounce/repeat counter */
        if (cnt && --cnt == 0) {
            /* we have a debounced or repeated action */

            if (bst & 1) { /* button press(ed) */
                /* bst is filled with 1's from the right
                 * for each repeat cycle. 8th bit is set
                 * after (6*162 + 25) = 997ms */
                if (bst == 0x7F) {
                    enc_button = pressed_long;
                }
                /* fill bst with 1's from the right */
                bst = (bst<<1) + 1;
                /* set repeat counter = 162ms */
                cnt = 162;
            }
            else {
                /* released after less than 6 repeats? */
                if ((bst & 0x80) == 0) {
                    enc_button = pressed_short;
                }
                /* clear status and disable counter */
                bst = 0;
                cnt = 0;
            }
        }
    }
}


/* half step with acceleration
 * speed goes up by ACCEL when encoder makes a step
 * and down by 1 each ms when encoder makes no step
 * min speed is SCALE, max is SCALE*VMAX
 * effective delta is +/- (speed/SCALE) */
void debounce_encoder(void)
{
    static uint8_t last; /* encoder status */
    static int16_t spd;  /* encoder speed */

    last = ((last << 2) | (phase_A ? 2 : 0) | (phase_B ? 1 : 0)) & 0x0F;
    int8_t delta = enc_table[last];

    if (delta) {
        /* accelerate, with saturation @ VMAX*SCALE */
        spd = (spd < (VMAX*SCALE-ACCEL) ? spd+ACCEL : VMAX*SCALE);

        if (delta > 0) {
            enc_delta += (spd/SCALE);
        } else {
            enc_delta -= (spd/SCALE);
        }

    } else {
        /* decelerate, but not below SCALE */
        spd = (spd > SCALE ? spd-1 : SCALE);
    }
}


/* initialize the hardware */
void hardware_init(void)
{
    /* port B is all inputs */
    PORTB = 0xFF;
    DDRB  = 0x00;

    /* 6 outputs at port D, bulb off
     * complementary levels at buzzer pins */
    PORTD = 0xFF & ~(_BV(bulb_pin) | _BV(buzzer_pin_a));
    DDRD  = (_BV(oe_pin) | _BV(clk_pin) | _BV(data_pin) |
             _BV(bulb_pin) | _BV(buzzer_pin_a) | _BV(buzzer_pin_b));

    /* turn off analog comparator */
    ACSR |= _BV(ACD);

#if defined (__AVR_AT90S2313__)

    /* timer0: free running @ 8MHz/8, or stopped */
    /* TCCR0 = _BV(CS01); */

    /* timer1: CTC mode for 1000Hz, prescaler 64 */
    TCCR1B = _BV(CTC1) | _BV(CS11) | _BV(CS10);
    OCR1   = F_CPU/64/1000;

    /* interrupt on timer0 overflow
     * and timer 1 compare match */
    TIMSK  = _BV(OCIE1A) | _BV(TOIE0);

#elif defined (__AVR_ATtiny2313__)

    /* timer0: CTC mode for 8000Hz, prescaler 8 */
    TCCR0A = _BV(WGM01);
    OCR0A  = F_CPU/8/8000 - 1;
    /* TCCR0B = _BV(CS01); */

    /* timer1: CTC mode for 1000Hz, prescaler 64 */
    TCCR1B = _BV(WGM12) | _BV(CS11) | _BV(CS10);
    OCR1A  = F_CPU/64/1000 - 1;

    /* interrupt on timer0 and timer 1
     * compare match A */
    TIMSK  = _BV(OCIE1A) | _BV(OCIE0A);
#else
#error unknown MCU!
#endif

    /* enable idle sleep */
    MCUCR  = _BV(SE);

    sei();
}


/* copy display buffer onto display */
void refresh(void)
{
    /* start data frame */
    clk_lo();
    oe_lo();

    /* send start bit */
    data_hi();
    wait(); clk_hi();
    wait(); clk_lo();

    /* followed by 35 data bits */
    for (uint8_t bit=0; bit<35; bit++) {
        if (display[bit/8] & (1<<(bit%8))) {
            data_hi();
        } else {
            data_lo();
        }
        wait(); clk_hi();
        wait(); clk_lo();
    }

    /* end data frame */
    oe_hi();
}


/* restart time measurement */
void restart_timer(void)
{
    ATOMIC_BLOCK(ATOMIC_RESTORESTATE)
    {
        vtick = 0;
        TCNT1 = 0;
        TIFR  = _BV(OCIE1A);
    }
}


/* setup display buffers from global variables */
void setup_display(void)
{
    /* setup digits */
    uint16_t t = (uint16_t)glob.time;
    display[3] = n_to_7[t%10]; t /= 10;
    display[2] = n_to_7[t%6];  t /= 6;
    display[1] = n_to_7[t%10]; t /= 10;
    display[0] = (t == 0 ? 0 : n_to_7[t%10]); /* suppress leading 0 */

    /* turn on colon between minutes and seconds */
    display[1]  |= 0x80;    /* bit #16 */

    /* setup status LEDs */
    display[4] = 0;
    if (glob.state == READY) {
        display[4] = 0x02;  /* bit #34 */
    }
    if (glob.state == RUNNING) {
        display[4] = 0x01;  /* bit #33 */
    }
    if (glob.state == DONE) {
        display[0] |= 0x80; /* bit #8 */
    }
}


void show_alert_state(void)
{
    display[0] = 0xF7; /* "A" + LED "Done" */
    display[1] = 0xB8; /* "L" + ":" */
    display[2] = 0;
    display[3] = n_to_7[glob.alert];
    display[4] = 0;
    refresh();
}


#if defined (__AVR_AT90S2313__)

/* once every millisecond */
ISR(TIMER1_COMP1_vect)
{
    sei(); /* allow buzzer interrupt */
    debounce_encoder();
    debounce_button();
    int16_t ltick= vtick;
    vtick = (ltick == 999 ? 0 : ltick+1);
}

/* toggle buzzer pins */
ISR(TIMER0_OVF0_vect)
{
    TCNT0 = 130;
    PORTD ^= (_BV(buzzer_pin_a) | _BV(buzzer_pin_b));
}

#elif defined (__AVR_ATtiny2313__)

/* once every millisecond */
ISR(TIMER1_COMPA_vect)
{
    sei(); /* allow buzzer interrupt */
    debounce_encoder();
    debounce_button();
    int16_t ltick= vtick;
    vtick = (ltick == 999 ? 0 : ltick+1);
}

/* toggle buzzer pins */
ISR(TIMER0_COMPA_vect)
{
    PORTD ^= (_BV(buzzer_pin_a) | _BV(buzzer_pin_b));
}

#else
#error unknown MCU!
#endif

