#include <avr/io.h>
#include <avr/eeprom.h>
#include <util/delay.h>

#include <stdio.h>
#include <stdlib.h>

#include "lib/lcd.h"
#include "lib/simpleprog.h"
#include "lib/ds18b20.h"

void welcome(float steilheit, float offset);
void clear(void);
void writeTemp(uint8_t digit, uint16_t decimal, uint8_t x, uint8_t y);
uint16_t adc_mw(uint8_t adc_port, uint8_t steps);
void kalibrierung(uint8_t pin, uint8_t port, float *steilheit, float *offset);
float ph(uint16_t adc, float *steilheit, float *offset);

int main(void) {
    uint8_t grad;
    uint16_t grad_dec;
    static float EEMEM steil = 0.0;
    static float EEMEM off = 0.0;
    float s = 0.0;
    float o = 0.0;
    float ph_w = 0.0;
    char buffer[20];
    
    pinMode(DDRB, PB1, 0);
    pinMode(DDRB, PB2, 0);
    pinMode(DDRB, PB5, 0);
    anSet(PC5);
    
    s = eeprom_read_float(&steil);
    o = eeprom_read_float(&off);
    welcome(s, o);
    
    while (1) {
        if(digRead(PINB, PB1)){
            kalibrierung(PINB, PB5, &s, &o);
            
            eeprom_update_float(&steil, s);
            eeprom_update_float(&off, o);
            
            s = eeprom_read_float(&steil);
            o = eeprom_read_float(&off);
        }
        
        if(digRead(PINB, PB2)){
            clear();
            lcd_setcursor(0,2);
            lcd_string("       Messung      ");
            ph_w = ph(adc_mw(PC5, 5), &s, &o);
            sprintf(buffer, "      pH=%4.2f      ", ph_w);
            lcd_setcursor(0, 3);
            lcd_string(buffer);
        }
    }
}

void welcome(float steilheit, float offset){
    char buffer[20];
    lcd_init();
    lcd_home();
    lcd_string("      pH-Meter      ");
    lcd_setcursor(0, 2);
    lcd_string("Kalibrierung?  (SW1)");
    lcd_setcursor(0, 3);
    sprintf(buffer, "s=%4.2f    o=%4.2f", steilheit, offset);
    lcd_string(buffer);
}

void clear(void){
    lcd_clear();
    lcd_home();
    lcd_string("      pH-Meter      ");
}

void writeTemp(uint8_t digit, uint16_t decimal, uint8_t x, uint8_t y) {
    char buffer[20];
    lcd_setcursor(x, y);
    sprintf(buffer, "%d.%02u", digit, decimal/100);
    lcd_string(buffer);
    lcd_data(0b11011111);
    lcd_string("C");
}

uint16_t adc_mw(uint8_t adc_port, uint8_t steps){
    anSet(adc_port);
    uint16_t adc;
    uint32_t adc_list = 0;
    lcd_setcursor(0,4);
    for(int i = 0; i<steps; i++){
        for(int j = 0; j < 20/steps; j++){
            lcd_data(0b11111111);
        }
        adc = anRead(adc_port);
        adc_list += adc;
        _delay_ms(200);
    }
    return adc_list / steps;
}

void kalibrierung(uint8_t pin, uint8_t port, float *steilheit, float *offset){
    uint16_t admw[2];
    char buffer[20];
    
    clear();
    lcd_setcursor(0, 2);
    lcd_string("    Kalibrierung    ");
    
    lcd_setcursor(0, 3);
    lcd_string("PH-4    Start");
    while(1){
        if(digRead(pin, port)){
            break;
        }
    }
    lcd_setcursor(0, 3);
    lcd_string("PH-4    Messe...");
    admw[0] = adc_mw(PC5, 5);
    lcd_setcursor(0, 3);
    lcd_string("PH-4    ADC-MW:");
    sprintf(buffer, "%i", admw[0]);
    lcd_string(buffer);
    
    while(1){
        if(digRead(pin, port)){
            break;
        }
    }
    
    clear();
    lcd_setcursor(0, 2);
    lcd_string("    Kalibrierung    ");
    
    lcd_setcursor(0, 3);
    lcd_string("PH-7    Start");
    while(1){
        if(digRead(pin, port)){
            break;
        }
    }
    lcd_setcursor(0, 3);
    lcd_string("PH-7    Messe...");
    admw[1] = adc_mw(PC5, 5);
    lcd_setcursor(0, 3);
    lcd_string("PH-7    ADC-MW:");
    sprintf(buffer, "%i", admw[1]);
    lcd_string(buffer);
    
    while(1){
        if(digRead(pin, port)){
            break;
        }
    }
    
    clear();
    lcd_setcursor(0, 2);
    lcd_string("    Kalibrierung    ");
    
    lcd_setcursor(0, 3);
    sprintf(buffer, "U4:%i    ", admw[0]);
    lcd_string(buffer);
    sprintf(buffer, "U7:%i", admw[1]);
    lcd_string(buffer);
    
    lcd_setcursor(0, 4);
    
    *steilheit = (7.0-4.0)/(admw[0]-admw[1]);   
    sprintf(buffer, "s=%4.2f    ", *steilheit);
    lcd_string(buffer);
    
    *offset = (-1.0)*(*steilheit*admw[1]-7.0);
    sprintf(buffer, "o=%4.2f", *offset);
    lcd_string(buffer);
}

float ph(uint16_t adc, float *steilheit, float *offset){
    float ph = 0.0;
    ph = *steilheit * adc + *offset;
    return ph;
}
