//************************************************
//*** Debouncing 8 Keys                        ***
//*** Sampling 4 Times                         ***
//*** With Repeat Function                     ***
//***                                          ***
//*** Author:      Peter Dannegger             ***
//*** Modified by: Thomas Finke                ***
//************************************************

#include <avr/io.h>
#include <avr/interrupt.h>
#include "debounce.h"

//************************************************
//*** Function for initialization              ***
//************************************************
void key_init(void)
{
	KEY_DDR &= ~ALL_KEYS;                				// configure key port for input
	KEY_PORT |= ALL_KEYS;               				// and turn on pull up resistors
}

//************************************************
//*** Function to debounce the keys            ***
//*** This function has to be called each 10ms ***
//************************************************
void key_debounce(void)
{
	static unsigned char ct0, ct1,rpt;
	unsigned char i;

	i = key_state ^ ~KEY_PIN;                       	// set the bit to "1" if the key is pressed and was not pressed before
	ct0 = ~( ct0 & i );                             	// Toggle ct0 if the key is pressed (Counter Bit0)
	ct1 = ct0 ^ (ct1 & i);                          	// Count if the key is pressed (Counter Bit 1)
	i &= ct0 & ct1;                                 	// set the bit to "1" if the key is pressed and the counter reaches "4"...
	key_state ^= i;                                 	// ...then toggle debounced state
	key_press |= key_state & i;

	if ((key_state & REPEAT_MASK) == 0)					// check repeat function
	{
		rpt = REPEAT_START;                      		// start delay
	}

	if (--rpt == 0)
	{
		rpt = REPEAT_NEXT;                          	// repeat delay
		key_rpt |= key_state & REPEAT_MASK;
	}
}

//************************************************
//*** Check if a key has been pressed          ***
//*** Each key is reported only once           ***
//************************************************
unsigned char get_key_press(unsigned char key_mask)
{
	cli();                                        	  	// read and clear atomic !

//	key_press &= ~KEY_PIN;								// clear keys that are not pressed any longer

	key_mask &= key_press;                         	 	// read key(s)
	key_press ^= key_mask;                         	 	// clear key(s)
	sei();
	return key_mask;
}

//************************************************
//*** Check if a key has been pressed long     ***
//*** enough such that the key repeat          ***
//*** functionality kicks in. After a small    ***
//*** setup delay the key is reported beeing   ***
//*** pressed in subsequent calls to this      ***
//*** function. This simulates the user        ***
//*** repeatedly pressing and releasing the key***
//************************************************
unsigned char get_key_rpt(unsigned char key_mask)
{
	cli();                                         	 	// read and clear atomic !
	key_mask &= key_rpt;                           	 	// read key(s)
	key_rpt ^= key_mask;                           	 	// clear key(s)
	sei();
	return key_mask;
}

//************************************************
//*** Check if a key has been pressed short    ***
//************************************************ 
unsigned char get_key_short(unsigned char key_mask)
{
	cli();                                          		// read key state and key press atomic !
	return get_key_press( ~key_state & key_mask );
}

//************************************************
//*** Check if a key has been pressed long     ***
//************************************************ 
unsigned char get_key_long(unsigned char key_mask)
{
	return get_key_press( get_key_rpt( key_mask ));
}
